/*
 * Copyright The OpenTelemetry Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import { ValueType } from '@opentelemetry/api';
import { hrTimeToNanoseconds } from '@opentelemetry/core';
import { AggregationTemporality, DataPointType, } from '@opentelemetry/sdk-metrics';
import { toAttributes } from '../common/internal';
export function toResourceMetrics(resourceMetrics) {
    return {
        resource: {
            attributes: toAttributes(resourceMetrics.resource.attributes),
            droppedAttributesCount: 0,
        },
        schemaUrl: undefined,
        scopeMetrics: toScopeMetrics(resourceMetrics.scopeMetrics),
    };
}
export function toScopeMetrics(scopeMetrics) {
    return Array.from(scopeMetrics.map(metrics => {
        const scopeMetrics = {
            scope: {
                name: metrics.scope.name,
                version: metrics.scope.version,
            },
            metrics: metrics.metrics.map(metricData => toMetric(metricData)),
            schemaUrl: metrics.scope.schemaUrl,
        };
        return scopeMetrics;
    }));
}
export function toMetric(metricData) {
    const out = {
        name: metricData.descriptor.name,
        description: metricData.descriptor.description,
        unit: metricData.descriptor.unit,
    };
    const aggregationTemporality = toAggregationTemporality(metricData.aggregationTemporality);
    if (metricData.dataPointType === DataPointType.SUM) {
        out.sum = {
            aggregationTemporality,
            isMonotonic: metricData.isMonotonic,
            dataPoints: toSingularDataPoints(metricData),
        };
    }
    else if (metricData.dataPointType === DataPointType.GAUGE) {
        // Instrument is a gauge.
        out.gauge = {
            dataPoints: toSingularDataPoints(metricData),
        };
    }
    else if (metricData.dataPointType === DataPointType.HISTOGRAM) {
        out.histogram = {
            aggregationTemporality,
            dataPoints: toHistogramDataPoints(metricData),
        };
    }
    else if (metricData.dataPointType === DataPointType.EXPONENTIAL_HISTOGRAM) {
        out.exponentialHistogram = {
            aggregationTemporality,
            dataPoints: toExponentialHistogramDataPoints(metricData),
        };
    }
    return out;
}
function toSingularDataPoint(dataPoint, valueType) {
    const out = {
        attributes: toAttributes(dataPoint.attributes),
        startTimeUnixNano: hrTimeToNanoseconds(dataPoint.startTime),
        timeUnixNano: hrTimeToNanoseconds(dataPoint.endTime),
    };
    if (valueType === ValueType.INT) {
        out.asInt = dataPoint.value;
    }
    else if (valueType === ValueType.DOUBLE) {
        out.asDouble = dataPoint.value;
    }
    return out;
}
function toSingularDataPoints(metricData) {
    return metricData.dataPoints.map(dataPoint => {
        return toSingularDataPoint(dataPoint, metricData.descriptor.valueType);
    });
}
function toHistogramDataPoints(metricData) {
    return metricData.dataPoints.map(dataPoint => {
        const histogram = dataPoint.value;
        return {
            attributes: toAttributes(dataPoint.attributes),
            bucketCounts: histogram.buckets.counts,
            explicitBounds: histogram.buckets.boundaries,
            count: histogram.count,
            sum: histogram.sum,
            min: histogram.min,
            max: histogram.max,
            startTimeUnixNano: hrTimeToNanoseconds(dataPoint.startTime),
            timeUnixNano: hrTimeToNanoseconds(dataPoint.endTime),
        };
    });
}
function toExponentialHistogramDataPoints(metricData) {
    return metricData.dataPoints.map(dataPoint => {
        const histogram = dataPoint.value;
        return {
            attributes: toAttributes(dataPoint.attributes),
            count: histogram.count,
            min: histogram.min,
            max: histogram.max,
            sum: histogram.sum,
            positive: {
                offset: histogram.positive.offset,
                bucketCounts: histogram.positive.bucketCounts,
            },
            negative: {
                offset: histogram.negative.offset,
                bucketCounts: histogram.negative.bucketCounts,
            },
            scale: histogram.scale,
            zeroCount: histogram.zeroCount,
            startTimeUnixNano: hrTimeToNanoseconds(dataPoint.startTime),
            timeUnixNano: hrTimeToNanoseconds(dataPoint.endTime),
        };
    });
}
function toAggregationTemporality(temporality) {
    if (temporality === AggregationTemporality.DELTA) {
        return 1 /* AGGREGATION_TEMPORALITY_DELTA */;
    }
    if (temporality === AggregationTemporality.CUMULATIVE) {
        return 2 /* AGGREGATION_TEMPORALITY_CUMULATIVE */;
    }
    return 0 /* AGGREGATION_TEMPORALITY_UNSPECIFIED */;
}
//# sourceMappingURL=internal.js.map