// src/protocols/webdriver.ts
var webdriver_default = {
  "/session": {
    POST: {
      command: "newSession",
      description: "The New Session command creates a new WebDriver session with the endpoint node. If the creation fails, a session not created error is returned.",
      ref: "https://w3c.github.io/webdriver/#dfn-new-sessions",
      parameters: [
        {
          name: "capabilities",
          type: "object",
          description: "a JSON object, the set of capabilities that was ultimately merged and matched in the capability processing algorithm",
          required: true
        }
      ],
      returns: {
        type: "Object",
        name: "session",
        description: "Object containing sessionId and capabilities of created WebDriver session."
      }
    }
  },
  "/session/:sessionId": {
    DELETE: {
      command: "deleteSession",
      description: "The Delete Session command closes any top-level browsing contexts associated with the current session, terminates the connection, and finally closes the current session.",
      ref: "https://w3c.github.io/webdriver/#dfn-delete-session",
      parameters: [{
        name: "deleteSessionOpts",
        type: "object",
        description: "Object containing options for the deleteSession command, e.g. `{ shutdownDriver: boolean }`",
        required: false
      }]
    }
  },
  "/status": {
    GET: {
      command: "status",
      description: "The Status command returns information about whether a remote end is in a state in which it can create new sessions and can additionally include arbitrary meta information that is specific to the implementation.",
      ref: "https://w3c.github.io/webdriver/#dfn-status",
      parameters: [],
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L5-L16"],
      returns: {
        type: "Object",
        name: "status",
        description: "Object containing status of the driver status."
      }
    }
  },
  "/session/:sessionId/timeouts": {
    GET: {
      command: "getTimeouts",
      description: "The Get Timeouts command gets timeout durations associated with the current session.",
      ref: "https://w3c.github.io/webdriver/#dfn-get-timeouts",
      parameters: [],
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L20-L24"],
      returns: {
        type: "Object",
        name: "timeouts",
        description: "Object containing timeout durations for `script`, `pageLoad` and `implicit` timeouts."
      }
    },
    POST: {
      command: "setTimeouts",
      description: "The Set Timeouts command sets timeout durations associated with the current session. The timeouts that can be controlled are listed in the table of session timeouts below.",
      ref: "https://w3c.github.io/webdriver/#dfn-set-timeouts",
      parameters: [
        {
          name: "implicit",
          type: "number",
          description: "integer in ms for session implicit wait timeout",
          required: false
        },
        {
          name: "pageLoad",
          type: "number",
          description: "integer in ms for session page load timeout",
          required: false
        },
        {
          name: "script",
          type: "number",
          description: "integer in ms for session script timeout",
          required: false
        }
      ],
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L28-L33"]
    }
  },
  "/session/:sessionId/url": {
    GET: {
      command: "getUrl",
      description: "The Get Current URL command returns the URL of the current top-level browsing context.",
      ref: "https://w3c.github.io/webdriver/#dfn-get-current-url",
      parameters: [],
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L39-L43"],
      returns: {
        type: "string",
        name: "url",
        description: "current top-level browsing context\u2019s active document\u2019s document URL"
      }
    },
    POST: {
      command: "navigateTo",
      description: "The navigateTo (go) command is used to cause the user agent to navigate the current top-level browsing context a new location.",
      ref: "https://w3c.github.io/webdriver/#dfn-navigate-to",
      parameters: [
        {
          name: "url",
          type: "string",
          description: "string representing an absolute URL (beginning with http(s)), possibly including a fragment (#...), could also be a local scheme (about: etc)",
          required: true
        }
      ],
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L47-L51"],
      alternativeCommands: ["browser/url"]
    }
  },
  "/session/:sessionId/back": {
    POST: {
      command: "back",
      description: "The Back command causes the browser to traverse one step backward in the joint session history of the current top-level browsing context. This is equivalent to pressing the back button in the browser chrome or calling `window.history.back`.",
      ref: "https://w3c.github.io/webdriver/#dfn-back",
      parameters: [],
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L55-L59"]
    }
  },
  "/session/:sessionId/forward": {
    POST: {
      command: "forward",
      description: "The Forward command causes the browser to traverse one step forwards in the joint session history of the current top-level browsing context.",
      ref: "https://w3c.github.io/webdriver/#dfn-forward",
      parameters: [],
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L63-L69"]
    }
  },
  "/session/:sessionId/refresh": {
    POST: {
      command: "refresh",
      description: "The Refresh command causes the browser to reload the page in current top-level browsing context.",
      ref: "https://w3c.github.io/webdriver/#dfn-refresh",
      parameters: [],
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L73-L78"]
    }
  },
  "/session/:sessionId/title": {
    GET: {
      command: "getTitle",
      description: "The Get Title command returns the document title of the current top-level browsing context, equivalent to calling `document.title`.",
      ref: "https://w3c.github.io/webdriver/#dfn-get-title",
      parameters: [],
      returns: {
        type: "string",
        name: "title",
        description: "Returns a string which is the same as `document.title` of the current top-level browsing context."
      },
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L82-L86"]
    }
  },
  "/session/:sessionId/window": {
    GET: {
      command: "getWindowHandle",
      description: "The Get Window Handle command returns the window handle for the current top-level browsing context. It can be used as an argument to Switch To Window.",
      ref: "https://w3c.github.io/webdriver/#dfn-get-window-handle",
      parameters: [],
      returns: {
        type: "string",
        name: "handle",
        description: "Returns a string which is the window handle for the current top-level browsing context."
      },
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L90-L93"]
    },
    DELETE: {
      command: "closeWindow",
      description: "The Close Window command closes the current top-level browsing context. Once done, if there are no more top-level browsing contexts open, the WebDriver session itself is closed.",
      ref: "https://w3c.github.io/webdriver/#dfn-close-window",
      parameters: [],
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L97-L117"]
    },
    POST: {
      command: "switchToWindow",
      description: "The Switch To Window command is used to select the current top-level browsing context for the current session, i.e. the one that will be used for processing commands.",
      ref: "https://w3c.github.io/webdriver/#dfn-switch-to-window",
      parameters: [
        {
          name: "handle",
          type: "string",
          description: "a string representing a window handle, should be one of the strings that was returned in a call to getWindowHandles",
          required: true
        }
      ],
      alternativeCommands: ["browser/switchWindow"],
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L121-L130"]
    }
  },
  "/session/:sessionId/window/new": {
    POST: {
      command: "createWindow",
      description: "Create a new top-level browsing context.",
      ref: "https://w3c.github.io/webdriver/#new-window",
      parameters: [
        {
          name: "type",
          type: "string",
          description: "Set to 'tab' if the newly created window shares an OS-level window with the current browsing context, or 'window' otherwise.",
          required: true
        }
      ],
      returns: {
        type: "Object",
        name: "window",
        description: "New window object containing 'handle' with the value of the handle and 'type' with the value of the created window type"
      },
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L134-L136"]
    }
  },
  "/session/:sessionId/window/handles": {
    GET: {
      command: "getWindowHandles",
      description: "The Get Window Handles command returns a list of window handles for every open top-level browsing context. The order in which the window handles are returned is arbitrary.",
      ref: "https://w3c.github.io/webdriver/#dfn-get-window-handles",
      parameters: [],
      returns: {
        type: "String[]",
        name: "handles",
        description: "An array which is a list of window handles."
      },
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L140-L143"]
    }
  },
  "/session/:sessionId/print": {
    POST: {
      command: "printPage",
      description: "The Print Page command renders the document to a paginated PDF document. __Note:__ Chrome currently only supports this in [headless mode](https://webdriver.io/docs/capabilities/#run-browser-headless), see [`crbug753118`](https://bugs.chromium.org/p/chromium/issues/detail?id=753118)).",
      ref: "https://w3c.github.io/webdriver/#print-page",
      parameters: [
        {
          name: "orientation",
          type: "string",
          description: "page orientation. Default: `portrait`",
          required: false
        },
        {
          name: "scale",
          type: "number",
          description: "page scale. Default: `1`",
          required: false
        },
        {
          name: "background",
          type: "boolean",
          description: "page background. Default: `false`",
          required: false
        },
        {
          name: "width",
          type: "number",
          description: "page width in cm. Default: `21.59` from page",
          required: false
        },
        {
          name: "height",
          type: "number",
          description: "page height in cm. Default: `27.94` from page",
          required: false
        },
        {
          name: "top",
          type: "number",
          description: "page margin in cm from top margin. Default: `1`",
          required: false
        },
        {
          name: "bottom",
          type: "number",
          description: "page margin in cm from bottom margin. Default: `1`",
          required: false
        },
        {
          name: "left",
          type: "number",
          description: "page margin in cm from left margin. Default: `1`",
          required: false
        },
        {
          name: "right",
          type: "number",
          description: "page margin in cm from right margin. Default: `1`",
          required: false
        },
        {
          name: "shrinkToFit",
          type: "boolean",
          description: "shrink pdf to fit in page. Default: `true`",
          required: false
        },
        {
          name: "pageRanges",
          type: "object[]",
          description: "page ranges. Default `[]`",
          required: false
        }
      ],
      returns: {
        type: "string",
        name: "pdf",
        description: "The base64-encoded PDF representation of the paginated document."
      },
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L150-L151"]
    }
  },
  "/session/:sessionId/frame": {
    POST: {
      command: "switchToFrame",
      deprecated: "This command is deprecated and we encourage everyone to use `switchFrame` instead for switching into frames. Read more about this command at https://webdriver.io/docs/api/browser/switchFrame.",
      description: "The Switch To Frame command is used to select the current top-level browsing context or a child browsing context of the current browsing context to use as the current browsing context for subsequent commands.",
      ref: "https://w3c.github.io/webdriver/#dfn-switch-to-frame",
      parameters: [
        {
          name: "id",
          type: "(number|object|null)",
          description: "one of three possible types: null: this represents the top-level browsing context (i.e., not an iframe), a Number, representing the index of the window object corresponding to a frame, an Element object received using `findElement`.",
          required: true
        }
      ],
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L155-L168"]
    }
  },
  "/session/:sessionId/frame/parent": {
    POST: {
      command: "switchToParentFrame",
      description: "The Switch to Parent Frame command sets the current browsing context for future commands to the parent of the current browsing context.",
      ref: "https://w3c.github.io/webdriver/#dfn-switch-to-parent-frame",
      parameters: [],
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L172-L189"]
    }
  },
  "/session/:sessionId/window/rect": {
    GET: {
      command: "getWindowRect",
      description: "The Get Window Rect command returns the size and position on the screen of the operating system window corresponding to the current top-level browsing context.",
      ref: "https://w3c.github.io/webdriver/#dfn-get-window-rect",
      parameters: [],
      returns: {
        type: "Object",
        name: "windowRect",
        description: 'A JSON representation of a "window rect" object. This has 4 properties: `x`, `y`, `width` and `height`.'
      },
      alternativeCommands: ["browser/getWindowSize"],
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L193-L196"]
    },
    POST: {
      command: "setWindowRect",
      description: "The Set Window Rect command alters the size and the position of the operating system window corresponding to the current top-level browsing context.",
      ref: "https://w3c.github.io/webdriver/#dfn-set-window-rect",
      parameters: [
        {
          name: "x",
          type: "(number|null)",
          description: "the screenX attribute of the window object",
          required: true
        },
        {
          name: "y",
          type: "(number|null)",
          description: "the screenY attribute of the window object",
          required: true
        },
        {
          name: "width",
          type: "(number|null)",
          description: "the width of the outer dimensions of the top-level browsing context, including browser chrome etc...",
          required: true
        },
        {
          name: "height",
          type: "(number|null)",
          description: "the height of the outer dimensions of the top-level browsing context, including browser chrome etc...",
          required: true
        }
      ],
      alternativeCommands: ["browser/setWindowSize"],
      returns: {
        type: "Object",
        name: "windowRect",
        description: 'A JSON representation of a "window rect" object based on the new window state.'
      },
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L200-L204"]
    }
  },
  "/session/:sessionId/window/maximize": {
    POST: {
      command: "maximizeWindow",
      description: 'The Maximize Window command invokes the window manager-specific "maximize" operation, if any, on the window containing the current top-level browsing context. This typically increases the window to the maximum available size without going full-screen.',
      ref: "https://w3c.github.io/webdriver/#dfn-maximize-window",
      parameters: [],
      returns: {
        type: "Object",
        name: "windowRect",
        description: 'A JSON representation of a "window rect" object based on the new window state.'
      },
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L208-L212"]
    }
  },
  "/session/:sessionId/window/minimize": {
    POST: {
      command: "minimizeWindow",
      description: 'The Minimize Window command invokes the window manager-specific "minimize" operation, if any, on the window containing the current top-level browsing context. This typically hides the window in the system tray.',
      ref: "https://w3c.github.io/webdriver/#dfn-minimize-window",
      parameters: [],
      returns: {
        type: "Object",
        name: "windowRect",
        description: 'A JSON representation of a "window rect" object of the (new) current top-level browsing context.'
      }
    }
  },
  "/session/:sessionId/window/fullscreen": {
    POST: {
      command: "fullscreenWindow",
      description: "The Fullscreen Window command invokes the window manager-specific \u201Cfull screen\u201D operation, if any, on the window containing the current top-level browsing context. This typically increases the window to the size of the physical display and can hide browser chrome elements such as toolbars.",
      ref: "https://w3c.github.io/webdriver/#dfn-fullscreen-window",
      parameters: [],
      returns: {
        type: "Object",
        name: "windowRect",
        description: 'A JSON representation of a "window rect" object of the (new) current top-level browsing context.'
      }
    }
  },
  "/session/:sessionId/element": {
    POST: {
      command: "findElement",
      description: "The Find Element command is used to find an element in the current browsing context that can be used for future commands. This command returns JSON representation of the element that can be passed to $ command to transform the reference to an extended WebdriverIO element.",
      ref: "https://w3c.github.io/webdriver/#dfn-find-element",
      parameters: [
        {
          name: "using",
          type: "string",
          description: "a valid element location strategy",
          required: true
        },
        {
          name: "value",
          type: "string",
          description: "the actual selector that will be used to find an element",
          required: true
        }
      ],
      returns: {
        type: "object",
        name: "element",
        description: "A JSON representation of an element object, e.g. `{ 'element-6066-11e4-a52e-4f735466cecf': 'ELEMENT_1' }`."
      },
      alternativeCommands: ["browser/$"],
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L226-L232"]
    }
  },
  "/session/:sessionId/shadow/:shadowId/element": {
    POST: {
      command: "findElementFromShadowRoot",
      description: "The Find Element From Shadow Root command is used to find an element within the shadow root of an element that can be used for future commands. This command returns JSON representation of the element that can be passed to $ command to transform the reference to an extended WebdriverIO element.",
      ref: "https://w3c.github.io/webdriver/#find-element-from-shadow-root",
      variables: [
        {
          name: "shadowId",
          description: "element id of a shadow root element"
        }
      ],
      parameters: [
        {
          name: "using",
          type: "string",
          description: "a valid element location strategy",
          required: true
        },
        {
          name: "value",
          type: "string",
          description: "the actual selector that will be used to find an element",
          required: true
        }
      ],
      alternativeCommands: ["element/shadow$"],
      returns: {
        type: "object",
        name: "element",
        description: "A JSON representation of an element shadow object, e.g. `{ 'element-6066-11e4-a52e-4f735466cecf': 'ELEMENT_1' }`."
      },
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L236-L248"]
    }
  },
  "/session/:sessionId/elements": {
    POST: {
      command: "findElements",
      description: "The Find Elements command is used to find elements in the current browsing context that can be used for future commands. This command returns array of JSON representation of the elements that can be passed to $ command to transform the reference to an extended WebdriverIO element (See findElement).",
      ref: "https://w3c.github.io/webdriver/#dfn-find-elements",
      parameters: [
        {
          name: "using",
          type: "string",
          description: "a valid element location strategy",
          required: true
        },
        {
          name: "value",
          type: "string",
          description: "the actual selector that will be used to find an element",
          required: true
        }
      ],
      alternativeCommands: ["browser/$$"],
      returns: {
        type: "object[]",
        name: "elements",
        description: "A (possibly empty) JSON list of representations of an element object, e.g. `[{ 'element-6066-11e4-a52e-4f735466cecf': 'ELEMENT_1' }]`."
      },
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L252-L254"]
    }
  },
  "/session/:sessionId/shadow/:shadowId/elements": {
    POST: {
      command: "findElementsFromShadowRoot",
      description: "The Find Elements command is used to find elements within the shadow root of an element that can be used for future commands. This command returns array of JSON representation of the elements that can be passed to $ command to transform the reference to an extended WebdriverIO element (See findElement).",
      ref: "https://w3c.github.io/webdriver/#find-elements-from-shadow-root",
      variables: [
        {
          name: "shadowId",
          description: "element id of a shadow root element"
        }
      ],
      alternativeCommands: ["element/shadow$$"],
      parameters: [
        {
          name: "using",
          type: "string",
          description: "a valid element location strategy",
          required: true
        },
        {
          name: "value",
          type: "string",
          description: "the actual selector that will be used to find an element",
          required: true
        }
      ],
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L258-L268"],
      returns: {
        type: "object[]",
        name: "elements",
        description: "A (possibly empty) JSON list of representations of an element object, e.g. `{ 'element-6066-11e4-a52e-4f735466cecf': 'ELEMENT_1' }`."
      }
    }
  },
  "/session/:sessionId/element/:elementId/element": {
    POST: {
      command: "findElementFromElement",
      description: "The Find Element From Element command is used to find an element from a web element in the current browsing context that can be used for future commands. This command returns JSON representation of the element that can be passed to $ command to transform the reference to an extended WebdriverIO element (See findElement).",
      ref: "https://w3c.github.io/webdriver/#dfn-find-element-from-element",
      variables: [
        {
          name: "elementId",
          description: "the id of an element returned in a previous call to Find Element(s)"
        }
      ],
      parameters: [
        {
          name: "using",
          type: "string",
          description: "a valid element location strategy",
          required: true
        },
        {
          name: "value",
          type: "string",
          description: "the actual selector that will be used to find an element",
          required: true
        }
      ],
      alternativeCommands: ["element/$"],
      returns: {
        type: "object",
        name: "element",
        description: "A JSON representation of an element object, e.g. `{ 'element-6066-11e4-a52e-4f735466cecf': 'ELEMENT_1' }`."
      },
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L272-L279"]
    }
  },
  "/session/:sessionId/element/:elementId/elements": {
    POST: {
      command: "findElementsFromElement",
      description: "The Find Elements From Element command is used to find elements from a web element in the current browsing context that can be used for future commands. This command returns array of JSON representation of the elements that can be passed to $ command to transform the reference to an extended WebdriverIO element (See findElement).",
      ref: "https://w3c.github.io/webdriver/#dfn-find-elements-from-element",
      variables: [
        {
          name: "elementId",
          description: "the id of an element returned in a previous call to Find Element(s)"
        }
      ],
      parameters: [
        {
          name: "using",
          type: "string",
          description: "a valid element location strategy",
          required: true
        },
        {
          name: "value",
          type: "string",
          description: "the actual selector that will be used to find an element",
          required: true
        }
      ],
      alternativeCommands: ["element/$$"],
      returns: {
        type: "object[]",
        name: "elements",
        description: "A (possibly empty) JSON list of representations of an element object, e.g. `[{ 'element-6066-11e4-a52e-4f735466cecf': 'ELEMENT_1' }]`."
      },
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L283-L290"]
    }
  },
  "/session/:sessionId/element/:elementId/shadow": {
    GET: {
      command: "getElementShadowRoot",
      description: "Get the shadow root object of an element. The result object can be used to fetch elements within this shadow root using e.g. findElementFromShadowRoots or findElementsFromShadowRoots.",
      ref: "https://w3c.github.io/webdriver/#dfn-get-active-element",
      variables: [
        {
          name: "elementId",
          description: "the id of an element returned in a previous call to Find Element(s)"
        }
      ],
      parameters: [],
      returns: {
        type: "string",
        name: "shadowRoot",
        description: "A JSON representation of an element shadow root, e.g. `{ 'shadow-6066-11e4-a52e-4f735466cecf': 'ELEMENT_1' }`."
      },
      alternativeCommands: ["element/shadow$"],
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L294-L305"]
    }
  },
  "/session/:sessionId/element/active": {
    GET: {
      command: "getActiveElement",
      description: "Get Active Element returns the active element of the current browsing context\u2019s document element. This command returns JSON representation of the element that can be passed to $ command to transform the reference to an extended WebdriverIO element (See findElement).",
      ref: "https://w3c.github.io/webdriver/#dfn-get-active-element",
      parameters: [],
      returns: {
        type: "string",
        name: "element",
        description: "A JSON representation of an element object, e.g. `{ 'element-6066-11e4-a52e-4f735466cecf': 'ELEMENT_1' }`."
      },
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L309-L316"]
    }
  },
  "/session/:sessionId/element/:elementId/selected": {
    GET: {
      command: "isElementSelected",
      description: "Is Element Selected determines if the referenced element is selected or not. This operation only makes sense on input elements of the Checkbox- and Radio Button states, or option elements.",
      ref: "https://w3c.github.io/webdriver/#dfn-is-element-selected",
      variables: [
        {
          name: "elementId",
          description: "the id of an element returned in a previous call to Find Element(s)"
        }
      ],
      parameters: [],
      returns: {
        type: "Boolean",
        name: "isSelected",
        description: "`true` or `false` based on the selected state."
      },
      alternativeCommands: ["element/isSelected"],
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L322-L325"]
    }
  },
  "/session/:sessionId/element/:elementId/displayed": {
    GET: {
      command: "isElementDisplayed",
      description: "Is Element Displayed determines the visibility of an element which is guided by what is perceptually visible to the human eye. In this context, an element's displayedness does not relate to the `visibility` or `display` style properties.",
      ref: "https://w3c.github.io/webdriver/#element-displayedness",
      variables: [
        {
          name: "elementId",
          description: "the id of an element returned in a previous call to Find Element(s)"
        }
      ],
      parameters: [],
      returns: {
        type: "Boolean",
        name: "isDisplayed",
        description: "`true` or `false` based on the visible state."
      },
      alternativeCommands: ["element/isDisplayed"],
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L331-L333"]
    }
  },
  "/session/:sessionId/element/:elementId/attribute/:name": {
    GET: {
      command: "getElementAttribute",
      description: "The Get Element Attribute command will return the attribute of a web element.",
      ref: "https://w3c.github.io/webdriver/#dfn-get-element-attribute",
      variables: [
        {
          name: "elementId",
          description: "the id of an element returned in a previous call to Find Element(s)"
        },
        {
          name: "name",
          description: "name of the attribute value to retrieve"
        }
      ],
      parameters: [],
      returns: {
        type: "string",
        name: "attribute",
        description: "The named attribute of the element."
      },
      alternativeCommands: ["element/getAttribute"],
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L339-L341"]
    }
  },
  "/session/:sessionId/element/:elementId/property/:name": {
    GET: {
      command: "getElementProperty",
      description: "The Get Element Property command will return the result of getting a property of an element.",
      ref: "https://w3c.github.io/webdriver/#dfn-get-element-property",
      variables: [
        {
          name: "elementId",
          description: "the id of an element returned in a previous call to Find Element(s)"
        },
        {
          name: "name",
          description: "name of the attribute property to retrieve"
        }
      ],
      parameters: [],
      returns: {
        type: "string",
        name: "property",
        description: "The named property of the element, accessed by calling GetOwnProperty on the element object."
      },
      alternativeCommands: ["element/getProperty"],
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L347-L349"]
    }
  },
  "/session/:sessionId/element/:elementId/css/:propertyName": {
    GET: {
      command: "getElementCSSValue",
      description: "The Get Element CSS Value command retrieves the computed value of the given CSS property of the given web element.",
      ref: "https://w3c.github.io/webdriver/#dfn-get-element-css-value",
      variables: [
        {
          name: "elementId",
          description: "the id of an element returned in a previous call to Find Element(s)"
        },
        {
          name: "propertyName",
          description: "name of the CSS property to retrieve"
        }
      ],
      parameters: [],
      alternativeCommands: ["element/getCSSProperty"],
      returns: {
        type: "string",
        name: "cssValue",
        description: "The computed value of the parameter corresponding to property name from the element's style declarations (unless the document type is xml, in which case the return value is simply the empty string)."
      },
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L355-L357"]
    }
  },
  "/session/:sessionId/element/:elementId/text": {
    GET: {
      command: "getElementText",
      description: `The Get Element Text command intends to return an element\u2019s text "as rendered". An element's rendered text is also used for locating a elements by their link text and partial link text.`,
      ref: "https://w3c.github.io/webdriver/#dfn-get-element-text",
      variables: [
        {
          name: "elementId",
          description: "the id of an element returned in a previous call to Find Element(s)"
        }
      ],
      parameters: [],
      returns: {
        type: "string",
        name: "text",
        description: "The visible text of the element (including child elements), following the algorithm defined in the Selenium Atoms for [`bot.dom.getVisibleText`](https://github.com/SeleniumHQ/selenium/blob/e09e28f016c9f53196cf68d6f71991c5af4a35d4/javascript/atoms/dom.js#L981)."
      },
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L363-L365"]
    }
  },
  "/session/:sessionId/element/:elementId/name": {
    GET: {
      command: "getElementTagName",
      description: "The Get Element Tag Name command returns the qualified element name of the given web element.",
      ref: "https://w3c.github.io/webdriver/#dfn-get-element-tag-name",
      variables: [
        {
          name: "elementId",
          description: "the id of an element returned in a previous call to Find Element(s)"
        }
      ],
      parameters: [],
      returns: {
        type: "string",
        name: "text",
        description: "The tagName attribute of the element."
      },
      alternativeCommands: ["element/getTagName"],
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L371-L373"]
    }
  },
  "/session/:sessionId/element/:elementId/rect": {
    GET: {
      command: "getElementRect",
      description: "The Get Element Rect command returns the dimensions and coordinates of the given web element.",
      ref: "https://w3c.github.io/webdriver/#dfn-get-element-rect",
      variables: [
        {
          name: "elementId",
          description: "the id of an element returned in a previous call to Find Element(s)"
        }
      ],
      parameters: [],
      returns: {
        type: "Object",
        name: "elementRect",
        description: "A JSON object representing the position and bounding rect of the element."
      },
      alternativeCommands: ["element/getSize", "element/getLocation"],
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L379-L381"]
    }
  },
  "/session/:sessionId/element/:elementId/enabled": {
    GET: {
      command: "isElementEnabled",
      description: "Is Element Enabled determines if the referenced element is enabled or not. This operation only makes sense on form controls.",
      ref: "https://w3c.github.io/webdriver/#dfn-is-element-enabled",
      variables: [
        {
          name: "elementId",
          description: "the id of an element returned in a previous call to Find Element(s)"
        }
      ],
      parameters: [],
      returns: {
        type: "Boolean",
        name: "isEnabled",
        description: "If the element is in an xml document, or is a disabled form control: `false`, otherwise, `true`."
      },
      alternativeCommands: ["element/isEnabled"],
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L387-L390"]
    }
  },
  "/session/:sessionId/element/:elementId/click": {
    POST: {
      command: "elementClick",
      description: "The Element Click command scrolls into view the element if it is not already pointer-interactable, and clicks its in-view center point. If the element's center point is obscured by another element, an element click intercepted error is returned. If the element is outside the viewport, an element not interactable error is returned.",
      ref: "https://w3c.github.io/webdriver/#dfn-element-click",
      variables: [
        {
          name: "elementId",
          description: "the id of an element returned in a previous call to Find Element(s)"
        }
      ],
      parameters: [],
      alternativeCommands: ["element/click"],
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L396-L398"]
    }
  },
  "/session/:sessionId/element/:elementId/clear": {
    POST: {
      command: "elementClear",
      description: "The Element Clear command scrolls into view an editable or resettable element and then attempts to clear its selected files or text content.",
      ref: "https://w3c.github.io/webdriver/#dfn-element-clear",
      variables: [
        {
          name: "elementId",
          description: "the id of an element returned in a previous call to Find Element(s)"
        }
      ],
      parameters: [],
      alternativeCommands: ["element/clearValue"],
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L404-L407"]
    }
  },
  "/session/:sessionId/element/:elementId/value": {
    POST: {
      command: "elementSendKeys",
      description: 'The Element Send Keys command scrolls into view the form control element and then sends the provided keys to the element. In case the element is not keyboard-interactable, an element not interactable error is returned.<br /><br />The key input state used for input may be cleared mid-way through "typing" by sending the null key, which is U+E000 (NULL).',
      ref: "https://w3c.github.io/webdriver/#dfn-element-send-keys",
      variables: [
        {
          name: "elementId",
          description: "the id of an element returned in a previous call to Find Element(s)"
        }
      ],
      parameters: [
        {
          name: "text",
          type: "string",
          description: "string to send as keystrokes to the element",
          required: true
        }
      ],
      alternativeCommands: ["element/addValue", "element/setValue"],
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L413-L416"]
    }
  },
  "/session/:sessionId/source": {
    GET: {
      command: "getPageSource",
      description: "The Get Page Source command returns a string serialization of the DOM of the current browsing context active document.",
      ref: "https://w3c.github.io/webdriver/#dfn-get-page-source",
      parameters: [],
      returns: {
        type: "string",
        name: "pageSource",
        description: "the DOM of the current browsing context active document"
      },
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L420-L421"]
    }
  },
  "/session/:sessionId/execute/sync": {
    POST: {
      command: "executeScript",
      description: "The Execute Script command executes a JavaScript function in the context of the current browsing context and returns the return value of the function.",
      ref: "https://w3c.github.io/webdriver/#dfn-execute-script",
      parameters: [
        {
          name: "script",
          type: "string",
          description: "a string, the Javascript function body you want executed",
          required: true
        },
        {
          name: "args",
          type: "(string|object|number|boolean|null|undefined)[]",
          description: "an array of JSON values which will be deserialized and passed as arguments to your function",
          required: true
        }
      ],
      returns: {
        type: "*",
        name: "result",
        description: "Either the return value of your script, the fulfillment of the Promise returned by your script, or the error which was the reason for your script's returned Promise's rejection."
      },
      alternativeCommands: ["browser/execute"],
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L425-L426"]
    }
  },
  "/session/:sessionId/execute/async": {
    POST: {
      command: "executeAsyncScript",
      description: "The Execute Async Script command causes JavaScript to execute as an anonymous function. Unlike the Execute Script command, the result of the function is ignored. Instead an additional argument is provided as the final argument to the function. This is a function that, when called, returns its first argument as the response.",
      ref: "https://w3c.github.io/webdriver/#dfn-execute-async-script",
      parameters: [
        {
          name: "script",
          type: "string",
          description: "a string, the Javascript function body you want executed",
          required: true
        },
        {
          name: "args",
          type: "(string|object|number|boolean|null|undefined)[]",
          description: "an array of JSON values which will be deserialized and passed as arguments to your function",
          required: true
        }
      ],
      returns: {
        type: "*",
        name: "result",
        description: "Either the return value of your script, the fulfillment of the Promise returned by your script, or the error which was the reason for your script's returned Promise's rejection."
      },
      alternativeCommands: ["browser/executeAsync"],
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L430-L434"]
    }
  },
  "/session/:sessionId/cookie": {
    GET: {
      command: "getAllCookies",
      description: "The Get All Cookies command returns all cookies associated with the address of the current browsing context\u2019s active document.",
      ref: "https://w3c.github.io/webdriver/#dfn-get-all-cookies",
      parameters: [],
      returns: {
        type: "Object[]",
        name: "cookies",
        description: "A list of serialized cookies. Each serialized cookie has a number of optional fields which may or may not be returned in addition to `name` and `value`."
      },
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L438-L455"]
    },
    POST: {
      command: "addCookie",
      description: "The Add Cookie command adds a single cookie to the cookie store associated with the active document's address.",
      ref: "https://w3c.github.io/webdriver/#dfn-adding-a-cookie",
      parameters: [
        {
          name: "cookie",
          type: "object",
          description: "A JSON object representing a cookie. It must have at least the name and value fields and could have more, including expiry-time and so on",
          required: true
        }
      ],
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L459-L477"]
    },
    DELETE: {
      command: "deleteAllCookies",
      description: "The Delete All Cookies command allows deletion of all cookies associated with the active document's address.",
      ref: "https://w3c.github.io/webdriver/#dfn-delete-all-cookies",
      parameters: [],
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L481-L485"]
    }
  },
  "/session/:sessionId/cookie/:name": {
    GET: {
      command: "getNamedCookie",
      description: "The Get Named Cookie command returns the cookie with the requested name from the associated cookies in the cookie store of the current browsing context's active document. If no cookie is found, a no such cookie error is returned.",
      ref: "https://w3c.github.io/webdriver/#dfn-get-named-cookie",
      variables: [
        {
          name: "name",
          description: "name of the cookie to retrieve"
        }
      ],
      parameters: [],
      returns: {
        type: "Object",
        name: "cookie",
        description: "A serialized cookie, with name and value fields. There are a number of optional fields like `path`, `domain`, and `expiry-time` which may also be present."
      },
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L489-L503"]
    },
    DELETE: {
      command: "deleteCookie",
      description: "The Delete Cookie command allows you to delete either a single cookie by parameter name, or all the cookies associated with the active document's address if name is undefined.",
      ref: "https://w3c.github.io/webdriver/#dfn-delete-cookie",
      variables: [
        {
          name: "name",
          description: "name of the cookie to delete"
        }
      ],
      parameters: [],
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L507-L512"]
    }
  },
  "/session/:sessionId/actions": {
    POST: {
      command: "performActions",
      description: "The Perform Actions command is used to execute complex user actions. See [spec](https://github.com/jlipps/simple-wd-spec#perform-actions) for more details.",
      ref: "https://w3c.github.io/webdriver/#dfn-perform-actions",
      parameters: [
        {
          name: "actions",
          type: "object[]",
          description: "a list of objects, each of which represents an input source and its associated actions",
          required: true
        }
      ]
    },
    DELETE: {
      command: "releaseActions",
      description: "The Release Actions command is used to release all the keys and pointer buttons that are currently depressed. This causes events to be fired as if the state was released by an explicit series of actions. It also clears all the internal state of the virtual devices.",
      ref: "https://w3c.github.io/webdriver/#dfn-release-actions",
      parameters: []
    }
  },
  "/session/:sessionId/alert/dismiss": {
    POST: {
      command: "dismissAlert",
      description: "The Dismiss Alert command dismisses a simple dialog if present, otherwise error. A request to dismiss an alert user prompt, which may not necessarily have a dismiss button, has the same effect as accepting it.",
      ref: "https://w3c.github.io/webdriver/#dfn-dismiss-alert",
      parameters: [],
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L516-L517"]
    }
  },
  "/session/:sessionId/alert/accept": {
    POST: {
      command: "acceptAlert",
      description: "The Accept Alert command accepts a simple dialog if present, otherwise error.",
      ref: "https://w3c.github.io/webdriver/#dfn-accept-alert",
      parameters: []
    }
  },
  "/session/:sessionId/alert/text": {
    GET: {
      command: "getAlertText",
      description: "The Get Alert Text command returns the message of the current user prompt. If there is no current user prompt, it returns an error.",
      ref: "https://w3c.github.io/webdriver/#dfn-get-alert-text",
      parameters: [],
      returns: {
        type: "string",
        name: "alertText",
        description: "The message of the user prompt."
      },
      exampleReferences: ["https://github.com/webdriverio/example-recipes/blob/e8b147e88e7a38351b0918b4f7efbd9ae292201d/api/webdriver/examples.js#L521-L522"]
    },
    POST: {
      command: "sendAlertText",
      description: "The Send Alert Text command sets the text field of a window.prompt user prompt to the given value.",
      ref: "https://w3c.github.io/webdriver/#dfn-send-alert-text",
      parameters: [
        {
          name: "text",
          type: "string",
          description: "string to set the prompt to",
          required: true
        }
      ]
    }
  },
  "/session/:sessionId/screenshot": {
    GET: {
      command: "takeScreenshot",
      description: "The Take Screenshot command takes a screenshot of the top-level browsing context's viewport.",
      ref: "https://w3c.github.io/webdriver/#dfn-take-screenshot",
      parameters: [],
      returns: {
        type: "string",
        name: "screenshot",
        description: "The base64-encoded PNG image data comprising the screenshot of the initial viewport."
      }
    }
  },
  "/session/:sessionId/element/:elementId/screenshot": {
    GET: {
      command: "takeElementScreenshot",
      description: "The Take Element Screenshot command takes a screenshot of the visible region encompassed by the bounding rectangle of an element.",
      ref: "https://w3c.github.io/webdriver/#dfn-take-element-screenshot",
      variables: [
        {
          name: "elementId",
          description: "the id of an element returned in a previous call to Find Element(s)"
        }
      ],
      parameters: [
        {
          name: "scroll",
          type: "boolean",
          description: "scroll into view the element. Default: true",
          required: false
        }
      ],
      returns: {
        type: "string",
        name: "screenshot",
        description: "The base64-encoded PNG image data comprising the screenshot of the visible region of an element\u2019s bounding rectangle after it has been scrolled into view."
      }
    }
  },
  "/session/:sessionId/element/:elementId/computedrole": {
    GET: {
      command: "getElementComputedRole",
      description: "Get the computed WAI-ARIA role of an element.",
      ref: "https://w3c.github.io/webdriver/#get-computed-role",
      variables: [
        {
          name: "elementId",
          description: "the id of an element returned in a previous call to Find Element(s)"
        }
      ],
      parameters: [],
      returns: {
        type: "string",
        name: "role",
        description: "The result of computing the WAI-ARIA role of element."
      }
    }
  },
  "/session/:sessionId/element/:elementId/computedlabel": {
    GET: {
      command: "getElementComputedLabel",
      description: "Get the accessible name of the element.",
      ref: "https://w3c.github.io/webdriver/#get-computed-label",
      variables: [
        {
          name: "elementId",
          description: "the id of an element returned in a previous call to Find Element(s)"
        }
      ],
      parameters: [],
      returns: {
        type: "string",
        name: "label",
        description: "The result of a Accessible Name and Description Computation for the Accessible Name of the element."
      }
    }
  },
  "/session/:sessionId/permissions": {
    POST: {
      command: "setPermissions",
      description: "Simulates user modification of a PermissionDescriptor's permission state. __Note:__ this feature has not landed in all browsers yet.",
      ref: "https://w3c.github.io/permissions/#set-permission-command",
      examples: [
        [
          "// set midi permissions",
          "browser.setPermissions(",
          "  { name: 'midi', sysex: true },",
          `  'granted' // can be also "denied" or "prompt"`,
          ");"
        ],
        [
          "// set clipboard permissions",
          "browser.setPermissions({ name: 'clipboard-read' }, 'granted');",
          "// now you can read the clipboard via, e.g.",
          "const clipboardText = await browser.execute(() => navigator.clipboard.readText());"
        ]
      ],
      parameters: [
        {
          name: "descriptor",
          type: "object",
          description: "Each powerful feature has one or more aspects that websites can request permission to access. To describe these aspects, each feature defines a subtype of PermissionDescriptor to be its permission descriptor type. __Note:__ this feature has not landed in all browsers yet.",
          required: true
        },
        {
          name: "state",
          type: "string",
          description: "Determines whether permission is granted, denied or prompted.",
          required: true
        },
        {
          name: "oneRealm",
          type: "boolean",
          description: "Whether or not to apply permissions to all execution contexts.",
          required: false
        }
      ]
    }
  },
  "/session/:sessionId/reporting/generate_test_report": {
    POST: {
      command: "generateTestReport",
      description: "Generates a report for testing. Extension for [Reporting API](https://developers.google.com/web/updates/2018/09/reportingapi). __Note:__ this feature has not landed in all browsers yet.",
      ref: "https://w3c.github.io/reporting/#automation",
      parameters: [
        {
          name: "message",
          type: "string",
          description: "Message to be displayed in the report.",
          required: true
        },
        {
          name: "group",
          type: "string",
          description: "Specifies the endpoint group to deliver the report to."
        }
      ]
    }
  },
  "/session/:sessionId/sensor": {
    POST: {
      command: "createMockSensor",
      description: "Creates a mock sensor to emulate sensors like Ambient Light Sensor. __Note:__ this feature has not landed in all browsers yet.",
      ref: "https://w3c.github.io/sensors/#create-mock-sensor-command",
      parameters: [
        {
          name: "mockSensorType",
          type: "string",
          description: "Type of sensor API to mock, e.g. 'ambient-light'",
          required: true
        },
        {
          name: "maxSamplingFrequency",
          type: "number",
          description: "A double representing frequency in Hz that is used to set maximum supported sampling frequency for the associated mock sensor."
        },
        {
          name: "minSamplingFrequency",
          type: "number",
          description: "A double representing frequency in Hz that is used to set minimum supported sampling frequency for the associated mock sensor."
        }
      ]
    }
  },
  "/session/:sessionId/sensor/:type": {
    GET: {
      command: "getMockSensor",
      description: "Retrieves information about a given type of mock sensor. __Note:__ this feature has not landed in all browsers yet.",
      ref: "https://w3c.github.io/sensors/#get-mock-sensor-command",
      variables: [
        {
          name: "type",
          description: "Mock sensor type to retrieve information from."
        }
      ],
      parameters: [],
      returns: {
        type: "object",
        name: "sensorReading",
        description: "Values of the mock sensor reading."
      }
    },
    POST: {
      command: "updateMockSensor",
      description: "Updates the mock sensor type. __Note:__ this feature has not landed in all browsers yet.",
      ref: "https://w3c.github.io/sensors/#update-mock-sensor-reading-command",
      variables: [
        {
          name: "type",
          description: "Mock sensor type to update information for."
        }
      ],
      parameters: [
        {
          name: "mockSensorType",
          type: "string",
          description: "Type of sensor API to mock, e.g. 'ambient-light'",
          required: true
        },
        {
          name: "maxSamplingFrequency",
          type: "number",
          description: "A double representing frequency in Hz that is used to set maximum supported sampling frequency for the associated mock sensor."
        },
        {
          name: "minSamplingFrequency",
          type: "number",
          description: "A double representing frequency in Hz that is used to set minimum supported sampling frequency for the associated mock sensor."
        }
      ]
    },
    DELETE: {
      command: "deleteMockSensor",
      description: "The Delete Session command closes any top-level browsing contexts associated with the current session, terminates the connection, and finally closes the current session. __Note:__ this feature has not landed in all browsers yet.",
      ref: "https://w3c.github.io/sensors/#delete-mock-sensor-command",
      variables: [
        {
          name: "type",
          description: "Mock sensor type to delete."
        }
      ],
      parameters: []
    }
  },
  "/session/:sessionId/time_zone": {
    POST: {
      command: "setTimeZone",
      description: "Simulates the changing of a time zone for the purposes of testing. __Note:__ this feature has not landed in all browsers yet.",
      ref: "https://w3c.github.io/sensors/#create-mock-sensor-command",
      parameters: [
        {
          name: "time_zone",
          type: "string",
          description: "Name of the timezone, e.g. Asia/Tokyo",
          required: true
        }
      ]
    }
  },
  "/session/:sessionId/webauthn/authenticator": {
    POST: {
      command: "addVirtualAuthenticator",
      description: "Creates a software [Virtual Authenticator](https://www.w3.org/TR/webauthn-2/#virtual-authenticators).",
      ref: "https://www.w3.org/TR/webauthn-2/#sctn-automation-add-virtual-authenticator",
      parameters: [
        {
          name: "protocol",
          type: "string",
          description: "Valid values: 'ctap1/u2f', 'ctap2', 'ctap2_1'.",
          required: false
        },
        {
          name: "transport",
          type: "string",
          description: "Valid values: 'usb', 'nfc', 'ble' or 'internal'.",
          required: false
        },
        {
          name: "hasResidentKey",
          type: "boolean",
          description: "Valid values: true, false.",
          required: false
        },
        {
          name: "hasUserVerification",
          type: "boolean",
          description: "Valid values: true, false.",
          required: false
        },
        {
          name: "isUserConsenting",
          type: "boolean",
          description: "Valid values: true, false.",
          required: false
        },
        {
          name: "isUserVerified",
          type: "boolean",
          description: "Valid values: An array containing extension identifiers.",
          required: false
        },
        {
          name: "extensions",
          type: "string[]",
          description: "Valid values: Up to 3 User Verification Method entries.",
          required: false
        },
        {
          name: "uvm",
          type: "object[]",
          description: "",
          required: false
        }
      ],
      returns: {
        type: "string",
        name: "authenticatorId",
        description: "Returns the string ID of the authenticator."
      }
    }
  },
  "/session/:sessionId/webauthn/authenticator/:authenticatorId": {
    DELETE: {
      command: "removeVirtualAuthenticator",
      description: "Removes a previously created Virtual Authenticator.",
      ref: "https://www.w3.org/TR/webauthn-2/#sctn-automation-remove-virtual-authenticator",
      variables: [
        {
          name: "authenticatorId",
          description: "id of authenticator"
        }
      ],
      parameters: []
    }
  },
  "/session/:sessionId/webauthn/authenticator/:authenticatorId/credential": {
    POST: {
      command: "addCredential",
      description: "Injects a Public Key Credential Source into an existing Virtual Authenticator.",
      ref: "https://www.w3.org/TR/webauthn-2/#sctn-automation-add-credential",
      variables: [
        {
          name: "authenticatorId",
          description: "ID of authenticator"
        }
      ],
      parameters: [
        {
          name: "credentialId",
          type: "string",
          description: "The Credential ID encoded using Base64url Encoding.",
          required: true
        },
        {
          name: "isResidentCredential",
          type: "boolean",
          description: "If set to true, a client-side discoverable credential is created. If set to false, a server-side credential is created instead.",
          required: true
        },
        {
          name: "rpId",
          type: "string",
          description: "The Relying Party ID the credential is scoped to.",
          required: true
        },
        {
          name: "privateKey",
          type: "string",
          description: "An asymmetric key package containing a single private key per [RFC5958], encoded using Base64url Encoding.",
          required: true
        },
        {
          name: "userHandle",
          type: "string",
          description: "The userHandle associated to the credential encoded using Base64url Encoding. This property may not be defined.",
          required: true
        },
        {
          name: "signCount",
          type: "number",
          description: "The initial value for a signature counter associated to the public key credential source.",
          required: true
        },
        {
          name: "largeBlob",
          type: "string",
          description: "The large, per-credential blob associated to the public key credential source, encoded using Base64url Encoding. This property may not be defined.",
          required: false
        }
      ]
    }
  },
  "/session/:sessionId/webauthn/authenticator/:authenticatorId/credentials": {
    GET: {
      command: "getCredentials",
      description: "Returns one Credential Parameters object for every Public Key Credential Source stored in a Virtual Authenticator, regardless of whether they were stored using Add Credential or `navigator.credentials.create()`.",
      ref: "https://www.w3.org/TR/webauthn-2/#sctn-automation-get-credentials",
      variables: [
        {
          name: "authenticatorId",
          description: "id of authenticator"
        }
      ],
      parameters: [],
      returns: {
        type: "object[]",
        name: "credentials",
        description: "Returns an array of credentials."
      }
    },
    DELETE: {
      command: "removeAllCredentials",
      description: "Removes all Public Key Credential Sources stored on a Virtual Authenticator.",
      ref: "https://www.w3.org/TR/webauthn-2/#sctn-automation-remove-all-credentials",
      variables: [
        {
          name: "authenticatorId",
          description: "id of authenticator"
        }
      ],
      parameters: []
    }
  },
  "/session/:sessionId/webauthn/authenticator/:authenticatorId/credentials/:credentialId": {
    DELETE: {
      command: "removeCredential",
      description: "Removes a Public Key Credential Source stored on a Virtual Authenticator.",
      ref: "https://www.w3.org/TR/webauthn-2/#sctn-automation-remove-credential",
      variables: [
        {
          name: "authenticatorId",
          description: "id of authenticator"
        },
        {
          name: "credentialId",
          description: "id of credential"
        }
      ],
      parameters: []
    }
  },
  "/session/:sessionId/webauthn/authenticator/:authenticatorId/uv": {
    POST: {
      command: "setUserVerified",
      description: "The Set User Verified extension command sets the isUserVerified property on the Virtual Authenticator.",
      ref: "https://www.w3.org/TR/webauthn-2/#sctn-automation-set-user-verified",
      variables: [
        {
          name: "authenticatorId",
          description: "id of authenticator"
        }
      ],
      parameters: []
    }
  }
};

// src/protocols/webdriverBidi.ts
var protocol = {
  "sendCommand": {
    "socket": {
      "command": "send",
      "description": "Send socket commands via WebDriver Bidi",
      "ref": "https://github.com/w3c/webdriver-bidi",
      "parameters": [
        {
          "name": "params",
          "type": "CommandData",
          "description": "socket payload",
          "required": true
        }
      ],
      "returns": {
        "type": "Object",
        "name": "CommandResponse",
        "description": "WebDriver Bidi response"
      }
    }
  },
  "sendAsyncCommand": {
    "socket": {
      "command": "sendAsync",
      "description": "Send asynchronous socket commands via WebDriver Bidi",
      "ref": "https://github.com/w3c/webdriver-bidi",
      "parameters": [
        {
          "name": "params",
          "type": "CommandData",
          "description": "socket payload",
          "required": true
        }
      ],
      "returns": {
        "type": "Number",
        "name": "id",
        "description": "id of WebDriver Bidi request"
      }
    }
  },
  "session.status": {
    "socket": {
      "command": "sessionStatus",
      "description": 'WebDriver Bidi command to send command method "session.status" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-session-status",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.EmptyParams`",
          "description": "<pre>\\{\\}</pre>",
          "required": true
        }
      ],
      "returns": {
        "type": "Object",
        "name": "local.SessionStatusResult",
        "description": "Command return value with the following interface:\n   ```ts\n   {\n     ready: boolean;\n     message: string;\n   }\n   ```"
      }
    }
  },
  "session.new": {
    "socket": {
      "command": "sessionNew",
      "description": 'WebDriver Bidi command to send command method "session.new" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-session-new",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.SessionNewParameters`",
          "description": "<pre>\\{<br />  capabilities: SessionCapabilitiesRequest;<br />\\}</pre>",
          "required": true
        }
      ],
      "returns": {
        "type": "Object",
        "name": "local.SessionNewResult",
        "description": "Command return value with the following interface:\n   ```ts\n   {\n     sessionId: string;\n     capabilities: {\n       acceptInsecureCerts: boolean;\n       browserName: string;\n       browserVersion: string;\n       platformName: string;\n       setWindowRect: boolean;\n       userAgent: string;\n       proxy?: SessionProxyConfiguration;\n       unhandledPromptBehavior?: SessionUserPromptHandler;\n       webSocketUrl?: string;\n     };\n   }\n   ```"
      }
    }
  },
  "session.end": {
    "socket": {
      "command": "sessionEnd",
      "description": 'WebDriver Bidi command to send command method "session.end" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-session-end",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.EmptyParams`",
          "description": "<pre>\\{\\}</pre>",
          "required": true
        }
      ]
    }
  },
  "session.subscribe": {
    "socket": {
      "command": "sessionSubscribe",
      "description": 'WebDriver Bidi command to send command method "session.subscribe" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-session-subscribe",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.SessionSubscriptionRequest`",
          "description": "<pre>\\{<br />  events: string[];<br />  contexts?: BrowsingContextBrowsingContext[];<br />  userContexts?: BrowserUserContext[];<br />\\}</pre>",
          "required": true
        }
      ],
      "returns": {
        "type": "Object",
        "name": "local.SessionSubscribeResult",
        "description": "Command return value with the following interface:\n   ```ts\n   {\n     subscription: SessionSubscription;\n   }\n   ```"
      }
    }
  },
  "session.unsubscribe": {
    "socket": {
      "command": "sessionUnsubscribe",
      "description": 'WebDriver Bidi command to send command method "session.unsubscribe" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-session-unsubscribe",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.SessionUnsubscribeParameters`",
          "description": "<pre>\\\\}</pre>",
          "required": true
        }
      ]
    }
  },
  "browser.close": {
    "socket": {
      "command": "browserClose",
      "description": 'WebDriver Bidi command to send command method "browser.close" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-browser-close",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.EmptyParams`",
          "description": "<pre>\\{\\}</pre>",
          "required": true
        }
      ]
    }
  },
  "browser.createUserContext": {
    "socket": {
      "command": "browserCreateUserContext",
      "description": 'WebDriver Bidi command to send command method "browser.createUserContext" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-browser-createUserContext",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.BrowserCreateUserContextParameters`",
          "description": "<pre>\\{<br />  acceptInsecureCerts?: boolean;<br />\\}</pre>",
          "required": true
        }
      ],
      "returns": {
        "type": "Object",
        "name": "local.BrowserCreateUserContextResult",
        "description": "Command return value with the following interface:\n   ```ts\n   ;\n   ```"
      }
    }
  },
  "browser.getClientWindows": {
    "socket": {
      "command": "browserGetClientWindows",
      "description": 'WebDriver Bidi command to send command method "browser.getClientWindows" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-browser-getClientWindows",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.EmptyParams`",
          "description": "<pre>\\{\\}</pre>",
          "required": true
        }
      ],
      "returns": {
        "type": "Object",
        "name": "local.BrowserGetClientWindowsResult",
        "description": "Command return value with the following interface:\n   ```ts\n   {\n     clientWindows: BrowserClientWindowInfo[];\n   }\n   ```"
      }
    }
  },
  "browser.getUserContexts": {
    "socket": {
      "command": "browserGetUserContexts",
      "description": 'WebDriver Bidi command to send command method "browser.getUserContexts" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-browser-getUserContexts",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.EmptyParams`",
          "description": "<pre>\\{\\}</pre>",
          "required": true
        }
      ],
      "returns": {
        "type": "Object",
        "name": "local.BrowserGetUserContextsResult",
        "description": "Command return value with the following interface:\n   ```ts\n   {\n     userContexts: BrowserUserContextInfo[];\n   }\n   ```"
      }
    }
  },
  "browser.removeUserContext": {
    "socket": {
      "command": "browserRemoveUserContext",
      "description": 'WebDriver Bidi command to send command method "browser.removeUserContext" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-browser-removeUserContext",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.BrowserRemoveUserContextParameters`",
          "description": "<pre>\\{<br />  userContext: BrowserUserContext;<br />\\}</pre>",
          "required": true
        }
      ]
    }
  },
  "browser.setClientWindowState": {
    "socket": {
      "command": "browserSetClientWindowState",
      "description": 'WebDriver Bidi command to send command method "browser.setClientWindowState" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-browser-setClientWindowState",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.BrowserSetClientWindowStateParameters`",
          "description": "<pre>\\{<br />  clientWindow: BrowserClientWindow;<br />\\}</pre>",
          "required": true
        }
      ]
    }
  },
  "browsingContext.activate": {
    "socket": {
      "command": "browsingContextActivate",
      "description": 'WebDriver Bidi command to send command method "browsingContext.activate" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-browsingContext-activate",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.BrowsingContextActivateParameters`",
          "description": "<pre>\\{<br />  context: BrowsingContextBrowsingContext;<br />\\}</pre>",
          "required": true
        }
      ]
    }
  },
  "browsingContext.captureScreenshot": {
    "socket": {
      "command": "browsingContextCaptureScreenshot",
      "description": 'WebDriver Bidi command to send command method "browsingContext.captureScreenshot" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-browsingContext-captureScreenshot",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.BrowsingContextCaptureScreenshotParameters`",
          "description": `<pre>\\{<br />  context: BrowsingContextBrowsingContext;<br />  /\\*\\*<br />   \\* @default 'viewport'<br />   \\*/<br />  origin?: "viewport" &#124; "document";<br />  format?: BrowsingContextImageFormat;<br />  clip?: BrowsingContextClipRectangle;<br />\\}</pre>`,
          "required": true
        }
      ],
      "returns": {
        "type": "Object",
        "name": "local.BrowsingContextCaptureScreenshotResult",
        "description": "Command return value with the following interface:\n   ```ts\n   {\n     data: string;\n   }\n   ```"
      }
    }
  },
  "browsingContext.close": {
    "socket": {
      "command": "browsingContextClose",
      "description": 'WebDriver Bidi command to send command method "browsingContext.close" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-browsingContext-close",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.BrowsingContextCloseParameters`",
          "description": "<pre>\\{<br />  context: BrowsingContextBrowsingContext;<br />  promptUnload?: boolean;<br />\\}</pre>",
          "required": true
        }
      ]
    }
  },
  "browsingContext.create": {
    "socket": {
      "command": "browsingContextCreate",
      "description": 'WebDriver Bidi command to send command method "browsingContext.create" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-browsingContext-create",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.BrowsingContextCreateParameters`",
          "description": "<pre>\\{<br />  type: BrowsingContextCreateType;<br />  referenceContext?: BrowsingContextBrowsingContext;<br />  background?: boolean;<br />  userContext?: BrowserUserContext;<br />\\}</pre>",
          "required": true
        }
      ],
      "returns": {
        "type": "Object",
        "name": "local.BrowsingContextCreateResult",
        "description": "Command return value with the following interface:\n   ```ts\n   {\n     context: BrowsingContextBrowsingContext;\n   }\n   ```"
      }
    }
  },
  "browsingContext.getTree": {
    "socket": {
      "command": "browsingContextGetTree",
      "description": 'WebDriver Bidi command to send command method "browsingContext.getTree" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-browsingContext-getTree",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.BrowsingContextGetTreeParameters`",
          "description": "<pre>\\{<br />  maxDepth?: JsUint;<br />  root?: BrowsingContextBrowsingContext;<br />\\}</pre>",
          "required": true
        }
      ],
      "returns": {
        "type": "Object",
        "name": "local.BrowsingContextGetTreeResult",
        "description": "Command return value with the following interface:\n   ```ts\n   {\n     contexts: BrowsingContextInfoList;\n   }\n   ```"
      }
    }
  },
  "browsingContext.handleUserPrompt": {
    "socket": {
      "command": "browsingContextHandleUserPrompt",
      "description": 'WebDriver Bidi command to send command method "browsingContext.handleUserPrompt" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-browsingContext-handleUserPrompt",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.BrowsingContextHandleUserPromptParameters`",
          "description": "<pre>\\{<br />  context: BrowsingContextBrowsingContext;<br />  accept?: boolean;<br />  userText?: string;<br />\\}</pre>",
          "required": true
        }
      ]
    }
  },
  "browsingContext.locateNodes": {
    "socket": {
      "command": "browsingContextLocateNodes",
      "description": 'WebDriver Bidi command to send command method "browsingContext.locateNodes" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-browsingContext-locateNodes",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.BrowsingContextLocateNodesParameters`",
          "description": "<pre>\\{<br />  context: BrowsingContextBrowsingContext;<br />  locator: BrowsingContextLocator;<br />  maxNodeCount?: JsUint;<br />  serializationOptions?: ScriptSerializationOptions;<br />  startNodes?: ScriptSharedReference[];<br />\\}</pre>",
          "required": true
        }
      ],
      "returns": {
        "type": "Object",
        "name": "local.BrowsingContextLocateNodesResult",
        "description": "Command return value with the following interface:\n   ```ts\n   {\n     nodes: ScriptNodeRemoteValue[];\n   }\n   ```"
      }
    }
  },
  "browsingContext.navigate": {
    "socket": {
      "command": "browsingContextNavigate",
      "description": 'WebDriver Bidi command to send command method "browsingContext.navigate" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-browsingContext-navigate",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.BrowsingContextNavigateParameters`",
          "description": "<pre>\\{<br />  context: BrowsingContextBrowsingContext;<br />  url: string;<br />  wait?: BrowsingContextReadinessState;<br />\\}</pre>",
          "required": true
        }
      ],
      "returns": {
        "type": "Object",
        "name": "local.BrowsingContextNavigateResult",
        "description": "Command return value with the following interface:\n   ```ts\n   {\n     navigation: BrowsingContextNavigation | null;\n     url: string;\n   }\n   ```"
      }
    }
  },
  "browsingContext.print": {
    "socket": {
      "command": "browsingContextPrint",
      "description": 'WebDriver Bidi command to send command method "browsingContext.print" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-browsingContext-print",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.BrowsingContextPrintParameters`",
          "description": `<pre>\\{<br />  context: BrowsingContextBrowsingContext;<br />  background?: boolean;<br />  margin?: BrowsingContextPrintMarginParameters;<br />  /\\*\\*<br />   \\* @default 'portrait'<br />   \\*/<br />  orientation?: "portrait" &#124; "landscape";<br />  page?: BrowsingContextPrintPageParameters;<br />  pageRanges?: (JsUint &#124; string)[];<br />  /\\*\\*<br />   \\* @default 1<br />   \\*/<br />  scale?: number;<br />  /\\*\\*<br />   \\* @default true<br />   \\*/<br />  shrinkToFit?: boolean;<br />\\}</pre>`,
          "required": true
        }
      ],
      "returns": {
        "type": "Object",
        "name": "local.BrowsingContextPrintResult",
        "description": "Command return value with the following interface:\n   ```ts\n   {\n     data: string;\n   }\n   ```"
      }
    }
  },
  "browsingContext.reload": {
    "socket": {
      "command": "browsingContextReload",
      "description": 'WebDriver Bidi command to send command method "browsingContext.reload" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-browsingContext-reload",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.BrowsingContextReloadParameters`",
          "description": "<pre>\\{<br />  context: BrowsingContextBrowsingContext;<br />  ignoreCache?: boolean;<br />  wait?: BrowsingContextReadinessState;<br />\\}</pre>",
          "required": true
        }
      ]
    }
  },
  "browsingContext.setViewport": {
    "socket": {
      "command": "browsingContextSetViewport",
      "description": 'WebDriver Bidi command to send command method "browsingContext.setViewport" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-browsingContext-setViewport",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.BrowsingContextSetViewportParameters`",
          "description": "<pre>\\{<br />  context?: BrowsingContextBrowsingContext;<br />  viewport?: BrowsingContextViewport &#124; null;<br />  devicePixelRatio?: number &#124; null;<br />  userContexts?: BrowserUserContext[];<br />\\}</pre>",
          "required": true
        }
      ]
    }
  },
  "browsingContext.traverseHistory": {
    "socket": {
      "command": "browsingContextTraverseHistory",
      "description": 'WebDriver Bidi command to send command method "browsingContext.traverseHistory" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-browsingContext-traverseHistory",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.BrowsingContextTraverseHistoryParameters`",
          "description": "<pre>\\{<br />  context: BrowsingContextBrowsingContext;<br />  delta: JsInt;<br />\\}</pre>",
          "required": true
        }
      ]
    }
  },
  "emulation.setGeolocationOverride": {
    "socket": {
      "command": "emulationSetGeolocationOverride",
      "description": 'WebDriver Bidi command to send command method "emulation.setGeolocationOverride" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-emulation-setGeolocationOverride",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.EmulationSetGeolocationOverrideParameters`",
          "description": "<pre>\\\\}</pre>",
          "required": true
        }
      ]
    }
  },
  "network.addIntercept": {
    "socket": {
      "command": "networkAddIntercept",
      "description": 'WebDriver Bidi command to send command method "network.addIntercept" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-network-addIntercept",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.NetworkAddInterceptParameters`",
          "description": "<pre>\\{<br />  phases: NetworkInterceptPhase[];<br />  contexts?: BrowsingContextBrowsingContext[];<br />  urlPatterns?: NetworkUrlPattern[];<br />\\}</pre>",
          "required": true
        }
      ],
      "returns": {
        "type": "Object",
        "name": "local.NetworkAddInterceptResult",
        "description": "Command return value with the following interface:\n   ```ts\n   {\n     intercept: NetworkIntercept;\n   }\n   ```"
      }
    }
  },
  "network.continueRequest": {
    "socket": {
      "command": "networkContinueRequest",
      "description": 'WebDriver Bidi command to send command method "network.continueRequest" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-network-continueRequest",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.NetworkContinueRequestParameters`",
          "description": "<pre>\\{<br />  request: NetworkRequest;<br />  body?: NetworkBytesValue;<br />  cookies?: NetworkCookieHeader[];<br />  headers?: NetworkHeader[];<br />  method?: string;<br />  url?: string;<br />\\}</pre>",
          "required": true
        }
      ]
    }
  },
  "network.continueResponse": {
    "socket": {
      "command": "networkContinueResponse",
      "description": 'WebDriver Bidi command to send command method "network.continueResponse" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-network-continueResponse",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.NetworkContinueResponseParameters`",
          "description": "<pre>\\{<br />  request: NetworkRequest;<br />  cookies?: NetworkSetCookieHeader[];<br />  credentials?: NetworkAuthCredentials;<br />  headers?: NetworkHeader[];<br />  reasonPhrase?: string;<br />  statusCode?: JsUint;<br />\\}</pre>",
          "required": true
        }
      ]
    }
  },
  "network.continueWithAuth": {
    "socket": {
      "command": "networkContinueWithAuth",
      "description": 'WebDriver Bidi command to send command method "network.continueWithAuth" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-network-continueWithAuth",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.NetworkContinueWithAuthParameters`",
          "description": "<pre>\\{<br />  request: NetworkRequest;<br />\\}</pre>",
          "required": true
        }
      ]
    }
  },
  "network.failRequest": {
    "socket": {
      "command": "networkFailRequest",
      "description": 'WebDriver Bidi command to send command method "network.failRequest" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-network-failRequest",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.NetworkFailRequestParameters`",
          "description": "<pre>\\{<br />  request: NetworkRequest;<br />\\}</pre>",
          "required": true
        }
      ]
    }
  },
  "network.provideResponse": {
    "socket": {
      "command": "networkProvideResponse",
      "description": 'WebDriver Bidi command to send command method "network.provideResponse" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-network-provideResponse",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.NetworkProvideResponseParameters`",
          "description": "<pre>\\{<br />  request: NetworkRequest;<br />  body?: NetworkBytesValue;<br />  cookies?: NetworkSetCookieHeader[];<br />  headers?: NetworkHeader[];<br />  reasonPhrase?: string;<br />  statusCode?: JsUint;<br />\\}</pre>",
          "required": true
        }
      ]
    }
  },
  "network.removeIntercept": {
    "socket": {
      "command": "networkRemoveIntercept",
      "description": 'WebDriver Bidi command to send command method "network.removeIntercept" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-network-removeIntercept",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.NetworkRemoveInterceptParameters`",
          "description": "<pre>\\{<br />  intercept: NetworkIntercept;<br />\\}</pre>",
          "required": true
        }
      ]
    }
  },
  "network.setCacheBehavior": {
    "socket": {
      "command": "networkSetCacheBehavior",
      "description": 'WebDriver Bidi command to send command method "network.setCacheBehavior" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-network-setCacheBehavior",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.NetworkSetCacheBehaviorParameters`",
          "description": '<pre>\\{<br />  cacheBehavior: "default" &#124; "bypass";<br />  contexts?: BrowsingContextBrowsingContext[];<br />\\}</pre>',
          "required": true
        }
      ]
    }
  },
  "script.addPreloadScript": {
    "socket": {
      "command": "scriptAddPreloadScript",
      "description": 'WebDriver Bidi command to send command method "script.addPreloadScript" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-script-addPreloadScript",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.ScriptAddPreloadScriptParameters`",
          "description": "<pre>\\{<br />  functionDeclaration: string;<br />  arguments?: ScriptChannelValue[];<br />  contexts?: BrowsingContextBrowsingContext[];<br />  userContexts?: BrowserUserContext[];<br />  sandbox?: string;<br />\\}</pre>",
          "required": true
        }
      ],
      "returns": {
        "type": "Object",
        "name": "local.ScriptAddPreloadScriptResult",
        "description": "Command return value with the following interface:\n   ```ts\n   {\n     script: ScriptPreloadScript;\n   }\n   ```"
      }
    }
  },
  "script.disown": {
    "socket": {
      "command": "scriptDisown",
      "description": 'WebDriver Bidi command to send command method "script.disown" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-script-disown",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.ScriptDisownParameters`",
          "description": "<pre>\\{<br />  handles: ScriptHandle[];<br />  target: ScriptTarget;<br />\\}</pre>",
          "required": true
        }
      ]
    }
  },
  "script.callFunction": {
    "socket": {
      "command": "scriptCallFunction",
      "description": 'WebDriver Bidi command to send command method "script.callFunction" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-script-callFunction",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.ScriptCallFunctionParameters`",
          "description": "<pre>\\{<br />  functionDeclaration: string;<br />  awaitPromise: boolean;<br />  target: ScriptTarget;<br />  arguments?: ScriptLocalValue[];<br />  resultOwnership?: ScriptResultOwnership;<br />  serializationOptions?: ScriptSerializationOptions;<br />  this?: ScriptLocalValue;<br />  userActivation?: boolean;<br />\\}</pre>",
          "required": true
        }
      ]
    }
  },
  "script.evaluate": {
    "socket": {
      "command": "scriptEvaluate",
      "description": 'WebDriver Bidi command to send command method "script.evaluate" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-script-evaluate",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.ScriptEvaluateParameters`",
          "description": "<pre>\\{<br />  expression: string;<br />  target: ScriptTarget;<br />  awaitPromise: boolean;<br />  resultOwnership?: ScriptResultOwnership;<br />  serializationOptions?: ScriptSerializationOptions;<br />  userActivation?: boolean;<br />\\}</pre>",
          "required": true
        }
      ],
      "returns": {
        "type": "Object",
        "name": "local.ScriptEvaluateResult",
        "description": "Command return value with the following interface:\n   ```ts\n   ;\n   ```"
      }
    }
  },
  "script.getRealms": {
    "socket": {
      "command": "scriptGetRealms",
      "description": 'WebDriver Bidi command to send command method "script.getRealms" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-script-getRealms",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.ScriptGetRealmsParameters`",
          "description": "<pre>\\{<br />  context?: BrowsingContextBrowsingContext;<br />  type?: ScriptRealmType;<br />\\}</pre>",
          "required": true
        }
      ],
      "returns": {
        "type": "Object",
        "name": "local.ScriptGetRealmsResult",
        "description": "Command return value with the following interface:\n   ```ts\n   {\n     realms: ScriptRealmInfo[];\n   }\n   ```"
      }
    }
  },
  "script.removePreloadScript": {
    "socket": {
      "command": "scriptRemovePreloadScript",
      "description": 'WebDriver Bidi command to send command method "script.removePreloadScript" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-script-removePreloadScript",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.ScriptRemovePreloadScriptParameters`",
          "description": "<pre>\\{<br />  script: ScriptPreloadScript;<br />\\}</pre>",
          "required": true
        }
      ]
    }
  },
  "storage.getCookies": {
    "socket": {
      "command": "storageGetCookies",
      "description": 'WebDriver Bidi command to send command method "storage.getCookies" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-storage-getCookies",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.StorageGetCookiesParameters`",
          "description": "<pre>\\{<br />  filter?: StorageCookieFilter;<br />  partition?: StoragePartitionDescriptor;<br />\\}</pre>",
          "required": true
        }
      ],
      "returns": {
        "type": "Object",
        "name": "local.StorageGetCookiesResult",
        "description": "Command return value with the following interface:\n   ```ts\n   {\n     cookies: NetworkCookie[];\n     partitionKey: StoragePartitionKey;\n   }\n   ```"
      }
    }
  },
  "storage.setCookie": {
    "socket": {
      "command": "storageSetCookie",
      "description": 'WebDriver Bidi command to send command method "storage.setCookie" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-storage-setCookie",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.StorageSetCookieParameters`",
          "description": "<pre>\\{<br />  cookie: StoragePartialCookie;<br />  partition?: StoragePartitionDescriptor;<br />\\}</pre>",
          "required": true
        }
      ],
      "returns": {
        "type": "Object",
        "name": "local.StorageSetCookieResult",
        "description": "Command return value with the following interface:\n   ```ts\n   {\n     partitionKey: StoragePartitionKey;\n   }\n   ```"
      }
    }
  },
  "storage.deleteCookies": {
    "socket": {
      "command": "storageDeleteCookies",
      "description": 'WebDriver Bidi command to send command method "storage.deleteCookies" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-storage-deleteCookies",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.StorageDeleteCookiesParameters`",
          "description": "<pre>\\{<br />  filter?: StorageCookieFilter;<br />  partition?: StoragePartitionDescriptor;<br />\\}</pre>",
          "required": true
        }
      ],
      "returns": {
        "type": "Object",
        "name": "local.StorageDeleteCookiesResult",
        "description": "Command return value with the following interface:\n   ```ts\n   {\n     partitionKey: StoragePartitionKey;\n   }\n   ```"
      }
    }
  },
  "input.performActions": {
    "socket": {
      "command": "inputPerformActions",
      "description": 'WebDriver Bidi command to send command method "input.performActions" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-input-performActions",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.InputPerformActionsParameters`",
          "description": "<pre>\\{<br />  context: BrowsingContextBrowsingContext;<br />  actions: InputSourceActions[];<br />\\}</pre>",
          "required": true
        }
      ]
    }
  },
  "input.releaseActions": {
    "socket": {
      "command": "inputReleaseActions",
      "description": 'WebDriver Bidi command to send command method "input.releaseActions" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-input-releaseActions",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.InputReleaseActionsParameters`",
          "description": "<pre>\\{<br />  context: BrowsingContextBrowsingContext;<br />\\}</pre>",
          "required": true
        }
      ]
    }
  },
  "input.setFiles": {
    "socket": {
      "command": "inputSetFiles",
      "description": 'WebDriver Bidi command to send command method "input.setFiles" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-input-setFiles",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.InputSetFilesParameters`",
          "description": "<pre>\\{<br />  context: BrowsingContextBrowsingContext;<br />  element: ScriptSharedReference;<br />  files: string[];<br />\\}</pre>",
          "required": true
        }
      ]
    }
  },
  "webExtension.install": {
    "socket": {
      "command": "webExtensionInstall",
      "description": 'WebDriver Bidi command to send command method "webExtension.install" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-webExtension-install",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.WebExtensionInstallParameters`",
          "description": "<pre>\\{<br />  extensionData: WebExtensionExtensionData;<br />\\}</pre>",
          "required": true
        }
      ],
      "returns": {
        "type": "Object",
        "name": "local.WebExtensionInstallResult",
        "description": "Command return value with the following interface:\n   ```ts\n   {\n     extension: WebExtensionExtension;\n   }\n   ```"
      }
    }
  },
  "webExtension.uninstall": {
    "socket": {
      "command": "webExtensionUninstall",
      "description": 'WebDriver Bidi command to send command method "webExtension.uninstall" with parameters.',
      "ref": "https://w3c.github.io/webdriver-bidi/#command-webExtension-uninstall",
      "parameters": [
        {
          "name": "params",
          "type": "`remote.WebExtensionUninstallParameters`",
          "description": "<pre>\\{<br />  extension: WebExtensionExtension;<br />\\}</pre>",
          "required": true
        }
      ]
    }
  }
};
var webdriverBidi_default = protocol;

// src/protocols/mjsonwp.ts
var mjsonwp_default = {
  "/session/:sessionId/element/:elementId/pageIndex": {
    GET: {
      command: "getPageIndex",
      ref: "https://github.com/appium/appium-base-driver/blob/master/docs/mjsonwp/protocol-methods.md#mobile-json-wire-protocol-endpoints",
      deprecated: "In Appium 2.0, this method is marked as deprecated and currently has no available alternatives.",
      parameters: [],
      returns: {
        type: "string",
        name: "pageIndex"
      }
    }
  },
  "/session/:sessionId/network_connection": {
    GET: {
      command: "getNetworkConnection",
      ref: "https://github.com/SeleniumHQ/mobile-spec/blob/master/spec-draft.md#device-modes",
      parameters: [],
      returns: {
        type: "number",
        name: "connectionType",
        description: "see https://appium.github.io/appium.io/docs/en/writing-running-appium/other/network-connection/"
      }
    },
    POST: {
      command: "setNetworkConnection",
      ref: "https://github.com/SeleniumHQ/mobile-spec/blob/master/spec-draft.md#device-modes",
      parameters: [
        {
          name: "type",
          type: "number",
          description: "a bit mask that should be translated to an integer value when serialized",
          required: true
        }
      ]
    }
  },
  "/session/:sessionId/touch/perform": {
    GET: {
      command: "touchPerform",
      ref: "https://github.com/SeleniumHQ/mobile-spec/blob/master/spec-draft.md#touch-gestures",
      parameters: [
        {
          name: "actions",
          type: "object[]",
          description: "a list of objects, each of which represents an input source and its associated actions",
          required: true
        }
      ]
    }
  },
  "/session/:sessionId/touch/multi/perform": {
    GET: {
      command: "multiTouchPerform",
      ref: "https://github.com/SeleniumHQ/mobile-spec/blob/master/spec-draft.md#touch-gestures",
      parameters: [
        {
          name: "actions",
          type: "object[]",
          description: "a list of objects, each of which represents an input source and its associated actions",
          required: true
        },
        {
          name: "elementId",
          type: "object[]",
          description: "the id of an element returned in a previous call to Find Element(s)"
        }
      ]
    }
  },
  "/session/:sessionId/receive_async_response": {
    POST: {
      command: "receiveAsyncResponse",
      ref: "https://github.com/appium/appium-base-driver/blob/master/docs/mjsonwp/protocol-methods.md#mobile-json-wire-protocol-endpoints",
      parameters: [
        {
          name: "status",
          type: "string",
          description: "the expected status of the response",
          required: true
        },
        {
          name: "value",
          type: "string",
          description: "the expected value of the response",
          required: true
        }
      ]
    }
  }
};

// src/protocols/appium.ts
var appium_default = {
  "/session/:sessionId/context": {
    GET: {
      command: "getAppiumContext",
      ref: "https://github.com/SeleniumHQ/mobile-spec/blob/master/spec-draft.md#webviews-and-other-contexts",
      parameters: [],
      returns: {
        type: "Context",
        name: "context",
        description: "a string representing the current context or null representing 'no context'"
      }
    },
    POST: {
      command: "switchAppiumContext",
      ref: "https://github.com/SeleniumHQ/mobile-spec/blob/master/spec-draft.md#webviews-and-other-contexts",
      parameters: [
        {
          name: "name",
          type: "string",
          description: "a string representing an available context",
          required: true
        }
      ]
    }
  },
  "/session/:sessionId/contexts": {
    GET: {
      command: "getAppiumContexts",
      ref: "https://github.com/SeleniumHQ/mobile-spec/blob/master/spec-draft.md#webviews-and-other-contexts",
      parameters: [],
      returns: {
        type: "Context[]",
        name: "contexts",
        description: "an array of strings representing available contexts, e.g. 'WEBVIEW', or 'NATIVE'"
      }
    }
  },
  "/session/:sessionId/appium/device/shake": {
    POST: {
      command: "shake",
      description: "Perform a shake action on the device.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/interactions/shake/",
      parameters: [],
      support: {
        ios: {
          XCUITest: "9.3+",
          UIAutomation: "8.0 to 9.3"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/lock": {
    POST: {
      command: "lock",
      description: "Lock the device.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/interactions/lock/",
      parameters: [
        {
          name: "seconds",
          type: "number",
          description: "how long to lock the screen (iOS only)",
          required: false
        }
      ],
      support: {
        ios: {
          UIAutomation: "8.0 to 9.3"
        },
        android: {
          UiAutomator: "4.2+"
        },
        windows: {
          Windows: "10+"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/unlock": {
    POST: {
      command: "unlock",
      description: "Unlock the device.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/interactions/unlock/",
      parameters: [],
      support: {
        android: {
          UiAutomator: "4.2+"
        },
        windows: {
          Windows: "10+"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/is_locked": {
    POST: {
      command: "isLocked",
      description: "Check whether the device is locked or not.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/interactions/is-locked/",
      parameters: [],
      returns: {
        type: "boolean",
        name: "isLocked",
        description: "True if the device is locked, false if not"
      },
      support: {
        android: {
          UiAutomator: "4.2+"
        },
        windows: {
          Windows: "10+"
        }
      }
    }
  },
  "/session/:sessionId/appium/start_recording_screen": {
    POST: {
      command: "startRecordingScreen",
      description: "Start recording the screen.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/recording-screen/start-recording-screen/",
      parameters: [
        {
          name: "options",
          type: "object",
          description: "command parameters that can contain keys like: remotePath, username, password, method, forceRestart, timeLimit, videoType, videoQuality, videoFps, bitRate, videoSize, bugReport (see more description in Appium docs)",
          required: false
        }
      ],
      support: {
        ios: {
          XCUITest: "9.3+"
        },
        android: {
          UiAutomator: "4.2+"
        },
        windows: {
          Windows: "10+"
        }
      }
    }
  },
  "/session/:sessionId/appium/stop_recording_screen": {
    POST: {
      command: "stopRecordingScreen",
      description: "Stop recording screen",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/recording-screen/stop-recording-screen/",
      parameters: [
        {
          name: "remotePath",
          type: "string",
          description: "The path to the remote location, where the resulting video should be uploaded. The following protocols are supported http/https, ftp. This option only has an effect if there is screen recording process in progreess and forceRestart parameter is not set to true. Null or empty string value (the default setting) means the content of resulting file should be encoded as Base64.",
          required: false
        },
        {
          name: "username",
          type: "string",
          description: "The name of the user for the remote authentication.",
          required: false
        },
        {
          name: "password",
          type: "string",
          description: "The password for the remote authentication.",
          required: false
        },
        {
          name: "method",
          type: "string",
          description: "The http multipart upload method name. The 'PUT' one is used by default.",
          required: false
        }
      ],
      returns: {
        type: "string",
        name: "response",
        description: "Base64 encoded string. If remote_path is set, the response is empty string"
      },
      support: {
        ios: {
          XCUITest: "9.3+"
        },
        android: {
          UiAutomator: "4.2+"
        },
        windows: {
          Windows: "10+"
        }
      }
    }
  },
  "/session/:sessionId/appium/performanceData/types": {
    POST: {
      command: "getPerformanceDataTypes",
      description: "Returns the information types of the system state which is supported to read as like cpu, memory, network traffic, and battery.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/performance-data/performance-data-types/",
      parameters: [],
      returns: {
        type: "string[]",
        name: "performanceTypes",
        description: "The available performance data types (cpuinfo|batteryinfo|networkinfo|memoryinfo)"
      },
      support: {
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/getPerformanceData": {
    POST: {
      command: "getPerformanceData",
      description: "Returns the information of the system state which is supported to read as like cpu, memory, network traffic, and battery.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/performance-data/get-performance-data/",
      parameters: [
        {
          name: "packageName",
          type: "string",
          description: "the package name of the application",
          required: true
        },
        {
          name: "dataType",
          type: "string",
          description: "the type of system state which wants to read. It should be one of the supported performance data types",
          required: true
        },
        {
          name: "dataReadTimeout",
          type: "number",
          description: "the number of attempts to read",
          required: false
        }
      ],
      returns: {
        type: "string[]",
        name: "performanceData",
        description: "The information type of the system state which is supported to read as like cpu, memory, network traffic, and battery"
      },
      support: {
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/press_keycode": {
    POST: {
      command: "pressKeyCode",
      description: "Press a particular key on the device.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/keys/press-keycode/",
      parameters: [
        {
          name: "keycode",
          type: "number",
          description: "keycode to press",
          required: true
        },
        {
          name: "metastate",
          type: "number",
          description: "meta state to press the keycode with",
          required: false
        },
        {
          name: "flags",
          type: "number",
          description: "flags for the keypress",
          required: false
        }
      ],
      support: {
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/long_press_keycode": {
    POST: {
      command: "longPressKeyCode",
      description: "Press and hold a particular key code on the device.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/keys/long-press-keycode/",
      parameters: [
        {
          name: "keycode",
          type: "number",
          description: "keycode to press on the device",
          required: true
        },
        {
          name: "metastate",
          type: "number",
          description: "metastate for the keypress",
          required: false
        },
        {
          name: "flags",
          type: "number",
          description: "flags for the keypress",
          required: false
        }
      ],
      support: {
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/keyevent": {
    POST: {
      command: "sendKeyEvent",
      description: "Send a key code to the device.",
      ref: "https://github.com/appium/appium-base-driver/blob/master/docs/mjsonwp/protocol-methods.md#appium-extension-endpoints",
      parameters: [
        {
          name: "keycode",
          type: "string",
          description: "keycode to press",
          required: true
        },
        {
          name: "metastate",
          type: "string",
          description: "meta state to press the keycode with",
          required: false
        }
      ],
      support: {
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/rotation": {
    POST: {
      command: "rotateDevice",
      description: "Rotate the device in three dimensions.",
      ref: "https://github.com/SeleniumHQ/mobile-spec/blob/master/spec-draft.md#device-rotation",
      parameters: [
        {
          name: "x",
          type: "number",
          description: "x offset to use for the center of the rotate gesture",
          required: true,
          default: 0
        },
        {
          name: "y",
          type: "number",
          description: "y offset to use for the center of the rotate gesture",
          required: true,
          default: 0
        },
        {
          name: "z",
          type: "number",
          description: "z offset to use for the center of the rotate gesture",
          required: true,
          default: 0
        }
      ],
      support: {
        ios: {
          UIAutomation: "8.0 to 9.3"
        },
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/current_activity": {
    GET: {
      command: "getCurrentActivity",
      description: "Get the name of the current Android activity.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/activity/current-activity/",
      parameters: [],
      returns: {
        type: "string",
        name: "activity",
        description: "Name of the current activity"
      },
      support: {
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/current_package": {
    GET: {
      command: "getCurrentPackage",
      description: "Get the name of the current Android package.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/activity/current-package/",
      parameters: [],
      returns: {
        type: "string",
        name: "package",
        description: "Name of the current package"
      },
      support: {
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/install_app": {
    POST: {
      command: "installApp",
      description: "Install the given app onto the device.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/app/install-app/",
      parameters: [
        {
          name: "appPath",
          type: "string",
          description: "path to application .apk file",
          required: true
        }
      ],
      support: {
        ios: {
          XCUITest: "9.3+",
          UIAutomation: "8.0 to 9.3"
        },
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/activate_app": {
    POST: {
      command: "activateApp",
      description: "Activate the given app onto the device",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/app/activate-app/",
      parameters: [
        {
          name: "appId",
          type: "string",
          description: "App ID (package ID for Android, bundle ID for iOS)",
          required: true
        }
      ],
      support: {
        ios: {
          XCUITest: "9.3+"
        },
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/remove_app": {
    POST: {
      command: "removeApp",
      description: "Remove an app from the device.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/app/remove-app/",
      parameters: [
        {
          name: "appId",
          type: "string",
          description: "App ID (package ID for Android, bundle ID for iOS)",
          required: true
        }
      ],
      support: {
        ios: {
          XCUITest: "9.3+",
          UIAutomation: "8.0 to 9.3"
        },
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/terminate_app": {
    POST: {
      command: "terminateApp",
      description: "Terminate the given app on the device",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/app/terminate-app/",
      parameters: [
        {
          name: "appId",
          type: "string",
          description: "App ID (package ID for Android, bundle ID for iOS)",
          required: true
        },
        {
          name: "options",
          type: "object",
          description: 'Command options. E.g. "timeout": (Only Android) Timeout to retry terminate the app (see more in Appium docs)'
        }
      ],
      support: {
        ios: {
          XCUITest: "9.3+"
        },
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/app_installed": {
    POST: {
      command: "isAppInstalled",
      description: "Check whether the specified app is installed on the device.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/app/is-app-installed/",
      parameters: [
        {
          name: "appId",
          type: "string",
          description: "App ID (package ID for Android, bundle ID for iOS)",
          required: true
        }
      ],
      returns: {
        type: "boolean",
        name: "isAppInstalled",
        description: "Return true if installed, false if not"
      },
      support: {
        ios: {
          XCUITest: "9.3+",
          UIAutomation: "8.0 to 9.3"
        },
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/app_state": {
    POST: {
      command: "queryAppState",
      description: "Get the given app status on the device",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/app/app-state/",
      parameters: [
        {
          name: "appId",
          type: "string",
          description: "App ID (package ID for Android, bundle ID for iOS)",
          required: true
        }
      ],
      returns: {
        type: "number",
        name: "appStatus",
        description: "0 is not installed. 1 is not running. 2 is running in background or suspended. 3 is running in background. 4 is running in foreground"
      },
      support: {
        ios: {
          XCUITest: "9.3+"
        },
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/hide_keyboard": {
    POST: {
      command: "hideKeyboard",
      description: "Hide soft keyboard.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/keys/hide-keyboard/",
      parameters: [
        {
          name: "strategy",
          type: "string",
          description: "hide keyboard strategy (UIAutomation only), available strategies - 'press', 'pressKey', 'swipeDown', 'tapOut', 'tapOutside', 'default'",
          required: false
        },
        {
          name: "key",
          type: "string",
          description: "key value if strategy is 'pressKey'",
          required: false
        },
        {
          name: "keyCode",
          type: "string",
          description: "key code if strategy is 'pressKey'",
          required: false
        },
        {
          name: "keyName",
          type: "string",
          description: "key name if strategy is 'pressKey'",
          required: false
        }
      ],
      support: {
        ios: {
          XCUITest: "9.3+",
          UIAutomation: "8.0 to 9.3"
        },
        android: {
          UiAutomator: "4.2+"
        },
        windows: {
          Windows: "10+"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/is_keyboard_shown": {
    GET: {
      command: "isKeyboardShown",
      description: "Whether or not the soft keyboard is shown.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/keys/is-keyboard-shown/",
      parameters: [],
      returns: {
        type: "boolean",
        name: "isKeyboardShown",
        description: "True if the keyboard is shown"
      },
      support: {
        ios: {
          XCUITest: "9.3+",
          UIAutomation: "8.0 to 9.3"
        },
        android: {
          UiAutomator: "4.2+"
        },
        windows: {
          Windows: "10+"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/push_file": {
    POST: {
      command: "pushFile",
      description: "Place a file onto the device in a particular place.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/files/push-file/",
      parameters: [
        {
          name: "path",
          type: "string",
          description: "path to install the data to",
          required: true
        },
        {
          name: "data",
          type: "string",
          description: "contents of file in base64",
          required: true
        }
      ],
      support: {
        ios: {
          XCUITest: "9.3+",
          UIAutomation: "8.0 to 9.3"
        },
        android: {
          UiAutomator: "4.2+"
        },
        windows: {
          Windows: "10+"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/pull_file": {
    POST: {
      command: "pullFile",
      description: "Retrieve a file from the device's file system.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/files/pull-file/",
      parameters: [
        {
          name: "path",
          type: "string",
          description: "path on the device to pull file from",
          required: true
        }
      ],
      support: {
        ios: {
          XCUITest: "9.3+",
          UIAutomation: "8.0 to 9.3"
        },
        android: {
          UiAutomator: "4.2+"
        },
        windows: {
          Windows: "10+"
        }
      },
      returns: {
        type: "string",
        name: "response",
        description: "Contents of file in base64"
      }
    }
  },
  "/session/:sessionId/appium/device/pull_folder": {
    POST: {
      command: "pullFolder",
      description: "Retrieve a folder from the device's file system.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/files/pull-folder/",
      parameters: [
        {
          name: "path",
          type: "string",
          description: "path to an entire folder on the device",
          required: true
        }
      ],
      support: {
        ios: {
          XCUITest: "9.3+",
          UIAutomation: "8.0 to 9.3"
        },
        android: {
          UiAutomator: "4.2+"
        },
        windows: {
          Windows: "10+"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/toggle_airplane_mode": {
    POST: {
      command: "toggleAirplaneMode",
      description: "Toggle airplane mode on device.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/network/toggle-airplane-mode/",
      parameters: [],
      support: {
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/toggle_data": {
    POST: {
      command: "toggleData",
      description: "Switch the state of data service.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/network/toggle-data/",
      parameters: [],
      support: {
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/toggle_wifi": {
    POST: {
      command: "toggleWiFi",
      description: "Switch the state of the wifi service.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/network/toggle-wifi/",
      parameters: [],
      support: {
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/toggle_location_services": {
    POST: {
      command: "toggleLocationServices",
      description: "Switch the state of the location service.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/network/toggle-location-services/",
      parameters: [],
      support: {
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/network_speed": {
    POST: {
      command: "toggleNetworkSpeed",
      description: "Set network speed (Emulator only)",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/network/network-speed/",
      parameters: [
        {
          name: "netspeed",
          type: "string",
          description: "Network type - 'full','gsm', 'edge', 'hscsd', 'gprs', 'umts', 'hsdpa', 'lte', 'evdo'",
          required: true
        }
      ],
      support: {
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/open_notifications": {
    POST: {
      command: "openNotifications",
      description: "Open Android notifications (Emulator only).",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/system/open-notifications/",
      parameters: [],
      support: {
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/start_activity": {
    POST: {
      command: "startActivity",
      description: "Start an Android activity by providing package name and activity name.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/activity/start-activity/",
      parameters: [
        {
          name: "appPackage",
          type: "string",
          description: "name of app",
          required: true
        },
        {
          name: "appActivity",
          type: "string",
          description: "name of activity",
          required: true
        },
        {
          name: "appWaitPackage",
          type: "string",
          description: "name of app to wait for",
          required: false
        },
        {
          name: "appWaitActivity",
          type: "string",
          description: "name of activity to wait for",
          required: false
        },
        {
          name: "intentAction",
          type: "string",
          description: "intent action which will be used to start activity",
          required: false,
          default: "android.intent.action.MAIN"
        },
        {
          name: "intentCategory",
          type: "string",
          description: "intent category which will be used to start activity",
          required: false,
          default: "android.intent.category.LAUNCHER"
        },
        {
          name: "intentFlags",
          type: "string",
          description: "flags that will be used to start activity",
          required: false,
          default: "0x10200000"
        },
        {
          name: "optionalIntentArguments",
          type: "string",
          description: "additional intent arguments that will be used to start activity",
          required: false
        },
        {
          name: "dontStopAppOnReset",
          type: "string",
          description: "doesn\u2019t stop the process of the app under test, before starting the app using adb",
          required: false
        }
      ],
      support: {
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/system_bars": {
    GET: {
      command: "getSystemBars",
      description: "Retrieve visibility and bounds information of the status and navigation bars.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/system/system-bars/",
      parameters: [],
      returns: {
        type: "object[]",
        name: "systemBars",
        description: "Information about visibility and bounds of status and navigation bar"
      },
      support: {
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/system_time": {
    GET: {
      command: "getDeviceTime",
      description: "Get the time on the device.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/system/system-time/",
      parameters: [],
      returns: {
        type: "string",
        name: "time",
        description: "Time on the device"
      },
      support: {
        ios: {
          XCUITest: "9.3+",
          UIAutomation: "8.0 to 9.3"
        },
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/display_density": {
    GET: {
      command: "getDisplayDensity",
      description: "Get display density from device.",
      ref: "https://github.com/appium/appium-base-driver/blob/master/docs/mjsonwp/protocol-methods.md#appium-extension-endpoints",
      parameters: [],
      returns: {
        type: "*",
        name: "displayDensity"
      },
      support: {
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/simulator/touch_id": {
    POST: {
      command: "touchId",
      description: "Simulate a [touch id](https://support.apple.com/en-ca/ht201371) event (iOS Simulator only). To enable this feature, the `allowTouchIdEnroll` desired capability must be set to true and the Simulator must be [enrolled](https://support.apple.com/en-ca/ht201371). When you set allowTouchIdEnroll to true, it will set the Simulator to be enrolled by default. The enrollment state can be [toggled](https://appium.github.io/appium.io/docs/en/commands/device/simulator/toggle-touch-id-enrollment/index.html). This call will only work if Appium process or its parent application (e.g. Terminal.app or Appium.app) has access to Mac OS accessibility in System Preferences > Security & Privacy > Privacy > Accessibility list.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/simulator/touch-id/",
      parameters: [
        {
          name: "match",
          type: "boolean",
          description: "are we simulating a successful touch (true) or a failed touch (false)",
          required: true
        }
      ],
      support: {
        ios: {
          XCUITest: "9.3+"
        }
      }
    }
  },
  "/session/:sessionId/appium/simulator/toggle_touch_id_enrollment": {
    POST: {
      command: "toggleEnrollTouchId",
      description: "Toggle the simulator being [enrolled](https://support.apple.com/en-ca/ht201371) to accept touchId (iOS Simulator only). To enable this feature, the `allowTouchIdEnroll` desired capability must be set to true. When `allowTouchIdEnroll` is set to true the Simulator will be enrolled by default, and the 'Toggle Touch ID Enrollment' changes the enrollment state. This call will only work if the Appium process or its parent application (e.g., Terminal.app or Appium.app) has access to Mac OS accessibility in System Preferences > Security & Privacy > Privacy > Accessibility list.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/simulator/toggle-touch-id-enrollment/",
      parameters: [
        {
          name: "enabled",
          type: "boolean",
          description: "equals to true if TouchID enrollment should be enabled",
          required: false,
          default: true
        }
      ],
      support: {
        ios: {
          XCUITest: "9.3+"
        }
      }
    }
  },
  "/session/:sessionId/appium/app/launch": {
    POST: {
      command: "launchApp",
      description: "Launch an app on device.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/app/launch-app/",
      deprecated: "For iOS, utilize `driver.execute('mobile: launchApp', { ... })`, and for Android, make use of `driver.execute('mobile: activateApp', { ... })`.",
      parameters: [],
      support: {
        ios: {
          XCUITest: "9.3+",
          UIAutomation: "8.0 to 9.3"
        },
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/app/close": {
    POST: {
      command: "closeApp",
      description: "Close an app on device.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/app/close-app/",
      deprecated: "Use `driver.execute('mobile: terminateApp', { ... })` instead",
      parameters: [],
      support: {
        ios: {
          XCUITest: "9.3+",
          UIAutomation: "8.0 to 9.3"
        },
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/app/background": {
    POST: {
      command: "background",
      description: "Send the currently running app for this session to the background.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/app/background-app/",
      deprecated: "Use `driver.execute('mobile: backgroundApp', { ... })` instead",
      parameters: [
        {
          name: "seconds",
          type: "(number|null)",
          description: "timeout to restore app, if 'null' app won't be restored",
          required: true,
          default: "null"
        }
      ],
      support: {
        ios: {
          XCUITest: "9.3+",
          UIAutomation: "8.0 to 9.3"
        },
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/app/end_test_coverage": {
    POST: {
      command: "endCoverage",
      description: "Get test coverage data.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/app/end-test-coverage/",
      parameters: [
        {
          name: "intent",
          type: "string",
          description: "intent to broadcast",
          required: true
        },
        {
          name: "path",
          type: "string",
          description: "path to .ec file",
          required: true
        }
      ],
      support: {
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/app/strings": {
    POST: {
      command: "getStrings",
      description: "Get app strings.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/app/get-app-strings/",
      parameters: [
        {
          name: "language",
          type: "string",
          description: "language code",
          required: false
        },
        {
          name: "stringFile",
          type: "string",
          description: "path to the string file",
          required: false
        }
      ],
      returns: {
        type: "object",
        name: "appStrings",
        description: "all defined Strings from an app for the specified language and strings filename"
      },
      support: {
        ios: {
          XCUITest: "9.3+",
          UIAutomation: "8.0 to 9.3"
        },
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/element/:elementId/value": {
    POST: {
      command: "setValueImmediate",
      ref: "https://github.com/appium/appium-base-driver/blob/master/docs/mjsonwp/protocol-methods.md#appium-extension-endpoints",
      variables: [
        {
          name: "elementId",
          description: "the id of an element returned in a previous call to Find Element(s)"
        }
      ],
      parameters: [
        {
          name: "text",
          type: "string",
          description: "text to set to an element",
          required: true
        }
      ],
      support: {
        ios: {
          XCUITest: "9.3+"
        },
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/element/:elementId/replace_value": {
    POST: {
      command: "replaceValue",
      description: "Replace the value to element directly.",
      ref: "https://github.com/appium/appium-base-driver/blob/master/docs/mjsonwp/protocol-methods.md#appium-extension-endpoints",
      variables: [
        {
          name: "elementId",
          description: "the id of an element returned in a previous call to Find Element(s)"
        }
      ],
      parameters: [
        {
          name: "value",
          type: "string",
          description: "value to replace on element",
          required: true
        }
      ],
      support: {
        ios: {
          XCUITest: "9.3+",
          UIAutomation: "8.0 to 9.3"
        },
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/settings": {
    GET: {
      command: "getSettings",
      description: "Retrieve the current settings on the device.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/session/settings/get-settings/",
      parameters: [],
      returns: {
        type: "object",
        name: "settings",
        description: "JSON hash of all the currently specified settings, see Settings API"
      },
      support: {
        ios: {
          XCUITest: "9.3+",
          UIAutomation: "8.0 to 9.3"
        },
        android: {
          UiAutomator: "4.2+"
        },
        windows: {
          Windows: "10+"
        }
      }
    },
    POST: {
      command: "updateSettings",
      description: "Update the current setting on the device.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/session/settings/update-settings/",
      parameters: [
        {
          name: "settings",
          type: "object",
          description: "key/value object with settings to update",
          required: true
        }
      ],
      support: {
        ios: {
          XCUITest: "9.3+",
          UIAutomation: "8.0 to 9.3"
        },
        android: {
          UiAutomator: "4.2+"
        },
        windows: {
          Windows: "10+"
        }
      }
    }
  },
  "/session/:sessionId/appium/receive_async_response": {
    POST: {
      command: "receiveAsyncResponse",
      description: "Callback url for asynchronous execution of JavaScript.",
      ref: "https://github.com/appium/appium-base-driver/blob/master/docs/mjsonwp/protocol-methods.md#appium-extension-endpoints",
      parameters: [
        {
          name: "response",
          type: "object",
          description: "response to receive on device",
          required: true
        }
      ],
      support: {
        ios: {
          XCUITest: "9.3+",
          UIAutomation: "8.0 to 9.3"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/gsm_call": {
    POST: {
      command: "gsmCall",
      description: "Make GSM call (Emulator only).",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/network/gsm-call/",
      parameters: [
        {
          name: "phoneNumber",
          type: "string",
          description: "the phone number to call to",
          required: true
        },
        {
          name: "action",
          type: "string",
          description: "The action - 'call', 'accept', 'cancel', 'hold'",
          required: true
        }
      ],
      support: {
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/gsm_signal": {
    POST: {
      command: "gsmSignal",
      description: "Set GSM signal strength (Emulator only).",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/network/gsm-signal/",
      parameters: [
        {
          name: "signalStrength",
          type: "string",
          description: "signal strength in the range [0, 4]",
          required: true
        },
        {
          name: "signalStrengh",
          type: "string",
          description: "signal strength in the range [0, 4]. Please also set this parameter with the same value if you use Appium v1.11.0 or lower (see https://github.com/appium/appium/issues/12234).",
          required: false
        }
      ],
      support: {
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/power_capacity": {
    POST: {
      command: "powerCapacity",
      description: "Set the battery percentage (Emulator only).",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/emulator/power_capacity/",
      parameters: [
        {
          name: "percent",
          type: "number",
          description: "percentage value in range [0, 100]",
          required: true
        }
      ],
      support: {
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/power_ac": {
    POST: {
      command: "powerAC",
      description: "Set the state of the battery charger to connected or not (Emulator only).",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/emulator/power_ac/",
      parameters: [
        {
          name: "state",
          type: "string",
          description: "set the state. on or off",
          required: true
        }
      ],
      support: {
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/gsm_voice": {
    POST: {
      command: "gsmVoice",
      description: "Set GSM voice state (Emulator only).",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/network/gsm-voice/",
      parameters: [
        {
          name: "state",
          type: "string",
          description: "state of GSM voice - 'unregistered', 'home', 'roaming', 'searching', 'denied', 'off', 'on'",
          required: true
        }
      ],
      support: {
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/send_sms": {
    POST: {
      command: "sendSms",
      description: "Simulate an SMS message (Emulator only).",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/network/send-sms/",
      parameters: [
        {
          name: "phoneNumber",
          type: "string",
          description: "the phone number to send the SMS too",
          required: true
        },
        {
          name: "message",
          type: "string",
          description: "the SMS message",
          required: true
        }
      ],
      support: {
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/finger_print": {
    POST: {
      command: "fingerPrint",
      description: "Authenticate users by using their finger print scans on supported emulators.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/authentication/finger-print/",
      parameters: [
        {
          name: "fingerprintId",
          type: "number",
          description: "finger prints stored in Android Keystore system (from 1 to 10)",
          required: true
        }
      ],
      support: {
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/set_clipboard": {
    POST: {
      command: "setClipboard",
      description: "Set the content of the system clipboard",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/clipboard/set-clipboard/",
      parameters: [
        {
          name: "content",
          type: "string",
          description: "The actual base64 encoded clipboard content",
          required: true
        },
        {
          name: "contentType",
          type: "string",
          description: "The type of the content to get. Plaintext, Image, URL. Android supports only plaintext",
          required: false
        },
        {
          name: "label",
          type: "string",
          description: "Clipboard data label for Android",
          required: false
        }
      ],
      returns: {
        type: "string",
        name: "response",
        description: "Response from Appium server"
      },
      support: {
        ios: {
          XCUITest: "9.3+"
        },
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/appium/device/get_clipboard": {
    POST: {
      command: "getClipboard",
      description: "Get the content of the system clipboard",
      ref: "https://appium.github.io/appium.io/docs/en/commands/device/clipboard/get-clipboard/",
      parameters: [
        {
          name: "contentType",
          type: "string",
          description: "The type of the content to get. Plaintext, Image, URL. Android supports only plaintext",
          required: false
        }
      ],
      returns: {
        type: "string",
        name: "response",
        description: "Clipboard content as base64-encoded string or an empty string if the clipboard is empty"
      },
      support: {
        ios: {
          XCUITest: "9.3+"
        },
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/touch/perform": {
    POST: {
      command: "touchPerform",
      description: "This functionality is only available from within a native context. 'Touch Perform' works similarly to the other singular touch interactions, except that this allows you to chain together more than one touch action as one command. This is useful because Appium commands are sent over the network and there's latency between commands. This latency can make certain touch interactions impossible because some interactions need to be performed in one sequence. Vertical, for example, requires pressing down, moving to a different y coordinate, and then releasing. For it to work, there can't be a delay between the interactions.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/interactions/touch/touch-perform/",
      parameters: [
        {
          name: "actions",
          type: "object[]",
          description: "The type of action to perform (e.g. moveTo, release, press, tap, wait)",
          required: true
        }
      ],
      examples: [
        [
          "// do a horizontal swipe by percentage",
          "const startPercentage = 10;",
          "const endPercentage = 90;",
          "const anchorPercentage = 50;",
          "",
          "const { width, height } = driver.getWindowSize();",
          "const anchor = height * anchorPercentage / 100;",
          "const startPoint = width * startPercentage / 100;",
          "const endPoint = width * endPercentage / 100;",
          "driver.touchPerform([",
          "  {",
          "    action: 'press',",
          "    options: {",
          "      x: startPoint,",
          "      y: anchor,",
          "    },",
          "  },",
          "  {",
          "    action: 'wait',",
          "    options: {",
          "      ms: 100,",
          "    },",
          "  },",
          "  {",
          "    action: 'moveTo',",
          "    options: {",
          "      x: endPoint,",
          "      y: anchor,",
          "    },",
          "  },",
          "  {",
          "    action: 'release',",
          "    options: {},",
          "  },",
          "]);"
        ]
      ],
      support: {
        android: {
          UiAutomator: "4.2+"
        },
        ios: {
          XCUITest: "9.3+",
          UIAutomation: "8.0 to 9.3"
        },
        windows: {
          Windows: "10+"
        }
      }
    }
  },
  "/session/:sessionId/touch/multi/perform": {
    POST: {
      command: "multiTouchPerform",
      description: "This functionality is only available from within a native context. Perform a multi touch action sequence.",
      ref: "https://appium.github.io/appium.io/docs/en/commands/interactions/touch/multi-touch-perform/",
      parameters: [
        {
          name: "actions",
          type: "object[]",
          description: "The type of action to perform (e.g. moveTo, release, press, tap, wait)",
          required: true
        }
      ],
      support: {
        android: {
          UiAutomator: "4.2+"
        },
        ios: {
          XCUITest: "9.3+",
          UIAutomation: "8.0 to 9.3"
        },
        windows: {
          Windows: "10+"
        }
      }
    }
  },
  "/session/:sessionId/appium/execute_driver": {
    POST: {
      command: "executeDriverScript",
      description: "This command enables you to specify a WebdriverIO script as a string and transmit it to the Appium server for local execution on the server itself. This approach helps minimize potential latency associated with each command. ***To utilize this command with Appium 2.0, you must have the [`execute-driver-plugin`](https://github.com/appium/appium/tree/master/packages/execute-driver-plugin) plugin installed.***",
      ref: "https://github.com/appium/appium/blob/master/docs/en/commands/session/execute-driver.md",
      parameters: [
        {
          name: "script",
          type: "string",
          description: "The script to execute. It has access to a 'driver' object which represents a WebdriverIO session attached to the current server.",
          required: true
        },
        {
          name: "type",
          type: "string",
          description: "The language/framework used in the script. Currently, only 'webdriverio' is supported and is the default.",
          required: false
        },
        {
          name: "timeout",
          type: "number",
          description: "The number of milliseconds the script should be allowed to run before being killed by the Appium server. Defaults to the equivalent of 1 hour.",
          required: false
        }
      ],
      returns: {
        type: "object",
        name: "result",
        description: "An object containing two fields: 'result', which is the return value of the script itself, and 'logs', which contains 3 inner fields, 'log', 'warn', and 'error', which hold an array of strings logged by console.log, console.warn, and console.error in the script's execution."
      }
    }
  },
  "/session/:sessionId/appium/events": {
    POST: {
      command: "getEvents",
      description: "Get events stored in appium server.",
      ref: "https://github.com/appium/appium/blob/master/docs/en/commands/session/events/get-events.md",
      parameters: [
        {
          name: "type",
          type: "string[]",
          description: "Get events which are filtered with the type if the type is provided.",
          required: true
        }
      ],
      returns: {
        type: "object",
        name: "result",
        description: "A JSON hash of events like `{'commands' => [{'cmd' => 123455, ....}], 'startTime' => 1572954894127, }`."
      },
      support: {
        android: {
          UiAutomator: "4.2+"
        },
        ios: {
          XCUITest: "9.3+"
        }
      }
    }
  },
  "/session/:sessionId/appium/log_event": {
    POST: {
      command: "logEvent",
      description: "Store a custom event.",
      ref: "https://github.com/appium/appium/blob/master/docs/en/commands/session/events/log-event.md",
      parameters: [
        {
          name: "vendor",
          type: "string",
          description: "The name of vendor. It will be `vendor` in `vendor:event`.",
          required: true
        },
        {
          name: "event",
          type: "string",
          description: "The name of event. It will be `event` in `vendor:event`.",
          required: true
        }
      ],
      support: {
        android: {
          UiAutomator: "4.2+"
        },
        ios: {
          XCUITest: "9.3+"
        }
      }
    }
  },
  "/session/:sessionId/appium/compare_images": {
    POST: {
      command: "compareImages",
      description: "This feature conducts image comparisons utilizing the capabilities of the OpenCV framework. Please note that for this functionality to work, both the OpenCV framework and the opencv4nodejs module must be installed on the machine where the Appium server is operational. ***Furthermore, you'll need to have the [`images-plugin`](https://github.com/appium/appium/tree/master/packages/images-plugin) plugin installed to use this feature with Appium 2.0.***",
      ref: "https://appium.github.io/appium.io/docs/en/writing-running-appium/image-comparison/",
      parameters: [
        {
          name: "mode",
          type: "string",
          description: "One of possible comparison modes: 'matchFeatures', 'getSimilarity', 'matchTemplate'. 'matchFeatures' is by default.",
          required: true,
          default: "matchFeatures"
        },
        {
          name: "firstImage",
          type: "string",
          description: "An image data. All image formats, that OpenCV library itself accepts, are supported.",
          required: true
        },
        {
          name: "secondImage",
          type: "string",
          description: "An image data. All image formats, that OpenCV library itself accepts, are supported.",
          required: true
        },
        {
          name: "options",
          type: "object",
          description: "The content of this dictionary depends on the actual `mode` value. See the documentation on `appium-support` module for more details. ",
          required: true,
          default: {}
        }
      ],
      returns: {
        type: "object",
        name: "result",
        description: "The content of the resulting dictionary depends on the actual `mode` and `options` values. See the documentation on `appium-support` module for more details."
      }
    }
  },
  "/session/:sessionId/timeouts/implicit_wait": {
    POST: {
      command: "implicitWait",
      description: "Set the amount of time the driver should wait when searching for elements. When searching for a single element, the driver should poll the page until an element is found or the timeout expires, whichever occurs first. When searching for multiple elements, the driver should poll the page until at least one element is found or the timeout expires, at which point it should return an empty list. If this command is never sent, the driver should default to an implicit wait of 0ms.",
      ref: "https://github.com/appium/appium/blob/master/packages/base-driver/docs/mjsonwp/protocol-methods.md#webdriver-endpoints",
      parameters: [
        {
          name: "ms",
          type: "number",
          description: "The amount of time, in milliseconds, to wait on an element.",
          required: true
        }
      ],
      support: {
        android: {
          UiAutomator: "4.2+"
        },
        ios: {
          XCUITest: "9.3+"
        }
      }
    }
  },
  "/session/:sessionId/element/:elementId/location_in_view": {
    GET: {
      command: "getLocationInView",
      description: "Determine an element's location on the screen once it has been scrolled into view.<br /><br />__Note:__ This is considered an internal command and should only be used to determine an element's location for correctly generating native events.",
      ref: "https://github.com/appium/appium/blob/master/packages/base-driver/docs/mjsonwp/protocol-methods.md#webdriver-endpoints",
      variables: [
        {
          name: "elementId",
          description: "ID of the element to route the command to"
        }
      ],
      parameters: [],
      returns: {
        type: "Object",
        name: "location",
        description: "The X and Y coordinates for the element on the page."
      },
      support: {
        android: {
          UiAutomator: "4.2+"
        },
        ios: {
          XCUITest: "9.3+"
        }
      }
    }
  },
  "/session/:sessionId/keys": {
    POST: {
      command: "sendKeys",
      description: "Send a sequence of key strokes to the active element",
      ref: "https://github.com/appium/appium/blob/master/packages/base-driver/docs/mjsonwp/protocol-methods.md#webdriver-endpoints",
      parameters: [
        {
          name: "value",
          type: "string[]",
          description: "The sequence of keys to type. An array must be provided.",
          required: true
        }
      ],
      support: {
        android: {
          UiAutomator: "4.2+"
        },
        ios: {
          XCUITest: "9.3+"
        }
      }
    }
  },
  "/session/:sessionId/ime/available_engines": {
    GET: {
      command: "availableIMEEngines",
      description: "List all available engines on the machine. To use an engine, it has to be present in this list.",
      ref: "https://github.com/appium/appium/blob/master/packages/base-driver/docs/mjsonwp/protocol-methods.md#webdriver-endpoints",
      parameters: [],
      returns: {
        type: "String[]",
        name: "engines",
        description: "A list of available engines"
      },
      support: {
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/ime/active_engine": {
    GET: {
      command: "getActiveIMEEngine",
      description: "Get the name of the active IME engine. The name string is platform specific.",
      ref: "https://github.com/appium/appium/blob/master/packages/base-driver/docs/mjsonwp/protocol-methods.md#webdriver-endpoints",
      parameters: [],
      returns: {
        type: "String",
        name: "engine",
        description: "The name of the active IME engine"
      },
      support: {
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/ime/activated": {
    GET: {
      command: "isIMEActivated",
      description: "Indicates whether IME input is active at the moment",
      ref: "https://github.com/appium/appium/blob/master/packages/base-driver/docs/mjsonwp/protocol-methods.md#webdriver-endpoints",
      parameters: [],
      returns: {
        type: "Boolean",
        name: "isActive",
        description: "true if IME input is available and currently active, false otherwise"
      },
      support: {
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/ime/deactivate": {
    POST: {
      command: "deactivateIMEEngine",
      description: "De-activates the currently-active IME engine.",
      ref: "https://github.com/appium/appium/blob/master/packages/base-driver/docs/mjsonwp/protocol-methods.md#webdriver-endpoints",
      parameters: [],
      support: {
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/ime/activate": {
    POST: {
      command: "activateIMEEngine",
      description: "Make an engines that is available",
      ref: "https://github.com/appium/appium/blob/master/packages/base-driver/docs/mjsonwp/protocol-methods.md#webdriver-endpoints",
      parameters: [
        {
          name: "engine",
          type: "string",
          description: "name of the engine to activate",
          required: true
        }
      ],
      support: {
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/timeouts/async_script": {
    POST: {
      command: "asyncScriptTimeout",
      description: "Set the amount of time, in milliseconds, that asynchronous scripts executed by `/session/:sessionId/execute_async` are permitted to run before they are aborted and a `Timeout` error is returned to the client.",
      ref: "https://github.com/appium/appium/blob/master/packages/base-driver/docs/mjsonwp/protocol-methods.md#webdriver-endpoints",
      parameters: [
        {
          name: "ms",
          type: "number",
          description: "The amount of time, in milliseconds, that time-limited commands are permitted to run",
          required: true
        }
      ],
      support: {
        ios: {
          XCUITest: "9.3+"
        }
      }
    }
  },
  "/session/:sessionId/element/:elementId/submit": {
    POST: {
      command: "submit",
      description: "Submit a form element.",
      ref: "https://github.com/appium/appium/blob/master/packages/base-driver/docs/mjsonwp/protocol-methods.md#webdriver-endpoints",
      variables: [
        {
          name: "elementId",
          description: "ID of the form element to be submitted"
        }
      ],
      parameters: [],
      support: {
        ios: {
          XCUITest: "9.3+"
        }
      }
    }
  },
  "/session/:sessionId/element/:elementId/size": {
    GET: {
      command: "getElementSize",
      description: "Determine an element's size in pixels. The size will be returned as a JSON object with `width` and `height` properties.",
      ref: "https://github.com/appium/appium/blob/master/packages/base-driver/docs/mjsonwp/protocol-methods.md#webdriver-endpoints",
      variables: [
        {
          name: "elementId",
          description: "ID of the element to route the command to"
        }
      ],
      parameters: [],
      returns: {
        type: "Object",
        name: "size",
        description: "The width and height of the element, in pixels."
      },
      support: {
        android: {
          UiAutomator: "4.2+"
        },
        ios: {
          XCUITest: "9.3+"
        }
      }
    }
  },
  "/session/:sessionId/element/:elementId/location": {
    GET: {
      command: "getElementLocation",
      description: "Determine an element's location on the page. The point `(0, 0)` refers to the upper-left corner of the page. The element's coordinates are returned as a JSON object with `x` and `y` properties.",
      ref: "https://github.com/appium/appium/blob/master/packages/base-driver/docs/mjsonwp/protocol-methods.md#webdriver-endpoints",
      variables: [
        {
          name: "elementId",
          description: "ID of the element to route the command to"
        }
      ],
      parameters: [],
      returns: {
        type: "Object",
        name: "location",
        description: "The X and Y coordinates for the element on the page."
      },
      support: {
        android: {
          UiAutomator: "4.2+"
        },
        ios: {
          XCUITest: "9.3+"
        }
      }
    }
  },
  "/session/:sessionId/touch/click": {
    POST: {
      command: "touchClick",
      description: "Single tap on the touch enabled device.",
      ref: "https://github.com/appium/appium/blob/master/packages/base-driver/docs/mjsonwp/protocol-methods.md#webdriver-endpoints",
      parameters: [
        {
          name: "element",
          type: "string",
          description: "ID of the element to single tap on.",
          required: true
        }
      ],
      support: {
        android: {
          UiAutomator: "4.2+"
        },
        ios: {
          XCUITest: "9.3+"
        }
      }
    }
  },
  "/session/:sessionId/touch/down": {
    POST: {
      command: "touchDown",
      description: "Finger down on the screen.",
      ref: "https://github.com/appium/appium/blob/master/packages/base-driver/docs/mjsonwp/protocol-methods.md#webdriver-endpoints",
      parameters: [
        {
          name: "x",
          type: "number",
          description: "x coordinate on the screen",
          required: true
        },
        {
          name: "y",
          type: "number",
          description: "y coordinate on the screen",
          required: true
        }
      ],
      support: {
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/touch/up": {
    POST: {
      command: "touchUp",
      description: "Finger up on the screen.",
      ref: "https://github.com/appium/appium/blob/master/packages/base-driver/docs/mjsonwp/protocol-methods.md#webdriver-endpoints",
      parameters: [
        {
          name: "x",
          type: "number",
          description: "x coordinate on the screen",
          required: true
        },
        {
          name: "y",
          type: "number",
          description: "y coordinate on the screen",
          required: true
        }
      ],
      support: {
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/touch/move": {
    POST: {
      command: "touchMove",
      description: "Finger move on the screen.",
      ref: "https://github.com/appium/appium/blob/master/packages/base-driver/docs/mjsonwp/protocol-methods.md#webdriver-endpoints",
      parameters: [
        {
          name: "x",
          type: "number",
          description: "x coordinate on the screen",
          required: true
        },
        {
          name: "y",
          type: "number",
          description: "y coordinate on the screen",
          required: true
        }
      ],
      support: {
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/touch/longclick": {
    POST: {
      command: "touchLongClick",
      description: "Long press on the touch screen using finger motion events.",
      ref: "https://github.com/appium/appium/blob/master/packages/base-driver/docs/mjsonwp/protocol-methods.md#webdriver-endpoints",
      parameters: [
        {
          name: "element",
          type: "string",
          description: "ID of the element to long press on",
          required: true
        }
      ],
      support: {
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/touch/flick": {
    POST: {
      command: "touchFlick",
      description: "Flick on the touch screen using finger motion events. This flick command starts at a particular screen location.",
      ref: "https://github.com/appium/appium/blob/master/packages/base-driver/docs/mjsonwp/protocol-methods.md#webdriver-endpoints",
      parameters: [
        {
          name: "xoffset",
          type: "number",
          description: "the x offset in pixels to flick by",
          required: false
        },
        {
          name: "yoffset",
          type: "number",
          description: "the y offset in pixels to flick by",
          required: false
        },
        {
          name: "element",
          type: "string",
          description: "ID of the element where the flick starts",
          required: false
        },
        {
          name: "speed",
          type: "number",
          description: "the speed in pixels per seconds",
          required: false
        },
        {
          name: "xspeed",
          type: "number",
          description: "the x speed in pixels per second",
          required: false
        },
        {
          name: "yspeed",
          type: "number",
          description: "the y speed in pixels per second",
          required: false
        }
      ],
      support: {
        android: {
          UiAutomator: "4.2+"
        }
      }
    }
  },
  "/session/:sessionId/orientation": {
    GET: {
      command: "getOrientation",
      description: "Get the current device orientation.",
      ref: "https://github.com/appium/appium/blob/master/packages/base-driver/docs/mjsonwp/protocol-methods.md#webdriver-endpoints",
      parameters: [],
      returns: {
        type: "String",
        name: "orientation",
        description: "The current orientation corresponding to a value defined in ScreenOrientation: `LANDSCAPE|PORTRAIT`."
      },
      support: {
        android: {
          UiAutomator: "4.2+"
        },
        ios: {
          XCUITest: "9.3+"
        }
      }
    },
    POST: {
      command: "setOrientation",
      description: "Set the device orientation",
      ref: "https://github.com/appium/appium/blob/master/packages/base-driver/docs/mjsonwp/protocol-methods.md#webdriver-endpoints",
      parameters: [
        {
          name: "orientation",
          type: "string",
          description: "the new browser orientation as defined in ScreenOrientation: `LANDSCAPE|PORTRAIT`",
          required: true
        }
      ],
      support: {
        android: {
          UiAutomator: "4.2+"
        },
        ios: {
          XCUITest: "9.3+"
        }
      }
    }
  },
  "/session/:sessionId/log": {
    POST: {
      command: "getLogs",
      description: "Get the log for a given log type. Log buffer is reset after each request.",
      ref: "https://github.com/appium/appium/blob/master/packages/base-driver/docs/mjsonwp/protocol-methods.md#webdriver-endpoints",
      parameters: [
        {
          name: "type",
          type: "string",
          description: "the log type",
          required: true
        }
      ],
      returns: {
        type: "Object[]",
        name: "logs",
        description: "The list of log entries."
      },
      support: {
        android: {
          UiAutomator: "4.2+"
        },
        ios: {
          XCUITest: "9.3+"
        }
      }
    }
  },
  "/session/:sessionId/log/types": {
    GET: {
      command: "getLogTypes",
      description: "Get available log types.",
      ref: "https://github.com/appium/appium/blob/master/packages/base-driver/docs/mjsonwp/protocol-methods.md#webdriver-endpoints",
      parameters: [],
      returns: {
        type: "String[]",
        name: "logTypes",
        description: "The list of available log types."
      },
      support: {
        android: {
          UiAutomator: "4.2+"
        },
        ios: {
          XCUITest: "9.3+"
        }
      }
    }
  }
};

// src/protocols/chromium.ts
var chromium_default = {
  "/session/:sessionId/alert": {
    GET: {
      command: "isAlertOpen",
      description: "Whether a simple dialog is currently open.",
      ref: "https://github.com/bayandin/chromedriver/blob/v2.45/alert_commands.cc#L42-L49",
      examples: [
        [
          "console.log(browser.isAlertOpen()); // outputs: false",
          "browser.execute('window.alert()');",
          "console.log(browser.isAlertOpen()); // outputs: true"
        ]
      ],
      parameters: [],
      returns: {
        type: "Boolean",
        name: "isAlertOpen",
        description: "`true` or `false` based on whether simple dialog is present or not."
      }
    }
  },
  "/session/:sessionId/autoreport": {
    GET: {
      command: "isAutoReporting",
      description: "Whether it should automatically raises errors on browser logs.",
      ref: "https://codereview.chromium.org/101203012",
      parameters: [],
      returns: {
        type: "Boolean",
        name: "isAutoReporting",
        description: "`true` or `false` based on whether auto reporting is enabled."
      }
    },
    POST: {
      command: "setAutoReporting",
      description: "Toggle whether to return response with unknown error with first browser error (e.g. failed to load resource due to 403/404 response) for all subsequent commands (once enabled).",
      ref: "https://codereview.chromium.org/101203012",
      examples: [
        [
          "// Enable auto reporting first thing after session was initiated with empty browser logs",
          "console.log(browser.setAutoReporting(true)); // outputs: null",
          "// Upon requesting an non-existing resource it will abort execution due to thrown unknown error",
          "browser.url('https://webdriver.io/img/404-does-not-exist.png');"
        ],
        [
          "// During the session do some operations which populate the browser logs",
          "browser.url('https://webdriver.io/img/404-does-not-exist.png');",
          "browser.url('https://webdriver.io/403/no-access');",
          "// Enable auto reporting which throws an unknown error for first browser log (404 response)",
          "browser.setAutoReporting(true);"
        ]
      ],
      parameters: [
        {
          name: "enabled",
          type: "boolean",
          description: "`true` if auto reporting should be enabled, use `false` to disable previously enabled auto reporting.",
          required: true
        }
      ],
      returns: {
        type: "Object|Null",
        name: "firstBrowserError",
        description: "In case first browser error already occured prior to executing this command it will throw unknown error as response, which is an object with 'message' key describing first browser error. Otherwise it returns `null` on success."
      }
    }
  },
  "/session/:sessionId/is_loading": {
    GET: {
      command: "isLoading",
      description: "Determines load status for active window handle.",
      ref: "https://github.com/bayandin/chromedriver/blob/v2.45/session_commands.cc#L783-L802",
      examples: [
        [
          "console.log(browser.isLoading()); // outputs: false",
          "browser.newWindow('https://webdriver.io');",
          "console.log(browser.isLoading()); // outputs: true"
        ]
      ],
      parameters: [],
      returns: {
        type: "Boolean",
        name: "isLoading",
        description: "`true` or `false` based on whether active window handle is loading or not."
      }
    }
  },
  "/session/:sessionId/chromium/heap_snapshot": {
    GET: {
      command: "takeHeapSnapshot",
      description: "Takes a heap snapshot of the current execution context.",
      ref: "https://github.com/bayandin/chromedriver/blob/v2.45/chrome/web_view.h#L198-L202",
      parameters: [],
      returns: {
        type: "Object",
        name: "heapSnapshot",
        description: "A JSON representation of the heap snapshot. Which can be inspected by loading as file into Chrome DevTools."
      }
    }
  },
  "/session/:sessionId/network_connection": {
    GET: {
      command: "getNetworkConnection",
      description: "Get the connection type for network emulation. This command is only applicable when remote end replies with `networkConnectionEnabled` capability set to `true`.",
      ref: "https://github.com/SeleniumHQ/mobile-spec/blob/master/spec-draft.md#device-modes",
      examples: [
        [
          "const browser = remote({",
          "    capabilities: {",
          "        browserName: 'chrome',",
          "        'goog:chromeOptions': {",
          "            // Network emulation requires device mode, which is only enabled when mobile emulation is on",
          "            mobileEmulation: { deviceName: 'iPad' },",
          "        },",
          "    }",
          "});",
          "console.log(browser.getNetworkConnection()); // outputs: 6 (Both Wi-Fi and data)"
        ]
      ],
      parameters: [],
      returns: {
        type: "Number",
        name: "connectionType",
        description: "A bitmask to represent the network connection type. Airplane Mode (`1`), Wi-Fi only (`2`), Wi-Fi and data (`6`), 4G (`8`), 3G (`10`), 2G (`20`). By default [Wi-Fi and data are enabled](https://github.com/bayandin/chromedriver/blob/v2.45/chrome/chrome_desktop_impl.cc#L36-L37)."
      }
    },
    POST: {
      command: "setNetworkConnection",
      description: "Change connection type for network connection. This command is only applicable when remote end replies with `networkConnectionEnabled` capability set to `true`.",
      ref: "https://github.com/SeleniumHQ/mobile-spec/blob/master/spec-draft.md#device-modes",
      examples: [
        [
          "const browser = remote({",
          "    capabilities: {",
          "        browserName: 'chrome',",
          "        'goog:chromeOptions': {",
          "            // Network emulation requires device mode, which is only enabled when mobile emulation is on",
          "            mobileEmulation: { deviceName: 'iPad' },",
          "        },",
          "    }",
          "});",
          "console.log(browser.setNetworkConnection({ type: 1 })); // outputs: 1 (Airplane Mode)"
        ]
      ],
      parameters: [
        {
          name: "parameters",
          type: "object",
          description: "Object containing ConnectionType, set bitmask as value for `type` key in object. Airplane Mode (`1`), Wi-Fi only (`2`), Wi-Fi and data (`6`), 4G (`8`), 3G (`10`), 2G (`20`).",
          required: true
        }
      ],
      returns: {
        type: "Number",
        name: "connectionType",
        description: "A bitmask to represent the network connection type. Value should match specified `type` in object, however device might not be capable of the network connection type requested."
      }
    }
  },
  "/session/:sessionId/chromium/network_conditions": {
    GET: {
      command: "getNetworkConditions",
      description: "Get current network conditions used for emulation.",
      ref: "https://github.com/bayandin/chromedriver/blob/v2.45/session_commands.cc#L839-L859",
      parameters: [],
      returns: {
        type: "Object",
        name: "networkConditions",
        description: "Object containing network conditions for `offline`, `latency`, `download_throughput` and `upload_throughput`. Network conditions must be set before it can be retrieved."
      }
    },
    POST: {
      command: "setNetworkConditions",
      description: "Set network conditions used for emulation by throttling connection.",
      ref: "https://github.com/bayandin/chromedriver/blob/v2.45/window_commands.cc#L1663-L1722",
      examples: [
        [
          "// Use different download (25kb/s) and upload (50kb/s) throughput values for throttling with a latency of 1000ms",
          "browser.setNetworkConditions({ latency: 1000, download_throughput: 25600, upload_throughput: 51200 });"
        ],
        [
          "// Force disconnected from network by setting 'offline' to true",
          "browser.setNetworkConditions({ latency: 0, throughput: 0, offline: true });"
        ],
        [
          "// When preset name (e.g. 'DSL') is specified it does not respect values in object (e.g. 'offline')",
          "browser.setNetworkConditions({ latency: 0, throughput: 0, offline: true }, 'DSL');"
        ],
        [
          "// Best practice for specifying network throttling preset is to use an empty object",
          "browser.setNetworkConditions({}, 'Good 3G');"
        ]
      ],
      parameters: [
        {
          name: "network_conditions",
          type: "object",
          description: "Object containing network conditions which are `latency`, `throughput` (or `download_throughput`/`upload_throughput`) and `offline` (optional).",
          required: true
        },
        {
          name: "network_name",
          type: "string",
          description: "Name of [network throttling preset](https://github.com/bayandin/chromedriver/blob/v2.45/chrome/network_list.cc#L12-L25). `GPRS`, `Regular 2G`, `Good 2G`, `Regular 3G`, `Good 3G`, `Regular 4G`, `DSL`, `WiFi` or `No throttling` to disable. When preset is specified values passed in first argument are not respected.",
          required: false
        }
      ]
    },
    DELETE: {
      command: "deleteNetworkConditions",
      description: "Disable any network throttling which might have been set. Equivalent of setting the `No throttling` preset.",
      ref: "https://github.com/bayandin/chromedriver/blob/v2.45/window_commands.cc#L1724-L1745",
      parameters: []
    }
  },
  "/session/:sessionId/chromium/send_command": {
    POST: {
      command: "sendCommand",
      description: "Send a command to the DevTools debugger.<br />For a list of available commands and their parameters refer to the [Chrome DevTools Protocol Viewer](https://chromedevtools.github.io/devtools-protocol/).",
      ref: "https://github.com/bayandin/chromedriver/blob/v2.45/window_commands.cc#L1290-L1304",
      parameters: [
        {
          name: "cmd",
          type: "string",
          description: "Name of the command (e.g. [`Browser.close`](https://chromedevtools.github.io/devtools-protocol/1-3/Browser#method-close)).",
          required: true
        },
        {
          name: "params",
          type: "object",
          description: "Parameters to the command. In case no parameters for command, specify an empty object.",
          required: true
        }
      ]
    }
  },
  "/session/:sessionId/chromium/send_command_and_get_result": {
    POST: {
      command: "sendCommandAndGetResult",
      description: "Send a command to the DevTools debugger and wait for the result.<br />For a list of available commands and their parameters refer to the [Chrome DevTools Protocol Viewer](https://chromedevtools.github.io/devtools-protocol/).",
      ref: "https://github.com/bayandin/chromedriver/blob/v2.45/window_commands.cc#L1306-L1320",
      parameters: [
        {
          name: "cmd",
          type: "string",
          description: "Name of the command which returns a result (e.g. [`Network.getAllCookies`](https://chromedevtools.github.io/devtools-protocol/1-3/Network#method-getAllCookies)).",
          required: true
        },
        {
          name: "params",
          type: "object",
          description: "Parameters to the command. In case no parameters for command, specify an empty object.",
          required: true
        }
      ],
      returns: {
        type: "*",
        name: "result",
        description: "Either the return value of your command, or the error which was the reason for your command's failure."
      }
    }
  },
  "/session/:sessionId/file": {
    POST: {
      command: "file",
      description: "Upload a file to remote machine on which the browser is running.",
      ref: "https://github.com/bayandin/chromedriver/blob/v2.45/session_commands.cc#L1037-L1065",
      parameters: [
        {
          name: "file",
          type: "string",
          description: "Base64-encoded zip archive containing __single__ file which to upload. In case base64-encoded data does not represent a zip archive or archive contains more than one file it will throw an unknown error.",
          required: true
        }
      ],
      returns: {
        type: "String",
        name: "path",
        description: "Absolute path of uploaded file on remote machine."
      }
    }
  },
  "/session/:sessionId/chromium/launch_app": {
    POST: {
      command: "launchChromeApp",
      description: "Launches a Chrome app by specified id.",
      ref: "https://github.com/bayandin/chromedriver/blob/v2.45/session_commands.cc#L521-L539",
      examples: [
        [
          "import fs from 'fs'",
          "const browser = remote({",
          "    capabilities: {",
          "        browserName: 'chrome',",
          "        'goog:chromeOptions': {",
          "            // Install upon starting browser in order to launch it",
          "            extensions: [",
          "              // Entry should be a base64-encoded packed Chrome app (.crx)",
          "              fs.readFileSync('/absolute/path/app.crx').toString('base64')",
          "            ]",
          "        }",
          "    }",
          "});",
          "browser.launchChromeApp('aohghmighlieiainnegkcijnfilokake')); // Google Docs (https://chrome.google.com/webstore/detail/docs/aohghmighlieiainnegkcijnfilokake)"
        ]
      ],
      parameters: [
        {
          name: "id",
          type: "string",
          description: "Extension id of app to be launched, as defined in chrome://extensions.",
          required: true
        }
      ]
    }
  },
  "/session/:sessionId/element/:elementId/value": {
    GET: {
      command: "getElementValue",
      description: "Retrieves the value of a given form control element.",
      ref: "https://github.com/bayandin/chromedriver/blob/v2.45/element_commands.cc#L431-L443",
      variables: [
        {
          name: "elementId",
          description: "id of element to get value from"
        }
      ],
      parameters: [],
      returns: {
        type: "String|Null",
        name: "value",
        description: "Current value of the element. In case specified element is not a form control element, it will return `null`."
      }
    }
  },
  "/session/:sessionId/element/:elementId/hover": {
    POST: {
      command: "elementHover",
      description: "Enable hover state for an element, which is reset upon next interaction.",
      ref: "https://github.com/bayandin/chromedriver/blob/v2.45/element_commands.cc#L126-L146",
      variables: [
        {
          name: "elementId",
          description: "id of element to hover over to"
        }
      ],
      parameters: []
    }
  },
  "/session/:sessionId/touch/pinch": {
    POST: {
      command: "touchPinch",
      description: "Trigger a pinch zoom effect.",
      ref: "https://github.com/bayandin/chromedriver/blob/v2.45/window_commands.cc#L813-L827",
      parameters: [
        {
          name: "x",
          type: "number",
          description: "x position to pinch on",
          required: true
        },
        {
          name: "y",
          type: "number",
          description: "y position to pinch on",
          required: true
        },
        {
          name: "scale",
          type: "number",
          description: "pinch zoom scale",
          required: true
        }
      ]
    }
  },
  "/session/:sessionId/goog/page/freeze": {
    POST: {
      command: "freeze",
      description: "Freeze the current page. Extension for [Page Lifecycle API](https://developers.google.com/web/updates/2018/07/page-lifecycle-api).",
      ref: "https://github.com/bayandin/chromedriver/blob/v2.45/window_commands.cc#L625-L633",
      parameters: []
    }
  },
  "/session/:sessionId/goog/page/resume": {
    POST: {
      command: "resume",
      description: "Resume the current page. Extension for [Page Lifecycle API](https://developers.google.com/web/updates/2018/07/page-lifecycle-api).",
      ref: "https://github.com/bayandin/chromedriver/blob/v2.45/window_commands.cc#L635-L645",
      parameters: []
    }
  },
  "/session/:sessionId/goog/cast/get_sinks": {
    GET: {
      command: "getCastSinks",
      description: "Returns the list of cast sinks (Cast devices) available to the Chrome media router.",
      ref: "https://chromium.googlesource.com/chromium/src/+/refs/tags/73.0.3683.121/chrome/test/chromedriver/server/http_handler.cc#748",
      parameters: [],
      returns: {
        type: "string[]",
        name: "sinks",
        description: "List of available sinks."
      }
    }
  },
  "/session/:sessionId/goog/cast/set_sink_to_use": {
    POST: {
      command: "selectCastSink",
      description: "Selects a cast sink (Cast device) as the recipient of media router intents (connect or play).",
      ref: "https://chromium.googlesource.com/chromium/src/+/refs/tags/73.0.3683.121/chrome/test/chromedriver/server/http_handler.cc#737",
      parameters: [
        {
          name: "sinkName",
          type: "string",
          description: "The name of the target device.",
          required: true
        }
      ]
    }
  },
  "/session/:sessionId/goog/cast/start_tab_mirroring": {
    POST: {
      command: "startCastTabMirroring",
      description: "Initiates tab mirroring for the current browser tab on the specified device.",
      ref: "https://chromium.googlesource.com/chromium/src/+/refs/tags/73.0.3683.121/chrome/test/chromedriver/server/http_handler.cc#741",
      parameters: [
        {
          name: "sinkName",
          type: "string",
          description: "The name of the target device.",
          required: true
        }
      ]
    }
  },
  "/session/:sessionId/goog/cast/get_issue_message": {
    GET: {
      command: "getCastIssueMessage",
      description: "Returns error message if there is any issue in a Cast session.",
      ref: "https://chromium.googlesource.com/chromium/src/+/refs/tags/73.0.3683.121/chrome/test/chromedriver/server/http_handler.cc#751",
      parameters: [],
      returns: {
        type: "String",
        name: "message",
        description: "Error message, if any."
      }
    }
  },
  "/session/:sessionId/goog/cast/stop_casting": {
    POST: {
      command: "stopCasting",
      description: "Stops casting from media router to the specified device, if connected.",
      ref: "https://chromium.googlesource.com/chromium/src/+/refs/tags/73.0.3683.121/chrome/test/chromedriver/server/http_handler.cc#744",
      parameters: [
        {
          name: "sinkName",
          type: "string",
          description: "The name of the target device.",
          required: true
        }
      ]
    }
  },
  "/shutdown": {
    POST: {
      command: "shutdown",
      description: "Shutdown ChromeDriver process and consequently terminating all active sessions.",
      ref: "https://github.com/bayandin/chromedriver/blob/v2.45/session_commands.cc#L489-L498",
      parameters: []
    }
  },
  "/session/:sessionId/element/:elementId/screenshot": {
    GET: {
      command: "takeElementScreenshot",
      description: "The Take Element Screenshot command takes a screenshot of the visible region encompassed by the bounding rectangle of an element.",
      ref: "https://w3c.github.io/webdriver/#dfn-take-element-screenshot",
      variables: [
        {
          name: "elementId",
          description: "the id of an element returned in a previous call to Find Element(s)"
        }
      ],
      parameters: [
        {
          name: "scroll",
          type: "boolean",
          description: "scroll into view the element. Default: true",
          required: false
        }
      ],
      returns: {
        type: "String",
        name: "screenshot",
        description: "The base64-encoded PNG image data comprising the screenshot of the visible region of an element\u2019s bounding rectangle after it has been scrolled into view."
      }
    }
  },
  "/session/:sessionId/se/log/types": {
    GET: {
      command: "getLogTypes",
      description: "Get available log types.",
      ref: "https://github.com/SeleniumHQ/selenium/wiki/JsonWireProtocol#sessionsessionidlogtypes",
      parameters: [],
      returns: {
        type: "String[]",
        name: "logTypes",
        description: "The list of available log types, example: browser, driver."
      }
    }
  },
  "/session/:sessionId/se/log": {
    POST: {
      command: "getLogs",
      description: "Get the log for a given log type. Log buffer is reset after each request.",
      ref: "https://github.com/SeleniumHQ/selenium/wiki/JsonWireProtocol#sessionsessionidlog",
      parameters: [
        {
          name: "type",
          type: "string",
          description: "the log type",
          required: true
        }
      ],
      returns: {
        type: "Object[]",
        name: "logs",
        description: "The list of log entries."
      }
    }
  }
};

// src/protocols/gecko.ts
var gecko_default = {
  "/session/:sessionId/moz/screenshot/full": {
    GET: {
      command: "fullPageScreenshot",
      description: "Captures a screenshot of the entire page.",
      ref: "https://phabricator.services.mozilla.com/source/mozilla-central/browse/default/testing/geckodriver/src/command.rs$43-46",
      parameters: [],
      returns: {
        type: "String",
        name: "screenshot",
        description: "The base64-encoded PNG image data comprising the screenshot of the full page."
      }
    }
  },
  "/session/:sessionId/moz/context": {
    GET: {
      command: "getMozContext",
      description: "Get the context that is currently in effect, e.g. `CHROME` or `CONTENT`.",
      ref: "https://github.com/SeleniumHQ/selenium/blob/586affe0cf675b1d5c8abc756defa4a46d95391b/javascript/node/selenium-webdriver/firefox.js#L615-L622",
      examples: [
        [
          "console.log(await browser.getMozContext()); // outputs: 'CHROME'"
        ]
      ],
      parameters: [],
      returns: {
        type: "String",
        name: "Context",
        description: "The browser context, either `CHROME` or `CONTENT`"
      }
    },
    POST: {
      command: "setMozContext",
      description: "Changes target context for commands between chrome- and content.<br /><br />Changing the current context has a stateful impact on all subsequent commands. The `CONTENT` context has normal web platform document permissions, as if you would evaluate arbitrary JavaScript. The `CHROME` context gets elevated permissions that lets you manipulate the browser chrome itself, with full access to the XUL toolkit.",
      ref: "https://github.com/SeleniumHQ/selenium/blob/586affe0cf675b1d5c8abc756defa4a46d95391b/javascript/node/selenium-webdriver/firefox.js#L615-L645",
      examples: [
        [
          "console.log(await browser.getMozContext()); // outputs: 'CHROME'",
          "browser.setMozContext('CONTENT');",
          "console.log(await browser.getMozContext()); // outputs: 'CONTENT'"
        ]
      ],
      parameters: [
        {
          name: "context",
          type: "string",
          description: "The browser context, either `CHROME` or `CONTENT`",
          required: true
        }
      ]
    }
  },
  "/session/:sessionId/moz/addon/install": {
    POST: {
      command: "installAddOn",
      description: "Installs a new addon with the current session. This function will return an ID that may later be used to uninstall the addon using `uninstallAddon`.",
      ref: "https://github.com/SeleniumHQ/selenium/blob/586affe0cf675b1d5c8abc756defa4a46d95391b/javascript/node/selenium-webdriver/firefox.js#L647-L668",
      examples: [
        [
          "// Create a buffer of the add on .zip file",
          "const extension = await fs.promises.readFile('/path/to/extension.zip')",
          "// Load extension in Firefox",
          "const id = await browser.installAddOn(extension.toString('base64'), false);"
        ]
      ],
      parameters: [
        {
          // cypress uses 'path' over 'addon' to avoid zipping the extension,
          // which is easier for cypress to incorporate and doesn't break users
          name: "path",
          type: "string",
          description: "path to the extension",
          required: true
        },
        {
          name: "temporary",
          type: "boolean",
          description: "temporary Flag indicating whether the extension should be installed temporarily - gets removed on restart",
          required: true
        }
      ],
      returns: {
        type: "String",
        name: "id",
        description: "A promise that will resolve to an ID for the newly installed addon."
      }
    }
  },
  "/session/:sessionId/moz/addon/uninstall": {
    GET: {
      command: "uninstallAddOn",
      description: "Uninstalls an addon from the current browser session's profile.",
      ref: "https://github.com/SeleniumHQ/selenium/blob/586affe0cf675b1d5c8abc756defa4a46d95391b/javascript/node/selenium-webdriver/firefox.js#L670-L687",
      examples: [
        [
          "// Create a buffer of the add on .zip file",
          "const extension = await fs.promises.readFile('/path/to/extension.zip')",
          "// Load extension in Firefox",
          "const id = await browser.installAddOn(extension.toString('base64'), false);",
          "// ...",
          "await browser.uninstallAddOn(id)"
        ]
      ],
      parameters: [
        {
          name: "id",
          type: "string",
          description: "id ID of the addon to uninstall.",
          required: true
        }
      ]
    }
  }
};

// src/protocols/saucelabs.ts
var saucelabs_default = {
  "/session/:sessionId/sauce/ondemand/log": {
    POST: {
      command: "getPageLogs",
      description: "Get webpage specific log information based on the last page load.",
      ref: "https://docs.saucelabs.com/insights/debug/#network-logs",
      examples: [
        [
          "// Get Network Logs",
          "console.log(browser.getPageLogs('sauce:network'));",
          "/**",
          " * outputs:",
          " * [{",
          ' *   "url": "https://app.saucelabs.com/dashboard",',
          ' *   "statusCode": 200,',
          ' *   "method": "GET",',
          ' *   "requestHeaders": {',
          " *     ...",
          " *   },",
          ' *   "responseHeaders": {',
          " *     ...",
          " *   },",
          ' *   "timing": {',
          " *     ...",
          " *   }",
          " * }, {,",
          " *   ...",
          " * }]",
          " */"
        ],
        [
          "// Get Performance Logs (needs capturePerformance capability see: https://docs.saucelabs.com/performance/transitions/#setting-performance-capabilities",
          "console.log(browser.getPageLogs('sauce:performance'));",
          "/**",
          " * outputs:",
          " * {",
          ' *   "speedIndex": 1472.023,',
          ' *   "timeToFirstInteractive": 1243.214,',
          ' *   "firstMeaningfulPaint": 892.643,',
          " *   ...",
          " * }",
          " */"
        ]
      ],
      parameters: [
        {
          name: "type",
          type: "string",
          description: "log type (e.g. sauce:network', 'sauce:performance')",
          required: true
        }
      ],
      returns: {
        type: "object",
        name: "log",
        description: "log output of desired type (see example)"
      }
    }
  },
  "/session/:sessionId/sauce/ondemand/throttle/network": {
    POST: {
      command: "sauceThrottleNetwork",
      description: "With network conditioning you can test your site on a variety of network connections, including Edge, 3G, and even offline. You can throttle the data throughput, including the maximum download and upload throughput, and use latency manipulation to enforce a minimum delay in connection round-trip time (RTT).",
      ref: "https://docs.saucelabs.com/insights/debug/#saucethrottlenetwork",
      examples: [
        [
          "// predefined network condition",
          "browser.sauceThrottleNetwork('offline')"
        ],
        [
          "// custom network condition",
          "browser.sauceThrottleNetwork({",
          "  download: 1000,",
          "  upload: 500,",
          "  latency: 40'",
          "})"
        ]
      ],
      parameters: [
        {
          name: "condition",
          type: "(string|object)",
          description: "network condition to set (e.g. 'online', 'offline', 'GPRS', 'Regular 2G', 'Good 2G', 'Regular 3G', 'Good 3G', 'Regular 4G', 'DSL', 'Wifi')",
          required: true
        }
      ]
    }
  },
  "/session/:sessionId/sauce/ondemand/throttle/cpu": {
    POST: {
      command: "throttleCPU",
      description: "You can throttle the CPU in DevTools to understand how your page performs under that constraint.",
      ref: "https://docs.saucelabs.com/insights/debug/#saucethrottlecpu",
      examples: [
        [
          "// throttle CPU and make it run 4x slower",
          "browser.throttleCPU(4)"
        ],
        ["// reset CPU throttling", "browser.throttleCPU(0)"]
      ],
      parameters: [
        {
          name: "rate",
          type: "number",
          description: "Rate on how much the CPU should get throttled.",
          required: true
        }
      ]
    }
  },
  "/session/:sessionId/sauce/ondemand/intercept": {
    POST: {
      command: "interceptRequest",
      description: "Allows modifying any request made by the browser. You can blacklist, modify, or redirect these as required for your tests.",
      ref: "https://docs.saucelabs.com/insights/debug/#intercept-network-requests",
      examples: [
        [
          "// redirect a request",
          "browser.interceptRequest({",
          "  url: 'https://saucelabs.com',",
          "  redirect: 'https://google.com'",
          "})"
        ],
        [
          "// Blacklist requests to 3rd party vendors",
          "browser.interceptRequest({",
          "  url: 'https://api.segment.io/v1/p',",
          "  error: 'Failed'",
          "})"
        ],
        [
          "// Modify requests to REST API (Mock REST API response)",
          "browser.interceptRequest({",
          "  url: 'http://sampleapp.appspot.com/api/todos',",
          "  response: {",
          "    headers: {",
          "      'x-custom-headers': 'foobar'",
          "    },",
          "    body: [{",
          "      title: 'My custom todo',",
          "      order: 1,",
          "      completed: false,",
          "      url: 'http://todo-backend-express.herokuapp.com/15727'",
          "    }]",
          "  }",
          "})"
        ]
      ],
      parameters: [
        {
          name: "rule",
          type: "object",
          description: "A rule describing the request interception.",
          required: true
        }
      ]
    }
  },
  "/session/:sessionId/sauce/ondemand/performance": {
    POST: {
      command: "assertPerformance",
      description: "Assert against the performance baseline of your app.",
      ref: "https://docs.saucelabs.com/performance/transitions/#setting-performance-capabilities",
      examples: [
        [
          "// test performance for a page",
          "browser.url('https://webdriver.io')",
          "const hasRegression = browser.assertPerformance({",
          "  name: 'my performance test', // make sure that the name is also set in the sauce options in your capabilities",
          "  metrics: ['score', 'firstPaint']",
          "})"
        ]
      ],
      parameters: [
        {
          name: "name",
          type: "string",
          description: "Name of the job you created your baseline with.",
          required: true
        },
        {
          name: "metrics",
          type: "string[]",
          description: "Name of metrics you want to assert agains the baseline.",
          required: false
        }
      ],
      returns: {
        type: "object",
        name: "hasRegression",
        description: "An object containing the result as well as metrics about the result."
      }
    }
  },
  "/session/:sessionId/sauce/ondemand/performance/scroll": {
    POST: {
      command: "jankinessCheck",
      description: "Perform a scroll test that evaluates the jankiness of the application.",
      ref: "https://docs.saucelabs.com/performance/motion/#implementing-the-jankiness-custom-command",
      examples: [
        [
          "// test performance for a page",
          "browser.url('https://webdriver.io')",
          "browser.jankinessCheck()"
        ]
      ],
      parameters: [],
      returns: {
        type: "object",
        name: "testResults",
        description: "An object containing the score as well as metrics around how smooth the UX of the page was during the test."
      }
    }
  },
  "/session/:sessionId/sauce/ondemand/mock": {
    POST: {
      command: "mockRequest",
      description: "Mocks a network resource.",
      ref: "https://docs.saucelabs.com/",
      parameters: [
        {
          name: "url",
          type: "string",
          description: "URL glob to match url to mock.",
          required: true
        },
        {
          name: "filterOptions",
          description: "Additional filter options for url to mock (e.g. headers, method).",
          type: "object",
          required: false
        }
      ],
      returns: {
        type: "object",
        name: "mockId",
        description: "An object containing the id of a mock resource."
      }
    }
  },
  "/session/:sessionId/sauce/ondemand/mock/:mockId": {
    GET: {
      command: "getMockCalls",
      description: "Receive request information about requests that match the mocked resource.",
      ref: "https://docs.saucelabs.com/",
      variables: [
        {
          name: "mockId",
          description: "the id of a mock"
        }
      ],
      parameters: [],
      returns: {
        type: "object",
        name: "requests",
        description: "A list of request information."
      }
    },
    DELETE: {
      command: "clearMockCalls",
      description: "Clear list of mock calls.",
      ref: "https://docs.saucelabs.com/",
      variables: [
        {
          name: "mockId",
          description: "the id of a mock"
        }
      ],
      parameters: [
        {
          type: "boolean",
          name: "restore",
          description: "Set to true if mock should be restored as well.",
          required: false
        }
      ]
    },
    POST: {
      command: "respondMock",
      description: "Respond if mock matches a specific resource.",
      ref: "https://docs.saucelabs.com/",
      variables: [
        {
          name: "mockId",
          description: "the id of a mock"
        }
      ],
      parameters: [
        {
          type: "object",
          name: "payload",
          description: "Information on mock response."
        }
      ]
    }
  }
};

// src/protocols/selenium.ts
var selenium_default = {
  "/session/:sessionId/se/file": {
    POST: {
      command: "file",
      description: "Upload a file to remote machine on which the browser is running.",
      ref: "https://www.seleniumhq.org/",
      parameters: [
        {
          name: "file",
          type: "string",
          description: "Base64-encoded zip archive containing __single__ file which to upload. In case base64-encoded data does not represent a zip archive or archive contains more than one file it will throw an unknown error.",
          required: true
        }
      ],
      returns: {
        type: "String",
        name: "path",
        description: "Absolute path of uploaded file on remote machine."
      }
    }
  },
  "/session/:sessionId/se/files": {
    GET: {
      command: "getDownloadableFiles",
      description: "List files from remote machine available for download.",
      ref: "https://www.seleniumhq.org/",
      parameters: [],
      returns: {
        type: "Object",
        name: "names",
        description: "Object containing a list of downloadable files on remote machine."
      }
    },
    POST: {
      command: "download",
      description: "Download a file from remote machine on which the browser is running.",
      ref: "https://www.seleniumhq.org/",
      parameters: [
        {
          name: "name",
          type: "string",
          description: "Name of the file to be downloaded",
          required: true
        }
      ],
      returns: {
        type: "Object",
        name: "data",
        description: "Object containing downloaded file name and its content"
      }
    },
    DELETE: {
      command: "deleteDownloadableFiles",
      description: "Remove all downloadable files from remote machine on which the browser is running.",
      ref: "https://www.seleniumhq.org/",
      parameters: []
    }
  },
  "/grid/api/hub/": {
    GET: {
      isHubCommand: true,
      command: "getHubConfig",
      description: "Receive hub config remotely.",
      ref: "https://github.com/nicegraham/selenium-grid2-api#gridapihub",
      parameters: [],
      returns: {
        type: "Object",
        name: "config",
        description: "Returns the hub config with slotCount, timeouts and other information."
      }
    }
  },
  "/grid/api/testsession?session=:session": {
    GET: {
      isHubCommand: true,
      command: "gridTestSession",
      description: "Get the details of the Selenium Grid node running a session.",
      ref: "https://github.com/nicegraham/selenium-grid2-api#gridapitestsession",
      parameters: [],
      variables: [
        {
          name: "session",
          description: "The id of the session to receive hub details for."
        }
      ],
      returns: {
        type: "Object",
        name: "details",
        description: "Object containing information about session details."
      }
    }
  },
  "/grid/api/proxy": {
    POST: {
      isHubCommand: true,
      command: "gridProxyDetails",
      description: "Get proxy details.",
      ref: "https://github.com/nicegraham/selenium-grid2-api#gridapiproxy",
      parameters: [
        {
          name: "id",
          type: "string",
          description: "the id of the proxy (can be received using gridTestSession command).",
          required: true
        }
      ],
      returns: {
        type: "Object",
        name: "details",
        description: "Object containing information about proxy."
      }
    }
  },
  "/lifecycle-manager?action=:action": {
    GET: {
      isHubCommand: true,
      command: "manageSeleniumHubLifecycle",
      description: "Manage lifecycle of hub node.",
      ref: "https://github.com/nicegraham/selenium-grid2-api#lifecycle-manager",
      parameters: [],
      variables: [
        {
          name: "action",
          description: "Command to call on Selenium Hub. The only implemented action is to 'shutdown' the hub."
        }
      ]
    }
  },
  "/graphql": {
    POST: {
      isHubCommand: true,
      command: "queryGrid",
      description: "Send GraphQL queries to the Selenium (hub or node) server to fetch data. (Only supported with Selenium v4 Server)",
      ref: "https://www.selenium.dev/documentation/grid/advanced_features/graphql_support/",
      parameters: [
        {
          name: "query",
          type: "string",
          description: "A GraphQL query to be send to the server.",
          required: true
        }
      ],
      examples: [
        [
          "const result = await browser.queryGrid('{ nodesInfo { nodes { status, uri } } }');",
          "console.log(JSON.stringify(result, null, 4))",
          "/**",
          " * outputs:",
          " * {",
          ' *   "data": {',
          ' *     "nodesInfo": {',
          ' *       "nodes": [{',
          ' *         "status": "UP",',
          ' *         "uri": "http://192.168.0.39:4444"',
          " *       }]",
          " *     }",
          " *   }",
          " * }",
          " */"
        ]
      ],
      returns: {
        type: "Object",
        name: "data",
        description: "Result of the GraphQL query."
      }
    }
  }
};

// src/index.ts
var CAPABILITY_KEYS = [
  "browserName",
  "browserVersion",
  "platformName",
  "acceptInsecureCerts",
  "pageLoadStrategy",
  "proxy",
  "setWindowRect",
  "timeouts",
  "strictFileInteractability",
  "unhandledPromptBehavior",
  "webSocketUrl"
];
export {
  appium_default as AppiumProtocol,
  CAPABILITY_KEYS,
  chromium_default as ChromiumProtocol,
  gecko_default as GeckoProtocol,
  mjsonwp_default as MJsonWProtocol,
  saucelabs_default as SauceLabsProtocol,
  selenium_default as SeleniumProtocol,
  webdriverBidi_default as WebDriverBidiProtocol,
  webdriver_default as WebDriverProtocol
};
