"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.fetchBuilder = void 0;
function sanitize(params, defaults) {
    const result = Object.assign(Object.assign({}, defaults), params);
    if (typeof result.retries === 'undefined') {
        result.retries = defaults.retries;
    }
    if (typeof result.retryDelay === 'undefined') {
        result.retryDelay = defaults.retryDelay;
    }
    if (typeof result.retryOn === 'undefined') {
        result.retryOn = defaults.retryOn;
    }
    return result;
}
// eslint-disable-next-line @typescript-eslint/no-explicit-any
function fetchBuilder(fetchFunc, params = {}) {
    const defaults = sanitize(params, { retries: 3, retryDelay: 500, retryOn: [419, 503, 504] });
    return function (input, init) {
        const frp = sanitize({
            retries: init === null || init === void 0 ? void 0 : init.retries,
            retryDelay: init === null || init === void 0 ? void 0 : init.retryDelay,
            retryOn: init === null || init === void 0 ? void 0 : init.retryOn,
        }, defaults);
        const retryDelayFn = typeof frp.retryDelay === 'function' ? frp.retryDelay : () => frp.retryDelay;
        const retryOnFn = typeof frp.retryOn === 'function'
            ? frp.retryOn
            : (attempt, retries, error, response) => (!!error || !response || frp.retryOn.indexOf(response.status) !== -1) &&
                attempt < retries;
        return new Promise(function (resolve, reject) {
            const extendedFetch = function (attempt) {
                fetchFunc(input, init)
                    .then(function (response) {
                    if (retryOnFn(attempt, frp.retries, null, response)) {
                        // eslint-disable-next-line @typescript-eslint/no-use-before-define
                        retry(attempt, null, response);
                    }
                    else {
                        resolve(response);
                    }
                })
                    .catch(function (error) {
                    if (retryOnFn(attempt, frp.retries, error, null)) {
                        // eslint-disable-next-line @typescript-eslint/no-use-before-define
                        retry(attempt, error, null);
                    }
                    else {
                        reject(error);
                    }
                });
            };
            function retry(attempt, error, response) {
                setTimeout(function () {
                    extendedFetch(++attempt);
                }, retryDelayFn(attempt, error, response));
            }
            extendedFetch(0);
        });
    };
}
exports.fetchBuilder = fetchBuilder;
exports.default = fetchBuilder;
