(function polyfill() {
  const relList = document.createElement("link").relList;
  if (relList && relList.supports && relList.supports("modulepreload")) {
    return;
  }
  for (const link3 of document.querySelectorAll('link[rel="modulepreload"]')) {
    processPreload(link3);
  }
  new MutationObserver((mutations) => {
    for (const mutation of mutations) {
      if (mutation.type !== "childList") {
        continue;
      }
      for (const node of mutation.addedNodes) {
        if (node.tagName === "LINK" && node.rel === "modulepreload")
          processPreload(node);
      }
    }
  }).observe(document, { childList: true, subtree: true });
  function getFetchOpts(link3) {
    const fetchOpts = {};
    if (link3.integrity) fetchOpts.integrity = link3.integrity;
    if (link3.referrerPolicy) fetchOpts.referrerPolicy = link3.referrerPolicy;
    if (link3.crossOrigin === "use-credentials")
      fetchOpts.credentials = "include";
    else if (link3.crossOrigin === "anonymous") fetchOpts.credentials = "omit";
    else fetchOpts.credentials = "same-origin";
    return fetchOpts;
  }
  function processPreload(link3) {
    if (link3.ep)
      return;
    link3.ep = true;
    const fetchOpts = getFetchOpts(link3);
    fetch(link3.href, fetchOpts);
  }
})();
function makeMap(str, expectsLowerCase) {
  const map2 = /* @__PURE__ */ Object.create(null);
  const list2 = str.split(",");
  for (let i2 = 0; i2 < list2.length; i2++) {
    map2[list2[i2]] = true;
  }
  return expectsLowerCase ? (val) => !!map2[val.toLowerCase()] : (val) => !!map2[val];
}
const PatchFlagNames = {
  [
    1
    /* PatchFlags.TEXT */
  ]: `TEXT`,
  [
    2
    /* PatchFlags.CLASS */
  ]: `CLASS`,
  [
    4
    /* PatchFlags.STYLE */
  ]: `STYLE`,
  [
    8
    /* PatchFlags.PROPS */
  ]: `PROPS`,
  [
    16
    /* PatchFlags.FULL_PROPS */
  ]: `FULL_PROPS`,
  [
    32
    /* PatchFlags.HYDRATE_EVENTS */
  ]: `HYDRATE_EVENTS`,
  [
    64
    /* PatchFlags.STABLE_FRAGMENT */
  ]: `STABLE_FRAGMENT`,
  [
    128
    /* PatchFlags.KEYED_FRAGMENT */
  ]: `KEYED_FRAGMENT`,
  [
    256
    /* PatchFlags.UNKEYED_FRAGMENT */
  ]: `UNKEYED_FRAGMENT`,
  [
    512
    /* PatchFlags.NEED_PATCH */
  ]: `NEED_PATCH`,
  [
    1024
    /* PatchFlags.DYNAMIC_SLOTS */
  ]: `DYNAMIC_SLOTS`,
  [
    2048
    /* PatchFlags.DEV_ROOT_FRAGMENT */
  ]: `DEV_ROOT_FRAGMENT`,
  [
    -1
    /* PatchFlags.HOISTED */
  ]: `HOISTED`,
  [
    -2
    /* PatchFlags.BAIL */
  ]: `BAIL`
};
const slotFlagsText = {
  [
    1
    /* SlotFlags.STABLE */
  ]: "STABLE",
  [
    2
    /* SlotFlags.DYNAMIC */
  ]: "DYNAMIC",
  [
    3
    /* SlotFlags.FORWARDED */
  ]: "FORWARDED"
};
const GLOBALS_WHITE_LISTED = "Infinity,undefined,NaN,isFinite,isNaN,parseFloat,parseInt,decodeURI,decodeURIComponent,encodeURI,encodeURIComponent,Math,Number,Date,Array,Object,Boolean,String,RegExp,Map,Set,JSON,Intl,BigInt";
const isGloballyWhitelisted = /* @__PURE__ */ makeMap(GLOBALS_WHITE_LISTED);
const range$1 = 2;
function generateCodeFrame(source, start = 0, end = source.length) {
  let lines = source.split(/(\r?\n)/);
  const newlineSequences = lines.filter((_2, idx) => idx % 2 === 1);
  lines = lines.filter((_2, idx) => idx % 2 === 0);
  let count = 0;
  const res = [];
  for (let i2 = 0; i2 < lines.length; i2++) {
    count += lines[i2].length + (newlineSequences[i2] && newlineSequences[i2].length || 0);
    if (count >= start) {
      for (let j2 = i2 - range$1; j2 <= i2 + range$1 || end > count; j2++) {
        if (j2 < 0 || j2 >= lines.length)
          continue;
        const line = j2 + 1;
        res.push(`${line}${" ".repeat(Math.max(3 - String(line).length, 0))}|  ${lines[j2]}`);
        const lineLength = lines[j2].length;
        const newLineSeqLength = newlineSequences[j2] && newlineSequences[j2].length || 0;
        if (j2 === i2) {
          const pad = start - (count - (lineLength + newLineSeqLength));
          const length = Math.max(1, end > count ? lineLength - pad : end - start);
          res.push(`   |  ` + " ".repeat(pad) + "^".repeat(length));
        } else if (j2 > i2) {
          if (end > count) {
            const length = Math.max(Math.min(end - count, lineLength), 1);
            res.push(`   |  ` + "^".repeat(length));
          }
          count += lineLength + newLineSeqLength;
        }
      }
      break;
    }
  }
  return res.join("\n");
}
function normalizeStyle(value) {
  if (isArray$2(value)) {
    const res = {};
    for (let i2 = 0; i2 < value.length; i2++) {
      const item = value[i2];
      const normalized = isString$4(item) ? parseStringStyle(item) : normalizeStyle(item);
      if (normalized) {
        for (const key in normalized) {
          res[key] = normalized[key];
        }
      }
    }
    return res;
  } else if (isString$4(value)) {
    return value;
  } else if (isObject$5(value)) {
    return value;
  }
}
const listDelimiterRE = /;(?![^(]*\))/g;
const propertyDelimiterRE = /:([^]+)/;
const styleCommentRE = /\/\*.*?\*\//gs;
function parseStringStyle(cssText) {
  const ret = {};
  cssText.replace(styleCommentRE, "").split(listDelimiterRE).forEach((item) => {
    if (item) {
      const tmp = item.split(propertyDelimiterRE);
      tmp.length > 1 && (ret[tmp[0].trim()] = tmp[1].trim());
    }
  });
  return ret;
}
function stringifyStyle(styles) {
  let ret = "";
  if (!styles || isString$4(styles)) {
    return ret;
  }
  for (const key in styles) {
    const value = styles[key];
    const normalizedKey = key.startsWith(`--`) ? key : hyphenate(key);
    if (isString$4(value) || typeof value === "number") {
      ret += `${normalizedKey}:${value};`;
    }
  }
  return ret;
}
function normalizeClass(value) {
  let res = "";
  if (isString$4(value)) {
    res = value;
  } else if (isArray$2(value)) {
    for (let i2 = 0; i2 < value.length; i2++) {
      const normalized = normalizeClass(value[i2]);
      if (normalized) {
        res += normalized + " ";
      }
    }
  } else if (isObject$5(value)) {
    for (const name2 in value) {
      if (value[name2]) {
        res += name2 + " ";
      }
    }
  }
  return res.trim();
}
function normalizeProps(props) {
  if (!props)
    return null;
  let { class: klass, style } = props;
  if (klass && !isString$4(klass)) {
    props.class = normalizeClass(klass);
  }
  if (style) {
    props.style = normalizeStyle(style);
  }
  return props;
}
const HTML_TAGS = "html,body,base,head,link,meta,style,title,address,article,aside,footer,header,hgroup,h1,h2,h3,h4,h5,h6,nav,section,div,dd,dl,dt,figcaption,figure,picture,hr,img,li,main,ol,p,pre,ul,a,b,abbr,bdi,bdo,br,cite,code,data,dfn,em,i,kbd,mark,q,rp,rt,ruby,s,samp,small,span,strong,sub,sup,time,u,var,wbr,area,audio,map,track,video,embed,object,param,source,canvas,script,noscript,del,ins,caption,col,colgroup,table,thead,tbody,td,th,tr,button,datalist,fieldset,form,input,label,legend,meter,optgroup,option,output,progress,select,textarea,details,dialog,menu,summary,template,blockquote,iframe,tfoot";
const SVG_TAGS = "svg,animate,animateMotion,animateTransform,circle,clipPath,color-profile,defs,desc,discard,ellipse,feBlend,feColorMatrix,feComponentTransfer,feComposite,feConvolveMatrix,feDiffuseLighting,feDisplacementMap,feDistantLight,feDropShadow,feFlood,feFuncA,feFuncB,feFuncG,feFuncR,feGaussianBlur,feImage,feMerge,feMergeNode,feMorphology,feOffset,fePointLight,feSpecularLighting,feSpotLight,feTile,feTurbulence,filter,foreignObject,g,hatch,hatchpath,image,line,linearGradient,marker,mask,mesh,meshgradient,meshpatch,meshrow,metadata,mpath,path,pattern,polygon,polyline,radialGradient,rect,set,solidcolor,stop,switch,symbol,text,textPath,title,tspan,unknown,use,view";
const VOID_TAGS = "area,base,br,col,embed,hr,img,input,link,meta,param,source,track,wbr";
const isHTMLTag = /* @__PURE__ */ makeMap(HTML_TAGS);
const isSVGTag = /* @__PURE__ */ makeMap(SVG_TAGS);
const isVoidTag = /* @__PURE__ */ makeMap(VOID_TAGS);
const specialBooleanAttrs = `itemscope,allowfullscreen,formnovalidate,ismap,nomodule,novalidate,readonly`;
const isSpecialBooleanAttr = /* @__PURE__ */ makeMap(specialBooleanAttrs);
const isBooleanAttr = /* @__PURE__ */ makeMap(specialBooleanAttrs + `,async,autofocus,autoplay,controls,default,defer,disabled,hidden,loop,open,required,reversed,scoped,seamless,checked,muted,multiple,selected`);
function includeBooleanAttr(value) {
  return !!value || value === "";
}
const unsafeAttrCharRE = /[>/="'\u0009\u000a\u000c\u0020]/;
const attrValidationCache = {};
function isSSRSafeAttrName(name2) {
  if (attrValidationCache.hasOwnProperty(name2)) {
    return attrValidationCache[name2];
  }
  const isUnsafe = unsafeAttrCharRE.test(name2);
  if (isUnsafe) {
    console.error(`unsafe attribute name: ${name2}`);
  }
  return attrValidationCache[name2] = !isUnsafe;
}
const propsToAttrMap = {
  acceptCharset: "accept-charset",
  className: "class",
  htmlFor: "for",
  httpEquiv: "http-equiv"
};
const isKnownHtmlAttr = /* @__PURE__ */ makeMap(`accept,accept-charset,accesskey,action,align,allow,alt,async,autocapitalize,autocomplete,autofocus,autoplay,background,bgcolor,border,buffered,capture,challenge,charset,checked,cite,class,code,codebase,color,cols,colspan,content,contenteditable,contextmenu,controls,coords,crossorigin,csp,data,datetime,decoding,default,defer,dir,dirname,disabled,download,draggable,dropzone,enctype,enterkeyhint,for,form,formaction,formenctype,formmethod,formnovalidate,formtarget,headers,height,hidden,high,href,hreflang,http-equiv,icon,id,importance,integrity,ismap,itemprop,keytype,kind,label,lang,language,loading,list,loop,low,manifest,max,maxlength,minlength,media,min,multiple,muted,name,novalidate,open,optimum,pattern,ping,placeholder,poster,preload,radiogroup,readonly,referrerpolicy,rel,required,reversed,rows,rowspan,sandbox,scope,scoped,selected,shape,size,sizes,slot,span,spellcheck,src,srcdoc,srclang,srcset,start,step,style,summary,tabindex,target,title,translate,type,usemap,value,width,wrap`);
const isKnownSvgAttr = /* @__PURE__ */ makeMap(`xmlns,accent-height,accumulate,additive,alignment-baseline,alphabetic,amplitude,arabic-form,ascent,attributeName,attributeType,azimuth,baseFrequency,baseline-shift,baseProfile,bbox,begin,bias,by,calcMode,cap-height,class,clip,clipPathUnits,clip-path,clip-rule,color,color-interpolation,color-interpolation-filters,color-profile,color-rendering,contentScriptType,contentStyleType,crossorigin,cursor,cx,cy,d,decelerate,descent,diffuseConstant,direction,display,divisor,dominant-baseline,dur,dx,dy,edgeMode,elevation,enable-background,end,exponent,fill,fill-opacity,fill-rule,filter,filterRes,filterUnits,flood-color,flood-opacity,font-family,font-size,font-size-adjust,font-stretch,font-style,font-variant,font-weight,format,from,fr,fx,fy,g1,g2,glyph-name,glyph-orientation-horizontal,glyph-orientation-vertical,glyphRef,gradientTransform,gradientUnits,hanging,height,href,hreflang,horiz-adv-x,horiz-origin-x,id,ideographic,image-rendering,in,in2,intercept,k,k1,k2,k3,k4,kernelMatrix,kernelUnitLength,kerning,keyPoints,keySplines,keyTimes,lang,lengthAdjust,letter-spacing,lighting-color,limitingConeAngle,local,marker-end,marker-mid,marker-start,markerHeight,markerUnits,markerWidth,mask,maskContentUnits,maskUnits,mathematical,max,media,method,min,mode,name,numOctaves,offset,opacity,operator,order,orient,orientation,origin,overflow,overline-position,overline-thickness,panose-1,paint-order,path,pathLength,patternContentUnits,patternTransform,patternUnits,ping,pointer-events,points,pointsAtX,pointsAtY,pointsAtZ,preserveAlpha,preserveAspectRatio,primitiveUnits,r,radius,referrerPolicy,refX,refY,rel,rendering-intent,repeatCount,repeatDur,requiredExtensions,requiredFeatures,restart,result,rotate,rx,ry,scale,seed,shape-rendering,slope,spacing,specularConstant,specularExponent,speed,spreadMethod,startOffset,stdDeviation,stemh,stemv,stitchTiles,stop-color,stop-opacity,strikethrough-position,strikethrough-thickness,string,stroke,stroke-dasharray,stroke-dashoffset,stroke-linecap,stroke-linejoin,stroke-miterlimit,stroke-opacity,stroke-width,style,surfaceScale,systemLanguage,tabindex,tableValues,target,targetX,targetY,text-anchor,text-decoration,text-rendering,textLength,to,transform,transform-origin,type,u1,u2,underline-position,underline-thickness,unicode,unicode-bidi,unicode-range,units-per-em,v-alphabetic,v-hanging,v-ideographic,v-mathematical,values,vector-effect,version,vert-adv-y,vert-origin-x,vert-origin-y,viewBox,viewTarget,visibility,width,widths,word-spacing,writing-mode,x,x-height,x1,x2,xChannelSelector,xlink:actuate,xlink:arcrole,xlink:href,xlink:role,xlink:show,xlink:title,xlink:type,xml:base,xml:lang,xml:space,y,y1,y2,yChannelSelector,z,zoomAndPan`);
const escapeRE = /["'&<>]/;
function escapeHtml$2(string) {
  const str = "" + string;
  const match2 = escapeRE.exec(str);
  if (!match2) {
    return str;
  }
  let html = "";
  let escaped;
  let index;
  let lastIndex = 0;
  for (index = match2.index; index < str.length; index++) {
    switch (str.charCodeAt(index)) {
      case 34:
        escaped = "&quot;";
        break;
      case 38:
        escaped = "&amp;";
        break;
      case 39:
        escaped = "&#39;";
        break;
      case 60:
        escaped = "&lt;";
        break;
      case 62:
        escaped = "&gt;";
        break;
      default:
        continue;
    }
    if (lastIndex !== index) {
      html += str.slice(lastIndex, index);
    }
    lastIndex = index + 1;
    html += escaped;
  }
  return lastIndex !== index ? html + str.slice(lastIndex, index) : html;
}
const commentStripRE = /^-?>|<!--|-->|--!>|<!-$/g;
function escapeHtmlComment(src) {
  return src.replace(commentStripRE, "");
}
function looseCompareArrays(a2, b2) {
  if (a2.length !== b2.length)
    return false;
  let equal = true;
  for (let i2 = 0; equal && i2 < a2.length; i2++) {
    equal = looseEqual(a2[i2], b2[i2]);
  }
  return equal;
}
function looseEqual(a2, b2) {
  if (a2 === b2)
    return true;
  let aValidType = isDate$1(a2);
  let bValidType = isDate$1(b2);
  if (aValidType || bValidType) {
    return aValidType && bValidType ? a2.getTime() === b2.getTime() : false;
  }
  aValidType = isSymbol$1(a2);
  bValidType = isSymbol$1(b2);
  if (aValidType || bValidType) {
    return a2 === b2;
  }
  aValidType = isArray$2(a2);
  bValidType = isArray$2(b2);
  if (aValidType || bValidType) {
    return aValidType && bValidType ? looseCompareArrays(a2, b2) : false;
  }
  aValidType = isObject$5(a2);
  bValidType = isObject$5(b2);
  if (aValidType || bValidType) {
    if (!aValidType || !bValidType) {
      return false;
    }
    const aKeysCount = Object.keys(a2).length;
    const bKeysCount = Object.keys(b2).length;
    if (aKeysCount !== bKeysCount) {
      return false;
    }
    for (const key in a2) {
      const aHasKey = a2.hasOwnProperty(key);
      const bHasKey = b2.hasOwnProperty(key);
      if (aHasKey && !bHasKey || !aHasKey && bHasKey || !looseEqual(a2[key], b2[key])) {
        return false;
      }
    }
  }
  return String(a2) === String(b2);
}
function looseIndexOf(arr, val) {
  return arr.findIndex((item) => looseEqual(item, val));
}
const toDisplayString$1 = (val) => {
  return isString$4(val) ? val : val == null ? "" : isArray$2(val) || isObject$5(val) && (val.toString === objectToString$2 || !isFunction$2(val.toString)) ? JSON.stringify(val, replacer, 2) : String(val);
};
const replacer = (_key, val) => {
  if (val && val.__v_isRef) {
    return replacer(_key, val.value);
  } else if (isMap(val)) {
    return {
      [`Map(${val.size})`]: [...val.entries()].reduce((entries, [key, val2]) => {
        entries[`${key} =>`] = val2;
        return entries;
      }, {})
    };
  } else if (isSet(val)) {
    return {
      [`Set(${val.size})`]: [...val.values()]
    };
  } else if (isObject$5(val) && !isArray$2(val) && !isPlainObject$2(val)) {
    return String(val);
  }
  return val;
};
const EMPTY_OBJ = {};
const EMPTY_ARR = [];
const NOOP = () => {
};
const NO = () => false;
const onRE = /^on[^a-z]/;
const isOn = (key) => onRE.test(key);
const isModelListener = (key) => key.startsWith("onUpdate:");
const extend = Object.assign;
const remove = (arr, el2) => {
  const i2 = arr.indexOf(el2);
  if (i2 > -1) {
    arr.splice(i2, 1);
  }
};
const hasOwnProperty$4 = Object.prototype.hasOwnProperty;
const hasOwn$1 = (val, key) => hasOwnProperty$4.call(val, key);
const isArray$2 = Array.isArray;
const isMap = (val) => toTypeString$1(val) === "[object Map]";
const isSet = (val) => toTypeString$1(val) === "[object Set]";
const isDate$1 = (val) => toTypeString$1(val) === "[object Date]";
const isRegExp$1 = (val) => toTypeString$1(val) === "[object RegExp]";
const isFunction$2 = (val) => typeof val === "function";
const isString$4 = (val) => typeof val === "string";
const isSymbol$1 = (val) => typeof val === "symbol";
const isObject$5 = (val) => val !== null && typeof val === "object";
const isPromise$1 = (val) => {
  return isObject$5(val) && isFunction$2(val.then) && isFunction$2(val.catch);
};
const objectToString$2 = Object.prototype.toString;
const toTypeString$1 = (value) => objectToString$2.call(value);
const toRawType = (value) => {
  return toTypeString$1(value).slice(8, -1);
};
const isPlainObject$2 = (val) => toTypeString$1(val) === "[object Object]";
const isIntegerKey = (key) => isString$4(key) && key !== "NaN" && key[0] !== "-" && "" + parseInt(key, 10) === key;
const isReservedProp = /* @__PURE__ */ makeMap(
  // the leading comma is intentional so empty string "" is also included
  ",key,ref,ref_for,ref_key,onVnodeBeforeMount,onVnodeMounted,onVnodeBeforeUpdate,onVnodeUpdated,onVnodeBeforeUnmount,onVnodeUnmounted"
);
const isBuiltInDirective = /* @__PURE__ */ makeMap("bind,cloak,else-if,else,for,html,if,model,on,once,pre,show,slot,text,memo");
const cacheStringFunction$1 = (fn) => {
  const cache2 = /* @__PURE__ */ Object.create(null);
  return ((str) => {
    const hit = cache2[str];
    return hit || (cache2[str] = fn(str));
  });
};
const camelizeRE$1 = /-(\w)/g;
const camelize$1 = cacheStringFunction$1((str) => {
  return str.replace(camelizeRE$1, (_2, c2) => c2 ? c2.toUpperCase() : "");
});
const hyphenateRE = /\B([A-Z])/g;
const hyphenate = cacheStringFunction$1((str) => str.replace(hyphenateRE, "-$1").toLowerCase());
const capitalize$2 = cacheStringFunction$1((str) => str.charAt(0).toUpperCase() + str.slice(1));
const toHandlerKey = cacheStringFunction$1((str) => str ? `on${capitalize$2(str)}` : ``);
const hasChanged = (value, oldValue) => !Object.is(value, oldValue);
const invokeArrayFns = (fns, arg) => {
  for (let i2 = 0; i2 < fns.length; i2++) {
    fns[i2](arg);
  }
};
const def = (obj, key, value) => {
  Object.defineProperty(obj, key, {
    configurable: true,
    enumerable: false,
    value
  });
};
const looseToNumber = (val) => {
  const n2 = parseFloat(val);
  return isNaN(n2) ? val : n2;
};
const toNumber = (val) => {
  const n2 = isString$4(val) ? Number(val) : NaN;
  return isNaN(n2) ? val : n2;
};
let _globalThis$1;
const getGlobalThis$1 = () => {
  return _globalThis$1 || (_globalThis$1 = typeof globalThis !== "undefined" ? globalThis : typeof self !== "undefined" ? self : typeof window !== "undefined" ? window : typeof global !== "undefined" ? global : {});
};
const identRE = /^[_$a-zA-Z\xA0-\uFFFF][_$a-zA-Z0-9\xA0-\uFFFF]*$/;
function genPropsAccessExp(name2) {
  return identRE.test(name2) ? `__props.${name2}` : `__props[${JSON.stringify(name2)}]`;
}
const shared_esmBundler = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  EMPTY_ARR,
  EMPTY_OBJ,
  NO,
  NOOP,
  PatchFlagNames,
  camelize: camelize$1,
  capitalize: capitalize$2,
  def,
  escapeHtml: escapeHtml$2,
  escapeHtmlComment,
  extend,
  genPropsAccessExp,
  generateCodeFrame,
  getGlobalThis: getGlobalThis$1,
  hasChanged,
  hasOwn: hasOwn$1,
  hyphenate,
  includeBooleanAttr,
  invokeArrayFns,
  isArray: isArray$2,
  isBooleanAttr,
  isBuiltInDirective,
  isDate: isDate$1,
  isFunction: isFunction$2,
  isGloballyWhitelisted,
  isHTMLTag,
  isIntegerKey,
  isKnownHtmlAttr,
  isKnownSvgAttr,
  isMap,
  isModelListener,
  isObject: isObject$5,
  isOn,
  isPlainObject: isPlainObject$2,
  isPromise: isPromise$1,
  isRegExp: isRegExp$1,
  isReservedProp,
  isSSRSafeAttrName,
  isSVGTag,
  isSet,
  isSpecialBooleanAttr,
  isString: isString$4,
  isSymbol: isSymbol$1,
  isVoidTag,
  looseEqual,
  looseIndexOf,
  looseToNumber,
  makeMap,
  normalizeClass,
  normalizeProps,
  normalizeStyle,
  objectToString: objectToString$2,
  parseStringStyle,
  propsToAttrMap,
  remove,
  slotFlagsText,
  stringifyStyle,
  toDisplayString: toDisplayString$1,
  toHandlerKey,
  toNumber,
  toRawType,
  toTypeString: toTypeString$1
}, Symbol.toStringTag, { value: "Module" }));
let activeEffectScope;
class EffectScope {
  constructor(detached = false) {
    this.detached = detached;
    this._active = true;
    this.effects = [];
    this.cleanups = [];
    this.parent = activeEffectScope;
    if (!detached && activeEffectScope) {
      this.index = (activeEffectScope.scopes || (activeEffectScope.scopes = [])).push(this) - 1;
    }
  }
  get active() {
    return this._active;
  }
  run(fn) {
    if (this._active) {
      const currentEffectScope = activeEffectScope;
      try {
        activeEffectScope = this;
        return fn();
      } finally {
        activeEffectScope = currentEffectScope;
      }
    }
  }
  /**
   * This should only be called on non-detached scopes
   * @internal
   */
  on() {
    activeEffectScope = this;
  }
  /**
   * This should only be called on non-detached scopes
   * @internal
   */
  off() {
    activeEffectScope = this.parent;
  }
  stop(fromParent) {
    if (this._active) {
      let i2, l2;
      for (i2 = 0, l2 = this.effects.length; i2 < l2; i2++) {
        this.effects[i2].stop();
      }
      for (i2 = 0, l2 = this.cleanups.length; i2 < l2; i2++) {
        this.cleanups[i2]();
      }
      if (this.scopes) {
        for (i2 = 0, l2 = this.scopes.length; i2 < l2; i2++) {
          this.scopes[i2].stop(true);
        }
      }
      if (!this.detached && this.parent && !fromParent) {
        const last2 = this.parent.scopes.pop();
        if (last2 && last2 !== this) {
          this.parent.scopes[this.index] = last2;
          last2.index = this.index;
        }
      }
      this.parent = void 0;
      this._active = false;
    }
  }
}
function effectScope(detached) {
  return new EffectScope(detached);
}
function recordEffectScope(effect2, scope = activeEffectScope) {
  if (scope && scope.active) {
    scope.effects.push(effect2);
  }
}
function getCurrentScope() {
  return activeEffectScope;
}
function onScopeDispose(fn) {
  if (activeEffectScope) {
    activeEffectScope.cleanups.push(fn);
  }
}
const createDep = (effects) => {
  const dep = new Set(effects);
  dep.w = 0;
  dep.n = 0;
  return dep;
};
const wasTracked = (dep) => (dep.w & trackOpBit) > 0;
const newTracked = (dep) => (dep.n & trackOpBit) > 0;
const initDepMarkers = ({ deps }) => {
  if (deps.length) {
    for (let i2 = 0; i2 < deps.length; i2++) {
      deps[i2].w |= trackOpBit;
    }
  }
};
const finalizeDepMarkers = (effect2) => {
  const { deps } = effect2;
  if (deps.length) {
    let ptr = 0;
    for (let i2 = 0; i2 < deps.length; i2++) {
      const dep = deps[i2];
      if (wasTracked(dep) && !newTracked(dep)) {
        dep.delete(effect2);
      } else {
        deps[ptr++] = dep;
      }
      dep.w &= ~trackOpBit;
      dep.n &= ~trackOpBit;
    }
    deps.length = ptr;
  }
};
const targetMap = /* @__PURE__ */ new WeakMap();
let effectTrackDepth = 0;
let trackOpBit = 1;
const maxMarkerBits = 30;
let activeEffect;
const ITERATE_KEY = Symbol("");
const MAP_KEY_ITERATE_KEY = Symbol("");
class ReactiveEffect {
  constructor(fn, scheduler2 = null, scope) {
    this.fn = fn;
    this.scheduler = scheduler2;
    this.active = true;
    this.deps = [];
    this.parent = void 0;
    recordEffectScope(this, scope);
  }
  run() {
    if (!this.active) {
      return this.fn();
    }
    let parent = activeEffect;
    let lastShouldTrack = shouldTrack;
    while (parent) {
      if (parent === this) {
        return;
      }
      parent = parent.parent;
    }
    try {
      this.parent = activeEffect;
      activeEffect = this;
      shouldTrack = true;
      trackOpBit = 1 << ++effectTrackDepth;
      if (effectTrackDepth <= maxMarkerBits) {
        initDepMarkers(this);
      } else {
        cleanupEffect(this);
      }
      return this.fn();
    } finally {
      if (effectTrackDepth <= maxMarkerBits) {
        finalizeDepMarkers(this);
      }
      trackOpBit = 1 << --effectTrackDepth;
      activeEffect = this.parent;
      shouldTrack = lastShouldTrack;
      this.parent = void 0;
      if (this.deferStop) {
        this.stop();
      }
    }
  }
  stop() {
    if (activeEffect === this) {
      this.deferStop = true;
    } else if (this.active) {
      cleanupEffect(this);
      if (this.onStop) {
        this.onStop();
      }
      this.active = false;
    }
  }
}
function cleanupEffect(effect2) {
  const { deps } = effect2;
  if (deps.length) {
    for (let i2 = 0; i2 < deps.length; i2++) {
      deps[i2].delete(effect2);
    }
    deps.length = 0;
  }
}
function effect(fn, options2) {
  if (fn.effect) {
    fn = fn.effect.fn;
  }
  const _effect = new ReactiveEffect(fn);
  if (options2) {
    extend(_effect, options2);
    if (options2.scope)
      recordEffectScope(_effect, options2.scope);
  }
  if (!options2 || !options2.lazy) {
    _effect.run();
  }
  const runner = _effect.run.bind(_effect);
  runner.effect = _effect;
  return runner;
}
function stop(runner) {
  runner.effect.stop();
}
let shouldTrack = true;
const trackStack = [];
function pauseTracking() {
  trackStack.push(shouldTrack);
  shouldTrack = false;
}
function resetTracking() {
  const last2 = trackStack.pop();
  shouldTrack = last2 === void 0 ? true : last2;
}
function track(target2, type2, key) {
  if (shouldTrack && activeEffect) {
    let depsMap = targetMap.get(target2);
    if (!depsMap) {
      targetMap.set(target2, depsMap = /* @__PURE__ */ new Map());
    }
    let dep = depsMap.get(key);
    if (!dep) {
      depsMap.set(key, dep = createDep());
    }
    trackEffects(dep);
  }
}
function trackEffects(dep, debuggerEventExtraInfo) {
  let shouldTrack2 = false;
  if (effectTrackDepth <= maxMarkerBits) {
    if (!newTracked(dep)) {
      dep.n |= trackOpBit;
      shouldTrack2 = !wasTracked(dep);
    }
  } else {
    shouldTrack2 = !dep.has(activeEffect);
  }
  if (shouldTrack2) {
    dep.add(activeEffect);
    activeEffect.deps.push(dep);
  }
}
function trigger(target2, type2, key, newValue, oldValue, oldTarget) {
  const depsMap = targetMap.get(target2);
  if (!depsMap) {
    return;
  }
  let deps = [];
  if (type2 === "clear") {
    deps = [...depsMap.values()];
  } else if (key === "length" && isArray$2(target2)) {
    const newLength = Number(newValue);
    depsMap.forEach((dep, key2) => {
      if (key2 === "length" || key2 >= newLength) {
        deps.push(dep);
      }
    });
  } else {
    if (key !== void 0) {
      deps.push(depsMap.get(key));
    }
    switch (type2) {
      case "add":
        if (!isArray$2(target2)) {
          deps.push(depsMap.get(ITERATE_KEY));
          if (isMap(target2)) {
            deps.push(depsMap.get(MAP_KEY_ITERATE_KEY));
          }
        } else if (isIntegerKey(key)) {
          deps.push(depsMap.get("length"));
        }
        break;
      case "delete":
        if (!isArray$2(target2)) {
          deps.push(depsMap.get(ITERATE_KEY));
          if (isMap(target2)) {
            deps.push(depsMap.get(MAP_KEY_ITERATE_KEY));
          }
        }
        break;
      case "set":
        if (isMap(target2)) {
          deps.push(depsMap.get(ITERATE_KEY));
        }
        break;
    }
  }
  if (deps.length === 1) {
    if (deps[0]) {
      {
        triggerEffects(deps[0]);
      }
    }
  } else {
    const effects = [];
    for (const dep of deps) {
      if (dep) {
        effects.push(...dep);
      }
    }
    {
      triggerEffects(createDep(effects));
    }
  }
}
function triggerEffects(dep, debuggerEventExtraInfo) {
  const effects = isArray$2(dep) ? dep : [...dep];
  for (const effect2 of effects) {
    if (effect2.computed) {
      triggerEffect(effect2);
    }
  }
  for (const effect2 of effects) {
    if (!effect2.computed) {
      triggerEffect(effect2);
    }
  }
}
function triggerEffect(effect2, debuggerEventExtraInfo) {
  if (effect2 !== activeEffect || effect2.allowRecurse) {
    if (effect2.scheduler) {
      effect2.scheduler();
    } else {
      effect2.run();
    }
  }
}
function getDepFromReactive(object, key) {
  var _a2;
  return (_a2 = targetMap.get(object)) === null || _a2 === void 0 ? void 0 : _a2.get(key);
}
const isNonTrackableKeys = /* @__PURE__ */ makeMap(`__proto__,__v_isRef,__isVue`);
const builtInSymbols = new Set(
  /* @__PURE__ */ Object.getOwnPropertyNames(Symbol).filter((key) => key !== "arguments" && key !== "caller").map((key) => Symbol[key]).filter(isSymbol$1)
);
const get$1 = /* @__PURE__ */ createGetter();
const shallowGet = /* @__PURE__ */ createGetter(false, true);
const readonlyGet = /* @__PURE__ */ createGetter(true);
const shallowReadonlyGet = /* @__PURE__ */ createGetter(true, true);
const arrayInstrumentations = /* @__PURE__ */ createArrayInstrumentations();
function createArrayInstrumentations() {
  const instrumentations = {};
  ["includes", "indexOf", "lastIndexOf"].forEach((key) => {
    instrumentations[key] = function(...args) {
      const arr = toRaw(this);
      for (let i2 = 0, l2 = this.length; i2 < l2; i2++) {
        track(arr, "get", i2 + "");
      }
      const res = arr[key](...args);
      if (res === -1 || res === false) {
        return arr[key](...args.map(toRaw));
      } else {
        return res;
      }
    };
  });
  ["push", "pop", "shift", "unshift", "splice"].forEach((key) => {
    instrumentations[key] = function(...args) {
      pauseTracking();
      const res = toRaw(this)[key].apply(this, args);
      resetTracking();
      return res;
    };
  });
  return instrumentations;
}
function hasOwnProperty$3(key) {
  const obj = toRaw(this);
  track(obj, "has", key);
  return obj.hasOwnProperty(key);
}
function createGetter(isReadonly2 = false, shallow = false) {
  return function get2(target2, key, receiver) {
    if (key === "__v_isReactive") {
      return !isReadonly2;
    } else if (key === "__v_isReadonly") {
      return isReadonly2;
    } else if (key === "__v_isShallow") {
      return shallow;
    } else if (key === "__v_raw" && receiver === (isReadonly2 ? shallow ? shallowReadonlyMap : readonlyMap : shallow ? shallowReactiveMap : reactiveMap).get(target2)) {
      return target2;
    }
    const targetIsArray = isArray$2(target2);
    if (!isReadonly2) {
      if (targetIsArray && hasOwn$1(arrayInstrumentations, key)) {
        return Reflect.get(arrayInstrumentations, key, receiver);
      }
      if (key === "hasOwnProperty") {
        return hasOwnProperty$3;
      }
    }
    const res = Reflect.get(target2, key, receiver);
    if (isSymbol$1(key) ? builtInSymbols.has(key) : isNonTrackableKeys(key)) {
      return res;
    }
    if (!isReadonly2) {
      track(target2, "get", key);
    }
    if (shallow) {
      return res;
    }
    if (isRef(res)) {
      return targetIsArray && isIntegerKey(key) ? res : res.value;
    }
    if (isObject$5(res)) {
      return isReadonly2 ? readonly(res) : reactive(res);
    }
    return res;
  };
}
const set$1 = /* @__PURE__ */ createSetter();
const shallowSet = /* @__PURE__ */ createSetter(true);
function createSetter(shallow = false) {
  return function set2(target2, key, value, receiver) {
    let oldValue = target2[key];
    if (isReadonly(oldValue) && isRef(oldValue) && !isRef(value)) {
      return false;
    }
    if (!shallow) {
      if (!isShallow(value) && !isReadonly(value)) {
        oldValue = toRaw(oldValue);
        value = toRaw(value);
      }
      if (!isArray$2(target2) && isRef(oldValue) && !isRef(value)) {
        oldValue.value = value;
        return true;
      }
    }
    const hadKey = isArray$2(target2) && isIntegerKey(key) ? Number(key) < target2.length : hasOwn$1(target2, key);
    const result = Reflect.set(target2, key, value, receiver);
    if (target2 === toRaw(receiver)) {
      if (!hadKey) {
        trigger(target2, "add", key, value);
      } else if (hasChanged(value, oldValue)) {
        trigger(target2, "set", key, value);
      }
    }
    return result;
  };
}
function deleteProperty(target2, key) {
  const hadKey = hasOwn$1(target2, key);
  target2[key];
  const result = Reflect.deleteProperty(target2, key);
  if (result && hadKey) {
    trigger(target2, "delete", key, void 0);
  }
  return result;
}
function has$1(target2, key) {
  const result = Reflect.has(target2, key);
  if (!isSymbol$1(key) || !builtInSymbols.has(key)) {
    track(target2, "has", key);
  }
  return result;
}
function ownKeys(target2) {
  track(target2, "iterate", isArray$2(target2) ? "length" : ITERATE_KEY);
  return Reflect.ownKeys(target2);
}
const mutableHandlers = {
  get: get$1,
  set: set$1,
  deleteProperty,
  has: has$1,
  ownKeys
};
const readonlyHandlers = {
  get: readonlyGet,
  set(target2, key) {
    return true;
  },
  deleteProperty(target2, key) {
    return true;
  }
};
const shallowReactiveHandlers = /* @__PURE__ */ extend({}, mutableHandlers, {
  get: shallowGet,
  set: shallowSet
});
const shallowReadonlyHandlers = /* @__PURE__ */ extend({}, readonlyHandlers, {
  get: shallowReadonlyGet
});
const toShallow = (value) => value;
const getProto = (v2) => Reflect.getPrototypeOf(v2);
function get(target2, key, isReadonly2 = false, isShallow2 = false) {
  target2 = target2[
    "__v_raw"
    /* ReactiveFlags.RAW */
  ];
  const rawTarget = toRaw(target2);
  const rawKey = toRaw(key);
  if (!isReadonly2) {
    if (key !== rawKey) {
      track(rawTarget, "get", key);
    }
    track(rawTarget, "get", rawKey);
  }
  const { has: has2 } = getProto(rawTarget);
  const wrap2 = isShallow2 ? toShallow : isReadonly2 ? toReadonly : toReactive;
  if (has2.call(rawTarget, key)) {
    return wrap2(target2.get(key));
  } else if (has2.call(rawTarget, rawKey)) {
    return wrap2(target2.get(rawKey));
  } else if (target2 !== rawTarget) {
    target2.get(key);
  }
}
function has(key, isReadonly2 = false) {
  const target2 = this[
    "__v_raw"
    /* ReactiveFlags.RAW */
  ];
  const rawTarget = toRaw(target2);
  const rawKey = toRaw(key);
  if (!isReadonly2) {
    if (key !== rawKey) {
      track(rawTarget, "has", key);
    }
    track(rawTarget, "has", rawKey);
  }
  return key === rawKey ? target2.has(key) : target2.has(key) || target2.has(rawKey);
}
function size$1(target2, isReadonly2 = false) {
  target2 = target2[
    "__v_raw"
    /* ReactiveFlags.RAW */
  ];
  !isReadonly2 && track(toRaw(target2), "iterate", ITERATE_KEY);
  return Reflect.get(target2, "size", target2);
}
function add(value) {
  value = toRaw(value);
  const target2 = toRaw(this);
  const proto = getProto(target2);
  const hadKey = proto.has.call(target2, value);
  if (!hadKey) {
    target2.add(value);
    trigger(target2, "add", value, value);
  }
  return this;
}
function set(key, value) {
  value = toRaw(value);
  const target2 = toRaw(this);
  const { has: has2, get: get2 } = getProto(target2);
  let hadKey = has2.call(target2, key);
  if (!hadKey) {
    key = toRaw(key);
    hadKey = has2.call(target2, key);
  }
  const oldValue = get2.call(target2, key);
  target2.set(key, value);
  if (!hadKey) {
    trigger(target2, "add", key, value);
  } else if (hasChanged(value, oldValue)) {
    trigger(target2, "set", key, value);
  }
  return this;
}
function deleteEntry(key) {
  const target2 = toRaw(this);
  const { has: has2, get: get2 } = getProto(target2);
  let hadKey = has2.call(target2, key);
  if (!hadKey) {
    key = toRaw(key);
    hadKey = has2.call(target2, key);
  }
  get2 ? get2.call(target2, key) : void 0;
  const result = target2.delete(key);
  if (hadKey) {
    trigger(target2, "delete", key, void 0);
  }
  return result;
}
function clear() {
  const target2 = toRaw(this);
  const hadItems = target2.size !== 0;
  const result = target2.clear();
  if (hadItems) {
    trigger(target2, "clear", void 0, void 0);
  }
  return result;
}
function createForEach(isReadonly2, isShallow2) {
  return function forEach(callback, thisArg) {
    const observed = this;
    const target2 = observed[
      "__v_raw"
      /* ReactiveFlags.RAW */
    ];
    const rawTarget = toRaw(target2);
    const wrap2 = isShallow2 ? toShallow : isReadonly2 ? toReadonly : toReactive;
    !isReadonly2 && track(rawTarget, "iterate", ITERATE_KEY);
    return target2.forEach((value, key) => {
      return callback.call(thisArg, wrap2(value), wrap2(key), observed);
    });
  };
}
function createIterableMethod(method, isReadonly2, isShallow2) {
  return function(...args) {
    const target2 = this[
      "__v_raw"
      /* ReactiveFlags.RAW */
    ];
    const rawTarget = toRaw(target2);
    const targetIsMap = isMap(rawTarget);
    const isPair = method === "entries" || method === Symbol.iterator && targetIsMap;
    const isKeyOnly = method === "keys" && targetIsMap;
    const innerIterator = target2[method](...args);
    const wrap2 = isShallow2 ? toShallow : isReadonly2 ? toReadonly : toReactive;
    !isReadonly2 && track(rawTarget, "iterate", isKeyOnly ? MAP_KEY_ITERATE_KEY : ITERATE_KEY);
    return {
      // iterator protocol
      next() {
        const { value, done } = innerIterator.next();
        return done ? { value, done } : {
          value: isPair ? [wrap2(value[0]), wrap2(value[1])] : wrap2(value),
          done
        };
      },
      // iterable protocol
      [Symbol.iterator]() {
        return this;
      }
    };
  };
}
function createReadonlyMethod(type2) {
  return function(...args) {
    return type2 === "delete" ? false : this;
  };
}
function createInstrumentations() {
  const mutableInstrumentations2 = {
    get(key) {
      return get(this, key);
    },
    get size() {
      return size$1(this);
    },
    has,
    add,
    set,
    delete: deleteEntry,
    clear,
    forEach: createForEach(false, false)
  };
  const shallowInstrumentations2 = {
    get(key) {
      return get(this, key, false, true);
    },
    get size() {
      return size$1(this);
    },
    has,
    add,
    set,
    delete: deleteEntry,
    clear,
    forEach: createForEach(false, true)
  };
  const readonlyInstrumentations2 = {
    get(key) {
      return get(this, key, true);
    },
    get size() {
      return size$1(this, true);
    },
    has(key) {
      return has.call(this, key, true);
    },
    add: createReadonlyMethod(
      "add"
      /* TriggerOpTypes.ADD */
    ),
    set: createReadonlyMethod(
      "set"
      /* TriggerOpTypes.SET */
    ),
    delete: createReadonlyMethod(
      "delete"
      /* TriggerOpTypes.DELETE */
    ),
    clear: createReadonlyMethod(
      "clear"
      /* TriggerOpTypes.CLEAR */
    ),
    forEach: createForEach(true, false)
  };
  const shallowReadonlyInstrumentations2 = {
    get(key) {
      return get(this, key, true, true);
    },
    get size() {
      return size$1(this, true);
    },
    has(key) {
      return has.call(this, key, true);
    },
    add: createReadonlyMethod(
      "add"
      /* TriggerOpTypes.ADD */
    ),
    set: createReadonlyMethod(
      "set"
      /* TriggerOpTypes.SET */
    ),
    delete: createReadonlyMethod(
      "delete"
      /* TriggerOpTypes.DELETE */
    ),
    clear: createReadonlyMethod(
      "clear"
      /* TriggerOpTypes.CLEAR */
    ),
    forEach: createForEach(true, true)
  };
  const iteratorMethods = ["keys", "values", "entries", Symbol.iterator];
  iteratorMethods.forEach((method) => {
    mutableInstrumentations2[method] = createIterableMethod(method, false, false);
    readonlyInstrumentations2[method] = createIterableMethod(method, true, false);
    shallowInstrumentations2[method] = createIterableMethod(method, false, true);
    shallowReadonlyInstrumentations2[method] = createIterableMethod(method, true, true);
  });
  return [
    mutableInstrumentations2,
    readonlyInstrumentations2,
    shallowInstrumentations2,
    shallowReadonlyInstrumentations2
  ];
}
const [mutableInstrumentations, readonlyInstrumentations, shallowInstrumentations, shallowReadonlyInstrumentations] = /* @__PURE__ */ createInstrumentations();
function createInstrumentationGetter(isReadonly2, shallow) {
  const instrumentations = shallow ? isReadonly2 ? shallowReadonlyInstrumentations : shallowInstrumentations : isReadonly2 ? readonlyInstrumentations : mutableInstrumentations;
  return (target2, key, receiver) => {
    if (key === "__v_isReactive") {
      return !isReadonly2;
    } else if (key === "__v_isReadonly") {
      return isReadonly2;
    } else if (key === "__v_raw") {
      return target2;
    }
    return Reflect.get(hasOwn$1(instrumentations, key) && key in target2 ? instrumentations : target2, key, receiver);
  };
}
const mutableCollectionHandlers = {
  get: /* @__PURE__ */ createInstrumentationGetter(false, false)
};
const shallowCollectionHandlers = {
  get: /* @__PURE__ */ createInstrumentationGetter(false, true)
};
const readonlyCollectionHandlers = {
  get: /* @__PURE__ */ createInstrumentationGetter(true, false)
};
const shallowReadonlyCollectionHandlers = {
  get: /* @__PURE__ */ createInstrumentationGetter(true, true)
};
const reactiveMap = /* @__PURE__ */ new WeakMap();
const shallowReactiveMap = /* @__PURE__ */ new WeakMap();
const readonlyMap = /* @__PURE__ */ new WeakMap();
const shallowReadonlyMap = /* @__PURE__ */ new WeakMap();
function targetTypeMap(rawType) {
  switch (rawType) {
    case "Object":
    case "Array":
      return 1;
    case "Map":
    case "Set":
    case "WeakMap":
    case "WeakSet":
      return 2;
    default:
      return 0;
  }
}
function getTargetType(value) {
  return value[
    "__v_skip"
    /* ReactiveFlags.SKIP */
  ] || !Object.isExtensible(value) ? 0 : targetTypeMap(toRawType(value));
}
function reactive(target2) {
  if (isReadonly(target2)) {
    return target2;
  }
  return createReactiveObject(target2, false, mutableHandlers, mutableCollectionHandlers, reactiveMap);
}
function shallowReactive(target2) {
  return createReactiveObject(target2, false, shallowReactiveHandlers, shallowCollectionHandlers, shallowReactiveMap);
}
function readonly(target2) {
  return createReactiveObject(target2, true, readonlyHandlers, readonlyCollectionHandlers, readonlyMap);
}
function shallowReadonly(target2) {
  return createReactiveObject(target2, true, shallowReadonlyHandlers, shallowReadonlyCollectionHandlers, shallowReadonlyMap);
}
function createReactiveObject(target2, isReadonly2, baseHandlers, collectionHandlers, proxyMap) {
  if (!isObject$5(target2)) {
    return target2;
  }
  if (target2[
    "__v_raw"
    /* ReactiveFlags.RAW */
  ] && !(isReadonly2 && target2[
    "__v_isReactive"
    /* ReactiveFlags.IS_REACTIVE */
  ])) {
    return target2;
  }
  const existingProxy = proxyMap.get(target2);
  if (existingProxy) {
    return existingProxy;
  }
  const targetType = getTargetType(target2);
  if (targetType === 0) {
    return target2;
  }
  const proxy = new Proxy(target2, targetType === 2 ? collectionHandlers : baseHandlers);
  proxyMap.set(target2, proxy);
  return proxy;
}
function isReactive(value) {
  if (isReadonly(value)) {
    return isReactive(value[
      "__v_raw"
      /* ReactiveFlags.RAW */
    ]);
  }
  return !!(value && value[
    "__v_isReactive"
    /* ReactiveFlags.IS_REACTIVE */
  ]);
}
function isReadonly(value) {
  return !!(value && value[
    "__v_isReadonly"
    /* ReactiveFlags.IS_READONLY */
  ]);
}
function isShallow(value) {
  return !!(value && value[
    "__v_isShallow"
    /* ReactiveFlags.IS_SHALLOW */
  ]);
}
function isProxy(value) {
  return isReactive(value) || isReadonly(value);
}
function toRaw(observed) {
  const raw = observed && observed[
    "__v_raw"
    /* ReactiveFlags.RAW */
  ];
  return raw ? toRaw(raw) : observed;
}
function markRaw(value) {
  def(value, "__v_skip", true);
  return value;
}
const toReactive = (value) => isObject$5(value) ? reactive(value) : value;
const toReadonly = (value) => isObject$5(value) ? readonly(value) : value;
function trackRefValue(ref2) {
  if (shouldTrack && activeEffect) {
    ref2 = toRaw(ref2);
    {
      trackEffects(ref2.dep || (ref2.dep = createDep()));
    }
  }
}
function triggerRefValue(ref2, newVal) {
  ref2 = toRaw(ref2);
  const dep = ref2.dep;
  if (dep) {
    {
      triggerEffects(dep);
    }
  }
}
function isRef(r2) {
  return !!(r2 && r2.__v_isRef === true);
}
function ref(value) {
  return createRef(value, false);
}
function shallowRef(value) {
  return createRef(value, true);
}
function createRef(rawValue, shallow) {
  if (isRef(rawValue)) {
    return rawValue;
  }
  return new RefImpl(rawValue, shallow);
}
class RefImpl {
  constructor(value, __v_isShallow) {
    this.__v_isShallow = __v_isShallow;
    this.dep = void 0;
    this.__v_isRef = true;
    this._rawValue = __v_isShallow ? value : toRaw(value);
    this._value = __v_isShallow ? value : toReactive(value);
  }
  get value() {
    trackRefValue(this);
    return this._value;
  }
  set value(newVal) {
    const useDirectValue = this.__v_isShallow || isShallow(newVal) || isReadonly(newVal);
    newVal = useDirectValue ? newVal : toRaw(newVal);
    if (hasChanged(newVal, this._rawValue)) {
      this._rawValue = newVal;
      this._value = useDirectValue ? newVal : toReactive(newVal);
      triggerRefValue(this);
    }
  }
}
function triggerRef(ref2) {
  triggerRefValue(ref2);
}
function unref(ref2) {
  return isRef(ref2) ? ref2.value : ref2;
}
const shallowUnwrapHandlers = {
  get: (target2, key, receiver) => unref(Reflect.get(target2, key, receiver)),
  set: (target2, key, value, receiver) => {
    const oldValue = target2[key];
    if (isRef(oldValue) && !isRef(value)) {
      oldValue.value = value;
      return true;
    } else {
      return Reflect.set(target2, key, value, receiver);
    }
  }
};
function proxyRefs(objectWithRefs) {
  return isReactive(objectWithRefs) ? objectWithRefs : new Proxy(objectWithRefs, shallowUnwrapHandlers);
}
class CustomRefImpl {
  constructor(factory) {
    this.dep = void 0;
    this.__v_isRef = true;
    const { get: get2, set: set2 } = factory(() => trackRefValue(this), () => triggerRefValue(this));
    this._get = get2;
    this._set = set2;
  }
  get value() {
    return this._get();
  }
  set value(newVal) {
    this._set(newVal);
  }
}
function customRef(factory) {
  return new CustomRefImpl(factory);
}
function toRefs(object) {
  const ret = isArray$2(object) ? new Array(object.length) : {};
  for (const key in object) {
    ret[key] = toRef(object, key);
  }
  return ret;
}
class ObjectRefImpl {
  constructor(_object, _key, _defaultValue) {
    this._object = _object;
    this._key = _key;
    this._defaultValue = _defaultValue;
    this.__v_isRef = true;
  }
  get value() {
    const val = this._object[this._key];
    return val === void 0 ? this._defaultValue : val;
  }
  set value(newVal) {
    this._object[this._key] = newVal;
  }
  get dep() {
    return getDepFromReactive(toRaw(this._object), this._key);
  }
}
function toRef(object, key, defaultValue) {
  const val = object[key];
  return isRef(val) ? val : new ObjectRefImpl(object, key, defaultValue);
}
var _a$1;
class ComputedRefImpl {
  constructor(getter, _setter, isReadonly2, isSSR) {
    this._setter = _setter;
    this.dep = void 0;
    this.__v_isRef = true;
    this[_a$1] = false;
    this._dirty = true;
    this.effect = new ReactiveEffect(getter, () => {
      if (!this._dirty) {
        this._dirty = true;
        triggerRefValue(this);
      }
    });
    this.effect.computed = this;
    this.effect.active = this._cacheable = !isSSR;
    this[
      "__v_isReadonly"
      /* ReactiveFlags.IS_READONLY */
    ] = isReadonly2;
  }
  get value() {
    const self2 = toRaw(this);
    trackRefValue(self2);
    if (self2._dirty || !self2._cacheable) {
      self2._dirty = false;
      self2._value = self2.effect.run();
    }
    return self2._value;
  }
  set value(newValue) {
    this._setter(newValue);
  }
}
_a$1 = "__v_isReadonly";
function computed$1(getterOrOptions, debugOptions, isSSR = false) {
  let getter;
  let setter;
  const onlyGetter = isFunction$2(getterOrOptions);
  if (onlyGetter) {
    getter = getterOrOptions;
    setter = NOOP;
  } else {
    getter = getterOrOptions.get;
    setter = getterOrOptions.set;
  }
  const cRef = new ComputedRefImpl(getter, setter, onlyGetter || !setter, isSSR);
  return cRef;
}
function warn$1(msg, ...args) {
  return;
}
function assertNumber(val, type2) {
  return;
}
function callWithErrorHandling(fn, instance, type2, args) {
  let res;
  try {
    res = args ? fn(...args) : fn();
  } catch (err) {
    handleError(err, instance, type2);
  }
  return res;
}
function callWithAsyncErrorHandling(fn, instance, type2, args) {
  if (isFunction$2(fn)) {
    const res = callWithErrorHandling(fn, instance, type2, args);
    if (res && isPromise$1(res)) {
      res.catch((err) => {
        handleError(err, instance, type2);
      });
    }
    return res;
  }
  const values = [];
  for (let i2 = 0; i2 < fn.length; i2++) {
    values.push(callWithAsyncErrorHandling(fn[i2], instance, type2, args));
  }
  return values;
}
function handleError(err, instance, type2, throwInDev = true) {
  const contextVNode = instance ? instance.vnode : null;
  if (instance) {
    let cur = instance.parent;
    const exposedInstance = instance.proxy;
    const errorInfo = type2;
    while (cur) {
      const errorCapturedHooks = cur.ec;
      if (errorCapturedHooks) {
        for (let i2 = 0; i2 < errorCapturedHooks.length; i2++) {
          if (errorCapturedHooks[i2](err, exposedInstance, errorInfo) === false) {
            return;
          }
        }
      }
      cur = cur.parent;
    }
    const appErrorHandler = instance.appContext.config.errorHandler;
    if (appErrorHandler) {
      callWithErrorHandling(appErrorHandler, null, 10, [err, exposedInstance, errorInfo]);
      return;
    }
  }
  logError(err, type2, contextVNode, throwInDev);
}
function logError(err, type2, contextVNode, throwInDev = true) {
  {
    console.error(err);
  }
}
let isFlushing = false;
let isFlushPending = false;
const queue = [];
let flushIndex = 0;
const pendingPostFlushCbs = [];
let activePostFlushCbs = null;
let postFlushIndex = 0;
const resolvedPromise = /* @__PURE__ */ Promise.resolve();
let currentFlushPromise = null;
function nextTick(fn) {
  const p2 = currentFlushPromise || resolvedPromise;
  return fn ? p2.then(this ? fn.bind(this) : fn) : p2;
}
function findInsertionIndex(id2) {
  let start = flushIndex + 1;
  let end = queue.length;
  while (start < end) {
    const middle = start + end >>> 1;
    const middleJobId = getId$1(queue[middle]);
    middleJobId < id2 ? start = middle + 1 : end = middle;
  }
  return start;
}
function queueJob(job) {
  if (!queue.length || !queue.includes(job, isFlushing && job.allowRecurse ? flushIndex + 1 : flushIndex)) {
    if (job.id == null) {
      queue.push(job);
    } else {
      queue.splice(findInsertionIndex(job.id), 0, job);
    }
    queueFlush();
  }
}
function queueFlush() {
  if (!isFlushing && !isFlushPending) {
    isFlushPending = true;
    currentFlushPromise = resolvedPromise.then(flushJobs);
  }
}
function invalidateJob(job) {
  const i2 = queue.indexOf(job);
  if (i2 > flushIndex) {
    queue.splice(i2, 1);
  }
}
function queuePostFlushCb(cb) {
  if (!isArray$2(cb)) {
    if (!activePostFlushCbs || !activePostFlushCbs.includes(cb, cb.allowRecurse ? postFlushIndex + 1 : postFlushIndex)) {
      pendingPostFlushCbs.push(cb);
    }
  } else {
    pendingPostFlushCbs.push(...cb);
  }
  queueFlush();
}
function flushPreFlushCbs(seen2, i2 = isFlushing ? flushIndex + 1 : 0) {
  for (; i2 < queue.length; i2++) {
    const cb = queue[i2];
    if (cb && cb.pre) {
      queue.splice(i2, 1);
      i2--;
      cb();
    }
  }
}
function flushPostFlushCbs(seen2) {
  if (pendingPostFlushCbs.length) {
    const deduped = [...new Set(pendingPostFlushCbs)];
    pendingPostFlushCbs.length = 0;
    if (activePostFlushCbs) {
      activePostFlushCbs.push(...deduped);
      return;
    }
    activePostFlushCbs = deduped;
    activePostFlushCbs.sort((a2, b2) => getId$1(a2) - getId$1(b2));
    for (postFlushIndex = 0; postFlushIndex < activePostFlushCbs.length; postFlushIndex++) {
      activePostFlushCbs[postFlushIndex]();
    }
    activePostFlushCbs = null;
    postFlushIndex = 0;
  }
}
const getId$1 = (job) => job.id == null ? Infinity : job.id;
const comparator = (a2, b2) => {
  const diff = getId$1(a2) - getId$1(b2);
  if (diff === 0) {
    if (a2.pre && !b2.pre)
      return -1;
    if (b2.pre && !a2.pre)
      return 1;
  }
  return diff;
};
function flushJobs(seen2) {
  isFlushPending = false;
  isFlushing = true;
  queue.sort(comparator);
  try {
    for (flushIndex = 0; flushIndex < queue.length; flushIndex++) {
      const job = queue[flushIndex];
      if (job && job.active !== false) {
        if (false) ;
        callWithErrorHandling(
          job,
          null,
          14
          /* ErrorCodes.SCHEDULER */
        );
      }
    }
  } finally {
    flushIndex = 0;
    queue.length = 0;
    flushPostFlushCbs();
    isFlushing = false;
    currentFlushPromise = null;
    if (queue.length || pendingPostFlushCbs.length) {
      flushJobs();
    }
  }
}
let devtools$1;
let buffer = [];
function setDevtoolsHook(hook, target2) {
  var _a2, _b2;
  devtools$1 = hook;
  if (devtools$1) {
    devtools$1.enabled = true;
    buffer.forEach(({ event, args }) => devtools$1.emit(event, ...args));
    buffer = [];
  } else if (
    // handle late devtools injection - only do this if we are in an actual
    // browser environment to avoid the timer handle stalling test runner exit
    // (#4815)
    typeof window !== "undefined" && // some envs mock window but not fully
    window.HTMLElement && // also exclude jsdom
    !((_b2 = (_a2 = window.navigator) === null || _a2 === void 0 ? void 0 : _a2.userAgent) === null || _b2 === void 0 ? void 0 : _b2.includes("jsdom"))
  ) {
    const replay = target2.__VUE_DEVTOOLS_HOOK_REPLAY__ = target2.__VUE_DEVTOOLS_HOOK_REPLAY__ || [];
    replay.push((newHook) => {
      setDevtoolsHook(newHook, target2);
    });
    setTimeout(() => {
      if (!devtools$1) {
        target2.__VUE_DEVTOOLS_HOOK_REPLAY__ = null;
        buffer = [];
      }
    }, 3e3);
  } else {
    buffer = [];
  }
}
function emit(instance, event, ...rawArgs) {
  if (instance.isUnmounted)
    return;
  const props = instance.vnode.props || EMPTY_OBJ;
  let args = rawArgs;
  const isModelListener2 = event.startsWith("update:");
  const modelArg = isModelListener2 && event.slice(7);
  if (modelArg && modelArg in props) {
    const modifiersKey = `${modelArg === "modelValue" ? "model" : modelArg}Modifiers`;
    const { number: number2, trim } = props[modifiersKey] || EMPTY_OBJ;
    if (trim) {
      args = rawArgs.map((a2) => isString$4(a2) ? a2.trim() : a2);
    }
    if (number2) {
      args = rawArgs.map(looseToNumber);
    }
  }
  let handlerName;
  let handler = props[handlerName = toHandlerKey(event)] || // also try camelCase event handler (#2249)
  props[handlerName = toHandlerKey(camelize$1(event))];
  if (!handler && isModelListener2) {
    handler = props[handlerName = toHandlerKey(hyphenate(event))];
  }
  if (handler) {
    callWithAsyncErrorHandling(handler, instance, 6, args);
  }
  const onceHandler = props[handlerName + `Once`];
  if (onceHandler) {
    if (!instance.emitted) {
      instance.emitted = {};
    } else if (instance.emitted[handlerName]) {
      return;
    }
    instance.emitted[handlerName] = true;
    callWithAsyncErrorHandling(onceHandler, instance, 6, args);
  }
}
function normalizeEmitsOptions(comp2, appContext, asMixin = false) {
  const cache2 = appContext.emitsCache;
  const cached = cache2.get(comp2);
  if (cached !== void 0) {
    return cached;
  }
  const raw = comp2.emits;
  let normalized = {};
  let hasExtends = false;
  if (!isFunction$2(comp2)) {
    const extendEmits = (raw2) => {
      const normalizedFromExtend = normalizeEmitsOptions(raw2, appContext, true);
      if (normalizedFromExtend) {
        hasExtends = true;
        extend(normalized, normalizedFromExtend);
      }
    };
    if (!asMixin && appContext.mixins.length) {
      appContext.mixins.forEach(extendEmits);
    }
    if (comp2.extends) {
      extendEmits(comp2.extends);
    }
    if (comp2.mixins) {
      comp2.mixins.forEach(extendEmits);
    }
  }
  if (!raw && !hasExtends) {
    if (isObject$5(comp2)) {
      cache2.set(comp2, null);
    }
    return null;
  }
  if (isArray$2(raw)) {
    raw.forEach((key) => normalized[key] = null);
  } else {
    extend(normalized, raw);
  }
  if (isObject$5(comp2)) {
    cache2.set(comp2, normalized);
  }
  return normalized;
}
function isEmitListener(options2, key) {
  if (!options2 || !isOn(key)) {
    return false;
  }
  key = key.slice(2).replace(/Once$/, "");
  return hasOwn$1(options2, key[0].toLowerCase() + key.slice(1)) || hasOwn$1(options2, hyphenate(key)) || hasOwn$1(options2, key);
}
let currentRenderingInstance = null;
let currentScopeId = null;
function setCurrentRenderingInstance(instance) {
  const prev = currentRenderingInstance;
  currentRenderingInstance = instance;
  currentScopeId = instance && instance.type.__scopeId || null;
  return prev;
}
function pushScopeId(id2) {
  currentScopeId = id2;
}
function popScopeId() {
  currentScopeId = null;
}
const withScopeId = (_id) => withCtx;
function withCtx(fn, ctx = currentRenderingInstance, isNonScopedSlot) {
  if (!ctx)
    return fn;
  if (fn._n) {
    return fn;
  }
  const renderFnWithContext = (...args) => {
    if (renderFnWithContext._d) {
      setBlockTracking(-1);
    }
    const prevInstance = setCurrentRenderingInstance(ctx);
    let res;
    try {
      res = fn(...args);
    } finally {
      setCurrentRenderingInstance(prevInstance);
      if (renderFnWithContext._d) {
        setBlockTracking(1);
      }
    }
    return res;
  };
  renderFnWithContext._n = true;
  renderFnWithContext._c = true;
  renderFnWithContext._d = true;
  return renderFnWithContext;
}
function markAttrsAccessed() {
}
function renderComponentRoot(instance) {
  const { type: Component, vnode, proxy, withProxy, props, propsOptions: [propsOptions], slots, attrs, emit: emit2, render: render2, renderCache, data, setupState, ctx, inheritAttrs } = instance;
  let result;
  let fallthroughAttrs;
  const prev = setCurrentRenderingInstance(instance);
  try {
    if (vnode.shapeFlag & 4) {
      const proxyToUse = withProxy || proxy;
      result = normalizeVNode(render2.call(proxyToUse, proxyToUse, renderCache, props, setupState, data, ctx));
      fallthroughAttrs = attrs;
    } else {
      const render22 = Component;
      if (false) ;
      result = normalizeVNode(render22.length > 1 ? render22(props, false ? {
        get attrs() {
          markAttrsAccessed();
          return attrs;
        },
        slots,
        emit: emit2
      } : { attrs, slots, emit: emit2 }) : render22(
        props,
        null
        /* we know it doesn't need it */
      ));
      fallthroughAttrs = Component.props ? attrs : getFunctionalFallthrough(attrs);
    }
  } catch (err) {
    blockStack.length = 0;
    handleError(
      err,
      instance,
      1
      /* ErrorCodes.RENDER_FUNCTION */
    );
    result = createVNode(Comment);
  }
  let root2 = result;
  if (fallthroughAttrs && inheritAttrs !== false) {
    const keys = Object.keys(fallthroughAttrs);
    const { shapeFlag } = root2;
    if (keys.length) {
      if (shapeFlag & (1 | 6)) {
        if (propsOptions && keys.some(isModelListener)) {
          fallthroughAttrs = filterModelListeners(fallthroughAttrs, propsOptions);
        }
        root2 = cloneVNode(root2, fallthroughAttrs);
      }
    }
  }
  if (vnode.dirs) {
    root2 = cloneVNode(root2);
    root2.dirs = root2.dirs ? root2.dirs.concat(vnode.dirs) : vnode.dirs;
  }
  if (vnode.transition) {
    root2.transition = vnode.transition;
  }
  {
    result = root2;
  }
  setCurrentRenderingInstance(prev);
  return result;
}
function filterSingleRoot(children) {
  let singleRoot;
  for (let i2 = 0; i2 < children.length; i2++) {
    const child = children[i2];
    if (isVNode$1(child)) {
      if (child.type !== Comment || child.children === "v-if") {
        if (singleRoot) {
          return;
        } else {
          singleRoot = child;
        }
      }
    } else {
      return;
    }
  }
  return singleRoot;
}
const getFunctionalFallthrough = (attrs) => {
  let res;
  for (const key in attrs) {
    if (key === "class" || key === "style" || isOn(key)) {
      (res || (res = {}))[key] = attrs[key];
    }
  }
  return res;
};
const filterModelListeners = (attrs, props) => {
  const res = {};
  for (const key in attrs) {
    if (!isModelListener(key) || !(key.slice(9) in props)) {
      res[key] = attrs[key];
    }
  }
  return res;
};
function shouldUpdateComponent(prevVNode, nextVNode, optimized) {
  const { props: prevProps, children: prevChildren, component } = prevVNode;
  const { props: nextProps, children: nextChildren, patchFlag } = nextVNode;
  const emits = component.emitsOptions;
  if (nextVNode.dirs || nextVNode.transition) {
    return true;
  }
  if (optimized && patchFlag >= 0) {
    if (patchFlag & 1024) {
      return true;
    }
    if (patchFlag & 16) {
      if (!prevProps) {
        return !!nextProps;
      }
      return hasPropsChanged(prevProps, nextProps, emits);
    } else if (patchFlag & 8) {
      const dynamicProps = nextVNode.dynamicProps;
      for (let i2 = 0; i2 < dynamicProps.length; i2++) {
        const key = dynamicProps[i2];
        if (nextProps[key] !== prevProps[key] && !isEmitListener(emits, key)) {
          return true;
        }
      }
    }
  } else {
    if (prevChildren || nextChildren) {
      if (!nextChildren || !nextChildren.$stable) {
        return true;
      }
    }
    if (prevProps === nextProps) {
      return false;
    }
    if (!prevProps) {
      return !!nextProps;
    }
    if (!nextProps) {
      return true;
    }
    return hasPropsChanged(prevProps, nextProps, emits);
  }
  return false;
}
function hasPropsChanged(prevProps, nextProps, emitsOptions) {
  const nextKeys = Object.keys(nextProps);
  if (nextKeys.length !== Object.keys(prevProps).length) {
    return true;
  }
  for (let i2 = 0; i2 < nextKeys.length; i2++) {
    const key = nextKeys[i2];
    if (nextProps[key] !== prevProps[key] && !isEmitListener(emitsOptions, key)) {
      return true;
    }
  }
  return false;
}
function updateHOCHostEl({ vnode, parent }, el2) {
  while (parent && parent.subTree === vnode) {
    (vnode = parent.vnode).el = el2;
    parent = parent.parent;
  }
}
const isSuspense = (type2) => type2.__isSuspense;
const SuspenseImpl = {
  name: "Suspense",
  // In order to make Suspense tree-shakable, we need to avoid importing it
  // directly in the renderer. The renderer checks for the __isSuspense flag
  // on a vnode's type and calls the `process` method, passing in renderer
  // internals.
  __isSuspense: true,
  process(n1, n2, container, anchor, parentComponent, parentSuspense, isSVG, slotScopeIds, optimized, rendererInternals) {
    if (n1 == null) {
      mountSuspense(n2, container, anchor, parentComponent, parentSuspense, isSVG, slotScopeIds, optimized, rendererInternals);
    } else {
      patchSuspense(n1, n2, container, anchor, parentComponent, isSVG, slotScopeIds, optimized, rendererInternals);
    }
  },
  hydrate: hydrateSuspense,
  create: createSuspenseBoundary,
  normalize: normalizeSuspenseChildren
};
const Suspense = SuspenseImpl;
function triggerEvent(vnode, name2) {
  const eventListener = vnode.props && vnode.props[name2];
  if (isFunction$2(eventListener)) {
    eventListener();
  }
}
function mountSuspense(vnode, container, anchor, parentComponent, parentSuspense, isSVG, slotScopeIds, optimized, rendererInternals) {
  const { p: patch, o: { createElement } } = rendererInternals;
  const hiddenContainer = createElement("div");
  const suspense = vnode.suspense = createSuspenseBoundary(vnode, parentSuspense, parentComponent, container, hiddenContainer, anchor, isSVG, slotScopeIds, optimized, rendererInternals);
  patch(null, suspense.pendingBranch = vnode.ssContent, hiddenContainer, null, parentComponent, suspense, isSVG, slotScopeIds);
  if (suspense.deps > 0) {
    triggerEvent(vnode, "onPending");
    triggerEvent(vnode, "onFallback");
    patch(
      null,
      vnode.ssFallback,
      container,
      anchor,
      parentComponent,
      null,
      // fallback tree will not have suspense context
      isSVG,
      slotScopeIds
    );
    setActiveBranch(suspense, vnode.ssFallback);
  } else {
    suspense.resolve();
  }
}
function patchSuspense(n1, n2, container, anchor, parentComponent, isSVG, slotScopeIds, optimized, { p: patch, um: unmount, o: { createElement } }) {
  const suspense = n2.suspense = n1.suspense;
  suspense.vnode = n2;
  n2.el = n1.el;
  const newBranch = n2.ssContent;
  const newFallback = n2.ssFallback;
  const { activeBranch, pendingBranch, isInFallback, isHydrating } = suspense;
  if (pendingBranch) {
    suspense.pendingBranch = newBranch;
    if (isSameVNodeType(newBranch, pendingBranch)) {
      patch(pendingBranch, newBranch, suspense.hiddenContainer, null, parentComponent, suspense, isSVG, slotScopeIds, optimized);
      if (suspense.deps <= 0) {
        suspense.resolve();
      } else if (isInFallback) {
        patch(
          activeBranch,
          newFallback,
          container,
          anchor,
          parentComponent,
          null,
          // fallback tree will not have suspense context
          isSVG,
          slotScopeIds,
          optimized
        );
        setActiveBranch(suspense, newFallback);
      }
    } else {
      suspense.pendingId++;
      if (isHydrating) {
        suspense.isHydrating = false;
        suspense.activeBranch = pendingBranch;
      } else {
        unmount(pendingBranch, parentComponent, suspense);
      }
      suspense.deps = 0;
      suspense.effects.length = 0;
      suspense.hiddenContainer = createElement("div");
      if (isInFallback) {
        patch(null, newBranch, suspense.hiddenContainer, null, parentComponent, suspense, isSVG, slotScopeIds, optimized);
        if (suspense.deps <= 0) {
          suspense.resolve();
        } else {
          patch(
            activeBranch,
            newFallback,
            container,
            anchor,
            parentComponent,
            null,
            // fallback tree will not have suspense context
            isSVG,
            slotScopeIds,
            optimized
          );
          setActiveBranch(suspense, newFallback);
        }
      } else if (activeBranch && isSameVNodeType(newBranch, activeBranch)) {
        patch(activeBranch, newBranch, container, anchor, parentComponent, suspense, isSVG, slotScopeIds, optimized);
        suspense.resolve(true);
      } else {
        patch(null, newBranch, suspense.hiddenContainer, null, parentComponent, suspense, isSVG, slotScopeIds, optimized);
        if (suspense.deps <= 0) {
          suspense.resolve();
        }
      }
    }
  } else {
    if (activeBranch && isSameVNodeType(newBranch, activeBranch)) {
      patch(activeBranch, newBranch, container, anchor, parentComponent, suspense, isSVG, slotScopeIds, optimized);
      setActiveBranch(suspense, newBranch);
    } else {
      triggerEvent(n2, "onPending");
      suspense.pendingBranch = newBranch;
      suspense.pendingId++;
      patch(null, newBranch, suspense.hiddenContainer, null, parentComponent, suspense, isSVG, slotScopeIds, optimized);
      if (suspense.deps <= 0) {
        suspense.resolve();
      } else {
        const { timeout, pendingId } = suspense;
        if (timeout > 0) {
          setTimeout(() => {
            if (suspense.pendingId === pendingId) {
              suspense.fallback(newFallback);
            }
          }, timeout);
        } else if (timeout === 0) {
          suspense.fallback(newFallback);
        }
      }
    }
  }
}
function createSuspenseBoundary(vnode, parent, parentComponent, container, hiddenContainer, anchor, isSVG, slotScopeIds, optimized, rendererInternals, isHydrating = false) {
  const { p: patch, m: move, um: unmount, n: next, o: { parentNode, remove: remove2 } } = rendererInternals;
  const timeout = vnode.props ? toNumber(vnode.props.timeout) : void 0;
  const suspense = {
    vnode,
    parent,
    parentComponent,
    isSVG,
    container,
    hiddenContainer,
    anchor,
    deps: 0,
    pendingId: 0,
    timeout: typeof timeout === "number" ? timeout : -1,
    activeBranch: null,
    pendingBranch: null,
    isInFallback: true,
    isHydrating,
    isUnmounted: false,
    effects: [],
    resolve(resume = false) {
      const { vnode: vnode2, activeBranch, pendingBranch, pendingId, effects, parentComponent: parentComponent2, container: container2 } = suspense;
      if (suspense.isHydrating) {
        suspense.isHydrating = false;
      } else if (!resume) {
        const delayEnter = activeBranch && pendingBranch.transition && pendingBranch.transition.mode === "out-in";
        if (delayEnter) {
          activeBranch.transition.afterLeave = () => {
            if (pendingId === suspense.pendingId) {
              move(
                pendingBranch,
                container2,
                anchor2,
                0
                /* MoveType.ENTER */
              );
            }
          };
        }
        let { anchor: anchor2 } = suspense;
        if (activeBranch) {
          anchor2 = next(activeBranch);
          unmount(activeBranch, parentComponent2, suspense, true);
        }
        if (!delayEnter) {
          move(
            pendingBranch,
            container2,
            anchor2,
            0
            /* MoveType.ENTER */
          );
        }
      }
      setActiveBranch(suspense, pendingBranch);
      suspense.pendingBranch = null;
      suspense.isInFallback = false;
      let parent2 = suspense.parent;
      let hasUnresolvedAncestor = false;
      while (parent2) {
        if (parent2.pendingBranch) {
          parent2.effects.push(...effects);
          hasUnresolvedAncestor = true;
          break;
        }
        parent2 = parent2.parent;
      }
      if (!hasUnresolvedAncestor) {
        queuePostFlushCb(effects);
      }
      suspense.effects = [];
      triggerEvent(vnode2, "onResolve");
    },
    fallback(fallbackVNode) {
      if (!suspense.pendingBranch) {
        return;
      }
      const { vnode: vnode2, activeBranch, parentComponent: parentComponent2, container: container2, isSVG: isSVG2 } = suspense;
      triggerEvent(vnode2, "onFallback");
      const anchor2 = next(activeBranch);
      const mountFallback = () => {
        if (!suspense.isInFallback) {
          return;
        }
        patch(
          null,
          fallbackVNode,
          container2,
          anchor2,
          parentComponent2,
          null,
          // fallback tree will not have suspense context
          isSVG2,
          slotScopeIds,
          optimized
        );
        setActiveBranch(suspense, fallbackVNode);
      };
      const delayEnter = fallbackVNode.transition && fallbackVNode.transition.mode === "out-in";
      if (delayEnter) {
        activeBranch.transition.afterLeave = mountFallback;
      }
      suspense.isInFallback = true;
      unmount(
        activeBranch,
        parentComponent2,
        null,
        // no suspense so unmount hooks fire now
        true
        // shouldRemove
      );
      if (!delayEnter) {
        mountFallback();
      }
    },
    move(container2, anchor2, type2) {
      suspense.activeBranch && move(suspense.activeBranch, container2, anchor2, type2);
      suspense.container = container2;
    },
    next() {
      return suspense.activeBranch && next(suspense.activeBranch);
    },
    registerDep(instance, setupRenderEffect) {
      const isInPendingSuspense = !!suspense.pendingBranch;
      if (isInPendingSuspense) {
        suspense.deps++;
      }
      const hydratedEl = instance.vnode.el;
      instance.asyncDep.catch((err) => {
        handleError(
          err,
          instance,
          0
          /* ErrorCodes.SETUP_FUNCTION */
        );
      }).then((asyncSetupResult) => {
        if (instance.isUnmounted || suspense.isUnmounted || suspense.pendingId !== instance.suspenseId) {
          return;
        }
        instance.asyncResolved = true;
        const { vnode: vnode2 } = instance;
        handleSetupResult(instance, asyncSetupResult, false);
        if (hydratedEl) {
          vnode2.el = hydratedEl;
        }
        const placeholder = !hydratedEl && instance.subTree.el;
        setupRenderEffect(
          instance,
          vnode2,
          // component may have been moved before resolve.
          // if this is not a hydration, instance.subTree will be the comment
          // placeholder.
          parentNode(hydratedEl || instance.subTree.el),
          // anchor will not be used if this is hydration, so only need to
          // consider the comment placeholder case.
          hydratedEl ? null : next(instance.subTree),
          suspense,
          isSVG,
          optimized
        );
        if (placeholder) {
          remove2(placeholder);
        }
        updateHOCHostEl(instance, vnode2.el);
        if (isInPendingSuspense && --suspense.deps === 0) {
          suspense.resolve();
        }
      });
    },
    unmount(parentSuspense, doRemove) {
      suspense.isUnmounted = true;
      if (suspense.activeBranch) {
        unmount(suspense.activeBranch, parentComponent, parentSuspense, doRemove);
      }
      if (suspense.pendingBranch) {
        unmount(suspense.pendingBranch, parentComponent, parentSuspense, doRemove);
      }
    }
  };
  return suspense;
}
function hydrateSuspense(node, vnode, parentComponent, parentSuspense, isSVG, slotScopeIds, optimized, rendererInternals, hydrateNode) {
  const suspense = vnode.suspense = createSuspenseBoundary(
    vnode,
    parentSuspense,
    parentComponent,
    node.parentNode,
    document.createElement("div"),
    null,
    isSVG,
    slotScopeIds,
    optimized,
    rendererInternals,
    true
    /* hydrating */
  );
  const result = hydrateNode(node, suspense.pendingBranch = vnode.ssContent, parentComponent, suspense, slotScopeIds, optimized);
  if (suspense.deps === 0) {
    suspense.resolve();
  }
  return result;
}
function normalizeSuspenseChildren(vnode) {
  const { shapeFlag, children } = vnode;
  const isSlotChildren = shapeFlag & 32;
  vnode.ssContent = normalizeSuspenseSlot(isSlotChildren ? children.default : children);
  vnode.ssFallback = isSlotChildren ? normalizeSuspenseSlot(children.fallback) : createVNode(Comment);
}
function normalizeSuspenseSlot(s2) {
  let block2;
  if (isFunction$2(s2)) {
    const trackBlock = isBlockTreeEnabled && s2._c;
    if (trackBlock) {
      s2._d = false;
      openBlock();
    }
    s2 = s2();
    if (trackBlock) {
      s2._d = true;
      block2 = currentBlock;
      closeBlock();
    }
  }
  if (isArray$2(s2)) {
    const singleChild = filterSingleRoot(s2);
    s2 = singleChild;
  }
  s2 = normalizeVNode(s2);
  if (block2 && !s2.dynamicChildren) {
    s2.dynamicChildren = block2.filter((c2) => c2 !== s2);
  }
  return s2;
}
function queueEffectWithSuspense(fn, suspense) {
  if (suspense && suspense.pendingBranch) {
    if (isArray$2(fn)) {
      suspense.effects.push(...fn);
    } else {
      suspense.effects.push(fn);
    }
  } else {
    queuePostFlushCb(fn);
  }
}
function setActiveBranch(suspense, branch) {
  suspense.activeBranch = branch;
  const { vnode, parentComponent } = suspense;
  const el2 = vnode.el = branch.el;
  if (parentComponent && parentComponent.subTree === vnode) {
    parentComponent.vnode.el = el2;
    updateHOCHostEl(parentComponent, el2);
  }
}
function provide(key, value) {
  if (!currentInstance) ;
  else {
    let provides = currentInstance.provides;
    const parentProvides = currentInstance.parent && currentInstance.parent.provides;
    if (parentProvides === provides) {
      provides = currentInstance.provides = Object.create(parentProvides);
    }
    provides[key] = value;
  }
}
function inject(key, defaultValue, treatDefaultAsFactory = false) {
  const instance = currentInstance || currentRenderingInstance;
  if (instance) {
    const provides = instance.parent == null ? instance.vnode.appContext && instance.vnode.appContext.provides : instance.parent.provides;
    if (provides && key in provides) {
      return provides[key];
    } else if (arguments.length > 1) {
      return treatDefaultAsFactory && isFunction$2(defaultValue) ? defaultValue.call(instance.proxy) : defaultValue;
    } else ;
  }
}
function watchEffect(effect2, options2) {
  return doWatch(effect2, null, options2);
}
function watchPostEffect(effect2, options2) {
  return doWatch(effect2, null, { flush: "post" });
}
function watchSyncEffect(effect2, options2) {
  return doWatch(effect2, null, { flush: "sync" });
}
const INITIAL_WATCHER_VALUE = {};
function watch(source, cb, options2) {
  return doWatch(source, cb, options2);
}
function doWatch(source, cb, { immediate, deep, flush, onTrack, onTrigger } = EMPTY_OBJ) {
  const instance = getCurrentScope() === (currentInstance === null || currentInstance === void 0 ? void 0 : currentInstance.scope) ? currentInstance : null;
  let getter;
  let forceTrigger = false;
  let isMultiSource = false;
  if (isRef(source)) {
    getter = () => source.value;
    forceTrigger = isShallow(source);
  } else if (isReactive(source)) {
    getter = () => source;
    deep = true;
  } else if (isArray$2(source)) {
    isMultiSource = true;
    forceTrigger = source.some((s2) => isReactive(s2) || isShallow(s2));
    getter = () => source.map((s2) => {
      if (isRef(s2)) {
        return s2.value;
      } else if (isReactive(s2)) {
        return traverse(s2);
      } else if (isFunction$2(s2)) {
        return callWithErrorHandling(
          s2,
          instance,
          2
          /* ErrorCodes.WATCH_GETTER */
        );
      } else ;
    });
  } else if (isFunction$2(source)) {
    if (cb) {
      getter = () => callWithErrorHandling(
        source,
        instance,
        2
        /* ErrorCodes.WATCH_GETTER */
      );
    } else {
      getter = () => {
        if (instance && instance.isUnmounted) {
          return;
        }
        if (cleanup) {
          cleanup();
        }
        return callWithAsyncErrorHandling(source, instance, 3, [onCleanup]);
      };
    }
  } else {
    getter = NOOP;
  }
  if (cb && deep) {
    const baseGetter = getter;
    getter = () => traverse(baseGetter());
  }
  let cleanup;
  let onCleanup = (fn) => {
    cleanup = effect2.onStop = () => {
      callWithErrorHandling(
        fn,
        instance,
        4
        /* ErrorCodes.WATCH_CLEANUP */
      );
    };
  };
  let ssrCleanup;
  if (isInSSRComponentSetup) {
    onCleanup = NOOP;
    if (!cb) {
      getter();
    } else if (immediate) {
      callWithAsyncErrorHandling(cb, instance, 3, [
        getter(),
        isMultiSource ? [] : void 0,
        onCleanup
      ]);
    }
    if (flush === "sync") {
      const ctx = useSSRContext();
      ssrCleanup = ctx.__watcherHandles || (ctx.__watcherHandles = []);
    } else {
      return NOOP;
    }
  }
  let oldValue = isMultiSource ? new Array(source.length).fill(INITIAL_WATCHER_VALUE) : INITIAL_WATCHER_VALUE;
  const job = () => {
    if (!effect2.active) {
      return;
    }
    if (cb) {
      const newValue = effect2.run();
      if (deep || forceTrigger || (isMultiSource ? newValue.some((v2, i2) => hasChanged(v2, oldValue[i2])) : hasChanged(newValue, oldValue)) || false) {
        if (cleanup) {
          cleanup();
        }
        callWithAsyncErrorHandling(cb, instance, 3, [
          newValue,
          // pass undefined as the old value when it's changed for the first time
          oldValue === INITIAL_WATCHER_VALUE ? void 0 : isMultiSource && oldValue[0] === INITIAL_WATCHER_VALUE ? [] : oldValue,
          onCleanup
        ]);
        oldValue = newValue;
      }
    } else {
      effect2.run();
    }
  };
  job.allowRecurse = !!cb;
  let scheduler;
  if (flush === "sync") {
    scheduler = job;
  } else if (flush === "post") {
    scheduler = () => queuePostRenderEffect(job, instance && instance.suspense);
  } else {
    job.pre = true;
    if (instance)
      job.id = instance.uid;
    scheduler = () => queueJob(job);
  }
  const effect2 = new ReactiveEffect(getter, scheduler);
  if (cb) {
    if (immediate) {
      job();
    } else {
      oldValue = effect2.run();
    }
  } else if (flush === "post") {
    queuePostRenderEffect(effect2.run.bind(effect2), instance && instance.suspense);
  } else {
    effect2.run();
  }
  const unwatch = () => {
    effect2.stop();
    if (instance && instance.scope) {
      remove(instance.scope.effects, effect2);
    }
  };
  if (ssrCleanup)
    ssrCleanup.push(unwatch);
  return unwatch;
}
function instanceWatch(source, value, options2) {
  const publicThis = this.proxy;
  const getter = isString$4(source) ? source.includes(".") ? createPathGetter(publicThis, source) : () => publicThis[source] : source.bind(publicThis, publicThis);
  let cb;
  if (isFunction$2(value)) {
    cb = value;
  } else {
    cb = value.handler;
    options2 = value;
  }
  const cur = currentInstance;
  setCurrentInstance(this);
  const res = doWatch(getter, cb.bind(publicThis), options2);
  if (cur) {
    setCurrentInstance(cur);
  } else {
    unsetCurrentInstance();
  }
  return res;
}
function createPathGetter(ctx, path) {
  const segments = path.split(".");
  return () => {
    let cur = ctx;
    for (let i2 = 0; i2 < segments.length && cur; i2++) {
      cur = cur[segments[i2]];
    }
    return cur;
  };
}
function traverse(value, seen2) {
  if (!isObject$5(value) || value[
    "__v_skip"
    /* ReactiveFlags.SKIP */
  ]) {
    return value;
  }
  seen2 = seen2 || /* @__PURE__ */ new Set();
  if (seen2.has(value)) {
    return value;
  }
  seen2.add(value);
  if (isRef(value)) {
    traverse(value.value, seen2);
  } else if (isArray$2(value)) {
    for (let i2 = 0; i2 < value.length; i2++) {
      traverse(value[i2], seen2);
    }
  } else if (isSet(value) || isMap(value)) {
    value.forEach((v2) => {
      traverse(v2, seen2);
    });
  } else if (isPlainObject$2(value)) {
    for (const key in value) {
      traverse(value[key], seen2);
    }
  }
  return value;
}
function useTransitionState() {
  const state = {
    isMounted: false,
    isLeaving: false,
    isUnmounting: false,
    leavingVNodes: /* @__PURE__ */ new Map()
  };
  onMounted(() => {
    state.isMounted = true;
  });
  onBeforeUnmount(() => {
    state.isUnmounting = true;
  });
  return state;
}
const TransitionHookValidator = [Function, Array];
const BaseTransitionImpl = {
  name: `BaseTransition`,
  props: {
    mode: String,
    appear: Boolean,
    persisted: Boolean,
    // enter
    onBeforeEnter: TransitionHookValidator,
    onEnter: TransitionHookValidator,
    onAfterEnter: TransitionHookValidator,
    onEnterCancelled: TransitionHookValidator,
    // leave
    onBeforeLeave: TransitionHookValidator,
    onLeave: TransitionHookValidator,
    onAfterLeave: TransitionHookValidator,
    onLeaveCancelled: TransitionHookValidator,
    // appear
    onBeforeAppear: TransitionHookValidator,
    onAppear: TransitionHookValidator,
    onAfterAppear: TransitionHookValidator,
    onAppearCancelled: TransitionHookValidator
  },
  setup(props, { slots }) {
    const instance = getCurrentInstance();
    const state = useTransitionState();
    let prevTransitionKey;
    return () => {
      const children = slots.default && getTransitionRawChildren(slots.default(), true);
      if (!children || !children.length) {
        return;
      }
      let child = children[0];
      if (children.length > 1) {
        for (const c2 of children) {
          if (c2.type !== Comment) {
            child = c2;
            break;
          }
        }
      }
      const rawProps = toRaw(props);
      const { mode } = rawProps;
      if (state.isLeaving) {
        return emptyPlaceholder(child);
      }
      const innerChild = getKeepAliveChild(child);
      if (!innerChild) {
        return emptyPlaceholder(child);
      }
      const enterHooks = resolveTransitionHooks(innerChild, rawProps, state, instance);
      setTransitionHooks(innerChild, enterHooks);
      const oldChild = instance.subTree;
      const oldInnerChild = oldChild && getKeepAliveChild(oldChild);
      let transitionKeyChanged = false;
      const { getTransitionKey } = innerChild.type;
      if (getTransitionKey) {
        const key = getTransitionKey();
        if (prevTransitionKey === void 0) {
          prevTransitionKey = key;
        } else if (key !== prevTransitionKey) {
          prevTransitionKey = key;
          transitionKeyChanged = true;
        }
      }
      if (oldInnerChild && oldInnerChild.type !== Comment && (!isSameVNodeType(innerChild, oldInnerChild) || transitionKeyChanged)) {
        const leavingHooks = resolveTransitionHooks(oldInnerChild, rawProps, state, instance);
        setTransitionHooks(oldInnerChild, leavingHooks);
        if (mode === "out-in") {
          state.isLeaving = true;
          leavingHooks.afterLeave = () => {
            state.isLeaving = false;
            if (instance.update.active !== false) {
              instance.update();
            }
          };
          return emptyPlaceholder(child);
        } else if (mode === "in-out" && innerChild.type !== Comment) {
          leavingHooks.delayLeave = (el2, earlyRemove, delayedLeave) => {
            const leavingVNodesCache = getLeavingNodesForType(state, oldInnerChild);
            leavingVNodesCache[String(oldInnerChild.key)] = oldInnerChild;
            el2._leaveCb = () => {
              earlyRemove();
              el2._leaveCb = void 0;
              delete enterHooks.delayedLeave;
            };
            enterHooks.delayedLeave = delayedLeave;
          };
        }
      }
      return child;
    };
  }
};
const BaseTransition = BaseTransitionImpl;
function getLeavingNodesForType(state, vnode) {
  const { leavingVNodes } = state;
  let leavingVNodesCache = leavingVNodes.get(vnode.type);
  if (!leavingVNodesCache) {
    leavingVNodesCache = /* @__PURE__ */ Object.create(null);
    leavingVNodes.set(vnode.type, leavingVNodesCache);
  }
  return leavingVNodesCache;
}
function resolveTransitionHooks(vnode, props, state, instance) {
  const { appear, mode, persisted = false, onBeforeEnter, onEnter, onAfterEnter, onEnterCancelled, onBeforeLeave, onLeave, onAfterLeave, onLeaveCancelled, onBeforeAppear, onAppear, onAfterAppear, onAppearCancelled } = props;
  const key = String(vnode.key);
  const leavingVNodesCache = getLeavingNodesForType(state, vnode);
  const callHook2 = (hook, args) => {
    hook && callWithAsyncErrorHandling(hook, instance, 9, args);
  };
  const callAsyncHook = (hook, args) => {
    const done = args[1];
    callHook2(hook, args);
    if (isArray$2(hook)) {
      if (hook.every((hook2) => hook2.length <= 1))
        done();
    } else if (hook.length <= 1) {
      done();
    }
  };
  const hooks = {
    mode,
    persisted,
    beforeEnter(el2) {
      let hook = onBeforeEnter;
      if (!state.isMounted) {
        if (appear) {
          hook = onBeforeAppear || onBeforeEnter;
        } else {
          return;
        }
      }
      if (el2._leaveCb) {
        el2._leaveCb(
          true
          /* cancelled */
        );
      }
      const leavingVNode = leavingVNodesCache[key];
      if (leavingVNode && isSameVNodeType(vnode, leavingVNode) && leavingVNode.el._leaveCb) {
        leavingVNode.el._leaveCb();
      }
      callHook2(hook, [el2]);
    },
    enter(el2) {
      let hook = onEnter;
      let afterHook = onAfterEnter;
      let cancelHook = onEnterCancelled;
      if (!state.isMounted) {
        if (appear) {
          hook = onAppear || onEnter;
          afterHook = onAfterAppear || onAfterEnter;
          cancelHook = onAppearCancelled || onEnterCancelled;
        } else {
          return;
        }
      }
      let called = false;
      const done = el2._enterCb = (cancelled) => {
        if (called)
          return;
        called = true;
        if (cancelled) {
          callHook2(cancelHook, [el2]);
        } else {
          callHook2(afterHook, [el2]);
        }
        if (hooks.delayedLeave) {
          hooks.delayedLeave();
        }
        el2._enterCb = void 0;
      };
      if (hook) {
        callAsyncHook(hook, [el2, done]);
      } else {
        done();
      }
    },
    leave(el2, remove2) {
      const key2 = String(vnode.key);
      if (el2._enterCb) {
        el2._enterCb(
          true
          /* cancelled */
        );
      }
      if (state.isUnmounting) {
        return remove2();
      }
      callHook2(onBeforeLeave, [el2]);
      let called = false;
      const done = el2._leaveCb = (cancelled) => {
        if (called)
          return;
        called = true;
        remove2();
        if (cancelled) {
          callHook2(onLeaveCancelled, [el2]);
        } else {
          callHook2(onAfterLeave, [el2]);
        }
        el2._leaveCb = void 0;
        if (leavingVNodesCache[key2] === vnode) {
          delete leavingVNodesCache[key2];
        }
      };
      leavingVNodesCache[key2] = vnode;
      if (onLeave) {
        callAsyncHook(onLeave, [el2, done]);
      } else {
        done();
      }
    },
    clone(vnode2) {
      return resolveTransitionHooks(vnode2, props, state, instance);
    }
  };
  return hooks;
}
function emptyPlaceholder(vnode) {
  if (isKeepAlive(vnode)) {
    vnode = cloneVNode(vnode);
    vnode.children = null;
    return vnode;
  }
}
function getKeepAliveChild(vnode) {
  return isKeepAlive(vnode) ? vnode.children ? vnode.children[0] : void 0 : vnode;
}
function setTransitionHooks(vnode, hooks) {
  if (vnode.shapeFlag & 6 && vnode.component) {
    setTransitionHooks(vnode.component.subTree, hooks);
  } else if (vnode.shapeFlag & 128) {
    vnode.ssContent.transition = hooks.clone(vnode.ssContent);
    vnode.ssFallback.transition = hooks.clone(vnode.ssFallback);
  } else {
    vnode.transition = hooks;
  }
}
function getTransitionRawChildren(children, keepComment = false, parentKey) {
  let ret = [];
  let keyedFragmentCount = 0;
  for (let i2 = 0; i2 < children.length; i2++) {
    let child = children[i2];
    const key = parentKey == null ? child.key : String(parentKey) + String(child.key != null ? child.key : i2);
    if (child.type === Fragment) {
      if (child.patchFlag & 128)
        keyedFragmentCount++;
      ret = ret.concat(getTransitionRawChildren(child.children, keepComment, key));
    } else if (keepComment || child.type !== Comment) {
      ret.push(key != null ? cloneVNode(child, { key }) : child);
    }
  }
  if (keyedFragmentCount > 1) {
    for (let i2 = 0; i2 < ret.length; i2++) {
      ret[i2].patchFlag = -2;
    }
  }
  return ret;
}
function defineComponent(options2) {
  return isFunction$2(options2) ? { setup: options2, name: options2.name } : options2;
}
const isAsyncWrapper = (i2) => !!i2.type.__asyncLoader;
function defineAsyncComponent(source) {
  if (isFunction$2(source)) {
    source = { loader: source };
  }
  const {
    loader,
    loadingComponent,
    errorComponent,
    delay = 200,
    timeout,
    // undefined = never times out
    suspensible = true,
    onError: userOnError
  } = source;
  let pendingRequest = null;
  let resolvedComp;
  let retries = 0;
  const retry = () => {
    retries++;
    pendingRequest = null;
    return load();
  };
  const load = () => {
    let thisRequest;
    return pendingRequest || (thisRequest = pendingRequest = loader().catch((err) => {
      err = err instanceof Error ? err : new Error(String(err));
      if (userOnError) {
        return new Promise((resolve22, reject) => {
          const userRetry = () => resolve22(retry());
          const userFail = () => reject(err);
          userOnError(err, userRetry, userFail, retries + 1);
        });
      } else {
        throw err;
      }
    }).then((comp2) => {
      if (thisRequest !== pendingRequest && pendingRequest) {
        return pendingRequest;
      }
      if (comp2 && (comp2.__esModule || comp2[Symbol.toStringTag] === "Module")) {
        comp2 = comp2.default;
      }
      resolvedComp = comp2;
      return comp2;
    }));
  };
  return defineComponent({
    name: "AsyncComponentWrapper",
    __asyncLoader: load,
    get __asyncResolved() {
      return resolvedComp;
    },
    setup() {
      const instance = currentInstance;
      if (resolvedComp) {
        return () => createInnerComp(resolvedComp, instance);
      }
      const onError = (err) => {
        pendingRequest = null;
        handleError(
          err,
          instance,
          13,
          !errorComponent
          /* do not throw in dev if user provided error component */
        );
      };
      if (suspensible && instance.suspense || isInSSRComponentSetup) {
        return load().then((comp2) => {
          return () => createInnerComp(comp2, instance);
        }).catch((err) => {
          onError(err);
          return () => errorComponent ? createVNode(errorComponent, {
            error: err
          }) : null;
        });
      }
      const loaded = ref(false);
      const error2 = ref();
      const delayed = ref(!!delay);
      if (delay) {
        setTimeout(() => {
          delayed.value = false;
        }, delay);
      }
      if (timeout != null) {
        setTimeout(() => {
          if (!loaded.value && !error2.value) {
            const err = new Error(`Async component timed out after ${timeout}ms.`);
            onError(err);
            error2.value = err;
          }
        }, timeout);
      }
      load().then(() => {
        loaded.value = true;
        if (instance.parent && isKeepAlive(instance.parent.vnode)) {
          queueJob(instance.parent.update);
        }
      }).catch((err) => {
        onError(err);
        error2.value = err;
      });
      return () => {
        if (loaded.value && resolvedComp) {
          return createInnerComp(resolvedComp, instance);
        } else if (error2.value && errorComponent) {
          return createVNode(errorComponent, {
            error: error2.value
          });
        } else if (loadingComponent && !delayed.value) {
          return createVNode(loadingComponent);
        }
      };
    }
  });
}
function createInnerComp(comp2, parent) {
  const { ref: ref3, props, children, ce } = parent.vnode;
  const vnode = createVNode(comp2, props, children);
  vnode.ref = ref3;
  vnode.ce = ce;
  delete parent.vnode.ce;
  return vnode;
}
const isKeepAlive = (vnode) => vnode.type.__isKeepAlive;
const KeepAliveImpl = {
  name: `KeepAlive`,
  // Marker for special handling inside the renderer. We are not using a ===
  // check directly on KeepAlive in the renderer, because importing it directly
  // would prevent it from being tree-shaken.
  __isKeepAlive: true,
  props: {
    include: [String, RegExp, Array],
    exclude: [String, RegExp, Array],
    max: [String, Number]
  },
  setup(props, { slots }) {
    const instance = getCurrentInstance();
    const sharedContext = instance.ctx;
    if (!sharedContext.renderer) {
      return () => {
        const children = slots.default && slots.default();
        return children && children.length === 1 ? children[0] : children;
      };
    }
    const cache2 = /* @__PURE__ */ new Map();
    const keys = /* @__PURE__ */ new Set();
    let current = null;
    const parentSuspense = instance.suspense;
    const { renderer: { p: patch, m: move, um: _unmount, o: { createElement } } } = sharedContext;
    const storageContainer = createElement("div");
    sharedContext.activate = (vnode, container, anchor, isSVG, optimized) => {
      const instance2 = vnode.component;
      move(vnode, container, anchor, 0, parentSuspense);
      patch(instance2.vnode, vnode, container, anchor, instance2, parentSuspense, isSVG, vnode.slotScopeIds, optimized);
      queuePostRenderEffect(() => {
        instance2.isDeactivated = false;
        if (instance2.a) {
          invokeArrayFns(instance2.a);
        }
        const vnodeHook = vnode.props && vnode.props.onVnodeMounted;
        if (vnodeHook) {
          invokeVNodeHook(vnodeHook, instance2.parent, vnode);
        }
      }, parentSuspense);
    };
    sharedContext.deactivate = (vnode) => {
      const instance2 = vnode.component;
      move(vnode, storageContainer, null, 1, parentSuspense);
      queuePostRenderEffect(() => {
        if (instance2.da) {
          invokeArrayFns(instance2.da);
        }
        const vnodeHook = vnode.props && vnode.props.onVnodeUnmounted;
        if (vnodeHook) {
          invokeVNodeHook(vnodeHook, instance2.parent, vnode);
        }
        instance2.isDeactivated = true;
      }, parentSuspense);
    };
    function unmount(vnode) {
      resetShapeFlag(vnode);
      _unmount(vnode, instance, parentSuspense, true);
    }
    function pruneCache(filter) {
      cache2.forEach((vnode, key) => {
        const name2 = getComponentName(vnode.type);
        if (name2 && (!filter || !filter(name2))) {
          pruneCacheEntry(key);
        }
      });
    }
    function pruneCacheEntry(key) {
      const cached = cache2.get(key);
      if (!current || !isSameVNodeType(cached, current)) {
        unmount(cached);
      } else if (current) {
        resetShapeFlag(current);
      }
      cache2.delete(key);
      keys.delete(key);
    }
    watch(
      () => [props.include, props.exclude],
      ([include, exclude]) => {
        include && pruneCache((name2) => matches$1(include, name2));
        exclude && pruneCache((name2) => !matches$1(exclude, name2));
      },
      // prune post-render after `current` has been updated
      { flush: "post", deep: true }
    );
    let pendingCacheKey = null;
    const cacheSubtree = () => {
      if (pendingCacheKey != null) {
        cache2.set(pendingCacheKey, getInnerChild(instance.subTree));
      }
    };
    onMounted(cacheSubtree);
    onUpdated(cacheSubtree);
    onBeforeUnmount(() => {
      cache2.forEach((cached) => {
        const { subTree, suspense } = instance;
        const vnode = getInnerChild(subTree);
        if (cached.type === vnode.type && cached.key === vnode.key) {
          resetShapeFlag(vnode);
          const da = vnode.component.da;
          da && queuePostRenderEffect(da, suspense);
          return;
        }
        unmount(cached);
      });
    });
    return () => {
      pendingCacheKey = null;
      if (!slots.default) {
        return null;
      }
      const children = slots.default();
      const rawVNode = children[0];
      if (children.length > 1) {
        current = null;
        return children;
      } else if (!isVNode$1(rawVNode) || !(rawVNode.shapeFlag & 4) && !(rawVNode.shapeFlag & 128)) {
        current = null;
        return rawVNode;
      }
      let vnode = getInnerChild(rawVNode);
      const comp2 = vnode.type;
      const name2 = getComponentName(isAsyncWrapper(vnode) ? vnode.type.__asyncResolved || {} : comp2);
      const { include, exclude, max: max2 } = props;
      if (include && (!name2 || !matches$1(include, name2)) || exclude && name2 && matches$1(exclude, name2)) {
        current = vnode;
        return rawVNode;
      }
      const key = vnode.key == null ? comp2 : vnode.key;
      const cachedVNode = cache2.get(key);
      if (vnode.el) {
        vnode = cloneVNode(vnode);
        if (rawVNode.shapeFlag & 128) {
          rawVNode.ssContent = vnode;
        }
      }
      pendingCacheKey = key;
      if (cachedVNode) {
        vnode.el = cachedVNode.el;
        vnode.component = cachedVNode.component;
        if (vnode.transition) {
          setTransitionHooks(vnode, vnode.transition);
        }
        vnode.shapeFlag |= 512;
        keys.delete(key);
        keys.add(key);
      } else {
        keys.add(key);
        if (max2 && keys.size > parseInt(max2, 10)) {
          pruneCacheEntry(keys.values().next().value);
        }
      }
      vnode.shapeFlag |= 256;
      current = vnode;
      return isSuspense(rawVNode.type) ? rawVNode : vnode;
    };
  }
};
const KeepAlive = KeepAliveImpl;
function matches$1(pattern, name2) {
  if (isArray$2(pattern)) {
    return pattern.some((p2) => matches$1(p2, name2));
  } else if (isString$4(pattern)) {
    return pattern.split(",").includes(name2);
  } else if (isRegExp$1(pattern)) {
    return pattern.test(name2);
  }
  return false;
}
function onActivated(hook, target2) {
  registerKeepAliveHook(hook, "a", target2);
}
function onDeactivated(hook, target2) {
  registerKeepAliveHook(hook, "da", target2);
}
function registerKeepAliveHook(hook, type2, target2 = currentInstance) {
  const wrappedHook = hook.__wdc || (hook.__wdc = () => {
    let current = target2;
    while (current) {
      if (current.isDeactivated) {
        return;
      }
      current = current.parent;
    }
    return hook();
  });
  injectHook(type2, wrappedHook, target2);
  if (target2) {
    let current = target2.parent;
    while (current && current.parent) {
      if (isKeepAlive(current.parent.vnode)) {
        injectToKeepAliveRoot(wrappedHook, type2, target2, current);
      }
      current = current.parent;
    }
  }
}
function injectToKeepAliveRoot(hook, type2, target2, keepAliveRoot) {
  const injected = injectHook(
    type2,
    hook,
    keepAliveRoot,
    true
    /* prepend */
  );
  onUnmounted(() => {
    remove(keepAliveRoot[type2], injected);
  }, target2);
}
function resetShapeFlag(vnode) {
  vnode.shapeFlag &= -257;
  vnode.shapeFlag &= -513;
}
function getInnerChild(vnode) {
  return vnode.shapeFlag & 128 ? vnode.ssContent : vnode;
}
function injectHook(type2, hook, target2 = currentInstance, prepend = false) {
  if (target2) {
    const hooks = target2[type2] || (target2[type2] = []);
    const wrappedHook = hook.__weh || (hook.__weh = (...args) => {
      if (target2.isUnmounted) {
        return;
      }
      pauseTracking();
      setCurrentInstance(target2);
      const res = callWithAsyncErrorHandling(hook, target2, type2, args);
      unsetCurrentInstance();
      resetTracking();
      return res;
    });
    if (prepend) {
      hooks.unshift(wrappedHook);
    } else {
      hooks.push(wrappedHook);
    }
    return wrappedHook;
  }
}
const createHook = (lifecycle) => (hook, target2 = currentInstance) => (
  // post-create lifecycle registrations are noops during SSR (except for serverPrefetch)
  (!isInSSRComponentSetup || lifecycle === "sp") && injectHook(lifecycle, (...args) => hook(...args), target2)
);
const onBeforeMount = createHook(
  "bm"
  /* LifecycleHooks.BEFORE_MOUNT */
);
const onMounted = createHook(
  "m"
  /* LifecycleHooks.MOUNTED */
);
const onBeforeUpdate = createHook(
  "bu"
  /* LifecycleHooks.BEFORE_UPDATE */
);
const onUpdated = createHook(
  "u"
  /* LifecycleHooks.UPDATED */
);
const onBeforeUnmount = createHook(
  "bum"
  /* LifecycleHooks.BEFORE_UNMOUNT */
);
const onUnmounted = createHook(
  "um"
  /* LifecycleHooks.UNMOUNTED */
);
const onServerPrefetch = createHook(
  "sp"
  /* LifecycleHooks.SERVER_PREFETCH */
);
const onRenderTriggered = createHook(
  "rtg"
  /* LifecycleHooks.RENDER_TRIGGERED */
);
const onRenderTracked = createHook(
  "rtc"
  /* LifecycleHooks.RENDER_TRACKED */
);
function onErrorCaptured(hook, target2 = currentInstance) {
  injectHook("ec", hook, target2);
}
function withDirectives(vnode, directives) {
  const internalInstance = currentRenderingInstance;
  if (internalInstance === null) {
    return vnode;
  }
  const instance = getExposeProxy(internalInstance) || internalInstance.proxy;
  const bindings = vnode.dirs || (vnode.dirs = []);
  for (let i2 = 0; i2 < directives.length; i2++) {
    let [dir, value, arg, modifiers = EMPTY_OBJ] = directives[i2];
    if (dir) {
      if (isFunction$2(dir)) {
        dir = {
          mounted: dir,
          updated: dir
        };
      }
      if (dir.deep) {
        traverse(value);
      }
      bindings.push({
        dir,
        instance,
        value,
        oldValue: void 0,
        arg,
        modifiers
      });
    }
  }
  return vnode;
}
function invokeDirectiveHook(vnode, prevVNode, instance, name2) {
  const bindings = vnode.dirs;
  const oldBindings = prevVNode && prevVNode.dirs;
  for (let i2 = 0; i2 < bindings.length; i2++) {
    const binding = bindings[i2];
    if (oldBindings) {
      binding.oldValue = oldBindings[i2].value;
    }
    let hook = binding.dir[name2];
    if (hook) {
      pauseTracking();
      callWithAsyncErrorHandling(hook, instance, 8, [
        vnode.el,
        binding,
        vnode,
        prevVNode
      ]);
      resetTracking();
    }
  }
}
const COMPONENTS = "components";
const DIRECTIVES = "directives";
function resolveComponent(name2, maybeSelfReference) {
  return resolveAsset(COMPONENTS, name2, true, maybeSelfReference) || name2;
}
const NULL_DYNAMIC_COMPONENT = Symbol();
function resolveDynamicComponent(component) {
  if (isString$4(component)) {
    return resolveAsset(COMPONENTS, component, false) || component;
  } else {
    return component || NULL_DYNAMIC_COMPONENT;
  }
}
function resolveDirective(name2) {
  return resolveAsset(DIRECTIVES, name2);
}
function resolveAsset(type2, name2, warnMissing = true, maybeSelfReference = false) {
  const instance = currentRenderingInstance || currentInstance;
  if (instance) {
    const Component = instance.type;
    if (type2 === COMPONENTS) {
      const selfName = getComponentName(
        Component,
        false
        /* do not include inferred name to avoid breaking existing code */
      );
      if (selfName && (selfName === name2 || selfName === camelize$1(name2) || selfName === capitalize$2(camelize$1(name2)))) {
        return Component;
      }
    }
    const res = (
      // local registration
      // check instance[type] first which is resolved for options API
      resolve(instance[type2] || Component[type2], name2) || // global registration
      resolve(instance.appContext[type2], name2)
    );
    if (!res && maybeSelfReference) {
      return Component;
    }
    return res;
  }
}
function resolve(registry, name2) {
  return registry && (registry[name2] || registry[camelize$1(name2)] || registry[capitalize$2(camelize$1(name2))]);
}
function renderList(source, renderItem, cache2, index) {
  let ret;
  const cached = cache2 && cache2[index];
  if (isArray$2(source) || isString$4(source)) {
    ret = new Array(source.length);
    for (let i2 = 0, l2 = source.length; i2 < l2; i2++) {
      ret[i2] = renderItem(source[i2], i2, void 0, cached && cached[i2]);
    }
  } else if (typeof source === "number") {
    ret = new Array(source);
    for (let i2 = 0; i2 < source; i2++) {
      ret[i2] = renderItem(i2 + 1, i2, void 0, cached && cached[i2]);
    }
  } else if (isObject$5(source)) {
    if (source[Symbol.iterator]) {
      ret = Array.from(source, (item, i2) => renderItem(item, i2, void 0, cached && cached[i2]));
    } else {
      const keys = Object.keys(source);
      ret = new Array(keys.length);
      for (let i2 = 0, l2 = keys.length; i2 < l2; i2++) {
        const key = keys[i2];
        ret[i2] = renderItem(source[key], key, i2, cached && cached[i2]);
      }
    }
  } else {
    ret = [];
  }
  if (cache2) {
    cache2[index] = ret;
  }
  return ret;
}
function createSlots(slots, dynamicSlots) {
  for (let i2 = 0; i2 < dynamicSlots.length; i2++) {
    const slot = dynamicSlots[i2];
    if (isArray$2(slot)) {
      for (let j2 = 0; j2 < slot.length; j2++) {
        slots[slot[j2].name] = slot[j2].fn;
      }
    } else if (slot) {
      slots[slot.name] = slot.key ? (...args) => {
        const res = slot.fn(...args);
        if (res)
          res.key = slot.key;
        return res;
      } : slot.fn;
    }
  }
  return slots;
}
function renderSlot(slots, name2, props = {}, fallback, noSlotted) {
  if (currentRenderingInstance.isCE || currentRenderingInstance.parent && isAsyncWrapper(currentRenderingInstance.parent) && currentRenderingInstance.parent.isCE) {
    if (name2 !== "default")
      props.name = name2;
    return createVNode("slot", props, fallback && fallback());
  }
  let slot = slots[name2];
  if (slot && slot._c) {
    slot._d = false;
  }
  openBlock();
  const validSlotContent = slot && ensureValidVNode(slot(props));
  const rendered = createBlock(
    Fragment,
    {
      key: props.key || // slot content array of a dynamic conditional slot may have a branch
      // key attached in the `createSlots` helper, respect that
      validSlotContent && validSlotContent.key || `_${name2}`
    },
    validSlotContent || (fallback ? fallback() : []),
    validSlotContent && slots._ === 1 ? 64 : -2
    /* PatchFlags.BAIL */
  );
  if (!noSlotted && rendered.scopeId) {
    rendered.slotScopeIds = [rendered.scopeId + "-s"];
  }
  if (slot && slot._c) {
    slot._d = true;
  }
  return rendered;
}
function ensureValidVNode(vnodes) {
  return vnodes.some((child) => {
    if (!isVNode$1(child))
      return true;
    if (child.type === Comment)
      return false;
    if (child.type === Fragment && !ensureValidVNode(child.children))
      return false;
    return true;
  }) ? vnodes : null;
}
function toHandlers(obj, preserveCaseIfNecessary) {
  const ret = {};
  for (const key in obj) {
    ret[preserveCaseIfNecessary && /[A-Z]/.test(key) ? `on:${key}` : toHandlerKey(key)] = obj[key];
  }
  return ret;
}
const getPublicInstance = (i2) => {
  if (!i2)
    return null;
  if (isStatefulComponent(i2))
    return getExposeProxy(i2) || i2.proxy;
  return getPublicInstance(i2.parent);
};
const publicPropertiesMap = (
  // Move PURE marker to new line to workaround compiler discarding it
  // due to type annotation
  /* @__PURE__ */ extend(/* @__PURE__ */ Object.create(null), {
    $: (i2) => i2,
    $el: (i2) => i2.vnode.el,
    $data: (i2) => i2.data,
    $props: (i2) => i2.props,
    $attrs: (i2) => i2.attrs,
    $slots: (i2) => i2.slots,
    $refs: (i2) => i2.refs,
    $parent: (i2) => getPublicInstance(i2.parent),
    $root: (i2) => getPublicInstance(i2.root),
    $emit: (i2) => i2.emit,
    $options: (i2) => resolveMergedOptions(i2),
    $forceUpdate: (i2) => i2.f || (i2.f = () => queueJob(i2.update)),
    $nextTick: (i2) => i2.n || (i2.n = nextTick.bind(i2.proxy)),
    $watch: (i2) => instanceWatch.bind(i2)
  })
);
const hasSetupBinding = (state, key) => state !== EMPTY_OBJ && !state.__isScriptSetup && hasOwn$1(state, key);
const PublicInstanceProxyHandlers = {
  get({ _: instance }, key) {
    const { ctx, setupState, data, props, accessCache, type: type2, appContext } = instance;
    let normalizedProps;
    if (key[0] !== "$") {
      const n2 = accessCache[key];
      if (n2 !== void 0) {
        switch (n2) {
          case 1:
            return setupState[key];
          case 2:
            return data[key];
          case 4:
            return ctx[key];
          case 3:
            return props[key];
        }
      } else if (hasSetupBinding(setupState, key)) {
        accessCache[key] = 1;
        return setupState[key];
      } else if (data !== EMPTY_OBJ && hasOwn$1(data, key)) {
        accessCache[key] = 2;
        return data[key];
      } else if (
        // only cache other properties when instance has declared (thus stable)
        // props
        (normalizedProps = instance.propsOptions[0]) && hasOwn$1(normalizedProps, key)
      ) {
        accessCache[key] = 3;
        return props[key];
      } else if (ctx !== EMPTY_OBJ && hasOwn$1(ctx, key)) {
        accessCache[key] = 4;
        return ctx[key];
      } else if (shouldCacheAccess) {
        accessCache[key] = 0;
      }
    }
    const publicGetter = publicPropertiesMap[key];
    let cssModule, globalProperties;
    if (publicGetter) {
      if (key === "$attrs") {
        track(instance, "get", key);
      }
      return publicGetter(instance);
    } else if (
      // css module (injected by vue-loader)
      (cssModule = type2.__cssModules) && (cssModule = cssModule[key])
    ) {
      return cssModule;
    } else if (ctx !== EMPTY_OBJ && hasOwn$1(ctx, key)) {
      accessCache[key] = 4;
      return ctx[key];
    } else if (
      // global properties
      globalProperties = appContext.config.globalProperties, hasOwn$1(globalProperties, key)
    ) {
      {
        return globalProperties[key];
      }
    } else ;
  },
  set({ _: instance }, key, value) {
    const { data, setupState, ctx } = instance;
    if (hasSetupBinding(setupState, key)) {
      setupState[key] = value;
      return true;
    } else if (data !== EMPTY_OBJ && hasOwn$1(data, key)) {
      data[key] = value;
      return true;
    } else if (hasOwn$1(instance.props, key)) {
      return false;
    }
    if (key[0] === "$" && key.slice(1) in instance) {
      return false;
    } else {
      {
        ctx[key] = value;
      }
    }
    return true;
  },
  has({ _: { data, setupState, accessCache, ctx, appContext, propsOptions } }, key) {
    let normalizedProps;
    return !!accessCache[key] || data !== EMPTY_OBJ && hasOwn$1(data, key) || hasSetupBinding(setupState, key) || (normalizedProps = propsOptions[0]) && hasOwn$1(normalizedProps, key) || hasOwn$1(ctx, key) || hasOwn$1(publicPropertiesMap, key) || hasOwn$1(appContext.config.globalProperties, key);
  },
  defineProperty(target2, key, descriptor) {
    if (descriptor.get != null) {
      target2._.accessCache[key] = 0;
    } else if (hasOwn$1(descriptor, "value")) {
      this.set(target2, key, descriptor.value, null);
    }
    return Reflect.defineProperty(target2, key, descriptor);
  }
};
const RuntimeCompiledPublicInstanceProxyHandlers = /* @__PURE__ */ extend({}, PublicInstanceProxyHandlers, {
  get(target2, key) {
    if (key === Symbol.unscopables) {
      return;
    }
    return PublicInstanceProxyHandlers.get(target2, key, target2);
  },
  has(_2, key) {
    const has2 = key[0] !== "_" && !isGloballyWhitelisted(key);
    return has2;
  }
});
let shouldCacheAccess = true;
function applyOptions(instance) {
  const options2 = resolveMergedOptions(instance);
  const publicThis = instance.proxy;
  const ctx = instance.ctx;
  shouldCacheAccess = false;
  if (options2.beforeCreate) {
    callHook$1(
      options2.beforeCreate,
      instance,
      "bc"
      /* LifecycleHooks.BEFORE_CREATE */
    );
  }
  const {
    // state
    data: dataOptions,
    computed: computedOptions,
    methods,
    watch: watchOptions,
    provide: provideOptions,
    inject: injectOptions,
    // lifecycle
    created,
    beforeMount,
    mounted,
    beforeUpdate,
    updated,
    activated,
    deactivated,
    beforeDestroy,
    beforeUnmount,
    destroyed,
    unmounted,
    render: render2,
    renderTracked,
    renderTriggered,
    errorCaptured,
    serverPrefetch,
    // public API
    expose,
    inheritAttrs,
    // assets
    components,
    directives,
    filters
  } = options2;
  const checkDuplicateProperties = null;
  if (injectOptions) {
    resolveInjections(injectOptions, ctx, checkDuplicateProperties, instance.appContext.config.unwrapInjectedRef);
  }
  if (methods) {
    for (const key in methods) {
      const methodHandler = methods[key];
      if (isFunction$2(methodHandler)) {
        {
          ctx[key] = methodHandler.bind(publicThis);
        }
      }
    }
  }
  if (dataOptions) {
    const data = dataOptions.call(publicThis, publicThis);
    if (!isObject$5(data)) ;
    else {
      instance.data = reactive(data);
    }
  }
  shouldCacheAccess = true;
  if (computedOptions) {
    for (const key in computedOptions) {
      const opt = computedOptions[key];
      const get2 = isFunction$2(opt) ? opt.bind(publicThis, publicThis) : isFunction$2(opt.get) ? opt.get.bind(publicThis, publicThis) : NOOP;
      const set2 = !isFunction$2(opt) && isFunction$2(opt.set) ? opt.set.bind(publicThis) : NOOP;
      const c2 = computed({
        get: get2,
        set: set2
      });
      Object.defineProperty(ctx, key, {
        enumerable: true,
        configurable: true,
        get: () => c2.value,
        set: (v2) => c2.value = v2
      });
    }
  }
  if (watchOptions) {
    for (const key in watchOptions) {
      createWatcher(watchOptions[key], ctx, publicThis, key);
    }
  }
  if (provideOptions) {
    const provides = isFunction$2(provideOptions) ? provideOptions.call(publicThis) : provideOptions;
    Reflect.ownKeys(provides).forEach((key) => {
      provide(key, provides[key]);
    });
  }
  if (created) {
    callHook$1(
      created,
      instance,
      "c"
      /* LifecycleHooks.CREATED */
    );
  }
  function registerLifecycleHook(register, hook) {
    if (isArray$2(hook)) {
      hook.forEach((_hook) => register(_hook.bind(publicThis)));
    } else if (hook) {
      register(hook.bind(publicThis));
    }
  }
  registerLifecycleHook(onBeforeMount, beforeMount);
  registerLifecycleHook(onMounted, mounted);
  registerLifecycleHook(onBeforeUpdate, beforeUpdate);
  registerLifecycleHook(onUpdated, updated);
  registerLifecycleHook(onActivated, activated);
  registerLifecycleHook(onDeactivated, deactivated);
  registerLifecycleHook(onErrorCaptured, errorCaptured);
  registerLifecycleHook(onRenderTracked, renderTracked);
  registerLifecycleHook(onRenderTriggered, renderTriggered);
  registerLifecycleHook(onBeforeUnmount, beforeUnmount);
  registerLifecycleHook(onUnmounted, unmounted);
  registerLifecycleHook(onServerPrefetch, serverPrefetch);
  if (isArray$2(expose)) {
    if (expose.length) {
      const exposed = instance.exposed || (instance.exposed = {});
      expose.forEach((key) => {
        Object.defineProperty(exposed, key, {
          get: () => publicThis[key],
          set: (val) => publicThis[key] = val
        });
      });
    } else if (!instance.exposed) {
      instance.exposed = {};
    }
  }
  if (render2 && instance.render === NOOP) {
    instance.render = render2;
  }
  if (inheritAttrs != null) {
    instance.inheritAttrs = inheritAttrs;
  }
  if (components)
    instance.components = components;
  if (directives)
    instance.directives = directives;
}
function resolveInjections(injectOptions, ctx, checkDuplicateProperties = NOOP, unwrapRef = false) {
  if (isArray$2(injectOptions)) {
    injectOptions = normalizeInject(injectOptions);
  }
  for (const key in injectOptions) {
    const opt = injectOptions[key];
    let injected;
    if (isObject$5(opt)) {
      if ("default" in opt) {
        injected = inject(
          opt.from || key,
          opt.default,
          true
          /* treat default function as factory */
        );
      } else {
        injected = inject(opt.from || key);
      }
    } else {
      injected = inject(opt);
    }
    if (isRef(injected)) {
      if (unwrapRef) {
        Object.defineProperty(ctx, key, {
          enumerable: true,
          configurable: true,
          get: () => injected.value,
          set: (v2) => injected.value = v2
        });
      } else {
        ctx[key] = injected;
      }
    } else {
      ctx[key] = injected;
    }
  }
}
function callHook$1(hook, instance, type2) {
  callWithAsyncErrorHandling(isArray$2(hook) ? hook.map((h2) => h2.bind(instance.proxy)) : hook.bind(instance.proxy), instance, type2);
}
function createWatcher(raw, ctx, publicThis, key) {
  const getter = key.includes(".") ? createPathGetter(publicThis, key) : () => publicThis[key];
  if (isString$4(raw)) {
    const handler = ctx[raw];
    if (isFunction$2(handler)) {
      watch(getter, handler);
    }
  } else if (isFunction$2(raw)) {
    watch(getter, raw.bind(publicThis));
  } else if (isObject$5(raw)) {
    if (isArray$2(raw)) {
      raw.forEach((r2) => createWatcher(r2, ctx, publicThis, key));
    } else {
      const handler = isFunction$2(raw.handler) ? raw.handler.bind(publicThis) : ctx[raw.handler];
      if (isFunction$2(handler)) {
        watch(getter, handler, raw);
      }
    }
  } else ;
}
function resolveMergedOptions(instance) {
  const base2 = instance.type;
  const { mixins, extends: extendsOptions } = base2;
  const { mixins: globalMixins, optionsCache: cache2, config: { optionMergeStrategies } } = instance.appContext;
  const cached = cache2.get(base2);
  let resolved;
  if (cached) {
    resolved = cached;
  } else if (!globalMixins.length && !mixins && !extendsOptions) {
    {
      resolved = base2;
    }
  } else {
    resolved = {};
    if (globalMixins.length) {
      globalMixins.forEach((m2) => mergeOptions(resolved, m2, optionMergeStrategies, true));
    }
    mergeOptions(resolved, base2, optionMergeStrategies);
  }
  if (isObject$5(base2)) {
    cache2.set(base2, resolved);
  }
  return resolved;
}
function mergeOptions(to, from, strats, asMixin = false) {
  const { mixins, extends: extendsOptions } = from;
  if (extendsOptions) {
    mergeOptions(to, extendsOptions, strats, true);
  }
  if (mixins) {
    mixins.forEach((m2) => mergeOptions(to, m2, strats, true));
  }
  for (const key in from) {
    if (asMixin && key === "expose") ;
    else {
      const strat = internalOptionMergeStrats[key] || strats && strats[key];
      to[key] = strat ? strat(to[key], from[key]) : from[key];
    }
  }
  return to;
}
const internalOptionMergeStrats = {
  data: mergeDataFn,
  props: mergeObjectOptions,
  emits: mergeObjectOptions,
  // objects
  methods: mergeObjectOptions,
  computed: mergeObjectOptions,
  // lifecycle
  beforeCreate: mergeAsArray$1,
  created: mergeAsArray$1,
  beforeMount: mergeAsArray$1,
  mounted: mergeAsArray$1,
  beforeUpdate: mergeAsArray$1,
  updated: mergeAsArray$1,
  beforeDestroy: mergeAsArray$1,
  beforeUnmount: mergeAsArray$1,
  destroyed: mergeAsArray$1,
  unmounted: mergeAsArray$1,
  activated: mergeAsArray$1,
  deactivated: mergeAsArray$1,
  errorCaptured: mergeAsArray$1,
  serverPrefetch: mergeAsArray$1,
  // assets
  components: mergeObjectOptions,
  directives: mergeObjectOptions,
  // watch
  watch: mergeWatchOptions,
  // provide / inject
  provide: mergeDataFn,
  inject: mergeInject
};
function mergeDataFn(to, from) {
  if (!from) {
    return to;
  }
  if (!to) {
    return from;
  }
  return function mergedDataFn() {
    return extend(isFunction$2(to) ? to.call(this, this) : to, isFunction$2(from) ? from.call(this, this) : from);
  };
}
function mergeInject(to, from) {
  return mergeObjectOptions(normalizeInject(to), normalizeInject(from));
}
function normalizeInject(raw) {
  if (isArray$2(raw)) {
    const res = {};
    for (let i2 = 0; i2 < raw.length; i2++) {
      res[raw[i2]] = raw[i2];
    }
    return res;
  }
  return raw;
}
function mergeAsArray$1(to, from) {
  return to ? [...new Set([].concat(to, from))] : from;
}
function mergeObjectOptions(to, from) {
  return to ? extend(extend(/* @__PURE__ */ Object.create(null), to), from) : from;
}
function mergeWatchOptions(to, from) {
  if (!to)
    return from;
  if (!from)
    return to;
  const merged = extend(/* @__PURE__ */ Object.create(null), to);
  for (const key in from) {
    merged[key] = mergeAsArray$1(to[key], from[key]);
  }
  return merged;
}
function initProps(instance, rawProps, isStateful, isSSR = false) {
  const props = {};
  const attrs = {};
  def(attrs, InternalObjectKey, 1);
  instance.propsDefaults = /* @__PURE__ */ Object.create(null);
  setFullProps(instance, rawProps, props, attrs);
  for (const key in instance.propsOptions[0]) {
    if (!(key in props)) {
      props[key] = void 0;
    }
  }
  if (isStateful) {
    instance.props = isSSR ? props : shallowReactive(props);
  } else {
    if (!instance.type.props) {
      instance.props = attrs;
    } else {
      instance.props = props;
    }
  }
  instance.attrs = attrs;
}
function updateProps(instance, rawProps, rawPrevProps, optimized) {
  const { props, attrs, vnode: { patchFlag } } = instance;
  const rawCurrentProps = toRaw(props);
  const [options2] = instance.propsOptions;
  let hasAttrsChanged = false;
  if (
    // always force full diff in dev
    // - #1942 if hmr is enabled with sfc component
    // - vite#872 non-sfc component used by sfc component
    (optimized || patchFlag > 0) && !(patchFlag & 16)
  ) {
    if (patchFlag & 8) {
      const propsToUpdate = instance.vnode.dynamicProps;
      for (let i2 = 0; i2 < propsToUpdate.length; i2++) {
        let key = propsToUpdate[i2];
        if (isEmitListener(instance.emitsOptions, key)) {
          continue;
        }
        const value = rawProps[key];
        if (options2) {
          if (hasOwn$1(attrs, key)) {
            if (value !== attrs[key]) {
              attrs[key] = value;
              hasAttrsChanged = true;
            }
          } else {
            const camelizedKey = camelize$1(key);
            props[camelizedKey] = resolvePropValue(
              options2,
              rawCurrentProps,
              camelizedKey,
              value,
              instance,
              false
              /* isAbsent */
            );
          }
        } else {
          if (value !== attrs[key]) {
            attrs[key] = value;
            hasAttrsChanged = true;
          }
        }
      }
    }
  } else {
    if (setFullProps(instance, rawProps, props, attrs)) {
      hasAttrsChanged = true;
    }
    let kebabKey;
    for (const key in rawCurrentProps) {
      if (!rawProps || // for camelCase
      !hasOwn$1(rawProps, key) && // it's possible the original props was passed in as kebab-case
      // and converted to camelCase (#955)
      ((kebabKey = hyphenate(key)) === key || !hasOwn$1(rawProps, kebabKey))) {
        if (options2) {
          if (rawPrevProps && // for camelCase
          (rawPrevProps[key] !== void 0 || // for kebab-case
          rawPrevProps[kebabKey] !== void 0)) {
            props[key] = resolvePropValue(
              options2,
              rawCurrentProps,
              key,
              void 0,
              instance,
              true
              /* isAbsent */
            );
          }
        } else {
          delete props[key];
        }
      }
    }
    if (attrs !== rawCurrentProps) {
      for (const key in attrs) {
        if (!rawProps || !hasOwn$1(rawProps, key) && true) {
          delete attrs[key];
          hasAttrsChanged = true;
        }
      }
    }
  }
  if (hasAttrsChanged) {
    trigger(instance, "set", "$attrs");
  }
}
function setFullProps(instance, rawProps, props, attrs) {
  const [options2, needCastKeys] = instance.propsOptions;
  let hasAttrsChanged = false;
  let rawCastValues;
  if (rawProps) {
    for (let key in rawProps) {
      if (isReservedProp(key)) {
        continue;
      }
      const value = rawProps[key];
      let camelKey;
      if (options2 && hasOwn$1(options2, camelKey = camelize$1(key))) {
        if (!needCastKeys || !needCastKeys.includes(camelKey)) {
          props[camelKey] = value;
        } else {
          (rawCastValues || (rawCastValues = {}))[camelKey] = value;
        }
      } else if (!isEmitListener(instance.emitsOptions, key)) {
        if (!(key in attrs) || value !== attrs[key]) {
          attrs[key] = value;
          hasAttrsChanged = true;
        }
      }
    }
  }
  if (needCastKeys) {
    const rawCurrentProps = toRaw(props);
    const castValues = rawCastValues || EMPTY_OBJ;
    for (let i2 = 0; i2 < needCastKeys.length; i2++) {
      const key = needCastKeys[i2];
      props[key] = resolvePropValue(options2, rawCurrentProps, key, castValues[key], instance, !hasOwn$1(castValues, key));
    }
  }
  return hasAttrsChanged;
}
function resolvePropValue(options2, props, key, value, instance, isAbsent) {
  const opt = options2[key];
  if (opt != null) {
    const hasDefault = hasOwn$1(opt, "default");
    if (hasDefault && value === void 0) {
      const defaultValue = opt.default;
      if (opt.type !== Function && isFunction$2(defaultValue)) {
        const { propsDefaults } = instance;
        if (key in propsDefaults) {
          value = propsDefaults[key];
        } else {
          setCurrentInstance(instance);
          value = propsDefaults[key] = defaultValue.call(null, props);
          unsetCurrentInstance();
        }
      } else {
        value = defaultValue;
      }
    }
    if (opt[
      0
      /* BooleanFlags.shouldCast */
    ]) {
      if (isAbsent && !hasDefault) {
        value = false;
      } else if (opt[
        1
        /* BooleanFlags.shouldCastTrue */
      ] && (value === "" || value === hyphenate(key))) {
        value = true;
      }
    }
  }
  return value;
}
function normalizePropsOptions(comp2, appContext, asMixin = false) {
  const cache2 = appContext.propsCache;
  const cached = cache2.get(comp2);
  if (cached) {
    return cached;
  }
  const raw = comp2.props;
  const normalized = {};
  const needCastKeys = [];
  let hasExtends = false;
  if (!isFunction$2(comp2)) {
    const extendProps = (raw2) => {
      hasExtends = true;
      const [props, keys] = normalizePropsOptions(raw2, appContext, true);
      extend(normalized, props);
      if (keys)
        needCastKeys.push(...keys);
    };
    if (!asMixin && appContext.mixins.length) {
      appContext.mixins.forEach(extendProps);
    }
    if (comp2.extends) {
      extendProps(comp2.extends);
    }
    if (comp2.mixins) {
      comp2.mixins.forEach(extendProps);
    }
  }
  if (!raw && !hasExtends) {
    if (isObject$5(comp2)) {
      cache2.set(comp2, EMPTY_ARR);
    }
    return EMPTY_ARR;
  }
  if (isArray$2(raw)) {
    for (let i2 = 0; i2 < raw.length; i2++) {
      const normalizedKey = camelize$1(raw[i2]);
      if (validatePropName(normalizedKey)) {
        normalized[normalizedKey] = EMPTY_OBJ;
      }
    }
  } else if (raw) {
    for (const key in raw) {
      const normalizedKey = camelize$1(key);
      if (validatePropName(normalizedKey)) {
        const opt = raw[key];
        const prop2 = normalized[normalizedKey] = isArray$2(opt) || isFunction$2(opt) ? { type: opt } : Object.assign({}, opt);
        if (prop2) {
          const booleanIndex = getTypeIndex(Boolean, prop2.type);
          const stringIndex = getTypeIndex(String, prop2.type);
          prop2[
            0
            /* BooleanFlags.shouldCast */
          ] = booleanIndex > -1;
          prop2[
            1
            /* BooleanFlags.shouldCastTrue */
          ] = stringIndex < 0 || booleanIndex < stringIndex;
          if (booleanIndex > -1 || hasOwn$1(prop2, "default")) {
            needCastKeys.push(normalizedKey);
          }
        }
      }
    }
  }
  const res = [normalized, needCastKeys];
  if (isObject$5(comp2)) {
    cache2.set(comp2, res);
  }
  return res;
}
function validatePropName(key) {
  if (key[0] !== "$") {
    return true;
  }
  return false;
}
function getType(ctor) {
  const match2 = ctor && ctor.toString().match(/^\s*(function|class) (\w+)/);
  return match2 ? match2[2] : ctor === null ? "null" : "";
}
function isSameType(a2, b2) {
  return getType(a2) === getType(b2);
}
function getTypeIndex(type2, expectedTypes) {
  if (isArray$2(expectedTypes)) {
    return expectedTypes.findIndex((t2) => isSameType(t2, type2));
  } else if (isFunction$2(expectedTypes)) {
    return isSameType(expectedTypes, type2) ? 0 : -1;
  }
  return -1;
}
const isInternalKey = (key) => key[0] === "_" || key === "$stable";
const normalizeSlotValue = (value) => isArray$2(value) ? value.map(normalizeVNode) : [normalizeVNode(value)];
const normalizeSlot = (key, rawSlot, ctx) => {
  if (rawSlot._n) {
    return rawSlot;
  }
  const normalized = withCtx((...args) => {
    if (false) ;
    return normalizeSlotValue(rawSlot(...args));
  }, ctx);
  normalized._c = false;
  return normalized;
};
const normalizeObjectSlots = (rawSlots, slots, instance) => {
  const ctx = rawSlots._ctx;
  for (const key in rawSlots) {
    if (isInternalKey(key))
      continue;
    const value = rawSlots[key];
    if (isFunction$2(value)) {
      slots[key] = normalizeSlot(key, value, ctx);
    } else if (value != null) {
      const normalized = normalizeSlotValue(value);
      slots[key] = () => normalized;
    }
  }
};
const normalizeVNodeSlots = (instance, children) => {
  const normalized = normalizeSlotValue(children);
  instance.slots.default = () => normalized;
};
const initSlots = (instance, children) => {
  if (instance.vnode.shapeFlag & 32) {
    const type2 = children._;
    if (type2) {
      instance.slots = toRaw(children);
      def(children, "_", type2);
    } else {
      normalizeObjectSlots(children, instance.slots = {});
    }
  } else {
    instance.slots = {};
    if (children) {
      normalizeVNodeSlots(instance, children);
    }
  }
  def(instance.slots, InternalObjectKey, 1);
};
const updateSlots = (instance, children, optimized) => {
  const { vnode, slots } = instance;
  let needDeletionCheck = true;
  let deletionComparisonTarget = EMPTY_OBJ;
  if (vnode.shapeFlag & 32) {
    const type2 = children._;
    if (type2) {
      if (optimized && type2 === 1) {
        needDeletionCheck = false;
      } else {
        extend(slots, children);
        if (!optimized && type2 === 1) {
          delete slots._;
        }
      }
    } else {
      needDeletionCheck = !children.$stable;
      normalizeObjectSlots(children, slots);
    }
    deletionComparisonTarget = children;
  } else if (children) {
    normalizeVNodeSlots(instance, children);
    deletionComparisonTarget = { default: 1 };
  }
  if (needDeletionCheck) {
    for (const key in slots) {
      if (!isInternalKey(key) && !(key in deletionComparisonTarget)) {
        delete slots[key];
      }
    }
  }
};
function createAppContext() {
  return {
    app: null,
    config: {
      isNativeTag: NO,
      performance: false,
      globalProperties: {},
      optionMergeStrategies: {},
      errorHandler: void 0,
      warnHandler: void 0,
      compilerOptions: {}
    },
    mixins: [],
    components: {},
    directives: {},
    provides: /* @__PURE__ */ Object.create(null),
    optionsCache: /* @__PURE__ */ new WeakMap(),
    propsCache: /* @__PURE__ */ new WeakMap(),
    emitsCache: /* @__PURE__ */ new WeakMap()
  };
}
let uid$1 = 0;
function createAppAPI(render2, hydrate2) {
  return function createApp2(rootComponent, rootProps = null) {
    if (!isFunction$2(rootComponent)) {
      rootComponent = Object.assign({}, rootComponent);
    }
    if (rootProps != null && !isObject$5(rootProps)) {
      rootProps = null;
    }
    const context = createAppContext();
    const installedPlugins = /* @__PURE__ */ new Set();
    let isMounted = false;
    const app2 = context.app = {
      _uid: uid$1++,
      _component: rootComponent,
      _props: rootProps,
      _container: null,
      _context: context,
      _instance: null,
      version: version$a,
      get config() {
        return context.config;
      },
      set config(v2) {
      },
      use(plugin, ...options2) {
        if (installedPlugins.has(plugin)) ;
        else if (plugin && isFunction$2(plugin.install)) {
          installedPlugins.add(plugin);
          plugin.install(app2, ...options2);
        } else if (isFunction$2(plugin)) {
          installedPlugins.add(plugin);
          plugin(app2, ...options2);
        } else ;
        return app2;
      },
      mixin(mixin) {
        {
          if (!context.mixins.includes(mixin)) {
            context.mixins.push(mixin);
          }
        }
        return app2;
      },
      component(name2, component) {
        if (!component) {
          return context.components[name2];
        }
        context.components[name2] = component;
        return app2;
      },
      directive(name2, directive) {
        if (!directive) {
          return context.directives[name2];
        }
        context.directives[name2] = directive;
        return app2;
      },
      mount(rootContainer, isHydrate, isSVG) {
        if (!isMounted) {
          const vnode = createVNode(rootComponent, rootProps);
          vnode.appContext = context;
          if (isHydrate && hydrate2) {
            hydrate2(vnode, rootContainer);
          } else {
            render2(vnode, rootContainer, isSVG);
          }
          isMounted = true;
          app2._container = rootContainer;
          rootContainer.__vue_app__ = app2;
          return getExposeProxy(vnode.component) || vnode.component.proxy;
        }
      },
      unmount() {
        if (isMounted) {
          render2(null, app2._container);
          delete app2._container.__vue_app__;
        }
      },
      provide(key, value) {
        context.provides[key] = value;
        return app2;
      }
    };
    return app2;
  };
}
function setRef(rawRef, oldRawRef, parentSuspense, vnode, isUnmount = false) {
  if (isArray$2(rawRef)) {
    rawRef.forEach((r2, i2) => setRef(r2, oldRawRef && (isArray$2(oldRawRef) ? oldRawRef[i2] : oldRawRef), parentSuspense, vnode, isUnmount));
    return;
  }
  if (isAsyncWrapper(vnode) && !isUnmount) {
    return;
  }
  const refValue = vnode.shapeFlag & 4 ? getExposeProxy(vnode.component) || vnode.component.proxy : vnode.el;
  const value = isUnmount ? null : refValue;
  const { i: owner, r: ref3 } = rawRef;
  const oldRef = oldRawRef && oldRawRef.r;
  const refs = owner.refs === EMPTY_OBJ ? owner.refs = {} : owner.refs;
  const setupState = owner.setupState;
  if (oldRef != null && oldRef !== ref3) {
    if (isString$4(oldRef)) {
      refs[oldRef] = null;
      if (hasOwn$1(setupState, oldRef)) {
        setupState[oldRef] = null;
      }
    } else if (isRef(oldRef)) {
      oldRef.value = null;
    }
  }
  if (isFunction$2(ref3)) {
    callWithErrorHandling(ref3, owner, 12, [value, refs]);
  } else {
    const _isString = isString$4(ref3);
    const _isRef = isRef(ref3);
    if (_isString || _isRef) {
      const doSet = () => {
        if (rawRef.f) {
          const existing = _isString ? hasOwn$1(setupState, ref3) ? setupState[ref3] : refs[ref3] : ref3.value;
          if (isUnmount) {
            isArray$2(existing) && remove(existing, refValue);
          } else {
            if (!isArray$2(existing)) {
              if (_isString) {
                refs[ref3] = [refValue];
                if (hasOwn$1(setupState, ref3)) {
                  setupState[ref3] = refs[ref3];
                }
              } else {
                ref3.value = [refValue];
                if (rawRef.k)
                  refs[rawRef.k] = ref3.value;
              }
            } else if (!existing.includes(refValue)) {
              existing.push(refValue);
            }
          }
        } else if (_isString) {
          refs[ref3] = value;
          if (hasOwn$1(setupState, ref3)) {
            setupState[ref3] = value;
          }
        } else if (_isRef) {
          ref3.value = value;
          if (rawRef.k)
            refs[rawRef.k] = value;
        } else ;
      };
      if (value) {
        doSet.id = -1;
        queuePostRenderEffect(doSet, parentSuspense);
      } else {
        doSet();
      }
    }
  }
}
let hasMismatch = false;
const isSVGContainer = (container) => /svg/.test(container.namespaceURI) && container.tagName !== "foreignObject";
const isComment = (node) => node.nodeType === 8;
function createHydrationFunctions(rendererInternals) {
  const { mt: mountComponent, p: patch, o: { patchProp: patchProp2, createText, nextSibling, parentNode, remove: remove2, insert, createComment } } = rendererInternals;
  const hydrate2 = (vnode, container) => {
    if (!container.hasChildNodes()) {
      patch(null, vnode, container);
      flushPostFlushCbs();
      container._vnode = vnode;
      return;
    }
    hasMismatch = false;
    hydrateNode(container.firstChild, vnode, null, null, null);
    flushPostFlushCbs();
    container._vnode = vnode;
    if (hasMismatch && true) {
      console.error(`Hydration completed but contains mismatches.`);
    }
  };
  const hydrateNode = (node, vnode, parentComponent, parentSuspense, slotScopeIds, optimized = false) => {
    const isFragmentStart = isComment(node) && node.data === "[";
    const onMismatch = () => handleMismatch(node, vnode, parentComponent, parentSuspense, slotScopeIds, isFragmentStart);
    const { type: type2, ref: ref3, shapeFlag, patchFlag } = vnode;
    let domType = node.nodeType;
    vnode.el = node;
    if (patchFlag === -2) {
      optimized = false;
      vnode.dynamicChildren = null;
    }
    let nextNode = null;
    switch (type2) {
      case Text:
        if (domType !== 3) {
          if (vnode.children === "") {
            insert(vnode.el = createText(""), parentNode(node), node);
            nextNode = node;
          } else {
            nextNode = onMismatch();
          }
        } else {
          if (node.data !== vnode.children) {
            hasMismatch = true;
            node.data = vnode.children;
          }
          nextNode = nextSibling(node);
        }
        break;
      case Comment:
        if (domType !== 8 || isFragmentStart) {
          nextNode = onMismatch();
        } else {
          nextNode = nextSibling(node);
        }
        break;
      case Static:
        if (isFragmentStart) {
          node = nextSibling(node);
          domType = node.nodeType;
        }
        if (domType === 1 || domType === 3) {
          nextNode = node;
          const needToAdoptContent = !vnode.children.length;
          for (let i2 = 0; i2 < vnode.staticCount; i2++) {
            if (needToAdoptContent)
              vnode.children += nextNode.nodeType === 1 ? nextNode.outerHTML : nextNode.data;
            if (i2 === vnode.staticCount - 1) {
              vnode.anchor = nextNode;
            }
            nextNode = nextSibling(nextNode);
          }
          return isFragmentStart ? nextSibling(nextNode) : nextNode;
        } else {
          onMismatch();
        }
        break;
      case Fragment:
        if (!isFragmentStart) {
          nextNode = onMismatch();
        } else {
          nextNode = hydrateFragment(node, vnode, parentComponent, parentSuspense, slotScopeIds, optimized);
        }
        break;
      default:
        if (shapeFlag & 1) {
          if (domType !== 1 || vnode.type.toLowerCase() !== node.tagName.toLowerCase()) {
            nextNode = onMismatch();
          } else {
            nextNode = hydrateElement(node, vnode, parentComponent, parentSuspense, slotScopeIds, optimized);
          }
        } else if (shapeFlag & 6) {
          vnode.slotScopeIds = slotScopeIds;
          const container = parentNode(node);
          mountComponent(vnode, container, null, parentComponent, parentSuspense, isSVGContainer(container), optimized);
          nextNode = isFragmentStart ? locateClosingAsyncAnchor(node) : nextSibling(node);
          if (nextNode && isComment(nextNode) && nextNode.data === "teleport end") {
            nextNode = nextSibling(nextNode);
          }
          if (isAsyncWrapper(vnode)) {
            let subTree;
            if (isFragmentStart) {
              subTree = createVNode(Fragment);
              subTree.anchor = nextNode ? nextNode.previousSibling : container.lastChild;
            } else {
              subTree = node.nodeType === 3 ? createTextVNode("") : createVNode("div");
            }
            subTree.el = node;
            vnode.component.subTree = subTree;
          }
        } else if (shapeFlag & 64) {
          if (domType !== 8) {
            nextNode = onMismatch();
          } else {
            nextNode = vnode.type.hydrate(node, vnode, parentComponent, parentSuspense, slotScopeIds, optimized, rendererInternals, hydrateChildren);
          }
        } else if (shapeFlag & 128) {
          nextNode = vnode.type.hydrate(node, vnode, parentComponent, parentSuspense, isSVGContainer(parentNode(node)), slotScopeIds, optimized, rendererInternals, hydrateNode);
        } else ;
    }
    if (ref3 != null) {
      setRef(ref3, null, parentSuspense, vnode);
    }
    return nextNode;
  };
  const hydrateElement = (el2, vnode, parentComponent, parentSuspense, slotScopeIds, optimized) => {
    optimized = optimized || !!vnode.dynamicChildren;
    const { type: type2, props, patchFlag, shapeFlag, dirs } = vnode;
    const forcePatchValue = type2 === "input" && dirs || type2 === "option";
    if (forcePatchValue || patchFlag !== -1) {
      if (dirs) {
        invokeDirectiveHook(vnode, null, parentComponent, "created");
      }
      if (props) {
        if (forcePatchValue || !optimized || patchFlag & (16 | 32)) {
          for (const key in props) {
            if (forcePatchValue && key.endsWith("value") || isOn(key) && !isReservedProp(key)) {
              patchProp2(el2, key, null, props[key], false, void 0, parentComponent);
            }
          }
        } else if (props.onClick) {
          patchProp2(el2, "onClick", null, props.onClick, false, void 0, parentComponent);
        }
      }
      let vnodeHooks;
      if (vnodeHooks = props && props.onVnodeBeforeMount) {
        invokeVNodeHook(vnodeHooks, parentComponent, vnode);
      }
      if (dirs) {
        invokeDirectiveHook(vnode, null, parentComponent, "beforeMount");
      }
      if ((vnodeHooks = props && props.onVnodeMounted) || dirs) {
        queueEffectWithSuspense(() => {
          vnodeHooks && invokeVNodeHook(vnodeHooks, parentComponent, vnode);
          dirs && invokeDirectiveHook(vnode, null, parentComponent, "mounted");
        }, parentSuspense);
      }
      if (shapeFlag & 16 && // skip if element has innerHTML / textContent
      !(props && (props.innerHTML || props.textContent))) {
        let next = hydrateChildren(el2.firstChild, vnode, el2, parentComponent, parentSuspense, slotScopeIds, optimized);
        while (next) {
          hasMismatch = true;
          const cur = next;
          next = next.nextSibling;
          remove2(cur);
        }
      } else if (shapeFlag & 8) {
        if (el2.textContent !== vnode.children) {
          hasMismatch = true;
          el2.textContent = vnode.children;
        }
      }
    }
    return el2.nextSibling;
  };
  const hydrateChildren = (node, parentVNode, container, parentComponent, parentSuspense, slotScopeIds, optimized) => {
    optimized = optimized || !!parentVNode.dynamicChildren;
    const children = parentVNode.children;
    const l2 = children.length;
    for (let i2 = 0; i2 < l2; i2++) {
      const vnode = optimized ? children[i2] : children[i2] = normalizeVNode(children[i2]);
      if (node) {
        node = hydrateNode(node, vnode, parentComponent, parentSuspense, slotScopeIds, optimized);
      } else if (vnode.type === Text && !vnode.children) {
        continue;
      } else {
        hasMismatch = true;
        patch(null, vnode, container, null, parentComponent, parentSuspense, isSVGContainer(container), slotScopeIds);
      }
    }
    return node;
  };
  const hydrateFragment = (node, vnode, parentComponent, parentSuspense, slotScopeIds, optimized) => {
    const { slotScopeIds: fragmentSlotScopeIds } = vnode;
    if (fragmentSlotScopeIds) {
      slotScopeIds = slotScopeIds ? slotScopeIds.concat(fragmentSlotScopeIds) : fragmentSlotScopeIds;
    }
    const container = parentNode(node);
    const next = hydrateChildren(nextSibling(node), vnode, container, parentComponent, parentSuspense, slotScopeIds, optimized);
    if (next && isComment(next) && next.data === "]") {
      return nextSibling(vnode.anchor = next);
    } else {
      hasMismatch = true;
      insert(vnode.anchor = createComment(`]`), container, next);
      return next;
    }
  };
  const handleMismatch = (node, vnode, parentComponent, parentSuspense, slotScopeIds, isFragment) => {
    hasMismatch = true;
    vnode.el = null;
    if (isFragment) {
      const end = locateClosingAsyncAnchor(node);
      while (true) {
        const next2 = nextSibling(node);
        if (next2 && next2 !== end) {
          remove2(next2);
        } else {
          break;
        }
      }
    }
    const next = nextSibling(node);
    const container = parentNode(node);
    remove2(node);
    patch(null, vnode, container, next, parentComponent, parentSuspense, isSVGContainer(container), slotScopeIds);
    return next;
  };
  const locateClosingAsyncAnchor = (node) => {
    let match2 = 0;
    while (node) {
      node = nextSibling(node);
      if (node && isComment(node)) {
        if (node.data === "[")
          match2++;
        if (node.data === "]") {
          if (match2 === 0) {
            return nextSibling(node);
          } else {
            match2--;
          }
        }
      }
    }
    return node;
  };
  return [hydrate2, hydrateNode];
}
const queuePostRenderEffect = queueEffectWithSuspense;
function createRenderer(options2) {
  return baseCreateRenderer(options2);
}
function createHydrationRenderer(options2) {
  return baseCreateRenderer(options2, createHydrationFunctions);
}
function baseCreateRenderer(options2, createHydrationFns) {
  const target2 = getGlobalThis$1();
  target2.__VUE__ = true;
  const { insert: hostInsert, remove: hostRemove, patchProp: hostPatchProp, createElement: hostCreateElement, createText: hostCreateText, createComment: hostCreateComment, setText: hostSetText, setElementText: hostSetElementText, parentNode: hostParentNode, nextSibling: hostNextSibling, setScopeId: hostSetScopeId = NOOP, insertStaticContent: hostInsertStaticContent } = options2;
  const patch = (n1, n2, container, anchor = null, parentComponent = null, parentSuspense = null, isSVG = false, slotScopeIds = null, optimized = !!n2.dynamicChildren) => {
    if (n1 === n2) {
      return;
    }
    if (n1 && !isSameVNodeType(n1, n2)) {
      anchor = getNextHostNode(n1);
      unmount(n1, parentComponent, parentSuspense, true);
      n1 = null;
    }
    if (n2.patchFlag === -2) {
      optimized = false;
      n2.dynamicChildren = null;
    }
    const { type: type2, ref: ref3, shapeFlag } = n2;
    switch (type2) {
      case Text:
        processText(n1, n2, container, anchor);
        break;
      case Comment:
        processCommentNode(n1, n2, container, anchor);
        break;
      case Static:
        if (n1 == null) {
          mountStaticNode(n2, container, anchor, isSVG);
        }
        break;
      case Fragment:
        processFragment(n1, n2, container, anchor, parentComponent, parentSuspense, isSVG, slotScopeIds, optimized);
        break;
      default:
        if (shapeFlag & 1) {
          processElement(n1, n2, container, anchor, parentComponent, parentSuspense, isSVG, slotScopeIds, optimized);
        } else if (shapeFlag & 6) {
          processComponent(n1, n2, container, anchor, parentComponent, parentSuspense, isSVG, slotScopeIds, optimized);
        } else if (shapeFlag & 64) {
          type2.process(n1, n2, container, anchor, parentComponent, parentSuspense, isSVG, slotScopeIds, optimized, internals);
        } else if (shapeFlag & 128) {
          type2.process(n1, n2, container, anchor, parentComponent, parentSuspense, isSVG, slotScopeIds, optimized, internals);
        } else ;
    }
    if (ref3 != null && parentComponent) {
      setRef(ref3, n1 && n1.ref, parentSuspense, n2 || n1, !n2);
    }
  };
  const processText = (n1, n2, container, anchor) => {
    if (n1 == null) {
      hostInsert(n2.el = hostCreateText(n2.children), container, anchor);
    } else {
      const el2 = n2.el = n1.el;
      if (n2.children !== n1.children) {
        hostSetText(el2, n2.children);
      }
    }
  };
  const processCommentNode = (n1, n2, container, anchor) => {
    if (n1 == null) {
      hostInsert(n2.el = hostCreateComment(n2.children || ""), container, anchor);
    } else {
      n2.el = n1.el;
    }
  };
  const mountStaticNode = (n2, container, anchor, isSVG) => {
    [n2.el, n2.anchor] = hostInsertStaticContent(n2.children, container, anchor, isSVG, n2.el, n2.anchor);
  };
  const moveStaticNode = ({ el: el2, anchor }, container, nextSibling) => {
    let next;
    while (el2 && el2 !== anchor) {
      next = hostNextSibling(el2);
      hostInsert(el2, container, nextSibling);
      el2 = next;
    }
    hostInsert(anchor, container, nextSibling);
  };
  const removeStaticNode = ({ el: el2, anchor }) => {
    let next;
    while (el2 && el2 !== anchor) {
      next = hostNextSibling(el2);
      hostRemove(el2);
      el2 = next;
    }
    hostRemove(anchor);
  };
  const processElement = (n1, n2, container, anchor, parentComponent, parentSuspense, isSVG, slotScopeIds, optimized) => {
    isSVG = isSVG || n2.type === "svg";
    if (n1 == null) {
      mountElement(n2, container, anchor, parentComponent, parentSuspense, isSVG, slotScopeIds, optimized);
    } else {
      patchElement(n1, n2, parentComponent, parentSuspense, isSVG, slotScopeIds, optimized);
    }
  };
  const mountElement = (vnode, container, anchor, parentComponent, parentSuspense, isSVG, slotScopeIds, optimized) => {
    let el2;
    let vnodeHook;
    const { type: type2, props, shapeFlag, transition, dirs } = vnode;
    el2 = vnode.el = hostCreateElement(vnode.type, isSVG, props && props.is, props);
    if (shapeFlag & 8) {
      hostSetElementText(el2, vnode.children);
    } else if (shapeFlag & 16) {
      mountChildren(vnode.children, el2, null, parentComponent, parentSuspense, isSVG && type2 !== "foreignObject", slotScopeIds, optimized);
    }
    if (dirs) {
      invokeDirectiveHook(vnode, null, parentComponent, "created");
    }
    setScopeId(el2, vnode, vnode.scopeId, slotScopeIds, parentComponent);
    if (props) {
      for (const key in props) {
        if (key !== "value" && !isReservedProp(key)) {
          hostPatchProp(el2, key, null, props[key], isSVG, vnode.children, parentComponent, parentSuspense, unmountChildren);
        }
      }
      if ("value" in props) {
        hostPatchProp(el2, "value", null, props.value);
      }
      if (vnodeHook = props.onVnodeBeforeMount) {
        invokeVNodeHook(vnodeHook, parentComponent, vnode);
      }
    }
    if (dirs) {
      invokeDirectiveHook(vnode, null, parentComponent, "beforeMount");
    }
    const needCallTransitionHooks = (!parentSuspense || parentSuspense && !parentSuspense.pendingBranch) && transition && !transition.persisted;
    if (needCallTransitionHooks) {
      transition.beforeEnter(el2);
    }
    hostInsert(el2, container, anchor);
    if ((vnodeHook = props && props.onVnodeMounted) || needCallTransitionHooks || dirs) {
      queuePostRenderEffect(() => {
        vnodeHook && invokeVNodeHook(vnodeHook, parentComponent, vnode);
        needCallTransitionHooks && transition.enter(el2);
        dirs && invokeDirectiveHook(vnode, null, parentComponent, "mounted");
      }, parentSuspense);
    }
  };
  const setScopeId = (el2, vnode, scopeId, slotScopeIds, parentComponent) => {
    if (scopeId) {
      hostSetScopeId(el2, scopeId);
    }
    if (slotScopeIds) {
      for (let i2 = 0; i2 < slotScopeIds.length; i2++) {
        hostSetScopeId(el2, slotScopeIds[i2]);
      }
    }
    if (parentComponent) {
      let subTree = parentComponent.subTree;
      if (vnode === subTree) {
        const parentVNode = parentComponent.vnode;
        setScopeId(el2, parentVNode, parentVNode.scopeId, parentVNode.slotScopeIds, parentComponent.parent);
      }
    }
  };
  const mountChildren = (children, container, anchor, parentComponent, parentSuspense, isSVG, slotScopeIds, optimized, start = 0) => {
    for (let i2 = start; i2 < children.length; i2++) {
      const child = children[i2] = optimized ? cloneIfMounted(children[i2]) : normalizeVNode(children[i2]);
      patch(null, child, container, anchor, parentComponent, parentSuspense, isSVG, slotScopeIds, optimized);
    }
  };
  const patchElement = (n1, n2, parentComponent, parentSuspense, isSVG, slotScopeIds, optimized) => {
    const el2 = n2.el = n1.el;
    let { patchFlag, dynamicChildren, dirs } = n2;
    patchFlag |= n1.patchFlag & 16;
    const oldProps = n1.props || EMPTY_OBJ;
    const newProps = n2.props || EMPTY_OBJ;
    let vnodeHook;
    parentComponent && toggleRecurse(parentComponent, false);
    if (vnodeHook = newProps.onVnodeBeforeUpdate) {
      invokeVNodeHook(vnodeHook, parentComponent, n2, n1);
    }
    if (dirs) {
      invokeDirectiveHook(n2, n1, parentComponent, "beforeUpdate");
    }
    parentComponent && toggleRecurse(parentComponent, true);
    const areChildrenSVG = isSVG && n2.type !== "foreignObject";
    if (dynamicChildren) {
      patchBlockChildren(n1.dynamicChildren, dynamicChildren, el2, parentComponent, parentSuspense, areChildrenSVG, slotScopeIds);
    } else if (!optimized) {
      patchChildren(n1, n2, el2, null, parentComponent, parentSuspense, areChildrenSVG, slotScopeIds, false);
    }
    if (patchFlag > 0) {
      if (patchFlag & 16) {
        patchProps(el2, n2, oldProps, newProps, parentComponent, parentSuspense, isSVG);
      } else {
        if (patchFlag & 2) {
          if (oldProps.class !== newProps.class) {
            hostPatchProp(el2, "class", null, newProps.class, isSVG);
          }
        }
        if (patchFlag & 4) {
          hostPatchProp(el2, "style", oldProps.style, newProps.style, isSVG);
        }
        if (patchFlag & 8) {
          const propsToUpdate = n2.dynamicProps;
          for (let i2 = 0; i2 < propsToUpdate.length; i2++) {
            const key = propsToUpdate[i2];
            const prev = oldProps[key];
            const next = newProps[key];
            if (next !== prev || key === "value") {
              hostPatchProp(el2, key, prev, next, isSVG, n1.children, parentComponent, parentSuspense, unmountChildren);
            }
          }
        }
      }
      if (patchFlag & 1) {
        if (n1.children !== n2.children) {
          hostSetElementText(el2, n2.children);
        }
      }
    } else if (!optimized && dynamicChildren == null) {
      patchProps(el2, n2, oldProps, newProps, parentComponent, parentSuspense, isSVG);
    }
    if ((vnodeHook = newProps.onVnodeUpdated) || dirs) {
      queuePostRenderEffect(() => {
        vnodeHook && invokeVNodeHook(vnodeHook, parentComponent, n2, n1);
        dirs && invokeDirectiveHook(n2, n1, parentComponent, "updated");
      }, parentSuspense);
    }
  };
  const patchBlockChildren = (oldChildren, newChildren, fallbackContainer, parentComponent, parentSuspense, isSVG, slotScopeIds) => {
    for (let i2 = 0; i2 < newChildren.length; i2++) {
      const oldVNode = oldChildren[i2];
      const newVNode = newChildren[i2];
      const container = (
        // oldVNode may be an errored async setup() component inside Suspense
        // which will not have a mounted element
        oldVNode.el && // - In the case of a Fragment, we need to provide the actual parent
        // of the Fragment itself so it can move its children.
        (oldVNode.type === Fragment || // - In the case of different nodes, there is going to be a replacement
        // which also requires the correct parent container
        !isSameVNodeType(oldVNode, newVNode) || // - In the case of a component, it could contain anything.
        oldVNode.shapeFlag & (6 | 64)) ? hostParentNode(oldVNode.el) : (
          // In other cases, the parent container is not actually used so we
          // just pass the block element here to avoid a DOM parentNode call.
          fallbackContainer
        )
      );
      patch(oldVNode, newVNode, container, null, parentComponent, parentSuspense, isSVG, slotScopeIds, true);
    }
  };
  const patchProps = (el2, vnode, oldProps, newProps, parentComponent, parentSuspense, isSVG) => {
    if (oldProps !== newProps) {
      if (oldProps !== EMPTY_OBJ) {
        for (const key in oldProps) {
          if (!isReservedProp(key) && !(key in newProps)) {
            hostPatchProp(el2, key, oldProps[key], null, isSVG, vnode.children, parentComponent, parentSuspense, unmountChildren);
          }
        }
      }
      for (const key in newProps) {
        if (isReservedProp(key))
          continue;
        const next = newProps[key];
        const prev = oldProps[key];
        if (next !== prev && key !== "value") {
          hostPatchProp(el2, key, prev, next, isSVG, vnode.children, parentComponent, parentSuspense, unmountChildren);
        }
      }
      if ("value" in newProps) {
        hostPatchProp(el2, "value", oldProps.value, newProps.value);
      }
    }
  };
  const processFragment = (n1, n2, container, anchor, parentComponent, parentSuspense, isSVG, slotScopeIds, optimized) => {
    const fragmentStartAnchor = n2.el = n1 ? n1.el : hostCreateText("");
    const fragmentEndAnchor = n2.anchor = n1 ? n1.anchor : hostCreateText("");
    let { patchFlag, dynamicChildren, slotScopeIds: fragmentSlotScopeIds } = n2;
    if (fragmentSlotScopeIds) {
      slotScopeIds = slotScopeIds ? slotScopeIds.concat(fragmentSlotScopeIds) : fragmentSlotScopeIds;
    }
    if (n1 == null) {
      hostInsert(fragmentStartAnchor, container, anchor);
      hostInsert(fragmentEndAnchor, container, anchor);
      mountChildren(n2.children, container, fragmentEndAnchor, parentComponent, parentSuspense, isSVG, slotScopeIds, optimized);
    } else {
      if (patchFlag > 0 && patchFlag & 64 && dynamicChildren && // #2715 the previous fragment could've been a BAILed one as a result
      // of renderSlot() with no valid children
      n1.dynamicChildren) {
        patchBlockChildren(n1.dynamicChildren, dynamicChildren, container, parentComponent, parentSuspense, isSVG, slotScopeIds);
        if (
          // #2080 if the stable fragment has a key, it's a <template v-for> that may
          //  get moved around. Make sure all root level vnodes inherit el.
          // #2134 or if it's a component root, it may also get moved around
          // as the component is being moved.
          n2.key != null || parentComponent && n2 === parentComponent.subTree
        ) {
          traverseStaticChildren(
            n1,
            n2,
            true
            /* shallow */
          );
        }
      } else {
        patchChildren(n1, n2, container, fragmentEndAnchor, parentComponent, parentSuspense, isSVG, slotScopeIds, optimized);
      }
    }
  };
  const processComponent = (n1, n2, container, anchor, parentComponent, parentSuspense, isSVG, slotScopeIds, optimized) => {
    n2.slotScopeIds = slotScopeIds;
    if (n1 == null) {
      if (n2.shapeFlag & 512) {
        parentComponent.ctx.activate(n2, container, anchor, isSVG, optimized);
      } else {
        mountComponent(n2, container, anchor, parentComponent, parentSuspense, isSVG, optimized);
      }
    } else {
      updateComponent(n1, n2, optimized);
    }
  };
  const mountComponent = (initialVNode, container, anchor, parentComponent, parentSuspense, isSVG, optimized) => {
    const instance = initialVNode.component = createComponentInstance(initialVNode, parentComponent, parentSuspense);
    if (isKeepAlive(initialVNode)) {
      instance.ctx.renderer = internals;
    }
    {
      setupComponent(instance);
    }
    if (instance.asyncDep) {
      parentSuspense && parentSuspense.registerDep(instance, setupRenderEffect);
      if (!initialVNode.el) {
        const placeholder = instance.subTree = createVNode(Comment);
        processCommentNode(null, placeholder, container, anchor);
      }
      return;
    }
    setupRenderEffect(instance, initialVNode, container, anchor, parentSuspense, isSVG, optimized);
  };
  const updateComponent = (n1, n2, optimized) => {
    const instance = n2.component = n1.component;
    if (shouldUpdateComponent(n1, n2, optimized)) {
      if (instance.asyncDep && !instance.asyncResolved) {
        updateComponentPreRender(instance, n2, optimized);
        return;
      } else {
        instance.next = n2;
        invalidateJob(instance.update);
        instance.update();
      }
    } else {
      n2.el = n1.el;
      instance.vnode = n2;
    }
  };
  const setupRenderEffect = (instance, initialVNode, container, anchor, parentSuspense, isSVG, optimized) => {
    const componentUpdateFn = () => {
      if (!instance.isMounted) {
        let vnodeHook;
        const { el: el2, props } = initialVNode;
        const { bm, m: m2, parent } = instance;
        const isAsyncWrapperVNode = isAsyncWrapper(initialVNode);
        toggleRecurse(instance, false);
        if (bm) {
          invokeArrayFns(bm);
        }
        if (!isAsyncWrapperVNode && (vnodeHook = props && props.onVnodeBeforeMount)) {
          invokeVNodeHook(vnodeHook, parent, initialVNode);
        }
        toggleRecurse(instance, true);
        if (el2 && hydrateNode) {
          const hydrateSubTree = () => {
            instance.subTree = renderComponentRoot(instance);
            hydrateNode(el2, instance.subTree, instance, parentSuspense, null);
          };
          if (isAsyncWrapperVNode) {
            initialVNode.type.__asyncLoader().then(
              // note: we are moving the render call into an async callback,
              // which means it won't track dependencies - but it's ok because
              // a server-rendered async wrapper is already in resolved state
              // and it will never need to change.
              () => !instance.isUnmounted && hydrateSubTree()
            );
          } else {
            hydrateSubTree();
          }
        } else {
          const subTree = instance.subTree = renderComponentRoot(instance);
          patch(null, subTree, container, anchor, instance, parentSuspense, isSVG);
          initialVNode.el = subTree.el;
        }
        if (m2) {
          queuePostRenderEffect(m2, parentSuspense);
        }
        if (!isAsyncWrapperVNode && (vnodeHook = props && props.onVnodeMounted)) {
          const scopedInitialVNode = initialVNode;
          queuePostRenderEffect(() => invokeVNodeHook(vnodeHook, parent, scopedInitialVNode), parentSuspense);
        }
        if (initialVNode.shapeFlag & 256 || parent && isAsyncWrapper(parent.vnode) && parent.vnode.shapeFlag & 256) {
          instance.a && queuePostRenderEffect(instance.a, parentSuspense);
        }
        instance.isMounted = true;
        initialVNode = container = anchor = null;
      } else {
        let { next, bu, u: u2, parent, vnode } = instance;
        let originNext = next;
        let vnodeHook;
        toggleRecurse(instance, false);
        if (next) {
          next.el = vnode.el;
          updateComponentPreRender(instance, next, optimized);
        } else {
          next = vnode;
        }
        if (bu) {
          invokeArrayFns(bu);
        }
        if (vnodeHook = next.props && next.props.onVnodeBeforeUpdate) {
          invokeVNodeHook(vnodeHook, parent, next, vnode);
        }
        toggleRecurse(instance, true);
        const nextTree = renderComponentRoot(instance);
        const prevTree = instance.subTree;
        instance.subTree = nextTree;
        patch(
          prevTree,
          nextTree,
          // parent may have changed if it's in a teleport
          hostParentNode(prevTree.el),
          // anchor may have changed if it's in a fragment
          getNextHostNode(prevTree),
          instance,
          parentSuspense,
          isSVG
        );
        next.el = nextTree.el;
        if (originNext === null) {
          updateHOCHostEl(instance, nextTree.el);
        }
        if (u2) {
          queuePostRenderEffect(u2, parentSuspense);
        }
        if (vnodeHook = next.props && next.props.onVnodeUpdated) {
          queuePostRenderEffect(() => invokeVNodeHook(vnodeHook, parent, next, vnode), parentSuspense);
        }
      }
    };
    const effect2 = instance.effect = new ReactiveEffect(
      componentUpdateFn,
      () => queueJob(update),
      instance.scope
      // track it in component's effect scope
    );
    const update = instance.update = () => effect2.run();
    update.id = instance.uid;
    toggleRecurse(instance, true);
    update();
  };
  const updateComponentPreRender = (instance, nextVNode, optimized) => {
    nextVNode.component = instance;
    const prevProps = instance.vnode.props;
    instance.vnode = nextVNode;
    instance.next = null;
    updateProps(instance, nextVNode.props, prevProps, optimized);
    updateSlots(instance, nextVNode.children, optimized);
    pauseTracking();
    flushPreFlushCbs();
    resetTracking();
  };
  const patchChildren = (n1, n2, container, anchor, parentComponent, parentSuspense, isSVG, slotScopeIds, optimized = false) => {
    const c1 = n1 && n1.children;
    const prevShapeFlag = n1 ? n1.shapeFlag : 0;
    const c2 = n2.children;
    const { patchFlag, shapeFlag } = n2;
    if (patchFlag > 0) {
      if (patchFlag & 128) {
        patchKeyedChildren(c1, c2, container, anchor, parentComponent, parentSuspense, isSVG, slotScopeIds, optimized);
        return;
      } else if (patchFlag & 256) {
        patchUnkeyedChildren(c1, c2, container, anchor, parentComponent, parentSuspense, isSVG, slotScopeIds, optimized);
        return;
      }
    }
    if (shapeFlag & 8) {
      if (prevShapeFlag & 16) {
        unmountChildren(c1, parentComponent, parentSuspense);
      }
      if (c2 !== c1) {
        hostSetElementText(container, c2);
      }
    } else {
      if (prevShapeFlag & 16) {
        if (shapeFlag & 16) {
          patchKeyedChildren(c1, c2, container, anchor, parentComponent, parentSuspense, isSVG, slotScopeIds, optimized);
        } else {
          unmountChildren(c1, parentComponent, parentSuspense, true);
        }
      } else {
        if (prevShapeFlag & 8) {
          hostSetElementText(container, "");
        }
        if (shapeFlag & 16) {
          mountChildren(c2, container, anchor, parentComponent, parentSuspense, isSVG, slotScopeIds, optimized);
        }
      }
    }
  };
  const patchUnkeyedChildren = (c1, c2, container, anchor, parentComponent, parentSuspense, isSVG, slotScopeIds, optimized) => {
    c1 = c1 || EMPTY_ARR;
    c2 = c2 || EMPTY_ARR;
    const oldLength = c1.length;
    const newLength = c2.length;
    const commonLength = Math.min(oldLength, newLength);
    let i2;
    for (i2 = 0; i2 < commonLength; i2++) {
      const nextChild = c2[i2] = optimized ? cloneIfMounted(c2[i2]) : normalizeVNode(c2[i2]);
      patch(c1[i2], nextChild, container, null, parentComponent, parentSuspense, isSVG, slotScopeIds, optimized);
    }
    if (oldLength > newLength) {
      unmountChildren(c1, parentComponent, parentSuspense, true, false, commonLength);
    } else {
      mountChildren(c2, container, anchor, parentComponent, parentSuspense, isSVG, slotScopeIds, optimized, commonLength);
    }
  };
  const patchKeyedChildren = (c1, c2, container, parentAnchor, parentComponent, parentSuspense, isSVG, slotScopeIds, optimized) => {
    let i2 = 0;
    const l2 = c2.length;
    let e1 = c1.length - 1;
    let e2 = l2 - 1;
    while (i2 <= e1 && i2 <= e2) {
      const n1 = c1[i2];
      const n2 = c2[i2] = optimized ? cloneIfMounted(c2[i2]) : normalizeVNode(c2[i2]);
      if (isSameVNodeType(n1, n2)) {
        patch(n1, n2, container, null, parentComponent, parentSuspense, isSVG, slotScopeIds, optimized);
      } else {
        break;
      }
      i2++;
    }
    while (i2 <= e1 && i2 <= e2) {
      const n1 = c1[e1];
      const n2 = c2[e2] = optimized ? cloneIfMounted(c2[e2]) : normalizeVNode(c2[e2]);
      if (isSameVNodeType(n1, n2)) {
        patch(n1, n2, container, null, parentComponent, parentSuspense, isSVG, slotScopeIds, optimized);
      } else {
        break;
      }
      e1--;
      e2--;
    }
    if (i2 > e1) {
      if (i2 <= e2) {
        const nextPos = e2 + 1;
        const anchor = nextPos < l2 ? c2[nextPos].el : parentAnchor;
        while (i2 <= e2) {
          patch(null, c2[i2] = optimized ? cloneIfMounted(c2[i2]) : normalizeVNode(c2[i2]), container, anchor, parentComponent, parentSuspense, isSVG, slotScopeIds, optimized);
          i2++;
        }
      }
    } else if (i2 > e2) {
      while (i2 <= e1) {
        unmount(c1[i2], parentComponent, parentSuspense, true);
        i2++;
      }
    } else {
      const s1 = i2;
      const s2 = i2;
      const keyToNewIndexMap = /* @__PURE__ */ new Map();
      for (i2 = s2; i2 <= e2; i2++) {
        const nextChild = c2[i2] = optimized ? cloneIfMounted(c2[i2]) : normalizeVNode(c2[i2]);
        if (nextChild.key != null) {
          keyToNewIndexMap.set(nextChild.key, i2);
        }
      }
      let j2;
      let patched = 0;
      const toBePatched = e2 - s2 + 1;
      let moved = false;
      let maxNewIndexSoFar = 0;
      const newIndexToOldIndexMap = new Array(toBePatched);
      for (i2 = 0; i2 < toBePatched; i2++)
        newIndexToOldIndexMap[i2] = 0;
      for (i2 = s1; i2 <= e1; i2++) {
        const prevChild = c1[i2];
        if (patched >= toBePatched) {
          unmount(prevChild, parentComponent, parentSuspense, true);
          continue;
        }
        let newIndex;
        if (prevChild.key != null) {
          newIndex = keyToNewIndexMap.get(prevChild.key);
        } else {
          for (j2 = s2; j2 <= e2; j2++) {
            if (newIndexToOldIndexMap[j2 - s2] === 0 && isSameVNodeType(prevChild, c2[j2])) {
              newIndex = j2;
              break;
            }
          }
        }
        if (newIndex === void 0) {
          unmount(prevChild, parentComponent, parentSuspense, true);
        } else {
          newIndexToOldIndexMap[newIndex - s2] = i2 + 1;
          if (newIndex >= maxNewIndexSoFar) {
            maxNewIndexSoFar = newIndex;
          } else {
            moved = true;
          }
          patch(prevChild, c2[newIndex], container, null, parentComponent, parentSuspense, isSVG, slotScopeIds, optimized);
          patched++;
        }
      }
      const increasingNewIndexSequence = moved ? getSequence(newIndexToOldIndexMap) : EMPTY_ARR;
      j2 = increasingNewIndexSequence.length - 1;
      for (i2 = toBePatched - 1; i2 >= 0; i2--) {
        const nextIndex = s2 + i2;
        const nextChild = c2[nextIndex];
        const anchor = nextIndex + 1 < l2 ? c2[nextIndex + 1].el : parentAnchor;
        if (newIndexToOldIndexMap[i2] === 0) {
          patch(null, nextChild, container, anchor, parentComponent, parentSuspense, isSVG, slotScopeIds, optimized);
        } else if (moved) {
          if (j2 < 0 || i2 !== increasingNewIndexSequence[j2]) {
            move(
              nextChild,
              container,
              anchor,
              2
              /* MoveType.REORDER */
            );
          } else {
            j2--;
          }
        }
      }
    }
  };
  const move = (vnode, container, anchor, moveType, parentSuspense = null) => {
    const { el: el2, type: type2, transition, children, shapeFlag } = vnode;
    if (shapeFlag & 6) {
      move(vnode.component.subTree, container, anchor, moveType);
      return;
    }
    if (shapeFlag & 128) {
      vnode.suspense.move(container, anchor, moveType);
      return;
    }
    if (shapeFlag & 64) {
      type2.move(vnode, container, anchor, internals);
      return;
    }
    if (type2 === Fragment) {
      hostInsert(el2, container, anchor);
      for (let i2 = 0; i2 < children.length; i2++) {
        move(children[i2], container, anchor, moveType);
      }
      hostInsert(vnode.anchor, container, anchor);
      return;
    }
    if (type2 === Static) {
      moveStaticNode(vnode, container, anchor);
      return;
    }
    const needTransition = moveType !== 2 && shapeFlag & 1 && transition;
    if (needTransition) {
      if (moveType === 0) {
        transition.beforeEnter(el2);
        hostInsert(el2, container, anchor);
        queuePostRenderEffect(() => transition.enter(el2), parentSuspense);
      } else {
        const { leave, delayLeave, afterLeave } = transition;
        const remove3 = () => hostInsert(el2, container, anchor);
        const performLeave = () => {
          leave(el2, () => {
            remove3();
            afterLeave && afterLeave();
          });
        };
        if (delayLeave) {
          delayLeave(el2, remove3, performLeave);
        } else {
          performLeave();
        }
      }
    } else {
      hostInsert(el2, container, anchor);
    }
  };
  const unmount = (vnode, parentComponent, parentSuspense, doRemove = false, optimized = false) => {
    const { type: type2, props, ref: ref3, children, dynamicChildren, shapeFlag, patchFlag, dirs } = vnode;
    if (ref3 != null) {
      setRef(ref3, null, parentSuspense, vnode, true);
    }
    if (shapeFlag & 256) {
      parentComponent.ctx.deactivate(vnode);
      return;
    }
    const shouldInvokeDirs = shapeFlag & 1 && dirs;
    const shouldInvokeVnodeHook = !isAsyncWrapper(vnode);
    let vnodeHook;
    if (shouldInvokeVnodeHook && (vnodeHook = props && props.onVnodeBeforeUnmount)) {
      invokeVNodeHook(vnodeHook, parentComponent, vnode);
    }
    if (shapeFlag & 6) {
      unmountComponent(vnode.component, parentSuspense, doRemove);
    } else {
      if (shapeFlag & 128) {
        vnode.suspense.unmount(parentSuspense, doRemove);
        return;
      }
      if (shouldInvokeDirs) {
        invokeDirectiveHook(vnode, null, parentComponent, "beforeUnmount");
      }
      if (shapeFlag & 64) {
        vnode.type.remove(vnode, parentComponent, parentSuspense, optimized, internals, doRemove);
      } else if (dynamicChildren && // #1153: fast path should not be taken for non-stable (v-for) fragments
      (type2 !== Fragment || patchFlag > 0 && patchFlag & 64)) {
        unmountChildren(dynamicChildren, parentComponent, parentSuspense, false, true);
      } else if (type2 === Fragment && patchFlag & (128 | 256) || !optimized && shapeFlag & 16) {
        unmountChildren(children, parentComponent, parentSuspense);
      }
      if (doRemove) {
        remove2(vnode);
      }
    }
    if (shouldInvokeVnodeHook && (vnodeHook = props && props.onVnodeUnmounted) || shouldInvokeDirs) {
      queuePostRenderEffect(() => {
        vnodeHook && invokeVNodeHook(vnodeHook, parentComponent, vnode);
        shouldInvokeDirs && invokeDirectiveHook(vnode, null, parentComponent, "unmounted");
      }, parentSuspense);
    }
  };
  const remove2 = (vnode) => {
    const { type: type2, el: el2, anchor, transition } = vnode;
    if (type2 === Fragment) {
      {
        removeFragment(el2, anchor);
      }
      return;
    }
    if (type2 === Static) {
      removeStaticNode(vnode);
      return;
    }
    const performRemove = () => {
      hostRemove(el2);
      if (transition && !transition.persisted && transition.afterLeave) {
        transition.afterLeave();
      }
    };
    if (vnode.shapeFlag & 1 && transition && !transition.persisted) {
      const { leave, delayLeave } = transition;
      const performLeave = () => leave(el2, performRemove);
      if (delayLeave) {
        delayLeave(vnode.el, performRemove, performLeave);
      } else {
        performLeave();
      }
    } else {
      performRemove();
    }
  };
  const removeFragment = (cur, end) => {
    let next;
    while (cur !== end) {
      next = hostNextSibling(cur);
      hostRemove(cur);
      cur = next;
    }
    hostRemove(end);
  };
  const unmountComponent = (instance, parentSuspense, doRemove) => {
    const { bum, scope, update, subTree, um } = instance;
    if (bum) {
      invokeArrayFns(bum);
    }
    scope.stop();
    if (update) {
      update.active = false;
      unmount(subTree, instance, parentSuspense, doRemove);
    }
    if (um) {
      queuePostRenderEffect(um, parentSuspense);
    }
    queuePostRenderEffect(() => {
      instance.isUnmounted = true;
    }, parentSuspense);
    if (parentSuspense && parentSuspense.pendingBranch && !parentSuspense.isUnmounted && instance.asyncDep && !instance.asyncResolved && instance.suspenseId === parentSuspense.pendingId) {
      parentSuspense.deps--;
      if (parentSuspense.deps === 0) {
        parentSuspense.resolve();
      }
    }
  };
  const unmountChildren = (children, parentComponent, parentSuspense, doRemove = false, optimized = false, start = 0) => {
    for (let i2 = start; i2 < children.length; i2++) {
      unmount(children[i2], parentComponent, parentSuspense, doRemove, optimized);
    }
  };
  const getNextHostNode = (vnode) => {
    if (vnode.shapeFlag & 6) {
      return getNextHostNode(vnode.component.subTree);
    }
    if (vnode.shapeFlag & 128) {
      return vnode.suspense.next();
    }
    return hostNextSibling(vnode.anchor || vnode.el);
  };
  const render2 = (vnode, container, isSVG) => {
    if (vnode == null) {
      if (container._vnode) {
        unmount(container._vnode, null, null, true);
      }
    } else {
      patch(container._vnode || null, vnode, container, null, null, null, isSVG);
    }
    flushPreFlushCbs();
    flushPostFlushCbs();
    container._vnode = vnode;
  };
  const internals = {
    p: patch,
    um: unmount,
    m: move,
    r: remove2,
    mt: mountComponent,
    mc: mountChildren,
    pc: patchChildren,
    pbc: patchBlockChildren,
    n: getNextHostNode,
    o: options2
  };
  let hydrate2;
  let hydrateNode;
  if (createHydrationFns) {
    [hydrate2, hydrateNode] = createHydrationFns(internals);
  }
  return {
    render: render2,
    hydrate: hydrate2,
    createApp: createAppAPI(render2, hydrate2)
  };
}
function toggleRecurse({ effect: effect2, update }, allowed) {
  effect2.allowRecurse = update.allowRecurse = allowed;
}
function traverseStaticChildren(n1, n2, shallow = false) {
  const ch1 = n1.children;
  const ch2 = n2.children;
  if (isArray$2(ch1) && isArray$2(ch2)) {
    for (let i2 = 0; i2 < ch1.length; i2++) {
      const c1 = ch1[i2];
      let c2 = ch2[i2];
      if (c2.shapeFlag & 1 && !c2.dynamicChildren) {
        if (c2.patchFlag <= 0 || c2.patchFlag === 32) {
          c2 = ch2[i2] = cloneIfMounted(ch2[i2]);
          c2.el = c1.el;
        }
        if (!shallow)
          traverseStaticChildren(c1, c2);
      }
      if (c2.type === Text) {
        c2.el = c1.el;
      }
    }
  }
}
function getSequence(arr) {
  const p2 = arr.slice();
  const result = [0];
  let i2, j2, u2, v2, c2;
  const len = arr.length;
  for (i2 = 0; i2 < len; i2++) {
    const arrI = arr[i2];
    if (arrI !== 0) {
      j2 = result[result.length - 1];
      if (arr[j2] < arrI) {
        p2[i2] = j2;
        result.push(i2);
        continue;
      }
      u2 = 0;
      v2 = result.length - 1;
      while (u2 < v2) {
        c2 = u2 + v2 >> 1;
        if (arr[result[c2]] < arrI) {
          u2 = c2 + 1;
        } else {
          v2 = c2;
        }
      }
      if (arrI < arr[result[u2]]) {
        if (u2 > 0) {
          p2[i2] = result[u2 - 1];
        }
        result[u2] = i2;
      }
    }
  }
  u2 = result.length;
  v2 = result[u2 - 1];
  while (u2-- > 0) {
    result[u2] = v2;
    v2 = p2[v2];
  }
  return result;
}
const isTeleport = (type2) => type2.__isTeleport;
const isTeleportDisabled = (props) => props && (props.disabled || props.disabled === "");
const isTargetSVG = (target2) => typeof SVGElement !== "undefined" && target2 instanceof SVGElement;
const resolveTarget = (props, select) => {
  const targetSelector = props && props.to;
  if (isString$4(targetSelector)) {
    if (!select) {
      return null;
    } else {
      const target2 = select(targetSelector);
      return target2;
    }
  } else {
    return targetSelector;
  }
};
const TeleportImpl = {
  __isTeleport: true,
  process(n1, n2, container, anchor, parentComponent, parentSuspense, isSVG, slotScopeIds, optimized, internals) {
    const { mc: mountChildren, pc: patchChildren, pbc: patchBlockChildren, o: { insert, querySelector, createText, createComment } } = internals;
    const disabled = isTeleportDisabled(n2.props);
    let { shapeFlag, children, dynamicChildren } = n2;
    if (n1 == null) {
      const placeholder = n2.el = createText("");
      const mainAnchor = n2.anchor = createText("");
      insert(placeholder, container, anchor);
      insert(mainAnchor, container, anchor);
      const target2 = n2.target = resolveTarget(n2.props, querySelector);
      const targetAnchor = n2.targetAnchor = createText("");
      if (target2) {
        insert(targetAnchor, target2);
        isSVG = isSVG || isTargetSVG(target2);
      }
      const mount = (container2, anchor2) => {
        if (shapeFlag & 16) {
          mountChildren(children, container2, anchor2, parentComponent, parentSuspense, isSVG, slotScopeIds, optimized);
        }
      };
      if (disabled) {
        mount(container, mainAnchor);
      } else if (target2) {
        mount(target2, targetAnchor);
      }
    } else {
      n2.el = n1.el;
      const mainAnchor = n2.anchor = n1.anchor;
      const target2 = n2.target = n1.target;
      const targetAnchor = n2.targetAnchor = n1.targetAnchor;
      const wasDisabled = isTeleportDisabled(n1.props);
      const currentContainer = wasDisabled ? container : target2;
      const currentAnchor = wasDisabled ? mainAnchor : targetAnchor;
      isSVG = isSVG || isTargetSVG(target2);
      if (dynamicChildren) {
        patchBlockChildren(n1.dynamicChildren, dynamicChildren, currentContainer, parentComponent, parentSuspense, isSVG, slotScopeIds);
        traverseStaticChildren(n1, n2, true);
      } else if (!optimized) {
        patchChildren(n1, n2, currentContainer, currentAnchor, parentComponent, parentSuspense, isSVG, slotScopeIds, false);
      }
      if (disabled) {
        if (!wasDisabled) {
          moveTeleport(
            n2,
            container,
            mainAnchor,
            internals,
            1
            /* TeleportMoveTypes.TOGGLE */
          );
        }
      } else {
        if ((n2.props && n2.props.to) !== (n1.props && n1.props.to)) {
          const nextTarget = n2.target = resolveTarget(n2.props, querySelector);
          if (nextTarget) {
            moveTeleport(
              n2,
              nextTarget,
              null,
              internals,
              0
              /* TeleportMoveTypes.TARGET_CHANGE */
            );
          }
        } else if (wasDisabled) {
          moveTeleport(
            n2,
            target2,
            targetAnchor,
            internals,
            1
            /* TeleportMoveTypes.TOGGLE */
          );
        }
      }
    }
    updateCssVars(n2);
  },
  remove(vnode, parentComponent, parentSuspense, optimized, { um: unmount, o: { remove: hostRemove } }, doRemove) {
    const { shapeFlag, children, anchor, targetAnchor, target: target2, props } = vnode;
    if (target2) {
      hostRemove(targetAnchor);
    }
    if (doRemove || !isTeleportDisabled(props)) {
      hostRemove(anchor);
      if (shapeFlag & 16) {
        for (let i2 = 0; i2 < children.length; i2++) {
          const child = children[i2];
          unmount(child, parentComponent, parentSuspense, true, !!child.dynamicChildren);
        }
      }
    }
  },
  move: moveTeleport,
  hydrate: hydrateTeleport
};
function moveTeleport(vnode, container, parentAnchor, { o: { insert }, m: move }, moveType = 2) {
  if (moveType === 0) {
    insert(vnode.targetAnchor, container, parentAnchor);
  }
  const { el: el2, anchor, shapeFlag, children, props } = vnode;
  const isReorder = moveType === 2;
  if (isReorder) {
    insert(el2, container, parentAnchor);
  }
  if (!isReorder || isTeleportDisabled(props)) {
    if (shapeFlag & 16) {
      for (let i2 = 0; i2 < children.length; i2++) {
        move(
          children[i2],
          container,
          parentAnchor,
          2
          /* MoveType.REORDER */
        );
      }
    }
  }
  if (isReorder) {
    insert(anchor, container, parentAnchor);
  }
}
function hydrateTeleport(node, vnode, parentComponent, parentSuspense, slotScopeIds, optimized, { o: { nextSibling, parentNode, querySelector } }, hydrateChildren) {
  const target2 = vnode.target = resolveTarget(vnode.props, querySelector);
  if (target2) {
    const targetNode = target2._lpa || target2.firstChild;
    if (vnode.shapeFlag & 16) {
      if (isTeleportDisabled(vnode.props)) {
        vnode.anchor = hydrateChildren(nextSibling(node), vnode, parentNode(node), parentComponent, parentSuspense, slotScopeIds, optimized);
        vnode.targetAnchor = targetNode;
      } else {
        vnode.anchor = nextSibling(node);
        let targetAnchor = targetNode;
        while (targetAnchor) {
          targetAnchor = nextSibling(targetAnchor);
          if (targetAnchor && targetAnchor.nodeType === 8 && targetAnchor.data === "teleport anchor") {
            vnode.targetAnchor = targetAnchor;
            target2._lpa = vnode.targetAnchor && nextSibling(vnode.targetAnchor);
            break;
          }
        }
        hydrateChildren(targetNode, vnode, target2, parentComponent, parentSuspense, slotScopeIds, optimized);
      }
    }
    updateCssVars(vnode);
  }
  return vnode.anchor && nextSibling(vnode.anchor);
}
const Teleport = TeleportImpl;
function updateCssVars(vnode) {
  const ctx = vnode.ctx;
  if (ctx && ctx.ut) {
    let node = vnode.children[0].el;
    while (node !== vnode.targetAnchor) {
      if (node.nodeType === 1)
        node.setAttribute("data-v-owner", ctx.uid);
      node = node.nextSibling;
    }
    ctx.ut();
  }
}
const Fragment = Symbol(void 0);
const Text = Symbol(void 0);
const Comment = Symbol(void 0);
const Static = Symbol(void 0);
const blockStack = [];
let currentBlock = null;
function openBlock(disableTracking = false) {
  blockStack.push(currentBlock = disableTracking ? null : []);
}
function closeBlock() {
  blockStack.pop();
  currentBlock = blockStack[blockStack.length - 1] || null;
}
let isBlockTreeEnabled = 1;
function setBlockTracking(value) {
  isBlockTreeEnabled += value;
}
function setupBlock(vnode) {
  vnode.dynamicChildren = isBlockTreeEnabled > 0 ? currentBlock || EMPTY_ARR : null;
  closeBlock();
  if (isBlockTreeEnabled > 0 && currentBlock) {
    currentBlock.push(vnode);
  }
  return vnode;
}
function createElementBlock(type2, props, children, patchFlag, dynamicProps, shapeFlag) {
  return setupBlock(createBaseVNode(
    type2,
    props,
    children,
    patchFlag,
    dynamicProps,
    shapeFlag,
    true
    /* isBlock */
  ));
}
function createBlock(type2, props, children, patchFlag, dynamicProps) {
  return setupBlock(createVNode(
    type2,
    props,
    children,
    patchFlag,
    dynamicProps,
    true
    /* isBlock: prevent a block from tracking itself */
  ));
}
function isVNode$1(value) {
  return value ? value.__v_isVNode === true : false;
}
function isSameVNodeType(n1, n2) {
  return n1.type === n2.type && n1.key === n2.key;
}
function transformVNodeArgs(transformer) {
}
const InternalObjectKey = `__vInternal`;
const normalizeKey = ({ key }) => key != null ? key : null;
const normalizeRef = ({ ref: ref3, ref_key, ref_for }) => {
  return ref3 != null ? isString$4(ref3) || isRef(ref3) || isFunction$2(ref3) ? { i: currentRenderingInstance, r: ref3, k: ref_key, f: !!ref_for } : ref3 : null;
};
function createBaseVNode(type2, props = null, children = null, patchFlag = 0, dynamicProps = null, shapeFlag = type2 === Fragment ? 0 : 1, isBlockNode = false, needFullChildrenNormalization = false) {
  const vnode = {
    __v_isVNode: true,
    __v_skip: true,
    type: type2,
    props,
    key: props && normalizeKey(props),
    ref: props && normalizeRef(props),
    scopeId: currentScopeId,
    slotScopeIds: null,
    children,
    component: null,
    suspense: null,
    ssContent: null,
    ssFallback: null,
    dirs: null,
    transition: null,
    el: null,
    anchor: null,
    target: null,
    targetAnchor: null,
    staticCount: 0,
    shapeFlag,
    patchFlag,
    dynamicProps,
    dynamicChildren: null,
    appContext: null,
    ctx: currentRenderingInstance
  };
  if (needFullChildrenNormalization) {
    normalizeChildren(vnode, children);
    if (shapeFlag & 128) {
      type2.normalize(vnode);
    }
  } else if (children) {
    vnode.shapeFlag |= isString$4(children) ? 8 : 16;
  }
  if (isBlockTreeEnabled > 0 && // avoid a block node from tracking itself
  !isBlockNode && // has current parent block
  currentBlock && // presence of a patch flag indicates this node needs patching on updates.
  // component nodes also should always be patched, because even if the
  // component doesn't need to update, it needs to persist the instance on to
  // the next vnode so that it can be properly unmounted later.
  (vnode.patchFlag > 0 || shapeFlag & 6) && // the EVENTS flag is only for hydration and if it is the only flag, the
  // vnode should not be considered dynamic due to handler caching.
  vnode.patchFlag !== 32) {
    currentBlock.push(vnode);
  }
  return vnode;
}
const createVNode = _createVNode;
function _createVNode(type2, props = null, children = null, patchFlag = 0, dynamicProps = null, isBlockNode = false) {
  if (!type2 || type2 === NULL_DYNAMIC_COMPONENT) {
    type2 = Comment;
  }
  if (isVNode$1(type2)) {
    const cloned = cloneVNode(
      type2,
      props,
      true
      /* mergeRef: true */
    );
    if (children) {
      normalizeChildren(cloned, children);
    }
    if (isBlockTreeEnabled > 0 && !isBlockNode && currentBlock) {
      if (cloned.shapeFlag & 6) {
        currentBlock[currentBlock.indexOf(type2)] = cloned;
      } else {
        currentBlock.push(cloned);
      }
    }
    cloned.patchFlag |= -2;
    return cloned;
  }
  if (isClassComponent(type2)) {
    type2 = type2.__vccOpts;
  }
  if (props) {
    props = guardReactiveProps(props);
    let { class: klass, style } = props;
    if (klass && !isString$4(klass)) {
      props.class = normalizeClass(klass);
    }
    if (isObject$5(style)) {
      if (isProxy(style) && !isArray$2(style)) {
        style = extend({}, style);
      }
      props.style = normalizeStyle(style);
    }
  }
  const shapeFlag = isString$4(type2) ? 1 : isSuspense(type2) ? 128 : isTeleport(type2) ? 64 : isObject$5(type2) ? 4 : isFunction$2(type2) ? 2 : 0;
  return createBaseVNode(type2, props, children, patchFlag, dynamicProps, shapeFlag, isBlockNode, true);
}
function guardReactiveProps(props) {
  if (!props)
    return null;
  return isProxy(props) || InternalObjectKey in props ? extend({}, props) : props;
}
function cloneVNode(vnode, extraProps, mergeRef = false) {
  const { props, ref: ref3, patchFlag, children } = vnode;
  const mergedProps = extraProps ? mergeProps(props || {}, extraProps) : props;
  const cloned = {
    __v_isVNode: true,
    __v_skip: true,
    type: vnode.type,
    props: mergedProps,
    key: mergedProps && normalizeKey(mergedProps),
    ref: extraProps && extraProps.ref ? (
      // #2078 in the case of <component :is="vnode" ref="extra"/>
      // if the vnode itself already has a ref, cloneVNode will need to merge
      // the refs so the single vnode can be set on multiple refs
      mergeRef && ref3 ? isArray$2(ref3) ? ref3.concat(normalizeRef(extraProps)) : [ref3, normalizeRef(extraProps)] : normalizeRef(extraProps)
    ) : ref3,
    scopeId: vnode.scopeId,
    slotScopeIds: vnode.slotScopeIds,
    children,
    target: vnode.target,
    targetAnchor: vnode.targetAnchor,
    staticCount: vnode.staticCount,
    shapeFlag: vnode.shapeFlag,
    // if the vnode is cloned with extra props, we can no longer assume its
    // existing patch flag to be reliable and need to add the FULL_PROPS flag.
    // note: preserve flag for fragments since they use the flag for children
    // fast paths only.
    patchFlag: extraProps && vnode.type !== Fragment ? patchFlag === -1 ? 16 : patchFlag | 16 : patchFlag,
    dynamicProps: vnode.dynamicProps,
    dynamicChildren: vnode.dynamicChildren,
    appContext: vnode.appContext,
    dirs: vnode.dirs,
    transition: vnode.transition,
    // These should technically only be non-null on mounted VNodes. However,
    // they *should* be copied for kept-alive vnodes. So we just always copy
    // them since them being non-null during a mount doesn't affect the logic as
    // they will simply be overwritten.
    component: vnode.component,
    suspense: vnode.suspense,
    ssContent: vnode.ssContent && cloneVNode(vnode.ssContent),
    ssFallback: vnode.ssFallback && cloneVNode(vnode.ssFallback),
    el: vnode.el,
    anchor: vnode.anchor,
    ctx: vnode.ctx,
    ce: vnode.ce
  };
  return cloned;
}
function createTextVNode(text2 = " ", flag = 0) {
  return createVNode(Text, null, text2, flag);
}
function createStaticVNode(content, numberOfNodes) {
  const vnode = createVNode(Static, null, content);
  vnode.staticCount = numberOfNodes;
  return vnode;
}
function createCommentVNode(text2 = "", asBlock = false) {
  return asBlock ? (openBlock(), createBlock(Comment, null, text2)) : createVNode(Comment, null, text2);
}
function normalizeVNode(child) {
  if (child == null || typeof child === "boolean") {
    return createVNode(Comment);
  } else if (isArray$2(child)) {
    return createVNode(
      Fragment,
      null,
      // #3666, avoid reference pollution when reusing vnode
      child.slice()
    );
  } else if (typeof child === "object") {
    return cloneIfMounted(child);
  } else {
    return createVNode(Text, null, String(child));
  }
}
function cloneIfMounted(child) {
  return child.el === null && child.patchFlag !== -1 || child.memo ? child : cloneVNode(child);
}
function normalizeChildren(vnode, children) {
  let type2 = 0;
  const { shapeFlag } = vnode;
  if (children == null) {
    children = null;
  } else if (isArray$2(children)) {
    type2 = 16;
  } else if (typeof children === "object") {
    if (shapeFlag & (1 | 64)) {
      const slot = children.default;
      if (slot) {
        slot._c && (slot._d = false);
        normalizeChildren(vnode, slot());
        slot._c && (slot._d = true);
      }
      return;
    } else {
      type2 = 32;
      const slotFlag = children._;
      if (!slotFlag && !(InternalObjectKey in children)) {
        children._ctx = currentRenderingInstance;
      } else if (slotFlag === 3 && currentRenderingInstance) {
        if (currentRenderingInstance.slots._ === 1) {
          children._ = 1;
        } else {
          children._ = 2;
          vnode.patchFlag |= 1024;
        }
      }
    }
  } else if (isFunction$2(children)) {
    children = { default: children, _ctx: currentRenderingInstance };
    type2 = 32;
  } else {
    children = String(children);
    if (shapeFlag & 64) {
      type2 = 16;
      children = [createTextVNode(children)];
    } else {
      type2 = 8;
    }
  }
  vnode.children = children;
  vnode.shapeFlag |= type2;
}
function mergeProps(...args) {
  const ret = {};
  for (let i2 = 0; i2 < args.length; i2++) {
    const toMerge = args[i2];
    for (const key in toMerge) {
      if (key === "class") {
        if (ret.class !== toMerge.class) {
          ret.class = normalizeClass([ret.class, toMerge.class]);
        }
      } else if (key === "style") {
        ret.style = normalizeStyle([ret.style, toMerge.style]);
      } else if (isOn(key)) {
        const existing = ret[key];
        const incoming = toMerge[key];
        if (incoming && existing !== incoming && !(isArray$2(existing) && existing.includes(incoming))) {
          ret[key] = existing ? [].concat(existing, incoming) : incoming;
        }
      } else if (key !== "") {
        ret[key] = toMerge[key];
      }
    }
  }
  return ret;
}
function invokeVNodeHook(hook, instance, vnode, prevVNode = null) {
  callWithAsyncErrorHandling(hook, instance, 7, [
    vnode,
    prevVNode
  ]);
}
const emptyAppContext = createAppContext();
let uid = 0;
function createComponentInstance(vnode, parent, suspense) {
  const type2 = vnode.type;
  const appContext = (parent ? parent.appContext : vnode.appContext) || emptyAppContext;
  const instance = {
    uid: uid++,
    vnode,
    type: type2,
    parent,
    appContext,
    root: null,
    next: null,
    subTree: null,
    effect: null,
    update: null,
    scope: new EffectScope(
      true
      /* detached */
    ),
    render: null,
    proxy: null,
    exposed: null,
    exposeProxy: null,
    withProxy: null,
    provides: parent ? parent.provides : Object.create(appContext.provides),
    accessCache: null,
    renderCache: [],
    // local resolved assets
    components: null,
    directives: null,
    // resolved props and emits options
    propsOptions: normalizePropsOptions(type2, appContext),
    emitsOptions: normalizeEmitsOptions(type2, appContext),
    // emit
    emit: null,
    emitted: null,
    // props default value
    propsDefaults: EMPTY_OBJ,
    // inheritAttrs
    inheritAttrs: type2.inheritAttrs,
    // state
    ctx: EMPTY_OBJ,
    data: EMPTY_OBJ,
    props: EMPTY_OBJ,
    attrs: EMPTY_OBJ,
    slots: EMPTY_OBJ,
    refs: EMPTY_OBJ,
    setupState: EMPTY_OBJ,
    setupContext: null,
    // suspense related
    suspense,
    suspenseId: suspense ? suspense.pendingId : 0,
    asyncDep: null,
    asyncResolved: false,
    // lifecycle hooks
    // not using enums here because it results in computed properties
    isMounted: false,
    isUnmounted: false,
    isDeactivated: false,
    bc: null,
    c: null,
    bm: null,
    m: null,
    bu: null,
    u: null,
    um: null,
    bum: null,
    da: null,
    a: null,
    rtg: null,
    rtc: null,
    ec: null,
    sp: null
  };
  {
    instance.ctx = { _: instance };
  }
  instance.root = parent ? parent.root : instance;
  instance.emit = emit.bind(null, instance);
  if (vnode.ce) {
    vnode.ce(instance);
  }
  return instance;
}
let currentInstance = null;
const getCurrentInstance = () => currentInstance || currentRenderingInstance;
const setCurrentInstance = (instance) => {
  currentInstance = instance;
  instance.scope.on();
};
const unsetCurrentInstance = () => {
  currentInstance && currentInstance.scope.off();
  currentInstance = null;
};
function isStatefulComponent(instance) {
  return instance.vnode.shapeFlag & 4;
}
let isInSSRComponentSetup = false;
function setupComponent(instance, isSSR = false) {
  isInSSRComponentSetup = isSSR;
  const { props, children } = instance.vnode;
  const isStateful = isStatefulComponent(instance);
  initProps(instance, props, isStateful, isSSR);
  initSlots(instance, children);
  const setupResult = isStateful ? setupStatefulComponent(instance, isSSR) : void 0;
  isInSSRComponentSetup = false;
  return setupResult;
}
function setupStatefulComponent(instance, isSSR) {
  const Component = instance.type;
  instance.accessCache = /* @__PURE__ */ Object.create(null);
  instance.proxy = markRaw(new Proxy(instance.ctx, PublicInstanceProxyHandlers));
  const { setup: setup8 } = Component;
  if (setup8) {
    const setupContext = instance.setupContext = setup8.length > 1 ? createSetupContext(instance) : null;
    setCurrentInstance(instance);
    pauseTracking();
    const setupResult = callWithErrorHandling(setup8, instance, 0, [instance.props, setupContext]);
    resetTracking();
    unsetCurrentInstance();
    if (isPromise$1(setupResult)) {
      setupResult.then(unsetCurrentInstance, unsetCurrentInstance);
      if (isSSR) {
        return setupResult.then((resolvedResult) => {
          handleSetupResult(instance, resolvedResult, isSSR);
        }).catch((e) => {
          handleError(
            e,
            instance,
            0
            /* ErrorCodes.SETUP_FUNCTION */
          );
        });
      } else {
        instance.asyncDep = setupResult;
      }
    } else {
      handleSetupResult(instance, setupResult, isSSR);
    }
  } else {
    finishComponentSetup(instance, isSSR);
  }
}
function handleSetupResult(instance, setupResult, isSSR) {
  if (isFunction$2(setupResult)) {
    if (instance.type.__ssrInlineRender) {
      instance.ssrRender = setupResult;
    } else {
      instance.render = setupResult;
    }
  } else if (isObject$5(setupResult)) {
    instance.setupState = proxyRefs(setupResult);
  } else ;
  finishComponentSetup(instance, isSSR);
}
let compile$2;
let installWithProxy;
function registerRuntimeCompiler(_compile) {
  compile$2 = _compile;
  installWithProxy = (i2) => {
    if (i2.render._rc) {
      i2.withProxy = new Proxy(i2.ctx, RuntimeCompiledPublicInstanceProxyHandlers);
    }
  };
}
const isRuntimeOnly = () => !compile$2;
function finishComponentSetup(instance, isSSR, skipOptions) {
  const Component = instance.type;
  if (!instance.render) {
    if (!isSSR && compile$2 && !Component.render) {
      const template = Component.template || resolveMergedOptions(instance).template;
      if (template) {
        const { isCustomElement, compilerOptions } = instance.appContext.config;
        const { delimiters, compilerOptions: componentCompilerOptions } = Component;
        const finalCompilerOptions = extend(extend({
          isCustomElement,
          delimiters
        }, compilerOptions), componentCompilerOptions);
        Component.render = compile$2(template, finalCompilerOptions);
      }
    }
    instance.render = Component.render || NOOP;
    if (installWithProxy) {
      installWithProxy(instance);
    }
  }
  {
    setCurrentInstance(instance);
    pauseTracking();
    applyOptions(instance);
    resetTracking();
    unsetCurrentInstance();
  }
}
function createAttrsProxy(instance) {
  return new Proxy(instance.attrs, {
    get(target2, key) {
      track(instance, "get", "$attrs");
      return target2[key];
    }
  });
}
function createSetupContext(instance) {
  const expose = (exposed) => {
    instance.exposed = exposed || {};
  };
  let attrs;
  {
    return {
      get attrs() {
        return attrs || (attrs = createAttrsProxy(instance));
      },
      slots: instance.slots,
      emit: instance.emit,
      expose
    };
  }
}
function getExposeProxy(instance) {
  if (instance.exposed) {
    return instance.exposeProxy || (instance.exposeProxy = new Proxy(proxyRefs(markRaw(instance.exposed)), {
      get(target2, key) {
        if (key in target2) {
          return target2[key];
        } else if (key in publicPropertiesMap) {
          return publicPropertiesMap[key](instance);
        }
      },
      has(target2, key) {
        return key in target2 || key in publicPropertiesMap;
      }
    }));
  }
}
function getComponentName(Component, includeInferred = true) {
  return isFunction$2(Component) ? Component.displayName || Component.name : Component.name || includeInferred && Component.__name;
}
function isClassComponent(value) {
  return isFunction$2(value) && "__vccOpts" in value;
}
const computed = ((getterOrOptions, debugOptions) => {
  return computed$1(getterOrOptions, debugOptions, isInSSRComponentSetup);
});
function defineProps() {
  return null;
}
function defineEmits() {
  return null;
}
function defineExpose(exposed) {
}
function withDefaults(props, defaults) {
  return null;
}
function useSlots() {
  return getContext().slots;
}
function useAttrs() {
  return getContext().attrs;
}
function getContext() {
  const i2 = getCurrentInstance();
  return i2.setupContext || (i2.setupContext = createSetupContext(i2));
}
function mergeDefaults(raw, defaults) {
  const props = isArray$2(raw) ? raw.reduce((normalized, p2) => (normalized[p2] = {}, normalized), {}) : raw;
  for (const key in defaults) {
    const opt = props[key];
    if (opt) {
      if (isArray$2(opt) || isFunction$2(opt)) {
        props[key] = { type: opt, default: defaults[key] };
      } else {
        opt.default = defaults[key];
      }
    } else if (opt === null) {
      props[key] = { default: defaults[key] };
    } else ;
  }
  return props;
}
function createPropsRestProxy(props, excludedKeys) {
  const ret = {};
  for (const key in props) {
    if (!excludedKeys.includes(key)) {
      Object.defineProperty(ret, key, {
        enumerable: true,
        get: () => props[key]
      });
    }
  }
  return ret;
}
function withAsyncContext(getAwaitable) {
  const ctx = getCurrentInstance();
  let awaitable = getAwaitable();
  unsetCurrentInstance();
  if (isPromise$1(awaitable)) {
    awaitable = awaitable.catch((e) => {
      setCurrentInstance(ctx);
      throw e;
    });
  }
  return [awaitable, () => setCurrentInstance(ctx)];
}
function h$2(type2, propsOrChildren, children) {
  const l2 = arguments.length;
  if (l2 === 2) {
    if (isObject$5(propsOrChildren) && !isArray$2(propsOrChildren)) {
      if (isVNode$1(propsOrChildren)) {
        return createVNode(type2, null, [propsOrChildren]);
      }
      return createVNode(type2, propsOrChildren);
    } else {
      return createVNode(type2, null, propsOrChildren);
    }
  } else {
    if (l2 > 3) {
      children = Array.prototype.slice.call(arguments, 2);
    } else if (l2 === 3 && isVNode$1(children)) {
      children = [children];
    }
    return createVNode(type2, propsOrChildren, children);
  }
}
const ssrContextKey = Symbol(``);
const useSSRContext = () => {
  {
    const ctx = inject(ssrContextKey);
    return ctx;
  }
};
function initCustomFormatter() {
  {
    return;
  }
}
function withMemo(memo, render2, cache2, index) {
  const cached = cache2[index];
  if (cached && isMemoSame(cached, memo)) {
    return cached;
  }
  const ret = render2();
  ret.memo = memo.slice();
  return cache2[index] = ret;
}
function isMemoSame(cached, memo) {
  const prev = cached.memo;
  if (prev.length != memo.length) {
    return false;
  }
  for (let i2 = 0; i2 < prev.length; i2++) {
    if (hasChanged(prev[i2], memo[i2])) {
      return false;
    }
  }
  if (isBlockTreeEnabled > 0 && currentBlock) {
    currentBlock.push(cached);
  }
  return true;
}
const version$a = "3.2.47";
const _ssrUtils = {
  createComponentInstance,
  setupComponent,
  renderComponentRoot,
  setCurrentRenderingInstance,
  isVNode: isVNode$1,
  normalizeVNode
};
const ssrUtils = _ssrUtils;
const resolveFilter = null;
const compatUtils = null;
const svgNS = "http://www.w3.org/2000/svg";
const doc = typeof document !== "undefined" ? document : null;
const templateContainer = doc && /* @__PURE__ */ doc.createElement("template");
const nodeOps = {
  insert: (child, parent, anchor) => {
    parent.insertBefore(child, anchor || null);
  },
  remove: (child) => {
    const parent = child.parentNode;
    if (parent) {
      parent.removeChild(child);
    }
  },
  createElement: (tag, isSVG, is, props) => {
    const el2 = isSVG ? doc.createElementNS(svgNS, tag) : doc.createElement(tag, is ? { is } : void 0);
    if (tag === "select" && props && props.multiple != null) {
      el2.setAttribute("multiple", props.multiple);
    }
    return el2;
  },
  createText: (text2) => doc.createTextNode(text2),
  createComment: (text2) => doc.createComment(text2),
  setText: (node, text2) => {
    node.nodeValue = text2;
  },
  setElementText: (el2, text2) => {
    el2.textContent = text2;
  },
  parentNode: (node) => node.parentNode,
  nextSibling: (node) => node.nextSibling,
  querySelector: (selector) => doc.querySelector(selector),
  setScopeId(el2, id2) {
    el2.setAttribute(id2, "");
  },
  // __UNSAFE__
  // Reason: innerHTML.
  // Static content here can only come from compiled templates.
  // As long as the user only uses trusted templates, this is safe.
  insertStaticContent(content, parent, anchor, isSVG, start, end) {
    const before = anchor ? anchor.previousSibling : parent.lastChild;
    if (start && (start === end || start.nextSibling)) {
      while (true) {
        parent.insertBefore(start.cloneNode(true), anchor);
        if (start === end || !(start = start.nextSibling))
          break;
      }
    } else {
      templateContainer.innerHTML = isSVG ? `<svg>${content}</svg>` : content;
      const template = templateContainer.content;
      if (isSVG) {
        const wrapper = template.firstChild;
        while (wrapper.firstChild) {
          template.appendChild(wrapper.firstChild);
        }
        template.removeChild(wrapper);
      }
      parent.insertBefore(template, anchor);
    }
    return [
      // first
      before ? before.nextSibling : parent.firstChild,
      // last
      anchor ? anchor.previousSibling : parent.lastChild
    ];
  }
};
function patchClass(el2, value, isSVG) {
  const transitionClasses = el2._vtc;
  if (transitionClasses) {
    value = (value ? [value, ...transitionClasses] : [...transitionClasses]).join(" ");
  }
  if (value == null) {
    el2.removeAttribute("class");
  } else if (isSVG) {
    el2.setAttribute("class", value);
  } else {
    el2.className = value;
  }
}
function patchStyle(el2, prev, next) {
  const style = el2.style;
  const isCssString = isString$4(next);
  if (next && !isCssString) {
    if (prev && !isString$4(prev)) {
      for (const key in prev) {
        if (next[key] == null) {
          setStyle(style, key, "");
        }
      }
    }
    for (const key in next) {
      setStyle(style, key, next[key]);
    }
  } else {
    const currentDisplay = style.display;
    if (isCssString) {
      if (prev !== next) {
        style.cssText = next;
      }
    } else if (prev) {
      el2.removeAttribute("style");
    }
    if ("_vod" in el2) {
      style.display = currentDisplay;
    }
  }
}
const importantRE = /\s*!important$/;
function setStyle(style, name2, val) {
  if (isArray$2(val)) {
    val.forEach((v2) => setStyle(style, name2, v2));
  } else {
    if (val == null)
      val = "";
    if (name2.startsWith("--")) {
      style.setProperty(name2, val);
    } else {
      const prefixed = autoPrefix(style, name2);
      if (importantRE.test(val)) {
        style.setProperty(hyphenate(prefixed), val.replace(importantRE, ""), "important");
      } else {
        style[prefixed] = val;
      }
    }
  }
}
const prefixes = ["Webkit", "Moz", "ms"];
const prefixCache = {};
function autoPrefix(style, rawName) {
  const cached = prefixCache[rawName];
  if (cached) {
    return cached;
  }
  let name2 = camelize$1(rawName);
  if (name2 !== "filter" && name2 in style) {
    return prefixCache[rawName] = name2;
  }
  name2 = capitalize$2(name2);
  for (let i2 = 0; i2 < prefixes.length; i2++) {
    const prefixed = prefixes[i2] + name2;
    if (prefixed in style) {
      return prefixCache[rawName] = prefixed;
    }
  }
  return rawName;
}
const xlinkNS = "http://www.w3.org/1999/xlink";
function patchAttr(el2, key, value, isSVG, instance) {
  if (isSVG && key.startsWith("xlink:")) {
    if (value == null) {
      el2.removeAttributeNS(xlinkNS, key.slice(6, key.length));
    } else {
      el2.setAttributeNS(xlinkNS, key, value);
    }
  } else {
    const isBoolean2 = isSpecialBooleanAttr(key);
    if (value == null || isBoolean2 && !includeBooleanAttr(value)) {
      el2.removeAttribute(key);
    } else {
      el2.setAttribute(key, isBoolean2 ? "" : value);
    }
  }
}
function patchDOMProp(el2, key, value, prevChildren, parentComponent, parentSuspense, unmountChildren) {
  if (key === "innerHTML" || key === "textContent") {
    if (prevChildren) {
      unmountChildren(prevChildren, parentComponent, parentSuspense);
    }
    el2[key] = value == null ? "" : value;
    return;
  }
  if (key === "value" && el2.tagName !== "PROGRESS" && // custom elements may use _value internally
  !el2.tagName.includes("-")) {
    el2._value = value;
    const newValue = value == null ? "" : value;
    if (el2.value !== newValue || // #4956: always set for OPTION elements because its value falls back to
    // textContent if no value attribute is present. And setting .value for
    // OPTION has no side effect
    el2.tagName === "OPTION") {
      el2.value = newValue;
    }
    if (value == null) {
      el2.removeAttribute(key);
    }
    return;
  }
  let needRemove = false;
  if (value === "" || value == null) {
    const type2 = typeof el2[key];
    if (type2 === "boolean") {
      value = includeBooleanAttr(value);
    } else if (value == null && type2 === "string") {
      value = "";
      needRemove = true;
    } else if (type2 === "number") {
      value = 0;
      needRemove = true;
    }
  }
  try {
    el2[key] = value;
  } catch (e) {
  }
  needRemove && el2.removeAttribute(key);
}
function addEventListener$1(el2, event, handler, options2) {
  el2.addEventListener(event, handler, options2);
}
function removeEventListener$1(el2, event, handler, options2) {
  el2.removeEventListener(event, handler, options2);
}
function patchEvent(el2, rawName, prevValue, nextValue, instance = null) {
  const invokers = el2._vei || (el2._vei = {});
  const existingInvoker = invokers[rawName];
  if (nextValue && existingInvoker) {
    existingInvoker.value = nextValue;
  } else {
    const [name2, options2] = parseName(rawName);
    if (nextValue) {
      const invoker = invokers[rawName] = createInvoker(nextValue, instance);
      addEventListener$1(el2, name2, invoker, options2);
    } else if (existingInvoker) {
      removeEventListener$1(el2, name2, existingInvoker, options2);
      invokers[rawName] = void 0;
    }
  }
}
const optionsModifierRE = /(?:Once|Passive|Capture)$/;
function parseName(name2) {
  let options2;
  if (optionsModifierRE.test(name2)) {
    options2 = {};
    let m2;
    while (m2 = name2.match(optionsModifierRE)) {
      name2 = name2.slice(0, name2.length - m2[0].length);
      options2[m2[0].toLowerCase()] = true;
    }
  }
  const event = name2[2] === ":" ? name2.slice(3) : hyphenate(name2.slice(2));
  return [event, options2];
}
let cachedNow = 0;
const p$3 = /* @__PURE__ */ Promise.resolve();
const getNow = () => cachedNow || (p$3.then(() => cachedNow = 0), cachedNow = Date.now());
function createInvoker(initialValue, instance) {
  const invoker = (e) => {
    if (!e._vts) {
      e._vts = Date.now();
    } else if (e._vts <= invoker.attached) {
      return;
    }
    callWithAsyncErrorHandling(patchStopImmediatePropagation(e, invoker.value), instance, 5, [e]);
  };
  invoker.value = initialValue;
  invoker.attached = getNow();
  return invoker;
}
function patchStopImmediatePropagation(e, value) {
  if (isArray$2(value)) {
    const originalStop = e.stopImmediatePropagation;
    e.stopImmediatePropagation = () => {
      originalStop.call(e);
      e._stopped = true;
    };
    return value.map((fn) => (e2) => !e2._stopped && fn && fn(e2));
  } else {
    return value;
  }
}
const nativeOnRE = /^on[a-z]/;
const patchProp = (el2, key, prevValue, nextValue, isSVG = false, prevChildren, parentComponent, parentSuspense, unmountChildren) => {
  if (key === "class") {
    patchClass(el2, nextValue, isSVG);
  } else if (key === "style") {
    patchStyle(el2, prevValue, nextValue);
  } else if (isOn(key)) {
    if (!isModelListener(key)) {
      patchEvent(el2, key, prevValue, nextValue, parentComponent);
    }
  } else if (key[0] === "." ? (key = key.slice(1), true) : key[0] === "^" ? (key = key.slice(1), false) : shouldSetAsProp(el2, key, nextValue, isSVG)) {
    patchDOMProp(el2, key, nextValue, prevChildren, parentComponent, parentSuspense, unmountChildren);
  } else {
    if (key === "true-value") {
      el2._trueValue = nextValue;
    } else if (key === "false-value") {
      el2._falseValue = nextValue;
    }
    patchAttr(el2, key, nextValue, isSVG);
  }
};
function shouldSetAsProp(el2, key, value, isSVG) {
  if (isSVG) {
    if (key === "innerHTML" || key === "textContent") {
      return true;
    }
    if (key in el2 && nativeOnRE.test(key) && isFunction$2(value)) {
      return true;
    }
    return false;
  }
  if (key === "spellcheck" || key === "draggable" || key === "translate") {
    return false;
  }
  if (key === "form") {
    return false;
  }
  if (key === "list" && el2.tagName === "INPUT") {
    return false;
  }
  if (key === "type" && el2.tagName === "TEXTAREA") {
    return false;
  }
  if (nativeOnRE.test(key) && isString$4(value)) {
    return false;
  }
  return key in el2;
}
function defineCustomElement(options2, hydrate2) {
  const Comp = defineComponent(options2);
  class VueCustomElement extends VueElement {
    constructor(initialProps) {
      super(Comp, initialProps, hydrate2);
    }
  }
  VueCustomElement.def = Comp;
  return VueCustomElement;
}
const defineSSRCustomElement = ((options2) => {
  return defineCustomElement(options2, hydrate);
});
const BaseClass = typeof HTMLElement !== "undefined" ? HTMLElement : class {
};
class VueElement extends BaseClass {
  constructor(_def, _props = {}, hydrate2) {
    super();
    this._def = _def;
    this._props = _props;
    this._instance = null;
    this._connected = false;
    this._resolved = false;
    this._numberProps = null;
    if (this.shadowRoot && hydrate2) {
      hydrate2(this._createVNode(), this.shadowRoot);
    } else {
      this.attachShadow({ mode: "open" });
      if (!this._def.__asyncLoader) {
        this._resolveProps(this._def);
      }
    }
  }
  connectedCallback() {
    this._connected = true;
    if (!this._instance) {
      if (this._resolved) {
        this._update();
      } else {
        this._resolveDef();
      }
    }
  }
  disconnectedCallback() {
    this._connected = false;
    nextTick(() => {
      if (!this._connected) {
        render$1c(null, this.shadowRoot);
        this._instance = null;
      }
    });
  }
  /**
   * resolve inner component definition (handle possible async component)
   */
  _resolveDef() {
    this._resolved = true;
    for (let i2 = 0; i2 < this.attributes.length; i2++) {
      this._setAttr(this.attributes[i2].name);
    }
    new MutationObserver((mutations) => {
      for (const m2 of mutations) {
        this._setAttr(m2.attributeName);
      }
    }).observe(this, { attributes: true });
    const resolve3 = (def2, isAsync = false) => {
      const { props, styles } = def2;
      let numberProps;
      if (props && !isArray$2(props)) {
        for (const key in props) {
          const opt = props[key];
          if (opt === Number || opt && opt.type === Number) {
            if (key in this._props) {
              this._props[key] = toNumber(this._props[key]);
            }
            (numberProps || (numberProps = /* @__PURE__ */ Object.create(null)))[camelize$1(key)] = true;
          }
        }
      }
      this._numberProps = numberProps;
      if (isAsync) {
        this._resolveProps(def2);
      }
      this._applyStyles(styles);
      this._update();
    };
    const asyncDef = this._def.__asyncLoader;
    if (asyncDef) {
      asyncDef().then((def2) => resolve3(def2, true));
    } else {
      resolve3(this._def);
    }
  }
  _resolveProps(def2) {
    const { props } = def2;
    const declaredPropKeys = isArray$2(props) ? props : Object.keys(props || {});
    for (const key of Object.keys(this)) {
      if (key[0] !== "_" && declaredPropKeys.includes(key)) {
        this._setProp(key, this[key], true, false);
      }
    }
    for (const key of declaredPropKeys.map(camelize$1)) {
      Object.defineProperty(this, key, {
        get() {
          return this._getProp(key);
        },
        set(val) {
          this._setProp(key, val);
        }
      });
    }
  }
  _setAttr(key) {
    let value = this.getAttribute(key);
    const camelKey = camelize$1(key);
    if (this._numberProps && this._numberProps[camelKey]) {
      value = toNumber(value);
    }
    this._setProp(camelKey, value, false);
  }
  /**
   * @internal
   */
  _getProp(key) {
    return this._props[key];
  }
  /**
   * @internal
   */
  _setProp(key, val, shouldReflect = true, shouldUpdate = true) {
    if (val !== this._props[key]) {
      this._props[key] = val;
      if (shouldUpdate && this._instance) {
        this._update();
      }
      if (shouldReflect) {
        if (val === true) {
          this.setAttribute(hyphenate(key), "");
        } else if (typeof val === "string" || typeof val === "number") {
          this.setAttribute(hyphenate(key), val + "");
        } else if (!val) {
          this.removeAttribute(hyphenate(key));
        }
      }
    }
  }
  _update() {
    render$1c(this._createVNode(), this.shadowRoot);
  }
  _createVNode() {
    const vnode = createVNode(this._def, extend({}, this._props));
    if (!this._instance) {
      vnode.ce = (instance) => {
        this._instance = instance;
        instance.isCE = true;
        const dispatch = (event, args) => {
          this.dispatchEvent(new CustomEvent(event, {
            detail: args
          }));
        };
        instance.emit = (event, ...args) => {
          dispatch(event, args);
          if (hyphenate(event) !== event) {
            dispatch(hyphenate(event), args);
          }
        };
        let parent = this;
        while (parent = parent && (parent.parentNode || parent.host)) {
          if (parent instanceof VueElement) {
            instance.parent = parent._instance;
            instance.provides = parent._instance.provides;
            break;
          }
        }
      };
    }
    return vnode;
  }
  _applyStyles(styles) {
    if (styles) {
      styles.forEach((css) => {
        const s2 = document.createElement("style");
        s2.textContent = css;
        this.shadowRoot.appendChild(s2);
      });
    }
  }
}
function useCssModule(name2 = "$style") {
  {
    const instance = getCurrentInstance();
    if (!instance) {
      return EMPTY_OBJ;
    }
    const modules = instance.type.__cssModules;
    if (!modules) {
      return EMPTY_OBJ;
    }
    const mod = modules[name2];
    if (!mod) {
      return EMPTY_OBJ;
    }
    return mod;
  }
}
function useCssVars(getter) {
  const instance = getCurrentInstance();
  if (!instance) {
    return;
  }
  const updateTeleports = instance.ut = (vars = getter(instance.proxy)) => {
    Array.from(document.querySelectorAll(`[data-v-owner="${instance.uid}"]`)).forEach((node) => setVarsOnNode(node, vars));
  };
  const setVars = () => {
    const vars = getter(instance.proxy);
    setVarsOnVNode(instance.subTree, vars);
    updateTeleports(vars);
  };
  watchPostEffect(setVars);
  onMounted(() => {
    const ob = new MutationObserver(setVars);
    ob.observe(instance.subTree.el.parentNode, { childList: true });
    onUnmounted(() => ob.disconnect());
  });
}
function setVarsOnVNode(vnode, vars) {
  if (vnode.shapeFlag & 128) {
    const suspense = vnode.suspense;
    vnode = suspense.activeBranch;
    if (suspense.pendingBranch && !suspense.isHydrating) {
      suspense.effects.push(() => {
        setVarsOnVNode(suspense.activeBranch, vars);
      });
    }
  }
  while (vnode.component) {
    vnode = vnode.component.subTree;
  }
  if (vnode.shapeFlag & 1 && vnode.el) {
    setVarsOnNode(vnode.el, vars);
  } else if (vnode.type === Fragment) {
    vnode.children.forEach((c2) => setVarsOnVNode(c2, vars));
  } else if (vnode.type === Static) {
    let { el: el2, anchor } = vnode;
    while (el2) {
      setVarsOnNode(el2, vars);
      if (el2 === anchor)
        break;
      el2 = el2.nextSibling;
    }
  }
}
function setVarsOnNode(el2, vars) {
  if (el2.nodeType === 1) {
    const style = el2.style;
    for (const key in vars) {
      style.setProperty(`--${key}`, vars[key]);
    }
  }
}
const TRANSITION$2 = "transition";
const ANIMATION = "animation";
const Transition = (props, { slots }) => h$2(BaseTransition, resolveTransitionProps(props), slots);
Transition.displayName = "Transition";
const DOMTransitionPropsValidators = {
  name: String,
  type: String,
  css: {
    type: Boolean,
    default: true
  },
  duration: [String, Number, Object],
  enterFromClass: String,
  enterActiveClass: String,
  enterToClass: String,
  appearFromClass: String,
  appearActiveClass: String,
  appearToClass: String,
  leaveFromClass: String,
  leaveActiveClass: String,
  leaveToClass: String
};
const TransitionPropsValidators = Transition.props = /* @__PURE__ */ extend({}, BaseTransition.props, DOMTransitionPropsValidators);
const callHook = (hook, args = []) => {
  if (isArray$2(hook)) {
    hook.forEach((h2) => h2(...args));
  } else if (hook) {
    hook(...args);
  }
};
const hasExplicitCallback = (hook) => {
  return hook ? isArray$2(hook) ? hook.some((h2) => h2.length > 1) : hook.length > 1 : false;
};
function resolveTransitionProps(rawProps) {
  const baseProps = {};
  for (const key in rawProps) {
    if (!(key in DOMTransitionPropsValidators)) {
      baseProps[key] = rawProps[key];
    }
  }
  if (rawProps.css === false) {
    return baseProps;
  }
  const { name: name2 = "v", type: type2, duration, enterFromClass = `${name2}-enter-from`, enterActiveClass = `${name2}-enter-active`, enterToClass = `${name2}-enter-to`, appearFromClass = enterFromClass, appearActiveClass = enterActiveClass, appearToClass = enterToClass, leaveFromClass = `${name2}-leave-from`, leaveActiveClass = `${name2}-leave-active`, leaveToClass = `${name2}-leave-to` } = rawProps;
  const durations = normalizeDuration(duration);
  const enterDuration = durations && durations[0];
  const leaveDuration = durations && durations[1];
  const { onBeforeEnter, onEnter, onEnterCancelled, onLeave, onLeaveCancelled, onBeforeAppear = onBeforeEnter, onAppear = onEnter, onAppearCancelled = onEnterCancelled } = baseProps;
  const finishEnter = (el2, isAppear, done) => {
    removeTransitionClass(el2, isAppear ? appearToClass : enterToClass);
    removeTransitionClass(el2, isAppear ? appearActiveClass : enterActiveClass);
    done && done();
  };
  const finishLeave = (el2, done) => {
    el2._isLeaving = false;
    removeTransitionClass(el2, leaveFromClass);
    removeTransitionClass(el2, leaveToClass);
    removeTransitionClass(el2, leaveActiveClass);
    done && done();
  };
  const makeEnterHook = (isAppear) => {
    return (el2, done) => {
      const hook = isAppear ? onAppear : onEnter;
      const resolve3 = () => finishEnter(el2, isAppear, done);
      callHook(hook, [el2, resolve3]);
      nextFrame$2(() => {
        removeTransitionClass(el2, isAppear ? appearFromClass : enterFromClass);
        addTransitionClass(el2, isAppear ? appearToClass : enterToClass);
        if (!hasExplicitCallback(hook)) {
          whenTransitionEnds(el2, type2, enterDuration, resolve3);
        }
      });
    };
  };
  return extend(baseProps, {
    onBeforeEnter(el2) {
      callHook(onBeforeEnter, [el2]);
      addTransitionClass(el2, enterFromClass);
      addTransitionClass(el2, enterActiveClass);
    },
    onBeforeAppear(el2) {
      callHook(onBeforeAppear, [el2]);
      addTransitionClass(el2, appearFromClass);
      addTransitionClass(el2, appearActiveClass);
    },
    onEnter: makeEnterHook(false),
    onAppear: makeEnterHook(true),
    onLeave(el2, done) {
      el2._isLeaving = true;
      const resolve3 = () => finishLeave(el2, done);
      addTransitionClass(el2, leaveFromClass);
      forceReflow();
      addTransitionClass(el2, leaveActiveClass);
      nextFrame$2(() => {
        if (!el2._isLeaving) {
          return;
        }
        removeTransitionClass(el2, leaveFromClass);
        addTransitionClass(el2, leaveToClass);
        if (!hasExplicitCallback(onLeave)) {
          whenTransitionEnds(el2, type2, leaveDuration, resolve3);
        }
      });
      callHook(onLeave, [el2, resolve3]);
    },
    onEnterCancelled(el2) {
      finishEnter(el2, false);
      callHook(onEnterCancelled, [el2]);
    },
    onAppearCancelled(el2) {
      finishEnter(el2, true);
      callHook(onAppearCancelled, [el2]);
    },
    onLeaveCancelled(el2) {
      finishLeave(el2);
      callHook(onLeaveCancelled, [el2]);
    }
  });
}
function normalizeDuration(duration) {
  if (duration == null) {
    return null;
  } else if (isObject$5(duration)) {
    return [NumberOf(duration.enter), NumberOf(duration.leave)];
  } else {
    const n2 = NumberOf(duration);
    return [n2, n2];
  }
}
function NumberOf(val) {
  const res = toNumber(val);
  return res;
}
function addTransitionClass(el2, cls) {
  cls.split(/\s+/).forEach((c2) => c2 && el2.classList.add(c2));
  (el2._vtc || (el2._vtc = /* @__PURE__ */ new Set())).add(cls);
}
function removeTransitionClass(el2, cls) {
  cls.split(/\s+/).forEach((c2) => c2 && el2.classList.remove(c2));
  const { _vtc } = el2;
  if (_vtc) {
    _vtc.delete(cls);
    if (!_vtc.size) {
      el2._vtc = void 0;
    }
  }
}
function nextFrame$2(cb) {
  requestAnimationFrame(() => {
    requestAnimationFrame(cb);
  });
}
let endId = 0;
function whenTransitionEnds(el2, expectedType, explicitTimeout, resolve3) {
  const id2 = el2._endId = ++endId;
  const resolveIfNotStale = () => {
    if (id2 === el2._endId) {
      resolve3();
    }
  };
  if (explicitTimeout) {
    return setTimeout(resolveIfNotStale, explicitTimeout);
  }
  const { type: type2, timeout, propCount } = getTransitionInfo(el2, expectedType);
  if (!type2) {
    return resolve3();
  }
  const endEvent = type2 + "end";
  let ended = 0;
  const end = () => {
    el2.removeEventListener(endEvent, onEnd);
    resolveIfNotStale();
  };
  const onEnd = (e) => {
    if (e.target === el2 && ++ended >= propCount) {
      end();
    }
  };
  setTimeout(() => {
    if (ended < propCount) {
      end();
    }
  }, timeout + 1);
  el2.addEventListener(endEvent, onEnd);
}
function getTransitionInfo(el2, expectedType) {
  const styles = window.getComputedStyle(el2);
  const getStyleProperties = (key) => (styles[key] || "").split(", ");
  const transitionDelays = getStyleProperties(`${TRANSITION$2}Delay`);
  const transitionDurations = getStyleProperties(`${TRANSITION$2}Duration`);
  const transitionTimeout = getTimeout(transitionDelays, transitionDurations);
  const animationDelays = getStyleProperties(`${ANIMATION}Delay`);
  const animationDurations = getStyleProperties(`${ANIMATION}Duration`);
  const animationTimeout = getTimeout(animationDelays, animationDurations);
  let type2 = null;
  let timeout = 0;
  let propCount = 0;
  if (expectedType === TRANSITION$2) {
    if (transitionTimeout > 0) {
      type2 = TRANSITION$2;
      timeout = transitionTimeout;
      propCount = transitionDurations.length;
    }
  } else if (expectedType === ANIMATION) {
    if (animationTimeout > 0) {
      type2 = ANIMATION;
      timeout = animationTimeout;
      propCount = animationDurations.length;
    }
  } else {
    timeout = Math.max(transitionTimeout, animationTimeout);
    type2 = timeout > 0 ? transitionTimeout > animationTimeout ? TRANSITION$2 : ANIMATION : null;
    propCount = type2 ? type2 === TRANSITION$2 ? transitionDurations.length : animationDurations.length : 0;
  }
  const hasTransform = type2 === TRANSITION$2 && /\b(transform|all)(,|$)/.test(getStyleProperties(`${TRANSITION$2}Property`).toString());
  return {
    type: type2,
    timeout,
    propCount,
    hasTransform
  };
}
function getTimeout(delays, durations) {
  while (delays.length < durations.length) {
    delays = delays.concat(delays);
  }
  return Math.max(...durations.map((d2, i2) => toMs(d2) + toMs(delays[i2])));
}
function toMs(s2) {
  return Number(s2.slice(0, -1).replace(",", ".")) * 1e3;
}
function forceReflow() {
  return document.body.offsetHeight;
}
const positionMap = /* @__PURE__ */ new WeakMap();
const newPositionMap = /* @__PURE__ */ new WeakMap();
const TransitionGroupImpl = {
  name: "TransitionGroup",
  props: /* @__PURE__ */ extend({}, TransitionPropsValidators, {
    tag: String,
    moveClass: String
  }),
  setup(props, { slots }) {
    const instance = getCurrentInstance();
    const state = useTransitionState();
    let prevChildren;
    let children;
    onUpdated(() => {
      if (!prevChildren.length) {
        return;
      }
      const moveClass = props.moveClass || `${props.name || "v"}-move`;
      if (!hasCSSTransform(prevChildren[0].el, instance.vnode.el, moveClass)) {
        return;
      }
      prevChildren.forEach(callPendingCbs);
      prevChildren.forEach(recordPosition);
      const movedChildren = prevChildren.filter(applyTranslation);
      forceReflow();
      movedChildren.forEach((c2) => {
        const el2 = c2.el;
        const style = el2.style;
        addTransitionClass(el2, moveClass);
        style.transform = style.webkitTransform = style.transitionDuration = "";
        const cb = el2._moveCb = (e) => {
          if (e && e.target !== el2) {
            return;
          }
          if (!e || /transform$/.test(e.propertyName)) {
            el2.removeEventListener("transitionend", cb);
            el2._moveCb = null;
            removeTransitionClass(el2, moveClass);
          }
        };
        el2.addEventListener("transitionend", cb);
      });
    });
    return () => {
      const rawProps = toRaw(props);
      const cssTransitionProps = resolveTransitionProps(rawProps);
      let tag = rawProps.tag || Fragment;
      prevChildren = children;
      children = slots.default ? getTransitionRawChildren(slots.default()) : [];
      for (let i2 = 0; i2 < children.length; i2++) {
        const child = children[i2];
        if (child.key != null) {
          setTransitionHooks(child, resolveTransitionHooks(child, cssTransitionProps, state, instance));
        }
      }
      if (prevChildren) {
        for (let i2 = 0; i2 < prevChildren.length; i2++) {
          const child = prevChildren[i2];
          setTransitionHooks(child, resolveTransitionHooks(child, cssTransitionProps, state, instance));
          positionMap.set(child, child.el.getBoundingClientRect());
        }
      }
      return createVNode(tag, null, children);
    };
  }
};
const removeMode = (props) => delete props.mode;
/* @__PURE__ */ removeMode(TransitionGroupImpl.props);
const TransitionGroup = TransitionGroupImpl;
function callPendingCbs(c2) {
  const el2 = c2.el;
  if (el2._moveCb) {
    el2._moveCb();
  }
  if (el2._enterCb) {
    el2._enterCb();
  }
}
function recordPosition(c2) {
  newPositionMap.set(c2, c2.el.getBoundingClientRect());
}
function applyTranslation(c2) {
  const oldPos = positionMap.get(c2);
  const newPos = newPositionMap.get(c2);
  const dx = oldPos.left - newPos.left;
  const dy = oldPos.top - newPos.top;
  if (dx || dy) {
    const s2 = c2.el.style;
    s2.transform = s2.webkitTransform = `translate(${dx}px,${dy}px)`;
    s2.transitionDuration = "0s";
    return c2;
  }
}
function hasCSSTransform(el2, root2, moveClass) {
  const clone = el2.cloneNode();
  if (el2._vtc) {
    el2._vtc.forEach((cls) => {
      cls.split(/\s+/).forEach((c2) => c2 && clone.classList.remove(c2));
    });
  }
  moveClass.split(/\s+/).forEach((c2) => c2 && clone.classList.add(c2));
  clone.style.display = "none";
  const container = root2.nodeType === 1 ? root2 : root2.parentNode;
  container.appendChild(clone);
  const { hasTransform } = getTransitionInfo(clone);
  container.removeChild(clone);
  return hasTransform;
}
const getModelAssigner = (vnode) => {
  const fn = vnode.props["onUpdate:modelValue"] || false;
  return isArray$2(fn) ? (value) => invokeArrayFns(fn, value) : fn;
};
function onCompositionStart(e) {
  e.target.composing = true;
}
function onCompositionEnd(e) {
  const target2 = e.target;
  if (target2.composing) {
    target2.composing = false;
    target2.dispatchEvent(new Event("input"));
  }
}
const vModelText = {
  created(el2, { modifiers: { lazy, trim, number: number2 } }, vnode) {
    el2._assign = getModelAssigner(vnode);
    const castToNumber = number2 || vnode.props && vnode.props.type === "number";
    addEventListener$1(el2, lazy ? "change" : "input", (e) => {
      if (e.target.composing)
        return;
      let domValue = el2.value;
      if (trim) {
        domValue = domValue.trim();
      }
      if (castToNumber) {
        domValue = looseToNumber(domValue);
      }
      el2._assign(domValue);
    });
    if (trim) {
      addEventListener$1(el2, "change", () => {
        el2.value = el2.value.trim();
      });
    }
    if (!lazy) {
      addEventListener$1(el2, "compositionstart", onCompositionStart);
      addEventListener$1(el2, "compositionend", onCompositionEnd);
      addEventListener$1(el2, "change", onCompositionEnd);
    }
  },
  // set value on mounted so it's after min/max for type="range"
  mounted(el2, { value }) {
    el2.value = value == null ? "" : value;
  },
  beforeUpdate(el2, { value, modifiers: { lazy, trim, number: number2 } }, vnode) {
    el2._assign = getModelAssigner(vnode);
    if (el2.composing)
      return;
    if (document.activeElement === el2 && el2.type !== "range") {
      if (lazy) {
        return;
      }
      if (trim && el2.value.trim() === value) {
        return;
      }
      if ((number2 || el2.type === "number") && looseToNumber(el2.value) === value) {
        return;
      }
    }
    const newValue = value == null ? "" : value;
    if (el2.value !== newValue) {
      el2.value = newValue;
    }
  }
};
const vModelCheckbox = {
  // #4096 array checkboxes need to be deep traversed
  deep: true,
  created(el2, _2, vnode) {
    el2._assign = getModelAssigner(vnode);
    addEventListener$1(el2, "change", () => {
      const modelValue = el2._modelValue;
      const elementValue = getValue(el2);
      const checked = el2.checked;
      const assign2 = el2._assign;
      if (isArray$2(modelValue)) {
        const index = looseIndexOf(modelValue, elementValue);
        const found = index !== -1;
        if (checked && !found) {
          assign2(modelValue.concat(elementValue));
        } else if (!checked && found) {
          const filtered = [...modelValue];
          filtered.splice(index, 1);
          assign2(filtered);
        }
      } else if (isSet(modelValue)) {
        const cloned = new Set(modelValue);
        if (checked) {
          cloned.add(elementValue);
        } else {
          cloned.delete(elementValue);
        }
        assign2(cloned);
      } else {
        assign2(getCheckboxValue(el2, checked));
      }
    });
  },
  // set initial checked on mount to wait for true-value/false-value
  mounted: setChecked,
  beforeUpdate(el2, binding, vnode) {
    el2._assign = getModelAssigner(vnode);
    setChecked(el2, binding, vnode);
  }
};
function setChecked(el2, { value, oldValue }, vnode) {
  el2._modelValue = value;
  if (isArray$2(value)) {
    el2.checked = looseIndexOf(value, vnode.props.value) > -1;
  } else if (isSet(value)) {
    el2.checked = value.has(vnode.props.value);
  } else if (value !== oldValue) {
    el2.checked = looseEqual(value, getCheckboxValue(el2, true));
  }
}
const vModelRadio = {
  created(el2, { value }, vnode) {
    el2.checked = looseEqual(value, vnode.props.value);
    el2._assign = getModelAssigner(vnode);
    addEventListener$1(el2, "change", () => {
      el2._assign(getValue(el2));
    });
  },
  beforeUpdate(el2, { value, oldValue }, vnode) {
    el2._assign = getModelAssigner(vnode);
    if (value !== oldValue) {
      el2.checked = looseEqual(value, vnode.props.value);
    }
  }
};
const vModelSelect = {
  // <select multiple> value need to be deep traversed
  deep: true,
  created(el2, { value, modifiers: { number: number2 } }, vnode) {
    const isSetModel = isSet(value);
    addEventListener$1(el2, "change", () => {
      const selectedVal = Array.prototype.filter.call(el2.options, (o2) => o2.selected).map((o2) => number2 ? looseToNumber(getValue(o2)) : getValue(o2));
      el2._assign(el2.multiple ? isSetModel ? new Set(selectedVal) : selectedVal : selectedVal[0]);
    });
    el2._assign = getModelAssigner(vnode);
  },
  // set value in mounted & updated because <select> relies on its children
  // <option>s.
  mounted(el2, { value }) {
    setSelected(el2, value);
  },
  beforeUpdate(el2, _binding, vnode) {
    el2._assign = getModelAssigner(vnode);
  },
  updated(el2, { value }) {
    setSelected(el2, value);
  }
};
function setSelected(el2, value) {
  const isMultiple = el2.multiple;
  if (isMultiple && !isArray$2(value) && !isSet(value)) {
    return;
  }
  for (let i2 = 0, l2 = el2.options.length; i2 < l2; i2++) {
    const option = el2.options[i2];
    const optionValue = getValue(option);
    if (isMultiple) {
      if (isArray$2(value)) {
        option.selected = looseIndexOf(value, optionValue) > -1;
      } else {
        option.selected = value.has(optionValue);
      }
    } else {
      if (looseEqual(getValue(option), value)) {
        if (el2.selectedIndex !== i2)
          el2.selectedIndex = i2;
        return;
      }
    }
  }
  if (!isMultiple && el2.selectedIndex !== -1) {
    el2.selectedIndex = -1;
  }
}
function getValue(el2) {
  return "_value" in el2 ? el2._value : el2.value;
}
function getCheckboxValue(el2, checked) {
  const key = checked ? "_trueValue" : "_falseValue";
  return key in el2 ? el2[key] : checked;
}
const vModelDynamic = {
  created(el2, binding, vnode) {
    callModelHook(el2, binding, vnode, null, "created");
  },
  mounted(el2, binding, vnode) {
    callModelHook(el2, binding, vnode, null, "mounted");
  },
  beforeUpdate(el2, binding, vnode, prevVNode) {
    callModelHook(el2, binding, vnode, prevVNode, "beforeUpdate");
  },
  updated(el2, binding, vnode, prevVNode) {
    callModelHook(el2, binding, vnode, prevVNode, "updated");
  }
};
function resolveDynamicModel(tagName, type2) {
  switch (tagName) {
    case "SELECT":
      return vModelSelect;
    case "TEXTAREA":
      return vModelText;
    default:
      switch (type2) {
        case "checkbox":
          return vModelCheckbox;
        case "radio":
          return vModelRadio;
        default:
          return vModelText;
      }
  }
}
function callModelHook(el2, binding, vnode, prevVNode, hook) {
  const modelToUse = resolveDynamicModel(el2.tagName, vnode.props && vnode.props.type);
  const fn = modelToUse[hook];
  fn && fn(el2, binding, vnode, prevVNode);
}
function initVModelForSSR() {
  vModelText.getSSRProps = ({ value }) => ({ value });
  vModelRadio.getSSRProps = ({ value }, vnode) => {
    if (vnode.props && looseEqual(vnode.props.value, value)) {
      return { checked: true };
    }
  };
  vModelCheckbox.getSSRProps = ({ value }, vnode) => {
    if (isArray$2(value)) {
      if (vnode.props && looseIndexOf(value, vnode.props.value) > -1) {
        return { checked: true };
      }
    } else if (isSet(value)) {
      if (vnode.props && value.has(vnode.props.value)) {
        return { checked: true };
      }
    } else if (value) {
      return { checked: true };
    }
  };
  vModelDynamic.getSSRProps = (binding, vnode) => {
    if (typeof vnode.type !== "string") {
      return;
    }
    const modelToUse = resolveDynamicModel(
      // resolveDynamicModel expects an uppercase tag name, but vnode.type is lowercase
      vnode.type.toUpperCase(),
      vnode.props && vnode.props.type
    );
    if (modelToUse.getSSRProps) {
      return modelToUse.getSSRProps(binding, vnode);
    }
  };
}
const systemModifiers = ["ctrl", "shift", "alt", "meta"];
const modifierGuards = {
  stop: (e) => e.stopPropagation(),
  prevent: (e) => e.preventDefault(),
  self: (e) => e.target !== e.currentTarget,
  ctrl: (e) => !e.ctrlKey,
  shift: (e) => !e.shiftKey,
  alt: (e) => !e.altKey,
  meta: (e) => !e.metaKey,
  left: (e) => "button" in e && e.button !== 0,
  middle: (e) => "button" in e && e.button !== 1,
  right: (e) => "button" in e && e.button !== 2,
  exact: (e, modifiers) => systemModifiers.some((m2) => e[`${m2}Key`] && !modifiers.includes(m2))
};
const withModifiers = (fn, modifiers) => {
  return (event, ...args) => {
    for (let i2 = 0; i2 < modifiers.length; i2++) {
      const guard = modifierGuards[modifiers[i2]];
      if (guard && guard(event, modifiers))
        return;
    }
    return fn(event, ...args);
  };
};
const keyNames = {
  esc: "escape",
  space: " ",
  up: "arrow-up",
  left: "arrow-left",
  right: "arrow-right",
  down: "arrow-down",
  delete: "backspace"
};
const withKeys = (fn, modifiers) => {
  return (event) => {
    if (!("key" in event)) {
      return;
    }
    const eventKey = hyphenate(event.key);
    if (modifiers.some((k2) => k2 === eventKey || keyNames[k2] === eventKey)) {
      return fn(event);
    }
  };
};
const vShow = {
  beforeMount(el2, { value }, { transition }) {
    el2._vod = el2.style.display === "none" ? "" : el2.style.display;
    if (transition && value) {
      transition.beforeEnter(el2);
    } else {
      setDisplay(el2, value);
    }
  },
  mounted(el2, { value }, { transition }) {
    if (transition && value) {
      transition.enter(el2);
    }
  },
  updated(el2, { value, oldValue }, { transition }) {
    if (!value === !oldValue)
      return;
    if (transition) {
      if (value) {
        transition.beforeEnter(el2);
        setDisplay(el2, true);
        transition.enter(el2);
      } else {
        transition.leave(el2, () => {
          setDisplay(el2, false);
        });
      }
    } else {
      setDisplay(el2, value);
    }
  },
  beforeUnmount(el2, { value }) {
    setDisplay(el2, value);
  }
};
function setDisplay(el2, value) {
  el2.style.display = value ? el2._vod : "none";
}
function initVShowForSSR() {
  vShow.getSSRProps = ({ value }) => {
    if (!value) {
      return { style: { display: "none" } };
    }
  };
}
const rendererOptions = /* @__PURE__ */ extend({ patchProp }, nodeOps);
let renderer$1;
let enabledHydration = false;
function ensureRenderer() {
  return renderer$1 || (renderer$1 = createRenderer(rendererOptions));
}
function ensureHydrationRenderer() {
  renderer$1 = enabledHydration ? renderer$1 : createHydrationRenderer(rendererOptions);
  enabledHydration = true;
  return renderer$1;
}
const render$1c = ((...args) => {
  ensureRenderer().render(...args);
});
const hydrate = ((...args) => {
  ensureHydrationRenderer().hydrate(...args);
});
const createApp = ((...args) => {
  const app2 = ensureRenderer().createApp(...args);
  const { mount } = app2;
  app2.mount = (containerOrSelector) => {
    const container = normalizeContainer(containerOrSelector);
    if (!container)
      return;
    const component = app2._component;
    if (!isFunction$2(component) && !component.render && !component.template) {
      component.template = container.innerHTML;
    }
    container.innerHTML = "";
    const proxy = mount(container, false, container instanceof SVGElement);
    if (container instanceof Element) {
      container.removeAttribute("v-cloak");
      container.setAttribute("data-v-app", "");
    }
    return proxy;
  };
  return app2;
});
const createSSRApp = ((...args) => {
  const app2 = ensureHydrationRenderer().createApp(...args);
  const { mount } = app2;
  app2.mount = (containerOrSelector) => {
    const container = normalizeContainer(containerOrSelector);
    if (container) {
      return mount(container, true, container instanceof SVGElement);
    }
  };
  return app2;
});
function normalizeContainer(container) {
  if (isString$4(container)) {
    const res = document.querySelector(container);
    return res;
  }
  return container;
}
let ssrDirectiveInitialized = false;
const initDirectivesForSSR = () => {
  if (!ssrDirectiveInitialized) {
    ssrDirectiveInitialized = true;
    initVModelForSSR();
    initVShowForSSR();
  }
};
const runtimeDom_esmBundler = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  BaseTransition,
  Comment,
  EffectScope,
  Fragment,
  KeepAlive,
  ReactiveEffect,
  Static,
  Suspense,
  Teleport,
  Text,
  Transition,
  TransitionGroup,
  VueElement,
  assertNumber,
  callWithAsyncErrorHandling,
  callWithErrorHandling,
  camelize: camelize$1,
  capitalize: capitalize$2,
  cloneVNode,
  compatUtils,
  computed,
  createApp,
  createBlock,
  createCommentVNode,
  createElementBlock,
  createElementVNode: createBaseVNode,
  createHydrationRenderer,
  createPropsRestProxy,
  createRenderer,
  createSSRApp,
  createSlots,
  createStaticVNode,
  createTextVNode,
  createVNode,
  customRef,
  defineAsyncComponent,
  defineComponent,
  defineCustomElement,
  defineEmits,
  defineExpose,
  defineProps,
  defineSSRCustomElement,
  get devtools() {
    return devtools$1;
  },
  effect,
  effectScope,
  getCurrentInstance,
  getCurrentScope,
  getTransitionRawChildren,
  guardReactiveProps,
  h: h$2,
  handleError,
  hydrate,
  initCustomFormatter,
  initDirectivesForSSR,
  inject,
  isMemoSame,
  isProxy,
  isReactive,
  isReadonly,
  isRef,
  isRuntimeOnly,
  isShallow,
  isVNode: isVNode$1,
  markRaw,
  mergeDefaults,
  mergeProps,
  nextTick,
  normalizeClass,
  normalizeProps,
  normalizeStyle,
  onActivated,
  onBeforeMount,
  onBeforeUnmount,
  onBeforeUpdate,
  onDeactivated,
  onErrorCaptured,
  onMounted,
  onRenderTracked,
  onRenderTriggered,
  onScopeDispose,
  onServerPrefetch,
  onUnmounted,
  onUpdated,
  openBlock,
  popScopeId,
  provide,
  proxyRefs,
  pushScopeId,
  queuePostFlushCb,
  reactive,
  readonly,
  ref,
  registerRuntimeCompiler,
  render: render$1c,
  renderList,
  renderSlot,
  resolveComponent,
  resolveDirective,
  resolveDynamicComponent,
  resolveFilter,
  resolveTransitionHooks,
  setBlockTracking,
  setDevtoolsHook,
  setTransitionHooks,
  shallowReactive,
  shallowReadonly,
  shallowRef,
  ssrContextKey,
  ssrUtils,
  stop,
  toDisplayString: toDisplayString$1,
  toHandlerKey,
  toHandlers,
  toRaw,
  toRef,
  toRefs,
  transformVNodeArgs,
  triggerRef,
  unref,
  useAttrs,
  useCssModule,
  useCssVars,
  useSSRContext,
  useSlots,
  useTransitionState,
  vModelCheckbox,
  vModelDynamic,
  vModelRadio,
  vModelSelect,
  vModelText,
  vShow,
  version: version$a,
  warn: warn$1,
  watch,
  watchEffect,
  watchPostEffect,
  watchSyncEffect,
  withAsyncContext,
  withCtx,
  withDefaults,
  withDirectives,
  withKeys,
  withMemo,
  withModifiers,
  withScopeId
}, Symbol.toStringTag, { value: "Module" }));
const HeaderBar_HeaderBarQueryDocument$1 = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "query", "name": { "kind": "Name", "value": "HeaderBar_HeaderBarQuery" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "currentProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "title" } }, { "kind": "Field", "name": { "kind": "Name", "value": "config" } }, { "kind": "Field", "name": { "kind": "Name", "value": "savedState" } }, { "kind": "Field", "name": { "kind": "Name", "value": "currentTestingType" } }, { "kind": "Field", "name": { "kind": "Name", "value": "branch" } }, { "kind": "Field", "name": { "kind": "Name", "value": "packageManager" } }, { "kind": "Field", "name": { "kind": "Name", "value": "activeBrowser" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "displayName" } }, { "kind": "Field", "name": { "kind": "Name", "value": "majorVersion" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "browsers" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isSelected" } }, { "kind": "Field", "name": { "kind": "Name", "value": "displayName" } }, { "kind": "Field", "name": { "kind": "Name", "value": "version" } }, { "kind": "Field", "name": { "kind": "Name", "value": "majorVersion" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isVersionSupported" } }, { "kind": "Field", "name": { "kind": "Name", "value": "warning" } }, { "kind": "Field", "name": { "kind": "Name", "value": "disabled" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "projectId" } }, { "kind": "Field", "name": { "kind": "Name", "value": "cloudProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "__typename" } }, { "kind": "InlineFragment", "typeCondition": { "kind": "NamedType", "name": { "kind": "Name", "value": "CloudProject" } }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }] } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "isGlobalMode" } }, { "kind": "Field", "name": { "kind": "Name", "value": "versions" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "current" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "version" } }, { "kind": "Field", "name": { "kind": "Name", "value": "released" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "latest" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "version" } }, { "kind": "Field", "name": { "kind": "Name", "value": "released" } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "cloudViewer" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "cloudOrganizationsUrl" } }, { "kind": "Field", "name": { "kind": "Name", "value": "organizations" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "first" }, "value": { "kind": "IntValue", "value": "100" } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "nodes" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "projects" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "first" }, "value": { "kind": "IntValue", "value": "100" } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "nodes" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "slug" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }] } }] } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "email" } }, { "kind": "Field", "name": { "kind": "Name", "value": "fullName" } }, { "kind": "Field", "alias": { "kind": "Name", "value": "firstOrganization" }, "name": { "kind": "Name", "value": "organizations" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "first" }, "value": { "kind": "IntValue", "value": "1" } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "nodes" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }] } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "authState" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "browserOpened" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "message" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "cachedUser" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "fullName" } }, { "kind": "Field", "name": { "kind": "Name", "value": "email" } }] } }] } }] };
const App_DevRelaunchDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "App_DevRelaunch" }, "variableDefinitions": [{ "kind": "VariableDefinition", "variable": { "kind": "Variable", "name": { "kind": "Name", "value": "action" } }, "type": { "kind": "NonNullType", "type": { "kind": "NamedType", "name": { "kind": "Name", "value": "DevRelaunchAction" } } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "devRelaunch" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "action" }, "value": { "kind": "Variable", "name": { "kind": "Name", "value": "action" } } }] }] } }] };
const MainLaunchpadQueryDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "query", "name": { "kind": "Name", "value": "MainLaunchpadQuery" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "currentProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isCTConfigured" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isE2EConfigured" } }, { "kind": "Field", "name": { "kind": "Name", "value": "currentTestingType" } }, { "kind": "Field", "name": { "kind": "Name", "value": "title" } }, { "kind": "Field", "name": { "kind": "Name", "value": "packageManager" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isLoadingConfigFile" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isLoadingNodeEvents" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isFullConfigReady" } }, { "kind": "Field", "name": { "kind": "Name", "value": "activeBrowser" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "invokedFromCli" } }, { "kind": "Field", "name": { "kind": "Name", "value": "scaffoldedFiles" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "status" } }, { "kind": "Field", "name": { "kind": "Name", "value": "description" } }, { "kind": "Field", "name": { "kind": "Name", "value": "file" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "absolute" } }, { "kind": "Field", "name": { "kind": "Name", "value": "relative" } }, { "kind": "Field", "name": { "kind": "Name", "value": "contents" } }, { "kind": "Field", "name": { "kind": "Name", "value": "fileExtension" } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "warnings" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "title" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errorType" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errorMessage" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "wizard" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "bundler" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "type" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isDetected" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "framework" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "type" } }, { "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isDetected" } }, { "kind": "Field", "name": { "kind": "Name", "value": "supportedBundlers" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "type" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isDetected" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "category" } }, { "kind": "Field", "name": { "kind": "Name", "value": "icon" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "frameworks" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "supportStatus" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isDetected" } }, { "kind": "Field", "name": { "kind": "Name", "value": "type" } }, { "kind": "Field", "name": { "kind": "Name", "value": "category" } }, { "kind": "Field", "name": { "kind": "Name", "value": "icon" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "allBundlers" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "type" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isDetected" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "erroredFrameworks" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "path" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "packagesToInstall" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "description" } }, { "kind": "Field", "name": { "kind": "Name", "value": "package" } }, { "kind": "Field", "name": { "kind": "Name", "value": "minVersion" } }, { "kind": "Field", "name": { "kind": "Name", "value": "detectedVersion" } }, { "kind": "Field", "name": { "kind": "Name", "value": "satisfied" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "installDependenciesCommand" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "baseError" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "title" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errorName" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errorStack" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errorType" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errorMessage" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isUserCodeError" } }, { "kind": "Field", "name": { "kind": "Name", "value": "codeFrame" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "line" } }, { "kind": "Field", "name": { "kind": "Name", "value": "column" } }, { "kind": "Field", "name": { "kind": "Name", "value": "codeBlock" } }, { "kind": "Field", "name": { "kind": "Name", "value": "file" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "absolute" } }, { "kind": "Field", "name": { "kind": "Name", "value": "relative" } }] } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "localSettings" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "preferences" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "majorVersionWelcomeDismissed" } }, { "kind": "Field", "name": { "kind": "Name", "value": "wasBrowserSetInCLI" } }, { "kind": "Field", "name": { "kind": "Name", "value": "shouldLaunchBrowserFromOpenBrowser" } }, { "kind": "Field", "name": { "kind": "Name", "value": "preferredEditorBinary" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "availableEditors" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "binary" } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "isGlobalMode" } }, { "kind": "Field", "name": { "kind": "Name", "value": "projects" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "InlineFragment", "typeCondition": { "kind": "NamedType", "name": { "kind": "Name", "value": "GlobalProject" } }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "title" } }, { "kind": "Field", "name": { "kind": "Name", "value": "projectRoot" } }] } }] } }] } }] };
const Main_ResetErrorsAndLoadConfigDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "Main_ResetErrorsAndLoadConfig" }, "variableDefinitions": [{ "kind": "VariableDefinition", "variable": { "kind": "Variable", "name": { "kind": "Name", "value": "id" } }, "type": { "kind": "NonNullType", "type": { "kind": "NamedType", "name": { "kind": "Name", "value": "ID" } } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "resetErrorAndLoadConfig" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "id" }, "value": { "kind": "Variable", "name": { "kind": "Name", "value": "id" } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "currentProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isCTConfigured" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isE2EConfigured" } }, { "kind": "Field", "name": { "kind": "Name", "value": "currentTestingType" } }, { "kind": "Field", "name": { "kind": "Name", "value": "title" } }, { "kind": "Field", "name": { "kind": "Name", "value": "packageManager" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isLoadingConfigFile" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isLoadingNodeEvents" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isFullConfigReady" } }, { "kind": "Field", "name": { "kind": "Name", "value": "activeBrowser" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "invokedFromCli" } }, { "kind": "Field", "name": { "kind": "Name", "value": "scaffoldedFiles" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "status" } }, { "kind": "Field", "name": { "kind": "Name", "value": "description" } }, { "kind": "Field", "name": { "kind": "Name", "value": "file" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "absolute" } }, { "kind": "Field", "name": { "kind": "Name", "value": "relative" } }, { "kind": "Field", "name": { "kind": "Name", "value": "contents" } }, { "kind": "Field", "name": { "kind": "Name", "value": "fileExtension" } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "warnings" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "title" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errorType" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errorMessage" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "wizard" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "bundler" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "type" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isDetected" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "framework" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "type" } }, { "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isDetected" } }, { "kind": "Field", "name": { "kind": "Name", "value": "supportedBundlers" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "type" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isDetected" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "category" } }, { "kind": "Field", "name": { "kind": "Name", "value": "icon" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "frameworks" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "supportStatus" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isDetected" } }, { "kind": "Field", "name": { "kind": "Name", "value": "type" } }, { "kind": "Field", "name": { "kind": "Name", "value": "category" } }, { "kind": "Field", "name": { "kind": "Name", "value": "icon" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "allBundlers" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "type" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isDetected" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "erroredFrameworks" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "path" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "packagesToInstall" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "description" } }, { "kind": "Field", "name": { "kind": "Name", "value": "package" } }, { "kind": "Field", "name": { "kind": "Name", "value": "minVersion" } }, { "kind": "Field", "name": { "kind": "Name", "value": "detectedVersion" } }, { "kind": "Field", "name": { "kind": "Name", "value": "satisfied" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "installDependenciesCommand" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "baseError" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "title" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errorName" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errorStack" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errorType" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errorMessage" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isUserCodeError" } }, { "kind": "Field", "name": { "kind": "Name", "value": "codeFrame" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "line" } }, { "kind": "Field", "name": { "kind": "Name", "value": "column" } }, { "kind": "Field", "name": { "kind": "Name", "value": "codeBlock" } }, { "kind": "Field", "name": { "kind": "Name", "value": "file" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "absolute" } }, { "kind": "Field", "name": { "kind": "Name", "value": "relative" } }] } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "localSettings" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "preferences" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "majorVersionWelcomeDismissed" } }, { "kind": "Field", "name": { "kind": "Name", "value": "wasBrowserSetInCLI" } }, { "kind": "Field", "name": { "kind": "Name", "value": "shouldLaunchBrowserFromOpenBrowser" } }, { "kind": "Field", "name": { "kind": "Name", "value": "preferredEditorBinary" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "availableEditors" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "binary" } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "isGlobalMode" } }, { "kind": "Field", "name": { "kind": "Name", "value": "projects" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "InlineFragment", "typeCondition": { "kind": "NamedType", "name": { "kind": "Name", "value": "GlobalProject" } }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "title" } }, { "kind": "Field", "name": { "kind": "Name", "value": "projectRoot" } }] } }] } }] } }] } }] };
const Main_LaunchProjectDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "Main_LaunchProject" }, "variableDefinitions": [{ "kind": "VariableDefinition", "variable": { "kind": "Variable", "name": { "kind": "Name", "value": "testingType" } }, "type": { "kind": "NonNullType", "type": { "kind": "NamedType", "name": { "kind": "Name", "value": "TestingTypeEnum" } } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "launchOpenProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "setProjectPreferencesInGlobalCache" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "testingType" }, "value": { "kind": "Variable", "name": { "kind": "Name", "value": "testingType" } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "currentProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "title" } }] } }] } }] } }] };
const GlobalPage_AddProjectDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "GlobalPage_addProject" }, "variableDefinitions": [{ "kind": "VariableDefinition", "variable": { "kind": "Variable", "name": { "kind": "Name", "value": "path" } }, "type": { "kind": "NamedType", "name": { "kind": "Name", "value": "String" } } }, { "kind": "VariableDefinition", "variable": { "kind": "Variable", "name": { "kind": "Name", "value": "open" } }, "type": { "kind": "NamedType", "name": { "kind": "Name", "value": "Boolean" } }, "defaultValue": { "kind": "BooleanValue", "value": true } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "addProject" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "path" }, "value": { "kind": "Variable", "name": { "kind": "Name", "value": "path" } } }, { "kind": "Argument", "name": { "kind": "Name", "value": "open" }, "value": { "kind": "Variable", "name": { "kind": "Name", "value": "open" } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "projects" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "InlineFragment", "typeCondition": { "kind": "NamedType", "name": { "kind": "Name", "value": "GlobalProject" } }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "title" } }, { "kind": "Field", "name": { "kind": "Name", "value": "projectRoot" } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "localSettings" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "availableEditors" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "binary" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "preferences" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "preferredEditorBinary" } }] } }] } }] } }] } }] };
const GlobalPage_RemoveProjectDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "GlobalPage_RemoveProject" }, "variableDefinitions": [{ "kind": "VariableDefinition", "variable": { "kind": "Variable", "name": { "kind": "Name", "value": "path" } }, "type": { "kind": "NonNullType", "type": { "kind": "NamedType", "name": { "kind": "Name", "value": "String" } } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "removeProject" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "path" }, "value": { "kind": "Variable", "name": { "kind": "Name", "value": "path" } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "projects" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "InlineFragment", "typeCondition": { "kind": "NamedType", "name": { "kind": "Name", "value": "GlobalProject" } }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "title" } }, { "kind": "Field", "name": { "kind": "Name", "value": "projectRoot" } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "localSettings" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "availableEditors" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "binary" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "preferences" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "preferredEditorBinary" } }] } }] } }] } }] } }] };
const GlobalPage_OpenDirectoryInIdeDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "GlobalPage_OpenDirectoryInIDE" }, "variableDefinitions": [{ "kind": "VariableDefinition", "variable": { "kind": "Variable", "name": { "kind": "Name", "value": "path" } }, "type": { "kind": "NonNullType", "type": { "kind": "NamedType", "name": { "kind": "Name", "value": "String" } } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "openDirectoryInIDE" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "path" }, "value": { "kind": "Variable", "name": { "kind": "Name", "value": "path" } } }] }] } }] };
const GlobalPage_OpenInFinderDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "GlobalPage_OpenInFinder" }, "variableDefinitions": [{ "kind": "VariableDefinition", "variable": { "kind": "Variable", "name": { "kind": "Name", "value": "path" } }, "type": { "kind": "NonNullType", "type": { "kind": "NamedType", "name": { "kind": "Name", "value": "String" } } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "openInFinder" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "path" }, "value": { "kind": "Variable", "name": { "kind": "Name", "value": "path" } } }] }] } }] };
const GlobalProjectCard_SetCurrentProjectDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "GlobalProjectCard_setCurrentProject" }, "variableDefinitions": [{ "kind": "VariableDefinition", "variable": { "kind": "Variable", "name": { "kind": "Name", "value": "path" } }, "type": { "kind": "NonNullType", "type": { "kind": "NamedType", "name": { "kind": "Name", "value": "String" } } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "setCurrentProject" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "path" }, "value": { "kind": "Variable", "name": { "kind": "Name", "value": "path" } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "currentProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isCTConfigured" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isE2EConfigured" } }, { "kind": "Field", "name": { "kind": "Name", "value": "currentTestingType" } }, { "kind": "Field", "name": { "kind": "Name", "value": "title" } }, { "kind": "Field", "name": { "kind": "Name", "value": "packageManager" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isLoadingConfigFile" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isLoadingNodeEvents" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isFullConfigReady" } }, { "kind": "Field", "name": { "kind": "Name", "value": "activeBrowser" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "invokedFromCli" } }, { "kind": "Field", "name": { "kind": "Name", "value": "scaffoldedFiles" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "status" } }, { "kind": "Field", "name": { "kind": "Name", "value": "description" } }, { "kind": "Field", "name": { "kind": "Name", "value": "file" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "absolute" } }, { "kind": "Field", "name": { "kind": "Name", "value": "relative" } }, { "kind": "Field", "name": { "kind": "Name", "value": "contents" } }, { "kind": "Field", "name": { "kind": "Name", "value": "fileExtension" } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "warnings" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "title" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errorType" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errorMessage" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "wizard" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "bundler" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "type" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isDetected" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "framework" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "type" } }, { "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isDetected" } }, { "kind": "Field", "name": { "kind": "Name", "value": "supportedBundlers" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "type" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isDetected" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "category" } }, { "kind": "Field", "name": { "kind": "Name", "value": "icon" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "frameworks" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "supportStatus" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isDetected" } }, { "kind": "Field", "name": { "kind": "Name", "value": "type" } }, { "kind": "Field", "name": { "kind": "Name", "value": "category" } }, { "kind": "Field", "name": { "kind": "Name", "value": "icon" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "allBundlers" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "type" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isDetected" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "erroredFrameworks" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "path" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "packagesToInstall" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "description" } }, { "kind": "Field", "name": { "kind": "Name", "value": "package" } }, { "kind": "Field", "name": { "kind": "Name", "value": "minVersion" } }, { "kind": "Field", "name": { "kind": "Name", "value": "detectedVersion" } }, { "kind": "Field", "name": { "kind": "Name", "value": "satisfied" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "installDependenciesCommand" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "baseError" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "title" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errorName" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errorStack" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errorType" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errorMessage" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isUserCodeError" } }, { "kind": "Field", "name": { "kind": "Name", "value": "codeFrame" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "line" } }, { "kind": "Field", "name": { "kind": "Name", "value": "column" } }, { "kind": "Field", "name": { "kind": "Name", "value": "codeBlock" } }, { "kind": "Field", "name": { "kind": "Name", "value": "file" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "absolute" } }, { "kind": "Field", "name": { "kind": "Name", "value": "relative" } }] } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "localSettings" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "preferences" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "majorVersionWelcomeDismissed" } }, { "kind": "Field", "name": { "kind": "Name", "value": "wasBrowserSetInCLI" } }, { "kind": "Field", "name": { "kind": "Name", "value": "shouldLaunchBrowserFromOpenBrowser" } }, { "kind": "Field", "name": { "kind": "Name", "value": "preferredEditorBinary" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "availableEditors" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "binary" } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "isGlobalMode" } }, { "kind": "Field", "name": { "kind": "Name", "value": "projects" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "InlineFragment", "typeCondition": { "kind": "NamedType", "name": { "kind": "Name", "value": "GlobalProject" } }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "title" } }, { "kind": "Field", "name": { "kind": "Name", "value": "projectRoot" } }] } }] } }] } }] } }] };
const EnvironmentSetup_DetectionChangeDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "subscription", "name": { "kind": "Name", "value": "EnvironmentSetup_DetectionChange" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "frameworkDetectionChange" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "bundler" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "type" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isDetected" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "framework" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "type" } }, { "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isDetected" } }, { "kind": "Field", "name": { "kind": "Name", "value": "supportedBundlers" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "type" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isDetected" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "category" } }, { "kind": "Field", "name": { "kind": "Name", "value": "icon" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "frameworks" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "supportStatus" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isDetected" } }, { "kind": "Field", "name": { "kind": "Name", "value": "type" } }, { "kind": "Field", "name": { "kind": "Name", "value": "category" } }, { "kind": "Field", "name": { "kind": "Name", "value": "icon" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "allBundlers" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "type" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isDetected" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "erroredFrameworks" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "path" } }] } }] } }] } }] };
const EnvironmentSetup_WizardUpdateDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "EnvironmentSetup_wizardUpdate" }, "variableDefinitions": [{ "kind": "VariableDefinition", "variable": { "kind": "Variable", "name": { "kind": "Name", "value": "input" } }, "type": { "kind": "NonNullType", "type": { "kind": "NamedType", "name": { "kind": "Name", "value": "WizardUpdateInput" } } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "wizardUpdate" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "input" }, "value": { "kind": "Variable", "name": { "kind": "Name", "value": "input" } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "bundler" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "type" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isDetected" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "framework" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "type" } }, { "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isDetected" } }, { "kind": "Field", "name": { "kind": "Name", "value": "supportedBundlers" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "type" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isDetected" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "category" } }, { "kind": "Field", "name": { "kind": "Name", "value": "icon" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "frameworks" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "supportStatus" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isDetected" } }, { "kind": "Field", "name": { "kind": "Name", "value": "type" } }, { "kind": "Field", "name": { "kind": "Name", "value": "category" } }, { "kind": "Field", "name": { "kind": "Name", "value": "icon" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "allBundlers" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "type" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isDetected" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "erroredFrameworks" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "path" } }] } }] } }] } }] };
const EnvironmentSetup_ClearTestingTypeDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "EnvironmentSetup_ClearTestingType" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "clearCurrentTestingType" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "currentProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "currentTestingType" } }] } }] } }] } }] };
const InstallDependencies_ScaffoldFilesDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "InstallDependencies_scaffoldFiles" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "scaffoldTestingType" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "scaffoldedFiles" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "status" } }, { "kind": "Field", "name": { "kind": "Name", "value": "description" } }, { "kind": "Field", "name": { "kind": "Name", "value": "file" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "absolute" } }, { "kind": "Field", "name": { "kind": "Name", "value": "relative" } }, { "kind": "Field", "name": { "kind": "Name", "value": "contents" } }, { "kind": "Field", "name": { "kind": "Name", "value": "fileExtension" } }] } }] } }] } }] } }] };
const Wizard_InstalledPackagesDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "query", "name": { "kind": "Name", "value": "Wizard_InstalledPackages" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "wizard" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "packagesToInstall" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "description" } }, { "kind": "Field", "name": { "kind": "Name", "value": "package" } }, { "kind": "Field", "name": { "kind": "Name", "value": "minVersion" } }, { "kind": "Field", "name": { "kind": "Name", "value": "detectedVersion" } }, { "kind": "Field", "name": { "kind": "Name", "value": "satisfied" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "installDependenciesCommand" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "currentProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "title" } }, { "kind": "Field", "name": { "kind": "Name", "value": "packageManager" } }] } }] } }] };
const OpenBrowserDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "query", "name": { "kind": "Name", "value": "OpenBrowser" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "currentProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "currentTestingType" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isLoadingConfigFile" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isLoadingNodeEvents" } }, { "kind": "Field", "name": { "kind": "Name", "value": "activeBrowser" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isFocusSupported" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "browsers" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "disabled" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "displayName" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isVersionSupported" } }, { "kind": "Field", "name": { "kind": "Name", "value": "warning" } }, { "kind": "Field", "name": { "kind": "Name", "value": "majorVersion" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "browserStatus" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "warnings" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "title" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errorType" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errorMessage" } }] } }] } }] };
const OpenBrowser_LocalSettingsDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "query", "name": { "kind": "Name", "value": "OpenBrowser_LocalSettings" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "localSettings" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "preferences" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "shouldLaunchBrowserFromOpenBrowser" } }] } }] } }] } }] };
const OpenBrowser_ClearTestingTypeDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "OpenBrowser_ClearTestingType" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "clearCurrentTestingType" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "currentProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "currentTestingType" } }] } }] } }] } }] };
const OpenBrowser_LaunchProjectDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "OpenBrowser_LaunchProject" }, "variableDefinitions": [{ "kind": "VariableDefinition", "variable": { "kind": "Variable", "name": { "kind": "Name", "value": "testingType" } }, "type": { "kind": "NonNullType", "type": { "kind": "NamedType", "name": { "kind": "Name", "value": "TestingTypeEnum" } } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "launchOpenProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "setProjectPreferencesInGlobalCache" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "testingType" }, "value": { "kind": "Variable", "name": { "kind": "Name", "value": "testingType" } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "currentProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "title" } }] } }] } }] } }] };
const OpenBrowser_CloseBrowserDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "OpenBrowser_CloseBrowser" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "closeBrowser" } }] } }] };
const OpenBrowser_FocusActiveBrowserWindowDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "OpenBrowser_FocusActiveBrowserWindow" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "focusActiveBrowserWindow" } }] } }] };
const OpenBrowser_ResetLatestVersionTelemetryDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "OpenBrowser_ResetLatestVersionTelemetry" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "resetLatestVersionTelemetry" } }] } }] };
const OpenBrowserList_SetBrowserDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "OpenBrowserList_SetBrowser" }, "variableDefinitions": [{ "kind": "VariableDefinition", "variable": { "kind": "Variable", "name": { "kind": "Name", "value": "id" } }, "type": { "kind": "NonNullType", "type": { "kind": "NamedType", "name": { "kind": "Name", "value": "ID" } } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "launchpadSetBrowser" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "id" }, "value": { "kind": "Variable", "name": { "kind": "Name", "value": "id" } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "activeBrowser" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isFocusSupported" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "browsers" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "disabled" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "displayName" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isVersionSupported" } }, { "kind": "Field", "name": { "kind": "Name", "value": "warning" } }, { "kind": "Field", "name": { "kind": "Name", "value": "majorVersion" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "currentTestingType" } }, { "kind": "Field", "name": { "kind": "Name", "value": "browserStatus" } }] } }] } }] };
const OpenBrowserList_BrowserStatusChangeDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "subscription", "name": { "kind": "Name", "value": "OpenBrowserList_browserStatusChange" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "browserStatusChange" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "browserStatus" } }, { "kind": "Field", "name": { "kind": "Name", "value": "activeBrowser" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isFocusSupported" } }] } }] } }] } }] };
const ScaffoldedFiles_CompleteSetupDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "ScaffoldedFiles_completeSetup" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "completeSetup" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "currentProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isLoadingConfigFile" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isLoadingNodeEvents" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isCTConfigured" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "scaffoldedFiles" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "status" } }] } }] } }] } }] };
const TestingTypeSelectionDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "TestingTypeSelection" }, "variableDefinitions": [{ "kind": "VariableDefinition", "variable": { "kind": "Variable", "name": { "kind": "Name", "value": "testingType" } }, "type": { "kind": "NonNullType", "type": { "kind": "NamedType", "name": { "kind": "Name", "value": "TestingTypeEnum" } } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "setAndLoadCurrentTestingType" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "testingType" }, "value": { "kind": "Variable", "name": { "kind": "Name", "value": "testingType" } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "currentProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "currentTestingType" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isCTConfigured" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isE2EConfigured" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isLoadingConfigFile" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isLoadingNodeEvents" } }, { "kind": "Field", "name": { "kind": "Name", "value": "title" } }, { "kind": "Field", "name": { "kind": "Name", "value": "packageManager" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "scaffoldedFiles" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "status" } }, { "kind": "Field", "name": { "kind": "Name", "value": "description" } }, { "kind": "Field", "name": { "kind": "Name", "value": "file" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "absolute" } }, { "kind": "Field", "name": { "kind": "Name", "value": "relative" } }, { "kind": "Field", "name": { "kind": "Name", "value": "contents" } }, { "kind": "Field", "name": { "kind": "Name", "value": "fileExtension" } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "warnings" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "title" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errorType" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errorMessage" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "wizard" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "bundler" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "type" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isDetected" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "framework" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "type" } }, { "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isDetected" } }, { "kind": "Field", "name": { "kind": "Name", "value": "supportedBundlers" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "type" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isDetected" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "category" } }, { "kind": "Field", "name": { "kind": "Name", "value": "icon" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "frameworks" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "supportStatus" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isDetected" } }, { "kind": "Field", "name": { "kind": "Name", "value": "type" } }, { "kind": "Field", "name": { "kind": "Name", "value": "category" } }, { "kind": "Field", "name": { "kind": "Name", "value": "icon" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "allBundlers" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "type" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isDetected" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "erroredFrameworks" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "path" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "packagesToInstall" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "description" } }, { "kind": "Field", "name": { "kind": "Name", "value": "package" } }, { "kind": "Field", "name": { "kind": "Name", "value": "minVersion" } }, { "kind": "Field", "name": { "kind": "Name", "value": "detectedVersion" } }, { "kind": "Field", "name": { "kind": "Name", "value": "satisfied" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "installDependenciesCommand" } }] } }] } }] } }] };
const WarningList_RemoveWarningDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "WarningList_removeWarning" }, "variableDefinitions": [{ "kind": "VariableDefinition", "variable": { "kind": "Variable", "name": { "kind": "Name", "value": "id" } }, "type": { "kind": "NonNullType", "type": { "kind": "NamedType", "name": { "kind": "Name", "value": "ID" } } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "dismissWarning" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "id" }, "value": { "kind": "Variable", "name": { "kind": "Name", "value": "id" } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "warnings" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "title" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errorType" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errorMessage" } }] } }] } }] } }] };
const WarningList_PingBaseUrlDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "WarningList_pingBaseUrl" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "pingBaseUrl" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "warnings" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "title" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errorType" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errorMessage" } }] } }] } }] } }] };
var nodejsCustomInspectSymbol = typeof Symbol === "function" && typeof Symbol.for === "function" ? Symbol.for("nodejs.util.inspect.custom") : void 0;
function _typeof$2(obj) {
  "@babel/helpers - typeof";
  if (typeof Symbol === "function" && typeof Symbol.iterator === "symbol") {
    _typeof$2 = function _typeof2(obj2) {
      return typeof obj2;
    };
  } else {
    _typeof$2 = function _typeof2(obj2) {
      return obj2 && typeof Symbol === "function" && obj2.constructor === Symbol && obj2 !== Symbol.prototype ? "symbol" : typeof obj2;
    };
  }
  return _typeof$2(obj);
}
var MAX_ARRAY_LENGTH = 10;
var MAX_RECURSIVE_DEPTH = 2;
function inspect(value) {
  return formatValue(value, []);
}
function formatValue(value, seenValues) {
  switch (_typeof$2(value)) {
    case "string":
      return JSON.stringify(value);
    case "function":
      return value.name ? "[function ".concat(value.name, "]") : "[function]";
    case "object":
      if (value === null) {
        return "null";
      }
      return formatObjectValue(value, seenValues);
    default:
      return String(value);
  }
}
function formatObjectValue(value, previouslySeenValues) {
  if (previouslySeenValues.indexOf(value) !== -1) {
    return "[Circular]";
  }
  var seenValues = [].concat(previouslySeenValues, [value]);
  var customInspectFn = getCustomFn(value);
  if (customInspectFn !== void 0) {
    var customValue = customInspectFn.call(value);
    if (customValue !== value) {
      return typeof customValue === "string" ? customValue : formatValue(customValue, seenValues);
    }
  } else if (Array.isArray(value)) {
    return formatArray(value, seenValues);
  }
  return formatObject(value, seenValues);
}
function formatObject(object, seenValues) {
  var keys = Object.keys(object);
  if (keys.length === 0) {
    return "{}";
  }
  if (seenValues.length > MAX_RECURSIVE_DEPTH) {
    return "[" + getObjectTag(object) + "]";
  }
  var properties = keys.map(function(key) {
    var value = formatValue(object[key], seenValues);
    return key + ": " + value;
  });
  return "{ " + properties.join(", ") + " }";
}
function formatArray(array, seenValues) {
  if (array.length === 0) {
    return "[]";
  }
  if (seenValues.length > MAX_RECURSIVE_DEPTH) {
    return "[Array]";
  }
  var len = Math.min(MAX_ARRAY_LENGTH, array.length);
  var remaining = array.length - len;
  var items = [];
  for (var i2 = 0; i2 < len; ++i2) {
    items.push(formatValue(array[i2], seenValues));
  }
  if (remaining === 1) {
    items.push("... 1 more item");
  } else if (remaining > 1) {
    items.push("... ".concat(remaining, " more items"));
  }
  return "[" + items.join(", ") + "]";
}
function getCustomFn(object) {
  var customInspectFn = object[String(nodejsCustomInspectSymbol)];
  if (typeof customInspectFn === "function") {
    return customInspectFn;
  }
  if (typeof object.inspect === "function") {
    return object.inspect;
  }
}
function getObjectTag(object) {
  var tag = Object.prototype.toString.call(object).replace(/^\[object /, "").replace(/]$/, "");
  if (tag === "Object" && typeof object.constructor === "function") {
    var name2 = object.constructor.name;
    if (typeof name2 === "string" && name2 !== "") {
      return name2;
    }
  }
  return tag;
}
function invariant$1(condition, message) {
  var booleanCondition = Boolean(condition);
  if (!booleanCondition) {
    throw new Error(message != null ? message : "Unexpected invariant triggered.");
  }
}
function defineInspect(classObject) {
  var fn = classObject.prototype.toJSON;
  typeof fn === "function" || invariant$1(0);
  classObject.prototype.inspect = fn;
  if (nodejsCustomInspectSymbol) {
    classObject.prototype[nodejsCustomInspectSymbol] = fn;
  }
}
var Location = /* @__PURE__ */ (function() {
  function Location2(startToken, endToken, source) {
    this.start = startToken.start;
    this.end = endToken.end;
    this.startToken = startToken;
    this.endToken = endToken;
    this.source = source;
  }
  var _proto = Location2.prototype;
  _proto.toJSON = function toJSON() {
    return {
      start: this.start,
      end: this.end
    };
  };
  return Location2;
})();
defineInspect(Location);
var Token = /* @__PURE__ */ (function() {
  function Token2(kind, start, end, line, column, prev, value) {
    this.kind = kind;
    this.start = start;
    this.end = end;
    this.line = line;
    this.column = column;
    this.value = value;
    this.prev = prev;
    this.next = null;
  }
  var _proto2 = Token2.prototype;
  _proto2.toJSON = function toJSON() {
    return {
      kind: this.kind,
      value: this.value,
      line: this.line,
      column: this.column
    };
  };
  return Token2;
})();
defineInspect(Token);
function isNode$1(maybeNode) {
  return maybeNode != null && typeof maybeNode.kind === "string";
}
var QueryDocumentKeys = {
  Name: [],
  Document: ["definitions"],
  OperationDefinition: ["name", "variableDefinitions", "directives", "selectionSet"],
  VariableDefinition: ["variable", "type", "defaultValue", "directives"],
  Variable: ["name"],
  SelectionSet: ["selections"],
  Field: ["alias", "name", "arguments", "directives", "selectionSet"],
  Argument: ["name", "value"],
  FragmentSpread: ["name", "directives"],
  InlineFragment: ["typeCondition", "directives", "selectionSet"],
  FragmentDefinition: [
    "name",
    // Note: fragment variable definitions are experimental and may be changed
    // or removed in the future.
    "variableDefinitions",
    "typeCondition",
    "directives",
    "selectionSet"
  ],
  IntValue: [],
  FloatValue: [],
  StringValue: [],
  BooleanValue: [],
  NullValue: [],
  EnumValue: [],
  ListValue: ["values"],
  ObjectValue: ["fields"],
  ObjectField: ["name", "value"],
  Directive: ["name", "arguments"],
  NamedType: ["name"],
  ListType: ["type"],
  NonNullType: ["type"],
  SchemaDefinition: ["description", "directives", "operationTypes"],
  OperationTypeDefinition: ["type"],
  ScalarTypeDefinition: ["description", "name", "directives"],
  ObjectTypeDefinition: ["description", "name", "interfaces", "directives", "fields"],
  FieldDefinition: ["description", "name", "arguments", "type", "directives"],
  InputValueDefinition: ["description", "name", "type", "defaultValue", "directives"],
  InterfaceTypeDefinition: ["description", "name", "interfaces", "directives", "fields"],
  UnionTypeDefinition: ["description", "name", "directives", "types"],
  EnumTypeDefinition: ["description", "name", "directives", "values"],
  EnumValueDefinition: ["description", "name", "directives"],
  InputObjectTypeDefinition: ["description", "name", "directives", "fields"],
  DirectiveDefinition: ["description", "name", "arguments", "locations"],
  SchemaExtension: ["directives", "operationTypes"],
  ScalarTypeExtension: ["name", "directives"],
  ObjectTypeExtension: ["name", "interfaces", "directives", "fields"],
  InterfaceTypeExtension: ["name", "interfaces", "directives", "fields"],
  UnionTypeExtension: ["name", "directives", "types"],
  EnumTypeExtension: ["name", "directives", "values"],
  InputObjectTypeExtension: ["name", "directives", "fields"]
};
var BREAK = Object.freeze({});
function visit$1(root2, visitor) {
  var visitorKeys = arguments.length > 2 && arguments[2] !== void 0 ? arguments[2] : QueryDocumentKeys;
  var stack = void 0;
  var inArray = Array.isArray(root2);
  var keys = [root2];
  var index = -1;
  var edits = [];
  var node = void 0;
  var key = void 0;
  var parent = void 0;
  var path = [];
  var ancestors = [];
  var newRoot = root2;
  do {
    index++;
    var isLeaving = index === keys.length;
    var isEdited = isLeaving && edits.length !== 0;
    if (isLeaving) {
      key = ancestors.length === 0 ? void 0 : path[path.length - 1];
      node = parent;
      parent = ancestors.pop();
      if (isEdited) {
        if (inArray) {
          node = node.slice();
        } else {
          var clone = {};
          for (var _i2 = 0, _Object$keys2 = Object.keys(node); _i2 < _Object$keys2.length; _i2++) {
            var k2 = _Object$keys2[_i2];
            clone[k2] = node[k2];
          }
          node = clone;
        }
        var editOffset = 0;
        for (var ii2 = 0; ii2 < edits.length; ii2++) {
          var editKey = edits[ii2][0];
          var editValue = edits[ii2][1];
          if (inArray) {
            editKey -= editOffset;
          }
          if (inArray && editValue === null) {
            node.splice(editKey, 1);
            editOffset++;
          } else {
            node[editKey] = editValue;
          }
        }
      }
      index = stack.index;
      keys = stack.keys;
      edits = stack.edits;
      inArray = stack.inArray;
      stack = stack.prev;
    } else {
      key = parent ? inArray ? index : keys[index] : void 0;
      node = parent ? parent[key] : newRoot;
      if (node === null || node === void 0) {
        continue;
      }
      if (parent) {
        path.push(key);
      }
    }
    var result = void 0;
    if (!Array.isArray(node)) {
      if (!isNode$1(node)) {
        throw new Error("Invalid AST Node: ".concat(inspect(node), "."));
      }
      var visitFn = getVisitFn(visitor, node.kind, isLeaving);
      if (visitFn) {
        result = visitFn.call(visitor, node, key, parent, path, ancestors);
        if (result === BREAK) {
          break;
        }
        if (result === false) {
          if (!isLeaving) {
            path.pop();
            continue;
          }
        } else if (result !== void 0) {
          edits.push([key, result]);
          if (!isLeaving) {
            if (isNode$1(result)) {
              node = result;
            } else {
              path.pop();
              continue;
            }
          }
        }
      }
    }
    if (result === void 0 && isEdited) {
      edits.push([key, node]);
    }
    if (isLeaving) {
      path.pop();
    } else {
      var _visitorKeys$node$kin;
      stack = {
        inArray,
        index,
        keys,
        edits,
        prev: stack
      };
      inArray = Array.isArray(node);
      keys = inArray ? node : (_visitorKeys$node$kin = visitorKeys[node.kind]) !== null && _visitorKeys$node$kin !== void 0 ? _visitorKeys$node$kin : [];
      index = -1;
      edits = [];
      if (parent) {
        ancestors.push(parent);
      }
      parent = node;
    }
  } while (stack !== void 0);
  if (edits.length !== 0) {
    newRoot = edits[edits.length - 1][1];
  }
  return newRoot;
}
function getVisitFn(visitor, kind, isLeaving) {
  var kindVisitor = visitor[kind];
  if (kindVisitor) {
    if (!isLeaving && typeof kindVisitor === "function") {
      return kindVisitor;
    }
    var kindSpecificVisitor = isLeaving ? kindVisitor.leave : kindVisitor.enter;
    if (typeof kindSpecificVisitor === "function") {
      return kindSpecificVisitor;
    }
  } else {
    var specificVisitor = isLeaving ? visitor.leave : visitor.enter;
    if (specificVisitor) {
      if (typeof specificVisitor === "function") {
        return specificVisitor;
      }
      var specificKindVisitor = specificVisitor[kind];
      if (typeof specificKindVisitor === "function") {
        return specificKindVisitor;
      }
    }
  }
}
var Kind = Object.freeze({
  // Name
  NAME: "Name",
  // Document
  DOCUMENT: "Document",
  OPERATION_DEFINITION: "OperationDefinition",
  VARIABLE_DEFINITION: "VariableDefinition",
  SELECTION_SET: "SelectionSet",
  FIELD: "Field",
  ARGUMENT: "Argument",
  // Fragments
  FRAGMENT_SPREAD: "FragmentSpread",
  INLINE_FRAGMENT: "InlineFragment",
  FRAGMENT_DEFINITION: "FragmentDefinition",
  // Values
  VARIABLE: "Variable",
  INT: "IntValue",
  FLOAT: "FloatValue",
  STRING: "StringValue",
  BOOLEAN: "BooleanValue",
  NULL: "NullValue",
  ENUM: "EnumValue",
  LIST: "ListValue",
  OBJECT: "ObjectValue",
  OBJECT_FIELD: "ObjectField",
  // Directives
  DIRECTIVE: "Directive",
  // Types
  NAMED_TYPE: "NamedType",
  LIST_TYPE: "ListType",
  NON_NULL_TYPE: "NonNullType",
  // Type System Definitions
  SCHEMA_DEFINITION: "SchemaDefinition",
  OPERATION_TYPE_DEFINITION: "OperationTypeDefinition",
  // Type Definitions
  SCALAR_TYPE_DEFINITION: "ScalarTypeDefinition",
  OBJECT_TYPE_DEFINITION: "ObjectTypeDefinition",
  FIELD_DEFINITION: "FieldDefinition",
  INPUT_VALUE_DEFINITION: "InputValueDefinition",
  INTERFACE_TYPE_DEFINITION: "InterfaceTypeDefinition",
  UNION_TYPE_DEFINITION: "UnionTypeDefinition",
  ENUM_TYPE_DEFINITION: "EnumTypeDefinition",
  ENUM_VALUE_DEFINITION: "EnumValueDefinition",
  INPUT_OBJECT_TYPE_DEFINITION: "InputObjectTypeDefinition",
  // Directive Definitions
  DIRECTIVE_DEFINITION: "DirectiveDefinition",
  // Type System Extensions
  SCHEMA_EXTENSION: "SchemaExtension",
  // Type Extensions
  SCALAR_TYPE_EXTENSION: "ScalarTypeExtension",
  OBJECT_TYPE_EXTENSION: "ObjectTypeExtension",
  INTERFACE_TYPE_EXTENSION: "InterfaceTypeExtension",
  UNION_TYPE_EXTENSION: "UnionTypeExtension",
  ENUM_TYPE_EXTENSION: "EnumTypeExtension",
  INPUT_OBJECT_TYPE_EXTENSION: "InputObjectTypeExtension"
});
function dedentBlockStringValue(rawString) {
  var lines = rawString.split(/\r\n|[\n\r]/g);
  var commonIndent = getBlockStringIndentation(rawString);
  if (commonIndent !== 0) {
    for (var i2 = 1; i2 < lines.length; i2++) {
      lines[i2] = lines[i2].slice(commonIndent);
    }
  }
  var startLine = 0;
  while (startLine < lines.length && isBlank(lines[startLine])) {
    ++startLine;
  }
  var endLine = lines.length;
  while (endLine > startLine && isBlank(lines[endLine - 1])) {
    --endLine;
  }
  return lines.slice(startLine, endLine).join("\n");
}
function isBlank(str) {
  for (var i2 = 0; i2 < str.length; ++i2) {
    if (str[i2] !== " " && str[i2] !== "	") {
      return false;
    }
  }
  return true;
}
function getBlockStringIndentation(value) {
  var _commonIndent;
  var isFirstLine = true;
  var isEmptyLine = true;
  var indent2 = 0;
  var commonIndent = null;
  for (var i2 = 0; i2 < value.length; ++i2) {
    switch (value.charCodeAt(i2)) {
      case 13:
        if (value.charCodeAt(i2 + 1) === 10) {
          ++i2;
        }
      // falls through
      case 10:
        isFirstLine = false;
        isEmptyLine = true;
        indent2 = 0;
        break;
      case 9:
      //   \t
      case 32:
        ++indent2;
        break;
      default:
        if (isEmptyLine && !isFirstLine && (commonIndent === null || indent2 < commonIndent)) {
          commonIndent = indent2;
        }
        isEmptyLine = false;
    }
  }
  return (_commonIndent = commonIndent) !== null && _commonIndent !== void 0 ? _commonIndent : 0;
}
function printBlockString(value) {
  var indentation = arguments.length > 1 && arguments[1] !== void 0 ? arguments[1] : "";
  var preferMultipleLines = arguments.length > 2 && arguments[2] !== void 0 ? arguments[2] : false;
  var isSingleLine = value.indexOf("\n") === -1;
  var hasLeadingSpace = value[0] === " " || value[0] === "	";
  var hasTrailingQuote = value[value.length - 1] === '"';
  var hasTrailingSlash = value[value.length - 1] === "\\";
  var printAsMultipleLines = !isSingleLine || hasTrailingQuote || hasTrailingSlash || preferMultipleLines;
  var result = "";
  if (printAsMultipleLines && !(isSingleLine && hasLeadingSpace)) {
    result += "\n" + indentation;
  }
  result += indentation ? value.replace(/\n/g, "\n" + indentation) : value;
  if (printAsMultipleLines) {
    result += "\n";
  }
  return '"""' + result.replace(/"""/g, '\\"""') + '"""';
}
function print$1(ast2) {
  return visit$1(ast2, {
    leave: printDocASTReducer
  });
}
var MAX_LINE_LENGTH = 80;
var printDocASTReducer = {
  Name: function Name(node) {
    return node.value;
  },
  Variable: function Variable(node) {
    return "$" + node.name;
  },
  // Document
  Document: function Document(node) {
    return join$3(node.definitions, "\n\n") + "\n";
  },
  OperationDefinition: function OperationDefinition(node) {
    var op = node.operation;
    var name2 = node.name;
    var varDefs = wrap("(", join$3(node.variableDefinitions, ", "), ")");
    var directives = join$3(node.directives, " ");
    var selectionSet = node.selectionSet;
    return !name2 && !directives && !varDefs && op === "query" ? selectionSet : join$3([op, join$3([name2, varDefs]), directives, selectionSet], " ");
  },
  VariableDefinition: function VariableDefinition(_ref) {
    var variable = _ref.variable, type2 = _ref.type, defaultValue = _ref.defaultValue, directives = _ref.directives;
    return variable + ": " + type2 + wrap(" = ", defaultValue) + wrap(" ", join$3(directives, " "));
  },
  SelectionSet: function SelectionSet(_ref2) {
    var selections = _ref2.selections;
    return block$2(selections);
  },
  Field: function Field(_ref3) {
    var alias = _ref3.alias, name2 = _ref3.name, args = _ref3.arguments, directives = _ref3.directives, selectionSet = _ref3.selectionSet;
    var prefix = wrap("", alias, ": ") + name2;
    var argsLine = prefix + wrap("(", join$3(args, ", "), ")");
    if (argsLine.length > MAX_LINE_LENGTH) {
      argsLine = prefix + wrap("(\n", indent(join$3(args, "\n")), "\n)");
    }
    return join$3([argsLine, join$3(directives, " "), selectionSet], " ");
  },
  Argument: function Argument(_ref4) {
    var name2 = _ref4.name, value = _ref4.value;
    return name2 + ": " + value;
  },
  // Fragments
  FragmentSpread: function FragmentSpread(_ref5) {
    var name2 = _ref5.name, directives = _ref5.directives;
    return "..." + name2 + wrap(" ", join$3(directives, " "));
  },
  InlineFragment: function InlineFragment(_ref6) {
    var typeCondition = _ref6.typeCondition, directives = _ref6.directives, selectionSet = _ref6.selectionSet;
    return join$3(["...", wrap("on ", typeCondition), join$3(directives, " "), selectionSet], " ");
  },
  FragmentDefinition: function FragmentDefinition(_ref7) {
    var name2 = _ref7.name, typeCondition = _ref7.typeCondition, variableDefinitions = _ref7.variableDefinitions, directives = _ref7.directives, selectionSet = _ref7.selectionSet;
    return (
      // Note: fragment variable definitions are experimental and may be changed
      // or removed in the future.
      "fragment ".concat(name2).concat(wrap("(", join$3(variableDefinitions, ", "), ")"), " ") + "on ".concat(typeCondition, " ").concat(wrap("", join$3(directives, " "), " ")) + selectionSet
    );
  },
  // Value
  IntValue: function IntValue(_ref8) {
    var value = _ref8.value;
    return value;
  },
  FloatValue: function FloatValue(_ref9) {
    var value = _ref9.value;
    return value;
  },
  StringValue: function StringValue(_ref10, key) {
    var value = _ref10.value, isBlockString = _ref10.block;
    return isBlockString ? printBlockString(value, key === "description" ? "" : "  ") : JSON.stringify(value);
  },
  BooleanValue: function BooleanValue(_ref11) {
    var value = _ref11.value;
    return value ? "true" : "false";
  },
  NullValue: function NullValue() {
    return "null";
  },
  EnumValue: function EnumValue(_ref12) {
    var value = _ref12.value;
    return value;
  },
  ListValue: function ListValue(_ref13) {
    var values = _ref13.values;
    return "[" + join$3(values, ", ") + "]";
  },
  ObjectValue: function ObjectValue(_ref14) {
    var fields = _ref14.fields;
    return "{" + join$3(fields, ", ") + "}";
  },
  ObjectField: function ObjectField(_ref15) {
    var name2 = _ref15.name, value = _ref15.value;
    return name2 + ": " + value;
  },
  // Directive
  Directive: function Directive(_ref16) {
    var name2 = _ref16.name, args = _ref16.arguments;
    return "@" + name2 + wrap("(", join$3(args, ", "), ")");
  },
  // Type
  NamedType: function NamedType(_ref17) {
    var name2 = _ref17.name;
    return name2;
  },
  ListType: function ListType(_ref18) {
    var type2 = _ref18.type;
    return "[" + type2 + "]";
  },
  NonNullType: function NonNullType(_ref19) {
    var type2 = _ref19.type;
    return type2 + "!";
  },
  // Type System Definitions
  SchemaDefinition: addDescription(function(_ref20) {
    var directives = _ref20.directives, operationTypes = _ref20.operationTypes;
    return join$3(["schema", join$3(directives, " "), block$2(operationTypes)], " ");
  }),
  OperationTypeDefinition: function OperationTypeDefinition(_ref21) {
    var operation = _ref21.operation, type2 = _ref21.type;
    return operation + ": " + type2;
  },
  ScalarTypeDefinition: addDescription(function(_ref22) {
    var name2 = _ref22.name, directives = _ref22.directives;
    return join$3(["scalar", name2, join$3(directives, " ")], " ");
  }),
  ObjectTypeDefinition: addDescription(function(_ref23) {
    var name2 = _ref23.name, interfaces = _ref23.interfaces, directives = _ref23.directives, fields = _ref23.fields;
    return join$3(["type", name2, wrap("implements ", join$3(interfaces, " & ")), join$3(directives, " "), block$2(fields)], " ");
  }),
  FieldDefinition: addDescription(function(_ref24) {
    var name2 = _ref24.name, args = _ref24.arguments, type2 = _ref24.type, directives = _ref24.directives;
    return name2 + (hasMultilineItems(args) ? wrap("(\n", indent(join$3(args, "\n")), "\n)") : wrap("(", join$3(args, ", "), ")")) + ": " + type2 + wrap(" ", join$3(directives, " "));
  }),
  InputValueDefinition: addDescription(function(_ref25) {
    var name2 = _ref25.name, type2 = _ref25.type, defaultValue = _ref25.defaultValue, directives = _ref25.directives;
    return join$3([name2 + ": " + type2, wrap("= ", defaultValue), join$3(directives, " ")], " ");
  }),
  InterfaceTypeDefinition: addDescription(function(_ref26) {
    var name2 = _ref26.name, interfaces = _ref26.interfaces, directives = _ref26.directives, fields = _ref26.fields;
    return join$3(["interface", name2, wrap("implements ", join$3(interfaces, " & ")), join$3(directives, " "), block$2(fields)], " ");
  }),
  UnionTypeDefinition: addDescription(function(_ref27) {
    var name2 = _ref27.name, directives = _ref27.directives, types = _ref27.types;
    return join$3(["union", name2, join$3(directives, " "), types && types.length !== 0 ? "= " + join$3(types, " | ") : ""], " ");
  }),
  EnumTypeDefinition: addDescription(function(_ref28) {
    var name2 = _ref28.name, directives = _ref28.directives, values = _ref28.values;
    return join$3(["enum", name2, join$3(directives, " "), block$2(values)], " ");
  }),
  EnumValueDefinition: addDescription(function(_ref29) {
    var name2 = _ref29.name, directives = _ref29.directives;
    return join$3([name2, join$3(directives, " ")], " ");
  }),
  InputObjectTypeDefinition: addDescription(function(_ref30) {
    var name2 = _ref30.name, directives = _ref30.directives, fields = _ref30.fields;
    return join$3(["input", name2, join$3(directives, " "), block$2(fields)], " ");
  }),
  DirectiveDefinition: addDescription(function(_ref31) {
    var name2 = _ref31.name, args = _ref31.arguments, repeatable = _ref31.repeatable, locations = _ref31.locations;
    return "directive @" + name2 + (hasMultilineItems(args) ? wrap("(\n", indent(join$3(args, "\n")), "\n)") : wrap("(", join$3(args, ", "), ")")) + (repeatable ? " repeatable" : "") + " on " + join$3(locations, " | ");
  }),
  SchemaExtension: function SchemaExtension(_ref32) {
    var directives = _ref32.directives, operationTypes = _ref32.operationTypes;
    return join$3(["extend schema", join$3(directives, " "), block$2(operationTypes)], " ");
  },
  ScalarTypeExtension: function ScalarTypeExtension(_ref33) {
    var name2 = _ref33.name, directives = _ref33.directives;
    return join$3(["extend scalar", name2, join$3(directives, " ")], " ");
  },
  ObjectTypeExtension: function ObjectTypeExtension(_ref34) {
    var name2 = _ref34.name, interfaces = _ref34.interfaces, directives = _ref34.directives, fields = _ref34.fields;
    return join$3(["extend type", name2, wrap("implements ", join$3(interfaces, " & ")), join$3(directives, " "), block$2(fields)], " ");
  },
  InterfaceTypeExtension: function InterfaceTypeExtension(_ref35) {
    var name2 = _ref35.name, interfaces = _ref35.interfaces, directives = _ref35.directives, fields = _ref35.fields;
    return join$3(["extend interface", name2, wrap("implements ", join$3(interfaces, " & ")), join$3(directives, " "), block$2(fields)], " ");
  },
  UnionTypeExtension: function UnionTypeExtension(_ref36) {
    var name2 = _ref36.name, directives = _ref36.directives, types = _ref36.types;
    return join$3(["extend union", name2, join$3(directives, " "), types && types.length !== 0 ? "= " + join$3(types, " | ") : ""], " ");
  },
  EnumTypeExtension: function EnumTypeExtension(_ref37) {
    var name2 = _ref37.name, directives = _ref37.directives, values = _ref37.values;
    return join$3(["extend enum", name2, join$3(directives, " "), block$2(values)], " ");
  },
  InputObjectTypeExtension: function InputObjectTypeExtension(_ref38) {
    var name2 = _ref38.name, directives = _ref38.directives, fields = _ref38.fields;
    return join$3(["extend input", name2, join$3(directives, " "), block$2(fields)], " ");
  }
};
function addDescription(cb) {
  return function(node) {
    return join$3([node.description, cb(node)], "\n");
  };
}
function join$3(maybeArray) {
  var _maybeArray$filter$jo;
  var separator = arguments.length > 1 && arguments[1] !== void 0 ? arguments[1] : "";
  return (_maybeArray$filter$jo = maybeArray === null || maybeArray === void 0 ? void 0 : maybeArray.filter(function(x2) {
    return x2;
  }).join(separator)) !== null && _maybeArray$filter$jo !== void 0 ? _maybeArray$filter$jo : "";
}
function block$2(array) {
  return wrap("{\n", indent(join$3(array, "\n")), "\n}");
}
function wrap(start, maybeString) {
  var end = arguments.length > 2 && arguments[2] !== void 0 ? arguments[2] : "";
  return maybeString != null && maybeString !== "" ? start + maybeString + end : "";
}
function indent(str) {
  return wrap("  ", str.replace(/\n/g, "\n  "));
}
function isMultiline(str) {
  return str.indexOf("\n") !== -1;
}
function hasMultilineItems(maybeArray) {
  return maybeArray != null && maybeArray.some(isMultiline);
}
function _typeof$1(obj) {
  "@babel/helpers - typeof";
  if (typeof Symbol === "function" && typeof Symbol.iterator === "symbol") {
    _typeof$1 = function _typeof2(obj2) {
      return typeof obj2;
    };
  } else {
    _typeof$1 = function _typeof2(obj2) {
      return obj2 && typeof Symbol === "function" && obj2.constructor === Symbol && obj2 !== Symbol.prototype ? "symbol" : typeof obj2;
    };
  }
  return _typeof$1(obj);
}
function isObjectLike$1(value) {
  return _typeof$1(value) == "object" && value !== null;
}
var SYMBOL_TO_STRING_TAG = typeof Symbol === "function" && Symbol.toStringTag != null ? Symbol.toStringTag : "@@toStringTag";
function getLocation(source, position) {
  var lineRegexp = /\r\n|[\n\r]/g;
  var line = 1;
  var column = position + 1;
  var match2;
  while ((match2 = lineRegexp.exec(source.body)) && match2.index < position) {
    line += 1;
    column = position + 1 - (match2.index + match2[0].length);
  }
  return {
    line,
    column
  };
}
function printLocation(location2) {
  return printSourceLocation(location2.source, getLocation(location2.source, location2.start));
}
function printSourceLocation(source, sourceLocation) {
  var firstLineColumnOffset = source.locationOffset.column - 1;
  var body = whitespace(firstLineColumnOffset) + source.body;
  var lineIndex = sourceLocation.line - 1;
  var lineOffset = source.locationOffset.line - 1;
  var lineNum = sourceLocation.line + lineOffset;
  var columnOffset = sourceLocation.line === 1 ? firstLineColumnOffset : 0;
  var columnNum = sourceLocation.column + columnOffset;
  var locationStr = "".concat(source.name, ":").concat(lineNum, ":").concat(columnNum, "\n");
  var lines = body.split(/\r\n|[\n\r]/g);
  var locationLine = lines[lineIndex];
  if (locationLine.length > 120) {
    var subLineIndex = Math.floor(columnNum / 80);
    var subLineColumnNum = columnNum % 80;
    var subLines = [];
    for (var i2 = 0; i2 < locationLine.length; i2 += 80) {
      subLines.push(locationLine.slice(i2, i2 + 80));
    }
    return locationStr + printPrefixedLines([["".concat(lineNum), subLines[0]]].concat(subLines.slice(1, subLineIndex + 1).map(function(subLine) {
      return ["", subLine];
    }), [[" ", whitespace(subLineColumnNum - 1) + "^"], ["", subLines[subLineIndex + 1]]]));
  }
  return locationStr + printPrefixedLines([
    // Lines specified like this: ["prefix", "string"],
    ["".concat(lineNum - 1), lines[lineIndex - 1]],
    ["".concat(lineNum), locationLine],
    ["", whitespace(columnNum - 1) + "^"],
    ["".concat(lineNum + 1), lines[lineIndex + 1]]
  ]);
}
function printPrefixedLines(lines) {
  var existingLines = lines.filter(function(_ref) {
    _ref[0];
    var line = _ref[1];
    return line !== void 0;
  });
  var padLen = Math.max.apply(Math, existingLines.map(function(_ref2) {
    var prefix = _ref2[0];
    return prefix.length;
  }));
  return existingLines.map(function(_ref3) {
    var prefix = _ref3[0], line = _ref3[1];
    return leftPad(padLen, prefix) + (line ? " | " + line : " |");
  }).join("\n");
}
function whitespace(len) {
  return Array(len + 1).join(" ");
}
function leftPad(len, str) {
  return whitespace(len - str.length) + str;
}
function _typeof(obj) {
  "@babel/helpers - typeof";
  if (typeof Symbol === "function" && typeof Symbol.iterator === "symbol") {
    _typeof = function _typeof2(obj2) {
      return typeof obj2;
    };
  } else {
    _typeof = function _typeof2(obj2) {
      return obj2 && typeof Symbol === "function" && obj2.constructor === Symbol && obj2 !== Symbol.prototype ? "symbol" : typeof obj2;
    };
  }
  return _typeof(obj);
}
function _classCallCheck(instance, Constructor) {
  if (!(instance instanceof Constructor)) {
    throw new TypeError("Cannot call a class as a function");
  }
}
function _defineProperties$1(target2, props) {
  for (var i2 = 0; i2 < props.length; i2++) {
    var descriptor = props[i2];
    descriptor.enumerable = descriptor.enumerable || false;
    descriptor.configurable = true;
    if ("value" in descriptor) descriptor.writable = true;
    Object.defineProperty(target2, descriptor.key, descriptor);
  }
}
function _createClass$1(Constructor, protoProps, staticProps) {
  if (protoProps) _defineProperties$1(Constructor.prototype, protoProps);
  return Constructor;
}
function _inherits(subClass, superClass) {
  if (typeof superClass !== "function" && superClass !== null) {
    throw new TypeError("Super expression must either be null or a function");
  }
  subClass.prototype = Object.create(superClass && superClass.prototype, { constructor: { value: subClass, writable: true, configurable: true } });
  if (superClass) _setPrototypeOf(subClass, superClass);
}
function _createSuper(Derived) {
  var hasNativeReflectConstruct = _isNativeReflectConstruct();
  return function _createSuperInternal() {
    var Super = _getPrototypeOf(Derived), result;
    if (hasNativeReflectConstruct) {
      var NewTarget = _getPrototypeOf(this).constructor;
      result = Reflect.construct(Super, arguments, NewTarget);
    } else {
      result = Super.apply(this, arguments);
    }
    return _possibleConstructorReturn(this, result);
  };
}
function _possibleConstructorReturn(self2, call) {
  if (call && (_typeof(call) === "object" || typeof call === "function")) {
    return call;
  }
  return _assertThisInitialized(self2);
}
function _assertThisInitialized(self2) {
  if (self2 === void 0) {
    throw new ReferenceError("this hasn't been initialised - super() hasn't been called");
  }
  return self2;
}
function _wrapNativeSuper(Class) {
  var _cache = typeof Map === "function" ? /* @__PURE__ */ new Map() : void 0;
  _wrapNativeSuper = function _wrapNativeSuper2(Class2) {
    if (Class2 === null || !_isNativeFunction(Class2)) return Class2;
    if (typeof Class2 !== "function") {
      throw new TypeError("Super expression must either be null or a function");
    }
    if (typeof _cache !== "undefined") {
      if (_cache.has(Class2)) return _cache.get(Class2);
      _cache.set(Class2, Wrapper);
    }
    function Wrapper() {
      return _construct(Class2, arguments, _getPrototypeOf(this).constructor);
    }
    Wrapper.prototype = Object.create(Class2.prototype, { constructor: { value: Wrapper, enumerable: false, writable: true, configurable: true } });
    return _setPrototypeOf(Wrapper, Class2);
  };
  return _wrapNativeSuper(Class);
}
function _construct(Parent, args, Class) {
  if (_isNativeReflectConstruct()) {
    _construct = Reflect.construct;
  } else {
    _construct = function _construct2(Parent2, args2, Class2) {
      var a2 = [null];
      a2.push.apply(a2, args2);
      var Constructor = Function.bind.apply(Parent2, a2);
      var instance = new Constructor();
      if (Class2) _setPrototypeOf(instance, Class2.prototype);
      return instance;
    };
  }
  return _construct.apply(null, arguments);
}
function _isNativeReflectConstruct() {
  if (typeof Reflect === "undefined" || !Reflect.construct) return false;
  if (Reflect.construct.sham) return false;
  if (typeof Proxy === "function") return true;
  try {
    Date.prototype.toString.call(Reflect.construct(Date, [], function() {
    }));
    return true;
  } catch (e) {
    return false;
  }
}
function _isNativeFunction(fn) {
  return Function.toString.call(fn).indexOf("[native code]") !== -1;
}
function _setPrototypeOf(o2, p2) {
  _setPrototypeOf = Object.setPrototypeOf || function _setPrototypeOf2(o3, p3) {
    o3.__proto__ = p3;
    return o3;
  };
  return _setPrototypeOf(o2, p2);
}
function _getPrototypeOf(o2) {
  _getPrototypeOf = Object.setPrototypeOf ? Object.getPrototypeOf : function _getPrototypeOf2(o3) {
    return o3.__proto__ || Object.getPrototypeOf(o3);
  };
  return _getPrototypeOf(o2);
}
var GraphQLError = /* @__PURE__ */ (function(_Error) {
  _inherits(GraphQLError2, _Error);
  var _super = _createSuper(GraphQLError2);
  function GraphQLError2(message, nodes, source, positions, path, originalError, extensions) {
    var _locations2, _source2, _positions2, _extensions2;
    var _this;
    _classCallCheck(this, GraphQLError2);
    _this = _super.call(this, message);
    var _nodes = Array.isArray(nodes) ? nodes.length !== 0 ? nodes : void 0 : nodes ? [nodes] : void 0;
    var _source = source;
    if (!_source && _nodes) {
      var _nodes$0$loc;
      _source = (_nodes$0$loc = _nodes[0].loc) === null || _nodes$0$loc === void 0 ? void 0 : _nodes$0$loc.source;
    }
    var _positions = positions;
    if (!_positions && _nodes) {
      _positions = _nodes.reduce(function(list2, node) {
        if (node.loc) {
          list2.push(node.loc.start);
        }
        return list2;
      }, []);
    }
    if (_positions && _positions.length === 0) {
      _positions = void 0;
    }
    var _locations;
    if (positions && source) {
      _locations = positions.map(function(pos) {
        return getLocation(source, pos);
      });
    } else if (_nodes) {
      _locations = _nodes.reduce(function(list2, node) {
        if (node.loc) {
          list2.push(getLocation(node.loc.source, node.loc.start));
        }
        return list2;
      }, []);
    }
    var _extensions = extensions;
    if (_extensions == null && originalError != null) {
      var originalExtensions = originalError.extensions;
      if (isObjectLike$1(originalExtensions)) {
        _extensions = originalExtensions;
      }
    }
    Object.defineProperties(_assertThisInitialized(_this), {
      name: {
        value: "GraphQLError"
      },
      message: {
        value: message,
        // By being enumerable, JSON.stringify will include `message` in the
        // resulting output. This ensures that the simplest possible GraphQL
        // service adheres to the spec.
        enumerable: true,
        writable: true
      },
      locations: {
        // Coercing falsy values to undefined ensures they will not be included
        // in JSON.stringify() when not provided.
        value: (_locations2 = _locations) !== null && _locations2 !== void 0 ? _locations2 : void 0,
        // By being enumerable, JSON.stringify will include `locations` in the
        // resulting output. This ensures that the simplest possible GraphQL
        // service adheres to the spec.
        enumerable: _locations != null
      },
      path: {
        // Coercing falsy values to undefined ensures they will not be included
        // in JSON.stringify() when not provided.
        value: path !== null && path !== void 0 ? path : void 0,
        // By being enumerable, JSON.stringify will include `path` in the
        // resulting output. This ensures that the simplest possible GraphQL
        // service adheres to the spec.
        enumerable: path != null
      },
      nodes: {
        value: _nodes !== null && _nodes !== void 0 ? _nodes : void 0
      },
      source: {
        value: (_source2 = _source) !== null && _source2 !== void 0 ? _source2 : void 0
      },
      positions: {
        value: (_positions2 = _positions) !== null && _positions2 !== void 0 ? _positions2 : void 0
      },
      originalError: {
        value: originalError
      },
      extensions: {
        // Coercing falsy values to undefined ensures they will not be included
        // in JSON.stringify() when not provided.
        value: (_extensions2 = _extensions) !== null && _extensions2 !== void 0 ? _extensions2 : void 0,
        // By being enumerable, JSON.stringify will include `path` in the
        // resulting output. This ensures that the simplest possible GraphQL
        // service adheres to the spec.
        enumerable: _extensions != null
      }
    });
    if (originalError !== null && originalError !== void 0 && originalError.stack) {
      Object.defineProperty(_assertThisInitialized(_this), "stack", {
        value: originalError.stack,
        writable: true,
        configurable: true
      });
      return _possibleConstructorReturn(_this);
    }
    if (Error.captureStackTrace) {
      Error.captureStackTrace(_assertThisInitialized(_this), GraphQLError2);
    } else {
      Object.defineProperty(_assertThisInitialized(_this), "stack", {
        value: Error().stack,
        writable: true,
        configurable: true
      });
    }
    return _this;
  }
  _createClass$1(GraphQLError2, [{
    key: "toString",
    value: function toString2() {
      return printError(this);
    }
    // FIXME: workaround to not break chai comparisons, should be remove in v16
    // $FlowFixMe[unsupported-syntax] Flow doesn't support computed properties yet
  }, {
    key: SYMBOL_TO_STRING_TAG,
    get: function get2() {
      return "Object";
    }
  }]);
  return GraphQLError2;
})(/* @__PURE__ */ _wrapNativeSuper(Error));
function printError(error2) {
  var output = error2.message;
  if (error2.nodes) {
    for (var _i2 = 0, _error$nodes2 = error2.nodes; _i2 < _error$nodes2.length; _i2++) {
      var node = _error$nodes2[_i2];
      if (node.loc) {
        output += "\n\n" + printLocation(node.loc);
      }
    }
  } else if (error2.source && error2.locations) {
    for (var _i4 = 0, _error$locations2 = error2.locations; _i4 < _error$locations2.length; _i4++) {
      var location2 = _error$locations2[_i4];
      output += "\n\n" + printSourceLocation(error2.source, location2);
    }
  }
  return output;
}
function syntaxError(source, position, description) {
  return new GraphQLError("Syntax Error: ".concat(description), void 0, source, [position]);
}
var TokenKind = Object.freeze({
  SOF: "<SOF>",
  EOF: "<EOF>",
  BANG: "!",
  DOLLAR: "$",
  AMP: "&",
  PAREN_L: "(",
  PAREN_R: ")",
  SPREAD: "...",
  COLON: ":",
  EQUALS: "=",
  AT: "@",
  BRACKET_L: "[",
  BRACKET_R: "]",
  BRACE_L: "{",
  PIPE: "|",
  BRACE_R: "}",
  NAME: "Name",
  INT: "Int",
  FLOAT: "Float",
  STRING: "String",
  BLOCK_STRING: "BlockString",
  COMMENT: "Comment"
});
function devAssert(condition, message) {
  var booleanCondition = Boolean(condition);
  if (!booleanCondition) {
    throw new Error(message);
  }
}
const instanceOf = (
  // istanbul ignore next (See: 'https://github.com/graphql/graphql-js/issues/2317')
  // eslint-disable-next-line no-shadow
  (function instanceOf2(value, constructor) {
    return value instanceof constructor;
  })
);
function _defineProperties(target2, props) {
  for (var i2 = 0; i2 < props.length; i2++) {
    var descriptor = props[i2];
    descriptor.enumerable = descriptor.enumerable || false;
    descriptor.configurable = true;
    if ("value" in descriptor) descriptor.writable = true;
    Object.defineProperty(target2, descriptor.key, descriptor);
  }
}
function _createClass(Constructor, protoProps, staticProps) {
  if (protoProps) _defineProperties(Constructor.prototype, protoProps);
  return Constructor;
}
var Source = /* @__PURE__ */ (function() {
  function Source2(body) {
    var name2 = arguments.length > 1 && arguments[1] !== void 0 ? arguments[1] : "GraphQL request";
    var locationOffset = arguments.length > 2 && arguments[2] !== void 0 ? arguments[2] : {
      line: 1,
      column: 1
    };
    typeof body === "string" || devAssert(0, "Body must be a string. Received: ".concat(inspect(body), "."));
    this.body = body;
    this.name = name2;
    this.locationOffset = locationOffset;
    this.locationOffset.line > 0 || devAssert(0, "line in locationOffset is 1-indexed and must be positive.");
    this.locationOffset.column > 0 || devAssert(0, "column in locationOffset is 1-indexed and must be positive.");
  }
  _createClass(Source2, [{
    key: SYMBOL_TO_STRING_TAG,
    get: function get2() {
      return "Source";
    }
  }]);
  return Source2;
})();
function isSource(source) {
  return instanceOf(source, Source);
}
var DirectiveLocation = Object.freeze({
  // Request Definitions
  QUERY: "QUERY",
  MUTATION: "MUTATION",
  SUBSCRIPTION: "SUBSCRIPTION",
  FIELD: "FIELD",
  FRAGMENT_DEFINITION: "FRAGMENT_DEFINITION",
  FRAGMENT_SPREAD: "FRAGMENT_SPREAD",
  INLINE_FRAGMENT: "INLINE_FRAGMENT",
  VARIABLE_DEFINITION: "VARIABLE_DEFINITION",
  // Type System Definitions
  SCHEMA: "SCHEMA",
  SCALAR: "SCALAR",
  OBJECT: "OBJECT",
  FIELD_DEFINITION: "FIELD_DEFINITION",
  ARGUMENT_DEFINITION: "ARGUMENT_DEFINITION",
  INTERFACE: "INTERFACE",
  UNION: "UNION",
  ENUM: "ENUM",
  ENUM_VALUE: "ENUM_VALUE",
  INPUT_OBJECT: "INPUT_OBJECT",
  INPUT_FIELD_DEFINITION: "INPUT_FIELD_DEFINITION"
});
var Lexer = /* @__PURE__ */ (function() {
  function Lexer2(source) {
    var startOfFileToken = new Token(TokenKind.SOF, 0, 0, 0, 0, null);
    this.source = source;
    this.lastToken = startOfFileToken;
    this.token = startOfFileToken;
    this.line = 1;
    this.lineStart = 0;
  }
  var _proto = Lexer2.prototype;
  _proto.advance = function advance() {
    this.lastToken = this.token;
    var token2 = this.token = this.lookahead();
    return token2;
  };
  _proto.lookahead = function lookahead() {
    var token2 = this.token;
    if (token2.kind !== TokenKind.EOF) {
      do {
        var _token$next;
        token2 = (_token$next = token2.next) !== null && _token$next !== void 0 ? _token$next : token2.next = readToken(this, token2);
      } while (token2.kind === TokenKind.COMMENT);
    }
    return token2;
  };
  return Lexer2;
})();
function isPunctuatorTokenKind(kind) {
  return kind === TokenKind.BANG || kind === TokenKind.DOLLAR || kind === TokenKind.AMP || kind === TokenKind.PAREN_L || kind === TokenKind.PAREN_R || kind === TokenKind.SPREAD || kind === TokenKind.COLON || kind === TokenKind.EQUALS || kind === TokenKind.AT || kind === TokenKind.BRACKET_L || kind === TokenKind.BRACKET_R || kind === TokenKind.BRACE_L || kind === TokenKind.PIPE || kind === TokenKind.BRACE_R;
}
function printCharCode(code2) {
  return (
    // NaN/undefined represents access beyond the end of the file.
    isNaN(code2) ? TokenKind.EOF : (
      // Trust JSON for ASCII.
      code2 < 127 ? JSON.stringify(String.fromCharCode(code2)) : (
        // Otherwise print the escaped form.
        '"\\u'.concat(("00" + code2.toString(16).toUpperCase()).slice(-4), '"')
      )
    )
  );
}
function readToken(lexer, prev) {
  var source = lexer.source;
  var body = source.body;
  var bodyLength = body.length;
  var pos = prev.end;
  while (pos < bodyLength) {
    var code2 = body.charCodeAt(pos);
    var _line = lexer.line;
    var _col = 1 + pos - lexer.lineStart;
    switch (code2) {
      case 65279:
      // <BOM>
      case 9:
      //   \t
      case 32:
      //  <space>
      case 44:
        ++pos;
        continue;
      case 10:
        ++pos;
        ++lexer.line;
        lexer.lineStart = pos;
        continue;
      case 13:
        if (body.charCodeAt(pos + 1) === 10) {
          pos += 2;
        } else {
          ++pos;
        }
        ++lexer.line;
        lexer.lineStart = pos;
        continue;
      case 33:
        return new Token(TokenKind.BANG, pos, pos + 1, _line, _col, prev);
      case 35:
        return readComment(source, pos, _line, _col, prev);
      case 36:
        return new Token(TokenKind.DOLLAR, pos, pos + 1, _line, _col, prev);
      case 38:
        return new Token(TokenKind.AMP, pos, pos + 1, _line, _col, prev);
      case 40:
        return new Token(TokenKind.PAREN_L, pos, pos + 1, _line, _col, prev);
      case 41:
        return new Token(TokenKind.PAREN_R, pos, pos + 1, _line, _col, prev);
      case 46:
        if (body.charCodeAt(pos + 1) === 46 && body.charCodeAt(pos + 2) === 46) {
          return new Token(TokenKind.SPREAD, pos, pos + 3, _line, _col, prev);
        }
        break;
      case 58:
        return new Token(TokenKind.COLON, pos, pos + 1, _line, _col, prev);
      case 61:
        return new Token(TokenKind.EQUALS, pos, pos + 1, _line, _col, prev);
      case 64:
        return new Token(TokenKind.AT, pos, pos + 1, _line, _col, prev);
      case 91:
        return new Token(TokenKind.BRACKET_L, pos, pos + 1, _line, _col, prev);
      case 93:
        return new Token(TokenKind.BRACKET_R, pos, pos + 1, _line, _col, prev);
      case 123:
        return new Token(TokenKind.BRACE_L, pos, pos + 1, _line, _col, prev);
      case 124:
        return new Token(TokenKind.PIPE, pos, pos + 1, _line, _col, prev);
      case 125:
        return new Token(TokenKind.BRACE_R, pos, pos + 1, _line, _col, prev);
      case 34:
        if (body.charCodeAt(pos + 1) === 34 && body.charCodeAt(pos + 2) === 34) {
          return readBlockString(source, pos, _line, _col, prev, lexer);
        }
        return readString(source, pos, _line, _col, prev);
      case 45:
      //  -
      case 48:
      //  0
      case 49:
      //  1
      case 50:
      //  2
      case 51:
      //  3
      case 52:
      //  4
      case 53:
      //  5
      case 54:
      //  6
      case 55:
      //  7
      case 56:
      //  8
      case 57:
        return readNumber(source, pos, code2, _line, _col, prev);
      case 65:
      //  A
      case 66:
      //  B
      case 67:
      //  C
      case 68:
      //  D
      case 69:
      //  E
      case 70:
      //  F
      case 71:
      //  G
      case 72:
      //  H
      case 73:
      //  I
      case 74:
      //  J
      case 75:
      //  K
      case 76:
      //  L
      case 77:
      //  M
      case 78:
      //  N
      case 79:
      //  O
      case 80:
      //  P
      case 81:
      //  Q
      case 82:
      //  R
      case 83:
      //  S
      case 84:
      //  T
      case 85:
      //  U
      case 86:
      //  V
      case 87:
      //  W
      case 88:
      //  X
      case 89:
      //  Y
      case 90:
      //  Z
      case 95:
      //  _
      case 97:
      //  a
      case 98:
      //  b
      case 99:
      //  c
      case 100:
      // d
      case 101:
      // e
      case 102:
      // f
      case 103:
      // g
      case 104:
      // h
      case 105:
      // i
      case 106:
      // j
      case 107:
      // k
      case 108:
      // l
      case 109:
      // m
      case 110:
      // n
      case 111:
      // o
      case 112:
      // p
      case 113:
      // q
      case 114:
      // r
      case 115:
      // s
      case 116:
      // t
      case 117:
      // u
      case 118:
      // v
      case 119:
      // w
      case 120:
      // x
      case 121:
      // y
      case 122:
        return readName(source, pos, _line, _col, prev);
    }
    throw syntaxError(source, pos, unexpectedCharacterMessage(code2));
  }
  var line = lexer.line;
  var col = 1 + pos - lexer.lineStart;
  return new Token(TokenKind.EOF, bodyLength, bodyLength, line, col, prev);
}
function unexpectedCharacterMessage(code2) {
  if (code2 < 32 && code2 !== 9 && code2 !== 10 && code2 !== 13) {
    return "Cannot contain the invalid character ".concat(printCharCode(code2), ".");
  }
  if (code2 === 39) {
    return `Unexpected single quote character ('), did you mean to use a double quote (")?`;
  }
  return "Cannot parse the unexpected character ".concat(printCharCode(code2), ".");
}
function readComment(source, start, line, col, prev) {
  var body = source.body;
  var code2;
  var position = start;
  do {
    code2 = body.charCodeAt(++position);
  } while (!isNaN(code2) && // SourceCharacter but not LineTerminator
  (code2 > 31 || code2 === 9));
  return new Token(TokenKind.COMMENT, start, position, line, col, prev, body.slice(start + 1, position));
}
function readNumber(source, start, firstCode, line, col, prev) {
  var body = source.body;
  var code2 = firstCode;
  var position = start;
  var isFloat = false;
  if (code2 === 45) {
    code2 = body.charCodeAt(++position);
  }
  if (code2 === 48) {
    code2 = body.charCodeAt(++position);
    if (code2 >= 48 && code2 <= 57) {
      throw syntaxError(source, position, "Invalid number, unexpected digit after 0: ".concat(printCharCode(code2), "."));
    }
  } else {
    position = readDigits(source, position, code2);
    code2 = body.charCodeAt(position);
  }
  if (code2 === 46) {
    isFloat = true;
    code2 = body.charCodeAt(++position);
    position = readDigits(source, position, code2);
    code2 = body.charCodeAt(position);
  }
  if (code2 === 69 || code2 === 101) {
    isFloat = true;
    code2 = body.charCodeAt(++position);
    if (code2 === 43 || code2 === 45) {
      code2 = body.charCodeAt(++position);
    }
    position = readDigits(source, position, code2);
    code2 = body.charCodeAt(position);
  }
  if (code2 === 46 || isNameStart(code2)) {
    throw syntaxError(source, position, "Invalid number, expected digit but got: ".concat(printCharCode(code2), "."));
  }
  return new Token(isFloat ? TokenKind.FLOAT : TokenKind.INT, start, position, line, col, prev, body.slice(start, position));
}
function readDigits(source, start, firstCode) {
  var body = source.body;
  var position = start;
  var code2 = firstCode;
  if (code2 >= 48 && code2 <= 57) {
    do {
      code2 = body.charCodeAt(++position);
    } while (code2 >= 48 && code2 <= 57);
    return position;
  }
  throw syntaxError(source, position, "Invalid number, expected digit but got: ".concat(printCharCode(code2), "."));
}
function readString(source, start, line, col, prev) {
  var body = source.body;
  var position = start + 1;
  var chunkStart = position;
  var code2 = 0;
  var value = "";
  while (position < body.length && !isNaN(code2 = body.charCodeAt(position)) && // not LineTerminator
  code2 !== 10 && code2 !== 13) {
    if (code2 === 34) {
      value += body.slice(chunkStart, position);
      return new Token(TokenKind.STRING, start, position + 1, line, col, prev, value);
    }
    if (code2 < 32 && code2 !== 9) {
      throw syntaxError(source, position, "Invalid character within String: ".concat(printCharCode(code2), "."));
    }
    ++position;
    if (code2 === 92) {
      value += body.slice(chunkStart, position - 1);
      code2 = body.charCodeAt(position);
      switch (code2) {
        case 34:
          value += '"';
          break;
        case 47:
          value += "/";
          break;
        case 92:
          value += "\\";
          break;
        case 98:
          value += "\b";
          break;
        case 102:
          value += "\f";
          break;
        case 110:
          value += "\n";
          break;
        case 114:
          value += "\r";
          break;
        case 116:
          value += "	";
          break;
        case 117: {
          var charCode = uniCharCode(body.charCodeAt(position + 1), body.charCodeAt(position + 2), body.charCodeAt(position + 3), body.charCodeAt(position + 4));
          if (charCode < 0) {
            var invalidSequence = body.slice(position + 1, position + 5);
            throw syntaxError(source, position, "Invalid character escape sequence: \\u".concat(invalidSequence, "."));
          }
          value += String.fromCharCode(charCode);
          position += 4;
          break;
        }
        default:
          throw syntaxError(source, position, "Invalid character escape sequence: \\".concat(String.fromCharCode(code2), "."));
      }
      ++position;
      chunkStart = position;
    }
  }
  throw syntaxError(source, position, "Unterminated string.");
}
function readBlockString(source, start, line, col, prev, lexer) {
  var body = source.body;
  var position = start + 3;
  var chunkStart = position;
  var code2 = 0;
  var rawValue = "";
  while (position < body.length && !isNaN(code2 = body.charCodeAt(position))) {
    if (code2 === 34 && body.charCodeAt(position + 1) === 34 && body.charCodeAt(position + 2) === 34) {
      rawValue += body.slice(chunkStart, position);
      return new Token(TokenKind.BLOCK_STRING, start, position + 3, line, col, prev, dedentBlockStringValue(rawValue));
    }
    if (code2 < 32 && code2 !== 9 && code2 !== 10 && code2 !== 13) {
      throw syntaxError(source, position, "Invalid character within String: ".concat(printCharCode(code2), "."));
    }
    if (code2 === 10) {
      ++position;
      ++lexer.line;
      lexer.lineStart = position;
    } else if (code2 === 13) {
      if (body.charCodeAt(position + 1) === 10) {
        position += 2;
      } else {
        ++position;
      }
      ++lexer.line;
      lexer.lineStart = position;
    } else if (
      // Escape Triple-Quote (\""")
      code2 === 92 && body.charCodeAt(position + 1) === 34 && body.charCodeAt(position + 2) === 34 && body.charCodeAt(position + 3) === 34
    ) {
      rawValue += body.slice(chunkStart, position) + '"""';
      position += 4;
      chunkStart = position;
    } else {
      ++position;
    }
  }
  throw syntaxError(source, position, "Unterminated string.");
}
function uniCharCode(a2, b2, c2, d2) {
  return char2hex(a2) << 12 | char2hex(b2) << 8 | char2hex(c2) << 4 | char2hex(d2);
}
function char2hex(a2) {
  return a2 >= 48 && a2 <= 57 ? a2 - 48 : a2 >= 65 && a2 <= 70 ? a2 - 55 : a2 >= 97 && a2 <= 102 ? a2 - 87 : -1;
}
function readName(source, start, line, col, prev) {
  var body = source.body;
  var bodyLength = body.length;
  var position = start + 1;
  var code2 = 0;
  while (position !== bodyLength && !isNaN(code2 = body.charCodeAt(position)) && (code2 === 95 || // _
  code2 >= 48 && code2 <= 57 || // 0-9
  code2 >= 65 && code2 <= 90 || // A-Z
  code2 >= 97 && code2 <= 122)) {
    ++position;
  }
  return new Token(TokenKind.NAME, start, position, line, col, prev, body.slice(start, position));
}
function isNameStart(code2) {
  return code2 === 95 || code2 >= 65 && code2 <= 90 || code2 >= 97 && code2 <= 122;
}
function parse$5(source, options2) {
  var parser = new Parser(source, options2);
  return parser.parseDocument();
}
var Parser = /* @__PURE__ */ (function() {
  function Parser2(source, options2) {
    var sourceObj = isSource(source) ? source : new Source(source);
    this._lexer = new Lexer(sourceObj);
    this._options = options2;
  }
  var _proto = Parser2.prototype;
  _proto.parseName = function parseName2() {
    var token2 = this.expectToken(TokenKind.NAME);
    return {
      kind: Kind.NAME,
      value: token2.value,
      loc: this.loc(token2)
    };
  };
  _proto.parseDocument = function parseDocument() {
    var start = this._lexer.token;
    return {
      kind: Kind.DOCUMENT,
      definitions: this.many(TokenKind.SOF, this.parseDefinition, TokenKind.EOF),
      loc: this.loc(start)
    };
  };
  _proto.parseDefinition = function parseDefinition() {
    if (this.peek(TokenKind.NAME)) {
      switch (this._lexer.token.value) {
        case "query":
        case "mutation":
        case "subscription":
          return this.parseOperationDefinition();
        case "fragment":
          return this.parseFragmentDefinition();
        case "schema":
        case "scalar":
        case "type":
        case "interface":
        case "union":
        case "enum":
        case "input":
        case "directive":
          return this.parseTypeSystemDefinition();
        case "extend":
          return this.parseTypeSystemExtension();
      }
    } else if (this.peek(TokenKind.BRACE_L)) {
      return this.parseOperationDefinition();
    } else if (this.peekDescription()) {
      return this.parseTypeSystemDefinition();
    }
    throw this.unexpected();
  };
  _proto.parseOperationDefinition = function parseOperationDefinition() {
    var start = this._lexer.token;
    if (this.peek(TokenKind.BRACE_L)) {
      return {
        kind: Kind.OPERATION_DEFINITION,
        operation: "query",
        name: void 0,
        variableDefinitions: [],
        directives: [],
        selectionSet: this.parseSelectionSet(),
        loc: this.loc(start)
      };
    }
    var operation = this.parseOperationType();
    var name2;
    if (this.peek(TokenKind.NAME)) {
      name2 = this.parseName();
    }
    return {
      kind: Kind.OPERATION_DEFINITION,
      operation,
      name: name2,
      variableDefinitions: this.parseVariableDefinitions(),
      directives: this.parseDirectives(false),
      selectionSet: this.parseSelectionSet(),
      loc: this.loc(start)
    };
  };
  _proto.parseOperationType = function parseOperationType() {
    var operationToken = this.expectToken(TokenKind.NAME);
    switch (operationToken.value) {
      case "query":
        return "query";
      case "mutation":
        return "mutation";
      case "subscription":
        return "subscription";
    }
    throw this.unexpected(operationToken);
  };
  _proto.parseVariableDefinitions = function parseVariableDefinitions() {
    return this.optionalMany(TokenKind.PAREN_L, this.parseVariableDefinition, TokenKind.PAREN_R);
  };
  _proto.parseVariableDefinition = function parseVariableDefinition() {
    var start = this._lexer.token;
    return {
      kind: Kind.VARIABLE_DEFINITION,
      variable: this.parseVariable(),
      type: (this.expectToken(TokenKind.COLON), this.parseTypeReference()),
      defaultValue: this.expectOptionalToken(TokenKind.EQUALS) ? this.parseValueLiteral(true) : void 0,
      directives: this.parseDirectives(true),
      loc: this.loc(start)
    };
  };
  _proto.parseVariable = function parseVariable() {
    var start = this._lexer.token;
    this.expectToken(TokenKind.DOLLAR);
    return {
      kind: Kind.VARIABLE,
      name: this.parseName(),
      loc: this.loc(start)
    };
  };
  _proto.parseSelectionSet = function parseSelectionSet() {
    var start = this._lexer.token;
    return {
      kind: Kind.SELECTION_SET,
      selections: this.many(TokenKind.BRACE_L, this.parseSelection, TokenKind.BRACE_R),
      loc: this.loc(start)
    };
  };
  _proto.parseSelection = function parseSelection() {
    return this.peek(TokenKind.SPREAD) ? this.parseFragment() : this.parseField();
  };
  _proto.parseField = function parseField() {
    var start = this._lexer.token;
    var nameOrAlias = this.parseName();
    var alias;
    var name2;
    if (this.expectOptionalToken(TokenKind.COLON)) {
      alias = nameOrAlias;
      name2 = this.parseName();
    } else {
      name2 = nameOrAlias;
    }
    return {
      kind: Kind.FIELD,
      alias,
      name: name2,
      arguments: this.parseArguments(false),
      directives: this.parseDirectives(false),
      selectionSet: this.peek(TokenKind.BRACE_L) ? this.parseSelectionSet() : void 0,
      loc: this.loc(start)
    };
  };
  _proto.parseArguments = function parseArguments(isConst) {
    var item = isConst ? this.parseConstArgument : this.parseArgument;
    return this.optionalMany(TokenKind.PAREN_L, item, TokenKind.PAREN_R);
  };
  _proto.parseArgument = function parseArgument() {
    var start = this._lexer.token;
    var name2 = this.parseName();
    this.expectToken(TokenKind.COLON);
    return {
      kind: Kind.ARGUMENT,
      name: name2,
      value: this.parseValueLiteral(false),
      loc: this.loc(start)
    };
  };
  _proto.parseConstArgument = function parseConstArgument() {
    var start = this._lexer.token;
    return {
      kind: Kind.ARGUMENT,
      name: this.parseName(),
      value: (this.expectToken(TokenKind.COLON), this.parseValueLiteral(true)),
      loc: this.loc(start)
    };
  };
  _proto.parseFragment = function parseFragment() {
    var start = this._lexer.token;
    this.expectToken(TokenKind.SPREAD);
    var hasTypeCondition = this.expectOptionalKeyword("on");
    if (!hasTypeCondition && this.peek(TokenKind.NAME)) {
      return {
        kind: Kind.FRAGMENT_SPREAD,
        name: this.parseFragmentName(),
        directives: this.parseDirectives(false),
        loc: this.loc(start)
      };
    }
    return {
      kind: Kind.INLINE_FRAGMENT,
      typeCondition: hasTypeCondition ? this.parseNamedType() : void 0,
      directives: this.parseDirectives(false),
      selectionSet: this.parseSelectionSet(),
      loc: this.loc(start)
    };
  };
  _proto.parseFragmentDefinition = function parseFragmentDefinition() {
    var _this$_options;
    var start = this._lexer.token;
    this.expectKeyword("fragment");
    if (((_this$_options = this._options) === null || _this$_options === void 0 ? void 0 : _this$_options.experimentalFragmentVariables) === true) {
      return {
        kind: Kind.FRAGMENT_DEFINITION,
        name: this.parseFragmentName(),
        variableDefinitions: this.parseVariableDefinitions(),
        typeCondition: (this.expectKeyword("on"), this.parseNamedType()),
        directives: this.parseDirectives(false),
        selectionSet: this.parseSelectionSet(),
        loc: this.loc(start)
      };
    }
    return {
      kind: Kind.FRAGMENT_DEFINITION,
      name: this.parseFragmentName(),
      typeCondition: (this.expectKeyword("on"), this.parseNamedType()),
      directives: this.parseDirectives(false),
      selectionSet: this.parseSelectionSet(),
      loc: this.loc(start)
    };
  };
  _proto.parseFragmentName = function parseFragmentName() {
    if (this._lexer.token.value === "on") {
      throw this.unexpected();
    }
    return this.parseName();
  };
  _proto.parseValueLiteral = function parseValueLiteral(isConst) {
    var token2 = this._lexer.token;
    switch (token2.kind) {
      case TokenKind.BRACKET_L:
        return this.parseList(isConst);
      case TokenKind.BRACE_L:
        return this.parseObject(isConst);
      case TokenKind.INT:
        this._lexer.advance();
        return {
          kind: Kind.INT,
          value: token2.value,
          loc: this.loc(token2)
        };
      case TokenKind.FLOAT:
        this._lexer.advance();
        return {
          kind: Kind.FLOAT,
          value: token2.value,
          loc: this.loc(token2)
        };
      case TokenKind.STRING:
      case TokenKind.BLOCK_STRING:
        return this.parseStringLiteral();
      case TokenKind.NAME:
        this._lexer.advance();
        switch (token2.value) {
          case "true":
            return {
              kind: Kind.BOOLEAN,
              value: true,
              loc: this.loc(token2)
            };
          case "false":
            return {
              kind: Kind.BOOLEAN,
              value: false,
              loc: this.loc(token2)
            };
          case "null":
            return {
              kind: Kind.NULL,
              loc: this.loc(token2)
            };
          default:
            return {
              kind: Kind.ENUM,
              value: token2.value,
              loc: this.loc(token2)
            };
        }
      case TokenKind.DOLLAR:
        if (!isConst) {
          return this.parseVariable();
        }
        break;
    }
    throw this.unexpected();
  };
  _proto.parseStringLiteral = function parseStringLiteral() {
    var token2 = this._lexer.token;
    this._lexer.advance();
    return {
      kind: Kind.STRING,
      value: token2.value,
      block: token2.kind === TokenKind.BLOCK_STRING,
      loc: this.loc(token2)
    };
  };
  _proto.parseList = function parseList(isConst) {
    var _this = this;
    var start = this._lexer.token;
    var item = function item2() {
      return _this.parseValueLiteral(isConst);
    };
    return {
      kind: Kind.LIST,
      values: this.any(TokenKind.BRACKET_L, item, TokenKind.BRACKET_R),
      loc: this.loc(start)
    };
  };
  _proto.parseObject = function parseObject(isConst) {
    var _this2 = this;
    var start = this._lexer.token;
    var item = function item2() {
      return _this2.parseObjectField(isConst);
    };
    return {
      kind: Kind.OBJECT,
      fields: this.any(TokenKind.BRACE_L, item, TokenKind.BRACE_R),
      loc: this.loc(start)
    };
  };
  _proto.parseObjectField = function parseObjectField(isConst) {
    var start = this._lexer.token;
    var name2 = this.parseName();
    this.expectToken(TokenKind.COLON);
    return {
      kind: Kind.OBJECT_FIELD,
      name: name2,
      value: this.parseValueLiteral(isConst),
      loc: this.loc(start)
    };
  };
  _proto.parseDirectives = function parseDirectives(isConst) {
    var directives = [];
    while (this.peek(TokenKind.AT)) {
      directives.push(this.parseDirective(isConst));
    }
    return directives;
  };
  _proto.parseDirective = function parseDirective(isConst) {
    var start = this._lexer.token;
    this.expectToken(TokenKind.AT);
    return {
      kind: Kind.DIRECTIVE,
      name: this.parseName(),
      arguments: this.parseArguments(isConst),
      loc: this.loc(start)
    };
  };
  _proto.parseTypeReference = function parseTypeReference() {
    var start = this._lexer.token;
    var type2;
    if (this.expectOptionalToken(TokenKind.BRACKET_L)) {
      type2 = this.parseTypeReference();
      this.expectToken(TokenKind.BRACKET_R);
      type2 = {
        kind: Kind.LIST_TYPE,
        type: type2,
        loc: this.loc(start)
      };
    } else {
      type2 = this.parseNamedType();
    }
    if (this.expectOptionalToken(TokenKind.BANG)) {
      return {
        kind: Kind.NON_NULL_TYPE,
        type: type2,
        loc: this.loc(start)
      };
    }
    return type2;
  };
  _proto.parseNamedType = function parseNamedType() {
    var start = this._lexer.token;
    return {
      kind: Kind.NAMED_TYPE,
      name: this.parseName(),
      loc: this.loc(start)
    };
  };
  _proto.parseTypeSystemDefinition = function parseTypeSystemDefinition() {
    var keywordToken = this.peekDescription() ? this._lexer.lookahead() : this._lexer.token;
    if (keywordToken.kind === TokenKind.NAME) {
      switch (keywordToken.value) {
        case "schema":
          return this.parseSchemaDefinition();
        case "scalar":
          return this.parseScalarTypeDefinition();
        case "type":
          return this.parseObjectTypeDefinition();
        case "interface":
          return this.parseInterfaceTypeDefinition();
        case "union":
          return this.parseUnionTypeDefinition();
        case "enum":
          return this.parseEnumTypeDefinition();
        case "input":
          return this.parseInputObjectTypeDefinition();
        case "directive":
          return this.parseDirectiveDefinition();
      }
    }
    throw this.unexpected(keywordToken);
  };
  _proto.peekDescription = function peekDescription() {
    return this.peek(TokenKind.STRING) || this.peek(TokenKind.BLOCK_STRING);
  };
  _proto.parseDescription = function parseDescription() {
    if (this.peekDescription()) {
      return this.parseStringLiteral();
    }
  };
  _proto.parseSchemaDefinition = function parseSchemaDefinition() {
    var start = this._lexer.token;
    var description = this.parseDescription();
    this.expectKeyword("schema");
    var directives = this.parseDirectives(true);
    var operationTypes = this.many(TokenKind.BRACE_L, this.parseOperationTypeDefinition, TokenKind.BRACE_R);
    return {
      kind: Kind.SCHEMA_DEFINITION,
      description,
      directives,
      operationTypes,
      loc: this.loc(start)
    };
  };
  _proto.parseOperationTypeDefinition = function parseOperationTypeDefinition() {
    var start = this._lexer.token;
    var operation = this.parseOperationType();
    this.expectToken(TokenKind.COLON);
    var type2 = this.parseNamedType();
    return {
      kind: Kind.OPERATION_TYPE_DEFINITION,
      operation,
      type: type2,
      loc: this.loc(start)
    };
  };
  _proto.parseScalarTypeDefinition = function parseScalarTypeDefinition() {
    var start = this._lexer.token;
    var description = this.parseDescription();
    this.expectKeyword("scalar");
    var name2 = this.parseName();
    var directives = this.parseDirectives(true);
    return {
      kind: Kind.SCALAR_TYPE_DEFINITION,
      description,
      name: name2,
      directives,
      loc: this.loc(start)
    };
  };
  _proto.parseObjectTypeDefinition = function parseObjectTypeDefinition() {
    var start = this._lexer.token;
    var description = this.parseDescription();
    this.expectKeyword("type");
    var name2 = this.parseName();
    var interfaces = this.parseImplementsInterfaces();
    var directives = this.parseDirectives(true);
    var fields = this.parseFieldsDefinition();
    return {
      kind: Kind.OBJECT_TYPE_DEFINITION,
      description,
      name: name2,
      interfaces,
      directives,
      fields,
      loc: this.loc(start)
    };
  };
  _proto.parseImplementsInterfaces = function parseImplementsInterfaces() {
    var _this$_options2;
    if (!this.expectOptionalKeyword("implements")) {
      return [];
    }
    if (((_this$_options2 = this._options) === null || _this$_options2 === void 0 ? void 0 : _this$_options2.allowLegacySDLImplementsInterfaces) === true) {
      var types = [];
      this.expectOptionalToken(TokenKind.AMP);
      do {
        types.push(this.parseNamedType());
      } while (this.expectOptionalToken(TokenKind.AMP) || this.peek(TokenKind.NAME));
      return types;
    }
    return this.delimitedMany(TokenKind.AMP, this.parseNamedType);
  };
  _proto.parseFieldsDefinition = function parseFieldsDefinition() {
    var _this$_options3;
    if (((_this$_options3 = this._options) === null || _this$_options3 === void 0 ? void 0 : _this$_options3.allowLegacySDLEmptyFields) === true && this.peek(TokenKind.BRACE_L) && this._lexer.lookahead().kind === TokenKind.BRACE_R) {
      this._lexer.advance();
      this._lexer.advance();
      return [];
    }
    return this.optionalMany(TokenKind.BRACE_L, this.parseFieldDefinition, TokenKind.BRACE_R);
  };
  _proto.parseFieldDefinition = function parseFieldDefinition() {
    var start = this._lexer.token;
    var description = this.parseDescription();
    var name2 = this.parseName();
    var args = this.parseArgumentDefs();
    this.expectToken(TokenKind.COLON);
    var type2 = this.parseTypeReference();
    var directives = this.parseDirectives(true);
    return {
      kind: Kind.FIELD_DEFINITION,
      description,
      name: name2,
      arguments: args,
      type: type2,
      directives,
      loc: this.loc(start)
    };
  };
  _proto.parseArgumentDefs = function parseArgumentDefs() {
    return this.optionalMany(TokenKind.PAREN_L, this.parseInputValueDef, TokenKind.PAREN_R);
  };
  _proto.parseInputValueDef = function parseInputValueDef() {
    var start = this._lexer.token;
    var description = this.parseDescription();
    var name2 = this.parseName();
    this.expectToken(TokenKind.COLON);
    var type2 = this.parseTypeReference();
    var defaultValue;
    if (this.expectOptionalToken(TokenKind.EQUALS)) {
      defaultValue = this.parseValueLiteral(true);
    }
    var directives = this.parseDirectives(true);
    return {
      kind: Kind.INPUT_VALUE_DEFINITION,
      description,
      name: name2,
      type: type2,
      defaultValue,
      directives,
      loc: this.loc(start)
    };
  };
  _proto.parseInterfaceTypeDefinition = function parseInterfaceTypeDefinition() {
    var start = this._lexer.token;
    var description = this.parseDescription();
    this.expectKeyword("interface");
    var name2 = this.parseName();
    var interfaces = this.parseImplementsInterfaces();
    var directives = this.parseDirectives(true);
    var fields = this.parseFieldsDefinition();
    return {
      kind: Kind.INTERFACE_TYPE_DEFINITION,
      description,
      name: name2,
      interfaces,
      directives,
      fields,
      loc: this.loc(start)
    };
  };
  _proto.parseUnionTypeDefinition = function parseUnionTypeDefinition() {
    var start = this._lexer.token;
    var description = this.parseDescription();
    this.expectKeyword("union");
    var name2 = this.parseName();
    var directives = this.parseDirectives(true);
    var types = this.parseUnionMemberTypes();
    return {
      kind: Kind.UNION_TYPE_DEFINITION,
      description,
      name: name2,
      directives,
      types,
      loc: this.loc(start)
    };
  };
  _proto.parseUnionMemberTypes = function parseUnionMemberTypes() {
    return this.expectOptionalToken(TokenKind.EQUALS) ? this.delimitedMany(TokenKind.PIPE, this.parseNamedType) : [];
  };
  _proto.parseEnumTypeDefinition = function parseEnumTypeDefinition() {
    var start = this._lexer.token;
    var description = this.parseDescription();
    this.expectKeyword("enum");
    var name2 = this.parseName();
    var directives = this.parseDirectives(true);
    var values = this.parseEnumValuesDefinition();
    return {
      kind: Kind.ENUM_TYPE_DEFINITION,
      description,
      name: name2,
      directives,
      values,
      loc: this.loc(start)
    };
  };
  _proto.parseEnumValuesDefinition = function parseEnumValuesDefinition() {
    return this.optionalMany(TokenKind.BRACE_L, this.parseEnumValueDefinition, TokenKind.BRACE_R);
  };
  _proto.parseEnumValueDefinition = function parseEnumValueDefinition() {
    var start = this._lexer.token;
    var description = this.parseDescription();
    var name2 = this.parseName();
    var directives = this.parseDirectives(true);
    return {
      kind: Kind.ENUM_VALUE_DEFINITION,
      description,
      name: name2,
      directives,
      loc: this.loc(start)
    };
  };
  _proto.parseInputObjectTypeDefinition = function parseInputObjectTypeDefinition() {
    var start = this._lexer.token;
    var description = this.parseDescription();
    this.expectKeyword("input");
    var name2 = this.parseName();
    var directives = this.parseDirectives(true);
    var fields = this.parseInputFieldsDefinition();
    return {
      kind: Kind.INPUT_OBJECT_TYPE_DEFINITION,
      description,
      name: name2,
      directives,
      fields,
      loc: this.loc(start)
    };
  };
  _proto.parseInputFieldsDefinition = function parseInputFieldsDefinition() {
    return this.optionalMany(TokenKind.BRACE_L, this.parseInputValueDef, TokenKind.BRACE_R);
  };
  _proto.parseTypeSystemExtension = function parseTypeSystemExtension() {
    var keywordToken = this._lexer.lookahead();
    if (keywordToken.kind === TokenKind.NAME) {
      switch (keywordToken.value) {
        case "schema":
          return this.parseSchemaExtension();
        case "scalar":
          return this.parseScalarTypeExtension();
        case "type":
          return this.parseObjectTypeExtension();
        case "interface":
          return this.parseInterfaceTypeExtension();
        case "union":
          return this.parseUnionTypeExtension();
        case "enum":
          return this.parseEnumTypeExtension();
        case "input":
          return this.parseInputObjectTypeExtension();
      }
    }
    throw this.unexpected(keywordToken);
  };
  _proto.parseSchemaExtension = function parseSchemaExtension() {
    var start = this._lexer.token;
    this.expectKeyword("extend");
    this.expectKeyword("schema");
    var directives = this.parseDirectives(true);
    var operationTypes = this.optionalMany(TokenKind.BRACE_L, this.parseOperationTypeDefinition, TokenKind.BRACE_R);
    if (directives.length === 0 && operationTypes.length === 0) {
      throw this.unexpected();
    }
    return {
      kind: Kind.SCHEMA_EXTENSION,
      directives,
      operationTypes,
      loc: this.loc(start)
    };
  };
  _proto.parseScalarTypeExtension = function parseScalarTypeExtension() {
    var start = this._lexer.token;
    this.expectKeyword("extend");
    this.expectKeyword("scalar");
    var name2 = this.parseName();
    var directives = this.parseDirectives(true);
    if (directives.length === 0) {
      throw this.unexpected();
    }
    return {
      kind: Kind.SCALAR_TYPE_EXTENSION,
      name: name2,
      directives,
      loc: this.loc(start)
    };
  };
  _proto.parseObjectTypeExtension = function parseObjectTypeExtension() {
    var start = this._lexer.token;
    this.expectKeyword("extend");
    this.expectKeyword("type");
    var name2 = this.parseName();
    var interfaces = this.parseImplementsInterfaces();
    var directives = this.parseDirectives(true);
    var fields = this.parseFieldsDefinition();
    if (interfaces.length === 0 && directives.length === 0 && fields.length === 0) {
      throw this.unexpected();
    }
    return {
      kind: Kind.OBJECT_TYPE_EXTENSION,
      name: name2,
      interfaces,
      directives,
      fields,
      loc: this.loc(start)
    };
  };
  _proto.parseInterfaceTypeExtension = function parseInterfaceTypeExtension() {
    var start = this._lexer.token;
    this.expectKeyword("extend");
    this.expectKeyword("interface");
    var name2 = this.parseName();
    var interfaces = this.parseImplementsInterfaces();
    var directives = this.parseDirectives(true);
    var fields = this.parseFieldsDefinition();
    if (interfaces.length === 0 && directives.length === 0 && fields.length === 0) {
      throw this.unexpected();
    }
    return {
      kind: Kind.INTERFACE_TYPE_EXTENSION,
      name: name2,
      interfaces,
      directives,
      fields,
      loc: this.loc(start)
    };
  };
  _proto.parseUnionTypeExtension = function parseUnionTypeExtension() {
    var start = this._lexer.token;
    this.expectKeyword("extend");
    this.expectKeyword("union");
    var name2 = this.parseName();
    var directives = this.parseDirectives(true);
    var types = this.parseUnionMemberTypes();
    if (directives.length === 0 && types.length === 0) {
      throw this.unexpected();
    }
    return {
      kind: Kind.UNION_TYPE_EXTENSION,
      name: name2,
      directives,
      types,
      loc: this.loc(start)
    };
  };
  _proto.parseEnumTypeExtension = function parseEnumTypeExtension() {
    var start = this._lexer.token;
    this.expectKeyword("extend");
    this.expectKeyword("enum");
    var name2 = this.parseName();
    var directives = this.parseDirectives(true);
    var values = this.parseEnumValuesDefinition();
    if (directives.length === 0 && values.length === 0) {
      throw this.unexpected();
    }
    return {
      kind: Kind.ENUM_TYPE_EXTENSION,
      name: name2,
      directives,
      values,
      loc: this.loc(start)
    };
  };
  _proto.parseInputObjectTypeExtension = function parseInputObjectTypeExtension() {
    var start = this._lexer.token;
    this.expectKeyword("extend");
    this.expectKeyword("input");
    var name2 = this.parseName();
    var directives = this.parseDirectives(true);
    var fields = this.parseInputFieldsDefinition();
    if (directives.length === 0 && fields.length === 0) {
      throw this.unexpected();
    }
    return {
      kind: Kind.INPUT_OBJECT_TYPE_EXTENSION,
      name: name2,
      directives,
      fields,
      loc: this.loc(start)
    };
  };
  _proto.parseDirectiveDefinition = function parseDirectiveDefinition() {
    var start = this._lexer.token;
    var description = this.parseDescription();
    this.expectKeyword("directive");
    this.expectToken(TokenKind.AT);
    var name2 = this.parseName();
    var args = this.parseArgumentDefs();
    var repeatable = this.expectOptionalKeyword("repeatable");
    this.expectKeyword("on");
    var locations = this.parseDirectiveLocations();
    return {
      kind: Kind.DIRECTIVE_DEFINITION,
      description,
      name: name2,
      arguments: args,
      repeatable,
      locations,
      loc: this.loc(start)
    };
  };
  _proto.parseDirectiveLocations = function parseDirectiveLocations() {
    return this.delimitedMany(TokenKind.PIPE, this.parseDirectiveLocation);
  };
  _proto.parseDirectiveLocation = function parseDirectiveLocation() {
    var start = this._lexer.token;
    var name2 = this.parseName();
    if (DirectiveLocation[name2.value] !== void 0) {
      return name2;
    }
    throw this.unexpected(start);
  };
  _proto.loc = function loc(startToken) {
    var _this$_options4;
    if (((_this$_options4 = this._options) === null || _this$_options4 === void 0 ? void 0 : _this$_options4.noLocation) !== true) {
      return new Location(startToken, this._lexer.lastToken, this._lexer.source);
    }
  };
  _proto.peek = function peek(kind) {
    return this._lexer.token.kind === kind;
  };
  _proto.expectToken = function expectToken(kind) {
    var token2 = this._lexer.token;
    if (token2.kind === kind) {
      this._lexer.advance();
      return token2;
    }
    throw syntaxError(this._lexer.source, token2.start, "Expected ".concat(getTokenKindDesc(kind), ", found ").concat(getTokenDesc(token2), "."));
  };
  _proto.expectOptionalToken = function expectOptionalToken(kind) {
    var token2 = this._lexer.token;
    if (token2.kind === kind) {
      this._lexer.advance();
      return token2;
    }
    return void 0;
  };
  _proto.expectKeyword = function expectKeyword(value) {
    var token2 = this._lexer.token;
    if (token2.kind === TokenKind.NAME && token2.value === value) {
      this._lexer.advance();
    } else {
      throw syntaxError(this._lexer.source, token2.start, 'Expected "'.concat(value, '", found ').concat(getTokenDesc(token2), "."));
    }
  };
  _proto.expectOptionalKeyword = function expectOptionalKeyword(value) {
    var token2 = this._lexer.token;
    if (token2.kind === TokenKind.NAME && token2.value === value) {
      this._lexer.advance();
      return true;
    }
    return false;
  };
  _proto.unexpected = function unexpected(atToken) {
    var token2 = atToken !== null && atToken !== void 0 ? atToken : this._lexer.token;
    return syntaxError(this._lexer.source, token2.start, "Unexpected ".concat(getTokenDesc(token2), "."));
  };
  _proto.any = function any(openKind, parseFn, closeKind) {
    this.expectToken(openKind);
    var nodes = [];
    while (!this.expectOptionalToken(closeKind)) {
      nodes.push(parseFn.call(this));
    }
    return nodes;
  };
  _proto.optionalMany = function optionalMany(openKind, parseFn, closeKind) {
    if (this.expectOptionalToken(openKind)) {
      var nodes = [];
      do {
        nodes.push(parseFn.call(this));
      } while (!this.expectOptionalToken(closeKind));
      return nodes;
    }
    return [];
  };
  _proto.many = function many(openKind, parseFn, closeKind) {
    this.expectToken(openKind);
    var nodes = [];
    do {
      nodes.push(parseFn.call(this));
    } while (!this.expectOptionalToken(closeKind));
    return nodes;
  };
  _proto.delimitedMany = function delimitedMany(delimiterKind, parseFn) {
    this.expectOptionalToken(delimiterKind);
    var nodes = [];
    do {
      nodes.push(parseFn.call(this));
    } while (this.expectOptionalToken(delimiterKind));
    return nodes;
  };
  return Parser2;
})();
function getTokenDesc(token2) {
  var value = token2.value;
  return getTokenKindDesc(token2.kind) + (value != null ? ' "'.concat(value, '"') : "");
}
function getTokenKindDesc(kind) {
  return isPunctuatorTokenKind(kind) ? '"'.concat(kind, '"') : kind;
}
function l$1(a2, b2) {
  b2.tag = a2;
  return b2;
}
function m$1() {
}
function p$2(a2) {
  return function(b2) {
    var c2 = a2.length;
    let d2 = false, e = false, f2 = false, g = 0;
    b2(
      l$1(0, [
        function(h2) {
          if (h2) {
            d2 = true;
          } else if (e) {
            f2 = true;
          } else {
            for (e = f2 = true; f2 && !d2; ) {
              g < c2 ? (h2 = a2[g], g = g + 1 | 0, f2 = false, b2(l$1(1, [h2]))) : (d2 = true, b2(0));
            }
            e = false;
          }
        }
      ])
    );
  };
}
function r$1() {
}
function t(a2) {
  a2(0);
}
function u$1(a2) {
  let b2 = false;
  a2(
    l$1(0, [
      function(c2) {
        c2 ? b2 = true : b2 || a2(0);
      }
    ])
  );
}
function C(a2) {
  return a2(0);
}
function D$1(a2) {
  return function(b2) {
    return function(c2) {
      let e = m$1, f2 = false, g = [], h2 = false;
      b2(function(b3) {
        "number" == typeof b3 ? h2 || (h2 = true, 0 === g.length && c2(0)) : b3.tag ? h2 || (f2 = false, (function(a3) {
          function b4(a4) {
            "number" == typeof a4 ? 0 !== g.length && (g = g.filter(d2), a4 = 0 === g.length, h2 && a4 ? c2(0) : !f2 && a4 && (f2 = true, e(0))) : a4.tag ? 0 !== g.length && (c2(l$1(1, [a4[0]])), k2(0)) : (k2 = a4 = a4[0], g = g.concat(a4), a4(0));
          }
          function d2(a4) {
            return a4 !== k2;
          }
          let k2 = m$1;
          1 === a3.length ? a3(b4) : a3.bind(null, b4);
        })(a2(b3[0])), f2 || (f2 = true, e(0))) : e = b3[0];
      });
      c2(
        l$1(0, [
          function(a3) {
            a3 ? (h2 || (h2 = true, e(a3)), g.forEach(function(c3) {
              return c3(a3);
            }), g = []) : (f2 || h2 ? f2 = false : (f2 = true, e(0)), g.forEach(C));
          }
        ])
      );
    };
  };
}
function E$1(a2) {
  return a2;
}
function H(a2) {
  return function(b2) {
    return function(c2) {
      let d2 = false;
      return b2(function(e) {
        if ("number" == typeof e) {
          d2 || (d2 = true, c2(e));
        } else if (e.tag) {
          d2 || (a2(e[0]), c2(e));
        } else {
          var g = e[0];
          c2(
            l$1(0, [
              function(a3) {
                if (!d2) {
                  return a3 && (d2 = true), g(a3);
                }
              }
            ])
          );
        }
      });
    };
  };
}
function J$1(a2) {
  a2(0);
}
function K(a2) {
  return function(b2) {
    return function(c2) {
      function d2(a3) {
        h2 && ("number" == typeof a3 ? (h2 = false, n2 ? c2(a3) : f2 || (f2 = true, e(0))) : a3.tag ? (c2(a3), k2 ? k2 = false : g(0)) : (g = a3 = a3[0], k2 = false, a3(0)));
      }
      let e = m$1, f2 = false, g = m$1, h2 = false, k2 = false, n2 = false;
      b2(function(b3) {
        "number" == typeof b3 ? n2 || (n2 = true, h2 || c2(0)) : b3.tag ? n2 || (h2 && (g(1), g = m$1), f2 ? f2 = false : (f2 = true, e(0)), b3 = a2(b3[0]), h2 = true, b3(d2)) : e = b3[0];
      });
      c2(
        l$1(0, [
          function(a3) {
            if (a3) {
              if (n2 || (n2 = true, e(1)), h2) {
                return h2 = false, g(1);
              }
            } else {
              n2 || f2 || (f2 = true, e(0)), h2 && !k2 && (k2 = true, g(0));
            }
          }
        ])
      );
    };
  };
}
function M(a2) {
  return function(b2) {
    return function(c2) {
      let d2 = [], e = m$1;
      return b2(function(b3) {
        "number" == typeof b3 ? p$2(d2)(c2) : b3.tag ? (d2.length >= a2 && 0 < a2 && d2.shift(), d2.push(b3[0]), e(0)) : (b3 = b3[0], e = b3, b3(0));
      });
    };
  };
}
function N$1(a2) {
  return function(b2) {
    let c2 = m$1, d2 = false;
    b2(function(e) {
      "number" == typeof e ? d2 = true : e.tag ? d2 || (a2(e[0]), c2(0)) : (c2 = e = e[0], e(0));
    });
    return {
      unsubscribe: function() {
        if (!d2) {
          return d2 = true, c2(1);
        }
      }
    };
  };
}
function O() {
}
function buffer$1(a2) {
  return function(b2) {
    return function(c2) {
      function d2(a3) {
        "number" == typeof a3 ? k2 || (k2 = true, f2(1), 0 < e.length && c2(l$1(1, [e])), c2(0)) : a3.tag ? !k2 && 0 < e.length && (a3 = e, e = [], c2(l$1(1, [a3]))) : g = a3[0];
      }
      let e = [], f2 = m$1, g = m$1, h2 = false, k2 = false;
      b2(function(b3) {
        "number" == typeof b3 ? k2 || (k2 = true, g(1), 0 < e.length && c2(l$1(1, [e])), c2(0)) : b3.tag ? k2 || (e.push(b3[0]), h2 ? h2 = false : (h2 = true, f2(0), g(0))) : (f2 = b3[0], a2(d2));
      });
      c2(
        l$1(0, [
          function(a3) {
            if (!k2) {
              if (a3) {
                return k2 = true, f2(1), g(1);
              }
              if (!h2) {
                return h2 = true, f2(0), g(0);
              }
            }
          }
        ])
      );
    };
  };
}
function filter$1(a2) {
  return function(b2) {
    return function(c2) {
      let d2 = m$1;
      return b2(function(b3) {
        "number" == typeof b3 ? c2(b3) : b3.tag ? a2(b3[0]) ? c2(b3) : d2(0) : (d2 = b3[0], c2(b3));
      });
    };
  };
}
function fromPromise$1(a2) {
  return function(b2) {
    let c2 = false;
    a2.then(function(a3) {
      c2 || (b2(l$1(1, [a3])), b2(0));
      return Promise.resolve(void 0);
    });
    b2(
      l$1(0, [
        function(a3) {
          a3 && (c2 = true);
        }
      ])
    );
  };
}
function fromValue$1(a2) {
  return function(b2) {
    let c2 = false;
    b2(
      l$1(0, [
        function(d2) {
          d2 ? c2 = true : c2 || (c2 = true, b2(l$1(1, [a2])), b2(0));
        }
      ])
    );
  };
}
function make$1(a2) {
  return function(b2) {
    let c2 = r$1, d2 = false;
    c2 = a2({
      next: function(a3) {
        d2 || b2(l$1(1, [a3]));
      },
      complete: function() {
        d2 || (d2 = true, b2(0));
      }
    });
    b2(
      l$1(0, [
        function(a3) {
          if (a3 && !d2) {
            return d2 = true, c2();
          }
        }
      ])
    );
  };
}
function makeSubject$1() {
  let a2 = [], b2 = false;
  return {
    source: function(c2) {
      function b3(a3) {
        return a3 !== c2;
      }
      a2 = a2.concat(c2);
      c2(
        l$1(0, [
          function(c3) {
            c3 && (a2 = a2.filter(b3));
          }
        ])
      );
    },
    next: function(c2) {
      b2 || a2.forEach(function(a3) {
        a3(l$1(1, [c2]));
      });
    },
    complete: function() {
      b2 || (b2 = true, a2.forEach(t));
    }
  };
}
function map$1$1(a2) {
  return function(b2) {
    return function(c2) {
      return b2(function(b3) {
        b3 = "number" == typeof b3 ? 0 : b3.tag ? l$1(1, [a2(b3[0])]) : l$1(0, [b3[0]]);
        c2(b3);
      });
    };
  };
}
function merge$1(a2) {
  return D$1(E$1)(p$2(a2));
}
function onEnd$1(a2) {
  return function(b2) {
    return function(c2) {
      let d2 = false;
      return b2(function(b3) {
        if ("number" == typeof b3) {
          if (d2) {
            return;
          }
          d2 = true;
          c2(b3);
          return a2();
        }
        if (b3.tag) {
          d2 || c2(b3);
        } else {
          var e = b3[0];
          c2(
            l$1(0, [
              function(c3) {
                if (!d2) {
                  return c3 ? (d2 = true, e(c3), a2()) : e(c3);
                }
              }
            ])
          );
        }
      });
    };
  };
}
function onStart$1(a2) {
  return function(b2) {
    return function(c2) {
      return b2(function(b3) {
        "number" == typeof b3 ? c2(b3) : b3.tag ? c2(b3) : (c2(b3), a2());
      });
    };
  };
}
function pipe() {
  for (var a2 = arguments, b2 = arguments[0], c2 = 1, d2 = arguments.length; c2 < d2; c2++) {
    b2 = a2[c2](b2);
  }
  return b2;
}
function publish$1(a2) {
  return N$1(O)(a2);
}
function share$1(a2) {
  function b2(a3) {
    "number" == typeof a3 ? (c2.forEach(J$1), c2 = []) : a3.tag ? (e = false, c2.forEach(function(b3) {
      b3(a3);
    })) : d2 = a3[0];
  }
  let c2 = [], d2 = m$1, e = false;
  return function(f2) {
    function g(a3) {
      return a3 !== f2;
    }
    c2 = c2.concat(f2);
    1 === c2.length && a2(b2);
    f2(
      l$1(0, [
        function(a3) {
          if (a3) {
            if (c2 = c2.filter(g), 0 === c2.length) {
              return d2(1);
            }
          } else {
            e || (e = true, d2(a3));
          }
        }
      ])
    );
  };
}
function skipUntil$1(a2) {
  return function(b2) {
    return function(c2) {
      function d2(a3) {
        "number" == typeof a3 ? g && (k2 = true, e(1)) : a3.tag ? (g = false, f2(1)) : (f2 = a3 = a3[0], a3(0));
      }
      let e = m$1, f2 = m$1, g = true, h2 = false, k2 = false;
      b2(function(b3) {
        "number" == typeof b3 ? (g && f2(1), k2 = true, c2(0)) : b3.tag ? g || k2 ? h2 ? h2 = false : (h2 = true, e(0), f2(0)) : (h2 = false, c2(b3)) : (e = b3[0], a2(d2));
      });
      c2(
        l$1(0, [
          function(a3) {
            if (!k2) {
              if (a3) {
                if (k2 = true, e(1), g) {
                  return f2(1);
                }
              } else {
                h2 || (h2 = true, g && f2(0), e(0));
              }
            }
          }
        ])
      );
    };
  };
}
function take$1(a2) {
  return function(b2) {
    return function(c2) {
      let d2 = false, e = 0, f2 = m$1;
      b2(function(b3) {
        "number" == typeof b3 ? d2 || (d2 = true, c2(0)) : b3.tag ? e < a2 && !d2 && (e = e + 1 | 0, c2(b3), !d2 && e >= a2 && (d2 = true, c2(0), f2(1))) : (b3 = b3[0], f2 = b3);
      });
      c2(
        l$1(0, [
          function(b3) {
            if (!d2) {
              if (b3) {
                return d2 = true, f2(1);
              }
              if (e < a2) {
                return f2(0);
              }
            }
          }
        ])
      );
    };
  };
}
function takeUntil$1(a2) {
  return function(b2) {
    return function(c2) {
      function d2(a3) {
        "number" != typeof a3 && (a3.tag ? (e = true, f2(1), c2(0)) : (g = a3 = a3[0], a3(0)));
      }
      let e = false, f2 = m$1, g = m$1;
      b2(function(b3) {
        "number" == typeof b3 ? e || (e = true, g(1), c2(0)) : b3.tag ? e || c2(b3) : (f2 = b3[0], a2(d2));
      });
      c2(
        l$1(0, [
          function(a3) {
            if (!e) {
              return a3 ? (e = true, f2(1), g(1)) : f2(0);
            }
          }
        ])
      );
    };
  };
}
function toPromise$1(a2) {
  return new Promise(function(b2) {
    M(1)(a2)(function(a3) {
      if ("number" != typeof a3) {
        if (a3.tag) {
          b2(a3[0]);
        } else {
          a3[0](0);
        }
      }
    });
  });
}
"function" == typeof Symbol ? Symbol.observable || (Symbol.observable = Symbol("observable")) : "@@observable";
function rehydrateGraphQlError(r2) {
  if ("string" == typeof r2) {
    return new GraphQLError(r2);
  } else if ("object" == typeof r2 && r2.message) {
    return new GraphQLError(r2.message, r2.nodes, r2.source, r2.positions, r2.path, r2, r2.extensions || {});
  } else {
    return r2;
  }
}
var o = (function(e) {
  function CombinedError(r2) {
    var t2 = r2.networkError;
    var n2 = r2.response;
    var a2 = (r2.graphQLErrors || []).map(rehydrateGraphQlError);
    var o2 = (function generateErrorMessage(e2, r3) {
      var t3 = "";
      if (void 0 !== e2) {
        return t3 = "[Network] " + e2.message;
      }
      if (void 0 !== r3) {
        r3.forEach((function(e3) {
          t3 += "[GraphQL] " + e3.message + "\n";
        }));
      }
      return t3.trim();
    })(t2, a2);
    e.call(this, o2);
    this.name = "CombinedError";
    this.message = o2;
    this.graphQLErrors = a2;
    this.networkError = t2;
    this.response = n2;
  }
  if (e) {
    CombinedError.__proto__ = e;
  }
  (CombinedError.prototype = Object.create(e && e.prototype)).constructor = CombinedError;
  CombinedError.prototype.toString = function toString2() {
    return this.message;
  };
  return CombinedError;
})(Error);
function phash(e, r2) {
  e |= 0;
  for (var t2 = 0, n2 = 0 | r2.length; t2 < n2; t2++) {
    e = (e << 5) + e + r2.charCodeAt(t2);
  }
  return e;
}
function hash$2(e) {
  return phash(5381, e) >>> 0;
}
var i$1 = /* @__PURE__ */ new Set();
var s = /* @__PURE__ */ new WeakMap();
function stringify(e) {
  if (null === e || i$1.has(e)) {
    return "null";
  } else if ("object" != typeof e) {
    return JSON.stringify(e) || "";
  } else if (e.toJSON) {
    return stringify(e.toJSON());
  } else if (Array.isArray(e)) {
    var r2 = "[";
    for (var t2 = 0, n2 = e.length; t2 < n2; t2++) {
      if (t2 > 0) {
        r2 += ",";
      }
      var a2 = stringify(e[t2]);
      r2 += a2.length > 0 ? a2 : "null";
    }
    return r2 += "]";
  }
  var o2 = Object.keys(e).sort();
  if (!o2.length && e.constructor && e.constructor !== Object) {
    var u2 = s.get(e) || Math.random().toString(36).slice(2);
    s.set(e, u2);
    return '{"__key":"' + u2 + '"}';
  }
  i$1.add(e);
  var f2 = "{";
  for (var c2 = 0, l2 = o2.length; c2 < l2; c2++) {
    var h2 = o2[c2];
    var p2 = stringify(e[h2]);
    if (p2) {
      if (f2.length > 1) {
        f2 += ",";
      }
      f2 += stringify(h2) + ":" + p2;
    }
  }
  i$1.delete(e);
  return f2 += "}";
}
function stringifyVariables(e) {
  i$1.clear();
  return stringify(e);
}
var u = /("{3}[\s\S]*"{3}|"(?:\\.|[^"])*")/g;
var f$1 = /([\s,]|#[^\n\r]+)+/g;
function replaceOutsideStrings(e, r2) {
  return r2 % 2 == 0 ? e.replace(f$1, " ").trim() : e;
}
function stringifyDocument(e) {
  var r2 = ("string" != typeof e ? e.loc && e.loc.source.body || print$1(e) : e).split(u).map(replaceOutsideStrings).join("");
  if ("string" != typeof e) {
    var t2 = "definitions" in e && getOperationName$1(e);
    if (t2) {
      r2 = "# " + t2 + "\n" + r2;
    }
    if (!e.loc) {
      e.loc = {
        start: 0,
        end: r2.length,
        source: {
          body: r2,
          name: "gql",
          locationOffset: {
            line: 1,
            column: 1
          }
        }
      };
    }
  }
  return r2;
}
var c$1 = /* @__PURE__ */ new Map();
function keyDocument(e) {
  var r2;
  var n2;
  if ("string" == typeof e) {
    r2 = hash$2(stringifyDocument(e));
    n2 = c$1.get(r2) || parse$5(e, {
      noLocation: true
    });
  } else {
    r2 = e.__key || hash$2(stringifyDocument(e));
    n2 = c$1.get(r2) || e;
  }
  if (!n2.loc) {
    stringifyDocument(n2);
  }
  n2.__key = r2;
  c$1.set(r2, n2);
  return n2;
}
function createRequest(e, r2) {
  if (!r2) {
    r2 = {};
  }
  var t2 = keyDocument(e);
  return {
    key: phash(t2.__key, stringifyVariables(r2)) >>> 0,
    query: t2,
    variables: r2
  };
}
function getOperationName$1(e) {
  for (var t2 = 0, n2 = e.definitions.length; t2 < n2; t2++) {
    var a2 = e.definitions[t2];
    if (a2.kind === Kind.OPERATION_DEFINITION && a2.name) {
      return a2.name.value;
    }
  }
}
function getOperationType(e) {
  for (var t2 = 0, n2 = e.definitions.length; t2 < n2; t2++) {
    var a2 = e.definitions[t2];
    if (a2.kind === Kind.OPERATION_DEFINITION) {
      return a2.operation;
    }
  }
}
function _extends$3() {
  return (_extends$3 = Object.assign || function(e) {
    for (var r2 = 1; r2 < arguments.length; r2++) {
      var t2 = arguments[r2];
      for (var n2 in t2) {
        if (Object.prototype.hasOwnProperty.call(t2, n2)) {
          e[n2] = t2[n2];
        }
      }
    }
    return e;
  }).apply(this, arguments);
}
function makeResult(e, r2, t2) {
  if (!("data" in r2) && !("errors" in r2) || "path" in r2) {
    throw new Error("No Content");
  }
  return {
    operation: e,
    data: r2.data,
    error: Array.isArray(r2.errors) ? new o({
      graphQLErrors: r2.errors,
      response: t2
    }) : void 0,
    extensions: "object" == typeof r2.extensions && r2.extensions || void 0,
    hasNext: !!r2.hasNext
  };
}
function mergeResultPatch(e, r2, t2) {
  var n2 = _extends$3({}, e);
  n2.hasNext = !!r2.hasNext;
  if (!("path" in r2)) {
    if ("data" in r2) {
      n2.data = r2.data;
    }
    return n2;
  }
  if (Array.isArray(r2.errors)) {
    n2.error = new o({
      graphQLErrors: n2.error ? n2.error.graphQLErrors.concat(r2.errors) : r2.errors,
      response: t2
    });
  }
  var a2 = n2.data = _extends$3({}, n2.data);
  var i2 = 0;
  var s2;
  while (i2 < r2.path.length) {
    a2 = a2[s2 = r2.path[i2++]] = Array.isArray(a2[s2]) ? [].concat(a2[s2]) : _extends$3({}, a2[s2]);
  }
  _extends$3(a2, r2.data);
  return n2;
}
function makeErrorResult(e, r2, t2) {
  return {
    operation: e,
    data: void 0,
    error: new o({
      networkError: r2,
      response: t2
    }),
    extensions: void 0
  };
}
function shouldUseGet(e) {
  return "query" === e.kind && !!e.context.preferGetMethod;
}
function makeFetchBody(e) {
  return {
    query: print$1(e.query),
    operationName: getOperationName$1(e.query),
    variables: e.variables || void 0,
    extensions: void 0
  };
}
function makeFetchURL(e, r2) {
  var t2 = shouldUseGet(e);
  var n2 = e.context.url;
  if (!t2 || !r2) {
    return n2;
  }
  var a2 = [];
  if (r2.operationName) {
    a2.push("operationName=" + encodeURIComponent(r2.operationName));
  }
  if (r2.query) {
    a2.push("query=" + encodeURIComponent(r2.query.replace(/#[^\n\r]+/g, " ").trim()));
  }
  if (r2.variables) {
    a2.push("variables=" + encodeURIComponent(stringifyVariables(r2.variables)));
  }
  if (r2.extensions) {
    a2.push("extensions=" + encodeURIComponent(stringifyVariables(r2.extensions)));
  }
  var o2 = n2 + "?" + a2.join("&");
  if (o2.length > 2047) {
    e.context.preferGetMethod = false;
    return n2;
  }
  return o2;
}
function makeFetchOptions(e, r2) {
  var t2 = shouldUseGet(e);
  var n2 = "function" == typeof e.context.fetchOptions ? e.context.fetchOptions() : e.context.fetchOptions || {};
  return _extends$3({}, n2, {
    body: !t2 && r2 ? JSON.stringify(r2) : void 0,
    method: t2 ? "GET" : "POST",
    headers: t2 ? n2.headers : _extends$3({}, {
      "content-type": "application/json"
    }, n2.headers)
  });
}
var l = "undefined" != typeof Symbol ? Symbol.asyncIterator : null;
var h$1 = "undefined" != typeof TextDecoder ? new TextDecoder() : null;
var p$1 = /content-type:[^\r\n]*application\/json/i;
var d$1 = /boundary="?([^=";]+)"?/i;
function makeFetchSource(e, r2, t2) {
  var n2 = "manual" === t2.redirect ? 400 : 300;
  var o2 = e.context.fetch;
  return make$1((function(a2) {
    var i2 = a2.next;
    var s2 = a2.complete;
    var u2 = "undefined" != typeof AbortController ? new AbortController() : null;
    if (u2) {
      t2.signal = u2.signal;
    }
    var f2 = false;
    function executeIncrementalFetch(e2, r3, t3) {
      var n3 = t3.headers && t3.headers.get("Content-Type") || "";
      if (!/multipart\/mixed/i.test(n3)) {
        return t3.json().then((function(n4) {
          var a4 = makeResult(r3, n4, t3);
          f2 = true;
          e2(a4);
        }));
      }
      var a3 = "---";
      var o3 = n3.match(d$1);
      if (o3) {
        a3 = "--" + o3[1];
      }
      var i3;
      var cancel = function() {
      };
      if (l && t3[l]) {
        var s3 = t3[l]();
        i3 = s3.next.bind(s3);
      } else if ("body" in t3 && t3.body) {
        var u3 = t3.body.getReader();
        cancel = u3.cancel.bind(u3);
        i3 = u3.read.bind(u3);
      } else {
        throw new TypeError("Streaming requests unsupported");
      }
      var c3 = "";
      var v3 = true;
      var m3 = null;
      var g = null;
      return i3().then((function next(n4) {
        if (!n4.done) {
          var o4 = (function toString2(e3) {
            return "Buffer" === e3.constructor.name ? e3.toString() : h$1.decode(e3);
          })(n4.value);
          var s4 = o4.indexOf(a3);
          if (s4 > -1) {
            s4 += c3.length;
          } else {
            s4 = c3.indexOf(a3);
          }
          c3 += o4;
          while (s4 > -1) {
            var u4 = c3.slice(0, s4);
            var l2 = c3.slice(s4 + a3.length);
            if (v3) {
              v3 = false;
            } else {
              var d2 = u4.indexOf("\r\n\r\n") + 4;
              var y2 = u4.slice(0, d2);
              var x2 = u4.slice(d2, u4.lastIndexOf("\r\n"));
              var b2 = void 0;
              if (p$1.test(y2)) {
                try {
                  b2 = JSON.parse(x2);
                  m3 = g = g ? mergeResultPatch(g, b2, t3) : makeResult(r3, b2, t3);
                } catch (e3) {
                }
              }
              if ("--" === l2.slice(0, 2) || b2 && !b2.hasNext) {
                if (!g) {
                  return e2(makeResult(r3, {}, t3));
                }
                break;
              }
            }
            s4 = (c3 = l2).indexOf(a3);
          }
        } else {
          f2 = true;
        }
        if (m3) {
          e2(m3);
          m3 = null;
        }
        if (!n4.done && (!g || g.hasNext)) {
          return i3().then(next);
        }
      })).finally(cancel);
    }
    var c2 = false;
    var v2 = false;
    var m2;
    Promise.resolve().then((function() {
      if (c2) {
        return;
      }
      return (o2 || fetch)(r2, t2);
    })).then((function(r3) {
      if (!r3) {
        return;
      }
      v2 = (m2 = r3).status < 200 || m2.status >= n2;
      return executeIncrementalFetch(i2, e, m2);
    })).then(s2).catch((function(r3) {
      if (f2) {
        throw r3;
      }
      if ("AbortError" !== r3.name) {
        var t3 = makeErrorResult(e, v2 ? new Error(m2.statusText) : r3, m2);
        i2(t3);
        s2();
      }
    }));
    return function() {
      c2 = true;
      if (u2) {
        u2.abort();
      }
    };
  }));
}
function collectTypes(e2, n2) {
  if (Array.isArray(e2)) {
    for (var r2 = 0; r2 < e2.length; r2++) {
      collectTypes(e2[r2], n2);
    }
  } else if ("object" == typeof e2 && null !== e2) {
    for (var t2 in e2) {
      if ("__typename" === t2 && "string" == typeof e2[t2]) {
        n2[e2[t2]] = 0;
      } else {
        collectTypes(e2[t2], n2);
      }
    }
  }
  return n2;
}
function collectTypesFromResponse(e2) {
  return Object.keys(collectTypes(e2, {}));
}
var formatNode = function(e2) {
  if (e2.selectionSet && !e2.selectionSet.selections.some((function(e3) {
    return e3.kind === Kind.FIELD && "__typename" === e3.name.value && !e3.alias;
  }))) {
    return _extends$3({}, e2, {
      selectionSet: _extends$3({}, e2.selectionSet, {
        selections: e2.selectionSet.selections.concat([{
          kind: Kind.FIELD,
          name: {
            kind: Kind.NAME,
            value: "__typename"
          }
        }])
      })
    });
  }
};
var F$1 = /* @__PURE__ */ new Map();
function formatDocument(n2) {
  var r2 = keyDocument(n2);
  var a2 = F$1.get(r2.__key);
  if (!a2) {
    a2 = visit$1(r2, {
      Field: formatNode,
      InlineFragment: formatNode
    });
    Object.defineProperty(a2, "__key", {
      value: r2.__key,
      enumerable: false
    });
    F$1.set(r2.__key, a2);
  }
  return a2;
}
function maskTypename(e2) {
  if (!e2 || "object" != typeof e2) {
    return e2;
  }
  return Object.keys(e2).reduce((function(n2, r2) {
    var t2 = e2[r2];
    if ("__typename" === r2) {
      Object.defineProperty(n2, "__typename", {
        enumerable: false,
        value: t2
      });
    } else if (Array.isArray(t2)) {
      n2[r2] = t2.map(maskTypename);
    } else if (t2 && "object" == typeof t2 && "__typename" in t2) {
      n2[r2] = maskTypename(t2);
    } else {
      n2[r2] = t2;
    }
    return n2;
  }), Array.isArray(e2) ? [] : {});
}
function withPromise(e2) {
  e2.toPromise = function() {
    return new Promise((function(n2) {
      var r2 = N$1((function(e3) {
        if (!e3.stale && !e3.hasNext) {
          Promise.resolve().then((function() {
            r2.unsubscribe();
            n2(e3);
          }));
        }
      }))(e2);
    }));
  };
  return e2;
}
function makeOperation(e2, n2, r2) {
  if (!r2) {
    r2 = n2.context;
  }
  return {
    key: n2.key,
    query: n2.query,
    variables: n2.variables,
    kind: e2,
    context: r2
  };
}
function addMetadata(e2, n2) {
  return makeOperation(e2.kind, e2, _extends$3({}, e2.context, {
    meta: _extends$3({}, e2.context.meta, n2)
  }));
}
function noop$3() {
}
function applyDefinitions(e2, r2, t2) {
  for (var a2 = 0; a2 < t2.length; a2++) {
    if (t2[a2].kind === Kind.FRAGMENT_DEFINITION) {
      var o2 = t2[a2].name.value;
      var i2 = stringifyDocument(t2[a2]);
      if (!e2.has(o2)) {
        e2.set(o2, i2);
        r2.push(t2[a2]);
      }
    } else {
      r2.push(t2[a2]);
    }
  }
}
function gql() {
  var e2 = arguments;
  var r2 = /* @__PURE__ */ new Map();
  var a2 = [];
  var o2 = [];
  var i2 = Array.isArray(arguments[0]) ? arguments[0][0] : arguments[0] || "";
  for (var u2 = 1; u2 < arguments.length; u2++) {
    var c2 = e2[u2];
    if (c2 && c2.definitions) {
      o2.push.apply(o2, c2.definitions);
    } else {
      i2 += c2;
    }
    i2 += e2[0][u2];
  }
  applyDefinitions(r2, a2, keyDocument(i2).definitions);
  applyDefinitions(r2, a2, o2);
  return keyDocument({
    kind: Kind.DOCUMENT,
    definitions: a2
  });
}
function shouldSkip(e2) {
  var n2 = e2.kind;
  return "mutation" !== n2 && "query" !== n2;
}
function cacheExchange$1(e2) {
  var n2 = e2.forward;
  var r2 = e2.client;
  e2.dispatchDebug;
  var a2 = /* @__PURE__ */ new Map();
  var i2 = /* @__PURE__ */ Object.create(null);
  function mapTypeNames(e3) {
    var n3 = makeOperation(e3.kind, e3);
    n3.query = formatDocument(e3.query);
    return n3;
  }
  function isOperationCached(e3) {
    var n3 = e3.context.requestPolicy;
    return "query" === e3.kind && "network-only" !== n3 && ("cache-only" === n3 || a2.has(e3.key));
  }
  return function(e3) {
    var u2 = share$1(e3);
    var c2 = map$1$1((function(e4) {
      var n3 = a2.get(e4.key);
      var i3 = _extends$3({}, n3, {
        operation: addMetadata(e4, {
          cacheOutcome: n3 ? "hit" : "miss"
        })
      });
      if ("cache-and-network" === e4.context.requestPolicy) {
        i3.stale = true;
        reexecuteOperation(r2, e4);
      }
      return i3;
    }))(filter$1((function(e4) {
      return !shouldSkip(e4) && isOperationCached(e4);
    }))(u2));
    var s2 = H((function(e4) {
      var n3 = e4.operation;
      if (!n3) {
        return;
      }
      var o2 = collectTypesFromResponse(e4.data).concat(n3.context.additionalTypenames || []);
      if ("mutation" === e4.operation.kind) {
        var u3 = /* @__PURE__ */ new Set();
        for (var c3 = 0; c3 < o2.length; c3++) {
          var s3 = o2[c3];
          var f2 = i2[s3] || (i2[s3] = /* @__PURE__ */ new Set());
          f2.forEach((function(e5) {
            u3.add(e5);
          }));
          f2.clear();
        }
        u3.forEach((function(e5) {
          if (a2.has(e5)) {
            n3 = a2.get(e5).operation;
            a2.delete(e5);
            reexecuteOperation(r2, n3);
          }
        }));
      } else if ("query" === n3.kind && e4.data) {
        a2.set(n3.key, e4);
        for (var p2 = 0; p2 < o2.length; p2++) {
          var l2 = o2[p2];
          (i2[l2] || (i2[l2] = /* @__PURE__ */ new Set())).add(n3.key);
        }
      }
    }))(n2(filter$1((function(e4) {
      return "query" !== e4.kind || "cache-only" !== e4.context.requestPolicy;
    }))(map$1$1((function(e4) {
      return addMetadata(e4, {
        cacheOutcome: "miss"
      });
    }))(merge$1([map$1$1(mapTypeNames)(filter$1((function(e4) {
      return !shouldSkip(e4) && !isOperationCached(e4);
    }))(u2)), filter$1((function(e4) {
      return shouldSkip(e4);
    }))(u2)])))));
    return merge$1([c2, s2]);
  };
}
function reexecuteOperation(e2, n2) {
  return e2.reexecuteOperation(makeOperation(n2.kind, n2, _extends$3({}, n2.context, {
    requestPolicy: "network-only"
  })));
}
function subscriptionExchange(e2) {
  var n2 = e2.forwardSubscription;
  var t2 = e2.enableAllOperations;
  return function(e3) {
    var a2 = e3.client;
    var i2 = e3.forward;
    function isSubscriptionOperation(e4) {
      var n3 = e4.kind;
      return "subscription" === n3 || !!t2 && ("query" === n3 || "mutation" === n3);
    }
    return function(e4) {
      var t3 = share$1(e4);
      var u2 = D$1((function(e5) {
        var i3 = e5.key;
        var u3 = filter$1((function(e6) {
          return "teardown" === e6.kind && e6.key === i3;
        }))(t3);
        return takeUntil$1(u3)((function createSubscriptionSource(e6) {
          var t4 = n2({
            key: e6.key.toString(36),
            query: print$1(e6.query),
            variables: e6.variables,
            context: _extends$3({}, e6.context)
          });
          return make$1((function(n3) {
            var r2 = n3.next;
            var o2 = n3.complete;
            var i4 = false;
            var u4;
            Promise.resolve().then((function() {
              if (i4) {
                return;
              }
              u4 = t4.subscribe({
                next: function(n4) {
                  return r2(makeResult(e6, n4));
                },
                error: function(n4) {
                  return r2(makeErrorResult(e6, n4));
                },
                complete: function() {
                  if (!i4) {
                    i4 = true;
                    if ("subscription" === e6.kind) {
                      a2.reexecuteOperation(makeOperation("teardown", e6, e6.context));
                    }
                    o2();
                  }
                }
              });
            }));
            return function() {
              i4 = true;
              if (u4) {
                u4.unsubscribe();
              }
            };
          }));
        })(e5));
      }))(filter$1(isSubscriptionOperation)(t3));
      var c2 = i2(filter$1((function(e5) {
        return !isSubscriptionOperation(e5);
      }))(t3));
      return merge$1([u2, c2]);
    };
  };
}
function dedupExchange(e2) {
  var n2 = e2.forward;
  e2.dispatchDebug;
  var t2 = /* @__PURE__ */ new Set();
  function filterIncomingOperation(e3) {
    var n3 = e3.key;
    var a2 = e3.kind;
    if ("teardown" === a2) {
      t2.delete(n3);
      return true;
    }
    if ("query" !== a2 && "subscription" !== a2) {
      return true;
    }
    var o2 = t2.has(n3);
    t2.add(n3);
    return !o2;
  }
  function afterOperationResult(e3) {
    if (!e3.hasNext) {
      t2.delete(e3.operation.key);
    }
  }
  return function(e3) {
    var r3 = filter$1(filterIncomingOperation)(e3);
    return H(afterOperationResult)(n2(r3));
  };
}
function fetchExchange(e2) {
  var n2 = e2.forward;
  e2.dispatchDebug;
  return function(e3) {
    var t2 = share$1(e3);
    var a2 = D$1((function(e4) {
      var n3 = e4.key;
      var a3 = filter$1((function(e5) {
        return "teardown" === e5.kind && e5.key === n3;
      }))(t2);
      var o3 = makeFetchBody(e4);
      var i2 = makeFetchURL(e4, o3);
      var u2 = makeFetchOptions(e4, o3);
      return H((function(n4) {
        !n4.data ? n4.error : void 0;
      }))(takeUntil$1(a3)(makeFetchSource(e4, i2, u2)));
    }))(filter$1((function(e4) {
      return "query" === e4.kind || "mutation" === e4.kind;
    }))(t2));
    var o2 = n2(filter$1((function(e4) {
      return "query" !== e4.kind && "mutation" !== e4.kind;
    }))(t2));
    return merge$1([a2, o2]);
  };
}
function fallbackExchange(e2) {
  e2.dispatchDebug;
  return function(e3) {
    return filter$1((function() {
      return false;
    }))(H((function(e4) {
      if ("teardown" !== e4.kind && false) ;
    }))(e3));
  };
}
fallbackExchange({
  dispatchDebug: noop$3
});
function composeExchanges(e2) {
  return function(n2) {
    var r2 = n2.client;
    n2.dispatchDebug;
    return e2.reduceRight((function(e3, n3) {
      return n3({
        client: r2,
        forward: e3,
        dispatchDebug: function dispatchDebug$1(e4) {
        }
      });
    }), n2.forward);
  };
}
function errorExchange(e2) {
  var n2 = e2.onError;
  return function(e3) {
    var r2 = e3.forward;
    return function(e4) {
      return H((function(e5) {
        var r3 = e5.error;
        if (r3) {
          n2(r3, e5.operation);
        }
      }))(r2(e4));
    };
  };
}
var L$1 = [dedupExchange, cacheExchange$1, fetchExchange];
var J = function Client(e2) {
  var n2 = /* @__PURE__ */ new Map();
  var r2 = /* @__PURE__ */ new Map();
  var t2 = [];
  var a2 = makeSubject$1();
  var i2 = a2.source;
  var u2 = a2.next;
  var c2 = false;
  function dispatchOperation(e3) {
    c2 = true;
    if (e3) {
      u2(e3);
    }
    while (e3 = t2.shift()) {
      u2(e3);
    }
    c2 = false;
  }
  function makeResultSource(e3) {
    var a3 = filter$1((function(n3) {
      return n3.operation.kind === e3.kind && n3.operation.key === e3.key && (!n3.operation.context._instance || n3.operation.context._instance === e3.context._instance);
    }))(y2);
    if (f2.maskTypename) {
      a3 = map$1$1((function(e4) {
        return _extends$3({}, e4, {
          data: maskTypename(e4.data)
        });
      }))(a3);
    }
    if ("mutation" === e3.kind) {
      return take$1(1)(onStart$1((function() {
        return dispatchOperation(e3);
      }))(a3));
    }
    return share$1(onEnd$1((function() {
      n2.delete(e3.key);
      r2.delete(e3.key);
      for (var a4 = t2.length - 1; a4 >= 0; a4--) {
        if (t2[a4].key === e3.key) {
          t2.splice(a4, 1);
        }
      }
      dispatchOperation(makeOperation("teardown", e3, e3.context));
    }))(H((function(r3) {
      n2.set(e3.key, r3);
    }))(K((function(n3) {
      if ("query" !== e3.kind || n3.stale) {
        return fromValue$1(n3);
      }
      return merge$1([fromValue$1(n3), map$1$1((function() {
        return _extends$3({}, n3, {
          stale: true
        });
      }))(take$1(1)(filter$1((function(n4) {
        return "query" === n4.kind && n4.key === e3.key && "cache-only" !== n4.context.requestPolicy;
      }))(i2)))]);
    }))(takeUntil$1(filter$1((function(n3) {
      return "teardown" === n3.kind && n3.key === e3.key;
    }))(i2))(a3)))));
  }
  var s2 = this instanceof Client ? this : Object.create(Client.prototype);
  var f2 = _extends$3(s2, {
    url: e2.url,
    fetchOptions: e2.fetchOptions,
    fetch: e2.fetch,
    suspense: !!e2.suspense,
    requestPolicy: e2.requestPolicy || "cache-first",
    preferGetMethod: !!e2.preferGetMethod,
    maskTypename: !!e2.maskTypename,
    operations$: i2,
    reexecuteOperation: function reexecuteOperation2(e3) {
      if ("mutation" === e3.kind || r2.has(e3.key)) {
        t2.push(e3);
        if (!c2) {
          Promise.resolve().then(dispatchOperation);
        }
      }
    },
    createOperationContext: function createOperationContext(e3) {
      if (!e3) {
        e3 = {};
      }
      return _extends$3({}, {
        _instance: void 0,
        url: f2.url,
        fetchOptions: f2.fetchOptions,
        fetch: f2.fetch,
        preferGetMethod: f2.preferGetMethod
      }, e3, {
        suspense: e3.suspense || false !== e3.suspense && f2.suspense,
        requestPolicy: e3.requestPolicy || f2.requestPolicy
      });
    },
    createRequestOperation: function createRequestOperation(e3, n3, r3) {
      getOperationType(n3.query);
      var a3 = f2.createOperationContext(r3);
      if ("mutation" === e3) {
        a3._instance = [];
      }
      return makeOperation(e3, n3, a3);
    },
    executeRequestOperation: function executeRequestOperation(e3) {
      if ("mutation" === e3.kind) {
        return makeResultSource(e3);
      }
      return make$1((function(t3) {
        var a3 = r2.get(e3.key);
        if (!a3) {
          r2.set(e3.key, a3 = makeResultSource(e3));
        }
        var i3 = "cache-and-network" === e3.context.requestPolicy || "network-only" === e3.context.requestPolicy;
        return N$1(t3.next)(onEnd$1(t3.complete)(onStart$1((function() {
          var r3 = n2.get(e3.key);
          if ("subscription" === e3.kind) {
            return dispatchOperation(e3);
          } else if (i3) {
            dispatchOperation(e3);
          }
          if (null != r3 && r3 === n2.get(e3.key)) {
            t3.next(i3 ? _extends$3({}, r3, {
              stale: true
            }) : r3);
          } else if (!i3) {
            dispatchOperation(e3);
          }
        }))(a3))).unsubscribe;
      }));
    },
    executeQuery: function executeQuery(e3, n3) {
      var r3 = f2.createRequestOperation("query", e3, n3);
      return f2.executeRequestOperation(r3);
    },
    executeSubscription: function executeSubscription(e3, n3) {
      var r3 = f2.createRequestOperation("subscription", e3, n3);
      return f2.executeRequestOperation(r3);
    },
    executeMutation: function executeMutation(e3, n3) {
      var r3 = f2.createRequestOperation("mutation", e3, n3);
      return f2.executeRequestOperation(r3);
    },
    query: function query2(e3, n3, r3) {
      if (!r3 || "boolean" != typeof r3.suspense) {
        r3 = _extends$3({}, r3, {
          suspense: false
        });
      }
      return withPromise(f2.executeQuery(createRequest(e3, n3), r3));
    },
    readQuery: function readQuery2(e3, n3, r3) {
      var t3 = null;
      N$1((function(e4) {
        t3 = e4;
      }))(f2.query(e3, n3, r3)).unsubscribe();
      return t3;
    },
    subscription: function subscription(e3, n3, r3) {
      return f2.executeSubscription(createRequest(e3, n3), r3);
    },
    mutation: function mutation(e3, n3, r3) {
      return withPromise(f2.executeMutation(createRequest(e3, n3), r3));
    }
  });
  var p2 = noop$3;
  var h2 = composeExchanges(void 0 !== e2.exchanges ? e2.exchanges : L$1);
  var y2 = share$1(h2({
    client: f2,
    dispatchDebug: p2,
    forward: fallbackExchange({
      dispatchDebug: p2
    })
  })(i2));
  publish$1(y2);
  return f2;
};
var W = J;
function _extends$2() {
  return (_extends$2 = Object.assign || function(e2) {
    for (var u2 = 1; u2 < arguments.length; u2++) {
      var n2 = arguments[u2];
      for (var r2 in n2) {
        if (Object.prototype.hasOwnProperty.call(n2, r2)) {
          e2[r2] = n2[r2];
        }
      }
    }
    return e2;
  }).apply(this, arguments);
}
function install(u2, t2) {
  var a2;
  if (!isRef(t2)) {
    a2 = ref(t2 instanceof J ? t2 : new J(t2));
  } else {
    a2 = t2;
  }
  u2.provide("$urql", a2);
}
function useClient() {
  var e2 = inject("$urql");
  return e2;
}
function unwrapPossibleProxy(e2) {
  return e2 && isRef(e2) ? e2.value : e2;
}
var y$1 = {
  flush: "pre"
};
function useQuery(e2) {
  return callUseQuery(e2);
}
function callUseQuery(e2, t2, a2) {
  if (void 0 === t2) {
    t2 = useClient();
  }
  if (void 0 === a2) {
    a2 = [];
  }
  var i2 = reactive(e2);
  var l2 = ref();
  var v2 = ref(false);
  var p2 = ref(false);
  var d2 = ref();
  var b2 = ref();
  var h2 = ref();
  var x2 = isRef(e2.pause) ? e2.pause : ref(!!e2.pause);
  var w2 = ref(createRequest(i2.query, unwrapPossibleProxy(i2.variables)));
  var P = ref();
  a2.push(watchEffect((function() {
    var e3 = createRequest(i2.query, unwrapPossibleProxy(i2.variables));
    if (w2.value.key !== e3.key) {
      w2.value = e3;
    }
  }), y$1));
  a2.push(watchEffect((function() {
    P.value = !x2.value ? t2.value.executeQuery(w2.value, _extends$2({}, {
      requestPolicy: i2.requestPolicy
    }, i2.context)) : void 0;
  }), y$1));
  var Q2 = {
    data: l2,
    stale: v2,
    error: d2,
    operation: b2,
    extensions: h2,
    fetching: p2,
    isPaused: x2,
    executeQuery: function executeQuery(e3) {
      var u2 = P.value = t2.value.executeQuery(w2.value, _extends$2({}, {
        requestPolicy: i2.requestPolicy
      }, i2.context, e3));
      return _extends$2({}, S3, {
        then: function then(e4, n2) {
          return new Promise((function(e5) {
            var n3 = false;
            var r2 = N$1((function() {
              if (!Q2.fetching.value && !Q2.stale.value) {
                if (r2) {
                  r2.unsubscribe();
                }
                n3 = true;
                e5(Q2);
              }
            }))(u2);
            if (n3) {
              r2.unsubscribe();
            }
          })).then(e4, n2);
        }
      });
    },
    pause: function pause() {
      x2.value = true;
    },
    resume: function resume() {
      x2.value = false;
    }
  };
  a2.push(watchEffect((function(e3) {
    if (P.value) {
      p2.value = true;
      v2.value = false;
      e3(N$1((function(e4) {
        l2.value = e4.data;
        v2.value = !!e4.stale;
        p2.value = false;
        d2.value = e4.error;
        b2.value = e4.operation;
        h2.value = e4.extensions;
      }))(onEnd$1((function() {
        p2.value = false;
        v2.value = false;
      }))(P.value)).unsubscribe);
    } else {
      p2.value = false;
      v2.value = false;
    }
  }), {
    flush: "sync"
  }));
  var S3 = _extends$2({}, Q2, {
    then: function then(e3, u2) {
      return new Promise((function(e4) {
        if (!P.value) {
          return e4(Q2);
        }
        var u3 = false;
        var n2 = N$1((function() {
          if (!Q2.fetching.value && !Q2.stale.value) {
            if (n2) {
              n2.unsubscribe();
            }
            u3 = true;
            e4(Q2);
          }
        }))(P.value);
        if (u3) {
          n2.unsubscribe();
        }
      })).then(e3, u2);
    }
  });
  return S3;
}
function useMutation(e2) {
  return callUseMutation(e2);
}
function callUseMutation(e2, n2) {
  if (void 0 === n2) {
    n2 = useClient();
  }
  var t2 = ref();
  var a2 = ref(false);
  var i2 = ref(false);
  var o2 = ref();
  var s2 = ref();
  var l2 = ref();
  return {
    data: t2,
    stale: a2,
    fetching: i2,
    error: o2,
    operation: s2,
    extensions: l2,
    executeMutation: function executeMutation(r2, v2) {
      i2.value = true;
      return toPromise$1(take$1(1)(n2.value.executeMutation(createRequest(e2, unwrapPossibleProxy(r2)), v2 || {}))).then((function(e3) {
        t2.value = e3.data;
        a2.value = !!e3.stale;
        i2.value = false;
        o2.value = e3.error;
        s2.value = e3.operation;
        l2.value = e3.extensions;
        return e3;
      }));
    }
  };
}
var b$2 = {
  flush: "pre"
};
function useSubscription(e2, u2) {
  return callUseSubscription(e2, u2);
}
function callUseSubscription(e2, t2, a2, i2) {
  if (void 0 === a2) {
    a2 = useClient();
  }
  if (void 0 === i2) {
    i2 = [];
  }
  var l2 = reactive(e2);
  var v2 = ref();
  var p2 = ref(false);
  var d2 = ref(false);
  var y2 = ref();
  var h2 = ref();
  var x2 = ref();
  var w2 = ref(t2);
  var P = isRef(e2.pause) ? e2.pause : ref(!!e2.pause);
  var Q2 = ref(createRequest(l2.query, unwrapPossibleProxy(l2.variables)));
  var S3 = ref();
  i2.push(watchEffect((function() {
    var e3 = createRequest(l2.query, unwrapPossibleProxy(l2.variables));
    if (Q2.value.key !== e3.key) {
      Q2.value = e3;
    }
  }), b$2));
  i2.push(watchEffect((function() {
    S3.value = !P.value ? a2.value.executeSubscription(Q2.value, _extends$2({}, l2.context)) : void 0;
  }), b$2));
  i2.push(watchEffect((function(e3) {
    if (S3.value) {
      d2.value = true;
      e3(N$1((function(e4) {
        d2.value = true;
        v2.value = void 0 !== e4.data ? "function" == typeof w2.value ? w2.value(v2.value, e4.data) : e4.data : e4.data, y2.value = e4.error;
        x2.value = e4.extensions;
        p2.value = !!e4.stale;
        h2.value = e4.operation;
      }))(onEnd$1((function() {
        d2.value = false;
      }))(S3.value)).unsubscribe);
    } else {
      d2.value = false;
    }
  }), b$2));
  var m2 = {
    data: v2,
    stale: p2,
    error: y2,
    operation: h2,
    extensions: x2,
    fetching: d2,
    isPaused: P,
    executeSubscription: function executeSubscription(e3) {
      S3.value = a2.value.executeSubscription(Q2.value, _extends$2({}, l2.context, e3));
      return m2;
    },
    pause: function pause() {
      P.value = true;
    },
    resume: function resume() {
      P.value = false;
    }
  };
  return m2;
}
const isFunction$1 = (value) => typeof value === "function";
const isString$3 = (value) => typeof value === "string";
const isNonEmptyString = (value) => isString$3(value) && value.trim().length > 0;
const isNumber$1 = (value) => typeof value === "number";
const isUndefined = (value) => typeof value === "undefined";
const isObject$4 = (value) => typeof value === "object" && value !== null;
const isJSX = (obj) => hasProp$1(obj, "tag") && isNonEmptyString(obj.tag);
const isTouchEvent = (event) => window.TouchEvent && event instanceof TouchEvent;
const isToastComponent = (obj) => hasProp$1(obj, "component") && isToastContent(obj.component);
const isVueComponent = (c2) => isFunction$1(c2) || isObject$4(c2);
const isToastContent = (obj) => (
  // Ignore undefined
  !isUndefined(obj) && // Is a string
  (isString$3(obj) || // Regular Vue component
  isVueComponent(obj) || // Nested object
  isToastComponent(obj))
);
const isDOMRect = (obj) => isObject$4(obj) && ["height", "width", "right", "left", "top", "bottom"].every((p2) => isNumber$1(obj[p2]));
const hasProp$1 = (obj, propKey) => (isObject$4(obj) || isFunction$1(obj)) && propKey in obj;
const getId = /* @__PURE__ */ ((i2) => () => i2++)(0);
function getX(event) {
  return isTouchEvent(event) ? event.targetTouches[0].clientX : event.clientX;
}
function getY(event) {
  return isTouchEvent(event) ? event.targetTouches[0].clientY : event.clientY;
}
const removeElement = (el2) => {
  if (!isUndefined(el2.remove)) {
    el2.remove();
  } else if (el2.parentNode) {
    el2.parentNode.removeChild(el2);
  }
};
const getVueComponentFromObj = (obj) => {
  if (isToastComponent(obj)) {
    return getVueComponentFromObj(obj.component);
  }
  if (isJSX(obj)) {
    return defineComponent({
      render() {
        return obj;
      }
    });
  }
  return typeof obj === "string" ? obj : toRaw(unref(obj));
};
const normalizeToastComponent = (obj) => {
  if (typeof obj === "string") {
    return obj;
  }
  const props = hasProp$1(obj, "props") && isObject$4(obj.props) ? obj.props : {};
  const listeners = hasProp$1(obj, "listeners") && isObject$4(obj.listeners) ? obj.listeners : {};
  return {
    component: getVueComponentFromObj(obj),
    props,
    listeners
  };
};
const isBrowser$1 = () => typeof window !== "undefined";
class EventBus {
  constructor() {
    this.allHandlers = {};
  }
  getHandlers(eventType) {
    return this.allHandlers[eventType] || [];
  }
  on(eventType, handler) {
    const handlers = this.getHandlers(eventType);
    handlers.push(handler);
    this.allHandlers[eventType] = handlers;
  }
  off(eventType, handler) {
    const handlers = this.getHandlers(eventType);
    handlers.splice(handlers.indexOf(handler) >>> 0, 1);
  }
  emit(eventType, event) {
    const handlers = this.getHandlers(eventType);
    handlers.forEach((handler) => handler(event));
  }
}
const isEventBusInterface = (e) => ["on", "off", "emit"].every((f2) => hasProp$1(e, f2) && isFunction$1(e[f2]));
var TYPE;
(function(TYPE2) {
  TYPE2["SUCCESS"] = "success";
  TYPE2["ERROR"] = "error";
  TYPE2["WARNING"] = "warning";
  TYPE2["INFO"] = "info";
  TYPE2["DEFAULT"] = "default";
})(TYPE || (TYPE = {}));
var POSITION;
(function(POSITION2) {
  POSITION2["TOP_LEFT"] = "top-left";
  POSITION2["TOP_CENTER"] = "top-center";
  POSITION2["TOP_RIGHT"] = "top-right";
  POSITION2["BOTTOM_LEFT"] = "bottom-left";
  POSITION2["BOTTOM_CENTER"] = "bottom-center";
  POSITION2["BOTTOM_RIGHT"] = "bottom-right";
})(POSITION || (POSITION = {}));
var EVENTS;
(function(EVENTS2) {
  EVENTS2["ADD"] = "add";
  EVENTS2["DISMISS"] = "dismiss";
  EVENTS2["UPDATE"] = "update";
  EVENTS2["CLEAR"] = "clear";
  EVENTS2["UPDATE_DEFAULTS"] = "update_defaults";
})(EVENTS || (EVENTS = {}));
const VT_NAMESPACE = "Vue-Toastification";
const COMMON = {
  type: {
    type: String,
    default: TYPE.DEFAULT
  },
  classNames: {
    type: [String, Array],
    default: () => []
  },
  trueBoolean: {
    type: Boolean,
    default: true
  }
};
const ICON = {
  type: COMMON.type,
  customIcon: {
    type: [String, Boolean, Object, Function],
    default: true
  }
};
const CLOSE_BUTTON = {
  component: {
    type: [String, Object, Function, Boolean],
    default: "button"
  },
  classNames: COMMON.classNames,
  showOnHover: {
    type: Boolean,
    default: false
  },
  ariaLabel: {
    type: String,
    default: "close"
  }
};
const PROGRESS_BAR = {
  timeout: {
    type: [Number, Boolean],
    default: 5e3
  },
  hideProgressBar: {
    type: Boolean,
    default: false
  },
  isRunning: {
    type: Boolean,
    default: false
  }
};
const TRANSITION$1 = {
  transition: {
    type: [Object, String],
    default: `${VT_NAMESPACE}__bounce`
  }
};
const CORE_TOAST = {
  position: {
    type: String,
    default: POSITION.TOP_RIGHT
  },
  draggable: COMMON.trueBoolean,
  draggablePercent: {
    type: Number,
    default: 0.6
  },
  pauseOnFocusLoss: COMMON.trueBoolean,
  pauseOnHover: COMMON.trueBoolean,
  closeOnClick: COMMON.trueBoolean,
  timeout: PROGRESS_BAR.timeout,
  hideProgressBar: PROGRESS_BAR.hideProgressBar,
  toastClassName: COMMON.classNames,
  bodyClassName: COMMON.classNames,
  icon: ICON.customIcon,
  closeButton: CLOSE_BUTTON.component,
  closeButtonClassName: CLOSE_BUTTON.classNames,
  showCloseButtonOnHover: CLOSE_BUTTON.showOnHover,
  accessibility: {
    type: Object,
    default: () => ({
      toastRole: "alert",
      closeButtonLabel: "close"
    })
  },
  rtl: {
    type: Boolean,
    default: false
  },
  eventBus: {
    type: Object,
    required: true,
    default: new EventBus()
  }
};
const TOAST = {
  id: {
    type: [String, Number],
    required: true,
    default: 0
  },
  type: COMMON.type,
  content: {
    type: [String, Object, Function],
    required: true,
    default: ""
  },
  onClick: {
    type: Function,
    default: () => {
    }
  },
  onClose: {
    type: Function,
    default: (
      /* istanbul ignore next */
      () => {
      }
    )
  }
};
const CONTAINER = {
  container: {
    type: [Object, Function],
    default: () => document.body
  },
  newestOnTop: COMMON.trueBoolean,
  maxToasts: {
    type: Number,
    default: 20
  },
  transition: TRANSITION$1.transition,
  toastDefaults: Object,
  filterBeforeCreate: {
    type: Function,
    default: (toast) => toast
  },
  filterToasts: {
    type: Function,
    default: (toasts) => toasts
  },
  containerClassName: COMMON.classNames,
  onMounted: Function
};
var PROPS = {
  CORE_TOAST,
  TOAST,
  CONTAINER,
  PROGRESS_BAR,
  ICON,
  TRANSITION: TRANSITION$1,
  CLOSE_BUTTON
};
var script$1 = defineComponent({
  name: "VtProgressBar",
  props: PROPS.PROGRESS_BAR,
  // TODO: The typescript compiler is not playing nice with emit types
  // Rollback this change once ts is able to infer emit types
  // emits: ["close-toast"],
  data() {
    return {
      hasClass: true
    };
  },
  computed: {
    style() {
      return {
        animationDuration: `${this.timeout}ms`,
        animationPlayState: this.isRunning ? "running" : "paused",
        opacity: this.hideProgressBar ? 0 : 1
      };
    },
    cpClass() {
      return this.hasClass ? `${VT_NAMESPACE}__progress-bar` : "";
    }
  },
  watch: {
    timeout() {
      this.hasClass = false;
      this.$nextTick(() => this.hasClass = true);
    }
  },
  mounted() {
    this.$el.addEventListener("animationend", this.animationEnded);
  },
  beforeUnmount() {
    this.$el.removeEventListener("animationend", this.animationEnded);
  },
  methods: {
    animationEnded() {
      this.$emit("close-toast");
    }
  }
});
function render$1b(_ctx, _cache, $props, $setup, $data, $options) {
  return openBlock(), createBlock(
    "div",
    {
      style: _ctx.style,
      class: _ctx.cpClass
    },
    null,
    6
    /* CLASS, STYLE */
  );
}
script$1.render = render$1b;
script$1.__file = "src/components/VtProgressBar.vue";
var script$1$1 = defineComponent({
  name: "VtCloseButton",
  props: PROPS.CLOSE_BUTTON,
  computed: {
    buttonComponent() {
      if (this.component !== false) {
        return getVueComponentFromObj(this.component);
      }
      return "button";
    },
    classes() {
      const classes = [`${VT_NAMESPACE}__close-button`];
      if (this.showOnHover) {
        classes.push("show-on-hover");
      }
      return classes.concat(this.classNames);
    }
  }
});
const _hoisted_1$2b = /* @__PURE__ */ createTextVNode(" × ");
function render$1$1(_ctx, _cache, $props, $setup, $data, $options) {
  return openBlock(), createBlock(resolveDynamicComponent(_ctx.buttonComponent), mergeProps({
    "aria-label": _ctx.ariaLabel,
    class: _ctx.classes
  }, _ctx.$attrs), {
    default: withCtx(() => [
      _hoisted_1$2b
    ]),
    _: 1
  }, 16, ["aria-label", "class"]);
}
script$1$1.render = render$1$1;
script$1$1.__file = "src/components/VtCloseButton.vue";
var script$2 = {};
const _hoisted_1$1$2 = {
  "aria-hidden": "true",
  focusable: "false",
  "data-prefix": "fas",
  "data-icon": "check-circle",
  class: "svg-inline--fa fa-check-circle fa-w-16",
  role: "img",
  xmlns: "http://www.w3.org/2000/svg",
  viewBox: "0 0 512 512"
};
const _hoisted_2$1X = /* @__PURE__ */ createVNode(
  "path",
  {
    fill: "currentColor",
    d: "M504 256c0 136.967-111.033 248-248 248S8 392.967 8 256 119.033 8 256 8s248 111.033 248 248zM227.314 387.314l184-184c6.248-6.248 6.248-16.379 0-22.627l-22.627-22.627c-6.248-6.249-16.379-6.249-22.628 0L216 308.118l-70.059-70.059c-6.248-6.248-16.379-6.248-22.628 0l-22.627 22.627c-6.248 6.248-6.248 16.379 0 22.627l104 104c6.249 6.249 16.379 6.249 22.628.001z"
  },
  null,
  -1
  /* HOISTED */
);
function render$2$1(_ctx, _cache, $props, $setup, $data, $options) {
  return openBlock(), createBlock("svg", _hoisted_1$1$2, [
    _hoisted_2$1X
  ]);
}
script$2.render = render$2$1;
script$2.__file = "src/components/icons/VtSuccessIcon.vue";
var script$3 = {};
const _hoisted_1$2$2 = {
  "aria-hidden": "true",
  focusable: "false",
  "data-prefix": "fas",
  "data-icon": "info-circle",
  class: "svg-inline--fa fa-info-circle fa-w-16",
  role: "img",
  xmlns: "http://www.w3.org/2000/svg",
  viewBox: "0 0 512 512"
};
const _hoisted_2$1$2 = /* @__PURE__ */ createVNode(
  "path",
  {
    fill: "currentColor",
    d: "M256 8C119.043 8 8 119.083 8 256c0 136.997 111.043 248 248 248s248-111.003 248-248C504 119.083 392.957 8 256 8zm0 110c23.196 0 42 18.804 42 42s-18.804 42-42 42-42-18.804-42-42 18.804-42 42-42zm56 254c0 6.627-5.373 12-12 12h-88c-6.627 0-12-5.373-12-12v-24c0-6.627 5.373-12 12-12h12v-64h-12c-6.627 0-12-5.373-12-12v-24c0-6.627 5.373-12 12-12h64c6.627 0 12 5.373 12 12v100h12c6.627 0 12 5.373 12 12v24z"
  },
  null,
  -1
  /* HOISTED */
);
function render$3$1(_ctx, _cache, $props, $setup, $data, $options) {
  return openBlock(), createBlock("svg", _hoisted_1$2$2, [
    _hoisted_2$1$2
  ]);
}
script$3.render = render$3$1;
script$3.__file = "src/components/icons/VtInfoIcon.vue";
var script$4 = {};
const _hoisted_1$3$1 = {
  "aria-hidden": "true",
  focusable: "false",
  "data-prefix": "fas",
  "data-icon": "exclamation-circle",
  class: "svg-inline--fa fa-exclamation-circle fa-w-16",
  role: "img",
  xmlns: "http://www.w3.org/2000/svg",
  viewBox: "0 0 512 512"
};
const _hoisted_2$2$1 = /* @__PURE__ */ createVNode(
  "path",
  {
    fill: "currentColor",
    d: "M504 256c0 136.997-111.043 248-248 248S8 392.997 8 256C8 119.083 119.043 8 256 8s248 111.083 248 248zm-248 50c-25.405 0-46 20.595-46 46s20.595 46 46 46 46-20.595 46-46-20.595-46-46-46zm-43.673-165.346l7.418 136c.347 6.364 5.609 11.346 11.982 11.346h48.546c6.373 0 11.635-4.982 11.982-11.346l7.418-136c.375-6.874-5.098-12.654-11.982-12.654h-63.383c-6.884 0-12.356 5.78-11.981 12.654z"
  },
  null,
  -1
  /* HOISTED */
);
function render$4$1(_ctx, _cache, $props, $setup, $data, $options) {
  return openBlock(), createBlock("svg", _hoisted_1$3$1, [
    _hoisted_2$2$1
  ]);
}
script$4.render = render$4$1;
script$4.__file = "src/components/icons/VtWarningIcon.vue";
var script$5 = {};
const _hoisted_1$4$1 = {
  "aria-hidden": "true",
  focusable: "false",
  "data-prefix": "fas",
  "data-icon": "exclamation-triangle",
  class: "svg-inline--fa fa-exclamation-triangle fa-w-18",
  role: "img",
  xmlns: "http://www.w3.org/2000/svg",
  viewBox: "0 0 576 512"
};
const _hoisted_2$3$1 = /* @__PURE__ */ createVNode(
  "path",
  {
    fill: "currentColor",
    d: "M569.517 440.013C587.975 472.007 564.806 512 527.94 512H48.054c-36.937 0-59.999-40.055-41.577-71.987L246.423 23.985c18.467-32.009 64.72-31.951 83.154 0l239.94 416.028zM288 354c-25.405 0-46 20.595-46 46s20.595 46 46 46 46-20.595 46-46-20.595-46-46-46zm-43.673-165.346l7.418 136c.347 6.364 5.609 11.346 11.982 11.346h48.546c6.373 0 11.635-4.982 11.982-11.346l7.418-136c.375-6.874-5.098-12.654-11.982-12.654h-63.383c-6.884 0-12.356 5.78-11.981 12.654z"
  },
  null,
  -1
  /* HOISTED */
);
function render$5$1(_ctx, _cache, $props, $setup, $data, $options) {
  return openBlock(), createBlock("svg", _hoisted_1$4$1, [
    _hoisted_2$3$1
  ]);
}
script$5.render = render$5$1;
script$5.__file = "src/components/icons/VtErrorIcon.vue";
var script$6 = defineComponent({
  name: "VtIcon",
  props: PROPS.ICON,
  computed: {
    customIconChildren() {
      return hasProp$1(this.customIcon, "iconChildren") ? this.trimValue(this.customIcon.iconChildren) : "";
    },
    customIconClass() {
      if (isString$3(this.customIcon)) {
        return this.trimValue(this.customIcon);
      } else if (hasProp$1(this.customIcon, "iconClass")) {
        return this.trimValue(this.customIcon.iconClass);
      }
      return "";
    },
    customIconTag() {
      if (hasProp$1(this.customIcon, "iconTag")) {
        return this.trimValue(this.customIcon.iconTag, "i");
      }
      return "i";
    },
    hasCustomIcon() {
      return this.customIconClass.length > 0;
    },
    component() {
      if (this.hasCustomIcon) {
        return this.customIconTag;
      }
      if (isToastContent(this.customIcon)) {
        return getVueComponentFromObj(this.customIcon);
      }
      return this.iconTypeComponent;
    },
    iconTypeComponent() {
      const types = {
        [TYPE.DEFAULT]: script$3,
        [TYPE.INFO]: script$3,
        [TYPE.SUCCESS]: script$2,
        [TYPE.ERROR]: script$5,
        [TYPE.WARNING]: script$4
      };
      return types[this.type];
    },
    iconClasses() {
      const classes = [`${VT_NAMESPACE}__icon`];
      if (this.hasCustomIcon) {
        return classes.concat(this.customIconClass);
      }
      return classes;
    }
  },
  methods: {
    trimValue(value, empty2 = "") {
      return isNonEmptyString(value) ? value.trim() : empty2;
    }
  }
});
function render$6$1(_ctx, _cache, $props, $setup, $data, $options) {
  return openBlock(), createBlock(resolveDynamicComponent(_ctx.component), { class: _ctx.iconClasses }, {
    default: withCtx(() => [
      createTextVNode(
        toDisplayString$1(_ctx.customIconChildren),
        1
        /* TEXT */
      )
    ]),
    _: 1
  }, 8, ["class"]);
}
script$6.render = render$6$1;
script$6.__file = "src/components/VtIcon.vue";
var script$7 = defineComponent({
  name: "VtToast",
  components: {
    ProgressBar: script$1,
    CloseButton: script$1$1,
    Icon: script$6
  },
  inheritAttrs: false,
  props: Object.assign({}, PROPS.CORE_TOAST, PROPS.TOAST),
  data() {
    const data = {
      isRunning: true,
      disableTransitions: false,
      beingDragged: false,
      dragStart: 0,
      dragPos: {
        x: 0,
        y: 0
      },
      dragRect: {}
    };
    return data;
  },
  computed: {
    classes() {
      const classes = [`${VT_NAMESPACE}__toast`, `${VT_NAMESPACE}__toast--${this.type}`, `${this.position}`].concat(this.toastClassName);
      if (this.disableTransitions) {
        classes.push("disable-transition");
      }
      if (this.rtl) {
        classes.push(`${VT_NAMESPACE}__toast--rtl`);
      }
      return classes;
    },
    bodyClasses() {
      const classes = [`${VT_NAMESPACE}__toast-${isString$3(this.content) ? "body" : "component-body"}`].concat(this.bodyClassName);
      return classes;
    },
    /* istanbul ignore next */
    draggableStyle() {
      if (this.dragStart === this.dragPos.x) {
        return {};
      } else if (this.beingDragged) {
        return {
          transform: `translateX(${this.dragDelta}px)`,
          opacity: 1 - Math.abs(this.dragDelta / this.removalDistance)
        };
      } else {
        return {
          transition: "transform 0.2s, opacity 0.2s",
          transform: "translateX(0)",
          opacity: 1
        };
      }
    },
    dragDelta() {
      return this.beingDragged ? this.dragPos.x - this.dragStart : 0;
    },
    removalDistance() {
      if (isDOMRect(this.dragRect)) {
        return (this.dragRect.right - this.dragRect.left) * this.draggablePercent;
      }
      return 0;
    }
  },
  mounted() {
    if (this.draggable) {
      this.draggableSetup();
    }
    if (this.pauseOnFocusLoss) {
      this.focusSetup();
    }
  },
  beforeUnmount() {
    if (this.draggable) {
      this.draggableCleanup();
    }
    if (this.pauseOnFocusLoss) {
      this.focusCleanup();
    }
  },
  methods: {
    getVueComponentFromObj,
    closeToast() {
      this.eventBus.emit(EVENTS.DISMISS, this.id);
    },
    clickHandler() {
      if (this.onClick) {
        this.onClick(this.closeToast);
      }
      if (this.closeOnClick) {
        if (!this.beingDragged || this.dragStart === this.dragPos.x) {
          this.closeToast();
        }
      }
    },
    timeoutHandler() {
      this.closeToast();
    },
    hoverPause() {
      if (this.pauseOnHover) {
        this.isRunning = false;
      }
    },
    hoverPlay() {
      if (this.pauseOnHover) {
        this.isRunning = true;
      }
    },
    focusPause() {
      this.isRunning = false;
    },
    focusPlay() {
      this.isRunning = true;
    },
    focusSetup() {
      addEventListener("blur", this.focusPause);
      addEventListener("focus", this.focusPlay);
    },
    focusCleanup() {
      removeEventListener("blur", this.focusPause);
      removeEventListener("focus", this.focusPlay);
    },
    draggableSetup() {
      const element = this.$el;
      element.addEventListener("touchstart", this.onDragStart, {
        passive: true
      });
      element.addEventListener("mousedown", this.onDragStart);
      addEventListener("touchmove", this.onDragMove, {
        passive: false
      });
      addEventListener("mousemove", this.onDragMove);
      addEventListener("touchend", this.onDragEnd);
      addEventListener("mouseup", this.onDragEnd);
    },
    draggableCleanup() {
      const element = this.$el;
      element.removeEventListener("touchstart", this.onDragStart);
      element.removeEventListener("mousedown", this.onDragStart);
      removeEventListener("touchmove", this.onDragMove);
      removeEventListener("mousemove", this.onDragMove);
      removeEventListener("touchend", this.onDragEnd);
      removeEventListener("mouseup", this.onDragEnd);
    },
    onDragStart(event) {
      this.beingDragged = true;
      this.dragPos = {
        x: getX(event),
        y: getY(event)
      };
      this.dragStart = getX(event);
      this.dragRect = this.$el.getBoundingClientRect();
    },
    onDragMove(event) {
      if (this.beingDragged) {
        event.preventDefault();
        if (this.isRunning) {
          this.isRunning = false;
        }
        this.dragPos = {
          x: getX(event),
          y: getY(event)
        };
      }
    },
    onDragEnd() {
      if (this.beingDragged) {
        if (Math.abs(this.dragDelta) >= this.removalDistance) {
          this.disableTransitions = true;
          this.$nextTick(() => this.closeToast());
        } else {
          setTimeout(() => {
            this.beingDragged = false;
            if (isDOMRect(this.dragRect) && this.pauseOnHover && this.dragRect.bottom >= this.dragPos.y && this.dragPos.y >= this.dragRect.top && this.dragRect.left <= this.dragPos.x && this.dragPos.x <= this.dragRect.right) {
              this.isRunning = false;
            } else {
              this.isRunning = true;
            }
          });
        }
      }
    }
  }
});
function render$7$1(_ctx, _cache, $props, $setup, $data, $options) {
  const _component_Icon = resolveComponent("Icon");
  const _component_CloseButton = resolveComponent("CloseButton");
  const _component_ProgressBar = resolveComponent("ProgressBar");
  return openBlock(), createBlock(
    "div",
    {
      class: _ctx.classes,
      style: _ctx.draggableStyle,
      onClick: _cache[1] || (_cache[1] = (...args) => _ctx.clickHandler(...args)),
      onMouseenter: _cache[2] || (_cache[2] = (...args) => _ctx.hoverPause(...args)),
      onMouseleave: _cache[3] || (_cache[3] = (...args) => _ctx.hoverPlay(...args))
    },
    [
      _ctx.icon ? (openBlock(), createBlock(_component_Icon, {
        key: 0,
        "custom-icon": _ctx.icon,
        type: _ctx.type
      }, null, 8, ["custom-icon", "type"])) : createCommentVNode("v-if", true),
      createVNode("div", {
        role: _ctx.accessibility.toastRole || "alert",
        class: _ctx.bodyClasses
      }, [
        typeof _ctx.content === "string" ? (openBlock(), createBlock(
          Fragment,
          { key: 0 },
          [
            createTextVNode(
              toDisplayString$1(_ctx.content),
              1
              /* TEXT */
            )
          ],
          64
          /* STABLE_FRAGMENT */
        )) : (openBlock(), createBlock(resolveDynamicComponent(_ctx.getVueComponentFromObj(_ctx.content)), mergeProps({
          key: 1,
          "toast-id": _ctx.id
        }, _ctx.content.props, toHandlers(_ctx.content.listeners), { onCloseToast: _ctx.closeToast }), null, 16, ["toast-id", "onCloseToast"]))
      ], 10, ["role"]),
      !!_ctx.closeButton ? (openBlock(), createBlock(_component_CloseButton, {
        key: 1,
        component: _ctx.closeButton,
        "class-names": _ctx.closeButtonClassName,
        "show-on-hover": _ctx.showCloseButtonOnHover,
        "aria-label": _ctx.accessibility.closeButtonLabel,
        onClick: withModifiers(_ctx.closeToast, ["stop"])
      }, null, 8, ["component", "class-names", "show-on-hover", "aria-label", "onClick"])) : createCommentVNode("v-if", true),
      _ctx.timeout ? (openBlock(), createBlock(_component_ProgressBar, {
        key: 2,
        "is-running": _ctx.isRunning,
        "hide-progress-bar": _ctx.hideProgressBar,
        timeout: _ctx.timeout,
        onCloseToast: _ctx.timeoutHandler
      }, null, 8, ["is-running", "hide-progress-bar", "timeout", "onCloseToast"])) : createCommentVNode("v-if", true)
    ],
    38
    /* CLASS, STYLE, HYDRATE_EVENTS */
  );
}
script$7.render = render$7$1;
script$7.__file = "src/components/VtToast.vue";
var script$8 = defineComponent({
  name: "VtTransition",
  props: PROPS.TRANSITION,
  emits: ["leave"],
  methods: {
    leave(el2) {
      el2.style.left = el2.offsetLeft + "px";
      el2.style.top = el2.offsetTop + "px";
      el2.style.width = getComputedStyle(el2).width;
      el2.style.position = "absolute";
    }
  }
});
function render$8$1(_ctx, _cache, $props, $setup, $data, $options) {
  return openBlock(), createBlock(TransitionGroup, {
    tag: "div",
    "enter-active-class": _ctx.transition.enter ? _ctx.transition.enter : `${_ctx.transition}-enter-active`,
    "move-class": _ctx.transition.move ? _ctx.transition.move : `${_ctx.transition}-move`,
    "leave-active-class": _ctx.transition.leave ? _ctx.transition.leave : `${_ctx.transition}-leave-active`,
    onLeave: _ctx.leave
  }, {
    default: withCtx(() => [
      renderSlot(_ctx.$slots, "default")
    ]),
    _: 3
  }, 8, ["enter-active-class", "move-class", "leave-active-class", "onLeave"]);
}
script$8.render = render$8$1;
script$8.__file = "src/components/VtTransition.vue";
var script$9 = defineComponent({
  name: "VueToastification",
  components: {
    Toast: script$7,
    VtTransition: script$8
  },
  props: Object.assign({}, PROPS.CORE_TOAST, PROPS.CONTAINER, PROPS.TRANSITION),
  data() {
    const data = {
      count: 0,
      positions: Object.values(POSITION),
      toasts: {},
      defaults: {}
    };
    return data;
  },
  computed: {
    toastArray() {
      return Object.values(this.toasts);
    },
    filteredToasts() {
      return this.defaults.filterToasts(this.toastArray);
    }
  },
  beforeMount() {
    const events = this.eventBus;
    events.on(EVENTS.ADD, this.addToast);
    events.on(EVENTS.CLEAR, this.clearToasts);
    events.on(EVENTS.DISMISS, this.dismissToast);
    events.on(EVENTS.UPDATE, this.updateToast);
    events.on(EVENTS.UPDATE_DEFAULTS, this.updateDefaults);
    this.defaults = this.$props;
  },
  mounted() {
    this.setup(this.container);
  },
  methods: {
    async setup(container) {
      if (isFunction$1(container)) {
        container = await container();
      }
      removeElement(this.$el);
      container.appendChild(this.$el);
    },
    setToast(props) {
      if (!isUndefined(props.id)) {
        this.toasts[props.id] = props;
      }
    },
    addToast(params) {
      params.content = normalizeToastComponent(params.content);
      const props = Object.assign({}, this.defaults, params.type && this.defaults.toastDefaults && this.defaults.toastDefaults[params.type], params);
      const toast = this.defaults.filterBeforeCreate(props, this.toastArray);
      toast && this.setToast(toast);
    },
    dismissToast(id2) {
      const toast = this.toasts[id2];
      if (!isUndefined(toast) && !isUndefined(toast.onClose)) {
        toast.onClose();
      }
      delete this.toasts[id2];
    },
    clearToasts() {
      Object.keys(this.toasts).forEach((id2) => {
        this.dismissToast(id2);
      });
    },
    getPositionToasts(position) {
      const toasts = this.filteredToasts.filter((toast) => toast.position === position).slice(0, this.defaults.maxToasts);
      return this.defaults.newestOnTop ? toasts.reverse() : toasts;
    },
    updateDefaults(update) {
      if (!isUndefined(update.container)) {
        this.setup(update.container);
      }
      this.defaults = Object.assign({}, this.defaults, update);
    },
    updateToast({
      id: id2,
      options: options2,
      create: create2
    }) {
      if (this.toasts[id2]) {
        if (options2.timeout && options2.timeout === this.toasts[id2].timeout) {
          options2.timeout++;
        }
        this.setToast(Object.assign({}, this.toasts[id2], options2));
      } else if (create2) {
        this.addToast(Object.assign({}, {
          id: id2
        }, options2));
      }
    },
    getClasses(position) {
      const classes = [`${VT_NAMESPACE}__container`, position];
      return classes.concat(this.defaults.containerClassName);
    }
  }
});
function render$9$1(_ctx, _cache, $props, $setup, $data, $options) {
  const _component_Toast = resolveComponent("Toast");
  const _component_VtTransition = resolveComponent("VtTransition");
  return openBlock(), createBlock("div", null, [
    (openBlock(true), createBlock(
      Fragment,
      null,
      renderList(_ctx.positions, (pos) => {
        return openBlock(), createBlock("div", { key: pos }, [
          createVNode(_component_VtTransition, {
            transition: _ctx.defaults.transition,
            class: _ctx.getClasses(pos)
          }, {
            default: withCtx(() => [
              (openBlock(true), createBlock(
                Fragment,
                null,
                renderList(_ctx.getPositionToasts(pos), (toast) => {
                  return openBlock(), createBlock(
                    _component_Toast,
                    mergeProps({
                      key: toast.id
                    }, toast),
                    null,
                    16
                    /* FULL_PROPS */
                  );
                }),
                128
                /* KEYED_FRAGMENT */
              ))
            ]),
            _: 2
          }, 1032, ["transition", "class"])
        ]);
      }),
      128
      /* KEYED_FRAGMENT */
    ))
  ]);
}
script$9.render = render$9$1;
script$9.__file = "src/components/VtToastContainer.vue";
const buildInterface = (globalOptions = {}, mountContainer = true) => {
  const events = globalOptions.eventBus = globalOptions.eventBus || new EventBus();
  if (mountContainer) {
    nextTick(() => {
      const app2 = createApp(script$9, Object.assign({}, globalOptions));
      const component = app2.mount(document.createElement("div"));
      const onMounted2 = globalOptions.onMounted;
      if (!isUndefined(onMounted2)) {
        onMounted2(component, app2);
      }
    });
  }
  const toast = (content, options2) => {
    const props = Object.assign({}, {
      id: getId(),
      type: TYPE.DEFAULT
    }, options2, {
      content
    });
    events.emit(EVENTS.ADD, props);
    return props.id;
  };
  toast.clear = () => events.emit(EVENTS.CLEAR, void 0);
  toast.updateDefaults = (update) => {
    events.emit(EVENTS.UPDATE_DEFAULTS, update);
  };
  toast.dismiss = (id2) => {
    events.emit(EVENTS.DISMISS, id2);
  };
  function updateToast(id2, {
    content,
    options: options2
  }, create2 = false) {
    const opt = Object.assign({}, options2, {
      content
    });
    events.emit(EVENTS.UPDATE, {
      id: id2,
      options: opt,
      create: create2
    });
  }
  toast.update = updateToast;
  toast.success = (content, options2) => toast(content, Object.assign({}, options2, {
    type: TYPE.SUCCESS
  }));
  toast.info = (content, options2) => toast(content, Object.assign({}, options2, {
    type: TYPE.INFO
  }));
  toast.error = (content, options2) => toast(content, Object.assign({}, options2, {
    type: TYPE.ERROR
  }));
  toast.warning = (content, options2) => toast(content, Object.assign({}, options2, {
    type: TYPE.WARNING
  }));
  return toast;
};
const createMockToastInterface = () => {
  const toast = () => console.warn("[Vue Toastification] This plugin does not support SSR!");
  return new Proxy(toast, {
    get() {
      return toast;
    }
  });
};
function createToastInterface(optionsOrEventBus) {
  if (!isBrowser$1()) {
    return createMockToastInterface();
  }
  if (isEventBusInterface(optionsOrEventBus)) {
    return buildInterface({
      eventBus: optionsOrEventBus
    }, false);
  }
  return buildInterface(optionsOrEventBus, true);
}
const toastInjectionKey = Symbol("VueToastification");
const globalEventBus = new EventBus();
const VueToastificationPlugin = (App, options2) => {
  const inter = createToastInterface(Object.assign({
    eventBus: globalEventBus
  }, options2));
  App.provide(toastInjectionKey, inter);
};
const useToast = (eventBus) => {
  const toast = getCurrentInstance() ? inject(toastInjectionKey) : void 0;
  return toast ? toast : createToastInterface(globalEventBus);
};
const _hoisted_1$2a = {
  style: { "min-width": "16px", "min-height": "16px" },
  width: "16",
  height: "16",
  viewBox: "0 0 16 16",
  fill: "none",
  xmlns: "http://www.w3.org/2000/svg"
};
const _hoisted_2$1W = /* @__PURE__ */ createBaseVNode("path", {
  d: "M9 12L13 8M13 8L9 4M13 8L3 8",
  stroke: "currentColor",
  class: "icon-dark",
  "stroke-width": "2",
  "stroke-linecap": "round",
  "stroke-linejoin": "round"
}, null, -1);
const _hoisted_3$1F = [
  _hoisted_2$1W
];
function render$1a(_ctx, _cache) {
  return openBlock(), createElementBlock("svg", _hoisted_1$2a, _hoisted_3$1F);
}
const __unplugin_components_0$m = { name: "cy-arrow-right_x16", render: render$1a };
const _hoisted_1$29 = {
  key: 0,
  class: "absolute top-0 right-0 text-teal-600 ribbon",
  "aria-hidden": "true"
};
const _hoisted_2$1V = ["disabled"];
const _hoisted_3$1E = { class: "tracking-tight text-gray-600 text-[14px] leading-[20px]" };
const _sfc_main$1l = /* @__PURE__ */ defineComponent({
  __name: "Card",
  props: {
    title: null,
    description: null,
    icon: null,
    hoverIcon: { default: void 0 },
    variant: null,
    iconSize: null,
    disabled: { type: Boolean, default: false },
    badgeText: { default: "" }
  },
  emits: ["click"],
  setup(__props, { emit: emits }) {
    const props = __props;
    const classMap = {
      indigo: "icon-dark-indigo-400 icon-light-indigo-100 icon-light-secondary-jade-200 icon-dark-secondary-jade-400",
      jade: "icon-dark-jade-400 icon-light-jade-100"
    };
    const iconDimensions = computed(() => {
      return props.iconSize === 48 ? `w-[48px] h-[48px]` : `w-[64px] h-[64px]`;
    });
    const classes = computed(() => {
      const iconMargin = props.iconSize === 64 ? "mb-[32px]" : "mb-[8px]";
      return [iconDimensions.value, iconMargin];
    });
    const iconClass = computed(() => {
      const colorClass = props.disabled ? "icon-dark-gray-600 icon-light-gray-100 icon-dark-secondary-gray-600 icon-light-secondary-gray-300" : classMap[props.variant];
      return [iconDimensions.value, colorClass].join(" ");
    });
    return (_ctx, _cache) => {
      return openBlock(), createElementBlock("div", {
        class: normalizeClass(["relative block h-auto text-center border border-gray-100 rounded outline-none group children:hyphens-manual", {
          "bg-gray-50 cursor-default": __props.disabled,
          "cursor-pointer focus-within-default hocus-default": !__props.disabled
        }]),
        "data-cy": "card",
        onClick: _cache[0] || (_cache[0] = ($event) => !__props.disabled && emits("click"))
      }, [
        __props.badgeText ? (openBlock(), createElementBlock("div", _hoisted_1$29, toDisplayString$1(__props.badgeText), 1)) : createCommentVNode("", true),
        createBaseVNode("div", {
          class: normalizeClass(["mx-auto children:transition-all children:duration-300", unref(classes)])
        }, [
          __props.hoverIcon ? (openBlock(), createBlock(resolveDynamicComponent(__props.hoverIcon), {
            key: 0,
            class: normalizeClass(["absolute opacity-0", [unref(iconClass), { "group-hover:opacity-100 group-focus:opacity-100": !__props.disabled }]]),
            "data-cy": "card-icon"
          }, null, 8, ["class"])) : createCommentVNode("", true),
          (openBlock(), createBlock(resolveDynamicComponent(__props.icon), {
            class: normalizeClass([
              "opacity-100",
              [
                __props.hoverIcon && !__props.disabled ? "group-hover:opacity-0" : void 0,
                unref(iconClass)
              ]
            ]),
            "data-cy": "card-icon"
          }, null, 8, ["class"]))
        ], 2),
        createBaseVNode("button", {
          class: normalizeClass(["font-medium mx-[8px] mb-[8px] text-[18px] leading-[24px] focus:outline-transparent", {
            "text-gray-700 cursor-default": __props.disabled,
            "text-indigo-500": !__props.disabled
          }]),
          disabled: __props.disabled
        }, toDisplayString$1(__props.title), 11, _hoisted_2$1V),
        createBaseVNode("p", _hoisted_3$1E, [
          renderSlot(_ctx.$slots, "default", {}, () => [
            createTextVNode(toDisplayString$1(__props.description), 1)
          ], true)
        ]),
        renderSlot(_ctx.$slots, "footer", {}, void 0, true)
      ], 2);
    };
  }
});
const _export_sfc$1 = (sfc, props) => {
  const target2 = sfc.__vccOpts || sfc;
  for (const [key, val] of props) {
    target2[key] = val;
  }
  return target2;
};
const Card = /* @__PURE__ */ _export_sfc$1(_sfc_main$1l, [["__scopeId", "data-v-b9303bc9"]]);
const _hoisted_1$28 = {
  style: { "min-width": "16px", "min-height": "16px" },
  width: "16",
  height: "16",
  fill: "none",
  xmlns: "http://www.w3.org/2000/svg"
};
const _hoisted_2$1U = /* @__PURE__ */ createBaseVNode("circle", {
  cx: "8",
  cy: "8",
  r: "3",
  "stroke-linecap": "round",
  "stroke-linejoin": "round",
  fill: "#F3F4FA",
  stroke: "#D0D2E0",
  class: "icon-dark-stroke icon-light-fill"
}, null, -1);
const _hoisted_3$1D = [
  _hoisted_2$1U
];
function render$19(_ctx, _cache) {
  return openBlock(), createElementBlock("svg", _hoisted_1$28, _hoisted_3$1D);
}
const __unplugin_components_0$l = { name: "cy-grommet_x16", render: render$19 };
const _sfc_main$1k = /* @__PURE__ */ defineComponent({
  __name: "StatusBadge",
  props: {
    status: { type: Boolean },
    title: null
  },
  setup(__props) {
    const props = __props;
    return (_ctx, _cache) => {
      const _component_i_cy_grommet_x16 = __unplugin_components_0$l;
      return openBlock(), createElementBlock("div", {
        class: normalizeClass(["border rounded-full font-medium p-[5px] pr-[16px] text-[14px] leading-[20px] inline-flex items-center relative", { "text-jade-500": props.status, "text-gray-600": !props.status }])
      }, [
        createVNode(_component_i_cy_grommet_x16, {
          class: normalizeClass(["h-[16px] mr-[4px] ml-[4px] w-[16px]", { "icon-light-jade-400 icon-dark-jade-400": props.status, "icon-dark-gray-200 icon-dark-gray-50": !props.status }])
        }, null, 8, ["class"]),
        createTextVNode(" " + toDisplayString$1(props.title), 1)
      ], 2);
    };
  }
});
const _hoisted_1$27 = {
  style: { "min-width": "64px", "min-height": "64px" },
  width: "64",
  height: "64",
  viewBox: "0 0 64 64",
  fill: "none",
  xmlns: "http://www.w3.org/2000/svg"
};
const _hoisted_2$1T = /* @__PURE__ */ createBaseVNode("path", {
  d: "M55.5 55C56.3284 55 57 54.3284 57 53.5V19.5C57 18.6716 56.3284 18 55.5 18H48V44.5C48 45.3285 47.3284 46 46.5 46H16V53.5C16 54.3284 16.6716 55 17.5 55H55.5Z",
  fill: "#69D3A7",
  class: "icon-light"
}, null, -1);
const _hoisted_3$1C = /* @__PURE__ */ createBaseVNode("path", {
  d: "M48 10.5C48 9.67157 47.3284 9 46.5 9H8.5C7.67157 9 7 9.67157 7 10.5V18H48V10.5Z",
  fill: "#69D3A7",
  class: "icon-light"
}, null, -1);
const _hoisted_4$$ = /* @__PURE__ */ createBaseVNode("path", {
  d: "M7 18V44.5C7 45.3285 7.67157 46 8.5 46H16M7 18V10.5C7 9.67157 7.67157 9 8.5 9H46.5C47.3284 9 48 9.67157 48 10.5V18M7 18H48M16 46H46.5C47.3284 46 48 45.3285 48 44.5V18M16 46V53.5C16 54.3284 16.6716 55 17.5 55H55.5C56.3284 55 57 54.3284 57 53.5V19.5C57 18.6716 56.3284 18 55.5 18H48M43 13.5H25M11.9987 13.5H12.0133M16 13.5H16.0133M20 13.5H20.0133",
  stroke: "#007780",
  "stroke-width": "2",
  "stroke-linecap": "round",
  "stroke-linejoin": "round",
  class: "icon-dark"
}, null, -1);
const _hoisted_5$C = [
  _hoisted_2$1T,
  _hoisted_3$1C,
  _hoisted_4$$
];
function render$18(_ctx, _cache) {
  return openBlock(), createElementBlock("svg", _hoisted_1$27, _hoisted_5$C);
}
const IconE2E = { name: "cy-testing-type-e2e_x64", render: render$18 };
const _hoisted_1$26 = {
  style: { "min-width": "64px", "min-height": "64px" },
  width: "64",
  height: "64",
  viewBox: "0 0 64 64",
  fill: "none",
  xmlns: "http://www.w3.org/2000/svg"
};
const _hoisted_2$1S = /* @__PURE__ */ createBaseVNode("path", {
  d: "M9 8C7.34315 8 6 9.34315 6 11V44C6 45.6569 7.34314 47 9 47H46C47.6569 47 49 45.6569 49 44V11C49 9.34315 47.6569 8 46 8H9Z",
  fill: "#69D3A7",
  class: "icon-light"
}, null, -1);
const _hoisted_3$1B = /* @__PURE__ */ createBaseVNode("path", {
  d: "M55 56C56.6569 56 58 54.6569 58 53V20C58 18.3431 56.6569 17 55 17H49V44C49 45.6569 47.6569 47 46 47H15V53C15 54.6569 16.3431 56 18 56H55Z",
  fill: "#00595D",
  class: "icon-dark"
}, null, -1);
const _hoisted_4$_ = /* @__PURE__ */ createBaseVNode("path", {
  "fill-rule": "evenodd",
  "clip-rule": "evenodd",
  d: "M49 11C49 9.34315 47.6569 8 46 8H9C7.34315 8 6 9.34315 6 11V17H49V11ZM44 11.5C44.5523 11.5 45 11.9477 45 12.5C45 13.0523 44.5523 13.5 44 13.5H25C24.4477 13.5 24 13.0523 24 12.5C24 11.9477 24.4477 11.5 25 11.5H44ZM11 13.5C10.4477 13.5 10 13.0523 10 12.5C10 11.9477 10.4477 11.5 11 11.5C11.5523 11.5 12 11.9477 12 12.5C12 13.0523 11.5523 13.5 11 13.5ZM15 11.5C14.4477 11.5 14 11.9477 14 12.5C14 13.0523 14.4477 13.5 15 13.5C15.5523 13.5 16 13.0523 16 12.5C16 11.9477 15.5523 11.5 15 11.5ZM19 13.5C18.4477 13.5 18 13.0523 18 12.5C18 11.9477 18.4477 11.5 19 11.5C19.5523 11.5 20 11.9477 20 12.5C20 13.0523 19.5523 13.5 19 13.5Z",
  fill: "#00595D",
  class: "icon-dark"
}, null, -1);
const _hoisted_5$B = [
  _hoisted_2$1S,
  _hoisted_3$1B,
  _hoisted_4$_
];
function render$17(_ctx, _cache) {
  return openBlock(), createElementBlock("svg", _hoisted_1$26, _hoisted_5$B);
}
const IconE2ESolid = { name: "cy-testing-type-e2e-solid_x64", render: render$17 };
const _hoisted_1$25 = {
  style: { "min-width": "64px", "min-height": "64px" },
  width: "64",
  height: "64",
  viewBox: "0 0 64 64",
  xmlns: "http://www.w3.org/2000/svg"
};
const _hoisted_2$1R = /* @__PURE__ */ createBaseVNode("path", {
  d: "M36 5H57.5C58.3284 5 59 5.67157 59 6.5V28H36V5Z",
  fill: "#69D3A7",
  class: "icon-light"
}, null, -1);
const _hoisted_3$1A = /* @__PURE__ */ createBaseVNode("path", {
  d: "M32 32H9V53.5C9 54.3284 9.67157 55 10.5 55H32V32Z",
  fill: "#69D3A7",
  class: "icon-light"
}, null, -1);
const _hoisted_4$Z = /* @__PURE__ */ createBaseVNode("path", {
  d: "M32 9H33V8H32V9ZM36 5V4H35V5H36ZM59 28V29H60V28H59ZM36 28H35V29H36V28ZM55 32H56V31H55V32ZM10.5 10H32V8H10.5V10ZM31 9V32H33V9H31ZM32 31H9V33H32V31ZM10 32V10.5H8V32H10ZM31 32V55H33V32H31ZM32 54H10.5V56H32V54ZM10 53.5V32H8V53.5H10ZM36 6H57.5V4H36V6ZM58 6.5V28H60V6.5H58ZM59 27H36V29H59V27ZM37 28V5H35V28H37ZM32 33H55V31H32V33ZM54 32V53.5H56V32H54ZM53.5 54H32V56H53.5V54ZM54 53.5C54 53.7761 53.7761 54 53.5 54V56C54.8807 56 56 54.8807 56 53.5H54ZM57.5 6C57.7762 6 58 6.22386 58 6.5H60C60 5.11929 58.8807 4 57.5 4V6ZM10.5 54C10.2239 54 10 53.7761 10 53.5H8C8 54.8807 9.11929 56 10.5 56V54ZM10.5 8C9.11929 8 8 9.11929 8 10.5H10C10 10.2239 10.2239 10 10.5 10V8Z",
  fill: "#007780",
  class: "icon-dark"
}, null, -1);
const _hoisted_5$A = [
  _hoisted_2$1R,
  _hoisted_3$1A,
  _hoisted_4$Z
];
function render$16(_ctx, _cache) {
  return openBlock(), createElementBlock("svg", _hoisted_1$25, _hoisted_5$A);
}
const IconComponent = { name: "cy-testing-type-component_x64", render: render$16 };
const _hoisted_1$24 = {
  style: { "min-width": "64px", "min-height": "64px" },
  "data-cy": "testing-type-component",
  width: "64",
  height: "64",
  viewBox: "0 0 64 64",
  fill: "none",
  xmlns: "http://www.w3.org/2000/svg"
};
const _hoisted_2$1Q = /* @__PURE__ */ createBaseVNode("path", {
  d: "M36 4H57C58.6569 4 60 5.34315 60 7V28H36V4Z",
  fill: "#00595D",
  class: "icon-dark"
}, null, -1);
const _hoisted_3$1z = /* @__PURE__ */ createBaseVNode("path", {
  d: "M32 56H11C9.34315 56 8 54.6568 8 53V32H32V56Z",
  fill: "#00595D",
  class: "icon-dark"
}, null, -1);
const _hoisted_4$Y = /* @__PURE__ */ createBaseVNode("path", {
  d: "M36 28H34C34 29.1046 34.8954 30 36 30V28ZM8 32V30C6.89543 30 6 30.8954 6 32H8ZM32 56V58C33.1046 58 34 57.1045 34 56H32ZM36 4V2H34V4H36ZM60 28V30H62V28H60ZM32 32H34C34 30.8954 33.1046 30 32 30V32ZM32 54H11V58H32V54ZM10 53V32H6V53H10ZM36 6H57V2H36V6ZM58 7V28H62V7H58ZM60 26H36V30H60V26ZM38 28V4H34V28H38ZM30 32V56H34V32H30ZM32 30H8V34H32V30ZM57 6C57.5523 6 58 6.44772 58 7H62C62 4.23858 59.7614 2 57 2V6ZM11 54C10.4477 54 10 53.5522 10 53H6C6 55.7614 8.23858 58 11 58V54Z",
  class: "icon-light"
}, null, -1);
const _hoisted_5$z = /* @__PURE__ */ createBaseVNode("path", {
  d: "M11 8H32V32H56V53C56 54.6569 54.6568 56 53 56H32L32 32H8V11C8 9.34315 9.34315 8 11 8Z",
  fill: "#69D3A7",
  class: "icon-light"
}, null, -1);
const _hoisted_6$k = [
  _hoisted_2$1Q,
  _hoisted_3$1z,
  _hoisted_4$Y,
  _hoisted_5$z
];
function render$15(_ctx, _cache) {
  return openBlock(), createElementBlock("svg", _hoisted_1$24, _hoisted_6$k);
}
const IconComponentSolid = { name: "cy-testing-type-component-solid_x64", render: render$15 };
/*!
  * shared v9.14.5
  * (c) 2025 kazuya kawaguchi
  * Released under the MIT License.
  */
function warn(msg, err) {
  if (typeof console !== "undefined") {
    console.warn(`[intlify] ` + msg);
    if (err) {
      console.warn(err.stack);
    }
  }
}
const inBrowser = typeof window !== "undefined";
const makeSymbol = (name2, shareable = false) => !shareable ? Symbol(name2) : Symbol.for(name2);
const generateFormatCacheKey = (locale, key, source) => friendlyJSONstringify({ l: locale, k: key, s: source });
const friendlyJSONstringify = (json) => JSON.stringify(json).replace(/\u2028/g, "\\u2028").replace(/\u2029/g, "\\u2029").replace(/\u0027/g, "\\u0027");
const isNumber = (val) => typeof val === "number" && isFinite(val);
const isDate = (val) => toTypeString(val) === "[object Date]";
const isRegExp = (val) => toTypeString(val) === "[object RegExp]";
const isEmptyObject = (val) => isPlainObject$1(val) && Object.keys(val).length === 0;
const assign$2 = Object.assign;
const _create = Object.create;
const create = (obj = null) => _create(obj);
let _globalThis;
const getGlobalThis = () => {
  return _globalThis || (_globalThis = typeof globalThis !== "undefined" ? globalThis : typeof self !== "undefined" ? self : typeof window !== "undefined" ? window : typeof global !== "undefined" ? global : create());
};
function escapeHtml$1(rawText) {
  return rawText.replace(/&/g, "&amp;").replace(/</g, "&lt;").replace(/>/g, "&gt;").replace(/"/g, "&quot;").replace(/'/g, "&apos;").replace(/\//g, "&#x2F;").replace(/=/g, "&#x3D;");
}
function escapeAttributeValue(value) {
  return value.replace(/&(?![a-zA-Z0-9#]{2,6};)/g, "&amp;").replace(/"/g, "&quot;").replace(/'/g, "&apos;").replace(/</g, "&lt;").replace(/>/g, "&gt;");
}
function sanitizeTranslatedHtml(html) {
  html = html.replace(/(\w+)\s*=\s*"([^"]*)"/g, (_2, attrName, attrValue) => `${attrName}="${escapeAttributeValue(attrValue)}"`);
  html = html.replace(/(\w+)\s*=\s*'([^']*)'/g, (_2, attrName, attrValue) => `${attrName}='${escapeAttributeValue(attrValue)}'`);
  const eventHandlerPattern = /\s*on\w+\s*=\s*["']?[^"'>]+["']?/gi;
  if (eventHandlerPattern.test(html)) {
    html = html.replace(/(\s+)(on)(\w+\s*=)/gi, "$1&#111;n$3");
  }
  const javascriptUrlPattern = [
    // In href, src, action, formaction attributes
    /(\s+(?:href|src|action|formaction)\s*=\s*["']?)\s*javascript:/gi,
    // In style attributes within url()
    /(style\s*=\s*["'][^"']*url\s*\(\s*)javascript:/gi
  ];
  javascriptUrlPattern.forEach((pattern) => {
    html = html.replace(pattern, "$1javascript&#58;");
  });
  return html;
}
const hasOwnProperty$2 = Object.prototype.hasOwnProperty;
function hasOwn(obj, key) {
  return hasOwnProperty$2.call(obj, key);
}
const isArray$1 = Array.isArray;
const isFunction = (val) => typeof val === "function";
const isString$2 = (val) => typeof val === "string";
const isBoolean = (val) => typeof val === "boolean";
const isObject$3 = (val) => val !== null && typeof val === "object";
const isPromise = (val) => {
  return isObject$3(val) && isFunction(val.then) && isFunction(val.catch);
};
const objectToString$1 = Object.prototype.toString;
const toTypeString = (value) => objectToString$1.call(value);
const isPlainObject$1 = (val) => {
  if (!isObject$3(val))
    return false;
  const proto = Object.getPrototypeOf(val);
  return proto === null || proto.constructor === Object;
};
const toDisplayString = (val) => {
  return val == null ? "" : isArray$1(val) || isPlainObject$1(val) && val.toString === objectToString$1 ? JSON.stringify(val, null, 2) : String(val);
};
function join$2(items, separator = "") {
  return items.reduce((str, item, index) => index === 0 ? str + item : str + separator + item, "");
}
function incrementer(code2) {
  let current = code2;
  return () => ++current;
}
const isNotObjectOrIsArray = (val) => !isObject$3(val) || isArray$1(val);
function deepCopy(src, des) {
  if (isNotObjectOrIsArray(src) || isNotObjectOrIsArray(des)) {
    throw new Error("Invalid value");
  }
  const stack = [{ src, des }];
  while (stack.length) {
    const { src: src2, des: des2 } = stack.pop();
    Object.keys(src2).forEach((key) => {
      if (key === "__proto__") {
        return;
      }
      if (isObject$3(src2[key]) && !isObject$3(des2[key])) {
        des2[key] = Array.isArray(src2[key]) ? [] : create();
      }
      if (isNotObjectOrIsArray(des2[key]) || isNotObjectOrIsArray(src2[key])) {
        des2[key] = src2[key];
      } else {
        stack.push({ src: src2[key], des: des2[key] });
      }
    });
  }
}
/*!
  * message-compiler v9.14.5
  * (c) 2025 kazuya kawaguchi
  * Released under the MIT License.
  */
function createPosition(line, column, offset2) {
  return { line, column, offset: offset2 };
}
function createLocation(start, end, source) {
  const loc = { start, end };
  return loc;
}
const RE_ARGS = /\{([0-9a-zA-Z]+)\}/g;
function format$2(message, ...args) {
  if (args.length === 1 && isObject$2(args[0])) {
    args = args[0];
  }
  if (!args || !args.hasOwnProperty) {
    args = {};
  }
  return message.replace(RE_ARGS, (match2, identifier) => {
    return args.hasOwnProperty(identifier) ? args[identifier] : "";
  });
}
const assign$1 = Object.assign;
const isString$1 = (val) => typeof val === "string";
const isObject$2 = (val) => val !== null && typeof val === "object";
function join$1(items, separator = "") {
  return items.reduce((str, item, index) => index === 0 ? str + item : str + separator + item, "");
}
const CompileWarnCodes = {
  USE_MODULO_SYNTAX: 1,
  __EXTEND_POINT__: 2
};
const warnMessages = {
  [CompileWarnCodes.USE_MODULO_SYNTAX]: `Use modulo before '{{0}}'.`
};
function createCompileWarn(code2, loc, ...args) {
  const msg = format$2(warnMessages[code2], ...args || []);
  const message = { message: String(msg), code: code2 };
  if (loc) {
    message.location = loc;
  }
  return message;
}
const CompileErrorCodes = {
  // tokenizer error codes
  EXPECTED_TOKEN: 1,
  INVALID_TOKEN_IN_PLACEHOLDER: 2,
  UNTERMINATED_SINGLE_QUOTE_IN_PLACEHOLDER: 3,
  UNKNOWN_ESCAPE_SEQUENCE: 4,
  INVALID_UNICODE_ESCAPE_SEQUENCE: 5,
  UNBALANCED_CLOSING_BRACE: 6,
  UNTERMINATED_CLOSING_BRACE: 7,
  EMPTY_PLACEHOLDER: 8,
  NOT_ALLOW_NEST_PLACEHOLDER: 9,
  INVALID_LINKED_FORMAT: 10,
  // parser error codes
  MUST_HAVE_MESSAGES_IN_PLURAL: 11,
  UNEXPECTED_EMPTY_LINKED_MODIFIER: 12,
  UNEXPECTED_EMPTY_LINKED_KEY: 13,
  UNEXPECTED_LEXICAL_ANALYSIS: 14,
  // generator error codes
  UNHANDLED_CODEGEN_NODE_TYPE: 15,
  // minifier error codes
  UNHANDLED_MINIFIER_NODE_TYPE: 16,
  // Special value for higher-order compilers to pick up the last code
  // to avoid collision of error codes. This should always be kept as the last
  // item.
  __EXTEND_POINT__: 17
};
const errorMessages = {
  // tokenizer error messages
  [CompileErrorCodes.EXPECTED_TOKEN]: `Expected token: '{0}'`,
  [CompileErrorCodes.INVALID_TOKEN_IN_PLACEHOLDER]: `Invalid token in placeholder: '{0}'`,
  [CompileErrorCodes.UNTERMINATED_SINGLE_QUOTE_IN_PLACEHOLDER]: `Unterminated single quote in placeholder`,
  [CompileErrorCodes.UNKNOWN_ESCAPE_SEQUENCE]: `Unknown escape sequence: \\{0}`,
  [CompileErrorCodes.INVALID_UNICODE_ESCAPE_SEQUENCE]: `Invalid unicode escape sequence: {0}`,
  [CompileErrorCodes.UNBALANCED_CLOSING_BRACE]: `Unbalanced closing brace`,
  [CompileErrorCodes.UNTERMINATED_CLOSING_BRACE]: `Unterminated closing brace`,
  [CompileErrorCodes.EMPTY_PLACEHOLDER]: `Empty placeholder`,
  [CompileErrorCodes.NOT_ALLOW_NEST_PLACEHOLDER]: `Not allowed nest placeholder`,
  [CompileErrorCodes.INVALID_LINKED_FORMAT]: `Invalid linked format`,
  // parser error messages
  [CompileErrorCodes.MUST_HAVE_MESSAGES_IN_PLURAL]: `Plural must have messages`,
  [CompileErrorCodes.UNEXPECTED_EMPTY_LINKED_MODIFIER]: `Unexpected empty linked modifier`,
  [CompileErrorCodes.UNEXPECTED_EMPTY_LINKED_KEY]: `Unexpected empty linked key`,
  [CompileErrorCodes.UNEXPECTED_LEXICAL_ANALYSIS]: `Unexpected lexical analysis in token: '{0}'`,
  // generator error messages
  [CompileErrorCodes.UNHANDLED_CODEGEN_NODE_TYPE]: `unhandled codegen node type: '{0}'`,
  // minimizer error messages
  [CompileErrorCodes.UNHANDLED_MINIFIER_NODE_TYPE]: `unhandled mimifier node type: '{0}'`
};
function createCompileError(code2, loc, options2 = {}) {
  const { domain, messages, args } = options2;
  const msg = format$2((messages || errorMessages)[code2] || "", ...args || []);
  const error2 = new SyntaxError(String(msg));
  error2.code = code2;
  if (loc) {
    error2.location = loc;
  }
  error2.domain = domain;
  return error2;
}
function defaultOnError$1(error2) {
  throw error2;
}
const CHAR_SP = " ";
const CHAR_CR = "\r";
const CHAR_LF = "\n";
const CHAR_LS = String.fromCharCode(8232);
const CHAR_PS = String.fromCharCode(8233);
function createScanner$1(str) {
  const _buf = str;
  let _index = 0;
  let _line = 1;
  let _column = 1;
  let _peekOffset = 0;
  const isCRLF = (index2) => _buf[index2] === CHAR_CR && _buf[index2 + 1] === CHAR_LF;
  const isLF = (index2) => _buf[index2] === CHAR_LF;
  const isPS = (index2) => _buf[index2] === CHAR_PS;
  const isLS = (index2) => _buf[index2] === CHAR_LS;
  const isLineEnd = (index2) => isCRLF(index2) || isLF(index2) || isPS(index2) || isLS(index2);
  const index = () => _index;
  const line = () => _line;
  const column = () => _column;
  const peekOffset = () => _peekOffset;
  const charAt = (offset2) => isCRLF(offset2) || isPS(offset2) || isLS(offset2) ? CHAR_LF : _buf[offset2];
  const currentChar = () => charAt(_index);
  const currentPeek = () => charAt(_index + _peekOffset);
  function next() {
    _peekOffset = 0;
    if (isLineEnd(_index)) {
      _line++;
      _column = 0;
    }
    if (isCRLF(_index)) {
      _index++;
    }
    _index++;
    _column++;
    return _buf[_index];
  }
  function peek() {
    if (isCRLF(_index + _peekOffset)) {
      _peekOffset++;
    }
    _peekOffset++;
    return _buf[_index + _peekOffset];
  }
  function reset() {
    _index = 0;
    _line = 1;
    _column = 1;
    _peekOffset = 0;
  }
  function resetPeek(offset2 = 0) {
    _peekOffset = offset2;
  }
  function skipToPeek() {
    const target2 = _index + _peekOffset;
    while (target2 !== _index) {
      next();
    }
    _peekOffset = 0;
  }
  return {
    index,
    line,
    column,
    peekOffset,
    charAt,
    currentChar,
    currentPeek,
    next,
    peek,
    reset,
    resetPeek,
    skipToPeek
  };
}
const EOF = void 0;
const DOT = ".";
const LITERAL_DELIMITER = "'";
const ERROR_DOMAIN$3 = "tokenizer";
function createTokenizer(source, options2 = {}) {
  const location2 = options2.location !== false;
  const _scnr = createScanner$1(source);
  const currentOffset = () => _scnr.index();
  const currentPosition = () => createPosition(_scnr.line(), _scnr.column(), _scnr.index());
  const _initLoc = currentPosition();
  const _initOffset = currentOffset();
  const _context = {
    currentType: 14,
    offset: _initOffset,
    startLoc: _initLoc,
    endLoc: _initLoc,
    lastType: 14,
    lastOffset: _initOffset,
    lastStartLoc: _initLoc,
    lastEndLoc: _initLoc,
    braceNest: 0,
    inLinked: false,
    text: ""
  };
  const context = () => _context;
  const { onError } = options2;
  function emitError2(code2, pos, offset2, ...args) {
    const ctx = context();
    pos.column += offset2;
    pos.offset += offset2;
    if (onError) {
      const loc = location2 ? createLocation(ctx.startLoc, pos) : null;
      const err = createCompileError(code2, loc, {
        domain: ERROR_DOMAIN$3,
        args
      });
      onError(err);
    }
  }
  function getToken(context2, type2, value) {
    context2.endLoc = currentPosition();
    context2.currentType = type2;
    const token2 = { type: type2 };
    if (location2) {
      token2.loc = createLocation(context2.startLoc, context2.endLoc);
    }
    if (value != null) {
      token2.value = value;
    }
    return token2;
  }
  const getEndToken = (context2) => getToken(
    context2,
    14
    /* TokenTypes.EOF */
  );
  function eat(scnr, ch) {
    if (scnr.currentChar() === ch) {
      scnr.next();
      return ch;
    } else {
      emitError2(CompileErrorCodes.EXPECTED_TOKEN, currentPosition(), 0, ch);
      return "";
    }
  }
  function peekSpaces(scnr) {
    let buf = "";
    while (scnr.currentPeek() === CHAR_SP || scnr.currentPeek() === CHAR_LF) {
      buf += scnr.currentPeek();
      scnr.peek();
    }
    return buf;
  }
  function skipSpaces(scnr) {
    const buf = peekSpaces(scnr);
    scnr.skipToPeek();
    return buf;
  }
  function isIdentifierStart(ch) {
    if (ch === EOF) {
      return false;
    }
    const cc = ch.charCodeAt(0);
    return cc >= 97 && cc <= 122 || // a-z
    cc >= 65 && cc <= 90 || // A-Z
    cc === 95;
  }
  function isNumberStart(ch) {
    if (ch === EOF) {
      return false;
    }
    const cc = ch.charCodeAt(0);
    return cc >= 48 && cc <= 57;
  }
  function isNamedIdentifierStart(scnr, context2) {
    const { currentType } = context2;
    if (currentType !== 2) {
      return false;
    }
    peekSpaces(scnr);
    const ret = isIdentifierStart(scnr.currentPeek());
    scnr.resetPeek();
    return ret;
  }
  function isListIdentifierStart(scnr, context2) {
    const { currentType } = context2;
    if (currentType !== 2) {
      return false;
    }
    peekSpaces(scnr);
    const ch = scnr.currentPeek() === "-" ? scnr.peek() : scnr.currentPeek();
    const ret = isNumberStart(ch);
    scnr.resetPeek();
    return ret;
  }
  function isLiteralStart(scnr, context2) {
    const { currentType } = context2;
    if (currentType !== 2) {
      return false;
    }
    peekSpaces(scnr);
    const ret = scnr.currentPeek() === LITERAL_DELIMITER;
    scnr.resetPeek();
    return ret;
  }
  function isLinkedDotStart(scnr, context2) {
    const { currentType } = context2;
    if (currentType !== 8) {
      return false;
    }
    peekSpaces(scnr);
    const ret = scnr.currentPeek() === ".";
    scnr.resetPeek();
    return ret;
  }
  function isLinkedModifierStart(scnr, context2) {
    const { currentType } = context2;
    if (currentType !== 9) {
      return false;
    }
    peekSpaces(scnr);
    const ret = isIdentifierStart(scnr.currentPeek());
    scnr.resetPeek();
    return ret;
  }
  function isLinkedDelimiterStart(scnr, context2) {
    const { currentType } = context2;
    if (!(currentType === 8 || currentType === 12)) {
      return false;
    }
    peekSpaces(scnr);
    const ret = scnr.currentPeek() === ":";
    scnr.resetPeek();
    return ret;
  }
  function isLinkedReferStart(scnr, context2) {
    const { currentType } = context2;
    if (currentType !== 10) {
      return false;
    }
    const fn = () => {
      const ch = scnr.currentPeek();
      if (ch === "{") {
        return isIdentifierStart(scnr.peek());
      } else if (ch === "@" || ch === "%" || ch === "|" || ch === ":" || ch === "." || ch === CHAR_SP || !ch) {
        return false;
      } else if (ch === CHAR_LF) {
        scnr.peek();
        return fn();
      } else {
        return isTextStart(scnr, false);
      }
    };
    const ret = fn();
    scnr.resetPeek();
    return ret;
  }
  function isPluralStart(scnr) {
    peekSpaces(scnr);
    const ret = scnr.currentPeek() === "|";
    scnr.resetPeek();
    return ret;
  }
  function detectModuloStart(scnr) {
    const spaces = peekSpaces(scnr);
    const ret = scnr.currentPeek() === "%" && scnr.peek() === "{";
    scnr.resetPeek();
    return {
      isModulo: ret,
      hasSpace: spaces.length > 0
    };
  }
  function isTextStart(scnr, reset = true) {
    const fn = (hasSpace = false, prev = "", detectModulo = false) => {
      const ch = scnr.currentPeek();
      if (ch === "{") {
        return prev === "%" ? false : hasSpace;
      } else if (ch === "@" || !ch) {
        return prev === "%" ? true : hasSpace;
      } else if (ch === "%") {
        scnr.peek();
        return fn(hasSpace, "%", true);
      } else if (ch === "|") {
        return prev === "%" || detectModulo ? true : !(prev === CHAR_SP || prev === CHAR_LF);
      } else if (ch === CHAR_SP) {
        scnr.peek();
        return fn(true, CHAR_SP, detectModulo);
      } else if (ch === CHAR_LF) {
        scnr.peek();
        return fn(true, CHAR_LF, detectModulo);
      } else {
        return true;
      }
    };
    const ret = fn();
    reset && scnr.resetPeek();
    return ret;
  }
  function takeChar(scnr, fn) {
    const ch = scnr.currentChar();
    if (ch === EOF) {
      return EOF;
    }
    if (fn(ch)) {
      scnr.next();
      return ch;
    }
    return null;
  }
  function isIdentifier(ch) {
    const cc = ch.charCodeAt(0);
    return cc >= 97 && cc <= 122 || // a-z
    cc >= 65 && cc <= 90 || // A-Z
    cc >= 48 && cc <= 57 || // 0-9
    cc === 95 || // _
    cc === 36;
  }
  function takeIdentifierChar(scnr) {
    return takeChar(scnr, isIdentifier);
  }
  function isNamedIdentifier(ch) {
    const cc = ch.charCodeAt(0);
    return cc >= 97 && cc <= 122 || // a-z
    cc >= 65 && cc <= 90 || // A-Z
    cc >= 48 && cc <= 57 || // 0-9
    cc === 95 || // _
    cc === 36 || // $
    cc === 45;
  }
  function takeNamedIdentifierChar(scnr) {
    return takeChar(scnr, isNamedIdentifier);
  }
  function isDigit2(ch) {
    const cc = ch.charCodeAt(0);
    return cc >= 48 && cc <= 57;
  }
  function takeDigit(scnr) {
    return takeChar(scnr, isDigit2);
  }
  function isHexDigit(ch) {
    const cc = ch.charCodeAt(0);
    return cc >= 48 && cc <= 57 || // 0-9
    cc >= 65 && cc <= 70 || // A-F
    cc >= 97 && cc <= 102;
  }
  function takeHexDigit(scnr) {
    return takeChar(scnr, isHexDigit);
  }
  function getDigits(scnr) {
    let ch = "";
    let num2 = "";
    while (ch = takeDigit(scnr)) {
      num2 += ch;
    }
    return num2;
  }
  function readModulo(scnr) {
    skipSpaces(scnr);
    const ch = scnr.currentChar();
    if (ch !== "%") {
      emitError2(CompileErrorCodes.EXPECTED_TOKEN, currentPosition(), 0, ch);
    }
    scnr.next();
    return "%";
  }
  function readText(scnr) {
    let buf = "";
    while (true) {
      const ch = scnr.currentChar();
      if (ch === "{" || ch === "}" || ch === "@" || ch === "|" || !ch) {
        break;
      } else if (ch === "%") {
        if (isTextStart(scnr)) {
          buf += ch;
          scnr.next();
        } else {
          break;
        }
      } else if (ch === CHAR_SP || ch === CHAR_LF) {
        if (isTextStart(scnr)) {
          buf += ch;
          scnr.next();
        } else if (isPluralStart(scnr)) {
          break;
        } else {
          buf += ch;
          scnr.next();
        }
      } else {
        buf += ch;
        scnr.next();
      }
    }
    return buf;
  }
  function readNamedIdentifier(scnr) {
    skipSpaces(scnr);
    let ch = "";
    let name2 = "";
    while (ch = takeNamedIdentifierChar(scnr)) {
      name2 += ch;
    }
    if (scnr.currentChar() === EOF) {
      emitError2(CompileErrorCodes.UNTERMINATED_CLOSING_BRACE, currentPosition(), 0);
    }
    return name2;
  }
  function readListIdentifier(scnr) {
    skipSpaces(scnr);
    let value = "";
    if (scnr.currentChar() === "-") {
      scnr.next();
      value += `-${getDigits(scnr)}`;
    } else {
      value += getDigits(scnr);
    }
    if (scnr.currentChar() === EOF) {
      emitError2(CompileErrorCodes.UNTERMINATED_CLOSING_BRACE, currentPosition(), 0);
    }
    return value;
  }
  function isLiteral2(ch) {
    return ch !== LITERAL_DELIMITER && ch !== CHAR_LF;
  }
  function readLiteral(scnr) {
    skipSpaces(scnr);
    eat(scnr, `'`);
    let ch = "";
    let literal = "";
    while (ch = takeChar(scnr, isLiteral2)) {
      if (ch === "\\") {
        literal += readEscapeSequence(scnr);
      } else {
        literal += ch;
      }
    }
    const current = scnr.currentChar();
    if (current === CHAR_LF || current === EOF) {
      emitError2(CompileErrorCodes.UNTERMINATED_SINGLE_QUOTE_IN_PLACEHOLDER, currentPosition(), 0);
      if (current === CHAR_LF) {
        scnr.next();
        eat(scnr, `'`);
      }
      return literal;
    }
    eat(scnr, `'`);
    return literal;
  }
  function readEscapeSequence(scnr) {
    const ch = scnr.currentChar();
    switch (ch) {
      case "\\":
      case `'`:
        scnr.next();
        return `\\${ch}`;
      case "u":
        return readUnicodeEscapeSequence(scnr, ch, 4);
      case "U":
        return readUnicodeEscapeSequence(scnr, ch, 6);
      default:
        emitError2(CompileErrorCodes.UNKNOWN_ESCAPE_SEQUENCE, currentPosition(), 0, ch);
        return "";
    }
  }
  function readUnicodeEscapeSequence(scnr, unicode, digits) {
    eat(scnr, unicode);
    let sequence = "";
    for (let i2 = 0; i2 < digits; i2++) {
      const ch = takeHexDigit(scnr);
      if (!ch) {
        emitError2(CompileErrorCodes.INVALID_UNICODE_ESCAPE_SEQUENCE, currentPosition(), 0, `\\${unicode}${sequence}${scnr.currentChar()}`);
        break;
      }
      sequence += ch;
    }
    return `\\${unicode}${sequence}`;
  }
  function isInvalidIdentifier(ch) {
    return ch !== "{" && ch !== "}" && ch !== CHAR_SP && ch !== CHAR_LF;
  }
  function readInvalidIdentifier(scnr) {
    skipSpaces(scnr);
    let ch = "";
    let identifiers2 = "";
    while (ch = takeChar(scnr, isInvalidIdentifier)) {
      identifiers2 += ch;
    }
    return identifiers2;
  }
  function readLinkedModifier(scnr) {
    let ch = "";
    let name2 = "";
    while (ch = takeIdentifierChar(scnr)) {
      name2 += ch;
    }
    return name2;
  }
  function readLinkedRefer(scnr) {
    const fn = (buf) => {
      const ch = scnr.currentChar();
      if (ch === "{" || ch === "%" || ch === "@" || ch === "|" || ch === "(" || ch === ")" || !ch) {
        return buf;
      } else if (ch === CHAR_SP) {
        return buf;
      } else if (ch === CHAR_LF || ch === DOT) {
        buf += ch;
        scnr.next();
        return fn(buf);
      } else {
        buf += ch;
        scnr.next();
        return fn(buf);
      }
    };
    return fn("");
  }
  function readPlural(scnr) {
    skipSpaces(scnr);
    const plural = eat(
      scnr,
      "|"
      /* TokenChars.Pipe */
    );
    skipSpaces(scnr);
    return plural;
  }
  function readTokenInPlaceholder(scnr, context2) {
    let token2 = null;
    const ch = scnr.currentChar();
    switch (ch) {
      case "{":
        if (context2.braceNest >= 1) {
          emitError2(CompileErrorCodes.NOT_ALLOW_NEST_PLACEHOLDER, currentPosition(), 0);
        }
        scnr.next();
        token2 = getToken(
          context2,
          2,
          "{"
          /* TokenChars.BraceLeft */
        );
        skipSpaces(scnr);
        context2.braceNest++;
        return token2;
      case "}":
        if (context2.braceNest > 0 && context2.currentType === 2) {
          emitError2(CompileErrorCodes.EMPTY_PLACEHOLDER, currentPosition(), 0);
        }
        scnr.next();
        token2 = getToken(
          context2,
          3,
          "}"
          /* TokenChars.BraceRight */
        );
        context2.braceNest--;
        context2.braceNest > 0 && skipSpaces(scnr);
        if (context2.inLinked && context2.braceNest === 0) {
          context2.inLinked = false;
        }
        return token2;
      case "@":
        if (context2.braceNest > 0) {
          emitError2(CompileErrorCodes.UNTERMINATED_CLOSING_BRACE, currentPosition(), 0);
        }
        token2 = readTokenInLinked(scnr, context2) || getEndToken(context2);
        context2.braceNest = 0;
        return token2;
      default: {
        let validNamedIdentifier = true;
        let validListIdentifier = true;
        let validLiteral = true;
        if (isPluralStart(scnr)) {
          if (context2.braceNest > 0) {
            emitError2(CompileErrorCodes.UNTERMINATED_CLOSING_BRACE, currentPosition(), 0);
          }
          token2 = getToken(context2, 1, readPlural(scnr));
          context2.braceNest = 0;
          context2.inLinked = false;
          return token2;
        }
        if (context2.braceNest > 0 && (context2.currentType === 5 || context2.currentType === 6 || context2.currentType === 7)) {
          emitError2(CompileErrorCodes.UNTERMINATED_CLOSING_BRACE, currentPosition(), 0);
          context2.braceNest = 0;
          return readToken2(scnr, context2);
        }
        if (validNamedIdentifier = isNamedIdentifierStart(scnr, context2)) {
          token2 = getToken(context2, 5, readNamedIdentifier(scnr));
          skipSpaces(scnr);
          return token2;
        }
        if (validListIdentifier = isListIdentifierStart(scnr, context2)) {
          token2 = getToken(context2, 6, readListIdentifier(scnr));
          skipSpaces(scnr);
          return token2;
        }
        if (validLiteral = isLiteralStart(scnr, context2)) {
          token2 = getToken(context2, 7, readLiteral(scnr));
          skipSpaces(scnr);
          return token2;
        }
        if (!validNamedIdentifier && !validListIdentifier && !validLiteral) {
          token2 = getToken(context2, 13, readInvalidIdentifier(scnr));
          emitError2(CompileErrorCodes.INVALID_TOKEN_IN_PLACEHOLDER, currentPosition(), 0, token2.value);
          skipSpaces(scnr);
          return token2;
        }
        break;
      }
    }
    return token2;
  }
  function readTokenInLinked(scnr, context2) {
    const { currentType } = context2;
    let token2 = null;
    const ch = scnr.currentChar();
    if ((currentType === 8 || currentType === 9 || currentType === 12 || currentType === 10) && (ch === CHAR_LF || ch === CHAR_SP)) {
      emitError2(CompileErrorCodes.INVALID_LINKED_FORMAT, currentPosition(), 0);
    }
    switch (ch) {
      case "@":
        scnr.next();
        token2 = getToken(
          context2,
          8,
          "@"
          /* TokenChars.LinkedAlias */
        );
        context2.inLinked = true;
        return token2;
      case ".":
        skipSpaces(scnr);
        scnr.next();
        return getToken(
          context2,
          9,
          "."
          /* TokenChars.LinkedDot */
        );
      case ":":
        skipSpaces(scnr);
        scnr.next();
        return getToken(
          context2,
          10,
          ":"
          /* TokenChars.LinkedDelimiter */
        );
      default:
        if (isPluralStart(scnr)) {
          token2 = getToken(context2, 1, readPlural(scnr));
          context2.braceNest = 0;
          context2.inLinked = false;
          return token2;
        }
        if (isLinkedDotStart(scnr, context2) || isLinkedDelimiterStart(scnr, context2)) {
          skipSpaces(scnr);
          return readTokenInLinked(scnr, context2);
        }
        if (isLinkedModifierStart(scnr, context2)) {
          skipSpaces(scnr);
          return getToken(context2, 12, readLinkedModifier(scnr));
        }
        if (isLinkedReferStart(scnr, context2)) {
          skipSpaces(scnr);
          if (ch === "{") {
            return readTokenInPlaceholder(scnr, context2) || token2;
          } else {
            return getToken(context2, 11, readLinkedRefer(scnr));
          }
        }
        if (currentType === 8) {
          emitError2(CompileErrorCodes.INVALID_LINKED_FORMAT, currentPosition(), 0);
        }
        context2.braceNest = 0;
        context2.inLinked = false;
        return readToken2(scnr, context2);
    }
  }
  function readToken2(scnr, context2) {
    let token2 = {
      type: 14
      /* TokenTypes.EOF */
    };
    if (context2.braceNest > 0) {
      return readTokenInPlaceholder(scnr, context2) || getEndToken(context2);
    }
    if (context2.inLinked) {
      return readTokenInLinked(scnr, context2) || getEndToken(context2);
    }
    const ch = scnr.currentChar();
    switch (ch) {
      case "{":
        return readTokenInPlaceholder(scnr, context2) || getEndToken(context2);
      case "}":
        emitError2(CompileErrorCodes.UNBALANCED_CLOSING_BRACE, currentPosition(), 0);
        scnr.next();
        return getToken(
          context2,
          3,
          "}"
          /* TokenChars.BraceRight */
        );
      case "@":
        return readTokenInLinked(scnr, context2) || getEndToken(context2);
      default: {
        if (isPluralStart(scnr)) {
          token2 = getToken(context2, 1, readPlural(scnr));
          context2.braceNest = 0;
          context2.inLinked = false;
          return token2;
        }
        const { isModulo, hasSpace } = detectModuloStart(scnr);
        if (isModulo) {
          return hasSpace ? getToken(context2, 0, readText(scnr)) : getToken(context2, 4, readModulo(scnr));
        }
        if (isTextStart(scnr)) {
          return getToken(context2, 0, readText(scnr));
        }
        break;
      }
    }
    return token2;
  }
  function nextToken() {
    const { currentType, offset: offset2, startLoc, endLoc } = _context;
    _context.lastType = currentType;
    _context.lastOffset = offset2;
    _context.lastStartLoc = startLoc;
    _context.lastEndLoc = endLoc;
    _context.offset = currentOffset();
    _context.startLoc = currentPosition();
    if (_scnr.currentChar() === EOF) {
      return getToken(
        _context,
        14
        /* TokenTypes.EOF */
      );
    }
    return readToken2(_scnr, _context);
  }
  return {
    nextToken,
    currentOffset,
    currentPosition,
    context
  };
}
const ERROR_DOMAIN$2 = "parser";
const KNOWN_ESCAPES = /(?:\\\\|\\'|\\u([0-9a-fA-F]{4})|\\U([0-9a-fA-F]{6}))/g;
function fromEscapeSequence(match2, codePoint4, codePoint6) {
  switch (match2) {
    case `\\\\`:
      return `\\`;
    // eslint-disable-next-line no-useless-escape
    case `\\'`:
      return `'`;
    default: {
      const codePoint = parseInt(codePoint4 || codePoint6, 16);
      if (codePoint <= 55295 || codePoint >= 57344) {
        return String.fromCodePoint(codePoint);
      }
      return "�";
    }
  }
}
function createParser(options2 = {}) {
  const location2 = options2.location !== false;
  const { onError, onWarn } = options2;
  function emitError2(tokenzer, code2, start, offset2, ...args) {
    const end = tokenzer.currentPosition();
    end.offset += offset2;
    end.column += offset2;
    if (onError) {
      const loc = location2 ? createLocation(start, end) : null;
      const err = createCompileError(code2, loc, {
        domain: ERROR_DOMAIN$2,
        args
      });
      onError(err);
    }
  }
  function emitWarn(tokenzer, code2, start, offset2, ...args) {
    const end = tokenzer.currentPosition();
    end.offset += offset2;
    end.column += offset2;
    if (onWarn) {
      const loc = location2 ? createLocation(start, end) : null;
      onWarn(createCompileWarn(code2, loc, args));
    }
  }
  function startNode(type2, offset2, loc) {
    const node = { type: type2 };
    if (location2) {
      node.start = offset2;
      node.end = offset2;
      node.loc = { start: loc, end: loc };
    }
    return node;
  }
  function endNode(node, offset2, pos, type2) {
    if (location2) {
      node.end = offset2;
      if (node.loc) {
        node.loc.end = pos;
      }
    }
  }
  function parseText2(tokenizer, value) {
    const context = tokenizer.context();
    const node = startNode(3, context.offset, context.startLoc);
    node.value = value;
    endNode(node, tokenizer.currentOffset(), tokenizer.currentPosition());
    return node;
  }
  function parseList(tokenizer, index) {
    const context = tokenizer.context();
    const { lastOffset: offset2, lastStartLoc: loc } = context;
    const node = startNode(5, offset2, loc);
    node.index = parseInt(index, 10);
    tokenizer.nextToken();
    endNode(node, tokenizer.currentOffset(), tokenizer.currentPosition());
    return node;
  }
  function parseNamed(tokenizer, key, modulo) {
    const context = tokenizer.context();
    const { lastOffset: offset2, lastStartLoc: loc } = context;
    const node = startNode(4, offset2, loc);
    node.key = key;
    if (modulo === true) {
      node.modulo = true;
    }
    tokenizer.nextToken();
    endNode(node, tokenizer.currentOffset(), tokenizer.currentPosition());
    return node;
  }
  function parseLiteral(tokenizer, value) {
    const context = tokenizer.context();
    const { lastOffset: offset2, lastStartLoc: loc } = context;
    const node = startNode(9, offset2, loc);
    node.value = value.replace(KNOWN_ESCAPES, fromEscapeSequence);
    tokenizer.nextToken();
    endNode(node, tokenizer.currentOffset(), tokenizer.currentPosition());
    return node;
  }
  function parseLinkedModifier(tokenizer) {
    const token2 = tokenizer.nextToken();
    const context = tokenizer.context();
    const { lastOffset: offset2, lastStartLoc: loc } = context;
    const node = startNode(8, offset2, loc);
    if (token2.type !== 12) {
      emitError2(tokenizer, CompileErrorCodes.UNEXPECTED_EMPTY_LINKED_MODIFIER, context.lastStartLoc, 0);
      node.value = "";
      endNode(node, offset2, loc);
      return {
        nextConsumeToken: token2,
        node
      };
    }
    if (token2.value == null) {
      emitError2(tokenizer, CompileErrorCodes.UNEXPECTED_LEXICAL_ANALYSIS, context.lastStartLoc, 0, getTokenCaption(token2));
    }
    node.value = token2.value || "";
    endNode(node, tokenizer.currentOffset(), tokenizer.currentPosition());
    return {
      node
    };
  }
  function parseLinkedKey(tokenizer, value) {
    const context = tokenizer.context();
    const node = startNode(7, context.offset, context.startLoc);
    node.value = value;
    endNode(node, tokenizer.currentOffset(), tokenizer.currentPosition());
    return node;
  }
  function parseLinked(tokenizer) {
    const context = tokenizer.context();
    const linkedNode = startNode(6, context.offset, context.startLoc);
    let token2 = tokenizer.nextToken();
    if (token2.type === 9) {
      const parsed = parseLinkedModifier(tokenizer);
      linkedNode.modifier = parsed.node;
      token2 = parsed.nextConsumeToken || tokenizer.nextToken();
    }
    if (token2.type !== 10) {
      emitError2(tokenizer, CompileErrorCodes.UNEXPECTED_LEXICAL_ANALYSIS, context.lastStartLoc, 0, getTokenCaption(token2));
    }
    token2 = tokenizer.nextToken();
    if (token2.type === 2) {
      token2 = tokenizer.nextToken();
    }
    switch (token2.type) {
      case 11:
        if (token2.value == null) {
          emitError2(tokenizer, CompileErrorCodes.UNEXPECTED_LEXICAL_ANALYSIS, context.lastStartLoc, 0, getTokenCaption(token2));
        }
        linkedNode.key = parseLinkedKey(tokenizer, token2.value || "");
        break;
      case 5:
        if (token2.value == null) {
          emitError2(tokenizer, CompileErrorCodes.UNEXPECTED_LEXICAL_ANALYSIS, context.lastStartLoc, 0, getTokenCaption(token2));
        }
        linkedNode.key = parseNamed(tokenizer, token2.value || "");
        break;
      case 6:
        if (token2.value == null) {
          emitError2(tokenizer, CompileErrorCodes.UNEXPECTED_LEXICAL_ANALYSIS, context.lastStartLoc, 0, getTokenCaption(token2));
        }
        linkedNode.key = parseList(tokenizer, token2.value || "");
        break;
      case 7:
        if (token2.value == null) {
          emitError2(tokenizer, CompileErrorCodes.UNEXPECTED_LEXICAL_ANALYSIS, context.lastStartLoc, 0, getTokenCaption(token2));
        }
        linkedNode.key = parseLiteral(tokenizer, token2.value || "");
        break;
      default: {
        emitError2(tokenizer, CompileErrorCodes.UNEXPECTED_EMPTY_LINKED_KEY, context.lastStartLoc, 0);
        const nextContext = tokenizer.context();
        const emptyLinkedKeyNode = startNode(7, nextContext.offset, nextContext.startLoc);
        emptyLinkedKeyNode.value = "";
        endNode(emptyLinkedKeyNode, nextContext.offset, nextContext.startLoc);
        linkedNode.key = emptyLinkedKeyNode;
        endNode(linkedNode, nextContext.offset, nextContext.startLoc);
        return {
          nextConsumeToken: token2,
          node: linkedNode
        };
      }
    }
    endNode(linkedNode, tokenizer.currentOffset(), tokenizer.currentPosition());
    return {
      node: linkedNode
    };
  }
  function parseMessage2(tokenizer) {
    const context = tokenizer.context();
    const startOffset = context.currentType === 1 ? tokenizer.currentOffset() : context.offset;
    const startLoc = context.currentType === 1 ? context.endLoc : context.startLoc;
    const node = startNode(2, startOffset, startLoc);
    node.items = [];
    let nextToken = null;
    let modulo = null;
    do {
      const token2 = nextToken || tokenizer.nextToken();
      nextToken = null;
      switch (token2.type) {
        case 0:
          if (token2.value == null) {
            emitError2(tokenizer, CompileErrorCodes.UNEXPECTED_LEXICAL_ANALYSIS, context.lastStartLoc, 0, getTokenCaption(token2));
          }
          node.items.push(parseText2(tokenizer, token2.value || ""));
          break;
        case 6:
          if (token2.value == null) {
            emitError2(tokenizer, CompileErrorCodes.UNEXPECTED_LEXICAL_ANALYSIS, context.lastStartLoc, 0, getTokenCaption(token2));
          }
          node.items.push(parseList(tokenizer, token2.value || ""));
          break;
        case 4:
          modulo = true;
          break;
        case 5:
          if (token2.value == null) {
            emitError2(tokenizer, CompileErrorCodes.UNEXPECTED_LEXICAL_ANALYSIS, context.lastStartLoc, 0, getTokenCaption(token2));
          }
          node.items.push(parseNamed(tokenizer, token2.value || "", !!modulo));
          if (modulo) {
            emitWarn(tokenizer, CompileWarnCodes.USE_MODULO_SYNTAX, context.lastStartLoc, 0, getTokenCaption(token2));
            modulo = null;
          }
          break;
        case 7:
          if (token2.value == null) {
            emitError2(tokenizer, CompileErrorCodes.UNEXPECTED_LEXICAL_ANALYSIS, context.lastStartLoc, 0, getTokenCaption(token2));
          }
          node.items.push(parseLiteral(tokenizer, token2.value || ""));
          break;
        case 8: {
          const parsed = parseLinked(tokenizer);
          node.items.push(parsed.node);
          nextToken = parsed.nextConsumeToken || null;
          break;
        }
      }
    } while (context.currentType !== 14 && context.currentType !== 1);
    const endOffset = context.currentType === 1 ? context.lastOffset : tokenizer.currentOffset();
    const endLoc = context.currentType === 1 ? context.lastEndLoc : tokenizer.currentPosition();
    endNode(node, endOffset, endLoc);
    return node;
  }
  function parsePlural(tokenizer, offset2, loc, msgNode) {
    const context = tokenizer.context();
    let hasEmptyMessage = msgNode.items.length === 0;
    const node = startNode(1, offset2, loc);
    node.cases = [];
    node.cases.push(msgNode);
    do {
      const msg = parseMessage2(tokenizer);
      if (!hasEmptyMessage) {
        hasEmptyMessage = msg.items.length === 0;
      }
      node.cases.push(msg);
    } while (context.currentType !== 14);
    if (hasEmptyMessage) {
      emitError2(tokenizer, CompileErrorCodes.MUST_HAVE_MESSAGES_IN_PLURAL, loc, 0);
    }
    endNode(node, tokenizer.currentOffset(), tokenizer.currentPosition());
    return node;
  }
  function parseResource(tokenizer) {
    const context = tokenizer.context();
    const { offset: offset2, startLoc } = context;
    const msgNode = parseMessage2(tokenizer);
    if (context.currentType === 14) {
      return msgNode;
    } else {
      return parsePlural(tokenizer, offset2, startLoc, msgNode);
    }
  }
  function parse2(source) {
    const tokenizer = createTokenizer(source, assign$1({}, options2));
    const context = tokenizer.context();
    const node = startNode(0, context.offset, context.startLoc);
    if (location2 && node.loc) {
      node.loc.source = source;
    }
    node.body = parseResource(tokenizer);
    if (options2.onCacheKey) {
      node.cacheKey = options2.onCacheKey(source);
    }
    if (context.currentType !== 14) {
      emitError2(tokenizer, CompileErrorCodes.UNEXPECTED_LEXICAL_ANALYSIS, context.lastStartLoc, 0, source[context.offset] || "");
    }
    endNode(node, tokenizer.currentOffset(), tokenizer.currentPosition());
    return node;
  }
  return { parse: parse2 };
}
function getTokenCaption(token2) {
  if (token2.type === 14) {
    return "EOF";
  }
  const name2 = (token2.value || "").replace(/\r?\n/gu, "\\n");
  return name2.length > 10 ? name2.slice(0, 9) + "…" : name2;
}
function createTransformer(ast2, options2 = {}) {
  const _context = {
    ast: ast2,
    helpers: /* @__PURE__ */ new Set()
  };
  const context = () => _context;
  const helper = (name2) => {
    _context.helpers.add(name2);
    return name2;
  };
  return { context, helper };
}
function traverseNodes(nodes, transformer) {
  for (let i2 = 0; i2 < nodes.length; i2++) {
    traverseNode$1(nodes[i2], transformer);
  }
}
function traverseNode$1(node, transformer) {
  switch (node.type) {
    case 1:
      traverseNodes(node.cases, transformer);
      transformer.helper(
        "plural"
        /* HelperNameMap.PLURAL */
      );
      break;
    case 2:
      traverseNodes(node.items, transformer);
      break;
    case 6: {
      const linked = node;
      traverseNode$1(linked.key, transformer);
      transformer.helper(
        "linked"
        /* HelperNameMap.LINKED */
      );
      transformer.helper(
        "type"
        /* HelperNameMap.TYPE */
      );
      break;
    }
    case 5:
      transformer.helper(
        "interpolate"
        /* HelperNameMap.INTERPOLATE */
      );
      transformer.helper(
        "list"
        /* HelperNameMap.LIST */
      );
      break;
    case 4:
      transformer.helper(
        "interpolate"
        /* HelperNameMap.INTERPOLATE */
      );
      transformer.helper(
        "named"
        /* HelperNameMap.NAMED */
      );
      break;
  }
}
function transform$1(ast2, options2 = {}) {
  const transformer = createTransformer(ast2);
  transformer.helper(
    "normalize"
    /* HelperNameMap.NORMALIZE */
  );
  ast2.body && traverseNode$1(ast2.body, transformer);
  const context = transformer.context();
  ast2.helpers = Array.from(context.helpers);
}
function optimize(ast2) {
  const body = ast2.body;
  if (body.type === 2) {
    optimizeMessageNode(body);
  } else {
    body.cases.forEach((c2) => optimizeMessageNode(c2));
  }
  return ast2;
}
function optimizeMessageNode(message) {
  if (message.items.length === 1) {
    const item = message.items[0];
    if (item.type === 3 || item.type === 9) {
      message.static = item.value;
      delete item.value;
    }
  } else {
    const values = [];
    for (let i2 = 0; i2 < message.items.length; i2++) {
      const item = message.items[i2];
      if (!(item.type === 3 || item.type === 9)) {
        break;
      }
      if (item.value == null) {
        break;
      }
      values.push(item.value);
    }
    if (values.length === message.items.length) {
      message.static = join$1(values);
      for (let i2 = 0; i2 < message.items.length; i2++) {
        const item = message.items[i2];
        if (item.type === 3 || item.type === 9) {
          delete item.value;
        }
      }
    }
  }
}
const ERROR_DOMAIN$1 = "minifier";
function minify(node) {
  node.t = node.type;
  switch (node.type) {
    case 0: {
      const resource = node;
      minify(resource.body);
      resource.b = resource.body;
      delete resource.body;
      break;
    }
    case 1: {
      const plural = node;
      const cases = plural.cases;
      for (let i2 = 0; i2 < cases.length; i2++) {
        minify(cases[i2]);
      }
      plural.c = cases;
      delete plural.cases;
      break;
    }
    case 2: {
      const message = node;
      const items = message.items;
      for (let i2 = 0; i2 < items.length; i2++) {
        minify(items[i2]);
      }
      message.i = items;
      delete message.items;
      if (message.static) {
        message.s = message.static;
        delete message.static;
      }
      break;
    }
    case 3:
    case 9:
    case 8:
    case 7: {
      const valueNode = node;
      if (valueNode.value) {
        valueNode.v = valueNode.value;
        delete valueNode.value;
      }
      break;
    }
    case 6: {
      const linked = node;
      minify(linked.key);
      linked.k = linked.key;
      delete linked.key;
      if (linked.modifier) {
        minify(linked.modifier);
        linked.m = linked.modifier;
        delete linked.modifier;
      }
      break;
    }
    case 5: {
      const list2 = node;
      list2.i = list2.index;
      delete list2.index;
      break;
    }
    case 4: {
      const named = node;
      named.k = named.key;
      delete named.key;
      break;
    }
    default: {
      throw createCompileError(CompileErrorCodes.UNHANDLED_MINIFIER_NODE_TYPE, null, {
        domain: ERROR_DOMAIN$1,
        args: [node.type]
      });
    }
  }
  delete node.type;
}
const ERROR_DOMAIN = "parser";
function createCodeGenerator(ast2, options2) {
  const { filename, breakLineCode, needIndent: _needIndent } = options2;
  const location2 = options2.location !== false;
  const _context = {
    filename,
    code: "",
    column: 1,
    line: 1,
    offset: 0,
    map: void 0,
    breakLineCode,
    needIndent: _needIndent,
    indentLevel: 0
  };
  if (location2 && ast2.loc) {
    _context.source = ast2.loc.source;
  }
  const context = () => _context;
  function push(code2, node) {
    _context.code += code2;
  }
  function _newline(n2, withBreakLine = true) {
    const _breakLineCode = withBreakLine ? breakLineCode : "";
    push(_needIndent ? _breakLineCode + `  `.repeat(n2) : _breakLineCode);
  }
  function indent2(withNewLine = true) {
    const level = ++_context.indentLevel;
    withNewLine && _newline(level);
  }
  function deindent(withNewLine = true) {
    const level = --_context.indentLevel;
    withNewLine && _newline(level);
  }
  function newline2() {
    _newline(_context.indentLevel);
  }
  const helper = (key) => `_${key}`;
  const needIndent = () => _context.needIndent;
  return {
    context,
    push,
    indent: indent2,
    deindent,
    newline: newline2,
    helper,
    needIndent
  };
}
function generateLinkedNode(generator, node) {
  const { helper } = generator;
  generator.push(`${helper(
    "linked"
    /* HelperNameMap.LINKED */
  )}(`);
  generateNode(generator, node.key);
  if (node.modifier) {
    generator.push(`, `);
    generateNode(generator, node.modifier);
    generator.push(`, _type`);
  } else {
    generator.push(`, undefined, _type`);
  }
  generator.push(`)`);
}
function generateMessageNode(generator, node) {
  const { helper, needIndent } = generator;
  generator.push(`${helper(
    "normalize"
    /* HelperNameMap.NORMALIZE */
  )}([`);
  generator.indent(needIndent());
  const length = node.items.length;
  for (let i2 = 0; i2 < length; i2++) {
    generateNode(generator, node.items[i2]);
    if (i2 === length - 1) {
      break;
    }
    generator.push(", ");
  }
  generator.deindent(needIndent());
  generator.push("])");
}
function generatePluralNode(generator, node) {
  const { helper, needIndent } = generator;
  if (node.cases.length > 1) {
    generator.push(`${helper(
      "plural"
      /* HelperNameMap.PLURAL */
    )}([`);
    generator.indent(needIndent());
    const length = node.cases.length;
    for (let i2 = 0; i2 < length; i2++) {
      generateNode(generator, node.cases[i2]);
      if (i2 === length - 1) {
        break;
      }
      generator.push(", ");
    }
    generator.deindent(needIndent());
    generator.push(`])`);
  }
}
function generateResource(generator, node) {
  if (node.body) {
    generateNode(generator, node.body);
  } else {
    generator.push("null");
  }
}
function generateNode(generator, node) {
  const { helper } = generator;
  switch (node.type) {
    case 0:
      generateResource(generator, node);
      break;
    case 1:
      generatePluralNode(generator, node);
      break;
    case 2:
      generateMessageNode(generator, node);
      break;
    case 6:
      generateLinkedNode(generator, node);
      break;
    case 8:
      generator.push(JSON.stringify(node.value), node);
      break;
    case 7:
      generator.push(JSON.stringify(node.value), node);
      break;
    case 5:
      generator.push(`${helper(
        "interpolate"
        /* HelperNameMap.INTERPOLATE */
      )}(${helper(
        "list"
        /* HelperNameMap.LIST */
      )}(${node.index}))`, node);
      break;
    case 4:
      generator.push(`${helper(
        "interpolate"
        /* HelperNameMap.INTERPOLATE */
      )}(${helper(
        "named"
        /* HelperNameMap.NAMED */
      )}(${JSON.stringify(node.key)}))`, node);
      break;
    case 9:
      generator.push(JSON.stringify(node.value), node);
      break;
    case 3:
      generator.push(JSON.stringify(node.value), node);
      break;
    default: {
      throw createCompileError(CompileErrorCodes.UNHANDLED_CODEGEN_NODE_TYPE, null, {
        domain: ERROR_DOMAIN,
        args: [node.type]
      });
    }
  }
}
const generate$1 = (ast2, options2 = {}) => {
  const mode = isString$1(options2.mode) ? options2.mode : "normal";
  const filename = isString$1(options2.filename) ? options2.filename : "message.intl";
  !!options2.sourceMap;
  const breakLineCode = options2.breakLineCode != null ? options2.breakLineCode : mode === "arrow" ? ";" : "\n";
  const needIndent = options2.needIndent ? options2.needIndent : mode !== "arrow";
  const helpers2 = ast2.helpers || [];
  const generator = createCodeGenerator(ast2, {
    filename,
    breakLineCode,
    needIndent
  });
  generator.push(mode === "normal" ? `function __msg__ (ctx) {` : `(ctx) => {`);
  generator.indent(needIndent);
  if (helpers2.length > 0) {
    generator.push(`const { ${join$1(helpers2.map((s2) => `${s2}: _${s2}`), ", ")} } = ctx`);
    generator.newline();
  }
  generator.push(`return `);
  generateNode(generator, ast2);
  generator.deindent(needIndent);
  generator.push(`}`);
  delete ast2.helpers;
  const { code: code2, map: map2 } = generator.context();
  return {
    ast: ast2,
    code: code2,
    map: map2 ? map2.toJSON() : void 0
    // eslint-disable-line @typescript-eslint/no-explicit-any
  };
};
function baseCompile$2(source, options2 = {}) {
  const assignedOptions = assign$1({}, options2);
  const jit = !!assignedOptions.jit;
  const enalbeMinify = !!assignedOptions.minify;
  const enambeOptimize = assignedOptions.optimize == null ? true : assignedOptions.optimize;
  const parser = createParser(assignedOptions);
  const ast2 = parser.parse(source);
  if (!jit) {
    transform$1(ast2, assignedOptions);
    return generate$1(ast2, assignedOptions);
  } else {
    enambeOptimize && optimize(ast2);
    enalbeMinify && minify(ast2);
    return { ast: ast2, code: "" };
  }
}
/*!
  * core-base v9.14.5
  * (c) 2025 kazuya kawaguchi
  * Released under the MIT License.
  */
function initFeatureFlags$1() {
  if (typeof __INTLIFY_PROD_DEVTOOLS__ !== "boolean") {
    getGlobalThis().__INTLIFY_PROD_DEVTOOLS__ = false;
  }
}
function isMessageAST(val) {
  return isObject$3(val) && resolveType(val) === 0 && (hasOwn(val, "b") || hasOwn(val, "body"));
}
const PROPS_BODY = ["b", "body"];
function resolveBody(node) {
  return resolveProps(node, PROPS_BODY);
}
const PROPS_CASES = ["c", "cases"];
function resolveCases(node) {
  return resolveProps(node, PROPS_CASES, []);
}
const PROPS_STATIC = ["s", "static"];
function resolveStatic(node) {
  return resolveProps(node, PROPS_STATIC);
}
const PROPS_ITEMS = ["i", "items"];
function resolveItems(node) {
  return resolveProps(node, PROPS_ITEMS, []);
}
const PROPS_TYPE = ["t", "type"];
function resolveType(node) {
  return resolveProps(node, PROPS_TYPE);
}
const PROPS_VALUE = ["v", "value"];
function resolveValue$1(node, type2) {
  const resolved = resolveProps(node, PROPS_VALUE);
  if (resolved != null) {
    return resolved;
  } else {
    throw createUnhandleNodeError(type2);
  }
}
const PROPS_MODIFIER = ["m", "modifier"];
function resolveLinkedModifier(node) {
  return resolveProps(node, PROPS_MODIFIER);
}
const PROPS_KEY = ["k", "key"];
function resolveLinkedKey(node) {
  const resolved = resolveProps(node, PROPS_KEY);
  if (resolved) {
    return resolved;
  } else {
    throw createUnhandleNodeError(
      6
      /* NodeTypes.Linked */
    );
  }
}
function resolveProps(node, props, defaultValue) {
  for (let i2 = 0; i2 < props.length; i2++) {
    const prop2 = props[i2];
    if (hasOwn(node, prop2) && node[prop2] != null) {
      return node[prop2];
    }
  }
  return defaultValue;
}
const AST_NODE_PROPS_KEYS = [
  ...PROPS_BODY,
  ...PROPS_CASES,
  ...PROPS_STATIC,
  ...PROPS_ITEMS,
  ...PROPS_KEY,
  ...PROPS_MODIFIER,
  ...PROPS_VALUE,
  ...PROPS_TYPE
];
function createUnhandleNodeError(type2) {
  return new Error(`unhandled node type: ${type2}`);
}
const pathStateMachine = [];
pathStateMachine[
  0
  /* States.BEFORE_PATH */
] = {
  [
    "w"
    /* PathCharTypes.WORKSPACE */
  ]: [
    0
    /* States.BEFORE_PATH */
  ],
  [
    "i"
    /* PathCharTypes.IDENT */
  ]: [
    3,
    0
    /* Actions.APPEND */
  ],
  [
    "["
    /* PathCharTypes.LEFT_BRACKET */
  ]: [
    4
    /* States.IN_SUB_PATH */
  ],
  [
    "o"
    /* PathCharTypes.END_OF_FAIL */
  ]: [
    7
    /* States.AFTER_PATH */
  ]
};
pathStateMachine[
  1
  /* States.IN_PATH */
] = {
  [
    "w"
    /* PathCharTypes.WORKSPACE */
  ]: [
    1
    /* States.IN_PATH */
  ],
  [
    "."
    /* PathCharTypes.DOT */
  ]: [
    2
    /* States.BEFORE_IDENT */
  ],
  [
    "["
    /* PathCharTypes.LEFT_BRACKET */
  ]: [
    4
    /* States.IN_SUB_PATH */
  ],
  [
    "o"
    /* PathCharTypes.END_OF_FAIL */
  ]: [
    7
    /* States.AFTER_PATH */
  ]
};
pathStateMachine[
  2
  /* States.BEFORE_IDENT */
] = {
  [
    "w"
    /* PathCharTypes.WORKSPACE */
  ]: [
    2
    /* States.BEFORE_IDENT */
  ],
  [
    "i"
    /* PathCharTypes.IDENT */
  ]: [
    3,
    0
    /* Actions.APPEND */
  ],
  [
    "0"
    /* PathCharTypes.ZERO */
  ]: [
    3,
    0
    /* Actions.APPEND */
  ]
};
pathStateMachine[
  3
  /* States.IN_IDENT */
] = {
  [
    "i"
    /* PathCharTypes.IDENT */
  ]: [
    3,
    0
    /* Actions.APPEND */
  ],
  [
    "0"
    /* PathCharTypes.ZERO */
  ]: [
    3,
    0
    /* Actions.APPEND */
  ],
  [
    "w"
    /* PathCharTypes.WORKSPACE */
  ]: [
    1,
    1
    /* Actions.PUSH */
  ],
  [
    "."
    /* PathCharTypes.DOT */
  ]: [
    2,
    1
    /* Actions.PUSH */
  ],
  [
    "["
    /* PathCharTypes.LEFT_BRACKET */
  ]: [
    4,
    1
    /* Actions.PUSH */
  ],
  [
    "o"
    /* PathCharTypes.END_OF_FAIL */
  ]: [
    7,
    1
    /* Actions.PUSH */
  ]
};
pathStateMachine[
  4
  /* States.IN_SUB_PATH */
] = {
  [
    "'"
    /* PathCharTypes.SINGLE_QUOTE */
  ]: [
    5,
    0
    /* Actions.APPEND */
  ],
  [
    '"'
    /* PathCharTypes.DOUBLE_QUOTE */
  ]: [
    6,
    0
    /* Actions.APPEND */
  ],
  [
    "["
    /* PathCharTypes.LEFT_BRACKET */
  ]: [
    4,
    2
    /* Actions.INC_SUB_PATH_DEPTH */
  ],
  [
    "]"
    /* PathCharTypes.RIGHT_BRACKET */
  ]: [
    1,
    3
    /* Actions.PUSH_SUB_PATH */
  ],
  [
    "o"
    /* PathCharTypes.END_OF_FAIL */
  ]: 8,
  [
    "l"
    /* PathCharTypes.ELSE */
  ]: [
    4,
    0
    /* Actions.APPEND */
  ]
};
pathStateMachine[
  5
  /* States.IN_SINGLE_QUOTE */
] = {
  [
    "'"
    /* PathCharTypes.SINGLE_QUOTE */
  ]: [
    4,
    0
    /* Actions.APPEND */
  ],
  [
    "o"
    /* PathCharTypes.END_OF_FAIL */
  ]: 8,
  [
    "l"
    /* PathCharTypes.ELSE */
  ]: [
    5,
    0
    /* Actions.APPEND */
  ]
};
pathStateMachine[
  6
  /* States.IN_DOUBLE_QUOTE */
] = {
  [
    '"'
    /* PathCharTypes.DOUBLE_QUOTE */
  ]: [
    4,
    0
    /* Actions.APPEND */
  ],
  [
    "o"
    /* PathCharTypes.END_OF_FAIL */
  ]: 8,
  [
    "l"
    /* PathCharTypes.ELSE */
  ]: [
    6,
    0
    /* Actions.APPEND */
  ]
};
const literalValueRE = /^\s?(?:true|false|-?[\d.]+|'[^']*'|"[^"]*")\s?$/;
function isLiteral(exp) {
  return literalValueRE.test(exp);
}
function stripQuotes(str) {
  const a2 = str.charCodeAt(0);
  const b2 = str.charCodeAt(str.length - 1);
  return a2 === b2 && (a2 === 34 || a2 === 39) ? str.slice(1, -1) : str;
}
function getPathCharType(ch) {
  if (ch === void 0 || ch === null) {
    return "o";
  }
  const code2 = ch.charCodeAt(0);
  switch (code2) {
    case 91:
    // [
    case 93:
    // ]
    case 46:
    // .
    case 34:
    // "
    case 39:
      return ch;
    case 95:
    // _
    case 36:
    // $
    case 45:
      return "i";
    case 9:
    // Tab (HT)
    case 10:
    // Newline (LF)
    case 13:
    // Return (CR)
    case 160:
    // No-break space (NBSP)
    case 65279:
    // Byte Order Mark (BOM)
    case 8232:
    // Line Separator (LS)
    case 8233:
      return "w";
  }
  return "i";
}
function formatSubPath(path) {
  const trimmed = path.trim();
  if (path.charAt(0) === "0" && isNaN(parseInt(path))) {
    return false;
  }
  return isLiteral(trimmed) ? stripQuotes(trimmed) : "*" + trimmed;
}
function parse$4(path) {
  const keys = [];
  let index = -1;
  let mode = 0;
  let subPathDepth = 0;
  let c2;
  let key;
  let newChar;
  let type2;
  let transition;
  let action;
  let typeMap;
  const actions = [];
  actions[
    0
    /* Actions.APPEND */
  ] = () => {
    if (key === void 0) {
      key = newChar;
    } else {
      key += newChar;
    }
  };
  actions[
    1
    /* Actions.PUSH */
  ] = () => {
    if (key !== void 0) {
      keys.push(key);
      key = void 0;
    }
  };
  actions[
    2
    /* Actions.INC_SUB_PATH_DEPTH */
  ] = () => {
    actions[
      0
      /* Actions.APPEND */
    ]();
    subPathDepth++;
  };
  actions[
    3
    /* Actions.PUSH_SUB_PATH */
  ] = () => {
    if (subPathDepth > 0) {
      subPathDepth--;
      mode = 4;
      actions[
        0
        /* Actions.APPEND */
      ]();
    } else {
      subPathDepth = 0;
      if (key === void 0) {
        return false;
      }
      key = formatSubPath(key);
      if (key === false) {
        return false;
      } else {
        actions[
          1
          /* Actions.PUSH */
        ]();
      }
    }
  };
  function maybeUnescapeQuote() {
    const nextChar = path[index + 1];
    if (mode === 5 && nextChar === "'" || mode === 6 && nextChar === '"') {
      index++;
      newChar = "\\" + nextChar;
      actions[
        0
        /* Actions.APPEND */
      ]();
      return true;
    }
  }
  while (mode !== null) {
    index++;
    c2 = path[index];
    if (c2 === "\\" && maybeUnescapeQuote()) {
      continue;
    }
    type2 = getPathCharType(c2);
    typeMap = pathStateMachine[mode];
    transition = typeMap[type2] || typeMap[
      "l"
      /* PathCharTypes.ELSE */
    ] || 8;
    if (transition === 8) {
      return;
    }
    mode = transition[0];
    if (transition[1] !== void 0) {
      action = actions[transition[1]];
      if (action) {
        newChar = c2;
        if (action() === false) {
          return;
        }
      }
    }
    if (mode === 7) {
      return keys;
    }
  }
}
const cache = /* @__PURE__ */ new Map();
function resolveWithKeyValue(obj, path) {
  return isObject$3(obj) ? obj[path] : null;
}
function resolveValue(obj, path) {
  if (!isObject$3(obj)) {
    return null;
  }
  let hit = cache.get(path);
  if (!hit) {
    hit = parse$4(path);
    if (hit) {
      cache.set(path, hit);
    }
  }
  if (!hit) {
    return null;
  }
  const len = hit.length;
  let last2 = obj;
  let i2 = 0;
  while (i2 < len) {
    const key = hit[i2];
    if (AST_NODE_PROPS_KEYS.includes(key) && isMessageAST(last2)) {
      return null;
    }
    const val = last2[key];
    if (val === void 0) {
      return null;
    }
    if (isFunction(last2)) {
      return null;
    }
    last2 = val;
    i2++;
  }
  return last2;
}
const DEFAULT_MODIFIER = (str) => str;
const DEFAULT_MESSAGE = (ctx) => "";
const DEFAULT_MESSAGE_DATA_TYPE = "text";
const DEFAULT_NORMALIZE = (values) => values.length === 0 ? "" : join$2(values);
const DEFAULT_INTERPOLATE = toDisplayString;
function pluralDefault(choice, choicesLength) {
  choice = Math.abs(choice);
  if (choicesLength === 2) {
    return choice ? choice > 1 ? 1 : 0 : 1;
  }
  return choice ? Math.min(choice, 2) : 0;
}
function getPluralIndex(options2) {
  const index = isNumber(options2.pluralIndex) ? options2.pluralIndex : -1;
  return options2.named && (isNumber(options2.named.count) || isNumber(options2.named.n)) ? isNumber(options2.named.count) ? options2.named.count : isNumber(options2.named.n) ? options2.named.n : index : index;
}
function normalizeNamed(pluralIndex, props) {
  if (!props.count) {
    props.count = pluralIndex;
  }
  if (!props.n) {
    props.n = pluralIndex;
  }
}
function createMessageContext(options2 = {}) {
  const locale = options2.locale;
  const pluralIndex = getPluralIndex(options2);
  const pluralRule = isObject$3(options2.pluralRules) && isString$2(locale) && isFunction(options2.pluralRules[locale]) ? options2.pluralRules[locale] : pluralDefault;
  const orgPluralRule = isObject$3(options2.pluralRules) && isString$2(locale) && isFunction(options2.pluralRules[locale]) ? pluralDefault : void 0;
  const plural = (messages) => {
    return messages[pluralRule(pluralIndex, messages.length, orgPluralRule)];
  };
  const _list = options2.list || [];
  const list2 = (index) => _list[index];
  const _named = options2.named || create();
  isNumber(options2.pluralIndex) && normalizeNamed(pluralIndex, _named);
  const named = (key) => _named[key];
  function message(key) {
    const msg = isFunction(options2.messages) ? options2.messages(key) : isObject$3(options2.messages) ? options2.messages[key] : false;
    return !msg ? options2.parent ? options2.parent.message(key) : DEFAULT_MESSAGE : msg;
  }
  const _modifier = (name2) => options2.modifiers ? options2.modifiers[name2] : DEFAULT_MODIFIER;
  const normalize2 = isPlainObject$1(options2.processor) && isFunction(options2.processor.normalize) ? options2.processor.normalize : DEFAULT_NORMALIZE;
  const interpolate = isPlainObject$1(options2.processor) && isFunction(options2.processor.interpolate) ? options2.processor.interpolate : DEFAULT_INTERPOLATE;
  const type2 = isPlainObject$1(options2.processor) && isString$2(options2.processor.type) ? options2.processor.type : DEFAULT_MESSAGE_DATA_TYPE;
  const linked = (key, ...args) => {
    const [arg1, arg2] = args;
    let type22 = "text";
    let modifier = "";
    if (args.length === 1) {
      if (isObject$3(arg1)) {
        modifier = arg1.modifier || modifier;
        type22 = arg1.type || type22;
      } else if (isString$2(arg1)) {
        modifier = arg1 || modifier;
      }
    } else if (args.length === 2) {
      if (isString$2(arg1)) {
        modifier = arg1 || modifier;
      }
      if (isString$2(arg2)) {
        type22 = arg2 || type22;
      }
    }
    const ret = message(key)(ctx);
    const msg = (
      // The message in vnode resolved with linked are returned as an array by processor.nomalize
      type22 === "vnode" && isArray$1(ret) && modifier ? ret[0] : ret
    );
    return modifier ? _modifier(modifier)(msg, type22) : msg;
  };
  const ctx = {
    [
      "list"
      /* HelperNameMap.LIST */
    ]: list2,
    [
      "named"
      /* HelperNameMap.NAMED */
    ]: named,
    [
      "plural"
      /* HelperNameMap.PLURAL */
    ]: plural,
    [
      "linked"
      /* HelperNameMap.LINKED */
    ]: linked,
    [
      "message"
      /* HelperNameMap.MESSAGE */
    ]: message,
    [
      "type"
      /* HelperNameMap.TYPE */
    ]: type2,
    [
      "interpolate"
      /* HelperNameMap.INTERPOLATE */
    ]: interpolate,
    [
      "normalize"
      /* HelperNameMap.NORMALIZE */
    ]: normalize2,
    [
      "values"
      /* HelperNameMap.VALUES */
    ]: assign$2(create(), _list, _named)
  };
  return ctx;
}
let devtools = null;
function setDevToolsHook(hook) {
  devtools = hook;
}
function initI18nDevTools(i18n, version2, meta) {
  devtools && devtools.emit("i18n:init", {
    timestamp: Date.now(),
    i18n,
    version: version2,
    meta
  });
}
const translateDevTools = /* @__PURE__ */ createDevToolsHook(
  "function:translate"
  /* IntlifyDevToolsHooks.FunctionTranslate */
);
function createDevToolsHook(hook) {
  return (payloads) => devtools && devtools.emit(hook, payloads);
}
const code$1$1 = CompileWarnCodes.__EXTEND_POINT__;
const inc$1$1 = incrementer(code$1$1);
const CoreWarnCodes = {
  // 2
  FALLBACK_TO_TRANSLATE: inc$1$1(),
  // 3
  CANNOT_FORMAT_NUMBER: inc$1$1(),
  // 4
  FALLBACK_TO_NUMBER_FORMAT: inc$1$1(),
  // 5
  CANNOT_FORMAT_DATE: inc$1$1(),
  // 6
  FALLBACK_TO_DATE_FORMAT: inc$1$1(),
  // 7
  EXPERIMENTAL_CUSTOM_MESSAGE_COMPILER: inc$1$1(),
  // 8
  __EXTEND_POINT__: inc$1$1()
  // 9
};
const code$3 = CompileErrorCodes.__EXTEND_POINT__;
const inc$2 = incrementer(code$3);
const CoreErrorCodes = {
  INVALID_ARGUMENT: code$3,
  // 17
  INVALID_DATE_ARGUMENT: inc$2(),
  // 18
  INVALID_ISO_DATE_ARGUMENT: inc$2(),
  // 19
  NOT_SUPPORT_NON_STRING_MESSAGE: inc$2(),
  // 20
  NOT_SUPPORT_LOCALE_PROMISE_VALUE: inc$2(),
  // 21
  NOT_SUPPORT_LOCALE_ASYNC_FUNCTION: inc$2(),
  // 22
  NOT_SUPPORT_LOCALE_TYPE: inc$2(),
  // 23
  __EXTEND_POINT__: inc$2()
  // 24
};
function createCoreError(code2) {
  return createCompileError(code2, null, void 0);
}
function getLocale(context, options2) {
  return options2.locale != null ? resolveLocale(options2.locale) : resolveLocale(context.locale);
}
let _resolveLocale;
function resolveLocale(locale) {
  if (isString$2(locale)) {
    return locale;
  } else {
    if (isFunction(locale)) {
      if (locale.resolvedOnce && _resolveLocale != null) {
        return _resolveLocale;
      } else if (locale.constructor.name === "Function") {
        const resolve3 = locale();
        if (isPromise(resolve3)) {
          throw createCoreError(CoreErrorCodes.NOT_SUPPORT_LOCALE_PROMISE_VALUE);
        }
        return _resolveLocale = resolve3;
      } else {
        throw createCoreError(CoreErrorCodes.NOT_SUPPORT_LOCALE_ASYNC_FUNCTION);
      }
    } else {
      throw createCoreError(CoreErrorCodes.NOT_SUPPORT_LOCALE_TYPE);
    }
  }
}
function fallbackWithSimple(ctx, fallback, start) {
  return [.../* @__PURE__ */ new Set([
    start,
    ...isArray$1(fallback) ? fallback : isObject$3(fallback) ? Object.keys(fallback) : isString$2(fallback) ? [fallback] : [start]
  ])];
}
function fallbackWithLocaleChain(ctx, fallback, start) {
  const startLocale = isString$2(start) ? start : DEFAULT_LOCALE;
  const context = ctx;
  if (!context.__localeChainCache) {
    context.__localeChainCache = /* @__PURE__ */ new Map();
  }
  let chain = context.__localeChainCache.get(startLocale);
  if (!chain) {
    chain = [];
    let block2 = [start];
    while (isArray$1(block2)) {
      block2 = appendBlockToChain(chain, block2, fallback);
    }
    const defaults = isArray$1(fallback) || !isPlainObject$1(fallback) ? fallback : fallback["default"] ? fallback["default"] : null;
    block2 = isString$2(defaults) ? [defaults] : defaults;
    if (isArray$1(block2)) {
      appendBlockToChain(chain, block2, false);
    }
    context.__localeChainCache.set(startLocale, chain);
  }
  return chain;
}
function appendBlockToChain(chain, block2, blocks) {
  let follow = true;
  for (let i2 = 0; i2 < block2.length && isBoolean(follow); i2++) {
    const locale = block2[i2];
    if (isString$2(locale)) {
      follow = appendLocaleToChain(chain, block2[i2], blocks);
    }
  }
  return follow;
}
function appendLocaleToChain(chain, locale, blocks) {
  let follow;
  const tokens = locale.split("-");
  do {
    const target2 = tokens.join("-");
    follow = appendItemToChain(chain, target2, blocks);
    tokens.splice(-1, 1);
  } while (tokens.length && follow === true);
  return follow;
}
function appendItemToChain(chain, target2, blocks) {
  let follow = false;
  if (!chain.includes(target2)) {
    follow = true;
    if (target2) {
      follow = target2[target2.length - 1] !== "!";
      const locale = target2.replace(/!/g, "");
      chain.push(locale);
      if ((isArray$1(blocks) || isPlainObject$1(blocks)) && blocks[locale]) {
        follow = blocks[locale];
      }
    }
  }
  return follow;
}
const VERSION$1 = "9.14.5";
const NOT_REOSLVED = -1;
const DEFAULT_LOCALE = "en-US";
const MISSING_RESOLVE_VALUE = "";
const capitalize$1 = (str) => `${str.charAt(0).toLocaleUpperCase()}${str.substr(1)}`;
function getDefaultLinkedModifiers() {
  return {
    upper: (val, type2) => {
      return type2 === "text" && isString$2(val) ? val.toUpperCase() : type2 === "vnode" && isObject$3(val) && "__v_isVNode" in val ? val.children.toUpperCase() : val;
    },
    lower: (val, type2) => {
      return type2 === "text" && isString$2(val) ? val.toLowerCase() : type2 === "vnode" && isObject$3(val) && "__v_isVNode" in val ? val.children.toLowerCase() : val;
    },
    capitalize: (val, type2) => {
      return type2 === "text" && isString$2(val) ? capitalize$1(val) : type2 === "vnode" && isObject$3(val) && "__v_isVNode" in val ? capitalize$1(val.children) : val;
    }
  };
}
let _compiler;
function registerMessageCompiler(compiler) {
  _compiler = compiler;
}
let _resolver;
function registerMessageResolver(resolver) {
  _resolver = resolver;
}
let _fallbacker;
function registerLocaleFallbacker(fallbacker) {
  _fallbacker = fallbacker;
}
let _additionalMeta = null;
const setAdditionalMeta = /* @__NO_SIDE_EFFECTS__ */ (meta) => {
  _additionalMeta = meta;
};
const getAdditionalMeta = /* @__NO_SIDE_EFFECTS__ */ () => _additionalMeta;
let _fallbackContext = null;
const setFallbackContext = (context) => {
  _fallbackContext = context;
};
const getFallbackContext = () => _fallbackContext;
let _cid = 0;
function createCoreContext(options2 = {}) {
  const onWarn = isFunction(options2.onWarn) ? options2.onWarn : warn;
  const version2 = isString$2(options2.version) ? options2.version : VERSION$1;
  const locale = isString$2(options2.locale) || isFunction(options2.locale) ? options2.locale : DEFAULT_LOCALE;
  const _locale = isFunction(locale) ? DEFAULT_LOCALE : locale;
  const fallbackLocale = isArray$1(options2.fallbackLocale) || isPlainObject$1(options2.fallbackLocale) || isString$2(options2.fallbackLocale) || options2.fallbackLocale === false ? options2.fallbackLocale : _locale;
  const messages = isPlainObject$1(options2.messages) ? options2.messages : createResources(_locale);
  const datetimeFormats = isPlainObject$1(options2.datetimeFormats) ? options2.datetimeFormats : createResources(_locale);
  const numberFormats = isPlainObject$1(options2.numberFormats) ? options2.numberFormats : createResources(_locale);
  const modifiers = assign$2(create(), options2.modifiers, getDefaultLinkedModifiers());
  const pluralRules = options2.pluralRules || create();
  const missing = isFunction(options2.missing) ? options2.missing : null;
  const missingWarn = isBoolean(options2.missingWarn) || isRegExp(options2.missingWarn) ? options2.missingWarn : true;
  const fallbackWarn = isBoolean(options2.fallbackWarn) || isRegExp(options2.fallbackWarn) ? options2.fallbackWarn : true;
  const fallbackFormat = !!options2.fallbackFormat;
  const unresolving = !!options2.unresolving;
  const postTranslation = isFunction(options2.postTranslation) ? options2.postTranslation : null;
  const processor = isPlainObject$1(options2.processor) ? options2.processor : null;
  const warnHtmlMessage = isBoolean(options2.warnHtmlMessage) ? options2.warnHtmlMessage : true;
  const escapeParameter = !!options2.escapeParameter;
  const messageCompiler = isFunction(options2.messageCompiler) ? options2.messageCompiler : _compiler;
  const messageResolver = isFunction(options2.messageResolver) ? options2.messageResolver : _resolver || resolveWithKeyValue;
  const localeFallbacker = isFunction(options2.localeFallbacker) ? options2.localeFallbacker : _fallbacker || fallbackWithSimple;
  const fallbackContext = isObject$3(options2.fallbackContext) ? options2.fallbackContext : void 0;
  const internalOptions = options2;
  const __datetimeFormatters = isObject$3(internalOptions.__datetimeFormatters) ? internalOptions.__datetimeFormatters : /* @__PURE__ */ new Map();
  const __numberFormatters = isObject$3(internalOptions.__numberFormatters) ? internalOptions.__numberFormatters : /* @__PURE__ */ new Map();
  const __meta = isObject$3(internalOptions.__meta) ? internalOptions.__meta : {};
  _cid++;
  const context = {
    version: version2,
    cid: _cid,
    locale,
    fallbackLocale,
    messages,
    modifiers,
    pluralRules,
    missing,
    missingWarn,
    fallbackWarn,
    fallbackFormat,
    unresolving,
    postTranslation,
    processor,
    warnHtmlMessage,
    escapeParameter,
    messageCompiler,
    messageResolver,
    localeFallbacker,
    fallbackContext,
    onWarn,
    __meta
  };
  {
    context.datetimeFormats = datetimeFormats;
    context.numberFormats = numberFormats;
    context.__datetimeFormatters = __datetimeFormatters;
    context.__numberFormatters = __numberFormatters;
  }
  if (__INTLIFY_PROD_DEVTOOLS__) {
    initI18nDevTools(context, version2, __meta);
  }
  return context;
}
const createResources = (locale) => ({ [locale]: create() });
function handleMissing(context, key, locale, missingWarn, type2) {
  const { missing, onWarn } = context;
  if (missing !== null) {
    const ret = missing(context, locale, key, type2);
    return isString$2(ret) ? ret : key;
  } else {
    return key;
  }
}
function updateFallbackLocale(ctx, locale, fallback) {
  const context = ctx;
  context.__localeChainCache = /* @__PURE__ */ new Map();
  ctx.localeFallbacker(ctx, fallback, locale);
}
function isAlmostSameLocale(locale, compareLocale) {
  if (locale === compareLocale)
    return false;
  return locale.split("-")[0] === compareLocale.split("-")[0];
}
function isImplicitFallback(targetLocale, locales) {
  const index = locales.indexOf(targetLocale);
  if (index === -1) {
    return false;
  }
  for (let i2 = index + 1; i2 < locales.length; i2++) {
    if (isAlmostSameLocale(targetLocale, locales[i2])) {
      return true;
    }
  }
  return false;
}
function format$1(ast2) {
  const msg = (ctx) => formatParts(ctx, ast2);
  return msg;
}
function formatParts(ctx, ast2) {
  const body = resolveBody(ast2);
  if (body == null) {
    throw createUnhandleNodeError(
      0
      /* NodeTypes.Resource */
    );
  }
  const type2 = resolveType(body);
  if (type2 === 1) {
    const plural = body;
    const cases = resolveCases(plural);
    return ctx.plural(cases.reduce((messages, c2) => [
      ...messages,
      formatMessageParts(ctx, c2)
    ], []));
  } else {
    return formatMessageParts(ctx, body);
  }
}
function formatMessageParts(ctx, node) {
  const static_ = resolveStatic(node);
  if (static_ != null) {
    return ctx.type === "text" ? static_ : ctx.normalize([static_]);
  } else {
    const messages = resolveItems(node).reduce((acm, c2) => [...acm, formatMessagePart(ctx, c2)], []);
    return ctx.normalize(messages);
  }
}
function formatMessagePart(ctx, node) {
  const type2 = resolveType(node);
  switch (type2) {
    case 3: {
      return resolveValue$1(node, type2);
    }
    case 9: {
      return resolveValue$1(node, type2);
    }
    case 4: {
      const named = node;
      if (hasOwn(named, "k") && named.k) {
        return ctx.interpolate(ctx.named(named.k));
      }
      if (hasOwn(named, "key") && named.key) {
        return ctx.interpolate(ctx.named(named.key));
      }
      throw createUnhandleNodeError(type2);
    }
    case 5: {
      const list2 = node;
      if (hasOwn(list2, "i") && isNumber(list2.i)) {
        return ctx.interpolate(ctx.list(list2.i));
      }
      if (hasOwn(list2, "index") && isNumber(list2.index)) {
        return ctx.interpolate(ctx.list(list2.index));
      }
      throw createUnhandleNodeError(type2);
    }
    case 6: {
      const linked = node;
      const modifier = resolveLinkedModifier(linked);
      const key = resolveLinkedKey(linked);
      return ctx.linked(formatMessagePart(ctx, key), modifier ? formatMessagePart(ctx, modifier) : void 0, ctx.type);
    }
    case 7: {
      return resolveValue$1(node, type2);
    }
    case 8: {
      return resolveValue$1(node, type2);
    }
    default:
      throw new Error(`unhandled node on format message part: ${type2}`);
  }
}
const defaultOnCacheKey = (message) => message;
let compileCache = create();
function baseCompile$1(message, options2 = {}) {
  let detectError = false;
  const onError = options2.onError || defaultOnError$1;
  options2.onError = (err) => {
    detectError = true;
    onError(err);
  };
  return { ...baseCompile$2(message, options2), detectError };
}
function compile$1(message, context) {
  if (isString$2(message)) {
    isBoolean(context.warnHtmlMessage) ? context.warnHtmlMessage : true;
    const onCacheKey = context.onCacheKey || defaultOnCacheKey;
    const cacheKey = onCacheKey(message);
    const cached = compileCache[cacheKey];
    if (cached) {
      return cached;
    }
    const { ast: ast2, detectError } = baseCompile$1(message, {
      ...context,
      location: false,
      jit: true
    });
    const msg = format$1(ast2);
    return !detectError ? compileCache[cacheKey] = msg : msg;
  } else {
    const cacheKey = message.cacheKey;
    if (cacheKey) {
      const cached = compileCache[cacheKey];
      if (cached) {
        return cached;
      }
      return compileCache[cacheKey] = format$1(message);
    } else {
      return format$1(message);
    }
  }
}
const NOOP_MESSAGE_FUNCTION = () => "";
const isMessageFunction = (val) => isFunction(val);
function translate(context, ...args) {
  const { fallbackFormat, postTranslation, unresolving, messageCompiler, fallbackLocale, messages } = context;
  const [key, options2] = parseTranslateArgs(...args);
  const missingWarn = isBoolean(options2.missingWarn) ? options2.missingWarn : context.missingWarn;
  const fallbackWarn = isBoolean(options2.fallbackWarn) ? options2.fallbackWarn : context.fallbackWarn;
  const escapeParameter = isBoolean(options2.escapeParameter) ? options2.escapeParameter : context.escapeParameter;
  const resolvedMessage = !!options2.resolvedMessage;
  const defaultMsgOrKey = isString$2(options2.default) || isBoolean(options2.default) ? !isBoolean(options2.default) ? options2.default : !messageCompiler ? () => key : key : fallbackFormat ? !messageCompiler ? () => key : key : "";
  const enableDefaultMsg = fallbackFormat || defaultMsgOrKey !== "";
  const locale = getLocale(context, options2);
  escapeParameter && escapeParams(options2);
  let [formatScope, targetLocale, message] = !resolvedMessage ? resolveMessageFormat(context, key, locale, fallbackLocale, fallbackWarn, missingWarn) : [
    key,
    locale,
    messages[locale] || create()
  ];
  let format2 = formatScope;
  let cacheBaseKey = key;
  if (!resolvedMessage && !(isString$2(format2) || isMessageAST(format2) || isMessageFunction(format2))) {
    if (enableDefaultMsg) {
      format2 = defaultMsgOrKey;
      cacheBaseKey = format2;
    }
  }
  if (!resolvedMessage && (!(isString$2(format2) || isMessageAST(format2) || isMessageFunction(format2)) || !isString$2(targetLocale))) {
    return unresolving ? NOT_REOSLVED : key;
  }
  let occurred = false;
  const onError = () => {
    occurred = true;
  };
  const msg = !isMessageFunction(format2) ? compileMessageFormat(context, key, targetLocale, format2, cacheBaseKey, onError) : format2;
  if (occurred) {
    return format2;
  }
  const ctxOptions = getMessageContextOptions(context, targetLocale, message, options2);
  const msgContext = createMessageContext(ctxOptions);
  const messaged = evaluateMessage(context, msg, msgContext);
  let ret = postTranslation ? postTranslation(messaged, key) : messaged;
  if (escapeParameter && isString$2(ret)) {
    ret = sanitizeTranslatedHtml(ret);
  }
  if (__INTLIFY_PROD_DEVTOOLS__) {
    const payloads = {
      timestamp: Date.now(),
      key: isString$2(key) ? key : isMessageFunction(format2) ? format2.key : "",
      locale: targetLocale || (isMessageFunction(format2) ? format2.locale : ""),
      format: isString$2(format2) ? format2 : isMessageFunction(format2) ? format2.source : "",
      message: ret
    };
    payloads.meta = assign$2({}, context.__meta, /* @__PURE__ */ getAdditionalMeta() || {});
    translateDevTools(payloads);
  }
  return ret;
}
function escapeParams(options2) {
  if (isArray$1(options2.list)) {
    options2.list = options2.list.map((item) => isString$2(item) ? escapeHtml$1(item) : item);
  } else if (isObject$3(options2.named)) {
    Object.keys(options2.named).forEach((key) => {
      if (isString$2(options2.named[key])) {
        options2.named[key] = escapeHtml$1(options2.named[key]);
      }
    });
  }
}
function resolveMessageFormat(context, key, locale, fallbackLocale, fallbackWarn, missingWarn) {
  const { messages, onWarn, messageResolver: resolveValue2, localeFallbacker } = context;
  const locales = localeFallbacker(context, fallbackLocale, locale);
  let message = create();
  let targetLocale;
  let format2 = null;
  const type2 = "translate";
  for (let i2 = 0; i2 < locales.length; i2++) {
    targetLocale = locales[i2];
    message = messages[targetLocale] || create();
    if ((format2 = resolveValue2(message, key)) === null) {
      format2 = message[key];
    }
    if (isString$2(format2) || isMessageAST(format2) || isMessageFunction(format2)) {
      break;
    }
    if (!isImplicitFallback(targetLocale, locales)) {
      const missingRet = handleMissing(
        context,
        // eslint-disable-line @typescript-eslint/no-explicit-any
        key,
        targetLocale,
        missingWarn,
        type2
      );
      if (missingRet !== key) {
        format2 = missingRet;
      }
    }
  }
  return [format2, targetLocale, message];
}
function compileMessageFormat(context, key, targetLocale, format2, cacheBaseKey, onError) {
  const { messageCompiler, warnHtmlMessage } = context;
  if (isMessageFunction(format2)) {
    const msg2 = format2;
    msg2.locale = msg2.locale || targetLocale;
    msg2.key = msg2.key || key;
    return msg2;
  }
  if (messageCompiler == null) {
    const msg2 = (() => format2);
    msg2.locale = targetLocale;
    msg2.key = key;
    return msg2;
  }
  const msg = messageCompiler(format2, getCompileContext(context, targetLocale, cacheBaseKey, format2, warnHtmlMessage, onError));
  msg.locale = targetLocale;
  msg.key = key;
  msg.source = format2;
  return msg;
}
function evaluateMessage(context, msg, msgCtx) {
  const messaged = msg(msgCtx);
  return messaged;
}
function parseTranslateArgs(...args) {
  const [arg1, arg2, arg3] = args;
  const options2 = create();
  if (!isString$2(arg1) && !isNumber(arg1) && !isMessageFunction(arg1) && !isMessageAST(arg1)) {
    throw createCoreError(CoreErrorCodes.INVALID_ARGUMENT);
  }
  const key = isNumber(arg1) ? String(arg1) : isMessageFunction(arg1) ? arg1 : arg1;
  if (isNumber(arg2)) {
    options2.plural = arg2;
  } else if (isString$2(arg2)) {
    options2.default = arg2;
  } else if (isPlainObject$1(arg2) && !isEmptyObject(arg2)) {
    options2.named = arg2;
  } else if (isArray$1(arg2)) {
    options2.list = arg2;
  }
  if (isNumber(arg3)) {
    options2.plural = arg3;
  } else if (isString$2(arg3)) {
    options2.default = arg3;
  } else if (isPlainObject$1(arg3)) {
    assign$2(options2, arg3);
  }
  return [key, options2];
}
function getCompileContext(context, locale, key, source, warnHtmlMessage, onError) {
  return {
    locale,
    key,
    warnHtmlMessage,
    onError: (err) => {
      onError && onError(err);
      {
        throw err;
      }
    },
    onCacheKey: (source2) => generateFormatCacheKey(locale, key, source2)
  };
}
function getMessageContextOptions(context, locale, message, options2) {
  const { modifiers, pluralRules, messageResolver: resolveValue2, fallbackLocale, fallbackWarn, missingWarn, fallbackContext } = context;
  const resolveMessage = (key) => {
    let val = resolveValue2(message, key);
    if (val == null && fallbackContext) {
      const [, , message2] = resolveMessageFormat(fallbackContext, key, locale, fallbackLocale, fallbackWarn, missingWarn);
      val = resolveValue2(message2, key);
    }
    if (isString$2(val) || isMessageAST(val)) {
      let occurred = false;
      const onError = () => {
        occurred = true;
      };
      const msg = compileMessageFormat(context, key, locale, val, key, onError);
      return !occurred ? msg : NOOP_MESSAGE_FUNCTION;
    } else if (isMessageFunction(val)) {
      return val;
    } else {
      return NOOP_MESSAGE_FUNCTION;
    }
  };
  const ctxOptions = {
    locale,
    modifiers,
    pluralRules,
    messages: resolveMessage
  };
  if (context.processor) {
    ctxOptions.processor = context.processor;
  }
  if (options2.list) {
    ctxOptions.list = options2.list;
  }
  if (options2.named) {
    ctxOptions.named = options2.named;
  }
  if (isNumber(options2.plural)) {
    ctxOptions.pluralIndex = options2.plural;
  }
  return ctxOptions;
}
function datetime(context, ...args) {
  const { datetimeFormats, unresolving, fallbackLocale, onWarn, localeFallbacker } = context;
  const { __datetimeFormatters } = context;
  const [key, value, options2, overrides] = parseDateTimeArgs(...args);
  const missingWarn = isBoolean(options2.missingWarn) ? options2.missingWarn : context.missingWarn;
  isBoolean(options2.fallbackWarn) ? options2.fallbackWarn : context.fallbackWarn;
  const part2 = !!options2.part;
  const locale = getLocale(context, options2);
  const locales = localeFallbacker(
    context,
    // eslint-disable-line @typescript-eslint/no-explicit-any
    fallbackLocale,
    locale
  );
  if (!isString$2(key) || key === "") {
    return new Intl.DateTimeFormat(locale, overrides).format(value);
  }
  let datetimeFormat = {};
  let targetLocale;
  let format2 = null;
  const type2 = "datetime format";
  for (let i2 = 0; i2 < locales.length; i2++) {
    targetLocale = locales[i2];
    datetimeFormat = datetimeFormats[targetLocale] || {};
    format2 = datetimeFormat[key];
    if (isPlainObject$1(format2))
      break;
    handleMissing(context, key, targetLocale, missingWarn, type2);
  }
  if (!isPlainObject$1(format2) || !isString$2(targetLocale)) {
    return unresolving ? NOT_REOSLVED : key;
  }
  let id2 = `${targetLocale}__${key}`;
  if (!isEmptyObject(overrides)) {
    id2 = `${id2}__${JSON.stringify(overrides)}`;
  }
  let formatter = __datetimeFormatters.get(id2);
  if (!formatter) {
    formatter = new Intl.DateTimeFormat(targetLocale, assign$2({}, format2, overrides));
    __datetimeFormatters.set(id2, formatter);
  }
  return !part2 ? formatter.format(value) : formatter.formatToParts(value);
}
const DATETIME_FORMAT_OPTIONS_KEYS = [
  "localeMatcher",
  "weekday",
  "era",
  "year",
  "month",
  "day",
  "hour",
  "minute",
  "second",
  "timeZoneName",
  "formatMatcher",
  "hour12",
  "timeZone",
  "dateStyle",
  "timeStyle",
  "calendar",
  "dayPeriod",
  "numberingSystem",
  "hourCycle",
  "fractionalSecondDigits"
];
function parseDateTimeArgs(...args) {
  const [arg1, arg2, arg3, arg4] = args;
  const options2 = create();
  let overrides = create();
  let value;
  if (isString$2(arg1)) {
    const matches2 = arg1.match(/(\d{4}-\d{2}-\d{2})(T|\s)?(.*)/);
    if (!matches2) {
      throw createCoreError(CoreErrorCodes.INVALID_ISO_DATE_ARGUMENT);
    }
    const dateTime = matches2[3] ? matches2[3].trim().startsWith("T") ? `${matches2[1].trim()}${matches2[3].trim()}` : `${matches2[1].trim()}T${matches2[3].trim()}` : matches2[1].trim();
    value = new Date(dateTime);
    try {
      value.toISOString();
    } catch (e) {
      throw createCoreError(CoreErrorCodes.INVALID_ISO_DATE_ARGUMENT);
    }
  } else if (isDate(arg1)) {
    if (isNaN(arg1.getTime())) {
      throw createCoreError(CoreErrorCodes.INVALID_DATE_ARGUMENT);
    }
    value = arg1;
  } else if (isNumber(arg1)) {
    value = arg1;
  } else {
    throw createCoreError(CoreErrorCodes.INVALID_ARGUMENT);
  }
  if (isString$2(arg2)) {
    options2.key = arg2;
  } else if (isPlainObject$1(arg2)) {
    Object.keys(arg2).forEach((key) => {
      if (DATETIME_FORMAT_OPTIONS_KEYS.includes(key)) {
        overrides[key] = arg2[key];
      } else {
        options2[key] = arg2[key];
      }
    });
  }
  if (isString$2(arg3)) {
    options2.locale = arg3;
  } else if (isPlainObject$1(arg3)) {
    overrides = arg3;
  }
  if (isPlainObject$1(arg4)) {
    overrides = arg4;
  }
  return [options2.key || "", value, options2, overrides];
}
function clearDateTimeFormat(ctx, locale, format2) {
  const context = ctx;
  for (const key in format2) {
    const id2 = `${locale}__${key}`;
    if (!context.__datetimeFormatters.has(id2)) {
      continue;
    }
    context.__datetimeFormatters.delete(id2);
  }
}
function number(context, ...args) {
  const { numberFormats, unresolving, fallbackLocale, onWarn, localeFallbacker } = context;
  const { __numberFormatters } = context;
  const [key, value, options2, overrides] = parseNumberArgs(...args);
  const missingWarn = isBoolean(options2.missingWarn) ? options2.missingWarn : context.missingWarn;
  isBoolean(options2.fallbackWarn) ? options2.fallbackWarn : context.fallbackWarn;
  const part2 = !!options2.part;
  const locale = getLocale(context, options2);
  const locales = localeFallbacker(
    context,
    // eslint-disable-line @typescript-eslint/no-explicit-any
    fallbackLocale,
    locale
  );
  if (!isString$2(key) || key === "") {
    return new Intl.NumberFormat(locale, overrides).format(value);
  }
  let numberFormat = {};
  let targetLocale;
  let format2 = null;
  const type2 = "number format";
  for (let i2 = 0; i2 < locales.length; i2++) {
    targetLocale = locales[i2];
    numberFormat = numberFormats[targetLocale] || {};
    format2 = numberFormat[key];
    if (isPlainObject$1(format2))
      break;
    handleMissing(context, key, targetLocale, missingWarn, type2);
  }
  if (!isPlainObject$1(format2) || !isString$2(targetLocale)) {
    return unresolving ? NOT_REOSLVED : key;
  }
  let id2 = `${targetLocale}__${key}`;
  if (!isEmptyObject(overrides)) {
    id2 = `${id2}__${JSON.stringify(overrides)}`;
  }
  let formatter = __numberFormatters.get(id2);
  if (!formatter) {
    formatter = new Intl.NumberFormat(targetLocale, assign$2({}, format2, overrides));
    __numberFormatters.set(id2, formatter);
  }
  return !part2 ? formatter.format(value) : formatter.formatToParts(value);
}
const NUMBER_FORMAT_OPTIONS_KEYS = [
  "localeMatcher",
  "style",
  "currency",
  "currencyDisplay",
  "currencySign",
  "useGrouping",
  "minimumIntegerDigits",
  "minimumFractionDigits",
  "maximumFractionDigits",
  "minimumSignificantDigits",
  "maximumSignificantDigits",
  "compactDisplay",
  "notation",
  "signDisplay",
  "unit",
  "unitDisplay",
  "roundingMode",
  "roundingPriority",
  "roundingIncrement",
  "trailingZeroDisplay"
];
function parseNumberArgs(...args) {
  const [arg1, arg2, arg3, arg4] = args;
  const options2 = create();
  let overrides = create();
  if (!isNumber(arg1)) {
    throw createCoreError(CoreErrorCodes.INVALID_ARGUMENT);
  }
  const value = arg1;
  if (isString$2(arg2)) {
    options2.key = arg2;
  } else if (isPlainObject$1(arg2)) {
    Object.keys(arg2).forEach((key) => {
      if (NUMBER_FORMAT_OPTIONS_KEYS.includes(key)) {
        overrides[key] = arg2[key];
      } else {
        options2[key] = arg2[key];
      }
    });
  }
  if (isString$2(arg3)) {
    options2.locale = arg3;
  } else if (isPlainObject$1(arg3)) {
    overrides = arg3;
  }
  if (isPlainObject$1(arg4)) {
    overrides = arg4;
  }
  return [options2.key || "", value, options2, overrides];
}
function clearNumberFormat(ctx, locale, format2) {
  const context = ctx;
  for (const key in format2) {
    const id2 = `${locale}__${key}`;
    if (!context.__numberFormatters.has(id2)) {
      continue;
    }
    context.__numberFormatters.delete(id2);
  }
}
{
  initFeatureFlags$1();
}
/*!
  * vue-i18n v9.14.5
  * (c) 2025 kazuya kawaguchi
  * Released under the MIT License.
  */
const VERSION = "9.14.5";
function initFeatureFlags() {
  if (typeof __INTLIFY_PROD_DEVTOOLS__ !== "boolean") {
    getGlobalThis().__INTLIFY_PROD_DEVTOOLS__ = false;
  }
}
const code$1 = CoreWarnCodes.__EXTEND_POINT__;
const inc$1 = incrementer(code$1);
({
  // 9
  NOT_SUPPORTED_PRESERVE: inc$1(),
  // 10
  NOT_SUPPORTED_FORMATTER: inc$1(),
  // 11
  NOT_SUPPORTED_PRESERVE_DIRECTIVE: inc$1(),
  // 12
  NOT_SUPPORTED_GET_CHOICE_INDEX: inc$1(),
  // 13
  COMPONENT_NAME_LEGACY_COMPATIBLE: inc$1(),
  // 14
  NOT_FOUND_PARENT_SCOPE: inc$1(),
  // 15
  IGNORE_OBJ_FLATTEN: inc$1(),
  // 16
  NOTICE_DROP_ALLOW_COMPOSITION: inc$1(),
  // 17
  NOTICE_DROP_TRANSLATE_EXIST_COMPATIBLE_FLAG: inc$1()
  // 18
});
const code$2 = CoreErrorCodes.__EXTEND_POINT__;
const inc = incrementer(code$2);
const I18nErrorCodes = {
  // composer module errors
  UNEXPECTED_RETURN_TYPE: code$2,
  // 24
  // legacy module errors
  INVALID_ARGUMENT: inc(),
  // 25
  // i18n module errors
  MUST_BE_CALL_SETUP_TOP: inc(),
  // 26
  NOT_INSTALLED: inc(),
  // 27
  NOT_AVAILABLE_IN_LEGACY_MODE: inc(),
  // 28
  // directive module errors
  REQUIRED_VALUE: inc(),
  // 29
  INVALID_VALUE: inc(),
  // 30
  // vue-devtools errors
  CANNOT_SETUP_VUE_DEVTOOLS_PLUGIN: inc(),
  // 31
  NOT_INSTALLED_WITH_PROVIDE: inc(),
  // 32
  // unexpected error
  UNEXPECTED_ERROR: inc(),
  // 33
  // not compatible legacy vue-i18n constructor
  NOT_COMPATIBLE_LEGACY_VUE_I18N: inc(),
  // 34
  // bridge support vue 2.x only
  BRIDGE_SUPPORT_VUE_2_ONLY: inc(),
  // 35
  // need to define `i18n` option in `allowComposition: true` and `useScope: 'local' at `useI18n``
  MUST_DEFINE_I18N_OPTION_IN_ALLOW_COMPOSITION: inc(),
  // 36
  // Not available Compostion API in Legacy API mode. Please make sure that the legacy API mode is working properly
  NOT_AVAILABLE_COMPOSITION_IN_LEGACY: inc(),
  // 37
  // for enhancement
  __EXTEND_POINT__: inc()
  // 38
};
function createI18nError(code2, ...args) {
  return createCompileError(code2, null, void 0);
}
const TranslateVNodeSymbol = /* @__PURE__ */ makeSymbol("__translateVNode");
const DatetimePartsSymbol = /* @__PURE__ */ makeSymbol("__datetimeParts");
const NumberPartsSymbol = /* @__PURE__ */ makeSymbol("__numberParts");
const SetPluralRulesSymbol = makeSymbol("__setPluralRules");
const InejctWithOptionSymbol = /* @__PURE__ */ makeSymbol("__injectWithOption");
const DisposeSymbol = /* @__PURE__ */ makeSymbol("__dispose");
function handleFlatJson(obj) {
  if (!isObject$3(obj)) {
    return obj;
  }
  if (isMessageAST(obj)) {
    return obj;
  }
  for (const key in obj) {
    if (!hasOwn(obj, key)) {
      continue;
    }
    if (!key.includes(".")) {
      if (isObject$3(obj[key])) {
        handleFlatJson(obj[key]);
      }
    } else {
      const subKeys = key.split(".");
      const lastIndex = subKeys.length - 1;
      let currentObj = obj;
      let hasStringValue = false;
      for (let i2 = 0; i2 < lastIndex; i2++) {
        if (subKeys[i2] === "__proto__") {
          throw new Error(`unsafe key: ${subKeys[i2]}`);
        }
        if (!(subKeys[i2] in currentObj)) {
          currentObj[subKeys[i2]] = create();
        }
        if (!isObject$3(currentObj[subKeys[i2]])) {
          hasStringValue = true;
          break;
        }
        currentObj = currentObj[subKeys[i2]];
      }
      if (!hasStringValue) {
        if (!isMessageAST(currentObj)) {
          currentObj[subKeys[lastIndex]] = obj[key];
          delete obj[key];
        } else {
          if (!AST_NODE_PROPS_KEYS.includes(subKeys[lastIndex])) {
            delete obj[key];
          }
        }
      }
      if (!isMessageAST(currentObj)) {
        const target2 = currentObj[subKeys[lastIndex]];
        if (isObject$3(target2)) {
          handleFlatJson(target2);
        }
      }
    }
  }
  return obj;
}
function getLocaleMessages(locale, options2) {
  const { messages, __i18n, messageResolver, flatJson } = options2;
  const ret = isPlainObject$1(messages) ? messages : isArray$1(__i18n) ? create() : { [locale]: create() };
  if (isArray$1(__i18n)) {
    __i18n.forEach((custom) => {
      if ("locale" in custom && "resource" in custom) {
        const { locale: locale2, resource } = custom;
        if (locale2) {
          ret[locale2] = ret[locale2] || create();
          deepCopy(resource, ret[locale2]);
        } else {
          deepCopy(resource, ret);
        }
      } else {
        isString$2(custom) && deepCopy(JSON.parse(custom), ret);
      }
    });
  }
  if (messageResolver == null && flatJson) {
    for (const key in ret) {
      if (hasOwn(ret, key)) {
        handleFlatJson(ret[key]);
      }
    }
  }
  return ret;
}
function getComponentOptions(instance) {
  return instance.type;
}
function adjustI18nResources(gl2, options2, componentOptions) {
  let messages = isObject$3(options2.messages) ? options2.messages : create();
  if ("__i18nGlobal" in componentOptions) {
    messages = getLocaleMessages(gl2.locale.value, {
      messages,
      __i18n: componentOptions.__i18nGlobal
    });
  }
  const locales = Object.keys(messages);
  if (locales.length) {
    locales.forEach((locale) => {
      gl2.mergeLocaleMessage(locale, messages[locale]);
    });
  }
  {
    if (isObject$3(options2.datetimeFormats)) {
      const locales2 = Object.keys(options2.datetimeFormats);
      if (locales2.length) {
        locales2.forEach((locale) => {
          gl2.mergeDateTimeFormat(locale, options2.datetimeFormats[locale]);
        });
      }
    }
    if (isObject$3(options2.numberFormats)) {
      const locales2 = Object.keys(options2.numberFormats);
      if (locales2.length) {
        locales2.forEach((locale) => {
          gl2.mergeNumberFormat(locale, options2.numberFormats[locale]);
        });
      }
    }
  }
}
function createTextNode(key) {
  return createVNode(Text, null, key, 0);
}
const DEVTOOLS_META = "__INTLIFY_META__";
const NOOP_RETURN_ARRAY = () => [];
const NOOP_RETURN_FALSE = () => false;
let composerID = 0;
function defineCoreMissingHandler(missing) {
  return ((ctx, locale, key, type2) => {
    return missing(locale, key, getCurrentInstance() || void 0, type2);
  });
}
const getMetaInfo = /* @__NO_SIDE_EFFECTS__ */ () => {
  const instance = getCurrentInstance();
  let meta = null;
  return instance && (meta = getComponentOptions(instance)[DEVTOOLS_META]) ? { [DEVTOOLS_META]: meta } : null;
};
function createComposer(options2 = {}, VueI18nLegacy) {
  const { __root, __injectWithOption } = options2;
  const _isGlobal = __root === void 0;
  const flatJson = options2.flatJson;
  const _ref = inBrowser ? ref : shallowRef;
  const translateExistCompatible = !!options2.translateExistCompatible;
  let _inheritLocale = isBoolean(options2.inheritLocale) ? options2.inheritLocale : true;
  const _locale = _ref(
    // prettier-ignore
    __root && _inheritLocale ? __root.locale.value : isString$2(options2.locale) ? options2.locale : DEFAULT_LOCALE
  );
  const _fallbackLocale = _ref(
    // prettier-ignore
    __root && _inheritLocale ? __root.fallbackLocale.value : isString$2(options2.fallbackLocale) || isArray$1(options2.fallbackLocale) || isPlainObject$1(options2.fallbackLocale) || options2.fallbackLocale === false ? options2.fallbackLocale : _locale.value
  );
  const _messages = _ref(getLocaleMessages(_locale.value, options2));
  const _datetimeFormats = _ref(isPlainObject$1(options2.datetimeFormats) ? options2.datetimeFormats : { [_locale.value]: {} });
  const _numberFormats = _ref(isPlainObject$1(options2.numberFormats) ? options2.numberFormats : { [_locale.value]: {} });
  let _missingWarn = __root ? __root.missingWarn : isBoolean(options2.missingWarn) || isRegExp(options2.missingWarn) ? options2.missingWarn : true;
  let _fallbackWarn = __root ? __root.fallbackWarn : isBoolean(options2.fallbackWarn) || isRegExp(options2.fallbackWarn) ? options2.fallbackWarn : true;
  let _fallbackRoot = __root ? __root.fallbackRoot : isBoolean(options2.fallbackRoot) ? options2.fallbackRoot : true;
  let _fallbackFormat = !!options2.fallbackFormat;
  let _missing = isFunction(options2.missing) ? options2.missing : null;
  let _runtimeMissing = isFunction(options2.missing) ? defineCoreMissingHandler(options2.missing) : null;
  let _postTranslation = isFunction(options2.postTranslation) ? options2.postTranslation : null;
  let _warnHtmlMessage = __root ? __root.warnHtmlMessage : isBoolean(options2.warnHtmlMessage) ? options2.warnHtmlMessage : true;
  let _escapeParameter = !!options2.escapeParameter;
  const _modifiers = __root ? __root.modifiers : isPlainObject$1(options2.modifiers) ? options2.modifiers : {};
  let _pluralRules = options2.pluralRules || __root && __root.pluralRules;
  let _context;
  const getCoreContext = () => {
    _isGlobal && setFallbackContext(null);
    const ctxOptions = {
      version: VERSION,
      locale: _locale.value,
      fallbackLocale: _fallbackLocale.value,
      messages: _messages.value,
      modifiers: _modifiers,
      pluralRules: _pluralRules,
      missing: _runtimeMissing === null ? void 0 : _runtimeMissing,
      missingWarn: _missingWarn,
      fallbackWarn: _fallbackWarn,
      fallbackFormat: _fallbackFormat,
      unresolving: true,
      postTranslation: _postTranslation === null ? void 0 : _postTranslation,
      warnHtmlMessage: _warnHtmlMessage,
      escapeParameter: _escapeParameter,
      messageResolver: options2.messageResolver,
      messageCompiler: options2.messageCompiler,
      __meta: { framework: "vue" }
    };
    {
      ctxOptions.datetimeFormats = _datetimeFormats.value;
      ctxOptions.numberFormats = _numberFormats.value;
      ctxOptions.__datetimeFormatters = isPlainObject$1(_context) ? _context.__datetimeFormatters : void 0;
      ctxOptions.__numberFormatters = isPlainObject$1(_context) ? _context.__numberFormatters : void 0;
    }
    const ctx = createCoreContext(ctxOptions);
    _isGlobal && setFallbackContext(ctx);
    return ctx;
  };
  _context = getCoreContext();
  updateFallbackLocale(_context, _locale.value, _fallbackLocale.value);
  function trackReactivityValues() {
    return [
      _locale.value,
      _fallbackLocale.value,
      _messages.value,
      _datetimeFormats.value,
      _numberFormats.value
    ];
  }
  const locale = computed({
    get: () => _locale.value,
    set: (val) => {
      _locale.value = val;
      _context.locale = _locale.value;
    }
  });
  const fallbackLocale = computed({
    get: () => _fallbackLocale.value,
    set: (val) => {
      _fallbackLocale.value = val;
      _context.fallbackLocale = _fallbackLocale.value;
      updateFallbackLocale(_context, _locale.value, val);
    }
  });
  const messages = computed(() => _messages.value);
  const datetimeFormats = /* @__PURE__ */ computed(() => _datetimeFormats.value);
  const numberFormats = /* @__PURE__ */ computed(() => _numberFormats.value);
  function getPostTranslationHandler() {
    return isFunction(_postTranslation) ? _postTranslation : null;
  }
  function setPostTranslationHandler(handler) {
    _postTranslation = handler;
    _context.postTranslation = handler;
  }
  function getMissingHandler() {
    return _missing;
  }
  function setMissingHandler(handler) {
    if (handler !== null) {
      _runtimeMissing = defineCoreMissingHandler(handler);
    }
    _missing = handler;
    _context.missing = _runtimeMissing;
  }
  const wrapWithDeps = (fn, argumentParser, warnType, fallbackSuccess, fallbackFail, successCondition) => {
    trackReactivityValues();
    let ret;
    try {
      if (__INTLIFY_PROD_DEVTOOLS__) {
        /* @__PURE__ */ setAdditionalMeta(/* @__PURE__ */ getMetaInfo());
      }
      if (!_isGlobal) {
        _context.fallbackContext = __root ? getFallbackContext() : void 0;
      }
      ret = fn(_context);
    } finally {
      if (__INTLIFY_PROD_DEVTOOLS__) ;
      if (!_isGlobal) {
        _context.fallbackContext = void 0;
      }
    }
    if (warnType !== "translate exists" && // for not `te` (e.g `t`)
    isNumber(ret) && ret === NOT_REOSLVED || warnType === "translate exists" && !ret) {
      const [key, arg2] = argumentParser();
      return __root && _fallbackRoot ? fallbackSuccess(__root) : fallbackFail(key);
    } else if (successCondition(ret)) {
      return ret;
    } else {
      throw createI18nError(I18nErrorCodes.UNEXPECTED_RETURN_TYPE);
    }
  };
  function t2(...args) {
    return wrapWithDeps((context) => Reflect.apply(translate, null, [context, ...args]), () => parseTranslateArgs(...args), "translate", (root2) => Reflect.apply(root2.t, root2, [...args]), (key) => key, (val) => isString$2(val));
  }
  function rt(...args) {
    const [arg1, arg2, arg3] = args;
    if (arg3 && !isObject$3(arg3)) {
      throw createI18nError(I18nErrorCodes.INVALID_ARGUMENT);
    }
    return t2(...[arg1, arg2, assign$2({ resolvedMessage: true }, arg3 || {})]);
  }
  function d2(...args) {
    return wrapWithDeps((context) => Reflect.apply(datetime, null, [context, ...args]), () => parseDateTimeArgs(...args), "datetime format", (root2) => Reflect.apply(root2.d, root2, [...args]), () => MISSING_RESOLVE_VALUE, (val) => isString$2(val));
  }
  function n2(...args) {
    return wrapWithDeps((context) => Reflect.apply(number, null, [context, ...args]), () => parseNumberArgs(...args), "number format", (root2) => Reflect.apply(root2.n, root2, [...args]), () => MISSING_RESOLVE_VALUE, (val) => isString$2(val));
  }
  function normalize2(values) {
    return values.map((val) => isString$2(val) || isNumber(val) || isBoolean(val) ? createTextNode(String(val)) : val);
  }
  const interpolate = (val) => val;
  const processor = {
    normalize: normalize2,
    interpolate,
    type: "vnode"
  };
  function translateVNode(...args) {
    return wrapWithDeps(
      (context) => {
        let ret;
        const _context2 = context;
        try {
          _context2.processor = processor;
          ret = Reflect.apply(translate, null, [_context2, ...args]);
        } finally {
          _context2.processor = null;
        }
        return ret;
      },
      () => parseTranslateArgs(...args),
      "translate",
      // eslint-disable-next-line @typescript-eslint/no-explicit-any
      (root2) => root2[TranslateVNodeSymbol](...args),
      (key) => [createTextNode(key)],
      (val) => isArray$1(val)
    );
  }
  function numberParts(...args) {
    return wrapWithDeps(
      (context) => Reflect.apply(number, null, [context, ...args]),
      () => parseNumberArgs(...args),
      "number format",
      // eslint-disable-next-line @typescript-eslint/no-explicit-any
      (root2) => root2[NumberPartsSymbol](...args),
      NOOP_RETURN_ARRAY,
      (val) => isString$2(val) || isArray$1(val)
    );
  }
  function datetimeParts(...args) {
    return wrapWithDeps(
      (context) => Reflect.apply(datetime, null, [context, ...args]),
      () => parseDateTimeArgs(...args),
      "datetime format",
      // eslint-disable-next-line @typescript-eslint/no-explicit-any
      (root2) => root2[DatetimePartsSymbol](...args),
      NOOP_RETURN_ARRAY,
      (val) => isString$2(val) || isArray$1(val)
    );
  }
  function setPluralRules(rules) {
    _pluralRules = rules;
    _context.pluralRules = _pluralRules;
  }
  function te(key, locale2) {
    return wrapWithDeps(() => {
      if (!key) {
        return false;
      }
      const targetLocale = isString$2(locale2) ? locale2 : _locale.value;
      const message = getLocaleMessage(targetLocale);
      const resolved = _context.messageResolver(message, key);
      return !translateExistCompatible ? isMessageAST(resolved) || isMessageFunction(resolved) || isString$2(resolved) : resolved != null;
    }, () => [key], "translate exists", (root2) => {
      return Reflect.apply(root2.te, root2, [key, locale2]);
    }, NOOP_RETURN_FALSE, (val) => isBoolean(val));
  }
  function resolveMessages(key) {
    let messages2 = null;
    const locales = fallbackWithLocaleChain(_context, _fallbackLocale.value, _locale.value);
    for (let i2 = 0; i2 < locales.length; i2++) {
      const targetLocaleMessages = _messages.value[locales[i2]] || {};
      const messageValue = _context.messageResolver(targetLocaleMessages, key);
      if (messageValue != null) {
        messages2 = messageValue;
        break;
      }
    }
    return messages2;
  }
  function tm(key) {
    const messages2 = resolveMessages(key);
    return messages2 != null ? messages2 : __root ? __root.tm(key) || {} : {};
  }
  function getLocaleMessage(locale2) {
    return _messages.value[locale2] || {};
  }
  function setLocaleMessage(locale2, message) {
    if (flatJson) {
      const _message = { [locale2]: message };
      for (const key in _message) {
        if (hasOwn(_message, key)) {
          handleFlatJson(_message[key]);
        }
      }
      message = _message[locale2];
    }
    _messages.value[locale2] = message;
    _context.messages = _messages.value;
  }
  function mergeLocaleMessage(locale2, message) {
    _messages.value[locale2] = _messages.value[locale2] || {};
    const _message = { [locale2]: message };
    if (flatJson) {
      for (const key in _message) {
        if (hasOwn(_message, key)) {
          handleFlatJson(_message[key]);
        }
      }
    }
    message = _message[locale2];
    deepCopy(message, _messages.value[locale2]);
    _context.messages = _messages.value;
  }
  function getDateTimeFormat(locale2) {
    return _datetimeFormats.value[locale2] || {};
  }
  function setDateTimeFormat(locale2, format2) {
    _datetimeFormats.value[locale2] = format2;
    _context.datetimeFormats = _datetimeFormats.value;
    clearDateTimeFormat(_context, locale2, format2);
  }
  function mergeDateTimeFormat(locale2, format2) {
    _datetimeFormats.value[locale2] = assign$2(_datetimeFormats.value[locale2] || {}, format2);
    _context.datetimeFormats = _datetimeFormats.value;
    clearDateTimeFormat(_context, locale2, format2);
  }
  function getNumberFormat(locale2) {
    return _numberFormats.value[locale2] || {};
  }
  function setNumberFormat(locale2, format2) {
    _numberFormats.value[locale2] = format2;
    _context.numberFormats = _numberFormats.value;
    clearNumberFormat(_context, locale2, format2);
  }
  function mergeNumberFormat(locale2, format2) {
    _numberFormats.value[locale2] = assign$2(_numberFormats.value[locale2] || {}, format2);
    _context.numberFormats = _numberFormats.value;
    clearNumberFormat(_context, locale2, format2);
  }
  composerID++;
  if (__root && inBrowser) {
    watch(__root.locale, (val) => {
      if (_inheritLocale) {
        _locale.value = val;
        _context.locale = val;
        updateFallbackLocale(_context, _locale.value, _fallbackLocale.value);
      }
    });
    watch(__root.fallbackLocale, (val) => {
      if (_inheritLocale) {
        _fallbackLocale.value = val;
        _context.fallbackLocale = val;
        updateFallbackLocale(_context, _locale.value, _fallbackLocale.value);
      }
    });
  }
  const composer = {
    id: composerID,
    locale,
    fallbackLocale,
    get inheritLocale() {
      return _inheritLocale;
    },
    set inheritLocale(val) {
      _inheritLocale = val;
      if (val && __root) {
        _locale.value = __root.locale.value;
        _fallbackLocale.value = __root.fallbackLocale.value;
        updateFallbackLocale(_context, _locale.value, _fallbackLocale.value);
      }
    },
    get availableLocales() {
      return Object.keys(_messages.value).sort();
    },
    messages,
    get modifiers() {
      return _modifiers;
    },
    get pluralRules() {
      return _pluralRules || {};
    },
    get isGlobal() {
      return _isGlobal;
    },
    get missingWarn() {
      return _missingWarn;
    },
    set missingWarn(val) {
      _missingWarn = val;
      _context.missingWarn = _missingWarn;
    },
    get fallbackWarn() {
      return _fallbackWarn;
    },
    set fallbackWarn(val) {
      _fallbackWarn = val;
      _context.fallbackWarn = _fallbackWarn;
    },
    get fallbackRoot() {
      return _fallbackRoot;
    },
    set fallbackRoot(val) {
      _fallbackRoot = val;
    },
    get fallbackFormat() {
      return _fallbackFormat;
    },
    set fallbackFormat(val) {
      _fallbackFormat = val;
      _context.fallbackFormat = _fallbackFormat;
    },
    get warnHtmlMessage() {
      return _warnHtmlMessage;
    },
    set warnHtmlMessage(val) {
      _warnHtmlMessage = val;
      _context.warnHtmlMessage = val;
    },
    get escapeParameter() {
      return _escapeParameter;
    },
    set escapeParameter(val) {
      _escapeParameter = val;
      _context.escapeParameter = val;
    },
    t: t2,
    getLocaleMessage,
    setLocaleMessage,
    mergeLocaleMessage,
    getPostTranslationHandler,
    setPostTranslationHandler,
    getMissingHandler,
    setMissingHandler,
    [SetPluralRulesSymbol]: setPluralRules
  };
  {
    composer.datetimeFormats = datetimeFormats;
    composer.numberFormats = numberFormats;
    composer.rt = rt;
    composer.te = te;
    composer.tm = tm;
    composer.d = d2;
    composer.n = n2;
    composer.getDateTimeFormat = getDateTimeFormat;
    composer.setDateTimeFormat = setDateTimeFormat;
    composer.mergeDateTimeFormat = mergeDateTimeFormat;
    composer.getNumberFormat = getNumberFormat;
    composer.setNumberFormat = setNumberFormat;
    composer.mergeNumberFormat = mergeNumberFormat;
    composer[InejctWithOptionSymbol] = __injectWithOption;
    composer[TranslateVNodeSymbol] = translateVNode;
    composer[DatetimePartsSymbol] = datetimeParts;
    composer[NumberPartsSymbol] = numberParts;
  }
  return composer;
}
const baseFormatProps = {
  tag: {
    type: [String, Object]
  },
  locale: {
    type: String
  },
  scope: {
    type: String,
    // NOTE: avoid https://github.com/microsoft/rushstack/issues/1050
    validator: (val) => val === "parent" || val === "global",
    default: "parent"
    /* ComponentI18nScope */
  },
  i18n: {
    type: Object
  }
};
function getInterpolateArg({ slots }, keys) {
  if (keys.length === 1 && keys[0] === "default") {
    const ret = slots.default ? slots.default() : [];
    return ret.reduce((slot, current) => {
      return [
        ...slot,
        // prettier-ignore
        ...current.type === Fragment ? current.children : [current]
      ];
    }, []);
  } else {
    return keys.reduce((arg, key) => {
      const slot = slots[key];
      if (slot) {
        arg[key] = slot();
      }
      return arg;
    }, create());
  }
}
function getFragmentableTag(tag) {
  return Fragment;
}
const TranslationImpl = /* @__PURE__ */ defineComponent({
  /* eslint-disable */
  name: "i18n-t",
  props: assign$2({
    keypath: {
      type: String,
      required: true
    },
    plural: {
      type: [Number, String],
      // eslint-disable-next-line @typescript-eslint/no-explicit-any
      validator: (val) => isNumber(val) || !isNaN(val)
    }
  }, baseFormatProps),
  /* eslint-enable */
  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  setup(props, context) {
    const { slots, attrs } = context;
    const i18n = props.i18n || useI18n$1({
      useScope: props.scope,
      __useComponent: true
    });
    return () => {
      const keys = Object.keys(slots).filter((key) => key !== "_");
      const options2 = create();
      if (props.locale) {
        options2.locale = props.locale;
      }
      if (props.plural !== void 0) {
        options2.plural = isString$2(props.plural) ? +props.plural : props.plural;
      }
      const arg = getInterpolateArg(context, keys);
      const children = i18n[TranslateVNodeSymbol](props.keypath, arg, options2);
      const assignedAttrs = assign$2(create(), attrs);
      const tag = isString$2(props.tag) || isObject$3(props.tag) ? props.tag : getFragmentableTag();
      return h$2(tag, assignedAttrs, children);
    };
  }
});
const Translation = TranslationImpl;
function isVNode(target2) {
  return isArray$1(target2) && !isString$2(target2[0]);
}
function renderFormatter(props, context, slotKeys, partFormatter) {
  const { slots, attrs } = context;
  return () => {
    const options2 = { part: true };
    let overrides = create();
    if (props.locale) {
      options2.locale = props.locale;
    }
    if (isString$2(props.format)) {
      options2.key = props.format;
    } else if (isObject$3(props.format)) {
      if (isString$2(props.format.key)) {
        options2.key = props.format.key;
      }
      overrides = Object.keys(props.format).reduce((options22, prop2) => {
        return slotKeys.includes(prop2) ? assign$2(create(), options22, { [prop2]: props.format[prop2] }) : options22;
      }, create());
    }
    const parts = partFormatter(...[props.value, options2, overrides]);
    let children = [options2.key];
    if (isArray$1(parts)) {
      children = parts.map((part2, index) => {
        const slot = slots[part2.type];
        const node = slot ? slot({ [part2.type]: part2.value, index, parts }) : [part2.value];
        if (isVNode(node)) {
          node[0].key = `${part2.type}-${index}`;
        }
        return node;
      });
    } else if (isString$2(parts)) {
      children = [parts];
    }
    const assignedAttrs = assign$2(create(), attrs);
    const tag = isString$2(props.tag) || isObject$3(props.tag) ? props.tag : getFragmentableTag();
    return h$2(tag, assignedAttrs, children);
  };
}
const NumberFormatImpl = /* @__PURE__ */ defineComponent({
  /* eslint-disable */
  name: "i18n-n",
  props: assign$2({
    value: {
      type: Number,
      required: true
    },
    format: {
      type: [String, Object]
    }
  }, baseFormatProps),
  /* eslint-enable */
  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  setup(props, context) {
    const i18n = props.i18n || useI18n$1({
      useScope: props.scope,
      __useComponent: true
    });
    return renderFormatter(props, context, NUMBER_FORMAT_OPTIONS_KEYS, (...args) => (
      // eslint-disable-next-line @typescript-eslint/no-explicit-any
      i18n[NumberPartsSymbol](...args)
    ));
  }
});
const NumberFormat = NumberFormatImpl;
const DatetimeFormatImpl = /* @__PURE__ */ defineComponent({
  /* eslint-disable */
  name: "i18n-d",
  props: assign$2({
    value: {
      type: [Number, Date],
      required: true
    },
    format: {
      type: [String, Object]
    }
  }, baseFormatProps),
  /* eslint-enable */
  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  setup(props, context) {
    const i18n = props.i18n || useI18n$1({
      useScope: props.scope,
      __useComponent: true
    });
    return renderFormatter(props, context, DATETIME_FORMAT_OPTIONS_KEYS, (...args) => (
      // eslint-disable-next-line @typescript-eslint/no-explicit-any
      i18n[DatetimePartsSymbol](...args)
    ));
  }
});
const DatetimeFormat = DatetimeFormatImpl;
function getComposer$2(i18n, instance) {
  const i18nInternal = i18n;
  if (i18n.mode === "composition") {
    return i18nInternal.__getInstance(instance) || i18n.global;
  } else {
    const vueI18n = i18nInternal.__getInstance(instance);
    return vueI18n != null ? vueI18n.__composer : i18n.global.__composer;
  }
}
function vTDirective(i18n) {
  const _process = (binding) => {
    const { instance, modifiers, value } = binding;
    if (!instance || !instance.$) {
      throw createI18nError(I18nErrorCodes.UNEXPECTED_ERROR);
    }
    const composer = getComposer$2(i18n, instance.$);
    const parsedValue = parseValue(value);
    return [
      Reflect.apply(composer.t, composer, [...makeParams(parsedValue)]),
      composer
    ];
  };
  const register = (el2, binding) => {
    const [textContent, composer] = _process(binding);
    if (inBrowser && i18n.global === composer) {
      el2.__i18nWatcher = watch(composer.locale, () => {
        binding.instance && binding.instance.$forceUpdate();
      });
    }
    el2.__composer = composer;
    el2.textContent = textContent;
  };
  const unregister = (el2) => {
    if (inBrowser && el2.__i18nWatcher) {
      el2.__i18nWatcher();
      el2.__i18nWatcher = void 0;
      delete el2.__i18nWatcher;
    }
    if (el2.__composer) {
      el2.__composer = void 0;
      delete el2.__composer;
    }
  };
  const update = (el2, { value }) => {
    if (el2.__composer) {
      const composer = el2.__composer;
      const parsedValue = parseValue(value);
      el2.textContent = Reflect.apply(composer.t, composer, [
        ...makeParams(parsedValue)
      ]);
    }
  };
  const getSSRProps = (binding) => {
    const [textContent] = _process(binding);
    return { textContent };
  };
  return {
    created: register,
    unmounted: unregister,
    beforeUpdate: update,
    getSSRProps
  };
}
function parseValue(value) {
  if (isString$2(value)) {
    return { path: value };
  } else if (isPlainObject$1(value)) {
    if (!("path" in value)) {
      throw createI18nError(I18nErrorCodes.REQUIRED_VALUE, "path");
    }
    return value;
  } else {
    throw createI18nError(I18nErrorCodes.INVALID_VALUE);
  }
}
function makeParams(value) {
  const { path, locale, args, choice, plural } = value;
  const options2 = {};
  const named = args || {};
  if (isString$2(locale)) {
    options2.locale = locale;
  }
  if (isNumber(choice)) {
    options2.plural = choice;
  }
  if (isNumber(plural)) {
    options2.plural = plural;
  }
  return [path, named, options2];
}
function apply(app2, i18n, ...options2) {
  const pluginOptions = isPlainObject$1(options2[0]) ? options2[0] : {};
  const useI18nComponentName = !!pluginOptions.useI18nComponentName;
  const globalInstall = isBoolean(pluginOptions.globalInstall) ? pluginOptions.globalInstall : true;
  if (globalInstall) {
    [!useI18nComponentName ? Translation.name : "i18n", "I18nT"].forEach((name2) => app2.component(name2, Translation));
    [NumberFormat.name, "I18nN"].forEach((name2) => app2.component(name2, NumberFormat));
    [DatetimeFormat.name, "I18nD"].forEach((name2) => app2.component(name2, DatetimeFormat));
  }
  {
    app2.directive("t", vTDirective(i18n));
  }
}
const I18nInjectionKey = /* @__PURE__ */ makeSymbol("global-vue-i18n");
function createI18n$1(options2 = {}, VueI18nLegacy) {
  const __globalInjection = isBoolean(options2.globalInjection) ? options2.globalInjection : true;
  const __allowComposition = true;
  const __instances = /* @__PURE__ */ new Map();
  const [globalScope, __global] = createGlobal(options2);
  const symbol = /* @__PURE__ */ makeSymbol("");
  function __getInstance(component) {
    return __instances.get(component) || null;
  }
  function __setInstance(component, instance) {
    __instances.set(component, instance);
  }
  function __deleteInstance(component) {
    __instances.delete(component);
  }
  {
    const i18n = {
      // mode
      get mode() {
        return "composition";
      },
      // allowComposition
      get allowComposition() {
        return __allowComposition;
      },
      // install plugin
      async install(app2, ...options22) {
        app2.__VUE_I18N_SYMBOL__ = symbol;
        app2.provide(app2.__VUE_I18N_SYMBOL__, i18n);
        if (isPlainObject$1(options22[0])) {
          const opts = options22[0];
          i18n.__composerExtend = opts.__composerExtend;
          i18n.__vueI18nExtend = opts.__vueI18nExtend;
        }
        let globalReleaseHandler = null;
        if (__globalInjection) {
          globalReleaseHandler = injectGlobalFields(app2, i18n.global);
        }
        {
          apply(app2, i18n, ...options22);
        }
        const unmountApp = app2.unmount;
        app2.unmount = () => {
          globalReleaseHandler && globalReleaseHandler();
          i18n.dispose();
          unmountApp();
        };
      },
      // global accessor
      get global() {
        return __global;
      },
      dispose() {
        globalScope.stop();
      },
      // @internal
      __instances,
      // @internal
      __getInstance,
      // @internal
      __setInstance,
      // @internal
      __deleteInstance
    };
    return i18n;
  }
}
function useI18n$1(options2 = {}) {
  const instance = getCurrentInstance();
  if (instance == null) {
    throw createI18nError(I18nErrorCodes.MUST_BE_CALL_SETUP_TOP);
  }
  if (!instance.isCE && instance.appContext.app != null && !instance.appContext.app.__VUE_I18N_SYMBOL__) {
    throw createI18nError(I18nErrorCodes.NOT_INSTALLED);
  }
  const i18n = getI18nInstance(instance);
  const gl2 = getGlobalComposer(i18n);
  const componentOptions = getComponentOptions(instance);
  const scope = getScope(options2, componentOptions);
  if (scope === "global") {
    adjustI18nResources(gl2, options2, componentOptions);
    return gl2;
  }
  if (scope === "parent") {
    let composer2 = getComposer(i18n, instance, options2.__useComponent);
    if (composer2 == null) {
      composer2 = gl2;
    }
    return composer2;
  }
  const i18nInternal = i18n;
  let composer = i18nInternal.__getInstance(instance);
  if (composer == null) {
    const composerOptions = assign$2({}, options2);
    if ("__i18n" in componentOptions) {
      composerOptions.__i18n = componentOptions.__i18n;
    }
    if (gl2) {
      composerOptions.__root = gl2;
    }
    composer = createComposer(composerOptions);
    if (i18nInternal.__composerExtend) {
      composer[DisposeSymbol] = i18nInternal.__composerExtend(composer);
    }
    setupLifeCycle(i18nInternal, instance, composer);
    i18nInternal.__setInstance(instance, composer);
  }
  return composer;
}
function createGlobal(options2, legacyMode, VueI18nLegacy) {
  const scope = effectScope();
  {
    const obj = scope.run(() => createComposer(options2));
    if (obj == null) {
      throw createI18nError(I18nErrorCodes.UNEXPECTED_ERROR);
    }
    return [scope, obj];
  }
}
function getI18nInstance(instance) {
  {
    const i18n = inject(!instance.isCE ? instance.appContext.app.__VUE_I18N_SYMBOL__ : I18nInjectionKey);
    if (!i18n) {
      throw createI18nError(!instance.isCE ? I18nErrorCodes.UNEXPECTED_ERROR : I18nErrorCodes.NOT_INSTALLED_WITH_PROVIDE);
    }
    return i18n;
  }
}
function getScope(options2, componentOptions) {
  return isEmptyObject(options2) ? "__i18n" in componentOptions ? "local" : "global" : !options2.useScope ? "local" : options2.useScope;
}
function getGlobalComposer(i18n) {
  return i18n.mode === "composition" ? i18n.global : i18n.global.__composer;
}
function getComposer(i18n, target2, useComponent = false) {
  let composer = null;
  const root2 = target2.root;
  let current = getParentComponentInstance(target2, useComponent);
  while (current != null) {
    const i18nInternal = i18n;
    if (i18n.mode === "composition") {
      composer = i18nInternal.__getInstance(current);
    }
    if (composer != null) {
      break;
    }
    if (root2 === current) {
      break;
    }
    current = current.parent;
  }
  return composer;
}
function getParentComponentInstance(target2, useComponent = false) {
  if (target2 == null) {
    return null;
  }
  {
    return !useComponent ? target2.parent : target2.vnode.ctx || target2.parent;
  }
}
function setupLifeCycle(i18n, target2, composer) {
  {
    onMounted(() => {
    }, target2);
    onUnmounted(() => {
      const _composer = composer;
      i18n.__deleteInstance(target2);
      const dispose = _composer[DisposeSymbol];
      if (dispose) {
        dispose();
        delete _composer[DisposeSymbol];
      }
    }, target2);
  }
}
const globalExportProps = [
  "locale",
  "fallbackLocale",
  "availableLocales"
];
const globalExportMethods = ["t", "rt", "d", "n", "tm", "te"];
function injectGlobalFields(app2, composer) {
  const i18n = /* @__PURE__ */ Object.create(null);
  globalExportProps.forEach((prop2) => {
    const desc = Object.getOwnPropertyDescriptor(composer, prop2);
    if (!desc) {
      throw createI18nError(I18nErrorCodes.UNEXPECTED_ERROR);
    }
    const wrap2 = isRef(desc.value) ? {
      get() {
        return desc.value.value;
      },
      // eslint-disable-next-line @typescript-eslint/no-explicit-any
      set(val) {
        desc.value.value = val;
      }
    } : {
      get() {
        return desc.get && desc.get();
      }
    };
    Object.defineProperty(i18n, prop2, wrap2);
  });
  app2.config.globalProperties.$i18n = i18n;
  globalExportMethods.forEach((method) => {
    const desc = Object.getOwnPropertyDescriptor(composer, method);
    if (!desc || !desc.value) {
      throw createI18nError(I18nErrorCodes.UNEXPECTED_ERROR);
    }
    Object.defineProperty(app2.config.globalProperties, `$${method}`, desc);
  });
  const dispose = () => {
    delete app2.config.globalProperties.$i18n;
    globalExportMethods.forEach((method) => {
      delete app2.config.globalProperties[`$${method}`];
    });
  };
  return dispose;
}
{
  initFeatureFlags();
}
{
  registerMessageCompiler(compile$1);
}
registerMessageResolver(resolveValue);
registerLocaleFallbacker(fallbackWithLocaleChain);
if (__INTLIFY_PROD_DEVTOOLS__) {
  const target2 = getGlobalThis();
  target2.__INTLIFY__ = true;
  setDevToolsHook(target2.__INTLIFY_DEVTOOLS_GLOBAL_HOOK__);
}
const isObject$1 = (item) => item && typeof item === "object" && !Array.isArray(item);
const mergeDeep = (target2, ...sources) => {
  if (!sources.length) return target2;
  const source = sources.shift();
  if (isObject$1(target2) && isObject$1(source)) {
    for (const key in source) {
      if (isObject$1(source[key])) {
        if (!target2[key]) Object.assign(target2, { [key]: {} });
        mergeDeep(target2[key], source[key]);
      } else {
        Object.assign(target2, { [key]: source[key] });
      }
    }
  }
  return mergeDeep(target2, ...sources);
};
const compiledMessages = mergeDeep(
  {},
  { "en-US": {
    "links": {
      "learnMore": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Learn more."]);
      },
      "learnMoreButton": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Learn more"]);
      },
      "needHelp": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Need help"]);
      }
    },
    "components": {
      "modal": {
        "dismiss": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Dismiss"]);
        }
      },
      "fileSearch": {
        "noMatchesForExtension": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["No files found when searching for the file extension"]);
        },
        "noMatchesForFileName": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize([]);
        },
        "byFilenameInput": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Search by filename..."]);
        },
        "byExtensionInput": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Extensions to include e.g. *.tsx, *.vue"]);
        },
        "expandToSearchByExtensions": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Search for files with extensions e.g. *.tsx, *.vue"]);
        },
        "matchesIndicatorEmptyFileSearch": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, named: _named, plural: _plural } = ctx;
          return _plural([_normalize(["No matches"]), _normalize(["1 match"]), _normalize([_interpolate(_named("denominator")), " matches"])]);
        },
        "matchesIndicator": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, named: _named, plural: _plural } = ctx;
          return _plural([_normalize(["No matches"]), _normalize([_interpolate(_named("numerator")), " of ", _interpolate(_named("denominator")), " matches"]), _normalize([_interpolate(_named("numerator")), " of ", _interpolate(_named("denominator")), " matches"])]);
        }
      },
      "button": {
        "cancel": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Cancel"]);
        },
        "back": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Back"]);
        }
      },
      "loading": {
        "config": {
          "title": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Initializing config..."]);
          },
          "description": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Please wait while we load your project and find browsers installed on your system"]);
          }
        }
      },
      "select": {
        "placeholder": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Choose an option..."]);
        }
      },
      "alert": {
        "dismissAriaLabel": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Dismiss"]);
        }
      },
      "specPattern": {
        "matches": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, named: _named, plural: _plural } = ctx;
          return _plural([_normalize(["No matches"]), _normalize([_interpolate(_named("n")), " match"]), _normalize([_interpolate(_named("n")), " matches"])]);
        }
      },
      "specPatternModal": {
        "title": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Spec pattern settings"]);
        }
      }
    },
    "clipboard": {
      "copy": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Copy"]);
      },
      "copied": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Copied!"]);
      }
    },
    "actions": {
      "close": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Close"]);
      }
    },
    "file": {
      "edit": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Edit"]);
      },
      "git": {
        "modified": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Modified"]);
        },
        "created": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Created"]);
        }
      }
    },
    "status": {
      "enabled": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Enabled"]);
      },
      "disabled": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Disabled"]);
      }
    },
    "createSpec": {
      "newSpec": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["New spec"]);
      },
      "createSpec": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Create spec"]);
      },
      "done": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Done!"]);
      },
      "noComponentsFound": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["No components found"]);
      },
      "unableToParseFile": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Unable to parse file"]);
      },
      "updateSpecPattern": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Update spec pattern"]);
      },
      "newSpecModalTitle": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Create new spec"]);
      },
      "successPage": {
        "header": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Great! The spec was successfully added"]);
        },
        "runSpecButton": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Okay, run the spec"]);
        },
        "createAnotherSpecButton": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Create another spec"]);
        }
      },
      "page": {
        "defaultPatternNoSpecs": {
          "title": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Create your first spec"]);
          },
          "component": {
            "description": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Since this project looks new, we recommend that you get started by creating a spec for one of your components."]);
            }
          },
          "e2e": {
            "description": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Since this project looks new, we recommend that you use the specs and tests that we've written for you to get started."]);
            }
          }
        },
        "customPatternNoSpecs": {
          "title": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["No specs found"]);
          },
          "description": (ctx) => {
            const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
            return _normalize(["We couldn't find any files matching the following spec patterns. You may need to move your specs or update your ", _interpolate(_list(0)), " to find them."]);
          }
        }
      },
      "noSpecsMessage": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["If you feel that you're seeing this screen in error, and there should be specs listed here, you likely need to update the spec pattern."]);
      },
      "viewSpecPatternButton": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["View spec pattern"]);
      },
      "e2e": {
        "importFromScaffold": {
          "header": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Scaffold example specs"]);
          },
          "description": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["We'll generate several example specs to help guide you on how to write tests in Cypress."]);
          },
          "specsAddedHeader": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Great! We added the following files to your project:"]);
          },
          "specsAddedButton": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Okay, I got it!"]);
          },
          "specsAddingHeader": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Adding specs..."]);
          }
        },
        "importTemplateSpec": {
          "header": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Create new spec"]);
          },
          "description": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["We'll generate a template spec file which can be used to start testing your application."]);
          },
          "chooseFilenameHeader": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Enter the path for your new spec"]);
          },
          "inputPlaceholder": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Enter a relative path..."]);
          },
          "invalidSpecWarning": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["This path is invalid because it doesn't match the following "]);
          },
          "specExtensionWarning": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["We recommend naming your spec to match the following pattern: "]);
          }
        }
      },
      "component": {
        "importFromComponent": {
          "header": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Create from component"]);
          },
          "description": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Generate a basic component test for any of the components in this project."]);
          },
          "chooseAComponentHeader": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Choose a component"]);
          }
        },
        "importTemplateSpec": {
          "header": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Create new spec"]);
          },
          "description": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["We'll generate a template spec file which can be used to start testing your components."]);
          },
          "invalidComponentWarning": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["We couldn't generate a valid path matching your custom "]);
          }
        }
      }
    },
    "specPage": {
      "pageTitle": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Specs"]);
      },
      "newSpecButton": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["New spec"]);
      },
      "searchPlaceholder": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Search specs"]);
      },
      "clearSearch": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Clear search field"]);
      },
      "componentSpecsHeader": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Component specs"]);
      },
      "e2eSpecsHeader": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["E2E specs"]);
      },
      "componentSpecsHeaderShort": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Component"]);
      },
      "e2eSpecsHeaderShort": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["E2E"]);
      },
      "lastUpdated": {
        "header": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Last updated"]);
        },
        "tooltip": {
          "gitStatus": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Git status"]);
          },
          "gitInfo": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["git info"]);
          },
          "gitInfoAvailable": (ctx) => {
            const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
            return _normalize([_interpolate(_list(0)), " of the spec files within this project"]);
          },
          "gitInfoUnavailable": (ctx) => {
            const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
            return _normalize(["Cypress is unable to detect ", _interpolate(_list(0)), " for this project and has defaulted to showing file system data instead"]);
          }
        }
      },
      "latestRuns": {
        "header": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Latest runs"]);
        },
        "headerShort": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Runs"]);
        },
        "tooltip": {
          "connected": (ctx) => {
            const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
            return _normalize(["The ", _interpolate(_list(0)), " in Cypress Cloud"]);
          },
          "notConnected": (ctx) => {
            const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
            return _normalize(["Connect to Cypress Cloud to see the ", _interpolate(_list(0))]);
          },
          "noAccess": (ctx) => {
            const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
            return _normalize(["Request access to this project in Cypress Cloud to view the ", _interpolate(_list(0))]);
          },
          "linkText": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["status of your latest runs"]);
          }
        }
      },
      "averageDuration": {
        "header": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Average duration"]);
        },
        "headerShort": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Duration"]);
        },
        "tooltip": {
          "connected": (ctx) => {
            const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
            return _normalize(["The ", _interpolate(_list(0)), " of your latest runs in Cypress Cloud"]);
          },
          "notConnected": (ctx) => {
            const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
            return _normalize(["Connect to Cypress Cloud to see the ", _interpolate(_list(0)), " of your latest runs"]);
          },
          "noAccess": (ctx) => {
            const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
            return _normalize(["Request access to this project in Cypress Cloud to view the ", _interpolate(_list(0)), " of your latest runs"]);
          },
          "linkText": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["average spec durations"]);
          }
        }
      },
      "flaky": {
        "badgeLabel": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Flaky"]);
        },
        "severityLow": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Low"]);
        },
        "severityMedium": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Medium"]);
        },
        "severityHigh": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["High"]);
        },
        "flakyRate": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
          return _normalize([_interpolate(_list(0)), "% flaky rate"]);
        },
        "flakyRuns": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, named: _named, plural: _plural } = ctx;
          return _plural([_normalize([_interpolate(_named("flakyRuns")), " flaky runs / ", _interpolate(_named("totalRuns")), " total"]), _normalize([_interpolate(_named("flakyRuns")), " flaky run / ", _interpolate(_named("totalRuns")), " total"]), _normalize([_interpolate(_named("flakyRuns")), " flaky runs / ", _interpolate(_named("totalRuns")), " total"])]);
        },
        "lastFlaky": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, named: _named, plural: _plural } = ctx;
          return _plural([_normalize(["Last run flaky"]), _normalize(["Last flaky ", _interpolate(_named("runsSinceLastFlake")), " run ago"]), _normalize(["Last flaky ", _interpolate(_named("runsSinceLastFlake")), " runs ago"])]);
        }
      },
      "hoverButton": {
        "connect": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Connect"]);
        },
        "connectProject": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Connect project"]);
        },
        "connectProjectShort": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Connect"]);
        },
        "requestAccess": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Request access"]);
        },
        "requestAccessShort": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Request"]);
        },
        "requestSent": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Request sent"]);
        },
        "requestSentShort": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Sent"]);
        }
      },
      "connectProjectButton": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Connect your project"]);
      },
      "cloudLoginButton": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Log in to Cypress Cloud"]);
      },
      "reconnectProjectButton": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Reconnect your project"]);
      },
      "unauthorizedBannerTitle": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Request access to project"]);
      },
      "requestAccessButton": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Request access"]);
      },
      "requestSentButton": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Request sent"]);
      },
      "rows": {
        "gitTooltipSubtext": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, named: _named } = ctx;
          return _normalize([_interpolate(_named("shortHash")), " by ", _interpolate(_named("author"))]);
        }
      },
      "noResultsMessage": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["No specs matched your search:"]);
      },
      "noSpecError": {
        "title": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Spec not found"]);
        },
        "intro": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["There is no spec matching the following location:"]);
        },
        "explainer": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["It is possible that the file has been moved or deleted. Please choose from the list of specs below."]);
        }
      },
      "offlineWarning": {
        "title": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["No internet connection"]);
        },
        "explainer": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Please check your internet connection to resolve this issue. When your internet connection is fixed, we will automatically attempt to fetch the run metrics from Cypress Cloud."]);
        }
      },
      "fetchFailedWarning": {
        "title": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Lost connection"]);
        },
        "explainer1": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["The request timed out or failed when trying to retrieve the recorded run metrics from Cypress Cloud. The information that you're seeing in the table below may be incomplete as a result."]);
        },
        "explainer2": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
          return _normalize(["Please refresh the page to try again and visit our ", _interpolate(_list(0)), " if this behavior continues."]);
        },
        "refreshButton": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Try again"]);
        }
      },
      "banners": {
        "login": {
          "title": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Optimize and record your CI test runs with Cypress Cloud"]);
          },
          "content": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["When you configure Cypress to record tests to Cypress Cloud, you'll see data from your latest recorded runs in the Cypress app. This increased visibility into your test history allows you to debug your tests faster and more effectively, all within your local workflow."]);
          },
          "buttonLabel": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Get started with Cypress Cloud"]);
          }
        },
        "createOrganization": {
          "title": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Finish setting up Cypress Cloud"]);
          },
          "content": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Join or create an organization in Cypress Cloud to access your projects and recorded test runs."]);
          },
          "buttonLabel": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Set up your organization"]);
          }
        },
        "connectProject": {
          "title": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Connect your project to Cypress Cloud"]);
          },
          "content": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Bring your recorded test results into your local development workflow to monitor, run, and fix tests all in the Cypress app."]);
          },
          "buttonLabel": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Connect your project"]);
          }
        },
        "record": {
          "title": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Record your first run"]);
          },
          "content": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Record a run to see your test results in Cypress Cloud. You can then optimize your test suite, debug failing and flaky tests, and integrate with your favorite tools."]);
          }
        },
        "componentTesting": {
          "title": (ctx) => {
            const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
            return _normalize([_interpolate(_list(0)), " component testing is available for this project"]);
          },
          "content": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["You can now use Cypress to develop and test individual components without running your whole application. Generate the config in just a few clicks."]);
          },
          "primaryAction": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Quick setup"]);
          },
          "secondaryAction": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Read our guide"]);
          },
          "dismissAction": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Give feedback"]);
          }
        },
        "enableNotifications": {
          "title": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Start debugging as soon as your first tests fail"]);
          },
          "subtitle": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Cypress can notify you about Cloud test runs for your branch."]);
          },
          "enableDesktopNotifications": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Enable desktop notifications"]);
          },
          "remindMeLater": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Remind me later"]);
          },
          "notificationsEnabledTitle": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Notifications Enabled"]);
          },
          "notificationsEnabledBody": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Nice, notifications are enabled!"]);
          },
          "dismissBanner": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Dismiss banner"]);
          }
        }
      },
      "runSelectedSpecs": (ctx) => {
        const { normalize: _normalize, interpolate: _interpolate, named: _named, plural: _plural } = ctx;
        return _plural([_normalize(["Run ", _interpolate(_named("n")), " spec"]), _normalize(["Run ", _interpolate(_named("n")), " specs"])]);
      },
      "testingTypePromo": {
        "e2e": {
          "header": {
            "title": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["End-to-end testing is not set up for this project"]);
            },
            "description": (ctx) => {
              const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
              return _normalize(["Try it out now, or ", _interpolate(_list(0)), " to learn more"]);
            },
            "link": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["read our guides"]);
            },
            "control": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Quick setup"]);
            }
          },
          "body": {
            "title": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["What is end-to-end testing?"]);
            },
            "description": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["End-to-end tests are perfect for testing your entire application from a user's perspective. Visit any URL and test full user journeys across pages and domains."]);
            },
            "listItem1": (ctx) => {
              const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
              return _normalize(["Test full user journeys across multiple pages using ", _interpolate(_list(0)), " to load up a web page for your test."]);
            },
            "listItem1Command": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["cy.visit()"]);
            },
            "listItem2": (ctx) => {
              const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
              return _normalize(["Work with live API calls or mock them with ", _interpolate(_list(0)), " to test all possible states of your app."]);
            },
            "listItem2Command": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["cy.intercept()"]);
            },
            "listItem3": (ctx) => {
              const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
              return _normalize(["Use ", _interpolate(_list(0)), " to maintain consistent browser sessions in your tests, and work with features like cookies and local storage, all while keeping your tests properly isolated."]);
            },
            "listItem3Command": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["cy.session()"]);
            }
          }
        },
        "ct": {
          "header": {
            "title": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Component testing is not set up for this project"]);
            },
            "description": (ctx) => {
              const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
              return _normalize(["Try it out now, or ", _interpolate(_list(0)), " to learn more"]);
            },
            "link": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["read our guides"]);
            },
            "control": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Quick setup"]);
            }
          },
          "body": {
            "title": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["What is Component Testing?"]);
            },
            "description": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Cypress Component Testing is a technique for developers to write, test, and debug components in isolation without ever leaving the browser - and without having to run your entire application."]);
            },
            "listItem1": (ctx) => {
              const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
              return _normalize(["Instead of visiting a page, use ", _interpolate(_list(0)), " to render your components directly in the browser."]);
            },
            "listItem1Command": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["cy.mount()"]);
            },
            "listItem2": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Test from a user's perspective with all the same Cypress commands that are available in end-to-end testing."]);
            },
            "listItem3": (ctx) => {
              const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
              return _normalize(["To get started, you can generate the config for ", _interpolate(_list(0)), ", ", _interpolate(_list(1)), ", and ", _interpolate(_list(2)), " (and ", _interpolate(_list(3)), "!)."]);
            },
            "listItem3Link": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["more"]);
            },
            "link": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Give feedback"]);
            }
          }
        }
      }
    },
    "noResults": {
      "defaultMessage": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["No results matched your search:"]);
      },
      "clearSearch": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Clear search"]);
      }
    },
    "sidebar": {
      "keyboardShortcuts": {
        "title": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Keyboard shortcuts"]);
        },
        "rerun": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Re-run tests"]);
        },
        "stop": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Stop tests"]);
        },
        "toggle": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Toggle specs list"]);
        },
        "studioSave": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Save Studio changes"]);
        }
      },
      "toggleLabel": {
        "expanded": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Collapse sidebar"]);
        },
        "collapsed": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Expand sidebar"]);
        }
      },
      "nav": {
        "ariaLabel": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Pages"]);
        }
      },
      "debug": {
        "debugFeature": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["New Debug feature"]);
        },
        "passed": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Relevant run passed"]);
        },
        "failed": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, named: _named, plural: _plural } = ctx;
          return _plural([_normalize(["Relevant run had ", _interpolate(_named("n")), " test failure"]), _normalize(["Relevant run had ", _interpolate(_named("n")), " test failures"])]);
        },
        "errored": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Relevant run had an error"]);
        },
        "erroredWithFailures": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, named: _named, plural: _plural } = ctx;
          return _plural([_normalize(["Relevant run had an error with ", _interpolate(_named("n")), " test failure"]), _normalize(["Relevant run had an error with ", _interpolate(_named("n")), " test failures"])]);
        },
        "passing": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Relevant run is passing"]);
        },
        "failing": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, named: _named, plural: _plural } = ctx;
          return _plural([_normalize(["Relevant run is failing with ", _interpolate(_named("n")), " test failure"]), _normalize(["Relevant run is failing with ", _interpolate(_named("n")), " test failures"])]);
        }
      },
      "runs": {
        "running": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Latest run is in progress"]);
        },
        "failing": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, named: _named, plural: _plural } = ctx;
          return _plural([_normalize(["Latest run is failing with ", _interpolate(_named("n")), " test failure"]), _normalize(["Latest run is failing with ", _interpolate(_named("n")), " test failures"])]);
        },
        "passed": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Latest run passed"]);
        },
        "failed": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, named: _named, plural: _plural } = ctx;
          return _plural([_normalize(["Latest run had ", _interpolate(_named("n")), " test failure"]), _normalize(["Latest run had ", _interpolate(_named("n")), " test failures"])]);
        },
        "cancelled": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Latest run has been cancelled"]);
        },
        "errored": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Latest run had an error"]);
        },
        "erroredWithFailures": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, named: _named, plural: _plural } = ctx;
          return _plural([_normalize(["Latest run had an error with ", _interpolate(_named("n")), " test failure"]), _normalize(["Latest run had an error with ", _interpolate(_named("n")), " test failures"])]);
        }
      }
    },
    "topNav": {
      "released": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Released"]);
      },
      "version": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Version"]);
      },
      "installed": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Installed"]);
      },
      "latest": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Latest"]);
      },
      "seeAllReleases": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["See all releases"]);
      },
      "runningOldVersion": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["You're currently running an old version of Cypress. Update to the latest version for the best experience."]);
      },
      "docsMenu": {
        "docsHeading": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Docs"]);
        },
        "gettingStartedTitle": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Getting started"]);
        },
        "firstTest": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Write your first test"]);
        },
        "testingApp": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Testing your app"]);
        },
        "organizingTests": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Organizing tests"]);
        },
        "referencesTitle": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["References"]);
        },
        "bestPractices": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Best practices"]);
        },
        "configuration": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Configuration"]);
        },
        "api": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["API"]);
        },
        "ciTitle": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Run in CI/CD"]);
        },
        "ciSetup": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Set up CI"]);
        },
        "fasterTests": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Run tests faster"]);
        },
        "prompts": {
          "ci1": {
            "title": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Configure CI"]);
            },
            "description": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Optimize your tests in CI by following these guides."]);
            },
            "seeOtherGuides": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["See other guides"]);
            },
            "intro": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Introduction to CI"]);
            },
            "learnTheBasics": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Learn the basics of running Cypress in CI."]);
            }
          },
          "orchestration1": {
            "title": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Run tests faster in CI"]);
            },
            "parallelTime": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["5 mins"]);
            },
            "noParallelTime": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["12 mins"]);
            },
            "withParallelization": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["with Parallelization"]);
            },
            "withoutParallelization": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["without Parallelization"]);
            },
            "intro": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["With Smart Orchestration, you'll be able to:"]);
            },
            "bullet1": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Run spec files in parallel"]);
            },
            "bullet2": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Prioritize failed specs to run first"]);
            },
            "bullet3": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Cancel CI runs on test failure"]);
            },
            "learnMore": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Learn more"]);
            }
          }
        }
      },
      "login": {
        "bodyInitial": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
          return _normalize(["Logging in gives you access to ", _interpolate(_list(0)), ". You can set up projects to be recorded and see test data from your project."]);
        },
        "bodySuccess": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
          return _normalize(["You are now logged in as ", _interpolate(_list(0)), "."]);
        },
        "bodyError": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["An error occurred while attempting to log in:"]);
        },
        "bodyBrowserError": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["The browser could not be opened."]);
        },
        "bodyBrowserErrorDetails": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Cypress was unable to open your installed browser. To continue logging in, please paste this URL into your web browser:"]);
        },
        "cloud": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Cypress Cloud"]);
        },
        "titleInitial": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Log in to Cypress"]);
        },
        "titleSuccess": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Login successful"]);
        },
        "titleFailed": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Login failed"]);
        },
        "titleBrowserError": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Browser error"]);
        },
        "actionLogin": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Log in"]);
        },
        "profileMenuLabel": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Profile and logout"]);
        },
        "actionLogout": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Log out"]);
        },
        "actionContinue": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Continue"]);
        },
        "actionOpening": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Opening browser"]);
        },
        "actionWaiting": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Waiting for you to log in"]);
        },
        "actionTryAgain": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Try again"]);
        },
        "actionCancel": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Cancel"]);
        }
      },
      "updateCypress": {
        "title": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Upgrade to Cypress"]);
        },
        "currentlyRunning": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
          return _normalize(["You are currently running Version ", _interpolate(_list(0)), " of Cypress. "]);
        },
        "pasteToUpgradeProject": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
          return _normalize(["To upgrade to the latest version for your project, first ", _interpolate(_list(0)), ", then paste the command below into your terminal:"]);
        },
        "pasteToUpgradeGlobal": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
          return _normalize(["To upgrade to the latest version, first ", _interpolate(_list(0)), ", then paste the command below into your terminal:"]);
        },
        "rememberToCloseInsert": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["close this app"]);
        }
      },
      "upgradeText": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Upgrade"]);
      },
      "global": {
        "projects": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Projects"]);
        }
      }
    },
    "launchpadErrors": {
      "generic": {
        "configErrorTitle": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Cypress configuration error"]);
        },
        "retryButton": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Try again"]);
        },
        "stackTraceLabel": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Stack trace"]);
        },
        "docsButton": {
          "configGuide": {
            "text": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Configuration guide"]);
            },
            "link": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["https://on.cypress.io/configuration"]);
            }
          },
          "cloudGuide": {
            "text": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Cypress Cloud guide"]);
            },
            "link": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["https://on.cypress.io/features-dashboard"]);
            }
          },
          "docsHomepage": {
            "text": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Cypress docs"]);
            },
            "link": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["https://docs.cypress.io"]);
            }
          }
        }
      },
      "noInternet": {
        "header": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["You have no internet connection"]);
        },
        "message": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Check your internet connection to pull the latest data from Cypress Cloud."]);
        },
        "connectProject": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Please check your internet connection to resolve this issue. When your internet connection is fixed, we will automatically attempt to fetch your latest runs for this project."]);
        }
      }
    },
    "setupPage": {
      "projectSetup": {
        "frameworkLabel": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Front-end framework"]);
        },
        "frameworkPlaceholder": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Pick a framework"]);
        },
        "bundlerLabel": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Bundler"]);
        },
        "bundlerPlaceholder": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Pick a bundler"]);
        },
        "languageLabel": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Language"]);
        },
        "configFileLanguageLabel": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Cypress config file"]);
        },
        "detected": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["(detected)"]);
        },
        "browseIntegrations": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Browse our list of third-party framework integrations"]);
        },
        "communityFrameworkDefinitionProblem": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Community framework definition problem"]);
        },
        "communityDependenciesCouldNotBeParsed": (ctx) => {
          const { normalize: _normalize, plural: _plural } = ctx;
          return _plural([_normalize(["This project has a community framework definition installed that could not be loaded. It is located at the following path:"]), _normalize(["This project has some community framework definitions installed that could not be loaded. They are located at the following paths:"])]);
        },
        "seeFrameworkDefinitionDocumentation": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
          return _normalize(["See the ", _interpolate(_list(0)), " for more information about creating, installing, and troubleshooting third party definitions."]);
        },
        "frameworkDefinitionDocumentation": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["framework definition documentation"]);
        }
      },
      "step": {
        "continue": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Continue"]);
        },
        "next": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Next step"]);
        },
        "back": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Back"]);
        },
        "skip": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Skip"]);
        }
      },
      "configFile": {
        "createManually": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Create file manually"]);
        },
        "skippedLabel": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Skipped"]);
        },
        "changesRequiredLabel": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Changes required"]);
        },
        "changesRequiredBadge": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Changes required"]);
        },
        "changesRequiredDescription": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
          return _normalize(["Please merge the code below with your existing ", _interpolate(_list(0))]);
        }
      },
      "testingCard": {
        "configured": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Configured"]);
        },
        "notConfigured": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Not Configured"]);
        },
        "running": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Running"]);
        },
        "chooseABrowser": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Choose a browser"]);
        },
        "reconfigure": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Reconfigure"]);
        },
        "disabled": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Disabled"]);
        }
      }
    },
    "globalPage": {
      "empty": {
        "title": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Add project"]);
        },
        "helper": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Get started by adding your first project below."]);
        },
        "dropText": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
          return _normalize(["Drag your project directory here or ", _interpolate(_list(0))]);
        },
        "browseManually": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["browse manually."]);
        }
      },
      "settings": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Settings"]);
      },
      "saveChanges": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Save changes"]);
      },
      "cancel": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Cancel"]);
      },
      "externalEditorPreferences": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["External editor preferences"]);
      },
      "externalEditorPreferencesDescription1": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Select your preference for opening files on your system."]);
      },
      "externalEditorPreferencesDescription2": (ctx) => {
        const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
        return _normalize(["You can change this later in the ", _interpolate(_list(0)), " page."]);
      },
      "recentProjectsHeader": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Recent projects"]);
      },
      "searchPlaceholder": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Search projects"]);
      },
      "addProjectButton": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Add project"]);
      },
      "projectActions": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Project actions"]);
      },
      "removeProject": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Remove project"]);
      },
      "openInIDE": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Open in IDE"]);
      },
      "openInFinder": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Open in Finder"]);
      },
      "noResultsMessage": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["No projects matched your search:"]);
      }
    },
    "welcomePage": {
      "review": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Review the differences between each testing type"]);
      },
      "codeExample": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Code example"]);
      },
      "compareTypes": {
        "content": {
          "e2eTitle": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["End-to-end tests"]);
          },
          "e2eBullet1": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Visit URLs via"]);
          },
          "e2eBullet2": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Test flows and functionality across multiple pages"]);
          },
          "e2eBullet3": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Ideal for testing integrated flows in CD workflows"]);
          },
          "componentTitle": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Component tests"]);
          },
          "ctBullet1": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Import components via"]);
          },
          "ctBullet2": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Test individual components of a design system in isolation"]);
          },
          "ctBullet3": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Ideal for testing isolated flows and components in CI"]);
          }
        },
        "modalTitle": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Key differences"]);
        }
      },
      "title": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Welcome to Cypress!"]);
      }
    },
    "settingsPage": {
      "config": {
        "title": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Resolved configuration"]);
        },
        "description": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
          return _normalize(["Since the options in the ", _interpolate(_list(0)), " can be set dynamically by your development environment, please refer to the legend below to understand how the configuration options are resolved."]);
        },
        "legend": {
          "env": {
            "label": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["env"]);
            },
            "description": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Set from environment variables"]);
            }
          },
          "default": {
            "label": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["default"]);
            },
            "description": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Default values"]);
            }
          },
          "dynamic": {
            "label": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["dynamic"]);
            },
            "description": (ctx) => {
              const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
              return _normalize(["Set by the ", _interpolate(_list(0)), " function"]);
            }
          },
          "cli": {
            "label": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["cli"]);
            },
            "description": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Set from CLI arguments"]);
            }
          },
          "config": {
            "label": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["config"]);
            },
            "description": (ctx) => {
              const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
              return _normalize(["Set from ", _interpolate(_list(0))]);
            }
          }
        }
      },
      "proxy": {
        "title": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Proxy settings"]);
        },
        "description": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Cypress auto-detected the following proxy settings from your operating system."]);
        },
        "bypassList": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Proxy bypass list"]);
        },
        "proxyServer": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Proxy server"]);
        }
      },
      "editor": {
        "title": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["External editor"]);
        },
        "description": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["External editor to use for editing code opened when using Cypress."]);
        },
        "noEditorSelectedPlaceholder": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Choose your editor..."]);
        },
        "customPathPlaceholder": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["/path/to/editor"]);
        }
      },
      "notifications": {
        "title": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Desktop notifications"]);
        },
        "description": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Cypress can notify you about Cloud test runs for your branch."]);
        },
        "notifyMeWhenRunStarts": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Notify me when a run starts"]);
        },
        "notifyMeWhenRunIsFailing": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Notify me when a run begins to fail"]);
        },
        "notifyMeWhenRunCompletes": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Notify me when a run completes"]);
        },
        "passed": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Passed"]);
        },
        "failed": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Failed"]);
        },
        "canceled": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Canceled"]);
        },
        "errored": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Errored"]);
        },
        "sendTestNotification": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Send a test notification"]);
        },
        "notReceivingNotifications": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
          return _normalize(["Not receiving notifications? You might need to allow notifications from Cypress in your computer's system preferences. ", _interpolate(_list(0))]);
        },
        "testNotificationTitle": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Hello From Cypress"]);
        },
        "testNotificationBody": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["This is a test notification"]);
        },
        "troubleshoot": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Troubleshoot"]);
        },
        "enableNotificationsLabel": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["To modify notification settings, enable desktop notifications"]);
        }
      },
      "projectId": {
        "title": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Project ID"]);
        },
        "description": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
          return _normalize(["The Project ID configured for this project inside of Cypress Cloud. ", _interpolate(_list(0))]);
        }
      },
      "specPattern": {
        "title": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Spec patterns"]);
        },
        "description": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
          return _normalize(["The globs pointing Cypress to your spec files configured for this project. ", _interpolate(_list(0))]);
        }
      },
      "recordKey": {
        "title": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Record key"]);
        },
        "description": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
          return _normalize(["The Record Key configured for this project inside of Cypress Cloud. ", _interpolate(_list(0))]);
        },
        "manageKeys": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Manage keys"]);
        },
        "errorEmpty": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["You do not have permission to record on this project"]);
        },
        "errorEmptyButton": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Request record rights"]);
        },
        "errorAccess": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["You do not have permission to view this project"]);
        },
        "errorAccessPending": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Your request for access to this project is pending approval"]);
        },
        "errorAccessButton": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Request access"]);
        },
        "errorNotFound": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Project not found"]);
        },
        "errorNotFoundButton": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Connect it again"]);
        },
        "errorNotLoggedIn": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["You are not connected to Cypress Cloud"]);
        },
        "errorNotLoggedInButton": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Log in to Cypress Cloud"]);
        }
      },
      "project": {
        "title": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Project settings"]);
        },
        "description": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Review the configuration settings currently in use for this project."]);
        }
      },
      "cloud": {
        "title": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Cypress Cloud settings"]);
        },
        "description": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Review the configuration settings for recording to Cypress Cloud."]);
        }
      },
      "experiments": {
        "title": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Experiments"]);
        },
        "description": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
          return _normalize(["If you'd like to try out new features that we're working on, you can enable beta features for your project by turning on the experimental features you'd like to try. ", _interpolate(_list(0))]);
        },
        "experimentalCspAllowList": {
          "name": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["CSP Allow List"]);
          },
          "description": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Enables Cypress to selectively permit Content-Security-Policy and Content-Security-Policy-Report-Only header directives, including those that might otherwise block Cypress from running."]);
          }
        },
        "experimentalInteractiveRunEvents": {
          "name": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Interactive run events"]);
          },
          "description": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Allows listening to the [`before:run`](https://on.cypress.io/before-run), [`after:run`](https://on.cypress.io/after-run), [`before:spec`](https://on.cypress.io/before-spec), and [`after:spec`](https://on.cypress.io/after-spec) events in plugins during interactive mode."]);
          }
        },
        "experimentalModifyObstructiveThirdPartyCode": {
          "name": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Modify obstructive third party code"]);
          },
          "description": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Applies `modifyObstructiveCode` to third party `.html` and `.js`, removes subresource integrity, and modifies the user agent in Electron."]);
          }
        },
        "retries": {
          "experimentalStrategy": {
            "name": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Retries Strategy"]);
            },
            "description": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(['Applies a strategy for test retries according to your "flake tolerance"; options are `detect-flake-but-always-fail` or `detect-flake-and-pass-on-threshold`.']);
            }
          },
          "experimentalOptions": {
            "name": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Retries Strategy Options"]);
            },
            "description": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Sets retries strategy-specific options like `maxRetries`, `passesRequired`, and `stopIfAnyPassed`."]);
            }
          }
        },
        "experimentalSingleTabRunMode": {
          "name": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Single tab run mode"]);
          },
          "description": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Runs all component specs in a single tab, trading spec isolation for faster run mode execution."]);
          }
        },
        "experimentalSourceRewriting": {
          "name": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Source rewriting"]);
          },
          "description": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Enables AST-based JS/HTML rewriting. This may fix issues caused by the existing regex-based JS/HTML replacement algorithm. See [#5273](https://github.com/cypress-io/cypress/issues/5273) for details."]);
          }
        },
        "experimentalPromptCommand": {
          "name": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Prompt command"]);
          },
          "description": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Enables support for `cy.prompt`, an AI-powered command that turns natural language steps into executable Cypress test code."]);
          }
        },
        "experimentalWebKitSupport": {
          "name": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["WebKit Support"]);
          },
          "description": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Adds support for testing in the WebKit browser engine used by Safari. See https://on.cypress.io/webkit-experiment for more information."]);
          }
        },
        "experimentalRunAllSpecs": {
          "name": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Run All Specs"]);
          },
          "description": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(['Enables the "Run All Specs" UI feature, allowing the execution of multiple specs sequentially.']);
          }
        },
        "experimentalOriginDependencies": {
          "name": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Origin Dependencies"]);
          },
          "description": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Enables support for `Cypress.require()` for including dependencies within the `cy.origin()` callback."]);
          }
        },
        "experimentalMemoryManagement": {
          "name": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Memory Management"]);
          },
          "description": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Enables support for improved memory management within Chromium-based browsers."]);
          }
        }
      },
      "device": {
        "title": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Device settings"]);
        },
        "description": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Review the configuration settings currently in use for this device."]);
        }
      },
      "testingPreferences": {
        "title": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Testing preferences"]);
        },
        "description": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Configure your testing environment with these flags"]);
        },
        "autoScrollingEnabled": {
          "title": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Auto scrolling enabled"]);
          },
          "description": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Scroll behavior when running tests."]);
          }
        }
      },
      "footer": {
        "text": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, named: _named } = ctx;
          return _normalize(["You can reconfigure the ", _interpolate(_named("testingType")), " testing settings for this project if you're experiencing issues with your Cypress configuration."]);
        },
        "button": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Configuration guide"]);
        },
        "buttonLink": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["https://on.cypress.io/configuration"]);
        }
      }
    },
    "runs": {
      "slideshow": {
        "tour": {
          "title": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Record your first test run to Cypress Cloud"]);
          },
          "description": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["You're almost there! Start recording your test runs to Cypress Cloud by running the command above in your local or CI terminal."]);
          }
        },
        "step1": {
          "title": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Monitor test failures in real time"]);
          },
          "description": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Uncover high-risk failed and flaky tests in CI before they become a problem in production."]);
          }
        },
        "step2": {
          "title": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Check if failed CI test runs also fail locally"]);
          },
          "description": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Use the Test Runner to run only the tests that failed in your last recorded run."]);
          }
        },
        "step3": {
          "title": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Build, test, and ship with confidence 🚀"]);
          },
          "description": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["With Cypress Cloud, you can proactively identify, investigate, and resolve failures, and be a quality champion."]);
          }
        },
        "controls": {
          "step": (ctx) => {
            const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
            return _normalize([_interpolate(_list(0)), " of ", _interpolate(_list(1))]);
          },
          "next": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Next"]);
          },
          "reset": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Reset"]);
          },
          "view": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["View Tour"]);
          }
        }
      },
      "connect": {
        "title": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Connect to view your recorded runs"]);
        },
        "description": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Cypress Cloud is built for teams who need to do testing at scale."]);
        },
        "buttonUser": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Connect to Cypress Cloud"]);
        },
        "buttonProject": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Connect a Cypress Cloud project"]);
        },
        "modal": {
          "title": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Connect project"]);
          },
          "cancel": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Cancel"]);
          },
          "createOrg": {
            "description": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["You need to create an organization in Cypress Cloud to continue."]);
            },
            "button": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Create organization"]);
            },
            "waitingButton": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Waiting for you to create an organization..."]);
            },
            "refreshButton": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Refresh organizations list"]);
            }
          },
          "selectProject": {
            "organization": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Organization"]);
            },
            "noOrganizationSelectedError": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["required when creating a new project"]);
            },
            "manageOrgs": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Manage organizations"]);
            },
            "project": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Project"]);
            },
            "projectName": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Project name"]);
            },
            "projectNameDisclaimer": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["(You can change this later)"]);
            },
            "createNewProject": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Create new"]);
            },
            "chooseExistingProject": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Choose an existing project"]);
            },
            "newProjectAccess": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Project access"]);
            },
            "privateLabel": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Private"]);
            },
            "privateDescription": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Only invited users can view recorded test results."]);
            },
            "publicLabel": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Public"]);
            },
            "publicDescription": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Anyone can view recorded test results."]);
            },
            "connectProject": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Connect project"]);
            },
            "createProject": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Create project"]);
            },
            "placeholderOrganizations": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Pick an organization"]);
            },
            "placeholderProjects": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Pick a project"]);
            },
            "placeholderProjectsPending": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Pick an organization first"]);
            }
          },
          "connectManually": {
            "title": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Connect project manually"]);
            },
            "warning": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["We couldn't add the projectId to your cypress.config.js file automatically."]);
            },
            "mainMessage": (ctx) => {
              const { normalize: _normalize, interpolate: _interpolate, named: _named } = ctx;
              return _normalize(["Manually add the ", _interpolate(_named("projectId")), " to the root of the config object in your ", _interpolate(_named("configFile")), " file."]);
            },
            "waitingButton": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Waiting for you to add the projectId..."]);
            }
          }
        },
        "errors": {
          "baseError": {
            "title": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Cypress Cloud error"]);
            }
          },
          "internalServerError": {
            "title": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Cannot connect to Cypress Cloud"]);
            },
            "description": (ctx) => {
              const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
              return _normalize(["The request times out when trying to retrieve the info from Cypress Cloud. Please refresh the page to try again and visit our ", _interpolate(_list(0)), " if this behavior continues."]);
            },
            "link": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Support Page"]);
            }
          }
        }
      },
      "connectSuccessAlert": {
        "title": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["This project is now connected to Cypress Cloud!"]);
        },
        "item1": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, named: _named } = ctx;
          return _normalize(["We automatically added the ", _interpolate(_named("projectId")), " to your ", _interpolate(_named("configFile")), " file."]);
        },
        "item2": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
          return _normalize(["Please ensure that your ", _interpolate(_list(0)), " file is checked into source control."]);
        }
      },
      "empty": {
        "title": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Copy the command below to record your first run"]);
        },
        "description": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
          return _normalize(["Review the ", _interpolate(_list(0)), " to Cypress Cloud"]);
        },
        "link": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["guide on recording runs"]);
        },
        "gitRepositoryNotDetected": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Git repository not detected"]);
        },
        "ensureGitSetupCorrectly": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Cypress uses Git to associate runs with your local state. As a fallback, all runs for your project are shown. Please ensure that version control is set up correctly."]);
        },
        "noRunsFoundForBranch": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["No runs found for your branch"]);
        },
        "noRunsForBranchMessage": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Cypress uses Git to show runs for your branch. As a fallback, all runs for your project are shown. Ensure that version control is properly configured and that you are sending Git information to Cypress Cloud."]);
        }
      },
      "results": {
        "skipped": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["skipped"]);
        },
        "pending": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["pending"]);
        },
        "passed": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["passed"]);
        },
        "failed": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["failed"]);
        }
      },
      "errors": {
        "notFound": {
          "title": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Couldn't find your project"]);
          },
          "description": (ctx) => {
            const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
            return _normalize(["We were unable to find an existing project matching the ", _interpolate(_list(0)), " set in your Cypress config file. You can reconnect with an existing project or create a new project."]);
          },
          "button": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Reconnect your project"]);
          }
        },
        "unauthorized": {
          "title": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Request access to view the recorded runs"]);
          },
          "description": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["This is a private project that you do not currently have access to. Please request access from the project owner in order to view the list of runs."]);
          },
          "button": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Request access"]);
          }
        },
        "unauthorizedRequested": {
          "title": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Your access request for this project has been sent."]);
          },
          "description": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["The owner of this project has been notified of your request. We'll notify you via email when your access request has been granted."]);
          },
          "button": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Request Sent"]);
          }
        },
        "connectionFailed": {
          "title": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Cannot connect to Cypress Cloud"]);
          },
          "description": (ctx) => {
            const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
            return _normalize(["The request times out when trying to retrieve the recorded runs from Cypress Cloud. Please refresh the page to try again and visit out ", _interpolate(_list(0)), " if this behavior continues."]);
          },
          "link": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Support page"]);
          },
          "button": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Try again"]);
          }
        }
      },
      "card": {
        "branchName": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Branch Name:"]);
        },
        "commitAuthor": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Commit Author:"]);
        },
        "totalDuration": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Run Total Duration:"]);
        },
        "debugLabel": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Debug"]);
        },
        "debugDescription": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, named: _named } = ctx;
          return _normalize(["View debug for run ", _interpolate(_named("runNumber"))]);
        },
        "noDebugAvailable": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["You can only debug runs for your most recent commits."]);
        }
      },
      "layout": {
        "viewCloudRuns": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["View runs in Cypress Cloud"]);
        },
        "checkedOut": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Checked out"]);
        }
      }
    },
    "debugPage": {
      "openFile": {
        "openInIDE": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Open in IDE"]);
        },
        "notFoundLocally": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Opening in IDE is disabled because the spec is not found in this project"]);
        }
      },
      "limit": {
        "title": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Cypress renders up to 100 failed test results"]);
        },
        "message": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, named: _named, plural: _plural } = ctx;
          return _plural([_normalize(["This run has ", _interpolate(_named("n")), " failed tests"]), _normalize(["This run has ", _interpolate(_named("n")), " failed test"]), _normalize(["This run has ", _interpolate(_named("n")), " failed tests"])]);
        },
        "link": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["See all test results in Cypress Cloud"]);
        }
      },
      "runFailures": {
        "btn": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Run Failures"]);
        },
        "notFoundLocally": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Spec was not found locally"]);
        },
        "switchTestingTypeInfo": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, named: _named, plural: _plural } = ctx;
          return _plural([_normalize(["There is ", _interpolate(_named("n")), " ", _interpolate(_named("testingType")), " test failing in this spec. To run it locally switch to ", _interpolate(_named("testingType")), " testing."]), _normalize(["There are ", _interpolate(_named("n")), " ", _interpolate(_named("testingType")), " tests failing in this spec. To run them locally switch to ", _interpolate(_named("testingType")), " testing."])]);
        },
        "switchTestingTypeAction": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, named: _named } = ctx;
          return _normalize(["Switch to ", _interpolate(_named("testingType")), " testing"]);
        }
      },
      "switchToRun": (ctx) => {
        const { normalize: _normalize, interpolate: _interpolate, named: _named } = ctx;
        return _normalize(["Switch to run number ", _interpolate(_named("runNumber"))]);
      },
      "overLimit": {
        "usageExceededTitle": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["You've reached your monthly test result limit"]);
        },
        "usageExceededUserMessage": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, named: _named } = ctx;
          return _normalize(["Your Free Cypress Cloud plan includes ", _interpolate(_named("numberTests")), " monthly recorded test results. Contact your Cypress Cloud admin to upgrade your plan and view more test results."]);
        },
        "usageExceededAdminMessage": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, named: _named } = ctx;
          return _normalize(["Your Free Cypress Cloud plan includes ", _interpolate(_named("numberTests")), " monthly recorded test results. Upgrade your plan now to view more test results."]);
        },
        "retentionExceededTitle": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Beyond data retention"]);
        },
        "retentionExceededMessage": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, named: _named } = ctx;
          return _normalize(["Your data retention limit is ", _interpolate(_named("limitDays")), " days and these tests ran ", _interpolate(_named("runAgeDays")), " days ago. Upgrade your plan to increase your data retention limit."]);
        },
        "upgradePlan": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Upgrade plan"]);
        },
        "contactAdmin": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Contact admin"]);
        }
      },
      "wellDone": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Well Done!"]);
      },
      "allYourTestsPassed": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["All your tests passed."]);
      },
      "incomplete": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Incomplete"]);
      },
      "specsSkipped": (ctx) => {
        const { normalize: _normalize, interpolate: _interpolate, named: _named, plural: _plural } = ctx;
        return _plural([_normalize([_interpolate(_named("totalSkippedSpecs")), " of ", _interpolate(_named("n")), " spec skipped"]), _normalize([_interpolate(_named("totalSkippedSpecs")), " of ", _interpolate(_named("n")), " specs skipped"])]);
      },
      "runHasNoTests": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Run has no tests"]);
      },
      "theRunStartedButNeverCompleted": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["The run started but never completed. This can happen when the run is cancelled from CI or when Cypress crashes while running tests."]);
      },
      "checkYourCiLogs": (ctx) => {
        const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
        return _normalize(["Check your ", _interpolate(_list(0)), " logs for more information."]);
      },
      "archiveThisRun": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Archive this run to remove it from the runs list and analytics."]);
      },
      "manuallyCancelled": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Manually cancelled"]);
      },
      "mostRecentRun": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["You are on the most recent run"]);
      },
      "switchToLatestRun": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Switch to latest run"]);
      },
      "switchRun": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Switch run"]);
      },
      "foundMoreThan100Runs": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["We found more than 100 runs."]);
      },
      "goToCypressCloud": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Go to Cypress Cloud to see all runs"]);
      },
      "emptyStates": {
        "debugDirectlyInCypress": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Debug failed CI runs directly in Cypress"]);
        },
        "reviewRerunAndDebug": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Review, rerun, and debug failed CI test runs that are recorded to Cypress Cloud – all from within your local Cypress app."]);
        },
        "connectToCypressCloud": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Connect to Cypress Cloud"]);
        },
        "connect": {
          "title": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Connect to debug your tests"]);
          },
          "description": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Cypress Cloud is built for teams who need to do testing at scale."]);
          }
        },
        "noProjectTestMessage": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Connect a Cypress Cloud project to locally debug failed CI test runs"]);
        },
        "noRuns": {
          "title": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Copy the command below to record your first run"]);
          },
          "description": (ctx) => {
            const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
            return _normalize(["Review the ", _interpolate(_list(0)), " to Cypress Cloud"]);
          },
          "link": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["guide on recording runs"]);
          }
        },
        "gitRepositoryNotDetected": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Git repository not detected"]);
        },
        "ensureGitSetupCorrectly": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Cypress uses Git to associate runs with your local state. Please ensure that Git is properly configured for your project."]);
        },
        "learnAboutRecordingSrText": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["about recording a run to Cypress Cloud"]);
        },
        "learnAboutDebuggingSrText": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["about debugging CI failures in Cypress"]);
        },
        "learnAboutProjectSetupSrText": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["about project setup in Cypress"]);
        },
        "noRunsFoundForBranch": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["No runs found for your branch"]);
        },
        "noRunsForBranchMessage": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Cypress uses Git to show runs for your branch. Ensure that version control is properly configured and that you are sending Git information to Cypress Cloud."]);
        },
        "slideshow": {
          "tour": {
            "title": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Record your first test run to Cypress Cloud"]);
            },
            "description": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["You're almost there! Start recording your test runs to Cypress Cloud by running the command below in your local or CI terminal."]);
            }
          },
          "step1": {
            "title": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Review how many tests failed during a CI test run"]);
            },
            "description": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["The Debug page shows the latest completed test run for your current checked out commit."]);
            }
          },
          "step2": {
            "title": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Check if failed CI test runs also fail locally"]);
            },
            "description": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Use the Test Runner to run only the tests that failed in your last recorded run."]);
            }
          },
          "step3": {
            "title": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Locally debug failed test runs with visual artifacts"]);
            },
            "description": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Easily review screenshots, videos and logs from your tests."]);
            }
          },
          "controls": {
            "step": (ctx) => {
              const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
              return _normalize([_interpolate(_list(0)), " of ", _interpolate(_list(1))]);
            },
            "next": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Next"]);
            },
            "reset": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["Reset"]);
            },
            "view": (ctx) => {
              const { normalize: _normalize } = ctx;
              return _normalize(["View Tour"]);
            }
          }
        }
      },
      "pending": {
        "title": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Testing in progress..."]);
        },
        "titleCompletionScheduled": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Scheduled to complete..."]);
        },
        "scheduledTime": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, named: _named } = ctx;
          return _normalize(["Scheduled to complete in ", _interpolate(_named("time"))]);
        },
        "scheduledCompletionExpired": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Scheduled to complete soon"]);
        },
        "failuresHere": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Failures will be displayed here"]);
        }
      },
      "specCounts": {
        "whenRunning": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, named: _named, plural: _plural } = ctx;
          return _plural([_normalize([_interpolate(_named("completed")), " of ", _interpolate(_named("total")), " specs completed"]), _normalize([_interpolate(_named("completed")), " of ", _interpolate(_named("total")), " spec completed"]), _normalize([_interpolate(_named("completed")), " of ", _interpolate(_named("total")), " specs completed"])]);
        },
        "whenCompleted": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, named: _named, plural: _plural } = ctx;
          return _plural([_normalize([_interpolate(_named("n")), " specs"]), _normalize([_interpolate(_named("n")), " spec"]), _normalize([_interpolate(_named("n")), " specs"])]);
        }
      },
      "header": {
        "runUrl": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["View in Cypress Cloud"]);
        },
        "commitsAhead": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, named: _named, plural: _plural } = ctx;
          return _plural([_normalize(["You are ", _interpolate(_named("n")), " commit ahead"]), _normalize(["You are ", _interpolate(_named("n")), " commits ahead"])]);
        }
      },
      "stats": {
        "groups": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, named: _named, plural: _plural } = ctx;
          return _plural([_normalize([_interpolate(_named("n")), " group"]), _normalize([_interpolate(_named("n")), " groups"])]);
        },
        "operatingSystems": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, named: _named, plural: _plural } = ctx;
          return _plural([_normalize([_interpolate(_named("n")), " operating system"]), _normalize([_interpolate(_named("n")), " operating systems"])]);
        },
        "browsers": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, named: _named, plural: _plural } = ctx;
          return _plural([_normalize([_interpolate(_named("n")), " browser"]), _normalize([_interpolate(_named("n")), " browsers"])]);
        }
      },
      "artifacts": {
        "replay": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["View Test Replay"]);
        },
        "stdout": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["View Log"]);
        },
        "screenshots": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["View Screenshots"]);
        },
        "video": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["View Video"]);
        }
      }
    },
    "testingType": {
      "modalTitle": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Choose a testing type"]);
      },
      "e2e": {
        "name": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["E2E Testing"]);
        },
        "description": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Build and test the entire experience of your application from end-to-end to ensure each flow matches your expectations."]);
        }
      },
      "component": {
        "name": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Component Testing"]);
        },
        "description": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Build and test your components from your design system in isolation in order to ensure each state matches your expectations."]);
        }
      },
      "componentDisabled": {
        "description": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, list: _list } = ctx;
          return _normalize(["To test your components you must ", _interpolate(_list(0)), " as an NPM dependency for this project."]);
        },
        "link": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["install Cypress"]);
        }
      }
    },
    "majorVersionWelcome": {
      "title": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["What's New in Cypress"]);
      },
      "actionContinue": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Continue"]);
      },
      "linkReleaseNotes": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["View full changelog"]);
      }
    },
    "setupWizard": {
      "selectFramework": {
        "title": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Project setup"]);
        },
        "description": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Confirm the front-end framework and bundler used in your project."]);
        }
      },
      "installDependencies": {
        "title": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Install dev dependencies"]);
        },
        "description": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Based on your previous selection, the following dependencies are required."]);
        },
        "pasteCommand": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Paste this command into your terminal to install the following packages:"]);
        },
        "waitForInstall": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Waiting for you to install the dependencies..."]);
        },
        "installed": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["installed"]);
        },
        "pendingInstall": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["pending installation"]);
        },
        "installationAlertSuccess": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["You've successfully installed all required dependencies."]);
        }
      },
      "configFiles": {
        "title": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Configuration files"]);
        },
        "description": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["We added the following files to your project:"]);
        }
      },
      "chooseBrowser": {
        "title": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Choose a browser"]);
        },
        "description": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, named: _named } = ctx;
          return _normalize(["Choose your preferred browser for ", _interpolate(_named("testingType")), " testing."]);
        }
      }
    },
    "e2eProjectSetup": {
      "title": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Project setup"]);
      },
      "description": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Confirm your project's preferred language."]);
      }
    },
    "openBrowser": {
      "startComponent": (ctx) => {
        const { normalize: _normalize, interpolate: _interpolate, named: _named } = ctx;
        return _normalize(["Start Component Testing in ", _interpolate(_named("browser"))]);
      },
      "startE2E": (ctx) => {
        const { normalize: _normalize, interpolate: _interpolate, named: _named } = ctx;
        return _normalize(["Start E2E Testing in ", _interpolate(_named("browser"))]);
      },
      "openingComponent": (ctx) => {
        const { normalize: _normalize, interpolate: _interpolate, named: _named } = ctx;
        return _normalize(["Opening Component Testing in ", _interpolate(_named("browser"))]);
      },
      "openingE2E": (ctx) => {
        const { normalize: _normalize, interpolate: _interpolate, named: _named } = ctx;
        return _normalize(["Opening E2E Testing in ", _interpolate(_named("browser"))]);
      },
      "running": (ctx) => {
        const { normalize: _normalize, interpolate: _interpolate, named: _named } = ctx;
        return _normalize(["Running ", _interpolate(_named("browser"))]);
      },
      "focus": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Focus"]);
      },
      "close": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Close"]);
      },
      "switchTestingType": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Switch testing type"]);
      }
    },
    "runner": {
      "header": {
        "reviewDocs": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Review the docs"]);
        },
        "troubleRendering": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["if you're having trouble rendering your components properly."]);
        }
      },
      "snapshot": {
        "highlightsLabel": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Highlights"]);
        },
        "testsRunningError": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Cannot show snapshot while tests are running"]);
        },
        "snapshotMissingError": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["The snapshot is missing. Displaying current state of the DOM."]);
        },
        "defaultTitle": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["DOM snapshot"]);
        },
        "pinnedTitle": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Pinned"]);
        }
      },
      "selectorPlayground": {
        "matches": (ctx) => {
          const { normalize: _normalize, interpolate: _interpolate, named: _named, plural: _plural } = ctx;
          return _plural([_normalize(["No matches"]), _normalize([_interpolate(_named("n")), " match"]), _normalize([_interpolate(_named("n")), " matches"])]);
        },
        "playgroundTooltip": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Click an element to see a suggested selector"]);
        },
        "copyTooltip": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Copy to clipboard"]);
        },
        "copyTooltipAction": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Copied!"]);
        },
        "printTooltip": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Print to console"]);
        },
        "printTooltipAction": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Printed!"]);
        },
        "invalidSelector": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Invalid"]);
        },
        "selectorMethodsLabel": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Selector methods"]);
        },
        "toggle": (ctx) => {
          const { normalize: _normalize } = ctx;
          return _normalize(["Toggle playground"]);
        }
      },
      "automation": {
        "disconnected": {
          "title": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["The Cypress extension has disconnected."]);
          },
          "description": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Cypress cannot run tests without this extension."]);
          },
          "reload": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Reload the browser"]);
          }
        },
        "missing": {
          "title": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["The Cypress extension is missing."]);
          },
          "description": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Cypress cannot run tests without this extension. Please choose another browser."]);
          }
        },
        "shared": {
          "link": (ctx) => {
            const { normalize: _normalize } = ctx;
            return _normalize(["Read more about browser management"]);
          }
        }
      }
    },
    "warnings": {
      "retry": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Try again"]);
      }
    },
    "versions": {
      "alpha": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Alpha"]);
      },
      "community": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Community"]);
      },
      "beta": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["Beta"]);
      },
      "new": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["New"]);
      },
      "communityNote": (ctx) => {
        const { normalize: _normalize } = ctx;
        return _normalize(["This is a community provided definition."]);
      }
    }
  } }
);
const rawJsonMessages = '{\n  "links": {\n    "learnMore": "Learn more.",\n    "learnMoreButton": "Learn more",\n    "needHelp": "Need help"\n  },\n  "components": {\n    "modal": {\n      "dismiss": "Dismiss"\n    },\n    "fileSearch": {\n      "noMatchesForExtension": "No files found when searching for the file extension",\n      "noMatchesForFileName": "",\n      "byFilenameInput": "Search by filename...",\n      "byExtensionInput": "Extensions to include e.g. *.tsx, *.vue",\n      "expandToSearchByExtensions": "Search for files with extensions e.g. *.tsx, *.vue",\n      "matchesIndicatorEmptyFileSearch": "No matches | 1 match | { denominator } matches",\n      "matchesIndicator": "No matches | {numerator} of {denominator} matches | {numerator} of {denominator} matches"\n    },\n    "button": {\n      "cancel": "Cancel",\n      "back": "Back"\n    },\n    "loading": {\n      "config": {\n        "title": "Initializing config...",\n        "description": "Please wait while we load your project and find browsers installed on your system"\n      }\n    },\n    "select": {\n      "placeholder": "Choose an option..."\n    },\n    "alert": {\n      "dismissAriaLabel": "Dismiss"\n    },\n    "specPattern": {\n      "matches": "No matches | {n} match | {n} matches"\n    },\n    "specPatternModal": {\n      "title": "Spec pattern settings"\n    }\n  },\n  "clipboard": {\n    "copy": "Copy",\n    "copied": "Copied!"\n  },\n  "actions": {\n    "close": "Close"\n  },\n  "file": {\n    "edit": "Edit",\n    "git": {\n      "modified": "Modified",\n      "created": "Created"\n    }\n  },\n  "status": {\n    "enabled": "Enabled",\n    "disabled": "Disabled"\n  },\n  "createSpec": {\n    "newSpec": "New spec",\n    "createSpec": "Create spec",\n    "done": "Done!",\n    "noComponentsFound": "No components found",\n    "unableToParseFile": "Unable to parse file",\n    "updateSpecPattern": "Update spec pattern",\n    "newSpecModalTitle": "Create new spec",\n    "successPage": {\n      "header": "Great! The spec was successfully added",\n      "runSpecButton": "Okay, run the spec",\n      "createAnotherSpecButton": "Create another spec"\n    },\n    "page": {\n      "defaultPatternNoSpecs": {\n        "title": "Create your first spec",\n        "component": {\n          "description": "Since this project looks new, we recommend that you get started by creating a spec for one of your components."\n        },\n        "e2e": {\n          "description": "Since this project looks new, we recommend that you use the specs and tests that we\'ve written for you to get started."\n        }\n      },\n      "customPatternNoSpecs": {\n        "title": "No specs found",\n        "description": "We couldn\'t find any files matching the following spec patterns. You may need to move your specs or update your {0} to find them."\n      }\n    },\n    "noSpecsMessage": "If you feel that you\'re seeing this screen in error, and there should be specs listed here, you likely need to update the spec pattern.",\n    "viewSpecPatternButton": "View spec pattern",\n    "e2e": {\n      "importFromScaffold": {\n        "header": "Scaffold example specs",\n        "description": "We\'ll generate several example specs to help guide you on how to write tests in Cypress.",\n        "specsAddedHeader": "Great! We added the following files to your project:",\n        "specsAddedButton": "Okay, I got it!",\n        "specsAddingHeader": "Adding specs..."\n      },\n      "importTemplateSpec": {\n        "header": "Create new spec",\n        "description": "We\'ll generate a template spec file which can be used to start testing your application.",\n        "chooseFilenameHeader": "Enter the path for your new spec",\n        "inputPlaceholder": "Enter a relative path...",\n        "invalidSpecWarning": "This path is invalid because it doesn\'t match the following ",\n        "specExtensionWarning": "We recommend naming your spec to match the following pattern: "\n      }\n    },\n    "component": {\n      "importFromComponent": {\n        "header": "Create from component",\n        "description": "Generate a basic component test for any of the components in this project.",\n        "chooseAComponentHeader": "Choose a component"\n      },\n      "importTemplateSpec": {\n        "header": "Create new spec",\n        "description": "We\'ll generate a template spec file which can be used to start testing your components.",\n        "invalidComponentWarning": "We couldn\'t generate a valid path matching your custom "\n      }\n    }\n  },\n  "specPage": {\n    "pageTitle": "Specs",\n    "newSpecButton": "New spec",\n    "searchPlaceholder": "Search specs",\n    "clearSearch": "Clear search field",\n    "componentSpecsHeader": "Component specs",\n    "e2eSpecsHeader": "E2E specs",\n    "componentSpecsHeaderShort": "Component",\n    "e2eSpecsHeaderShort": "E2E",\n    "lastUpdated": {\n      "header": "Last updated",\n      "tooltip": {\n        "gitStatus": "Git status",\n        "gitInfo": "git info",\n        "gitInfoAvailable": "{0} of the spec files within this project",\n        "gitInfoUnavailable": "Cypress is unable to detect {0} for this project and has defaulted to showing file system data instead"\n      }\n    },\n    "latestRuns": {\n      "header": "Latest runs",\n      "headerShort": "Runs",\n      "tooltip": {\n        "connected": "The {0} in Cypress Cloud",\n        "notConnected": "Connect to Cypress Cloud to see the {0}",\n        "noAccess": "Request access to this project in Cypress Cloud to view the {0}",\n        "linkText": "status of your latest runs"\n      }\n    },\n    "averageDuration": {\n      "header": "Average duration",\n      "headerShort": "Duration",\n      "tooltip": {\n        "connected": "The {0} of your latest runs in Cypress Cloud",\n        "notConnected": "Connect to Cypress Cloud to see the {0} of your latest runs",\n        "noAccess": "Request access to this project in Cypress Cloud to view the {0} of your latest runs",\n        "linkText": "average spec durations"\n      }\n    },\n    "flaky": {\n      "badgeLabel": "Flaky",\n      "severityLow": "Low",\n      "severityMedium": "Medium",\n      "severityHigh": "High",\n      "flakyRate": "{0}% flaky rate",\n      "flakyRuns": "{flakyRuns} flaky runs / {totalRuns} total | {flakyRuns} flaky run / {totalRuns} total | {flakyRuns} flaky runs / {totalRuns} total",\n      "lastFlaky": "Last run flaky | Last flaky {runsSinceLastFlake} run ago | Last flaky {runsSinceLastFlake} runs ago"\n    },\n    "hoverButton": {\n      "connect": "Connect",\n      "connectProject": "Connect project",\n      "connectProjectShort": "Connect",\n      "requestAccess": "Request access",\n      "requestAccessShort": "Request",\n      "requestSent": "Request sent",\n      "requestSentShort": "Sent"\n    },\n    "connectProjectButton": "Connect your project",\n    "cloudLoginButton": "Log in to Cypress Cloud",\n    "reconnectProjectButton": "Reconnect your project",\n    "unauthorizedBannerTitle": "Request access to project",\n    "requestAccessButton": "Request access",\n    "requestSentButton": "Request sent",\n    "rows": {\n      "gitTooltipSubtext": "{shortHash} by {author}"\n    },\n    "noResultsMessage": "No specs matched your search:",\n    "noSpecError": {\n      "title": "Spec not found",\n      "intro": "There is no spec matching the following location:",\n      "explainer": "It is possible that the file has been moved or deleted. Please choose from the list of specs below."\n    },\n    "offlineWarning": {\n      "title": "No internet connection",\n      "explainer": "Please check your internet connection to resolve this issue. When your internet connection is fixed, we will automatically attempt to fetch the run metrics from Cypress Cloud."\n    },\n    "fetchFailedWarning": {\n      "title": "Lost connection",\n      "explainer1": "The request timed out or failed when trying to retrieve the recorded run metrics from Cypress Cloud. The information that you\'re seeing in the table below may be incomplete as a result.",\n      "explainer2": "Please refresh the page to try again and visit our {0} if this behavior continues.",\n      "refreshButton": "Try again"\n    },\n    "banners": {\n      "login": {\n        "title": "Optimize and record your CI test runs with Cypress Cloud",\n        "content": "When you configure Cypress to record tests to Cypress Cloud, you\'ll see data from your latest recorded runs in the Cypress app. This increased visibility into your test history allows you to debug your tests faster and more effectively, all within your local workflow.",\n        "buttonLabel": "Get started with Cypress Cloud"\n      },\n      "createOrganization": {\n        "title": "Finish setting up Cypress Cloud",\n        "content": "Join or create an organization in Cypress Cloud to access your projects and recorded test runs.",\n        "buttonLabel": "Set up your organization"\n      },\n      "connectProject": {\n        "title": "Connect your project to Cypress Cloud",\n        "content": "Bring your recorded test results into your local development workflow to monitor, run, and fix tests all in the Cypress app.",\n        "buttonLabel": "Connect your project"\n      },\n      "record": {\n        "title": "Record your first run",\n        "content": "Record a run to see your test results in Cypress Cloud. You can then optimize your test suite, debug failing and flaky tests, and integrate with your favorite tools."\n      },\n      "componentTesting": {\n        "title": "{0} component testing is available for this project",\n        "content": "You can now use Cypress to develop and test individual components without running your whole application. Generate the config in just a few clicks.",\n        "primaryAction": "Quick setup",\n        "secondaryAction": "Read our guide",\n        "dismissAction": "Give feedback"\n      },\n      "enableNotifications": {\n        "title": "Start debugging as soon as your first tests fail",\n        "subtitle": "Cypress can notify you about Cloud test runs for your branch.",\n        "enableDesktopNotifications": "Enable desktop notifications",\n        "remindMeLater": "Remind me later",\n        "notificationsEnabledTitle": "Notifications Enabled",\n        "notificationsEnabledBody": "Nice, notifications are enabled!",\n        "dismissBanner": "Dismiss banner"\n      }\n    },\n    "runSelectedSpecs": "Run {n} spec | Run {n} specs",\n    "testingTypePromo": {\n      "e2e": {\n        "header": {\n          "title": "End-to-end testing is not set up for this project",\n          "description": "Try it out now, or {0} to learn more",\n          "link": "read our guides",\n          "control": "Quick setup"\n        },\n        "body": {\n          "title": "What is end-to-end testing?",\n          "description": "End-to-end tests are perfect for testing your entire application from a user\'s perspective. Visit any URL and test full user journeys across pages and domains.",\n          "listItem1": "Test full user journeys across multiple pages using {0} to load up a web page for your test.",\n          "listItem1Command": "cy.visit()",\n          "listItem2": "Work with live API calls or mock them with {0} to test all possible states of your app.",\n          "listItem2Command": "cy.intercept()",\n          "listItem3": "Use {0} to maintain consistent browser sessions in your tests, and work with features like cookies and local storage, all while keeping your tests properly isolated.",\n          "listItem3Command": "cy.session()"\n        }\n      },\n      "ct": {\n        "header": {\n          "title": "Component testing is not set up for this project",\n          "description": "Try it out now, or {0} to learn more",\n          "link": "read our guides",\n          "control": "Quick setup"\n        },\n        "body": {\n          "title": "What is Component Testing?",\n          "description": "Cypress Component Testing is a technique for developers to write, test, and debug components in isolation without ever leaving the browser - and without having to run your entire application.",\n          "listItem1": "Instead of visiting a page, use {0} to render your components directly in the browser.",\n          "listItem1Command": "cy.mount()",\n          "listItem2": "Test from a user\'s perspective with all the same Cypress commands that are available in end-to-end testing.",\n          "listItem3": "To get started, you can generate the config for {0}, {1}, and {2} (and {3}!).",\n          "listItem3Link": "more",\n          "link": "Give feedback"\n        }\n      }\n    }\n  },\n  "noResults": {\n    "defaultMessage": "No results matched your search:",\n    "clearSearch": "Clear search"\n  },\n  "sidebar": {\n    "keyboardShortcuts": {\n      "title": "Keyboard shortcuts",\n      "rerun": "Re-run tests",\n      "stop": "Stop tests",\n      "toggle": "Toggle specs list",\n      "studioSave": "Save Studio changes"\n    },\n    "toggleLabel": {\n      "expanded": "Collapse sidebar",\n      "collapsed": "Expand sidebar"\n    },\n    "nav": {\n      "ariaLabel": "Pages"\n    },\n    "debug": {\n      "debugFeature": "New Debug feature",\n      "passed": "Relevant run passed",\n      "failed": "Relevant run had {n} test failure | Relevant run had {n} test failures",\n      "errored": "Relevant run had an error",\n      "erroredWithFailures": "Relevant run had an error with {n} test failure | Relevant run had an error with {n} test failures",\n      "passing": "Relevant run is passing",\n      "failing": "Relevant run is failing with {n} test failure | Relevant run is failing with {n} test failures"\n    },\n    "runs": {\n      "running": "Latest run is in progress",\n      "failing": "Latest run is failing with {n} test failure | Latest run is failing with {n} test failures",\n      "passed": "Latest run passed",\n      "failed": "Latest run had {n} test failure | Latest run had {n} test failures",\n      "cancelled": "Latest run has been cancelled",\n      "errored": "Latest run had an error",\n      "erroredWithFailures": "Latest run had an error with {n} test failure | Latest run had an error with {n} test failures"\n    }\n  },\n  "topNav": {\n    "released": "Released",\n    "version": "Version",\n    "installed": "Installed",\n    "latest": "Latest",\n    "seeAllReleases": "See all releases",\n    "runningOldVersion": "You\'re currently running an old version of Cypress. Update to the latest version for the best experience.",\n    "docsMenu": {\n      "docsHeading": "Docs",\n      "gettingStartedTitle": "Getting started",\n      "firstTest": "Write your first test",\n      "testingApp": "Testing your app",\n      "organizingTests": "Organizing tests",\n      "referencesTitle": "References",\n      "bestPractices": "Best practices",\n      "configuration": "Configuration",\n      "api": "API",\n      "ciTitle": "Run in CI/CD",\n      "ciSetup": "Set up CI",\n      "fasterTests": "Run tests faster",\n      "prompts": {\n        "ci1": {\n          "title": "Configure CI",\n          "description": "Optimize your tests in CI by following these guides.",\n          "seeOtherGuides": "See other guides",\n          "intro": "Introduction to CI",\n          "learnTheBasics": "Learn the basics of running Cypress in CI."\n        },\n        "orchestration1": {\n          "title": "Run tests faster in CI",\n          "parallelTime": "5 mins",\n          "noParallelTime": "12 mins",\n          "withParallelization": "with Parallelization",\n          "withoutParallelization": "without Parallelization",\n          "intro": "With Smart Orchestration, you\'ll be able to:",\n          "bullet1": "Run spec files in parallel",\n          "bullet2": "Prioritize failed specs to run first",\n          "bullet3": "Cancel CI runs on test failure",\n          "learnMore": "Learn more"\n        }\n      }\n    },\n    "login": {\n      "bodyInitial": "Logging in gives you access to {0}. You can set up projects to be recorded and see test data from your project.",\n      "bodySuccess": "You are now logged in as {0}.",\n      "bodyError": "An error occurred while attempting to log in:",\n      "bodyBrowserError": "The browser could not be opened.",\n      "bodyBrowserErrorDetails": "Cypress was unable to open your installed browser. To continue logging in, please paste this URL into your web browser:",\n      "cloud": "Cypress Cloud",\n      "titleInitial": "Log in to Cypress",\n      "titleSuccess": "Login successful",\n      "titleFailed": "Login failed",\n      "titleBrowserError": "Browser error",\n      "actionLogin": "Log in",\n      "profileMenuLabel": "Profile and logout",\n      "actionLogout": "Log out",\n      "actionContinue": "Continue",\n      "actionOpening": "Opening browser",\n      "actionWaiting": "Waiting for you to log in",\n      "actionTryAgain": "Try again",\n      "actionCancel": "Cancel"\n    },\n    "updateCypress": {\n      "title": "Upgrade to Cypress",\n      "currentlyRunning": "You are currently running Version {0} of Cypress. ",\n      "pasteToUpgradeProject": "To upgrade to the latest version for your project, first {0}, then paste the command below into your terminal:",\n      "pasteToUpgradeGlobal": "To upgrade to the latest version, first {0}, then paste the command below into your terminal:",\n      "rememberToCloseInsert": "close this app"\n    },\n    "upgradeText": "Upgrade",\n    "global": {\n      "projects": "Projects"\n    }\n  },\n  "launchpadErrors": {\n    "generic": {\n      "configErrorTitle": "Cypress configuration error",\n      "retryButton": "Try again",\n      "stackTraceLabel": "Stack trace",\n      "docsButton": {\n        "configGuide": {\n          "text": "Configuration guide",\n          "link": "https://on.cypress.io/configuration"\n        },\n        "cloudGuide": {\n          "text": "Cypress Cloud guide",\n          "link": "https://on.cypress.io/features-dashboard"\n        },\n        "docsHomepage": {\n          "text": "Cypress docs",\n          "link": "https://docs.cypress.io"\n        }\n      }\n    },\n    "noInternet": {\n      "header": "You have no internet connection",\n      "message": "Check your internet connection to pull the latest data from Cypress Cloud.",\n      "connectProject": "Please check your internet connection to resolve this issue. When your internet connection is fixed, we will automatically attempt to fetch your latest runs for this project."\n    }\n  },\n  "setupPage": {\n    "projectSetup": {\n      "frameworkLabel": "Front-end framework",\n      "frameworkPlaceholder": "Pick a framework",\n      "bundlerLabel": "Bundler",\n      "bundlerPlaceholder": "Pick a bundler",\n      "languageLabel": "Language",\n      "configFileLanguageLabel": "Cypress config file",\n      "detected": "(detected)",\n      "browseIntegrations": "Browse our list of third-party framework integrations",\n      "communityFrameworkDefinitionProblem": "Community framework definition problem",\n      "communityDependenciesCouldNotBeParsed": "This project has a community framework definition installed that could not be loaded. It is located at the following path: | This project has some community framework definitions installed that could not be loaded. They are located at the following paths:",\n      "seeFrameworkDefinitionDocumentation": "See the {0} for more information about creating, installing, and troubleshooting third party definitions.",\n      "frameworkDefinitionDocumentation": "framework definition documentation"\n    },\n    "step": {\n      "continue": "Continue",\n      "next": "Next step",\n      "back": "Back",\n      "skip": "Skip"\n    },\n    "configFile": {\n      "createManually": "Create file manually",\n      "skippedLabel": "Skipped",\n      "changesRequiredLabel": "Changes required",\n      "changesRequiredBadge": "Changes required",\n      "changesRequiredDescription": "Please merge the code below with your existing {0}"\n    },\n    "testingCard": {\n      "configured": "Configured",\n      "notConfigured": "Not Configured",\n      "running": "Running",\n      "chooseABrowser": "Choose a browser",\n      "reconfigure": "Reconfigure",\n      "disabled": "Disabled"\n    }\n  },\n  "globalPage": {\n    "empty": {\n      "title": "Add project",\n      "helper": "Get started by adding your first project below.",\n      "dropText": "Drag your project directory here or {0}",\n      "browseManually": "browse manually."\n    },\n    "settings": "Settings",\n    "saveChanges": "Save changes",\n    "cancel": "Cancel",\n    "externalEditorPreferences": "External editor preferences",\n    "externalEditorPreferencesDescription1": "Select your preference for opening files on your system.",\n    "externalEditorPreferencesDescription2": "You can change this later in the {0} page.",\n    "recentProjectsHeader": "Recent projects",\n    "searchPlaceholder": "Search projects",\n    "addProjectButton": "Add project",\n    "projectActions": "Project actions",\n    "removeProject": "Remove project",\n    "openInIDE": "Open in IDE",\n    "openInFinder": "Open in Finder",\n    "noResultsMessage": "No projects matched your search:"\n  },\n  "welcomePage": {\n    "review": "Review the differences between each testing type",\n    "codeExample": "Code example",\n    "compareTypes": {\n      "content": {\n        "e2eTitle": "End-to-end tests",\n        "e2eBullet1": "Visit URLs via",\n        "e2eBullet2": "Test flows and functionality across multiple pages",\n        "e2eBullet3": "Ideal for testing integrated flows in CD workflows",\n        "componentTitle": "Component tests",\n        "ctBullet1": "Import components via",\n        "ctBullet2": "Test individual components of a design system in isolation",\n        "ctBullet3": "Ideal for testing isolated flows and components in CI"\n      },\n      "modalTitle": "Key differences"\n    },\n    "title": "Welcome to Cypress!"\n  },\n  "settingsPage": {\n    "config": {\n      "title": "Resolved configuration",\n      "description": "Since the options in the {0} can be set dynamically by your development environment, please refer to the legend below to understand how the configuration options are resolved.",\n      "legend": {\n        "env": {\n          "label": "env",\n          "description": "Set from environment variables"\n        },\n        "default": {\n          "label": "default",\n          "description": "Default values"\n        },\n        "dynamic": {\n          "label": "dynamic",\n          "description": "Set by the {0} function"\n        },\n        "cli": {\n          "label": "cli",\n          "description": "Set from CLI arguments"\n        },\n        "config": {\n          "label": "config",\n          "description": "Set from {0}"\n        }\n      }\n    },\n    "proxy": {\n      "title": "Proxy settings",\n      "description": "Cypress auto-detected the following proxy settings from your operating system.",\n      "bypassList": "Proxy bypass list",\n      "proxyServer": "Proxy server"\n    },\n    "editor": {\n      "title": "External editor",\n      "description": "External editor to use for editing code opened when using Cypress.",\n      "noEditorSelectedPlaceholder": "Choose your editor...",\n      "customPathPlaceholder": "/path/to/editor"\n    },\n    "notifications": {\n      "title": "Desktop notifications",\n      "description": "Cypress can notify you about Cloud test runs for your branch.",\n      "notifyMeWhenRunStarts": "Notify me when a run starts",\n      "notifyMeWhenRunIsFailing": "Notify me when a run begins to fail",\n      "notifyMeWhenRunCompletes": "Notify me when a run completes",\n      "passed": "Passed",\n      "failed": "Failed",\n      "canceled": "Canceled",\n      "errored": "Errored",\n      "sendTestNotification": "Send a test notification",\n      "notReceivingNotifications": "Not receiving notifications? You might need to allow notifications from Cypress in your computer\'s system preferences. {0}",\n      "testNotificationTitle": "Hello From Cypress",\n      "testNotificationBody": "This is a test notification",\n      "troubleshoot": "Troubleshoot",\n      "enableNotificationsLabel": "To modify notification settings, enable desktop notifications"\n    },\n    "projectId": {\n      "title": "Project ID",\n      "description": "The Project ID configured for this project inside of Cypress Cloud. {0}"\n    },\n    "specPattern": {\n      "title": "Spec patterns",\n      "description": "The globs pointing Cypress to your spec files configured for this project. {0}"\n    },\n    "recordKey": {\n      "title": "Record key",\n      "description": "The Record Key configured for this project inside of Cypress Cloud. {0}",\n      "manageKeys": "Manage keys",\n      "errorEmpty": "You do not have permission to record on this project",\n      "errorEmptyButton": "Request record rights",\n      "errorAccess": "You do not have permission to view this project",\n      "errorAccessPending": "Your request for access to this project is pending approval",\n      "errorAccessButton": "Request access",\n      "errorNotFound": "Project not found",\n      "errorNotFoundButton": "Connect it again",\n      "errorNotLoggedIn": "You are not connected to Cypress Cloud",\n      "errorNotLoggedInButton": "Log in to Cypress Cloud"\n    },\n    "project": {\n      "title": "Project settings",\n      "description": "Review the configuration settings currently in use for this project."\n    },\n    "cloud": {\n      "title": "Cypress Cloud settings",\n      "description": "Review the configuration settings for recording to Cypress Cloud."\n    },\n    "experiments": {\n      "title": "Experiments",\n      "description": "If you\'d like to try out new features that we\'re working on, you can enable beta features for your project by turning on the experimental features you\'d like to try. {0}",\n      "experimentalCspAllowList": {\n        "name": "CSP Allow List",\n        "description": "Enables Cypress to selectively permit Content-Security-Policy and Content-Security-Policy-Report-Only header directives, including those that might otherwise block Cypress from running."\n      },\n      "experimentalInteractiveRunEvents": {\n        "name": "Interactive run events",\n        "description": "Allows listening to the [`before:run`](https://on.cypress.io/before-run), [`after:run`](https://on.cypress.io/after-run), [`before:spec`](https://on.cypress.io/before-spec), and [`after:spec`](https://on.cypress.io/after-spec) events in plugins during interactive mode."\n      },\n      "experimentalModifyObstructiveThirdPartyCode": {\n        "name": "Modify obstructive third party code",\n        "description": "Applies `modifyObstructiveCode` to third party `.html` and `.js`, removes subresource integrity, and modifies the user agent in Electron."\n      },\n      "retries": {\n        "experimentalStrategy": {\n          "name": "Retries Strategy",\n          "description": "Applies a strategy for test retries according to your \\"flake tolerance\\"; options are `detect-flake-but-always-fail` or `detect-flake-and-pass-on-threshold`."\n        },\n        "experimentalOptions": {\n          "name": "Retries Strategy Options",\n          "description": "Sets retries strategy-specific options like `maxRetries`, `passesRequired`, and `stopIfAnyPassed`."\n        }\n      },\n      "experimentalSingleTabRunMode": {\n        "name": "Single tab run mode",\n        "description": "Runs all component specs in a single tab, trading spec isolation for faster run mode execution."\n      },\n      "experimentalSourceRewriting": {\n        "name": "Source rewriting",\n        "description": "Enables AST-based JS/HTML rewriting. This may fix issues caused by the existing regex-based JS/HTML replacement algorithm. See [#5273](https://github.com/cypress-io/cypress/issues/5273) for details."\n      },\n      "experimentalPromptCommand": {\n        "name": "Prompt command",\n        "description": "Enables support for `cy.prompt`, an AI-powered command that turns natural language steps into executable Cypress test code."\n      },\n      "experimentalWebKitSupport": {\n        "name": "WebKit Support",\n        "description": "Adds support for testing in the WebKit browser engine used by Safari. See https://on.cypress.io/webkit-experiment for more information."\n      },\n      "experimentalRunAllSpecs": {\n        "name": "Run All Specs",\n        "description": "Enables the \\"Run All Specs\\" UI feature, allowing the execution of multiple specs sequentially."\n      },\n      "experimentalOriginDependencies": {\n        "name": "Origin Dependencies",\n        "description": "Enables support for `Cypress.require()` for including dependencies within the `cy.origin()` callback."\n      },\n      "experimentalMemoryManagement": {\n        "name": "Memory Management",\n        "description": "Enables support for improved memory management within Chromium-based browsers."\n      }\n    },\n    "device": {\n      "title": "Device settings",\n      "description": "Review the configuration settings currently in use for this device."\n    },\n    "testingPreferences": {\n      "title": "Testing preferences",\n      "description": "Configure your testing environment with these flags",\n      "autoScrollingEnabled": {\n        "title": "Auto scrolling enabled",\n        "description": "Scroll behavior when running tests."\n      }\n    },\n    "footer": {\n      "text": "You can reconfigure the {testingType} testing settings for this project if you\'re experiencing issues with your Cypress configuration.",\n      "button": "Configuration guide",\n      "buttonLink": "https://on.cypress.io/configuration"\n    }\n  },\n  "runs": {\n    "slideshow": {\n      "tour": {\n        "title": "Record your first test run to Cypress Cloud",\n        "description": "You\'re almost there! Start recording your test runs to Cypress Cloud by running the command above in your local or CI terminal."\n      },\n      "step1": {\n        "title": "Monitor test failures in real time",\n        "description": "Uncover high-risk failed and flaky tests in CI before they become a problem in production."\n      },\n      "step2": {\n        "title": "Check if failed CI test runs also fail locally",\n        "description": "Use the Test Runner to run only the tests that failed in your last recorded run."\n      },\n      "step3": {\n        "title": "Build, test, and ship with confidence 🚀",\n        "description": "With Cypress Cloud, you can proactively identify, investigate, and resolve failures, and be a quality champion."\n      },\n      "controls": {\n        "step": "{0} of {1}",\n        "next": "Next",\n        "reset": "Reset",\n        "view": "View Tour"\n      }\n    },\n    "connect": {\n      "title": "Connect to view your recorded runs",\n      "description": "Cypress Cloud is built for teams who need to do testing at scale.",\n      "buttonUser": "Connect to Cypress Cloud",\n      "buttonProject": "Connect a Cypress Cloud project",\n      "modal": {\n        "title": "Connect project",\n        "cancel": "Cancel",\n        "createOrg": {\n          "description": "You need to create an organization in Cypress Cloud to continue.",\n          "button": "Create organization",\n          "waitingButton": "Waiting for you to create an organization...",\n          "refreshButton": "Refresh organizations list"\n        },\n        "selectProject": {\n          "organization": "Organization",\n          "noOrganizationSelectedError": "required when creating a new project",\n          "manageOrgs": "Manage organizations",\n          "project": "Project",\n          "projectName": "Project name",\n          "projectNameDisclaimer": "(You can change this later)",\n          "createNewProject": "Create new",\n          "chooseExistingProject": "Choose an existing project",\n          "newProjectAccess": "Project access",\n          "privateLabel": "Private",\n          "privateDescription": "Only invited users can view recorded test results.",\n          "publicLabel": "Public",\n          "publicDescription": "Anyone can view recorded test results.",\n          "connectProject": "Connect project",\n          "createProject": "Create project",\n          "placeholderOrganizations": "Pick an organization",\n          "placeholderProjects": "Pick a project",\n          "placeholderProjectsPending": "Pick an organization first"\n        },\n        "connectManually": {\n          "title": "Connect project manually",\n          "warning": "We couldn\'t add the projectId to your cypress.config.js file automatically.",\n          "mainMessage": "Manually add the {projectId} to the root of the config object in your {configFile} file.",\n          "waitingButton": "Waiting for you to add the projectId..."\n        }\n      },\n      "errors": {\n        "baseError": {\n          "title": "Cypress Cloud error"\n        },\n        "internalServerError": {\n          "title": "Cannot connect to Cypress Cloud",\n          "description": "The request times out when trying to retrieve the info from Cypress Cloud. Please refresh the page to try again and visit our {0} if this behavior continues.",\n          "link": "Support Page"\n        }\n      }\n    },\n    "connectSuccessAlert": {\n      "title": "This project is now connected to Cypress Cloud!",\n      "item1": "We automatically added the {projectId} to your {configFile} file.",\n      "item2": "Please ensure that your {0} file is checked into source control."\n    },\n    "empty": {\n      "title": "Copy the command below to record your first run",\n      "description": "Review the {0} to Cypress Cloud",\n      "link": "guide on recording runs",\n      "gitRepositoryNotDetected": "Git repository not detected",\n      "ensureGitSetupCorrectly": "Cypress uses Git to associate runs with your local state. As a fallback, all runs for your project are shown. Please ensure that version control is set up correctly.",\n      "noRunsFoundForBranch": "No runs found for your branch",\n      "noRunsForBranchMessage": "Cypress uses Git to show runs for your branch. As a fallback, all runs for your project are shown. Ensure that version control is properly configured and that you are sending Git information to Cypress Cloud."\n    },\n    "results": {\n      "skipped": "skipped",\n      "pending": "pending",\n      "passed": "passed",\n      "failed": "failed"\n    },\n    "errors": {\n      "notFound": {\n        "title": "Couldn\'t find your project",\n        "description": "We were unable to find an existing project matching the {0} set in your Cypress config file. You can reconnect with an existing project or create a new project.",\n        "button": "Reconnect your project"\n      },\n      "unauthorized": {\n        "title": "Request access to view the recorded runs",\n        "description": "This is a private project that you do not currently have access to. Please request access from the project owner in order to view the list of runs.",\n        "button": "Request access"\n      },\n      "unauthorizedRequested": {\n        "title": "Your access request for this project has been sent.",\n        "description": "The owner of this project has been notified of your request. We\'ll notify you via email when your access request has been granted.",\n        "button": "Request Sent"\n      },\n      "connectionFailed": {\n        "title": "Cannot connect to Cypress Cloud",\n        "description": "The request times out when trying to retrieve the recorded runs from Cypress Cloud. Please refresh the page to try again and visit out {0} if this behavior continues.",\n        "link": "Support page",\n        "button": "Try again"\n      }\n    },\n    "card": {\n      "branchName": "Branch Name:",\n      "commitAuthor": "Commit Author:",\n      "totalDuration": "Run Total Duration:",\n      "debugLabel": "Debug",\n      "debugDescription": "View debug for run {runNumber}",\n      "noDebugAvailable": "You can only debug runs for your most recent commits."\n    },\n    "layout": {\n      "viewCloudRuns": "View runs in Cypress Cloud",\n      "checkedOut": "Checked out"\n    }\n  },\n  "debugPage": {\n    "openFile": {\n      "openInIDE": "Open in IDE",\n      "notFoundLocally": "Opening in IDE is disabled because the spec is not found in this project"\n    },\n    "limit": {\n      "title": "Cypress renders up to 100 failed test results",\n      "message": "This run has {n} failed tests | This run has {n} failed test | This run has {n} failed tests",\n      "link": "See all test results in Cypress Cloud"\n    },\n    "runFailures": {\n      "btn": "Run Failures",\n      "notFoundLocally": "Spec was not found locally",\n      "switchTestingTypeInfo": "There is {n} {testingType} test failing in this spec. To run it locally switch to {testingType} testing. | There are {n} {testingType} tests failing in this spec. To run them locally switch to {testingType} testing.",\n      "switchTestingTypeAction": "Switch to {testingType} testing"\n    },\n    "switchToRun": "Switch to run number {runNumber}",\n    "overLimit": {\n      "usageExceededTitle": "You\'ve reached your monthly test result limit",\n      "usageExceededUserMessage": "Your Free Cypress Cloud plan includes {numberTests} monthly recorded test results. Contact your Cypress Cloud admin to upgrade your plan and view more test results.",\n      "usageExceededAdminMessage": "Your Free Cypress Cloud plan includes {numberTests} monthly recorded test results. Upgrade your plan now to view more test results.",\n      "retentionExceededTitle": "Beyond data retention",\n      "retentionExceededMessage": "Your data retention limit is {limitDays} days and these tests ran {runAgeDays} days ago. Upgrade your plan to increase your data retention limit.",\n      "upgradePlan": "Upgrade plan",\n      "contactAdmin": "Contact admin"\n    },\n    "wellDone": "Well Done!",\n    "allYourTestsPassed": "All your tests passed.",\n    "incomplete": "Incomplete",\n    "specsSkipped": "{totalSkippedSpecs} of {n} spec skipped | {totalSkippedSpecs} of {n} specs skipped",\n    "runHasNoTests": "Run has no tests",\n    "theRunStartedButNeverCompleted": "The run started but never completed. This can happen when the run is cancelled from CI or when Cypress crashes while running tests.",\n    "checkYourCiLogs": "Check your {0} logs for more information.",\n    "archiveThisRun": "Archive this run to remove it from the runs list and analytics.",\n    "manuallyCancelled": "Manually cancelled",\n    "mostRecentRun": "You are on the most recent run",\n    "switchToLatestRun": "Switch to latest run",\n    "switchRun": "Switch run",\n    "foundMoreThan100Runs": "We found more than 100 runs.",\n    "goToCypressCloud": "Go to Cypress Cloud to see all runs",\n    "emptyStates": {\n      "debugDirectlyInCypress": "Debug failed CI runs directly in Cypress",\n      "reviewRerunAndDebug": "Review, rerun, and debug failed CI test runs that are recorded to Cypress Cloud – all from within your local Cypress app.",\n      "connectToCypressCloud": "Connect to Cypress Cloud",\n      "connect": {\n        "title": "Connect to debug your tests",\n        "description": "Cypress Cloud is built for teams who need to do testing at scale."\n      },\n      "noProjectTestMessage": "Connect a Cypress Cloud project to locally debug failed CI test runs",\n      "noRuns": {\n        "title": "Copy the command below to record your first run",\n        "description": "Review the {0} to Cypress Cloud",\n        "link": "guide on recording runs"\n      },\n      "gitRepositoryNotDetected": "Git repository not detected",\n      "ensureGitSetupCorrectly": "Cypress uses Git to associate runs with your local state. Please ensure that Git is properly configured for your project.",\n      "learnAboutRecordingSrText": "about recording a run to Cypress Cloud",\n      "learnAboutDebuggingSrText": "about debugging CI failures in Cypress",\n      "learnAboutProjectSetupSrText": "about project setup in Cypress",\n      "noRunsFoundForBranch": "No runs found for your branch",\n      "noRunsForBranchMessage": "Cypress uses Git to show runs for your branch. Ensure that version control is properly configured and that you are sending Git information to Cypress Cloud.",\n      "slideshow": {\n        "tour": {\n          "title": "Record your first test run to Cypress Cloud",\n          "description": "You\'re almost there! Start recording your test runs to Cypress Cloud by running the command below in your local or CI terminal."\n        },\n        "step1": {\n          "title": "Review how many tests failed during a CI test run",\n          "description": "The Debug page shows the latest completed test run for your current checked out commit."\n        },\n        "step2": {\n          "title": "Check if failed CI test runs also fail locally",\n          "description": "Use the Test Runner to run only the tests that failed in your last recorded run."\n        },\n        "step3": {\n          "title": "Locally debug failed test runs with visual artifacts",\n          "description": "Easily review screenshots, videos and logs from your tests."\n        },\n        "controls": {\n          "step": "{0} of {1}",\n          "next": "Next",\n          "reset": "Reset",\n          "view": "View Tour"\n        }\n      }\n    },\n    "pending": {\n      "title": "Testing in progress...",\n      "titleCompletionScheduled": "Scheduled to complete...",\n      "scheduledTime": "Scheduled to complete in {time}",\n      "scheduledCompletionExpired": "Scheduled to complete soon",\n      "failuresHere": "Failures will be displayed here"\n    },\n    "specCounts": {\n      "whenRunning": "{completed} of {total} specs completed | {completed} of {total} spec completed | {completed} of {total} specs completed",\n      "whenCompleted": "{n} specs | {n} spec | {n} specs"\n    },\n    "header": {\n      "runUrl": "View in Cypress Cloud",\n      "commitsAhead": "You are {n} commit ahead | You are {n} commits ahead"\n    },\n    "stats": {\n      "groups": "{n} group | {n} groups",\n      "operatingSystems": "{n} operating system | {n} operating systems",\n      "browsers": "{n} browser | {n} browsers"\n    },\n    "artifacts": {\n      "replay": "View Test Replay",\n      "stdout": "View Log",\n      "screenshots": "View Screenshots",\n      "video": "View Video"\n    }\n  },\n  "testingType": {\n    "modalTitle": "Choose a testing type",\n    "e2e": {\n      "name": "E2E Testing",\n      "description": "Build and test the entire experience of your application from end-to-end to ensure each flow matches your expectations."\n    },\n    "component": {\n      "name": "Component Testing",\n      "description": "Build and test your components from your design system in isolation in order to ensure each state matches your expectations."\n    },\n    "componentDisabled": {\n      "description": "To test your components you must {0} as an NPM dependency for this project.",\n      "link": "install Cypress"\n    }\n  },\n  "majorVersionWelcome": {\n    "title": "What\'s New in Cypress",\n    "actionContinue": "Continue",\n    "linkReleaseNotes": "View full changelog"\n  },\n  "setupWizard": {\n    "selectFramework": {\n      "title": "Project setup",\n      "description": "Confirm the front-end framework and bundler used in your project."\n    },\n    "installDependencies": {\n      "title": "Install dev dependencies",\n      "description": "Based on your previous selection, the following dependencies are required.",\n      "pasteCommand": "Paste this command into your terminal to install the following packages:",\n      "waitForInstall": "Waiting for you to install the dependencies...",\n      "installed": "installed",\n      "pendingInstall": "pending installation",\n      "installationAlertSuccess": "You\'ve successfully installed all required dependencies."\n    },\n    "configFiles": {\n      "title": "Configuration files",\n      "description": "We added the following files to your project:"\n    },\n    "chooseBrowser": {\n      "title": "Choose a browser",\n      "description": "Choose your preferred browser for {testingType} testing."\n    }\n  },\n  "e2eProjectSetup": {\n    "title": "Project setup",\n    "description": "Confirm your project\'s preferred language."\n  },\n  "openBrowser": {\n    "startComponent": "Start Component Testing in {browser}",\n    "startE2E": "Start E2E Testing in {browser}",\n    "openingComponent": "Opening Component Testing in {browser}",\n    "openingE2E": "Opening E2E Testing in {browser}",\n    "running": "Running {browser}",\n    "focus": "Focus",\n    "close": "Close",\n    "switchTestingType": "Switch testing type"\n  },\n  "runner": {\n    "header": {\n      "reviewDocs": "Review the docs",\n      "troubleRendering": "if you\'re having trouble rendering your components properly."\n    },\n    "snapshot": {\n      "highlightsLabel": "Highlights",\n      "testsRunningError": "Cannot show snapshot while tests are running",\n      "snapshotMissingError": "The snapshot is missing. Displaying current state of the DOM.",\n      "defaultTitle": "DOM snapshot",\n      "pinnedTitle": "Pinned"\n    },\n    "selectorPlayground": {\n      "matches": "No matches | {n} match | {n} matches",\n      "playgroundTooltip": "Click an element to see a suggested selector",\n      "copyTooltip": "Copy to clipboard",\n      "copyTooltipAction": "Copied!",\n      "printTooltip": "Print to console",\n      "printTooltipAction": "Printed!",\n      "invalidSelector": "Invalid",\n      "selectorMethodsLabel": "Selector methods",\n      "toggle": "Toggle playground"\n    },\n    "automation": {\n      "disconnected": {\n        "title": "The Cypress extension has disconnected.",\n        "description": "Cypress cannot run tests without this extension.",\n        "reload": "Reload the browser"\n      },\n      "missing": {\n        "title": "The Cypress extension is missing.",\n        "description": "Cypress cannot run tests without this extension. Please choose another browser."\n      },\n      "shared": {\n        "link": "Read more about browser management"\n      }\n    }\n  },\n  "warnings": {\n    "retry": "Try again"\n  },\n  "versions": {\n    "alpha": "Alpha",\n    "community": "Community",\n    "beta": "Beta",\n    "new": "New",\n    "communityNote": "This is a community provided definition."\n  }\n}\n';
const defaultMessages = JSON.parse(rawJsonMessages);
createI18n();
function createI18n(opts = {}) {
  return createI18n$1({
    locale: "en-US",
    /**
    * precompiled messages from unplugin-vue-i18n do not include explicit keys derived from
    * the filenames of the raw message sources, so it must be coerced.
    */
    messages: compiledMessages,
    ...opts
  });
}
function useI18n() {
  return useI18n$1({ useScope: "global" });
}
const _hoisted_1$23 = ["href"];
const _sfc_main$1j = /* @__PURE__ */ defineComponent({
  __name: "BaseLink",
  props: {
    href: { default: "" },
    useDefaultHocus: { type: Boolean, default: true }
  },
  setup(__props) {
    const props = __props;
    const attrs = useAttrs();
    const classes = computed(() => {
      const hocus = props.useDefaultHocus ? "hocus-link-default" : "";
      return `${attrs.class ?? "text-indigo-500"} ${hocus}`;
    });
    return (_ctx, _cache) => {
      return openBlock(), createElementBlock("a", {
        href: props.href,
        class: normalizeClass(unref(classes))
      }, [
        renderSlot(_ctx.$slots, "default")
      ], 10, _hoisted_1$23);
    };
  }
});
const Auth_LogoutDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "Auth_Logout" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "logout" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "cloudViewer" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "cloudOrganizationsUrl" } }, { "kind": "Field", "name": { "kind": "Name", "value": "organizations" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "first" }, "value": { "kind": "IntValue", "value": "100" } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "nodes" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "projects" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "first" }, "value": { "kind": "IntValue", "value": "100" } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "nodes" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "slug" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }] } }] } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "email" } }, { "kind": "Field", "name": { "kind": "Name", "value": "fullName" } }, { "kind": "Field", "alias": { "kind": "Name", "value": "firstOrganization" }, "name": { "kind": "Name", "value": "organizations" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "first" }, "value": { "kind": "IntValue", "value": "1" } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "nodes" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }] } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "currentProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "title" } }, { "kind": "Field", "name": { "kind": "Name", "value": "projectId" } }, { "kind": "Field", "name": { "kind": "Name", "value": "cloudProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "__typename" } }, { "kind": "InlineFragment", "typeCondition": { "kind": "NamedType", "name": { "kind": "Name", "value": "CloudProject" } }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }] } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "authState" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "browserOpened" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "message" } }] } }] } }] } }] };
const Auth_LoginDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "Auth_Login" }, "variableDefinitions": [{ "kind": "VariableDefinition", "variable": { "kind": "Variable", "name": { "kind": "Name", "value": "utmSource" } }, "type": { "kind": "NonNullType", "type": { "kind": "NamedType", "name": { "kind": "Name", "value": "String" } } } }, { "kind": "VariableDefinition", "variable": { "kind": "Variable", "name": { "kind": "Name", "value": "utmMedium" } }, "type": { "kind": "NonNullType", "type": { "kind": "NamedType", "name": { "kind": "Name", "value": "String" } } } }, { "kind": "VariableDefinition", "variable": { "kind": "Variable", "name": { "kind": "Name", "value": "utmContent" } }, "type": { "kind": "NamedType", "name": { "kind": "Name", "value": "String" } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "login" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "utmSource" }, "value": { "kind": "Variable", "name": { "kind": "Name", "value": "utmSource" } } }, { "kind": "Argument", "name": { "kind": "Name", "value": "utmContent" }, "value": { "kind": "Variable", "name": { "kind": "Name", "value": "utmContent" } } }, { "kind": "Argument", "name": { "kind": "Name", "value": "utmMedium" }, "value": { "kind": "Variable", "name": { "kind": "Name", "value": "utmMedium" } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "cloudViewer" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "cloudOrganizationsUrl" } }, { "kind": "Field", "name": { "kind": "Name", "value": "organizations" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "first" }, "value": { "kind": "IntValue", "value": "100" } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "nodes" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "projects" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "first" }, "value": { "kind": "IntValue", "value": "100" } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "nodes" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "slug" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }] } }] } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "email" } }, { "kind": "Field", "name": { "kind": "Name", "value": "fullName" } }, { "kind": "Field", "alias": { "kind": "Name", "value": "firstOrganization" }, "name": { "kind": "Name", "value": "organizations" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "first" }, "value": { "kind": "IntValue", "value": "1" } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "nodes" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }] } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "currentProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "title" } }, { "kind": "Field", "name": { "kind": "Name", "value": "projectId" } }, { "kind": "Field", "name": { "kind": "Name", "value": "cloudProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "__typename" } }, { "kind": "InlineFragment", "typeCondition": { "kind": "NamedType", "name": { "kind": "Name", "value": "CloudProject" } }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }] } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "authState" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "browserOpened" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "message" } }] } }] } }] } }] };
const Auth_ResetAuthStateDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "Auth_ResetAuthState" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "resetAuthState" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "cloudViewer" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "cloudOrganizationsUrl" } }, { "kind": "Field", "name": { "kind": "Name", "value": "organizations" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "first" }, "value": { "kind": "IntValue", "value": "100" } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "nodes" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "projects" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "first" }, "value": { "kind": "IntValue", "value": "100" } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "nodes" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "slug" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }] } }] } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "email" } }, { "kind": "Field", "name": { "kind": "Name", "value": "fullName" } }, { "kind": "Field", "alias": { "kind": "Name", "value": "firstOrganization" }, "name": { "kind": "Name", "value": "organizations" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "first" }, "value": { "kind": "IntValue", "value": "1" } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "nodes" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }] } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "currentProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "title" } }, { "kind": "Field", "name": { "kind": "Name", "value": "projectId" } }, { "kind": "Field", "name": { "kind": "Name", "value": "cloudProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "__typename" } }, { "kind": "InlineFragment", "typeCondition": { "kind": "NamedType", "name": { "kind": "Name", "value": "CloudProject" } }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }] } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "authState" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "browserOpened" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "message" } }] } }] } }] } }] };
const ChooseExternalEditorModal_SetPreferredEditorBinaryDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "ChooseExternalEditorModal_SetPreferredEditorBinary" }, "variableDefinitions": [{ "kind": "VariableDefinition", "variable": { "kind": "Variable", "name": { "kind": "Name", "value": "value" } }, "type": { "kind": "NonNullType", "type": { "kind": "NamedType", "name": { "kind": "Name", "value": "String" } } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "setPreferences" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "value" }, "value": { "kind": "Variable", "name": { "kind": "Name", "value": "value" } } }, { "kind": "Argument", "name": { "kind": "Name", "value": "type" }, "value": { "kind": "EnumValue", "value": "global" } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "localSettings" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "preferences" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "preferredEditorBinary" } }] } }] } }] } }] } }] };
const CloudViewerAndProject_RequiredDataDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "query", "name": { "kind": "Name", "value": "CloudViewerAndProject_RequiredData" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "cloudViewer" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "fullName" } }, { "kind": "Field", "name": { "kind": "Name", "value": "email" } }, { "kind": "Field", "alias": { "kind": "Name", "value": "firstOrganization" }, "name": { "kind": "Name", "value": "organizations" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "first" }, "value": { "kind": "IntValue", "value": "1" } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "nodes" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }] } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "cachedUser" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "fullName" } }, { "kind": "Field", "name": { "kind": "Name", "value": "email" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "authState" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "name" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "currentProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "projectId" } }, { "kind": "Field", "name": { "kind": "Name", "value": "config" } }, { "kind": "Field", "name": { "kind": "Name", "value": "currentTestingType" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isFullConfigReady" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isCTConfigured" } }, { "kind": "Field", "name": { "kind": "Name", "value": "hasNonExampleSpec" } }, { "kind": "Field", "name": { "kind": "Name", "value": "savedState" } }, { "kind": "Field", "name": { "kind": "Name", "value": "branch" } }, { "kind": "Field", "name": { "kind": "Name", "value": "cloudProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "__typename" } }, { "kind": "InlineFragment", "typeCondition": { "kind": "NamedType", "name": { "kind": "Name", "value": "CloudProject" } }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "runs" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "first" }, "value": { "kind": "IntValue", "value": "10" } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "nodes" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "status" } }, { "kind": "Field", "name": { "kind": "Name", "value": "url" } }] } }] } }] } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "wizard" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "framework" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "icon" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isDetected" } }] } }] } }] } }] };
const CloudViewerAndProject_CheckCloudOrgMembershipDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "subscription", "name": { "kind": "Name", "value": "CloudViewerAndProject_CheckCloudOrgMembership" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "cloudViewerChange" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "cloudViewer" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "fullName" } }, { "kind": "Field", "name": { "kind": "Name", "value": "email" } }, { "kind": "Field", "alias": { "kind": "Name", "value": "firstOrganization" }, "name": { "kind": "Name", "value": "organizations" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "first" }, "value": { "kind": "IntValue", "value": "1" } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "nodes" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }] } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "cachedUser" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "fullName" } }, { "kind": "Field", "name": { "kind": "Name", "value": "email" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "authState" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "name" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "currentProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "projectId" } }, { "kind": "Field", "name": { "kind": "Name", "value": "config" } }, { "kind": "Field", "name": { "kind": "Name", "value": "currentTestingType" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isFullConfigReady" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isCTConfigured" } }, { "kind": "Field", "name": { "kind": "Name", "value": "hasNonExampleSpec" } }, { "kind": "Field", "name": { "kind": "Name", "value": "savedState" } }, { "kind": "Field", "name": { "kind": "Name", "value": "branch" } }, { "kind": "Field", "name": { "kind": "Name", "value": "cloudProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "__typename" } }, { "kind": "InlineFragment", "typeCondition": { "kind": "NamedType", "name": { "kind": "Name", "value": "CloudProject" } }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "runs" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "first" }, "value": { "kind": "IntValue", "value": "10" } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "nodes" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "status" } }, { "kind": "Field", "name": { "kind": "Name", "value": "url" } }] } }] } }] } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "wizard" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "framework" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "icon" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isDetected" } }] } }] } }] } }] } }] };
const CloudViewerAndProject_DetectCtFrameworksDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "CloudViewerAndProject_DetectCtFrameworks" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "initializeCtFrameworks" } }] } }] };
const HeaderBar_HeaderBarQueryDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "query", "name": { "kind": "Name", "value": "HeaderBar_HeaderBarQuery" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "currentProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "title" } }, { "kind": "Field", "name": { "kind": "Name", "value": "config" } }, { "kind": "Field", "name": { "kind": "Name", "value": "savedState" } }, { "kind": "Field", "name": { "kind": "Name", "value": "currentTestingType" } }, { "kind": "Field", "name": { "kind": "Name", "value": "branch" } }, { "kind": "Field", "name": { "kind": "Name", "value": "packageManager" } }, { "kind": "Field", "name": { "kind": "Name", "value": "activeBrowser" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "displayName" } }, { "kind": "Field", "name": { "kind": "Name", "value": "majorVersion" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "browsers" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isSelected" } }, { "kind": "Field", "name": { "kind": "Name", "value": "displayName" } }, { "kind": "Field", "name": { "kind": "Name", "value": "version" } }, { "kind": "Field", "name": { "kind": "Name", "value": "majorVersion" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isVersionSupported" } }, { "kind": "Field", "name": { "kind": "Name", "value": "warning" } }, { "kind": "Field", "name": { "kind": "Name", "value": "disabled" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "projectId" } }, { "kind": "Field", "name": { "kind": "Name", "value": "cloudProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "__typename" } }, { "kind": "InlineFragment", "typeCondition": { "kind": "NamedType", "name": { "kind": "Name", "value": "CloudProject" } }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }] } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "isGlobalMode" } }, { "kind": "Field", "name": { "kind": "Name", "value": "versions" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "current" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "version" } }, { "kind": "Field", "name": { "kind": "Name", "value": "released" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "latest" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "version" } }, { "kind": "Field", "name": { "kind": "Name", "value": "released" } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "cloudViewer" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "cloudOrganizationsUrl" } }, { "kind": "Field", "name": { "kind": "Name", "value": "organizations" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "first" }, "value": { "kind": "IntValue", "value": "100" } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "nodes" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "projects" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "first" }, "value": { "kind": "IntValue", "value": "100" } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "nodes" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "slug" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }] } }] } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "email" } }, { "kind": "Field", "name": { "kind": "Name", "value": "fullName" } }, { "kind": "Field", "alias": { "kind": "Name", "value": "firstOrganization" }, "name": { "kind": "Name", "value": "organizations" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "first" }, "value": { "kind": "IntValue", "value": "1" } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "nodes" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }] } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "authState" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "browserOpened" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "message" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "cachedUser" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "fullName" } }, { "kind": "Field", "name": { "kind": "Name", "value": "email" } }] } }] } }] };
const HeaderBarContent_AuthChangeDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "subscription", "name": { "kind": "Name", "value": "HeaderBarContent_authChange" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "authChange" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "cloudViewer" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "cloudOrganizationsUrl" } }, { "kind": "Field", "name": { "kind": "Name", "value": "organizations" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "first" }, "value": { "kind": "IntValue", "value": "100" } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "nodes" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "projects" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "first" }, "value": { "kind": "IntValue", "value": "100" } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "nodes" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "slug" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }] } }] } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "email" } }, { "kind": "Field", "name": { "kind": "Name", "value": "fullName" } }, { "kind": "Field", "alias": { "kind": "Name", "value": "firstOrganization" }, "name": { "kind": "Name", "value": "organizations" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "first" }, "value": { "kind": "IntValue", "value": "1" } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "nodes" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }] } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "currentProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "title" } }, { "kind": "Field", "name": { "kind": "Name", "value": "projectId" } }, { "kind": "Field", "name": { "kind": "Name", "value": "cloudProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "__typename" } }, { "kind": "InlineFragment", "typeCondition": { "kind": "NamedType", "name": { "kind": "Name", "value": "CloudProject" } }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }] } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "authState" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "browserOpened" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "message" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "cachedUser" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "fullName" } }, { "kind": "Field", "name": { "kind": "Name", "value": "email" } }] } }] } }] } }] };
const GlobalPageHeader_ClearCurrentProjectDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "GlobalPageHeader_clearCurrentProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "clearCurrentProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "baseError" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "title" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errorName" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errorStack" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errorType" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errorMessage" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isUserCodeError" } }, { "kind": "Field", "name": { "kind": "Name", "value": "codeFrame" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "line" } }, { "kind": "Field", "name": { "kind": "Name", "value": "column" } }, { "kind": "Field", "name": { "kind": "Name", "value": "codeBlock" } }, { "kind": "Field", "name": { "kind": "Name", "value": "file" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "absolute" } }, { "kind": "Field", "name": { "kind": "Name", "value": "relative" } }] } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "warnings" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "currentProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }] } }] } }] } }] };
const LoginConnectModals_LoginConnectModalsQueryDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "query", "name": { "kind": "Name", "value": "LoginConnectModals_LoginConnectModalsQuery" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "cloudViewer" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "cloudOrganizationsUrl" } }, { "kind": "Field", "name": { "kind": "Name", "value": "organizations" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "first" }, "value": { "kind": "IntValue", "value": "100" } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "nodes" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "projects" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "first" }, "value": { "kind": "IntValue", "value": "100" } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "nodes" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "slug" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }] } }] } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "email" } }, { "kind": "Field", "name": { "kind": "Name", "value": "fullName" } }, { "kind": "Field", "alias": { "kind": "Name", "value": "firstOrganization" }, "name": { "kind": "Name", "value": "organizations" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "first" }, "value": { "kind": "IntValue", "value": "1" } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "nodes" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "createCloudOrganizationUrl" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "currentProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "title" } }, { "kind": "Field", "name": { "kind": "Name", "value": "projectId" } }, { "kind": "Field", "name": { "kind": "Name", "value": "cloudProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "__typename" } }, { "kind": "InlineFragment", "typeCondition": { "kind": "NamedType", "name": { "kind": "Name", "value": "CloudProject" } }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "currentTestingType" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "authState" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "browserOpened" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "message" } }] } }] } }] };
const OpenFileInIdeDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "query", "name": { "kind": "Name", "value": "OpenFileInIDE" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "localSettings" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "preferences" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "preferredEditorBinary" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "availableEditors" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "binary" } }] } }] } }] } }] };
const OpenFileInIde_MutationDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "OpenFileInIDE_Mutation" }, "variableDefinitions": [{ "kind": "VariableDefinition", "variable": { "kind": "Variable", "name": { "kind": "Name", "value": "input" } }, "type": { "kind": "NonNullType", "type": { "kind": "NamedType", "name": { "kind": "Name", "value": "FileDetailsInput" } } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "openFileInIDE" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "input" }, "value": { "kind": "Variable", "name": { "kind": "Name", "value": "input" } } }] }] } }] };
const RecordPromptAdapterDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "query", "name": { "kind": "Name", "value": "RecordPromptAdapter" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "currentProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "title" } }, { "kind": "Field", "name": { "kind": "Name", "value": "currentTestingType" } }, { "kind": "Field", "name": { "kind": "Name", "value": "cloudProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "__typename" } }, { "kind": "InlineFragment", "typeCondition": { "kind": "NamedType", "name": { "kind": "Name", "value": "CloudProject" } }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "recordKeys" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "key" } }] } }] } }] } }] } }] } }] };
const UsePromptManager_SetProjectPreferencesDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "UsePromptManager_SetProjectPreferences" }, "variableDefinitions": [{ "kind": "VariableDefinition", "variable": { "kind": "Variable", "name": { "kind": "Name", "value": "value" } }, "type": { "kind": "NonNullType", "type": { "kind": "NamedType", "name": { "kind": "Name", "value": "String" } } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "setPreferences" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "type" }, "value": { "kind": "EnumValue", "value": "project" } }, { "kind": "Argument", "name": { "kind": "Name", "value": "value" }, "value": { "kind": "Variable", "name": { "kind": "Name", "value": "value" } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "currentProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "savedState" } }] } }] } }] } }] };
const UsePromptManager_SetGlobalPreferencesDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "UsePromptManager_SetGlobalPreferences" }, "variableDefinitions": [{ "kind": "VariableDefinition", "variable": { "kind": "Variable", "name": { "kind": "Name", "value": "value" } }, "type": { "kind": "NonNullType", "type": { "kind": "NamedType", "name": { "kind": "Name", "value": "String" } } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "setPreferences" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "type" }, "value": { "kind": "EnumValue", "value": "global" } }, { "kind": "Argument", "name": { "kind": "Name", "value": "value" }, "value": { "kind": "Variable", "name": { "kind": "Name", "value": "value" } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "localSettings" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "preferences" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "majorVersionWelcomeDismissed" } }] } }] } }] } }] } }] };
const CloudConnectModals_MonitorCloudViewerDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "subscription", "name": { "kind": "Name", "value": "CloudConnectModals_MonitorCloudViewer" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "cloudViewerChange" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "cloudViewer" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "cloudOrganizationsUrl" } }, { "kind": "Field", "name": { "kind": "Name", "value": "organizations" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "first" }, "value": { "kind": "IntValue", "value": "100" } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "nodes" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "projects" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "first" }, "value": { "kind": "IntValue", "value": "100" } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "nodes" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "slug" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }] } }] } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "createCloudOrganizationUrl" } }, { "kind": "Field", "alias": { "kind": "Name", "value": "firstOrganization" }, "name": { "kind": "Name", "value": "organizations" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "first" }, "value": { "kind": "IntValue", "value": "1" } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "nodes" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }] } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "currentProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "title" } }, { "kind": "Field", "name": { "kind": "Name", "value": "projectId" } }] } }] } }] } }] };
const CreateCloudOrgModal_CloudOrganizationsCheckDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "CreateCloudOrgModal_CloudOrganizationsCheck" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "refreshCloudViewer" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "cloudViewer" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "cloudOrganizationsUrl" } }, { "kind": "Field", "name": { "kind": "Name", "value": "organizations" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "first" }, "value": { "kind": "IntValue", "value": "100" } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "nodes" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }, { "kind": "Field", "name": { "kind": "Name", "value": "projects" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "first" }, "value": { "kind": "IntValue", "value": "100" } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "nodes" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "slug" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }] } }] } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "createCloudOrganizationUrl" } }, { "kind": "Field", "alias": { "kind": "Name", "value": "firstOrganization" }, "name": { "kind": "Name", "value": "organizations" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "first" }, "value": { "kind": "IntValue", "value": "1" } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "nodes" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }] } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "currentProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "title" } }, { "kind": "Field", "name": { "kind": "Name", "value": "projectId" } }] } }] } }] } }] };
const SelectCloudProjectModal_SetProjectIdDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "SelectCloudProjectModal_SetProjectId" }, "variableDefinitions": [{ "kind": "VariableDefinition", "variable": { "kind": "Variable", "name": { "kind": "Name", "value": "projectId" } }, "type": { "kind": "NonNullType", "type": { "kind": "NamedType", "name": { "kind": "Name", "value": "String" } } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "setProjectIdInConfigFile" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "projectId" }, "value": { "kind": "Variable", "name": { "kind": "Name", "value": "projectId" } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "currentProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "projectId" } }, { "kind": "Field", "name": { "kind": "Name", "value": "cloudProject" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "__typename" } }, { "kind": "InlineFragment", "typeCondition": { "kind": "NamedType", "name": { "kind": "Name", "value": "CloudProject" } }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "runs" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "first" }, "value": { "kind": "IntValue", "value": "10" } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "nodes" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "createdAt" } }, { "kind": "Field", "name": { "kind": "Name", "value": "status" } }, { "kind": "Field", "name": { "kind": "Name", "value": "totalDuration" } }, { "kind": "Field", "name": { "kind": "Name", "value": "url" } }, { "kind": "Field", "name": { "kind": "Name", "value": "tags" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "name" } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "totalPassed" } }, { "kind": "Field", "name": { "kind": "Name", "value": "totalFailed" } }, { "kind": "Field", "name": { "kind": "Name", "value": "totalPending" } }, { "kind": "Field", "name": { "kind": "Name", "value": "totalSkipped" } }, { "kind": "Field", "name": { "kind": "Name", "value": "totalFlakyTests" } }, { "kind": "Field", "name": { "kind": "Name", "value": "commitInfo" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "authorName" } }, { "kind": "Field", "name": { "kind": "Name", "value": "authorEmail" } }, { "kind": "Field", "name": { "kind": "Name", "value": "summary" } }, { "kind": "Field", "name": { "kind": "Name", "value": "branch" } }] } }] } }] } }] } }] } }] } }] } }] } }] };
const SelectCloudProjectModal_CreateCloudProjectDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "SelectCloudProjectModal_CreateCloudProject" }, "variableDefinitions": [{ "kind": "VariableDefinition", "variable": { "kind": "Variable", "name": { "kind": "Name", "value": "name" } }, "type": { "kind": "NonNullType", "type": { "kind": "NamedType", "name": { "kind": "Name", "value": "String" } } } }, { "kind": "VariableDefinition", "variable": { "kind": "Variable", "name": { "kind": "Name", "value": "orgId" } }, "type": { "kind": "NonNullType", "type": { "kind": "NamedType", "name": { "kind": "Name", "value": "ID" } } } }, { "kind": "VariableDefinition", "variable": { "kind": "Variable", "name": { "kind": "Name", "value": "public" } }, "type": { "kind": "NonNullType", "type": { "kind": "NamedType", "name": { "kind": "Name", "value": "Boolean" } } } }, { "kind": "VariableDefinition", "variable": { "kind": "Variable", "name": { "kind": "Name", "value": "campaign" } }, "type": { "kind": "NamedType", "name": { "kind": "Name", "value": "String" } } }, { "kind": "VariableDefinition", "variable": { "kind": "Variable", "name": { "kind": "Name", "value": "cohort" } }, "type": { "kind": "NamedType", "name": { "kind": "Name", "value": "String" } } }, { "kind": "VariableDefinition", "variable": { "kind": "Variable", "name": { "kind": "Name", "value": "medium" } }, "type": { "kind": "NamedType", "name": { "kind": "Name", "value": "String" } } }, { "kind": "VariableDefinition", "variable": { "kind": "Variable", "name": { "kind": "Name", "value": "source" } }, "type": { "kind": "NonNullType", "type": { "kind": "NamedType", "name": { "kind": "Name", "value": "String" } } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "cloudProjectCreate" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "name" }, "value": { "kind": "Variable", "name": { "kind": "Name", "value": "name" } } }, { "kind": "Argument", "name": { "kind": "Name", "value": "orgId" }, "value": { "kind": "Variable", "name": { "kind": "Name", "value": "orgId" } } }, { "kind": "Argument", "name": { "kind": "Name", "value": "public" }, "value": { "kind": "Variable", "name": { "kind": "Name", "value": "public" } } }, { "kind": "Argument", "name": { "kind": "Name", "value": "campaign" }, "value": { "kind": "Variable", "name": { "kind": "Name", "value": "campaign" } } }, { "kind": "Argument", "name": { "kind": "Name", "value": "cohort" }, "value": { "kind": "Variable", "name": { "kind": "Name", "value": "cohort" } } }, { "kind": "Argument", "name": { "kind": "Name", "value": "medium" }, "value": { "kind": "Variable", "name": { "kind": "Name", "value": "medium" } } }, { "kind": "Argument", "name": { "kind": "Name", "value": "source" }, "value": { "kind": "Variable", "name": { "kind": "Name", "value": "source" } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "slug" } }] } }] } }] };
const TopNav_SetPromptShownDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "TopNav_SetPromptShown" }, "variableDefinitions": [{ "kind": "VariableDefinition", "variable": { "kind": "Variable", "name": { "kind": "Name", "value": "slug" } }, "type": { "kind": "NonNullType", "type": { "kind": "NamedType", "name": { "kind": "Name", "value": "String" } } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "setPromptShown" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "slug" }, "value": { "kind": "Variable", "name": { "kind": "Name", "value": "slug" } } }] }] } }] };
const VerticalBrowserListItems_SetBrowserDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "VerticalBrowserListItems_SetBrowser" }, "variableDefinitions": [{ "kind": "VariableDefinition", "variable": { "kind": "Variable", "name": { "kind": "Name", "value": "id" } }, "type": { "kind": "NonNullType", "type": { "kind": "NamedType", "name": { "kind": "Name", "value": "ID" } } } }, { "kind": "VariableDefinition", "variable": { "kind": "Variable", "name": { "kind": "Name", "value": "specPath" } }, "type": { "kind": "NamedType", "name": { "kind": "Name", "value": "String" } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "launchpadSetBrowser" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "id" }, "value": { "kind": "Variable", "name": { "kind": "Name", "value": "id" } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "browsers" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isSelected" } }, { "kind": "Field", "name": { "kind": "Name", "value": "displayName" } }, { "kind": "Field", "name": { "kind": "Name", "value": "version" } }, { "kind": "Field", "name": { "kind": "Name", "value": "majorVersion" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isVersionSupported" } }, { "kind": "Field", "name": { "kind": "Name", "value": "warning" } }, { "kind": "Field", "name": { "kind": "Name", "value": "disabled" } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "launchOpenProject" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "specPath" }, "value": { "kind": "Variable", "name": { "kind": "Name", "value": "specPath" } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }] } }] } }] };
const Clipboard_CopyToClipboardDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "Clipboard_CopyToClipboard" }, "variableDefinitions": [{ "kind": "VariableDefinition", "variable": { "kind": "Variable", "name": { "kind": "Name", "value": "text" } }, "type": { "kind": "NonNullType", "type": { "kind": "NamedType", "name": { "kind": "Name", "value": "String" } } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "copyTextToClipboard" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "text" }, "value": { "kind": "Variable", "name": { "kind": "Name", "value": "text" } } }] }] } }] };
const ExternalLink_OpenExternalDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "mutation", "name": { "kind": "Name", "value": "ExternalLink_OpenExternal" }, "variableDefinitions": [{ "kind": "VariableDefinition", "variable": { "kind": "Variable", "name": { "kind": "Name", "value": "url" } }, "type": { "kind": "NonNullType", "type": { "kind": "NamedType", "name": { "kind": "Name", "value": "String" } } } }, { "kind": "VariableDefinition", "variable": { "kind": "Variable", "name": { "kind": "Name", "value": "includeGraphqlPort" } }, "type": { "kind": "NamedType", "name": { "kind": "Name", "value": "Boolean" } } }], "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "openExternal" }, "arguments": [{ "kind": "Argument", "name": { "kind": "Name", "value": "url" }, "value": { "kind": "Variable", "name": { "kind": "Name", "value": "url" } } }, { "kind": "Argument", "name": { "kind": "Name", "value": "includeGraphqlPort" }, "value": { "kind": "Variable", "name": { "kind": "Name", "value": "includeGraphqlPort" } } }] }] } }] };
const GlobalSubscriptions_PushFragmentDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "subscription", "name": { "kind": "Name", "value": "GlobalSubscriptions_PushFragment" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "pushFragment" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "target" } }, { "kind": "Field", "name": { "kind": "Name", "value": "fragment" } }, { "kind": "Field", "name": { "kind": "Name", "value": "data" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errors" } }, { "kind": "Field", "name": { "kind": "Name", "value": "variables" } }, { "kind": "Field", "name": { "kind": "Name", "value": "invalidateCache" } }] } }] } }] };
const GlobalSubscriptions_ErrorWarningChangeDocument = { "kind": "Document", "definitions": [{ "kind": "OperationDefinition", "operation": "subscription", "name": { "kind": "Name", "value": "GlobalSubscriptions_ErrorWarningChange" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "errorWarningChange" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "baseError" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "title" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errorName" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errorStack" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errorType" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errorMessage" } }, { "kind": "Field", "name": { "kind": "Name", "value": "isUserCodeError" } }, { "kind": "Field", "name": { "kind": "Name", "value": "codeFrame" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "line" } }, { "kind": "Field", "name": { "kind": "Name", "value": "column" } }, { "kind": "Field", "name": { "kind": "Name", "value": "codeBlock" } }, { "kind": "Field", "name": { "kind": "Name", "value": "file" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "absolute" } }, { "kind": "Field", "name": { "kind": "Name", "value": "relative" } }] } }] } }] } }, { "kind": "Field", "name": { "kind": "Name", "value": "warnings" }, "selectionSet": { "kind": "SelectionSet", "selections": [{ "kind": "Field", "name": { "kind": "Name", "value": "id" } }, { "kind": "Field", "name": { "kind": "Name", "value": "title" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errorType" } }, { "kind": "Field", "name": { "kind": "Name", "value": "errorMessage" } }] } }] } }] } }] };
gql`
mutation ExternalLink_OpenExternal ($url: String!, $includeGraphqlPort: Boolean) {
  openExternal(url: $url, includeGraphqlPort: $includeGraphqlPort)
}
`;
const useExternalLink = ($href, includeGraphqlPort = false) => {
  const openExternalMutation = useMutation(ExternalLink_OpenExternalDocument);
  return (href) => {
    const resolvedHref = unref(typeof href === "string" ? href : $href);
    if (!resolvedHref) {
      return new Error(`Cannot open external link. Possible urls passed in were ${{ localHref: href, initialHref: unref($href) }}`);
    }
    return openExternalMutation.executeMutation({ url: resolvedHref, includeGraphqlPort });
  };
};
const __default__$4 = defineComponent({
  inheritAttrs: true
});
const _sfc_main$1i = /* @__PURE__ */ defineComponent({
  ...__default__$4,
  __name: "ExternalLink",
  props: {
    href: { default: "" },
    useDefaultHocus: { type: Boolean, default: true },
    includeGraphqlPort: { type: Boolean, default: false }
  },
  setup(__props) {
    const props = __props;
    const open = useExternalLink(props.href, props.includeGraphqlPort);
    return (_ctx, _cache) => {
      return openBlock(), createBlock(_sfc_main$1j, {
        "data-cy": "external",
        href: props.href,
        "use-default-hocus": props.useDefaultHocus,
        onClick: withModifiers(unref(open), ["prevent"]),
        onKeypress: withKeys(withModifiers(unref(open), ["prevent"]), ["enter"])
      }, {
        default: withCtx(() => [
          renderSlot(_ctx.$slots, "default")
        ]),
        _: 3
      }, 8, ["href", "use-default-hocus", "onClick", "onKeypress"]);
    };
  }
});
const _hoisted_1$22 = {
  key: 0,
  class: "flex m-[24px] justify-center"
};
const _sfc_main$1h = /* @__PURE__ */ defineComponent({
  __name: "TestingTypePicker",
  props: {
    gql: null
  },
  emits: ["pick"],
  setup(__props, { emit: emits }) {
    const props = __props;
    const { t: t2 } = useI18n();
    gql`
fragment TestingTypePicker on Query {
  currentProject {
    id
    isCTConfigured
    isE2EConfigured
    currentTestingType
  }
  invokedFromCli
}
`;
    const e2eStatus = computed(() => {
      var _a2;
      if (!((_a2 = props.gql.currentProject) == null ? void 0 : _a2.isE2EConfigured)) return "notConfigured";
      return props.gql.currentProject.currentTestingType === "e2e" ? "running" : "configured";
    });
    const componentStatus = computed(() => {
      var _a2;
      if (!props.gql.invokedFromCli) return "disabled";
      if (!((_a2 = props.gql.currentProject) == null ? void 0 : _a2.isCTConfigured)) return "notConfigured";
      return props.gql.currentProject.currentTestingType === "component" ? "running" : "configured";
    });
    const testingTypes = computed(() => {
      return [
        {
          key: "e2e",
          name: t2("testingType.e2e.name"),
          description: t2("testingType.e2e.description"),
          icon: IconE2E,
          iconSolid: IconE2ESolid,
          status: e2eStatus.value,
          badgeText: ""
        },
        {
          key: "component",
          name: t2("testingType.component.name"),
          description: t2("testingType.component.description"),
          icon: IconComponent,
          iconSolid: IconComponentSolid,
          status: componentStatus.value,
          badgeText: ""
        }
      ];
    });
    const currentTestingType = computed(() => {
      var _a2;
      return (_a2 = props.gql.currentProject) == null ? void 0 : _a2.currentTestingType;
    });
    return (_ctx, _cache) => {
      const _component_i18n_t = resolveComponent("i18n-t");
      return props.gql.currentProject ? (openBlock(), createElementBlock("div", _hoisted_1$22, [
        (openBlock(true), createElementBlock(Fragment, null, renderList(unref(testingTypes), (tt) => {
          return openBlock(), createBlock(Card, {
            key: tt.key,
            "data-cy-testingType": tt.key,
            title: tt.name,
            description: tt.description,
            class: "m-[24px] px-[24px] pt-[36px] pb-[36px] w-[336px]",
            icon: tt.icon,
            "hover-icon": tt.iconSolid,
            "icon-size": 64,
            disabled: tt.status === "disabled",
            variant: "jade",
            "badge-text": tt.badgeText,
            onClick: ($event) => emits("pick", tt.key, unref(currentTestingType)),
            onKeyup: [
              withKeys(($event) => emits("pick", tt.key, unref(currentTestingType)), ["enter"]),
              withKeys(($event) => emits("pick", tt.key, unref(currentTestingType)), ["space"])
            ]
          }, createSlots({
            footer: withCtx(() => [
              createVNode(_sfc_main$1k, {
                class: "mt-[16px]",
                title: unref(t2)(`setupPage.testingCard.${tt.status}`),
                status: tt.status === "configured" || tt.status === "running",
                onChooseABrowser: ($event) => emits("pick", tt.key, unref(currentTestingType))
              }, null, 8, ["title", "status", "onChooseABrowser"])
            ]),
            _: 2
          }, [
            tt.key === "component" && tt.status === "disabled" ? {
              name: "default",
              fn: withCtx(() => [
                createVNode(_component_i18n_t, {
                  scope: "global",
                  keypath: "testingType.componentDisabled.description"
                }, {
                  default: withCtx(() => [
                    createVNode(_sfc_main$1i, { href: "https://on.cypress.io/installing-cypress" }, {
                      default: withCtx(() => [
                        createTextVNode(toDisplayString$1(unref(t2)("testingType.componentDisabled.link")), 1)
                      ]),
                      _: 1
                    })
                  ]),
                  _: 1
                })
              ]),
              key: "0"
            } : void 0
          ]), 1032, ["data-cy-testingType", "title", "description", "icon", "hover-icon", "disabled", "badge-text", "onClick", "onKeyup"]);
        }), 128))
      ])) : createCommentVNode("", true);
    };
  }
});
const _sfc_main$1g = /* @__PURE__ */ defineComponent({
  __name: "TestingTypeCards",
  props: {
    gql: null
  },
  setup(__props) {
    const props = __props;
    gql`
fragment TestingTypeCards on Query {
  ...TestingTypePicker
}
`;
    gql`
mutation TestingTypeSelection($testingType: TestingTypeEnum!) {
  setAndLoadCurrentTestingType(testingType: $testingType) {
    currentProject {
      id
      currentTestingType
      isCTConfigured
      isE2EConfigured
      isLoadingConfigFile
      isLoadingNodeEvents
    }
    ...Wizard
  }
}
`;
    const mutation = useMutation(TestingTypeSelectionDocument);
    function selectTestingType(testingType) {
      if (!mutation.fetching.value) {
        mutation.executeMutation({ testingType });
      }
    }
    return (_ctx, _cache) => {
      return openBlock(), createBlock(_sfc_main$1h, {
        gql: props.gql,
        onPick: selectTestingType
      }, null, 8, ["gql"]);
    };
  }
});
const _hoisted_1$21 = {
  style: { "min-width": "16px", "min-height": "16px" },
  width: "16",
  height: "16",
  viewBox: "0 0 16 16",
  fill: "none",
  xmlns: "http://www.w3.org/2000/svg"
};
const _hoisted_2$1P = /* @__PURE__ */ createBaseVNode("circle", {
  id: "Circle",
  cx: "8",
  cy: "8",
  r: "6",
  stroke: "#E1E3ED",
  class: "icon-light",
  "stroke-width": "2",
  "stroke-linecap": "round",
  "stroke-linejoin": "round"
}, null, -1);
const _hoisted_3$1y = /* @__PURE__ */ createBaseVNode("path", {
  id: "Progress",
  d: "M14 8C14 4.68629 11.3137 2 8 2",
  stroke: "#6470F3",
  class: "icon-dark",
  "stroke-width": "2",
  "stroke-linecap": "round",
  "stroke-linejoin": "round"
}, null, -1);
const _hoisted_4$X = [
  _hoisted_2$1P,
  _hoisted_3$1y
];
function render$14(_ctx, _cache) {
  return openBlock(), createElementBlock("svg", _hoisted_1$21, _hoisted_4$X);
}
const __unplugin_components_0$k = { name: "cy-loading_x16", render: render$14 };
const b$1 = "border border-solid rounded rounded-[4px] flex cy-button-width font-medium items-center transition duration-150 hover:shadow-ring-hover focus:shadow-ring-focus active:shadow-ring-focus disabled:hover:shadow-none disabled:cursor-not-allowed focus-visible:ring-1 focus-visible:ring-offset-0 focus-visible:outline-none", c = {
  link: "text-indigo-600 border-transparent ring-transparent hover:underline disabled:text-gray-500 disabled:hocus:shadow-none disabled:no-underline hocus:shadow-transparent focus-visible:ring-gray-100 focus-visible:underline",
  white: "text-indigo-500 bg-white border-gray-100 hover:border-gray-200 disabled:hocus:shadow-none hocus:shadow-gray-50 disabled:text-gray-500 focus:ring-gray-200",
  disabled: "text-gray-500 bg-gray-100 border-gray-100 hover:shadow-none",
  // outline variants
  "outline-indigo": "border-indigo-500 text-indigo-500 disabled:hocus:shadow-none hocus:shadow-indigo-300/[.35] disabled:text-gray-500 focus:ring-indigo-600",
  "outline-purple": "text-purple-500 border-purple-500 disabled:hocus:shadow-none hocus:shadow-purple-100 disabled:text-gray-500 focus:ring-purple-600",
  "outline-red": "text-red-500 border-red-500 disabled:hocus:shadow-none hocus:shadow-red-100 disabled:text-gray-500 focus:ring-red-600",
  "outline-gray-dark": "text-gray-1000 border-gray-1000 disabled:hocus:shadow-none hocus:shadow-gray-100 disabled:text-gray-500 focus:ring-gray-1000",
  "outline-light": "text-indigo-500 border-gray-100 hocus:border-gray-200 disabled:border-gray-100 disabled:hocus:shadow-none hocus:shadow-gray-50 disabled:text-gray-500 focus:ring-gray-200",
  "outline-gray-light": "text-gray-700 border-gray-100 hocus:border-gray-200 disabled:border-gray-100 disabled:hocus:shadow-none hocus:shadow-gray-50 disabled:text-gray-300 focus:ring-gray-200",
  "outline-orange-dark": "border-orange-500 text-orange-500 disabled:hocus:shadow-none hocus:shadow-orange-100 disabled:text-gray-500 focus:ring-orange-500",
  "outline-orange-light": "border-orange-200 text-orange-500 disabled:hocus:shadow-none hocus:shadow-orange-50 disabled:text-gray-500 focus:ring-orange-400",
  "outline-disabled": "text-gray-500 border-gray-100 hover:shadow-none bg-white",
  // light variants
  "indigo-light": "text-indigo-500 bg-indigo-50 border-indigo-50 hocus:border-indigo-200 disabled:hocus:shadow-none hocus:shadow-indigo-50 focus:ring-transparent",
  "jade-light": "text-teal-600 bg-jade-200 border-transparent hocus:border-jade-300 disabled:hocus:shadow-none hocus:shadow-jade-50 focus:ring-transparent",
  // dark variants
  "jade-dark": "text-white bg-jade-500 border-jade-500 disabled:hocus:shadow-none hocus:shadow-jade-100 focus:ring-transparent focus:border-jade-600 focus:bg-jade-600",
  "indigo-dark": "text-white bg-indigo-500 border-indigo-500 disabled:hocus:shadow-none hocus:shadow-indigo-100 focus:bg-indigo-600 focus:border-indigo-600 focus:ring-transparent",
  "teal-dark": "text-white bg-teal-500 border-teal-500 disabled:hocus:shadow-none hocus:shadow-teal-100 focus:bg-teal-600 focus:bg-teal-600 focus:border-teal-600 focus:ring-transparent",
  "purple-dark": "text-white bg-purple-500 border-purple-500 disabled:hocus:shadow-none hocus:shadow-purple-100 focus:bg-purple-600 focus:border-purple-600 focus:ring-transparent",
  "red-dark": "text-white bg-red-500 border-red-500 disabled:hocus:shadow-none hocus:shadow-red-100 focus:bg-red-600 focus:border-red-600 focus:ring-transparent",
  "gray-dark": "text-white bg-gray-800 border-gray-800 disabled:hocus:shadow-none hocus:shadow-gray-100 focus:bg-gray-900 focus:border-gray-900 focus:ring-transparent",
  "gray-darkest": "text-white bg-gray-1000 border-gray-1000 disabled:hocus:shadow-none hocus:shadow-gray-100 focus:bg-gray-900 focus:border-gray-900 focus:ring-transparent",
  // Dark mode
  "outline-red-dark-mode": "text-red-400 border-red-400 disabled:hocus:shadow-none hocus:shadow-red-300/35 disabled:text-gray-700 disabled:border-white/20 focus:ring-red-600",
  "outline-jade-dark-mode": "text-jade-400 border-jade-400 disabled:hocus:shadow-none hocus:shadow-jade-300/35 disabled:text-gray-700 disabled:border-white/20 focus:ring-jade-600",
  "outline-indigo-dark-mode": "text-indigo-300 border-indigo-400 disabled:hocus:shadow-none hocus:shadow-indigo-400/40 disabled:text-gray-700 disabled:border-white/20 focus:ring-indigo-300",
  "outline-dark": "text-white border-white/20 hocus:border-white/60 disabled:hocus:shadow-none hocus:shadow-white/20 disabled:border-white/20 disabled:hover:border-white/20 disabled:text-white/50 focus:ring-gray-200",
  "red-dark-mode": "text-white bg-red-500 border-red-500 disabled:hocus:shadow-none disabled:bg-gray-1000 disabled:text-gray-800 disabled:border-none hocus:shadow-red-400/40 disabled:mix-blend-screen focus:bg-red-600 focus:border-red-600 focus:ring-transparent"
}, f = "indigo-dark", w$1 = {
  20: "text-[14px] leading-[18px] min-h-[20px]",
  24: "text-[14px] leading-[20px] min-h-[24px]",
  32: "text-[14px] leading-[20px] min-h-[32px]",
  40: "text-[16px] leading-[24px] min-h-[40px]",
  48: "text-[16px] leading-[24px] min-h-[48px]"
}, x$1 = {
  20: {
    wide: "px-[4px] py-0",
    square: "h-[20px] w-[20px] flex items-center justify-center"
  },
  24: {
    wide: "px-[8px] py-[1px]",
    square: "h-[24px] w-[24px] flex items-center justify-center"
  },
  32: {
    wide: "px-[12px] py-[5px]",
    square: "h-[32px] w-[32px] flex items-center justify-center"
  },
  40: {
    wide: "px-[16px] py-[7px]",
    square: "h-[40px] w-[40px] flex items-center justify-center"
  },
  48: {
    wide: "px-[20px] py-[11px]",
    square: "h-[48px] w-[48px] flex items-center justify-center"
  }
}, p = "40", y = defineComponent({
  props: {
    variant: {
      type: String,
      default: f
    },
    size: {
      type: String,
      default: p
    },
    disabled: {
      type: Boolean,
      default: false
    },
    href: {
      type: String
    },
    type: {
      type: String,
      default: "button"
    },
    square: {
      type: Boolean,
      default: false
    }
  },
  emits: {
    click(e) {
      return !!e;
    }
  },
  slots: Object,
  setup(e) {
    const d2 = computed(() => e.disabled && !e.variant.includes("dark-mode") && !e.variant.includes("outline") && e.variant !== "white" ? "disabled" : e.variant), s2 = computed(
      () => e.disabled || e.variant === "disabled" || e.variant === "outline-disabled"
    ), o2 = computed(
      () => c[d2.value]
    ), r2 = computed(() => [
      w$1[e.size],
      x$1[e.size][e.square ? "square" : "wide"]
    ]), t2 = computed(() => [
      b$1,
      o2.value,
      r2.value
    ]);
    return {
      href: computed(() => e.href),
      type: computed(() => e.type),
      finalDisabled: s2,
      allClasses: t2
    };
  }
}), m = (e, d2) => {
  const s2 = e.__vccOpts || e;
  for (const [o2, r2] of d2)
    s2[o2] = r2;
  return s2;
}, v = ["href", "aria-disabled"], k = ["disabled", "type"];
function j(e, d2, s2, o2, r2, t2) {
  return e.href ? (openBlock(), createElementBlock("a", {
    key: 0,
    href: e.href,
    class: normalizeClass(e.allClasses),
    "aria-disabled": e.finalDisabled ? "true" : void 0,
    onClick: d2[0] || (d2[0] = (i2) => e.$emit("click", i2))
  }, [
    renderSlot(e.$slots, "default")
  ], 10, v)) : (openBlock(), createElementBlock("button", {
    key: 1,
    class: normalizeClass(e.allClasses),
    disabled: e.finalDisabled,
    type: e.type,
    onClick: d2[1] || (d2[1] = (i2) => e.$emit("click", i2))
  }, [
    renderSlot(e.$slots, "default")
  ], 10, k));
}
const S$1 = /* @__PURE__ */ m(y, [["render", j]]);
const _hoisted_1$20 = ["aria-labelledby", "disabled", "aria-checked"];
const _sfc_main$1f = /* @__PURE__ */ defineComponent({
  __name: "Switch",
  props: {
    value: { type: Boolean, default: false },
    size: { default: "lg" },
    labelId: null,
    disabled: { type: Boolean, default: false }
  },
  emits: ["update"],
  setup(__props) {
    const sizeClasses = {
      "sm": {
        container: "w-[16px] h-[10px]",
        indicator: "w-[6px] h-[6px] ml-[2px]",
        translate: "translate-x-[6px]"
      },
      "md": {
        container: "w-[24px] h-[12px]",
        indicator: "w-[8px] h-[8px] ml-[2px]",
        translate: "translate-x-[12px]"
      },
      "lg": {
        container: "w-[32px] h-[16px]",
        indicator: "w-[12px] h-[12px] ml-[2px]",
        translate: "translate-x-[14px]"
      },
      "xl": {
        container: "w-[48px] h-[24px]",
        indicator: "w-[16px] h-[16px] ml-[4px]",
        translate: "translate-x-[24px]"
      }
    };
    return (_ctx, _cache) => {
      return openBlock(), createElementBlock("button", {
        "aria-labelledby": __props.labelId,
        class: normalizeClass(["border-transparent border rounded-[50px] relative hocus-default disabled:ring-0 disabled:outline-0 disabled:bg-gray-100", [__props.value ? "bg-jade-400" : "bg-gray-300", sizeClasses[__props.size].container, {
          "!hocus:ring-0": __props.size === "sm"
        }]]),
        disabled: __props.disabled,
        role: "switch",
        "aria-checked": __props.value,
        onClick: _cache[0] || (_cache[0] = ($event) => _ctx.$emit("update", !__props.value))
      }, [
        createBaseVNode("span", {
          class: normalizeClass(["bg-white rounded-[50px] transform transition-transform ease-out duration-200 block toggle", [{ [sizeClasses[__props.size].translate]: __props.value }, sizeClasses[__props.size].indicator]])
        }, null, 2)
      ], 10, _hoisted_1$20);
    };
  }
});
const _hoisted_1$1$ = { class: "rounded-b flex bg-gray-50 border-t border-t-gray-100 py-[16px] px-[24px] gap-3" };
const _hoisted_2$1O = /* @__PURE__ */ createBaseVNode("div", { class: "grow" }, null, -1);
const _hoisted_3$1x = {
  key: 2,
  class: "flex px-3 items-center"
};
const _sfc_main$1e = /* @__PURE__ */ defineComponent({
  __name: "ButtonBar",
  props: {
    next: { default: void 0 },
    back: { default: void 0 },
    skip: { default: void 0 },
    nextFn: { type: Function, default: void 0 },
    backFn: { type: Function, default: void 0 },
    skipFn: { type: Function, default: void 0 },
    alt: { default: void 0 },
    altFn: { type: Function, default: void 0 },
    canNavigateForward: { type: Boolean },
    mainVariant: { default: "indigo-dark" }
  },
  setup(__props) {
    const props = __props;
    const altValue = ref(false);
    const handleAlt = () => {
      var _a2;
      altValue.value = !altValue.value;
      (_a2 = props.altFn) == null ? void 0 : _a2.call(props, altValue.value);
    };
    return (_ctx, _cache) => {
      const _component_i_cy_loading_x16 = __unplugin_components_0$k;
      const _component_i_cy_arrow_right_x16 = __unplugin_components_0$m;
      return openBlock(), createElementBlock("div", _hoisted_1$1$, [
        renderSlot(_ctx.$slots, "default", {}, () => [
          __props.nextFn ? (openBlock(), createBlock(unref(S$1), {
            key: 0,
            size: "40",
            disabled: !__props.canNavigateForward,
            variant: __props.mainVariant === "pending" ? "disabled" : __props.mainVariant,
            onClick: __props.nextFn
          }, {
            default: withCtx(() => [
              __props.mainVariant === "pending" ? (openBlock(), createBlock(_component_i_cy_loading_x16, {
                key: 0,
                class: "animate-spin icon-dark-white icon-light-gray-400 mr-[8px]"
              })) : createCommentVNode("", true),
              createTextVNode(" " + toDisplayString$1(__props.next), 1)
            ]),
            _: 1
          }, 8, ["disabled", "variant", "onClick"])) : createCommentVNode("", true),
          __props.backFn ? (openBlock(), createBlock(unref(S$1), {
            key: 1,
            size: "40",
            variant: "outline-light",
            onClick: __props.backFn
          }, {
            default: withCtx(() => [
              createTextVNode(toDisplayString$1(__props.back), 1)
            ]),
            _: 1
          }, 8, ["onClick"])) : createCommentVNode("", true),
          _hoisted_2$1O,
          __props.altFn && __props.alt ? (openBlock(), createElementBlock("div", _hoisted_3$1x, [
            createBaseVNode("label", {
              id: "altFn",
              class: "px-3 text-gray-500",
              onClick: handleAlt
            }, toDisplayString$1(__props.alt), 1),
            createVNode(_sfc_main$1f, {
              size: "lg",
              "label-id": "altFn",
              value: altValue.value,
              onUpdate: handleAlt
            }, null, 8, ["value"])
          ])) : createCommentVNode("", true),
          __props.skipFn ? (openBlock(), createBlock(unref(S$1), {
            key: 3,
            size: "40",
            variant: "link",
            class: "text-gray-500",
            onClick: __props.skipFn
          }, {
            default: withCtx(() => [
              createTextVNode(toDisplayString$1(__props.skip) + " ", 1),
              createVNode(_component_i_cy_arrow_right_x16, { class: "inline-block h-[16px] w-[16px] icon-dark-gray-500" })
            ]),
            _: 1
          }, 8, ["onClick"])) : createCommentVNode("", true)
        ])
      ]);
    };
  }
});
const _hoisted_1$1_ = { class: "mx-auto my-[32px]" };
const _hoisted_2$1N = {
  key: 0,
  class: "w-full mb-[24px]"
};
const _hoisted_3$1w = { class: "w-full border border-gray-100 rounded flex flex-col" };
const _hoisted_4$W = { class: "grow" };
const _sfc_main$1d = /* @__PURE__ */ defineComponent({
  __name: "WizardLayout",
  props: {
    next: { default: void 0 },
    back: { default: void 0 },
    skip: { default: void 0 },
    alt: { default: void 0 },
    showNext: { type: Boolean, default: true },
    canNavigateForward: { type: Boolean, default: void 0 },
    altFn: { type: Function, default: void 0 },
    nextFn: { type: Function, default: void 0 },
    backFn: { type: Function, default: void 0 },
    skipFn: { type: Function, default: void 0 },
    mainButtonVariant: { default: "indigo-dark" }
  },
  setup(__props) {
    const props = __props;
    const { t: t2 } = useI18n();
    const nextLabel = computed(() => props.next || t2("setupPage.step.next"));
    const backLabel = computed(() => props.back || t2("setupPage.step.back"));
    const skipLabel = computed(() => props.skip || t2("setupPage.step.skip"));
    return (_ctx, _cache) => {
      return openBlock(), createElementBlock("div", _hoisted_1$1_, [
        _ctx.$slots.accessory ? (openBlock(), createElementBlock("div", _hoisted_2$1N, [
          renderSlot(_ctx.$slots, "accessory")
        ])) : createCommentVNode("", true),
        createBaseVNode("div", _hoisted_3$1w, [
          createBaseVNode("div", _hoisted_4$W, [
            renderSlot(_ctx.$slots, "default", { backFn: __props.backFn })
          ]),
          createVNode(_sfc_main$1e, {
            "next-fn": props.nextFn,
            "can-navigate-forward": __props.canNavigateForward,
            "back-fn": __props.backFn,
            "alt-fn": __props.altFn,
            next: unref(nextLabel),
            "show-next": __props.showNext,
            back: unref(backLabel),
            alt: __props.alt,
            "main-variant": __props.mainButtonVariant,
            skip: unref(skipLabel),
            "skip-fn": __props.skipFn
          }, {
            default: withCtx(() => [
              renderSlot(_ctx.$slots, "button-bar")
            ]),
            _: 3
          }, 8, ["next-fn", "can-navigate-forward", "back-fn", "alt-fn", "next", "show-next", "back", "alt", "main-variant", "skip", "skip-fn"])
        ])
      ]);
    };
  }
});
const _hoisted_1$1Z = {
  xmlns: "http://www.w3.org/2000/svg",
  "data-cy": "webpack-logo",
  viewBox: "0 0 774 875.7"
};
const _hoisted_2$1M = /* @__PURE__ */ createBaseVNode("path", {
  fill: "#FFF",
  d: "m387 0 387 218.9v437.9L387 875.7 0 656.8V218.9z"
}, null, -1);
const _hoisted_3$1v = /* @__PURE__ */ createBaseVNode("path", {
  fill: "#8ed6fb",
  d: "M704.9 641.7 399.8 814.3V679.9l190.1-104.6 115 66.4zm20.9-18.9V261.9l-111.6 64.5v232l111.6 64.4zM67.9 641.7 373 814.3V679.9L182.8 575.3 67.9 641.7zM47 622.8V261.9l111.6 64.5v232L47 622.8zm13.1-384.3L373 61.5v129.9L172.5 301.7l-1.6.9-110.8-64.1zm652.6 0-312.9-177v129.9l200.5 110.2 1.6.9 110.8-64z"
}, null, -1);
const _hoisted_4$V = /* @__PURE__ */ createBaseVNode("path", {
  fill: "#1c78c0",
  d: "M373 649.3 185.4 546.1V341.8L373 450.1v199.2zm26.8 0 187.6-103.1V341.8L399.8 450.1v199.2zm-13.4-207zM198.1 318.2l188.3-103.5 188.3 103.5-188.3 108.7-188.3-108.7z"
}, null, -1);
const _hoisted_5$y = [
  _hoisted_2$1M,
  _hoisted_3$1v,
  _hoisted_4$V
];
function render$13(_ctx, _cache) {
  return openBlock(), createElementBlock("svg", _hoisted_1$1Z, _hoisted_5$y);
}
const LogoWebpack = { render: render$13 };
const _hoisted_1$1Y = {
  xmlns: "http://www.w3.org/2000/svg",
  fill: "none",
  "data-cy": "vite-logo",
  viewBox: "0 0 410 404"
};
const _hoisted_2$1L = /* @__PURE__ */ createStaticVNode('<path fill="url(#a)" d="m399.641 59.525-183.998 329.02c-3.799 6.793-13.559 6.833-17.415.073L10.582 59.556C6.38 52.19 12.68 43.266 21.028 44.76l184.195 32.923c1.175.21 2.378.208 3.553-.006l180.343-32.87c8.32-1.517 14.649 7.337 10.522 14.719Z"></path><path fill="url(#b)" d="M292.965 1.574 156.801 28.255a5 5 0 0 0-4.03 4.611l-8.376 141.464c-.197 3.332 2.863 5.918 6.115 5.168l37.91-8.749c3.547-.818 6.752 2.306 6.023 5.873l-11.263 55.153c-.758 3.712 2.727 6.886 6.352 5.785l23.415-7.114c3.63-1.102 7.118 2.081 6.35 5.796l-17.899 86.633c-1.12 5.419 6.088 8.374 9.094 3.728l2.008-3.103 110.954-221.428c1.858-3.707-1.346-7.935-5.418-7.15l-39.022 7.532c-3.667.707-6.787-2.708-5.752-6.296l25.469-88.291c1.036-3.594-2.095-7.012-5.766-6.293Z"></path><defs><linearGradient id="a" x1="6" x2="235" y1="33" y2="344" gradientUnits="userSpaceOnUse"><stop stop-color="#41D1FF"></stop><stop offset="1" stop-color="#BD34FE"></stop></linearGradient><linearGradient id="b" x1="194.651" x2="236.076" y1="8.818" y2="292.989" gradientUnits="userSpaceOnUse"><stop stop-color="#FFEA83"></stop><stop offset=".083" stop-color="#FFDD35"></stop><stop offset="1" stop-color="#FFA800"></stop></linearGradient></defs>', 3);
const _hoisted_5$x = [
  _hoisted_2$1L
];
function render$12(_ctx, _cache) {
  return openBlock(), createElementBlock("svg", _hoisted_1$1Y, _hoisted_5$x);
}
const LogoVite = { render: render$12 };
const _hoisted_1$1X = {
  xmlns: "http://www.w3.org/2000/svg",
  "data-cy": "nextjs-logo",
  viewBox: "0 0 512 512"
};
const _hoisted_2$1K = /* @__PURE__ */ createBaseVNode("path", {
  d: "M386.399 35.508C217.06-64.061 1.885 57.55.012 253.882c-1.828 191.716 201.063 315.545 370.02 231.163L185.56 213.636v167.997c0 18.614-35.619 18.614-35.619 0V156.421c0-14.776 27.448-15.989 35.226-3.145L395.43 470.572c157.95-101.737 155.817-338.136-9.031-435.064zm-23.756 317.939L326.91 298.87V149.458c0-13.932 35.732-13.932 35.732 0v203.989z",
  class: "icon-dark"
}, null, -1);
const _hoisted_3$1u = [
  _hoisted_2$1K
];
function render$11(_ctx, _cache) {
  return openBlock(), createElementBlock("svg", _hoisted_1$1X, _hoisted_3$1u);
}
const LogoNext = { render: render$11 };
const _hoisted_1$1W = {
  xmlns: "http://www.w3.org/2000/svg",
  "data-cy": "vue-logo",
  viewBox: "0 0 261.76 226.69"
};
const _hoisted_2$1J = /* @__PURE__ */ createBaseVNode("path", {
  fill: "#41b883",
  d: "m161.096.001-30.224 52.35L100.647.002H-.005L130.872 226.69 261.749 0z"
}, null, -1);
const _hoisted_3$1t = /* @__PURE__ */ createBaseVNode("path", {
  fill: "#34495e",
  d: "m161.096.001-30.224 52.35L100.647.002H52.346l78.526 136.01L209.398.001z"
}, null, -1);
const _hoisted_4$U = [
  _hoisted_2$1J,
  _hoisted_3$1t
];
function render$10(_ctx, _cache) {
  return openBlock(), createElementBlock("svg", _hoisted_1$1W, _hoisted_4$U);
}
const LogoVue = { render: render$10 };
const _hoisted_1$1V = {
  xmlns: "http://www.w3.org/2000/svg",
  "data-cy": "react-logo",
  viewBox: "0 0 500 500"
};
const _hoisted_2$1I = /* @__PURE__ */ createBaseVNode("g", { fill: "#3077c6" }, [
  /* @__PURE__ */ createBaseVNode("path", { d: "M495.3 248.5c0-32.5-40.7-63.3-103.1-82.4 14.4-63.6 8-114.2-20.2-130.4-6.5-3.8-14.1-5.6-22.4-5.6v22.3c4.6 0 8.3.9 11.4 2.6 13.6 7.8 19.5 37.5 14.9 75.7-1.1 9.4-2.9 19.3-5.1 29.4-19.6-4.8-41-8.5-63.5-10.9-13.5-18.5-27.5-35.3-41.6-50 32.6-30.3 63.2-46.9 84-46.9V30c-27.5 0-63.5 19.6-99.9 53.6-36.4-33.8-72.4-53.2-99.9-53.2v22.3c20.7 0 51.4 16.5 84 46.6-14 14.7-28 31.4-41.3 49.9-22.6 2.4-44 6.1-63.6 11-2.3-10-4-19.7-5.2-29-4.7-38.2 1.1-67.9 14.6-75.8 3-1.8 6.9-2.6 11.5-2.6V30.5c-8.4 0-16 1.8-22.6 5.6-28.1 16.2-34.4 66.7-19.9 130.1C45.2 185.4 4.7 216.1 4.7 248.5c0 32.5 40.7 63.3 103.1 82.4-14.4 63.6-8 114.2 20.2 130.4 6.5 3.8 14.1 5.6 22.5 5.6 27.5 0 63.5-19.6 99.9-53.6 36.4 33.8 72.4 53.2 99.9 53.2 8.4 0 16-1.8 22.6-5.6 28.1-16.2 34.4-66.7 19.9-130.1 62-19.1 102.5-49.9 102.5-82.3Zm-130.2-66.7c-3.7 12.9-8.3 26.2-13.5 39.5-4.1-8-8.4-16-13.1-24-4.6-8-9.5-15.8-14.4-23.4 14.2 2.1 27.9 4.7 41 7.9Zm-45.8 106.5c-7.8 13.5-15.8 26.3-24.1 38.2-14.9 1.3-30 2-45.2 2-15.1 0-30.2-.7-45-1.9-8.3-11.9-16.4-24.6-24.2-38-7.6-13.1-14.5-26.4-20.8-39.8 6.2-13.4 13.2-26.8 20.7-39.9 7.8-13.5 15.8-26.3 24.1-38.2 14.9-1.3 30-2 45.2-2 15.1 0 30.2.7 45 1.9 8.3 11.9 16.4 24.6 24.2 38 7.6 13.1 14.5 26.4 20.8 39.8-6.3 13.4-13.2 26.8-20.7 39.9Zm32.3-13c5.4 13.4 10 26.8 13.8 39.8-13.1 3.2-26.9 5.9-41.2 8 4.9-7.7 9.8-15.6 14.4-23.7 4.6-8 8.9-16.1 13-24.1ZM250.2 382c-9.3-9.6-18.6-20.3-27.8-32 9 .4 18.2.7 27.5.7 9.4 0 18.7-.2 27.8-.7-9 11.7-18.3 22.4-27.5 32Zm-74.4-58.9c-14.2-2.1-27.9-4.7-41-7.9 3.7-12.9 8.3-26.2 13.5-39.5 4.1 8 8.4 16 13.1 24 4.7 8 9.5 15.8 14.4 23.4ZM249.7 115c9.3 9.6 18.6 20.3 27.8 32-9-.4-18.2-.7-27.5-.7-9.4 0-18.7.2-27.8.7 9-11.7 18.3-22.4 27.5-32Zm-74 58.9c-4.9 7.7-9.8 15.6-14.4 23.7-4.6 8-8.9 16-13 24-5.4-13.4-10-26.8-13.8-39.8 13.1-3.1 26.9-5.8 41.2-7.9ZM85.2 299.1c-35.4-15.1-58.3-34.9-58.3-50.6 0-15.7 22.9-35.6 58.3-50.6 8.6-3.7 18-7 27.7-10.1 5.7 19.6 13.2 40 22.5 60.9-9.2 20.8-16.6 41.1-22.2 60.6-9.9-3.1-19.3-6.5-28-10.2ZM139 442c-13.6-7.8-19.5-37.5-14.9-75.7 1.1-9.4 2.9-19.3 5.1-29.4 19.6 4.8 41 8.5 63.5 10.9 13.5 18.5 27.5 35.3 41.6 50-32.6 30.3-63.2 46.9-84 46.9-4.5-.1-8.3-1-11.3-2.7Zm237.2-76.2c4.7 38.2-1.1 67.9-14.6 75.8-3 1.8-6.9 2.6-11.5 2.6-20.7 0-51.4-16.5-84-46.6 14-14.7 28-31.4 41.3-49.9 22.6-2.4 44-6.1 63.6-11 2.3 10.1 4.1 19.8 5.2 29.1Zm38.5-66.7c-8.6 3.7-18 7-27.7 10.1-5.7-19.6-13.2-40-22.5-60.9 9.2-20.8 16.6-41.1 22.2-60.6 9.9 3.1 19.3 6.5 28.1 10.2 35.4 15.1 58.3 34.9 58.3 50.6-.1 15.7-23 35.6-58.4 50.6Z" }),
  /* @__PURE__ */ createBaseVNode("path", { d: "M295.6 248.5c0 25.239-20.461 45.7-45.7 45.7-25.239 0-45.7-20.461-45.7-45.7 0-25.239 20.461-45.7 45.7-45.7 25.239 0 45.7 20.461 45.7 45.7Z" })
], -1);
const _hoisted_3$1s = [
  _hoisted_2$1I
];
function render$$(_ctx, _cache) {
  return openBlock(), createElementBlock("svg", _hoisted_1$1V, _hoisted_3$1s);
}
const LogoReact = { render: render$$ };
const _hoisted_1$1U = {
  xmlns: "http://www.w3.org/2000/svg",
  "xml:space": "preserve",
  style: { "enable-background": "new 0 0 250 250" },
  viewBox: "0 0 250 250"
};
const _hoisted_2$1H = /* @__PURE__ */ createBaseVNode("path", {
  d: "M125 30 31.9 63.2l14.2 123.1L125 230l78.9-43.7 14.2-123.1z",
  style: { "fill": "#dd0031" }
}, null, -1);
const _hoisted_3$1r = /* @__PURE__ */ createBaseVNode("path", {
  d: "M125 30v22.2-.1V230l78.9-43.7 14.2-123.1L125 30z",
  style: { "fill": "#c3002f" }
}, null, -1);
const _hoisted_4$T = /* @__PURE__ */ createBaseVNode("path", {
  d: "M125 52.1 66.8 182.6h21.7l11.7-29.2h49.4l11.7 29.2H183L125 52.1zm17 83.3h-34l17-40.9 17 40.9z",
  style: { "fill": "#fff" }
}, null, -1);
const _hoisted_5$w = [
  _hoisted_2$1H,
  _hoisted_3$1r,
  _hoisted_4$T
];
function render$_(_ctx, _cache) {
  return openBlock(), createElementBlock("svg", _hoisted_1$1U, _hoisted_5$w);
}
const LogoAngular = { render: render$_ };
const _hoisted_1$1T = {
  xmlns: "http://www.w3.org/2000/svg",
  width: ".84em",
  height: "1em",
  "aria-hidden": "true",
  viewBox: "0 0 256 308"
};
const _hoisted_2$1G = /* @__PURE__ */ createBaseVNode("path", {
  fill: "#FF3E00",
  d: "M239.682 40.707C211.113-.182 154.69-12.301 113.895 13.69L42.247 59.356a82.198 82.198 0 0 0-37.135 55.056 86.566 86.566 0 0 0 8.536 55.576 82.425 82.425 0 0 0-12.296 30.719 87.596 87.596 0 0 0 14.964 66.244c28.574 40.893 84.997 53.007 125.787 27.016l71.648-45.664a82.182 82.182 0 0 0 37.135-55.057 86.601 86.601 0 0 0-8.53-55.577 82.409 82.409 0 0 0 12.29-30.718 87.573 87.573 0 0 0-14.963-66.244"
}, null, -1);
const _hoisted_3$1q = /* @__PURE__ */ createBaseVNode("path", {
  fill: "#FFF",
  d: "M106.889 270.841c-23.102 6.007-47.497-3.036-61.103-22.648a52.685 52.685 0 0 1-9.003-39.85 49.978 49.978 0 0 1 1.713-6.693l1.35-4.115 3.671 2.697a92.447 92.447 0 0 0 28.036 14.007l2.663.808-.245 2.659a16.067 16.067 0 0 0 2.89 10.656 17.143 17.143 0 0 0 18.397 6.828 15.786 15.786 0 0 0 4.403-1.935l71.67-45.672a14.922 14.922 0 0 0 6.734-9.977 15.923 15.923 0 0 0-2.713-12.011 17.156 17.156 0 0 0-18.404-6.832 15.78 15.78 0 0 0-4.396 1.933l-27.35 17.434a52.298 52.298 0 0 1-14.553 6.391c-23.101 6.007-47.497-3.036-61.101-22.649a52.681 52.681 0 0 1-9.004-39.849 49.428 49.428 0 0 1 22.34-33.114l71.664-45.677a52.218 52.218 0 0 1 14.563-6.398c23.101-6.007 47.497 3.036 61.101 22.648a52.685 52.685 0 0 1 9.004 39.85 50.559 50.559 0 0 1-1.713 6.692l-1.35 4.116-3.67-2.693a92.373 92.373 0 0 0-28.037-14.013l-2.664-.809.246-2.658a16.099 16.099 0 0 0-2.89-10.656 17.143 17.143 0 0 0-18.398-6.828 15.786 15.786 0 0 0-4.402 1.935l-71.67 45.674a14.898 14.898 0 0 0-6.73 9.975 15.9 15.9 0 0 0 2.709 12.012 17.156 17.156 0 0 0 18.404 6.832 15.841 15.841 0 0 0 4.402-1.935l27.345-17.427a52.147 52.147 0 0 1 14.552-6.397c23.101-6.006 47.497 3.037 61.102 22.65a52.681 52.681 0 0 1 9.003 39.848 49.453 49.453 0 0 1-22.34 33.12l-71.664 45.673a52.218 52.218 0 0 1-14.563 6.398"
}, null, -1);
const _hoisted_4$S = [
  _hoisted_2$1G,
  _hoisted_3$1q
];
function render$Z(_ctx, _cache) {
  return openBlock(), createElementBlock("svg", _hoisted_1$1T, _hoisted_4$S);
}
const LogoSvelte = { render: render$Z };
const FrameworkBundlerLogos = {
  webpack: LogoWebpack,
  vite: LogoVite,
  vue3: LogoVue,
  vueclivue3: LogoVue,
  nextjs: LogoNext,
  react: LogoReact,
  angular: LogoAngular,
  svelte: LogoSvelte
};
const _hoisted_1$1S = {
  viewBox: "0 0 24 24",
  width: "1em",
  height: "1em"
};
const _hoisted_2$1F = /* @__PURE__ */ createBaseVNode("path", {
  fill: "currentColor",
  d: "M21 7L9 19l-5.5-5.5l1.41-1.41L9 16.17L19.59 5.59z"
}, null, -1);
const _hoisted_3$1p = [
  _hoisted_2$1F
];
function render$Y(_ctx, _cache) {
  return openBlock(), createElementBlock("svg", _hoisted_1$1S, _hoisted_3$1p);
}
const __unplugin_components_1$6 = { name: "mdi-check", render: render$Y };
const _hoisted_1$1R = {
  width: "16",
  height: "16",
  viewBox: "0 0 16 16",
  fill: "none",
  xmlns: "http://www.w3.org/2000/svg"
};
const _hoisted_2$1E = /* @__PURE__ */ createBaseVNode("path", {
  d: "M2 5L8 11L14 5",
  stroke: "#1B1E2E",
  class: "icon-dark",
  "stroke-width": "2",
  "stroke-linecap": "round",
  "stroke-linejoin": "round"
}, null, -1);
const _hoisted_3$1o = [
  _hoisted_2$1E
];
function render$X(_ctx, _cache) {
  return openBlock(), createElementBlock("svg", _hoisted_1$1R, _hoisted_3$1o);
}
const __unplugin_components_0$j = { name: "cy-chevron-down", render: render$X };
function _extends$1() {
  _extends$1 = Object.assign || function(target2) {
    for (var i2 = 1; i2 < arguments.length; i2++) {
      var source = arguments[i2];
      for (var key in source) {
        if (Object.prototype.hasOwnProperty.call(source, key)) {
          target2[key] = source[key];
        }
      }
    }
    return target2;
  };
  return _extends$1.apply(this, arguments);
}
function _objectWithoutPropertiesLoose(source, excluded) {
  if (source == null) return {};
  var target2 = {};
  var sourceKeys = Object.keys(source);
  var key, i2;
  for (i2 = 0; i2 < sourceKeys.length; i2++) {
    key = sourceKeys[i2];
    if (excluded.indexOf(key) >= 0) continue;
    target2[key] = source[key];
  }
  return target2;
}
function _unsupportedIterableToArray(o2, minLen) {
  if (!o2) return;
  if (typeof o2 === "string") return _arrayLikeToArray(o2, minLen);
  var n2 = Object.prototype.toString.call(o2).slice(8, -1);
  if (n2 === "Object" && o2.constructor) n2 = o2.constructor.name;
  if (n2 === "Map" || n2 === "Set") return Array.from(o2);
  if (n2 === "Arguments" || /^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(n2)) return _arrayLikeToArray(o2, minLen);
}
function _arrayLikeToArray(arr, len) {
  if (len == null || len > arr.length) len = arr.length;
  for (var i2 = 0, arr2 = new Array(len); i2 < len; i2++) arr2[i2] = arr[i2];
  return arr2;
}
function _createForOfIteratorHelperLoose(o2, allowArrayLike) {
  var it2;
  if (typeof Symbol === "undefined" || o2[Symbol.iterator] == null) {
    if (Array.isArray(o2) || (it2 = _unsupportedIterableToArray(o2)) || allowArrayLike) {
      if (it2) o2 = it2;
      var i2 = 0;
      return function() {
        if (i2 >= o2.length) return {
          done: true
        };
        return {
          done: false,
          value: o2[i2++]
        };
      };
    }
    throw new TypeError("Invalid attempt to iterate non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method.");
  }
  it2 = o2[Symbol.iterator]();
  return it2.next.bind(it2);
}
function match(value, lookup) {
  if (value in lookup) {
    var returnValue = lookup[value];
    for (var _len = arguments.length, args = new Array(_len > 2 ? _len - 2 : 0), _key = 2; _key < _len; _key++) {
      args[_key - 2] = arguments[_key];
    }
    return typeof returnValue === "function" ? returnValue.apply(void 0, args) : returnValue;
  }
  var error2 = new Error('Tried to handle "' + value + '" but there is no handler defined. Only defined handlers are: ' + Object.keys(lookup).map(function(key) {
    return '"' + key + '"';
  }).join(", ") + ".");
  if (Error.captureStackTrace) Error.captureStackTrace(error2, match);
  throw error2;
}
var Features;
(function(Features2) {
  Features2[Features2["None"] = 0] = "None";
  Features2[Features2["RenderStrategy"] = 1] = "RenderStrategy";
  Features2[Features2["Static"] = 2] = "Static";
})(Features || (Features = {}));
var RenderStrategy;
(function(RenderStrategy2) {
  RenderStrategy2[RenderStrategy2["Unmount"] = 0] = "Unmount";
  RenderStrategy2[RenderStrategy2["Hidden"] = 1] = "Hidden";
})(RenderStrategy || (RenderStrategy = {}));
function render$W(_ref) {
  var _ref$visible = _ref.visible, visible = _ref$visible === void 0 ? true : _ref$visible, _ref$features = _ref.features, features = _ref$features === void 0 ? Features.None : _ref$features, main2 = _objectWithoutPropertiesLoose(_ref, ["visible", "features"]);
  if (visible) return _render(main2);
  if (features & Features.Static) {
    if (main2.props["static"]) return _render(main2);
  }
  if (features & Features.RenderStrategy) {
    var _main$props$unmount, _match;
    var strategy = ((_main$props$unmount = main2.props.unmount) != null ? _main$props$unmount : true) ? RenderStrategy.Unmount : RenderStrategy.Hidden;
    return match(strategy, (_match = {}, _match[RenderStrategy.Unmount] = function() {
      return null;
    }, _match[RenderStrategy.Hidden] = function() {
      return _render(_extends$1({}, main2, {
        props: _extends$1({}, main2.props, {
          hidden: true,
          style: {
            display: "none"
          }
        })
      }));
    }, _match));
  }
  return _render(main2);
}
function _render(_ref2) {
  var props = _ref2.props, attrs = _ref2.attrs, slots = _ref2.slots, slot = _ref2.slot, name2 = _ref2.name;
  var _omit = omit(props, ["unmount", "static"]), as = _omit.as, passThroughProps = _objectWithoutPropertiesLoose(_omit, ["as"]);
  var children = slots["default"] == null ? void 0 : slots["default"](slot);
  if (as === "template") {
    if (Object.keys(passThroughProps).length > 0 || Object.keys(attrs).length > 0) {
      var _ref3 = children != null ? children : [], firstChild = _ref3[0], other = _ref3.slice(1);
      if (!isValidElement(firstChild) || other.length > 0) {
        throw new Error(['Passing props on "template"!', "", "The current component <" + name2 + ' /> is rendering a "template".', "However we need to passthrough the following props:", Object.keys(passThroughProps).concat(Object.keys(attrs)).map(function(line) {
          return "  - " + line;
        }).join("\n"), "", "You can apply a few solutions:", ['Add an `as="..."` prop, to ensure that we render an actual element instead of a "template".', "Render a single element as the child so that we can forward the props onto that element."].map(function(line) {
          return "  - " + line;
        }).join("\n")].join("\n"));
      }
      return cloneVNode(firstChild, passThroughProps);
    }
    if (Array.isArray(children) && children.length === 1) {
      return children[0];
    }
    return children;
  }
  return h$2(as, passThroughProps, children);
}
function omit(object, keysToOmit) {
  if (keysToOmit === void 0) {
    keysToOmit = [];
  }
  var clone = Object.assign({}, object);
  for (var _iterator = _createForOfIteratorHelperLoose(keysToOmit), _step; !(_step = _iterator()).done; ) {
    var key = _step.value;
    if (key in clone) delete clone[key];
  }
  return clone;
}
function isValidElement(input) {
  if (input == null) return false;
  if (typeof input.type === "string") return true;
  if (typeof input.type === "object") return true;
  if (typeof input.type === "function") return true;
  return false;
}
var Keys;
(function(Keys2) {
  Keys2["Space"] = " ";
  Keys2["Enter"] = "Enter";
  Keys2["Escape"] = "Escape";
  Keys2["Backspace"] = "Backspace";
  Keys2["ArrowLeft"] = "ArrowLeft";
  Keys2["ArrowUp"] = "ArrowUp";
  Keys2["ArrowRight"] = "ArrowRight";
  Keys2["ArrowDown"] = "ArrowDown";
  Keys2["Home"] = "Home";
  Keys2["End"] = "End";
  Keys2["PageUp"] = "PageUp";
  Keys2["PageDown"] = "PageDown";
  Keys2["Tab"] = "Tab";
})(Keys || (Keys = {}));
var id = 0;
function generateId() {
  return ++id;
}
function useId() {
  return generateId();
}
var focusableSelector = /* @__PURE__ */ ["[contentEditable=true]", "[tabindex]", "a[href]", "area[href]", "button:not([disabled])", "iframe", "input:not([disabled])", "select:not([disabled])", "textarea:not([disabled])"].map(function(selector) {
  return selector + ":not([tabindex='-1'])";
}).join(",");
var Focus;
(function(Focus2) {
  Focus2[Focus2["First"] = 1] = "First";
  Focus2[Focus2["Previous"] = 2] = "Previous";
  Focus2[Focus2["Next"] = 4] = "Next";
  Focus2[Focus2["Last"] = 8] = "Last";
  Focus2[Focus2["WrapAround"] = 16] = "WrapAround";
  Focus2[Focus2["NoScroll"] = 32] = "NoScroll";
})(Focus || (Focus = {}));
var FocusResult;
(function(FocusResult2) {
  FocusResult2[FocusResult2["Error"] = 0] = "Error";
  FocusResult2[FocusResult2["Overflow"] = 1] = "Overflow";
  FocusResult2[FocusResult2["Success"] = 2] = "Success";
  FocusResult2[FocusResult2["Underflow"] = 3] = "Underflow";
})(FocusResult || (FocusResult = {}));
var Direction;
(function(Direction2) {
  Direction2[Direction2["Previous"] = -1] = "Previous";
  Direction2[Direction2["Next"] = 1] = "Next";
})(Direction || (Direction = {}));
function getFocusableElements(container) {
  if (container === void 0) {
    container = document.body;
  }
  if (container == null) return [];
  return Array.from(container.querySelectorAll(focusableSelector));
}
var FocusableMode;
(function(FocusableMode2) {
  FocusableMode2[FocusableMode2["Strict"] = 0] = "Strict";
  FocusableMode2[FocusableMode2["Loose"] = 1] = "Loose";
})(FocusableMode || (FocusableMode = {}));
function isFocusableElement(element, mode) {
  var _match;
  if (mode === void 0) {
    mode = FocusableMode.Strict;
  }
  if (element === document.body) return false;
  return match(mode, (_match = {}, _match[FocusableMode.Strict] = function() {
    return element.matches(focusableSelector);
  }, _match[FocusableMode.Loose] = function() {
    var next = element;
    while (next !== null) {
      if (next.matches(focusableSelector)) return true;
      next = next.parentElement;
    }
    return false;
  }, _match));
}
function focusElement(element) {
  element == null ? void 0 : element.focus({
    preventScroll: true
  });
}
function focusIn(container, focus) {
  var elements = Array.isArray(container) ? container : getFocusableElements(container);
  var active = document.activeElement;
  var direction = (function() {
    if (focus & (Focus.First | Focus.Next)) return Direction.Next;
    if (focus & (Focus.Previous | Focus.Last)) return Direction.Previous;
    throw new Error("Missing Focus.First, Focus.Previous, Focus.Next or Focus.Last");
  })();
  var startIndex = (function() {
    if (focus & Focus.First) return 0;
    if (focus & Focus.Previous) return Math.max(0, elements.indexOf(active)) - 1;
    if (focus & Focus.Next) return Math.max(0, elements.indexOf(active)) + 1;
    if (focus & Focus.Last) return elements.length - 1;
    throw new Error("Missing Focus.First, Focus.Previous, Focus.Next or Focus.Last");
  })();
  var focusOptions = focus & Focus.NoScroll ? {
    preventScroll: true
  } : {};
  var offset2 = 0;
  var total = elements.length;
  var next = void 0;
  do {
    var _next;
    if (offset2 >= total || offset2 + total <= 0) return FocusResult.Error;
    var nextIdx = startIndex + offset2;
    if (focus & Focus.WrapAround) {
      nextIdx = (nextIdx + total) % total;
    } else {
      if (nextIdx < 0) return FocusResult.Underflow;
      if (nextIdx >= total) return FocusResult.Overflow;
    }
    next = elements[nextIdx];
    (_next = next) == null ? void 0 : _next.focus(focusOptions);
    offset2 += direction;
  } while (next !== document.activeElement);
  if (!next.hasAttribute("tabindex")) next.setAttribute("tabindex", "0");
  return FocusResult.Success;
}
function useWindowEvent(type2, listener, options2) {
  window.addEventListener(type2, listener, options2);
  onUnmounted(function() {
    return window.removeEventListener(type2, listener, options2);
  });
}
function contains$1(containers, element) {
  for (var _iterator = _createForOfIteratorHelperLoose(containers), _step; !(_step = _iterator()).done; ) {
    var container = _step.value;
    if (container.contains(element)) return true;
  }
  return false;
}
function useFocusTrap(containers, enabled, options2) {
  if (enabled === void 0) {
    enabled = ref(true);
  }
  if (options2 === void 0) {
    options2 = ref({});
  }
  var restoreElement = ref(typeof window !== "undefined" ? document.activeElement : null);
  var previousActiveElement = ref(null);
  function handleFocus() {
    if (!enabled.value) return;
    if (containers.value.size !== 1) return;
    var initialFocus = options2.value.initialFocus;
    var activeElement = document.activeElement;
    if (initialFocus) {
      if (initialFocus === activeElement) {
        return;
      }
    } else if (contains$1(containers.value, activeElement)) {
      return;
    }
    restoreElement.value = activeElement;
    if (initialFocus) {
      focusElement(initialFocus);
    } else {
      var couldFocus = false;
      for (var _iterator = _createForOfIteratorHelperLoose(containers.value), _step; !(_step = _iterator()).done; ) {
        var container = _step.value;
        var result = focusIn(container, Focus.First);
        if (result === FocusResult.Success) {
          couldFocus = true;
          break;
        }
      }
      if (!couldFocus) throw new Error("There are no focusable elements inside the <FocusTrap />");
    }
    previousActiveElement.value = document.activeElement;
  }
  function restore2() {
    focusElement(restoreElement.value);
    restoreElement.value = null;
    previousActiveElement.value = null;
  }
  watchEffect(handleFocus);
  onUpdated(function() {
    enabled.value ? handleFocus() : restore2();
  });
  onUnmounted(restore2);
  useWindowEvent("keydown", function(event) {
    if (!enabled.value) return;
    if (event.key !== Keys.Tab) return;
    if (!document.activeElement) return;
    if (containers.value.size !== 1) return;
    event.preventDefault();
    for (var _iterator2 = _createForOfIteratorHelperLoose(containers.value), _step2; !(_step2 = _iterator2()).done; ) {
      var element = _step2.value;
      var result = focusIn(element, (event.shiftKey ? Focus.Previous : Focus.Next) | Focus.WrapAround);
      if (result === FocusResult.Success) {
        previousActiveElement.value = document.activeElement;
        break;
      }
    }
  });
  useWindowEvent("focus", function(event) {
    if (!enabled.value) return;
    if (containers.value.size !== 1) return;
    var previous = previousActiveElement.value;
    if (!previous) return;
    var toElement = event.target;
    if (toElement && toElement instanceof HTMLElement) {
      if (!contains$1(containers.value, toElement)) {
        event.preventDefault();
        event.stopPropagation();
        focusElement(previous);
      } else {
        previousActiveElement.value = toElement;
        focusElement(toElement);
      }
    } else {
      focusElement(previousActiveElement.value);
    }
  }, true);
}
var CHILDREN_SELECTOR = "body > *";
var interactables = /* @__PURE__ */ new Set();
var originals = /* @__PURE__ */ new Map();
function inert(element) {
  element.setAttribute("aria-hidden", "true");
  element.inert = true;
}
function restore(element) {
  var original = originals.get(element);
  if (!original) return;
  if (original["aria-hidden"] === null) element.removeAttribute("aria-hidden");
  else element.setAttribute("aria-hidden", original["aria-hidden"]);
  element.inert = original.inert;
}
function useInertOthers(container, enabled) {
  if (enabled === void 0) {
    enabled = ref(true);
  }
  watchEffect(function(onInvalidate) {
    if (!enabled.value) return;
    if (!container.value) return;
    var element = container.value;
    interactables.add(element);
    for (var _iterator = _createForOfIteratorHelperLoose(originals.keys()), _step; !(_step = _iterator()).done; ) {
      var original = _step.value;
      if (original.contains(element)) {
        restore(original);
        originals["delete"](original);
      }
    }
    document.querySelectorAll(CHILDREN_SELECTOR).forEach(function(child) {
      if (!(child instanceof HTMLElement)) return;
      for (var _iterator2 = _createForOfIteratorHelperLoose(interactables), _step2; !(_step2 = _iterator2()).done; ) {
        var interactable = _step2.value;
        if (child.contains(interactable)) return;
      }
      if (interactables.size === 1) {
        originals.set(child, {
          "aria-hidden": child.getAttribute("aria-hidden"),
          // @ts-expect-error `inert` does not exist on HTMLElement (yet!)
          inert: child.inert
        });
        inert(child);
      }
    });
    onInvalidate(function() {
      interactables["delete"](element);
      if (interactables.size > 0) {
        document.querySelectorAll(CHILDREN_SELECTOR).forEach(function(child) {
          if (!(child instanceof HTMLElement)) return;
          if (originals.has(child)) return;
          for (var _iterator3 = _createForOfIteratorHelperLoose(interactables), _step3; !(_step3 = _iterator3()).done; ) {
            var interactable = _step3.value;
            if (child.contains(interactable)) return;
          }
          originals.set(child, {
            "aria-hidden": child.getAttribute("aria-hidden"),
            // @ts-expect-error `inert` does not exist on HTMLElement (yet!)
            inert: child.inert
          });
          inert(child);
        });
      } else {
        for (var _iterator4 = _createForOfIteratorHelperLoose(originals.keys()), _step4; !(_step4 = _iterator4()).done; ) {
          var _element = _step4.value;
          restore(_element);
          originals["delete"](_element);
        }
      }
    });
  });
}
var StackContext = /* @__PURE__ */ Symbol("StackContext");
var StackMessage;
(function(StackMessage2) {
  StackMessage2[StackMessage2["AddElement"] = 0] = "AddElement";
  StackMessage2[StackMessage2["RemoveElement"] = 1] = "RemoveElement";
})(StackMessage || (StackMessage = {}));
function useStackContext() {
  return inject(StackContext, function() {
  });
}
function useElemenStack(element) {
  var notify = useStackContext();
  watchEffect(function(onInvalidate) {
    var domElement = element == null ? void 0 : element.value;
    if (!domElement) return;
    notify(StackMessage.AddElement, domElement);
    onInvalidate(function() {
      return notify(StackMessage.RemoveElement, domElement);
    });
  });
}
function useStackProvider(onUpdate) {
  var parentUpdate = useStackContext();
  function notify() {
    for (var _len = arguments.length, args = new Array(_len), _key = 0; _key < _len; _key++) {
      args[_key] = arguments[_key];
    }
    onUpdate == null ? void 0 : onUpdate.apply(void 0, args);
    parentUpdate.apply(void 0, args);
  }
  provide(StackContext, notify);
}
var ForcePortalRootContext = /* @__PURE__ */ Symbol("ForcePortalRootContext");
function usePortalRoot() {
  return inject(ForcePortalRootContext, false);
}
var ForcePortalRoot = /* @__PURE__ */ defineComponent({
  name: "ForcePortalRoot",
  props: {
    as: {
      type: [Object, String],
      "default": "template"
    },
    force: {
      type: Boolean,
      "default": false
    }
  },
  setup: function setup(props, _ref) {
    var slots = _ref.slots, attrs = _ref.attrs;
    provide(ForcePortalRootContext, props.force);
    return function() {
      var passThroughProps = _objectWithoutPropertiesLoose(props, ["force"]);
      return render$W({
        props: passThroughProps,
        slot: {},
        slots,
        attrs,
        name: "ForcePortalRoot"
      });
    };
  }
});
function getPortalRoot() {
  var existingRoot = document.getElementById("headlessui-portal-root");
  if (existingRoot) return existingRoot;
  var root2 = document.createElement("div");
  root2.setAttribute("id", "headlessui-portal-root");
  return document.body.appendChild(root2);
}
var Portal = /* @__PURE__ */ defineComponent({
  name: "Portal",
  props: {
    as: {
      type: [Object, String],
      "default": "div"
    }
  },
  setup: function setup2(props, _ref) {
    var slots = _ref.slots, attrs = _ref.attrs;
    var forcePortalRoot = usePortalRoot();
    var groupContext = inject(PortalGroupContext, null);
    var myTarget = ref(forcePortalRoot === true ? getPortalRoot() : groupContext === null ? getPortalRoot() : groupContext.resolveTarget());
    watchEffect(function() {
      if (forcePortalRoot) return;
      if (groupContext === null) return;
      myTarget.value = groupContext.resolveTarget();
    });
    var element = ref(null);
    useElemenStack(element);
    onUnmounted(function() {
      var root2 = document.getElementById("headlessui-portal-root");
      if (!root2) return;
      if (myTarget.value !== root2) return;
      if (myTarget.value.children.length <= 0) {
        var _myTarget$value$paren;
        (_myTarget$value$paren = myTarget.value.parentElement) == null ? void 0 : _myTarget$value$paren.removeChild(myTarget.value);
      }
    });
    useStackProvider();
    return function() {
      if (myTarget.value === null) return null;
      var propsWeControl = {
        ref: element
      };
      return h$2(
        // @ts-expect-error Children can be an object, but TypeScript is not happy
        // with it. Once this is fixed upstream we can remove this assertion.
        Teleport,
        {
          to: myTarget.value
        },
        render$W({
          props: _extends$1({}, props, propsWeControl),
          slot: {},
          attrs,
          slots,
          name: "Portal"
        })
      );
    };
  }
});
var PortalGroupContext = /* @__PURE__ */ Symbol("PortalGroupContext");
var PortalGroup = /* @__PURE__ */ defineComponent({
  name: "PortalGroup",
  props: {
    as: {
      type: [Object, String],
      "default": "template"
    },
    target: {
      type: Object,
      "default": null
    }
  },
  setup: function setup3(props, _ref2) {
    var attrs = _ref2.attrs, slots = _ref2.slots;
    var api = reactive({
      resolveTarget: function resolveTarget2() {
        return props.target;
      }
    });
    provide(PortalGroupContext, api);
    return function() {
      var passThroughProps = _objectWithoutPropertiesLoose(props, ["target"]);
      return render$W({
        props: passThroughProps,
        slot: {},
        attrs,
        slots,
        name: "PortalGroup"
      });
    };
  }
});
var DescriptionContext = /* @__PURE__ */ Symbol("DescriptionContext");
function useDescriptionContext() {
  var context = inject(DescriptionContext, null);
  if (context === null) {
    throw new Error("Missing parent");
  }
  return context;
}
function useDescriptions(_temp) {
  var _ref = _temp === void 0 ? {} : _temp, _ref$slot = _ref.slot, slot = _ref$slot === void 0 ? ref({}) : _ref$slot, _ref$name = _ref.name, name2 = _ref$name === void 0 ? "Description" : _ref$name, _ref$props = _ref.props, props = _ref$props === void 0 ? {} : _ref$props;
  var descriptionIds = ref([]);
  function register(value) {
    descriptionIds.value.push(value);
    return function() {
      var idx = descriptionIds.value.indexOf(value);
      if (idx === -1) return;
      descriptionIds.value.splice(idx, 1);
    };
  }
  provide(DescriptionContext, {
    register,
    slot,
    name: name2,
    props
  });
  return computed(function() {
    return descriptionIds.value.length > 0 ? descriptionIds.value.join(" ") : void 0;
  });
}
var Description = /* @__PURE__ */ defineComponent({
  name: "Description",
  props: {
    as: {
      type: [Object, String],
      "default": "p"
    }
  },
  render: function render$1() {
    var _this$context = this.context, _this$context$name = _this$context.name, name2 = _this$context$name === void 0 ? "Description" : _this$context$name, _this$context$slot = _this$context.slot, slot = _this$context$slot === void 0 ? ref({}) : _this$context$slot, _this$context$props = _this$context.props, props = _this$context$props === void 0 ? {} : _this$context$props;
    var passThroughProps = this.$props;
    var propsWeControl = _extends$1({}, Object.entries(props).reduce(function(acc, _ref2) {
      var _Object$assign;
      var key = _ref2[0], value = _ref2[1];
      return Object.assign(acc, (_Object$assign = {}, _Object$assign[key] = unref(value), _Object$assign));
    }, {}), {
      id: this.id
    });
    return render$W({
      props: _extends$1({}, passThroughProps, propsWeControl),
      slot: slot.value,
      attrs: this.$attrs,
      slots: this.$slots,
      name: name2
    });
  },
  setup: function setup4() {
    var context = useDescriptionContext();
    var id2 = "headlessui-description-" + useId();
    onMounted(function() {
      return onUnmounted(context.register(id2));
    });
    return {
      id: id2,
      context
    };
  }
});
function dom(ref2) {
  var _ref$value$$el;
  if (ref2 == null) return null;
  if (ref2.value == null) return null;
  return (_ref$value$$el = ref2.value.$el) != null ? _ref$value$$el : ref2.value;
}
var Context = /* @__PURE__ */ Symbol("Context");
var State;
(function(State2) {
  State2[State2["Open"] = 0] = "Open";
  State2[State2["Closed"] = 1] = "Closed";
})(State || (State = {}));
function useOpenClosed() {
  return inject(Context, null);
}
function useOpenClosedProvider(value) {
  provide(Context, value);
}
var DialogStates;
(function(DialogStates2) {
  DialogStates2[DialogStates2["Open"] = 0] = "Open";
  DialogStates2[DialogStates2["Closed"] = 1] = "Closed";
})(DialogStates || (DialogStates = {}));
var DialogContext = /* @__PURE__ */ Symbol("DialogContext");
function useDialogContext(component) {
  var context = inject(DialogContext, null);
  if (context === null) {
    var err = new Error("<" + component + " /> is missing a parent <Dialog /> component.");
    if (Error.captureStackTrace) Error.captureStackTrace(err, useDialogContext);
    throw err;
  }
  return context;
}
var Missing = "DC8F892D-2EBD-447C-A4C8-A03058436FF4";
var Dialog = /* @__PURE__ */ defineComponent({
  name: "Dialog",
  inheritAttrs: false,
  props: {
    as: {
      type: [Object, String],
      "default": "div"
    },
    "static": {
      type: Boolean,
      "default": false
    },
    unmount: {
      type: Boolean,
      "default": true
    },
    open: {
      type: [Boolean, String],
      "default": Missing
    },
    initialFocus: {
      type: Object,
      "default": null
    }
  },
  emits: ["close"],
  render: function render$122() {
    var _this = this;
    var propsWeControl = _extends$1({}, this.$attrs, {
      ref: "el",
      id: this.id,
      role: "dialog",
      "aria-modal": this.dialogState === DialogStates.Open ? true : void 0,
      "aria-labelledby": this.titleId,
      "aria-describedby": this.describedby,
      onClick: this.handleClick,
      onKeydown: this.handleKeyDown
    });
    var _this$$props = this.$props, passThroughProps = _objectWithoutPropertiesLoose(_this$$props, ["open", "initialFocus"]);
    var slot = {
      open: this.dialogState === DialogStates.Open
    };
    return h$2(ForcePortalRoot, {
      force: true
    }, function() {
      return h$2(Portal, function() {
        return h$2(PortalGroup, {
          target: _this.dialogRef
        }, function() {
          return h$2(ForcePortalRoot, {
            force: false
          }, function() {
            return render$W({
              props: _extends$1({}, passThroughProps, propsWeControl),
              slot,
              attrs: _this.$attrs,
              slots: _this.$slots,
              visible: _this.visible,
              features: Features.RenderStrategy | Features.Static,
              name: "Dialog"
            });
          });
        });
      });
    });
  },
  setup: function setup5(props, _ref) {
    var emit2 = _ref.emit;
    var containers = ref(/* @__PURE__ */ new Set());
    var usesOpenClosedState = useOpenClosed();
    var open = computed(function() {
      if (props.open === Missing && usesOpenClosedState !== null) {
        var _match;
        return match(usesOpenClosedState.value, (_match = {}, _match[State.Open] = true, _match[State.Closed] = false, _match));
      }
      return props.open;
    });
    var hasOpen = props.open !== Missing || usesOpenClosedState !== null;
    if (!hasOpen) {
      throw new Error("You forgot to provide an `open` prop to the `Dialog`.");
    }
    if (typeof open.value !== "boolean") {
      throw new Error("You provided an `open` prop to the `Dialog`, but the value is not a boolean. Received: " + (open.value === Missing ? void 0 : props.open));
    }
    var dialogState = computed(function() {
      return props.open ? DialogStates.Open : DialogStates.Closed;
    });
    var visible = computed(function() {
      if (usesOpenClosedState !== null) {
        return usesOpenClosedState.value === State.Open;
      }
      return dialogState.value === DialogStates.Open;
    });
    var internalDialogRef = ref(null);
    var enabled = ref(dialogState.value === DialogStates.Open);
    onUpdated(function() {
      enabled.value = dialogState.value === DialogStates.Open;
    });
    var id2 = "headlessui-dialog-" + useId();
    var focusTrapOptions = computed(function() {
      return {
        initialFocus: props.initialFocus
      };
    });
    useFocusTrap(containers, enabled, focusTrapOptions);
    useInertOthers(internalDialogRef, enabled);
    useStackProvider(function(message, element) {
      var _match2;
      return match(message, (_match2 = {}, _match2[StackMessage.AddElement] = function() {
        containers.value.add(element);
      }, _match2[StackMessage.RemoveElement] = function() {
        containers.value["delete"](element);
      }, _match2));
    });
    var describedby = useDescriptions({
      name: "DialogDescription",
      slot: computed(function() {
        return {
          open: open.value
        };
      })
    });
    var titleId = ref(null);
    var api = {
      titleId,
      dialogState,
      setTitleId: function setTitleId(id3) {
        if (titleId.value === id3) return;
        titleId.value = id3;
      },
      close: function close() {
        emit2("close", false);
      }
    };
    provide(DialogContext, api);
    useWindowEvent("mousedown", function(event) {
      var target2 = event.target;
      if (dialogState.value !== DialogStates.Open) return;
      if (containers.value.size !== 1) return;
      if (contains$1(containers.value, target2)) return;
      api.close();
      nextTick(function() {
        return target2 == null ? void 0 : target2.focus();
      });
    });
    watchEffect(function(onInvalidate) {
      if (dialogState.value !== DialogStates.Open) return;
      var overflow = document.documentElement.style.overflow;
      var paddingRight = document.documentElement.style.paddingRight;
      var scrollbarWidth = window.innerWidth - document.documentElement.clientWidth;
      document.documentElement.style.overflow = "hidden";
      document.documentElement.style.paddingRight = scrollbarWidth + "px";
      onInvalidate(function() {
        document.documentElement.style.overflow = overflow;
        document.documentElement.style.paddingRight = paddingRight;
      });
    });
    watchEffect(function(onInvalidate) {
      if (dialogState.value !== DialogStates.Open) return;
      var container = dom(internalDialogRef);
      if (!container) return;
      var observer = new IntersectionObserver(function(entries) {
        for (var _iterator = _createForOfIteratorHelperLoose(entries), _step; !(_step = _iterator()).done; ) {
          var entry = _step.value;
          if (entry.boundingClientRect.x === 0 && entry.boundingClientRect.y === 0 && entry.boundingClientRect.width === 0 && entry.boundingClientRect.height === 0) {
            api.close();
          }
        }
      });
      observer.observe(container);
      onInvalidate(function() {
        return observer.disconnect();
      });
    });
    return {
      id: id2,
      el: internalDialogRef,
      dialogRef: internalDialogRef,
      containers,
      dialogState,
      titleId,
      describedby,
      visible,
      open,
      handleClick: function handleClick(event) {
        event.stopPropagation();
      },
      // Handle `Escape` to close
      handleKeyDown: function handleKeyDown(event) {
        if (event.key !== Keys.Escape) return;
        if (dialogState.value !== DialogStates.Open) return;
        if (containers.value.size > 1) return;
        event.preventDefault();
        event.stopPropagation();
        api.close();
      }
    };
  }
});
var DialogOverlay = /* @__PURE__ */ defineComponent({
  name: "DialogOverlay",
  props: {
    as: {
      type: [Object, String],
      "default": "div"
    }
  },
  render: function render$132() {
    var api = useDialogContext("DialogOverlay");
    var propsWeControl = {
      ref: "el",
      id: this.id,
      "aria-hidden": true,
      onClick: this.handleClick
    };
    var passThroughProps = this.$props;
    return render$W({
      props: _extends$1({}, passThroughProps, propsWeControl),
      slot: {
        open: api.dialogState.value === DialogStates.Open
      },
      attrs: this.$attrs,
      slots: this.$slots,
      name: "DialogOverlay"
    });
  },
  setup: function setup6() {
    var api = useDialogContext("DialogOverlay");
    var id2 = "headlessui-dialog-overlay-" + useId();
    return {
      id: id2,
      handleClick: function handleClick(event) {
        event.preventDefault();
        event.stopPropagation();
        api.close();
      }
    };
  }
});
var DialogTitle = /* @__PURE__ */ defineComponent({
  name: "DialogTitle",
  props: {
    as: {
      type: [Object, String],
      "default": "h2"
    }
  },
  render: function render$142() {
    var api = useDialogContext("DialogTitle");
    var propsWeControl = {
      id: this.id
    };
    var passThroughProps = this.$props;
    return render$W({
      props: _extends$1({}, passThroughProps, propsWeControl),
      slot: {
        open: api.dialogState.value === DialogStates.Open
      },
      attrs: this.$attrs,
      slots: this.$slots,
      name: "DialogTitle"
    });
  },
  setup: function setup7() {
    var api = useDialogContext("DialogTitle");
    var id2 = "headlessui-dialog-title-" + useId();
    onMounted(function() {
      api.setTitleId(id2);
      onUnmounted(function() {
        return api.setTitleId(null);
      });
    });
    return {
      id: id2
    };
  }
});
var DialogDescription = Description;
var DisclosureStates;
(function(DisclosureStates2) {
  DisclosureStates2[DisclosureStates2["Open"] = 0] = "Open";
  DisclosureStates2[DisclosureStates2["Closed"] = 1] = "Closed";
})(DisclosureStates || (DisclosureStates = {}));
function assertNever(x2) {
  throw new Error("Unexpected object: " + x2);
}
var Focus$1;
(function(Focus2) {
  Focus2[Focus2["First"] = 0] = "First";
  Focus2[Focus2["Previous"] = 1] = "Previous";
  Focus2[Focus2["Next"] = 2] = "Next";
  Focus2[Focus2["Last"] = 3] = "Last";
  Focus2[Focus2["Specific"] = 4] = "Specific";
  Focus2[Focus2["Nothing"] = 5] = "Nothing";
})(Focus$1 || (Focus$1 = {}));
function calculateActiveIndex(action, resolvers) {
  var items = resolvers.resolveItems();
  if (items.length <= 0) return null;
  var currentActiveIndex = resolvers.resolveActiveIndex();
  var activeIndex = currentActiveIndex != null ? currentActiveIndex : -1;
  var nextActiveIndex = (function() {
    switch (action.focus) {
      case Focus$1.First:
        return items.findIndex(function(item) {
          return !resolvers.resolveDisabled(item);
        });
      case Focus$1.Previous: {
        var idx = items.slice().reverse().findIndex(function(item, idx2, all) {
          if (activeIndex !== -1 && all.length - idx2 - 1 >= activeIndex) return false;
          return !resolvers.resolveDisabled(item);
        });
        if (idx === -1) return idx;
        return items.length - 1 - idx;
      }
      case Focus$1.Next:
        return items.findIndex(function(item, idx2) {
          if (idx2 <= activeIndex) return false;
          return !resolvers.resolveDisabled(item);
        });
      case Focus$1.Last: {
        var _idx = items.slice().reverse().findIndex(function(item) {
          return !resolvers.resolveDisabled(item);
        });
        if (_idx === -1) return _idx;
        return items.length - 1 - _idx;
      }
      case Focus$1.Specific:
        return items.findIndex(function(item) {
          return resolvers.resolveId(item) === action.id;
        });
      case Focus$1.Nothing:
        return null;
      default:
        assertNever(action);
    }
  })();
  return nextActiveIndex === -1 ? currentActiveIndex : nextActiveIndex;
}
var ListboxStates;
(function(ListboxStates2) {
  ListboxStates2[ListboxStates2["Open"] = 0] = "Open";
  ListboxStates2[ListboxStates2["Closed"] = 1] = "Closed";
})(ListboxStates || (ListboxStates = {}));
function nextFrame$1(cb) {
  requestAnimationFrame(function() {
    return requestAnimationFrame(cb);
  });
}
var ListboxContext = /* @__PURE__ */ Symbol("ListboxContext");
function useListboxContext(component) {
  var context = inject(ListboxContext, null);
  if (context === null) {
    var err = new Error("<" + component + " /> is missing a parent <Listbox /> component.");
    if (Error.captureStackTrace) Error.captureStackTrace(err, useListboxContext);
    throw err;
  }
  return context;
}
var Listbox = /* @__PURE__ */ defineComponent({
  name: "Listbox",
  emits: ["update:modelValue"],
  props: {
    as: {
      type: [Object, String],
      "default": "template"
    },
    disabled: {
      type: [Boolean],
      "default": false
    },
    horizontal: {
      type: [Boolean],
      "default": false
    },
    modelValue: {
      type: [Object, String, Number, Boolean]
    }
  },
  setup: function setup12(props, _ref) {
    var slots = _ref.slots, attrs = _ref.attrs, emit2 = _ref.emit;
    var listboxState = ref(ListboxStates.Closed);
    var labelRef = ref(null);
    var buttonRef = ref(null);
    var optionsRef = ref(null);
    var options2 = ref([]);
    var searchQuery = ref("");
    var activeOptionIndex = ref(null);
    var value = computed(function() {
      return props.modelValue;
    });
    var api = {
      listboxState,
      value,
      orientation: computed(function() {
        return props.horizontal ? "horizontal" : "vertical";
      }),
      labelRef,
      buttonRef,
      optionsRef,
      disabled: computed(function() {
        return props.disabled;
      }),
      options: options2,
      searchQuery,
      activeOptionIndex,
      closeListbox: function closeListbox() {
        if (props.disabled) return;
        if (listboxState.value === ListboxStates.Closed) return;
        listboxState.value = ListboxStates.Closed;
        activeOptionIndex.value = null;
      },
      openListbox: function openListbox() {
        if (props.disabled) return;
        if (listboxState.value === ListboxStates.Open) return;
        listboxState.value = ListboxStates.Open;
      },
      goToOption: function goToOption(focus, id2) {
        if (props.disabled) return;
        if (listboxState.value === ListboxStates.Closed) return;
        var nextActiveOptionIndex = calculateActiveIndex(focus === Focus$1.Specific ? {
          focus: Focus$1.Specific,
          id: id2
        } : {
          focus
        }, {
          resolveItems: function resolveItems2() {
            return options2.value;
          },
          resolveActiveIndex: function resolveActiveIndex() {
            return activeOptionIndex.value;
          },
          resolveId: function resolveId(option) {
            return option.id;
          },
          resolveDisabled: function resolveDisabled(option) {
            return option.dataRef.disabled;
          }
        });
        if (searchQuery.value === "" && activeOptionIndex.value === nextActiveOptionIndex) return;
        searchQuery.value = "";
        activeOptionIndex.value = nextActiveOptionIndex;
      },
      search: function search(value2) {
        if (props.disabled) return;
        if (listboxState.value === ListboxStates.Closed) return;
        searchQuery.value += value2.toLowerCase();
        var match2 = options2.value.findIndex(function(option) {
          return !option.dataRef.disabled && option.dataRef.textValue.startsWith(searchQuery.value);
        });
        if (match2 === -1 || match2 === activeOptionIndex.value) return;
        activeOptionIndex.value = match2;
      },
      clearSearch: function clearSearch() {
        if (props.disabled) return;
        if (listboxState.value === ListboxStates.Closed) return;
        if (searchQuery.value === "") return;
        searchQuery.value = "";
      },
      registerOption: function registerOption(id2, dataRef) {
        options2.value.push({
          id: id2,
          dataRef
        });
      },
      unregisterOption: function unregisterOption(id2) {
        var nextOptions = options2.value.slice();
        var currentActiveOption = activeOptionIndex.value !== null ? nextOptions[activeOptionIndex.value] : null;
        var idx = nextOptions.findIndex(function(a2) {
          return a2.id === id2;
        });
        if (idx !== -1) nextOptions.splice(idx, 1);
        options2.value = nextOptions;
        activeOptionIndex.value = (function() {
          if (idx === activeOptionIndex.value) return null;
          if (currentActiveOption === null) return null;
          return nextOptions.indexOf(currentActiveOption);
        })();
      },
      select: function select(value2) {
        if (props.disabled) return;
        emit2("update:modelValue", value2);
      }
    };
    useWindowEvent("mousedown", function(event) {
      var _dom, _dom2, _dom3;
      var target2 = event.target;
      var active = document.activeElement;
      if (listboxState.value !== ListboxStates.Open) return;
      if ((_dom = dom(buttonRef)) == null ? void 0 : _dom.contains(target2)) return;
      if (!((_dom2 = dom(optionsRef)) == null ? void 0 : _dom2.contains(target2))) api.closeListbox();
      if (active !== document.body && (active == null ? void 0 : active.contains(target2))) return;
      if (!event.defaultPrevented) (_dom3 = dom(buttonRef)) == null ? void 0 : _dom3.focus({
        preventScroll: true
      });
    });
    provide(ListboxContext, api);
    useOpenClosedProvider(computed(function() {
      var _match;
      return match(listboxState.value, (_match = {}, _match[ListboxStates.Open] = State.Open, _match[ListboxStates.Closed] = State.Closed, _match));
    }));
    return function() {
      var slot = {
        open: listboxState.value === ListboxStates.Open,
        disabled: props.disabled
      };
      return render$W({
        props: omit(props, ["modelValue", "onUpdate:modelValue", "disabled", "horizontal"]),
        slot,
        slots,
        attrs,
        name: "Listbox"
      });
    };
  }
});
var ListboxLabel = /* @__PURE__ */ defineComponent({
  name: "ListboxLabel",
  props: {
    as: {
      type: [Object, String],
      "default": "label"
    }
  },
  render: function render$182() {
    var api = useListboxContext("ListboxLabel");
    var slot = {
      open: api.listboxState.value === ListboxStates.Open,
      disabled: api.disabled.value
    };
    var propsWeControl = {
      id: this.id,
      ref: "el",
      onClick: this.handleClick
    };
    return render$W({
      props: _extends$1({}, this.$props, propsWeControl),
      slot,
      attrs: this.$attrs,
      slots: this.$slots,
      name: "ListboxLabel"
    });
  },
  setup: function setup13() {
    var api = useListboxContext("ListboxLabel");
    var id2 = "headlessui-listbox-label-" + useId();
    return {
      id: id2,
      el: api.labelRef,
      handleClick: function handleClick() {
        var _dom4;
        (_dom4 = dom(api.buttonRef)) == null ? void 0 : _dom4.focus({
          preventScroll: true
        });
      }
    };
  }
});
var ListboxButton = /* @__PURE__ */ defineComponent({
  name: "ListboxButton",
  props: {
    as: {
      type: [Object, String],
      "default": "button"
    }
  },
  render: function render$192() {
    var _dom5, _dom6;
    var api = useListboxContext("ListboxButton");
    var slot = {
      open: api.listboxState.value === ListboxStates.Open,
      disabled: api.disabled.value
    };
    var propsWeControl = {
      ref: "el",
      id: this.id,
      type: "button",
      "aria-haspopup": true,
      "aria-controls": (_dom5 = dom(api.optionsRef)) == null ? void 0 : _dom5.id,
      "aria-expanded": api.disabled.value ? void 0 : api.listboxState.value === ListboxStates.Open,
      "aria-labelledby": api.labelRef.value ? [(_dom6 = dom(api.labelRef)) == null ? void 0 : _dom6.id, this.id].join(" ") : void 0,
      disabled: api.disabled.value === true ? true : void 0,
      onKeydown: this.handleKeyDown,
      onKeyup: this.handleKeyUp,
      onClick: this.handleClick
    };
    return render$W({
      props: _extends$1({}, this.$props, propsWeControl),
      slot,
      attrs: this.$attrs,
      slots: this.$slots,
      name: "ListboxButton"
    });
  },
  setup: function setup14() {
    var api = useListboxContext("ListboxButton");
    var id2 = "headlessui-listbox-button-" + useId();
    function handleKeyDown(event) {
      switch (event.key) {
        // Ref: https://www.w3.org/TR/wai-aria-practices-1.2/#keyboard-interaction-13
        case Keys.Space:
        case Keys.Enter:
        case Keys.ArrowDown:
          event.preventDefault();
          api.openListbox();
          nextTick(function() {
            var _dom7;
            (_dom7 = dom(api.optionsRef)) == null ? void 0 : _dom7.focus({
              preventScroll: true
            });
            if (!api.value.value) api.goToOption(Focus$1.First);
          });
          break;
        case Keys.ArrowUp:
          event.preventDefault();
          api.openListbox();
          nextTick(function() {
            var _dom8;
            (_dom8 = dom(api.optionsRef)) == null ? void 0 : _dom8.focus({
              preventScroll: true
            });
            if (!api.value.value) api.goToOption(Focus$1.Last);
          });
          break;
      }
    }
    function handleKeyUp(event) {
      switch (event.key) {
        case Keys.Space:
          event.preventDefault();
          break;
      }
    }
    function handleClick(event) {
      if (api.disabled.value) return;
      if (api.listboxState.value === ListboxStates.Open) {
        api.closeListbox();
        nextTick(function() {
          var _dom9;
          return (_dom9 = dom(api.buttonRef)) == null ? void 0 : _dom9.focus({
            preventScroll: true
          });
        });
      } else {
        event.preventDefault();
        api.openListbox();
        nextFrame$1(function() {
          var _dom10;
          return (_dom10 = dom(api.optionsRef)) == null ? void 0 : _dom10.focus({
            preventScroll: true
          });
        });
      }
    }
    return {
      id: id2,
      el: api.buttonRef,
      handleKeyDown,
      handleKeyUp,
      handleClick
    };
  }
});
var ListboxOptions = /* @__PURE__ */ defineComponent({
  name: "ListboxOptions",
  props: {
    as: {
      type: [Object, String],
      "default": "ul"
    },
    "static": {
      type: Boolean,
      "default": false
    },
    unmount: {
      type: Boolean,
      "default": true
    }
  },
  render: function render$110() {
    var _api$options$value$ap, _dom$id, _dom11, _dom12;
    var api = useListboxContext("ListboxOptions");
    var slot = {
      open: api.listboxState.value === ListboxStates.Open
    };
    var propsWeControl = {
      "aria-activedescendant": api.activeOptionIndex.value === null ? void 0 : (_api$options$value$ap = api.options.value[api.activeOptionIndex.value]) == null ? void 0 : _api$options$value$ap.id,
      "aria-labelledby": (_dom$id = (_dom11 = dom(api.labelRef)) == null ? void 0 : _dom11.id) != null ? _dom$id : (_dom12 = dom(api.buttonRef)) == null ? void 0 : _dom12.id,
      "aria-orientation": api.orientation.value,
      id: this.id,
      onKeydown: this.handleKeyDown,
      role: "listbox",
      tabIndex: 0,
      ref: "el"
    };
    var passThroughProps = this.$props;
    return render$W({
      props: _extends$1({}, passThroughProps, propsWeControl),
      slot,
      attrs: this.$attrs,
      slots: this.$slots,
      features: Features.RenderStrategy | Features.Static,
      visible: this.visible,
      name: "ListboxOptions"
    });
  },
  setup: function setup15() {
    var api = useListboxContext("ListboxOptions");
    var id2 = "headlessui-listbox-options-" + useId();
    var searchDebounce = ref(null);
    function handleKeyDown(event) {
      if (searchDebounce.value) clearTimeout(searchDebounce.value);
      switch (event.key) {
        // Ref: https://www.w3.org/TR/wai-aria-practices-1.2/#keyboard-interaction-12
        // @ts-expect-error Fallthrough is expected here
        case Keys.Space:
          if (api.searchQuery.value !== "") {
            event.preventDefault();
            event.stopPropagation();
            return api.search(event.key);
          }
        // When in type ahead mode, fallthrough
        case Keys.Enter:
          event.preventDefault();
          event.stopPropagation();
          if (api.activeOptionIndex.value !== null) {
            var dataRef = api.options.value[api.activeOptionIndex.value].dataRef;
            api.select(dataRef.value);
          }
          api.closeListbox();
          nextTick(function() {
            var _dom13;
            return (_dom13 = dom(api.buttonRef)) == null ? void 0 : _dom13.focus({
              preventScroll: true
            });
          });
          break;
        case match(api.orientation.value, {
          vertical: Keys.ArrowDown,
          horizontal: Keys.ArrowRight
        }):
          event.preventDefault();
          event.stopPropagation();
          return api.goToOption(Focus$1.Next);
        case match(api.orientation.value, {
          vertical: Keys.ArrowUp,
          horizontal: Keys.ArrowLeft
        }):
          event.preventDefault();
          event.stopPropagation();
          return api.goToOption(Focus$1.Previous);
        case Keys.Home:
        case Keys.PageUp:
          event.preventDefault();
          event.stopPropagation();
          return api.goToOption(Focus$1.First);
        case Keys.End:
        case Keys.PageDown:
          event.preventDefault();
          event.stopPropagation();
          return api.goToOption(Focus$1.Last);
        case Keys.Escape:
          event.preventDefault();
          event.stopPropagation();
          api.closeListbox();
          nextTick(function() {
            var _dom14;
            return (_dom14 = dom(api.buttonRef)) == null ? void 0 : _dom14.focus({
              preventScroll: true
            });
          });
          break;
        case Keys.Tab:
          event.preventDefault();
          event.stopPropagation();
          break;
        default:
          if (event.key.length === 1) {
            api.search(event.key);
            searchDebounce.value = setTimeout(function() {
              return api.clearSearch();
            }, 350);
          }
          break;
      }
    }
    var usesOpenClosedState = useOpenClosed();
    var visible = computed(function() {
      if (usesOpenClosedState !== null) {
        return usesOpenClosedState.value === State.Open;
      }
      return api.listboxState.value === ListboxStates.Open;
    });
    return {
      id: id2,
      el: api.optionsRef,
      handleKeyDown,
      visible
    };
  }
});
var ListboxOption = /* @__PURE__ */ defineComponent({
  name: "ListboxOption",
  props: {
    as: {
      type: [Object, String],
      "default": "li"
    },
    value: {
      type: [Object, String, Number, Boolean]
    },
    disabled: {
      type: Boolean,
      "default": false
    }
  },
  setup: function setup16(props, _ref2) {
    var slots = _ref2.slots, attrs = _ref2.attrs;
    var api = useListboxContext("ListboxOption");
    var id2 = "headlessui-listbox-option-" + useId();
    var active = computed(function() {
      return api.activeOptionIndex.value !== null ? api.options.value[api.activeOptionIndex.value].id === id2 : false;
    });
    var selected = computed(function() {
      return toRaw(api.value.value) === toRaw(props.value);
    });
    var dataRef = ref({
      disabled: props.disabled,
      value: props.value,
      textValue: ""
    });
    onMounted(function() {
      var _document$getElementB, _document$getElementB2;
      var textValue = (_document$getElementB = document.getElementById(id2)) == null ? void 0 : (_document$getElementB2 = _document$getElementB.textContent) == null ? void 0 : _document$getElementB2.toLowerCase().trim();
      if (textValue !== void 0) dataRef.value.textValue = textValue;
    });
    onMounted(function() {
      return api.registerOption(id2, dataRef);
    });
    onUnmounted(function() {
      return api.unregisterOption(id2);
    });
    onMounted(function() {
      watch([api.listboxState, selected], function() {
        var _document$getElementB3;
        if (api.listboxState.value !== ListboxStates.Open) return;
        if (!selected.value) return;
        api.goToOption(Focus$1.Specific, id2);
        (_document$getElementB3 = document.getElementById(id2)) == null ? void 0 : _document$getElementB3.focus == null ? void 0 : _document$getElementB3.focus();
      }, {
        immediate: true
      });
    });
    watchEffect(function() {
      if (api.listboxState.value !== ListboxStates.Open) return;
      if (!active.value) return;
      nextTick(function() {
        var _document$getElementB4;
        return (_document$getElementB4 = document.getElementById(id2)) == null ? void 0 : _document$getElementB4.scrollIntoView == null ? void 0 : _document$getElementB4.scrollIntoView({
          block: "nearest"
        });
      });
    });
    function handleClick(event) {
      if (props.disabled) return event.preventDefault();
      api.select(props.value);
      api.closeListbox();
      nextTick(function() {
        var _dom15;
        return (_dom15 = dom(api.buttonRef)) == null ? void 0 : _dom15.focus({
          preventScroll: true
        });
      });
    }
    function handleFocus() {
      if (props.disabled) return api.goToOption(Focus$1.Nothing);
      api.goToOption(Focus$1.Specific, id2);
    }
    function handleMove() {
      if (props.disabled) return;
      if (active.value) return;
      api.goToOption(Focus$1.Specific, id2);
    }
    function handleLeave() {
      if (props.disabled) return;
      if (!active.value) return;
      api.goToOption(Focus$1.Nothing);
    }
    return function() {
      var disabled = props.disabled;
      var slot = {
        active: active.value,
        selected: selected.value,
        disabled
      };
      var propsWeControl = {
        id: id2,
        role: "option",
        tabIndex: disabled === true ? void 0 : -1,
        "aria-disabled": disabled === true ? true : void 0,
        "aria-selected": selected.value === true ? selected.value : void 0,
        disabled: void 0,
        onClick: handleClick,
        onFocus: handleFocus,
        onPointermove: handleMove,
        onMousemove: handleMove,
        onPointerleave: handleLeave,
        onMouseleave: handleLeave
      };
      return render$W({
        props: _extends$1({}, props, propsWeControl),
        slot,
        attrs,
        slots,
        name: "ListboxOption"
      });
    };
  }
});
function useTreeWalker(_ref) {
  var container = _ref.container, accept = _ref.accept, walk2 = _ref.walk, enabled = _ref.enabled;
  watchEffect(function() {
    var root2 = container.value;
    if (!root2) return;
    if (enabled !== void 0 && !enabled.value) return;
    var acceptNode = Object.assign(function(node) {
      return accept(node);
    }, {
      acceptNode: accept
    });
    var walker = document.createTreeWalker(root2, NodeFilter.SHOW_ELEMENT, acceptNode, false);
    while (walker.nextNode()) {
      walk2(walker.currentNode);
    }
  });
}
var MenuStates;
(function(MenuStates2) {
  MenuStates2[MenuStates2["Open"] = 0] = "Open";
  MenuStates2[MenuStates2["Closed"] = 1] = "Closed";
})(MenuStates || (MenuStates = {}));
function nextFrame$1$1(cb) {
  requestAnimationFrame(function() {
    return requestAnimationFrame(cb);
  });
}
var MenuContext = /* @__PURE__ */ Symbol("MenuContext");
function useMenuContext(component) {
  var context = inject(MenuContext, null);
  if (context === null) {
    var err = new Error("<" + component + " /> is missing a parent <Menu /> component.");
    if (Error.captureStackTrace) Error.captureStackTrace(err, useMenuContext);
    throw err;
  }
  return context;
}
var Menu$1 = /* @__PURE__ */ defineComponent({
  name: "Menu",
  props: {
    as: {
      type: [Object, String],
      "default": "template"
    }
  },
  setup: function setup17(props, _ref) {
    var slots = _ref.slots, attrs = _ref.attrs;
    var menuState = ref(MenuStates.Closed);
    var buttonRef = ref(null);
    var itemsRef = ref(null);
    var items = ref([]);
    var searchQuery = ref("");
    var activeItemIndex = ref(null);
    var api = {
      menuState,
      buttonRef,
      itemsRef,
      items,
      searchQuery,
      activeItemIndex,
      closeMenu: function closeMenu() {
        menuState.value = MenuStates.Closed;
        activeItemIndex.value = null;
      },
      openMenu: function openMenu() {
        return menuState.value = MenuStates.Open;
      },
      goToItem: function goToItem(focus, id2) {
        var nextActiveItemIndex = calculateActiveIndex(focus === Focus$1.Specific ? {
          focus: Focus$1.Specific,
          id: id2
        } : {
          focus
        }, {
          resolveItems: function resolveItems2() {
            return items.value;
          },
          resolveActiveIndex: function resolveActiveIndex() {
            return activeItemIndex.value;
          },
          resolveId: function resolveId(item) {
            return item.id;
          },
          resolveDisabled: function resolveDisabled(item) {
            return item.dataRef.disabled;
          }
        });
        if (searchQuery.value === "" && activeItemIndex.value === nextActiveItemIndex) return;
        searchQuery.value = "";
        activeItemIndex.value = nextActiveItemIndex;
      },
      search: function search(value) {
        searchQuery.value += value.toLowerCase();
        var match2 = items.value.findIndex(function(item) {
          return item.dataRef.textValue.startsWith(searchQuery.value) && !item.dataRef.disabled;
        });
        if (match2 === -1 || match2 === activeItemIndex.value) return;
        activeItemIndex.value = match2;
      },
      clearSearch: function clearSearch() {
        searchQuery.value = "";
      },
      registerItem: function registerItem(id2, dataRef) {
        items.value.push({
          id: id2,
          dataRef
        });
      },
      unregisterItem: function unregisterItem(id2) {
        var nextItems = items.value.slice();
        var currentActiveItem = activeItemIndex.value !== null ? nextItems[activeItemIndex.value] : null;
        var idx = nextItems.findIndex(function(a2) {
          return a2.id === id2;
        });
        if (idx !== -1) nextItems.splice(idx, 1);
        items.value = nextItems;
        activeItemIndex.value = (function() {
          if (idx === activeItemIndex.value) return null;
          if (currentActiveItem === null) return null;
          return nextItems.indexOf(currentActiveItem);
        })();
      }
    };
    useWindowEvent("mousedown", function(event) {
      var _dom, _dom2, _dom3;
      var target2 = event.target;
      var active = document.activeElement;
      if (menuState.value !== MenuStates.Open) return;
      if ((_dom = dom(buttonRef)) == null ? void 0 : _dom.contains(target2)) return;
      if (!((_dom2 = dom(itemsRef)) == null ? void 0 : _dom2.contains(target2))) api.closeMenu();
      if (active !== document.body && (active == null ? void 0 : active.contains(target2))) return;
      if (!event.defaultPrevented) (_dom3 = dom(buttonRef)) == null ? void 0 : _dom3.focus({
        preventScroll: true
      });
    });
    provide(MenuContext, api);
    useOpenClosedProvider(computed(function() {
      var _match;
      return match(menuState.value, (_match = {}, _match[MenuStates.Open] = State.Open, _match[MenuStates.Closed] = State.Closed, _match));
    }));
    return function() {
      var slot = {
        open: menuState.value === MenuStates.Open
      };
      return render$W({
        props,
        slot,
        slots,
        attrs,
        name: "Menu"
      });
    };
  }
});
var MenuButton = /* @__PURE__ */ defineComponent({
  name: "MenuButton",
  props: {
    disabled: {
      type: Boolean,
      "default": false
    },
    as: {
      type: [Object, String],
      "default": "button"
    }
  },
  render: function render$111() {
    var _dom4;
    var api = useMenuContext("MenuButton");
    var slot = {
      open: api.menuState.value === MenuStates.Open
    };
    var propsWeControl = {
      ref: "el",
      id: this.id,
      type: "button",
      "aria-haspopup": true,
      "aria-controls": (_dom4 = dom(api.itemsRef)) == null ? void 0 : _dom4.id,
      "aria-expanded": this.$props.disabled ? void 0 : api.menuState.value === MenuStates.Open,
      onKeydown: this.handleKeyDown,
      onKeyup: this.handleKeyUp,
      onClick: this.handleClick
    };
    return render$W({
      props: _extends$1({}, this.$props, propsWeControl),
      slot,
      attrs: this.$attrs,
      slots: this.$slots,
      name: "MenuButton"
    });
  },
  setup: function setup18(props) {
    var api = useMenuContext("MenuButton");
    var id2 = "headlessui-menu-button-" + useId();
    function handleKeyDown(event) {
      switch (event.key) {
        // Ref: https://www.w3.org/TR/wai-aria-practices-1.2/#keyboard-interaction-13
        case Keys.Space:
        case Keys.Enter:
        case Keys.ArrowDown:
          event.preventDefault();
          event.stopPropagation();
          api.openMenu();
          nextTick(function() {
            var _dom5;
            (_dom5 = dom(api.itemsRef)) == null ? void 0 : _dom5.focus({
              preventScroll: true
            });
            api.goToItem(Focus$1.First);
          });
          break;
        case Keys.ArrowUp:
          event.preventDefault();
          event.stopPropagation();
          api.openMenu();
          nextTick(function() {
            var _dom6;
            (_dom6 = dom(api.itemsRef)) == null ? void 0 : _dom6.focus({
              preventScroll: true
            });
            api.goToItem(Focus$1.Last);
          });
          break;
      }
    }
    function handleKeyUp(event) {
      switch (event.key) {
        case Keys.Space:
          event.preventDefault();
          break;
      }
    }
    function handleClick(event) {
      if (props.disabled) return;
      if (api.menuState.value === MenuStates.Open) {
        api.closeMenu();
        nextTick(function() {
          var _dom7;
          return (_dom7 = dom(api.buttonRef)) == null ? void 0 : _dom7.focus({
            preventScroll: true
          });
        });
      } else {
        event.preventDefault();
        event.stopPropagation();
        api.openMenu();
        nextFrame$1$1(function() {
          var _dom8;
          return (_dom8 = dom(api.itemsRef)) == null ? void 0 : _dom8.focus({
            preventScroll: true
          });
        });
      }
    }
    return {
      id: id2,
      el: api.buttonRef,
      handleKeyDown,
      handleKeyUp,
      handleClick
    };
  }
});
var MenuItems = /* @__PURE__ */ defineComponent({
  name: "MenuItems",
  props: {
    as: {
      type: [Object, String],
      "default": "div"
    },
    "static": {
      type: Boolean,
      "default": false
    },
    unmount: {
      type: Boolean,
      "default": true
    }
  },
  render: function render$112() {
    var _api$items$value$api$, _dom9;
    var api = useMenuContext("MenuItems");
    var slot = {
      open: api.menuState.value === MenuStates.Open
    };
    var propsWeControl = {
      "aria-activedescendant": api.activeItemIndex.value === null ? void 0 : (_api$items$value$api$ = api.items.value[api.activeItemIndex.value]) == null ? void 0 : _api$items$value$api$.id,
      "aria-labelledby": (_dom9 = dom(api.buttonRef)) == null ? void 0 : _dom9.id,
      id: this.id,
      onKeydown: this.handleKeyDown,
      onKeyup: this.handleKeyUp,
      role: "menu",
      tabIndex: 0,
      ref: "el"
    };
    var passThroughProps = this.$props;
    return render$W({
      props: _extends$1({}, passThroughProps, propsWeControl),
      slot,
      attrs: this.$attrs,
      slots: this.$slots,
      features: Features.RenderStrategy | Features.Static,
      visible: this.visible,
      name: "MenuItems"
    });
  },
  setup: function setup19() {
    var api = useMenuContext("MenuItems");
    var id2 = "headlessui-menu-items-" + useId();
    var searchDebounce = ref(null);
    useTreeWalker({
      container: computed(function() {
        return dom(api.itemsRef);
      }),
      enabled: computed(function() {
        return api.menuState.value === MenuStates.Open;
      }),
      accept: function accept(node) {
        if (node.getAttribute("role") === "menuitem") return NodeFilter.FILTER_REJECT;
        if (node.hasAttribute("role")) return NodeFilter.FILTER_SKIP;
        return NodeFilter.FILTER_ACCEPT;
      },
      walk: function walk2(node) {
        node.setAttribute("role", "none");
      }
    });
    function handleKeyDown(event) {
      if (searchDebounce.value) clearTimeout(searchDebounce.value);
      switch (event.key) {
        // Ref: https://www.w3.org/TR/wai-aria-practices-1.2/#keyboard-interaction-12
        // @ts-expect-error Fallthrough is expected here
        case Keys.Space:
          if (api.searchQuery.value !== "") {
            event.preventDefault();
            event.stopPropagation();
            return api.search(event.key);
          }
        // When in type ahead mode, fallthrough
        case Keys.Enter:
          event.preventDefault();
          event.stopPropagation();
          if (api.activeItemIndex.value !== null) {
            var _document$getElementB;
            var _id = api.items.value[api.activeItemIndex.value].id;
            (_document$getElementB = document.getElementById(_id)) == null ? void 0 : _document$getElementB.click();
          }
          api.closeMenu();
          nextTick(function() {
            var _dom10;
            return (_dom10 = dom(api.buttonRef)) == null ? void 0 : _dom10.focus({
              preventScroll: true
            });
          });
          break;
        case Keys.ArrowDown:
          event.preventDefault();
          event.stopPropagation();
          return api.goToItem(Focus$1.Next);
        case Keys.ArrowUp:
          event.preventDefault();
          event.stopPropagation();
          return api.goToItem(Focus$1.Previous);
        case Keys.Home:
        case Keys.PageUp:
          event.preventDefault();
          event.stopPropagation();
          return api.goToItem(Focus$1.First);
        case Keys.End:
        case Keys.PageDown:
          event.preventDefault();
          event.stopPropagation();
          return api.goToItem(Focus$1.Last);
        case Keys.Escape:
          event.preventDefault();
          event.stopPropagation();
          api.closeMenu();
          nextTick(function() {
            var _dom11;
            return (_dom11 = dom(api.buttonRef)) == null ? void 0 : _dom11.focus({
              preventScroll: true
            });
          });
          break;
        case Keys.Tab:
          event.preventDefault();
          event.stopPropagation();
          break;
        default:
          if (event.key.length === 1) {
            api.search(event.key);
            searchDebounce.value = setTimeout(function() {
              return api.clearSearch();
            }, 350);
          }
          break;
      }
    }
    function handleKeyUp(event) {
      switch (event.key) {
        case Keys.Space:
          event.preventDefault();
          break;
      }
    }
    var usesOpenClosedState = useOpenClosed();
    var visible = computed(function() {
      if (usesOpenClosedState !== null) {
        return usesOpenClosedState.value === State.Open;
      }
      return api.menuState.value === MenuStates.Open;
    });
    return {
      id: id2,
      el: api.itemsRef,
      handleKeyDown,
      handleKeyUp,
      visible
    };
  }
});
var MenuItem = /* @__PURE__ */ defineComponent({
  name: "MenuItem",
  props: {
    as: {
      type: [Object, String],
      "default": "template"
    },
    disabled: {
      type: Boolean,
      "default": false
    }
  },
  setup: function setup20(props, _ref2) {
    var slots = _ref2.slots, attrs = _ref2.attrs;
    var api = useMenuContext("MenuItem");
    var id2 = "headlessui-menu-item-" + useId();
    var active = computed(function() {
      return api.activeItemIndex.value !== null ? api.items.value[api.activeItemIndex.value].id === id2 : false;
    });
    var dataRef = ref({
      disabled: props.disabled,
      textValue: ""
    });
    onMounted(function() {
      var _document$getElementB2, _document$getElementB3;
      var textValue = (_document$getElementB2 = document.getElementById(id2)) == null ? void 0 : (_document$getElementB3 = _document$getElementB2.textContent) == null ? void 0 : _document$getElementB3.toLowerCase().trim();
      if (textValue !== void 0) dataRef.value.textValue = textValue;
    });
    onMounted(function() {
      return api.registerItem(id2, dataRef);
    });
    onUnmounted(function() {
      return api.unregisterItem(id2);
    });
    watchEffect(function() {
      if (api.menuState.value !== MenuStates.Open) return;
      if (!active.value) return;
      nextTick(function() {
        var _document$getElementB4;
        return (_document$getElementB4 = document.getElementById(id2)) == null ? void 0 : _document$getElementB4.scrollIntoView == null ? void 0 : _document$getElementB4.scrollIntoView({
          block: "nearest"
        });
      });
    });
    function handleClick(event) {
      if (props.disabled) return event.preventDefault();
      api.closeMenu();
      nextTick(function() {
        var _dom12;
        return (_dom12 = dom(api.buttonRef)) == null ? void 0 : _dom12.focus({
          preventScroll: true
        });
      });
    }
    function handleFocus() {
      if (props.disabled) return api.goToItem(Focus$1.Nothing);
      api.goToItem(Focus$1.Specific, id2);
    }
    function handleMove() {
      if (props.disabled) return;
      if (active.value) return;
      api.goToItem(Focus$1.Specific, id2);
    }
    function handleLeave() {
      if (props.disabled) return;
      if (!active.value) return;
      api.goToItem(Focus$1.Nothing);
    }
    return function() {
      var disabled = props.disabled;
      var slot = {
        active: active.value,
        disabled
      };
      var propsWeControl = {
        id: id2,
        role: "menuitem",
        tabIndex: disabled === true ? void 0 : -1,
        "aria-disabled": disabled === true ? true : void 0,
        onClick: handleClick,
        onFocus: handleFocus,
        onPointermove: handleMove,
        onMousemove: handleMove,
        onPointerleave: handleLeave,
        onMouseleave: handleLeave
      };
      return render$W({
        props: _extends$1({}, props, propsWeControl),
        slot,
        attrs,
        slots,
        name: "MenuItem"
      });
    };
  }
});
var PopoverStates;
(function(PopoverStates2) {
  PopoverStates2[PopoverStates2["Open"] = 0] = "Open";
  PopoverStates2[PopoverStates2["Closed"] = 1] = "Closed";
})(PopoverStates || (PopoverStates = {}));
var PopoverContext = /* @__PURE__ */ Symbol("PopoverContext");
function usePopoverContext(component) {
  var context = inject(PopoverContext, null);
  if (context === null) {
    var err = new Error("<" + component + " /> is missing a parent <" + Popover.name + " /> component.");
    if (Error.captureStackTrace) Error.captureStackTrace(err, usePopoverContext);
    throw err;
  }
  return context;
}
var PopoverGroupContext = /* @__PURE__ */ Symbol("PopoverGroupContext");
function usePopoverGroupContext() {
  return inject(PopoverGroupContext, null);
}
var PopoverPanelContext = /* @__PURE__ */ Symbol("PopoverPanelContext");
function usePopoverPanelContext() {
  return inject(PopoverPanelContext, null);
}
var Popover = /* @__PURE__ */ defineComponent({
  name: "Popover",
  props: {
    as: {
      type: [Object, String],
      "default": "div"
    }
  },
  setup: function setup21(props, _ref) {
    var slots = _ref.slots, attrs = _ref.attrs;
    var buttonId = "headlessui-popover-button-" + useId();
    var panelId = "headlessui-popover-panel-" + useId();
    var popoverState = ref(PopoverStates.Closed);
    var button = ref(null);
    var panel = ref(null);
    var api = {
      popoverState,
      buttonId,
      panelId,
      panel,
      button,
      togglePopover: function togglePopover() {
        var _match;
        popoverState.value = match(popoverState.value, (_match = {}, _match[PopoverStates.Open] = PopoverStates.Closed, _match[PopoverStates.Closed] = PopoverStates.Open, _match));
      },
      closePopover: function closePopover() {
        if (popoverState.value === PopoverStates.Closed) return;
        popoverState.value = PopoverStates.Closed;
      },
      close: function close(focusableElement) {
        api.closePopover();
        var restoreElement = (function() {
          if (!focusableElement) return dom(api.button);
          if (focusableElement instanceof HTMLElement) return focusableElement;
          if (focusableElement.value instanceof HTMLElement) return dom(focusableElement);
          return dom(api.button);
        })();
        restoreElement == null ? void 0 : restoreElement.focus();
      }
    };
    provide(PopoverContext, api);
    useOpenClosedProvider(computed(function() {
      var _match2;
      return match(popoverState.value, (_match2 = {}, _match2[PopoverStates.Open] = State.Open, _match2[PopoverStates.Closed] = State.Closed, _match2));
    }));
    var registerBag = {
      buttonId,
      panelId,
      close: function close() {
        api.closePopover();
      }
    };
    var groupContext = usePopoverGroupContext();
    var registerPopover = groupContext == null ? void 0 : groupContext.registerPopover;
    function isFocusWithinPopoverGroup() {
      var _groupContext$isFocus, _dom, _dom2;
      return (_groupContext$isFocus = groupContext == null ? void 0 : groupContext.isFocusWithinPopoverGroup()) != null ? _groupContext$isFocus : ((_dom = dom(button)) == null ? void 0 : _dom.contains(document.activeElement)) || ((_dom2 = dom(panel)) == null ? void 0 : _dom2.contains(document.activeElement));
    }
    watchEffect(function() {
      return registerPopover == null ? void 0 : registerPopover(registerBag);
    });
    useWindowEvent("focus", function() {
      if (popoverState.value !== PopoverStates.Open) return;
      if (isFocusWithinPopoverGroup()) return;
      if (!button) return;
      if (!panel) return;
      api.closePopover();
    }, true);
    useWindowEvent("mousedown", function(event) {
      var _dom3, _dom4;
      var target2 = event.target;
      if (popoverState.value !== PopoverStates.Open) return;
      if ((_dom3 = dom(button)) == null ? void 0 : _dom3.contains(target2)) return;
      if ((_dom4 = dom(panel)) == null ? void 0 : _dom4.contains(target2)) return;
      api.closePopover();
      if (!isFocusableElement(target2, FocusableMode.Loose)) {
        var _dom5;
        event.preventDefault();
        (_dom5 = dom(button)) == null ? void 0 : _dom5.focus();
      }
    });
    return function() {
      var slot = {
        open: popoverState.value === PopoverStates.Open,
        close: api.close
      };
      return render$W({
        props,
        slot,
        slots,
        attrs,
        name: "Popover"
      });
    };
  }
});
var PopoverButton = /* @__PURE__ */ defineComponent({
  name: "PopoverButton",
  props: {
    as: {
      type: [Object, String],
      "default": "button"
    },
    disabled: {
      type: [Boolean],
      "default": false
    }
  },
  render: function render$113() {
    var api = usePopoverContext("PopoverButton");
    var slot = {
      open: api.popoverState.value === PopoverStates.Open
    };
    var propsWeControl = this.isWithinPanel ? {
      type: "button",
      onKeydown: this.handleKeyDown,
      onClick: this.handleClick
    } : {
      ref: "el",
      id: api.buttonId,
      type: "button",
      "aria-expanded": this.$props.disabled ? void 0 : api.popoverState.value === PopoverStates.Open,
      "aria-controls": dom(api.panel) ? api.panelId : void 0,
      disabled: this.$props.disabled ? true : void 0,
      onKeydown: this.handleKeyDown,
      onKeyup: this.handleKeyUp,
      onClick: this.handleClick
    };
    return render$W({
      props: _extends$1({}, this.$props, propsWeControl),
      slot,
      attrs: this.$attrs,
      slots: this.$slots,
      name: "PopoverButton"
    });
  },
  setup: function setup22(props) {
    var api = usePopoverContext("PopoverButton");
    var groupContext = usePopoverGroupContext();
    var closeOthers = groupContext == null ? void 0 : groupContext.closeOthers;
    var panelContext = usePopoverPanelContext();
    var isWithinPanel = panelContext === null ? false : panelContext === api.panelId;
    var activeElementRef = ref(null);
    var previousActiveElementRef = ref(typeof window === "undefined" ? null : document.activeElement);
    useWindowEvent("focus", function() {
      previousActiveElementRef.value = activeElementRef.value;
      activeElementRef.value = document.activeElement;
    }, true);
    return {
      isWithinPanel,
      el: isWithinPanel ? null : api.button,
      handleKeyDown: function handleKeyDown(event) {
        var _dom6, _dom7;
        if (isWithinPanel) {
          if (api.popoverState.value === PopoverStates.Closed) return;
          switch (event.key) {
            case Keys.Space:
            case Keys.Enter:
              event.preventDefault();
              event.stopPropagation();
              api.closePopover();
              (_dom6 = dom(api.button)) == null ? void 0 : _dom6.focus();
              break;
          }
        } else {
          switch (event.key) {
            case Keys.Space:
            case Keys.Enter:
              event.preventDefault();
              event.stopPropagation();
              if (api.popoverState.value === PopoverStates.Closed) closeOthers == null ? void 0 : closeOthers(api.buttonId);
              api.togglePopover();
              break;
            case Keys.Escape:
              if (api.popoverState.value !== PopoverStates.Open) return closeOthers == null ? void 0 : closeOthers(api.buttonId);
              if (!dom(api.button)) return;
              if (!((_dom7 = dom(api.button)) == null ? void 0 : _dom7.contains(document.activeElement))) return;
              api.closePopover();
              break;
            case Keys.Tab:
              if (api.popoverState.value !== PopoverStates.Open) return;
              if (!api.panel) return;
              if (!api.button) return;
              if (event.shiftKey) {
                var _dom8, _dom9;
                if (!previousActiveElementRef.value) return;
                if ((_dom8 = dom(api.button)) == null ? void 0 : _dom8.contains(previousActiveElementRef.value)) return;
                if ((_dom9 = dom(api.panel)) == null ? void 0 : _dom9.contains(previousActiveElementRef.value)) return;
                var focusableElements = getFocusableElements();
                var previousIdx = focusableElements.indexOf(previousActiveElementRef.value);
                var buttonIdx = focusableElements.indexOf(dom(api.button));
                if (buttonIdx > previousIdx) return;
                event.preventDefault();
                event.stopPropagation();
                focusIn(dom(api.panel), Focus.Last);
              } else {
                event.preventDefault();
                event.stopPropagation();
                focusIn(dom(api.panel), Focus.First);
              }
              break;
          }
        }
      },
      handleKeyUp: function handleKeyUp(event) {
        var _dom10, _dom11;
        if (isWithinPanel) return;
        if (event.key === Keys.Space) {
          event.preventDefault();
        }
        if (api.popoverState.value !== PopoverStates.Open) return;
        if (!api.panel) return;
        if (!api.button) return;
        switch (event.key) {
          case Keys.Tab:
            if (!previousActiveElementRef.value) return;
            if ((_dom10 = dom(api.button)) == null ? void 0 : _dom10.contains(previousActiveElementRef.value)) return;
            if ((_dom11 = dom(api.panel)) == null ? void 0 : _dom11.contains(previousActiveElementRef.value)) return;
            var focusableElements = getFocusableElements();
            var previousIdx = focusableElements.indexOf(previousActiveElementRef.value);
            var buttonIdx = focusableElements.indexOf(dom(api.button));
            if (buttonIdx > previousIdx) return;
            event.preventDefault();
            event.stopPropagation();
            focusIn(dom(api.panel), Focus.Last);
            break;
        }
      },
      handleClick: function handleClick() {
        if (props.disabled) return;
        if (isWithinPanel) {
          var _dom12;
          api.closePopover();
          (_dom12 = dom(api.button)) == null ? void 0 : _dom12.focus();
        } else {
          var _dom13;
          if (api.popoverState.value === PopoverStates.Closed) closeOthers == null ? void 0 : closeOthers(api.buttonId);
          (_dom13 = dom(api.button)) == null ? void 0 : _dom13.focus();
          api.togglePopover();
        }
      },
      handleFocus: function handleFocus() {
      }
    };
  }
});
var PopoverPanel = /* @__PURE__ */ defineComponent({
  name: "PopoverPanel",
  props: {
    as: {
      type: [Object, String],
      "default": "div"
    },
    "static": {
      type: Boolean,
      "default": false
    },
    unmount: {
      type: Boolean,
      "default": true
    },
    focus: {
      type: Boolean,
      "default": false
    }
  },
  render: function render$115() {
    var api = usePopoverContext("PopoverPanel");
    var slot = {
      open: api.popoverState.value === PopoverStates.Open,
      close: api.close
    };
    var propsWeControl = {
      ref: "el",
      id: this.id,
      onKeydown: this.handleKeyDown
    };
    return render$W({
      props: _extends$1({}, this.$props, propsWeControl),
      slot,
      attrs: this.$attrs,
      slots: this.$slots,
      features: Features.RenderStrategy | Features.Static,
      visible: this.visible,
      name: "PopoverPanel"
    });
  },
  setup: function setup24(props) {
    var focus = props.focus;
    var api = usePopoverContext("PopoverPanel");
    provide(PopoverPanelContext, api.panelId);
    onUnmounted(function() {
      api.panel.value = null;
    });
    watchEffect(function() {
      var _dom14;
      if (!focus) return;
      if (api.popoverState.value !== PopoverStates.Open) return;
      if (!api.panel) return;
      var activeElement = document.activeElement;
      if ((_dom14 = dom(api.panel)) == null ? void 0 : _dom14.contains(activeElement)) return;
      focusIn(dom(api.panel), Focus.First);
    });
    useWindowEvent("keydown", function(event) {
      var _dom15;
      if (api.popoverState.value !== PopoverStates.Open) return;
      if (!dom(api.panel)) return;
      if (event.key !== Keys.Tab) return;
      if (!document.activeElement) return;
      if (!((_dom15 = dom(api.panel)) == null ? void 0 : _dom15.contains(document.activeElement))) return;
      event.preventDefault();
      var result = focusIn(dom(api.panel), event.shiftKey ? Focus.Previous : Focus.Next);
      if (result === FocusResult.Underflow) {
        var _dom16;
        return (_dom16 = dom(api.button)) == null ? void 0 : _dom16.focus();
      } else if (result === FocusResult.Overflow) {
        if (!dom(api.button)) return;
        var elements = getFocusableElements();
        var buttonIdx = elements.indexOf(dom(api.button));
        var nextElements = elements.splice(buttonIdx + 1).filter(function(element) {
          var _dom17;
          return !((_dom17 = dom(api.panel)) == null ? void 0 : _dom17.contains(element));
        });
        if (focusIn(nextElements, Focus.First) === FocusResult.Error) {
          focusIn(document.body, Focus.First);
        }
      }
    });
    useWindowEvent("focus", function() {
      var _dom18;
      if (!focus) return;
      if (api.popoverState.value !== PopoverStates.Open) return;
      if (!dom(api.panel)) return;
      if ((_dom18 = dom(api.panel)) == null ? void 0 : _dom18.contains(document.activeElement)) return;
      api.closePopover();
    }, true);
    var usesOpenClosedState = useOpenClosed();
    var visible = computed(function() {
      if (usesOpenClosedState !== null) {
        return usesOpenClosedState.value === State.Open;
      }
      return api.popoverState.value === PopoverStates.Open;
    });
    return {
      id: api.panelId,
      el: api.panel,
      handleKeyDown: function handleKeyDown(event) {
        var _dom19, _dom20;
        switch (event.key) {
          case Keys.Escape:
            if (api.popoverState.value !== PopoverStates.Open) return;
            if (!dom(api.panel)) return;
            if (!((_dom19 = dom(api.panel)) == null ? void 0 : _dom19.contains(document.activeElement))) return;
            event.preventDefault();
            api.closePopover();
            (_dom20 = dom(api.button)) == null ? void 0 : _dom20.focus();
            break;
        }
      },
      visible
    };
  }
});
var LabelContext = /* @__PURE__ */ Symbol("LabelContext");
function useLabelContext() {
  var context = inject(LabelContext, null);
  if (context === null) {
    var err = new Error("You used a <Label /> component, but it is not inside a parent.");
    if (Error.captureStackTrace) Error.captureStackTrace(err, useLabelContext);
    throw err;
  }
  return context;
}
function useLabels(_temp) {
  var _ref = _temp === void 0 ? {} : _temp, _ref$slot = _ref.slot, slot = _ref$slot === void 0 ? {} : _ref$slot, _ref$name = _ref.name, name2 = _ref$name === void 0 ? "Label" : _ref$name, _ref$props = _ref.props, props = _ref$props === void 0 ? {} : _ref$props;
  var labelIds = ref([]);
  function register(value) {
    labelIds.value.push(value);
    return function() {
      var idx = labelIds.value.indexOf(value);
      if (idx === -1) return;
      labelIds.value.splice(idx, 1);
    };
  }
  provide(LabelContext, {
    register,
    slot,
    name: name2,
    props
  });
  return computed(function() {
    return labelIds.value.length > 0 ? labelIds.value.join(" ") : void 0;
  });
}
var Label = /* @__PURE__ */ defineComponent({
  name: "Label",
  props: {
    as: {
      type: [Object, String],
      "default": "label"
    },
    passive: {
      type: [Boolean],
      "default": false
    }
  },
  render: function render$117() {
    var _this$context = this.context, _this$context$name = _this$context.name, name2 = _this$context$name === void 0 ? "Label" : _this$context$name, _this$context$slot = _this$context.slot, slot = _this$context$slot === void 0 ? {} : _this$context$slot, _this$context$props = _this$context.props, props = _this$context$props === void 0 ? {} : _this$context$props;
    var _this$$props = this.$props, passive = _this$$props.passive, passThroughProps = _objectWithoutPropertiesLoose(_this$$props, ["passive"]);
    var propsWeControl = _extends$1({}, Object.entries(props).reduce(function(acc, _ref2) {
      var _Object$assign;
      var key = _ref2[0], value = _ref2[1];
      return Object.assign(acc, (_Object$assign = {}, _Object$assign[key] = unref(value), _Object$assign));
    }, {}), {
      id: this.id
    });
    var allProps = _extends$1({}, passThroughProps, propsWeControl);
    if (passive) delete allProps["onClick"];
    return render$W({
      props: allProps,
      slot,
      attrs: this.$attrs,
      slots: this.$slots,
      name: name2
    });
  },
  setup: function setup26() {
    var context = useLabelContext();
    var id2 = "headlessui-label-" + useId();
    onMounted(function() {
      return onUnmounted(context.register(id2));
    });
    return {
      id: id2,
      context
    };
  }
});
var RadioGroupContext = /* @__PURE__ */ Symbol("RadioGroupContext");
function useRadioGroupContext(component) {
  var context = inject(RadioGroupContext, null);
  if (context === null) {
    var err = new Error("<" + component + " /> is missing a parent <RadioGroup /> component.");
    if (Error.captureStackTrace) Error.captureStackTrace(err, useRadioGroupContext);
    throw err;
  }
  return context;
}
var RadioGroup = /* @__PURE__ */ defineComponent({
  name: "RadioGroup",
  emits: ["update:modelValue"],
  props: {
    as: {
      type: [Object, String],
      "default": "div"
    },
    disabled: {
      type: [Boolean],
      "default": false
    },
    modelValue: {
      type: [Object, String, Number, Boolean]
    }
  },
  render: function render$118() {
    var _this$$props = this.$props, passThroughProps = _objectWithoutPropertiesLoose(_this$$props, ["modelValue", "disabled"]);
    var propsWeControl = {
      ref: "el",
      id: this.id,
      role: "radiogroup",
      "aria-labelledby": this.labelledby,
      "aria-describedby": this.describedby,
      onKeydown: this.handleKeyDown
    };
    return render$W({
      props: _extends$1({}, passThroughProps, propsWeControl),
      slot: {},
      attrs: this.$attrs,
      slots: this.$slots,
      name: "RadioGroup"
    });
  },
  setup: function setup27(props, _ref) {
    var emit2 = _ref.emit;
    var radioGroupRef = ref(null);
    var options2 = ref([]);
    var labelledby = useLabels({
      name: "RadioGroupLabel"
    });
    var describedby = useDescriptions({
      name: "RadioGroupDescription"
    });
    var value = computed(function() {
      return props.modelValue;
    });
    var api = {
      options: options2,
      value,
      disabled: computed(function() {
        return props.disabled;
      }),
      firstOption: computed(function() {
        return options2.value.find(function(option) {
          if (option.propsRef.disabled) return false;
          return true;
        });
      }),
      containsCheckedOption: computed(function() {
        return options2.value.some(function(option) {
          return toRaw(option.propsRef.value) === toRaw(props.modelValue);
        });
      }),
      change: function change(nextValue) {
        var _options$value$find;
        if (props.disabled) return false;
        if (value.value === nextValue) return false;
        var nextOption = (_options$value$find = options2.value.find(function(option) {
          return toRaw(option.propsRef.value) === toRaw(nextValue);
        })) == null ? void 0 : _options$value$find.propsRef;
        if (nextOption == null ? void 0 : nextOption.disabled) return false;
        emit2("update:modelValue", nextValue);
        return true;
      },
      registerOption: function registerOption(action) {
        var _radioGroupRef$value;
        var orderMap = Array.from((_radioGroupRef$value = radioGroupRef.value) == null ? void 0 : _radioGroupRef$value.querySelectorAll('[id^="headlessui-radiogroup-option-"]')).reduce(function(lookup, element, index) {
          var _Object$assign;
          return Object.assign(lookup, (_Object$assign = {}, _Object$assign[element.id] = index, _Object$assign));
        }, {});
        options2.value.push(action);
        options2.value.sort(function(a2, z) {
          return orderMap[a2.id] - orderMap[z.id];
        });
      },
      unregisterOption: function unregisterOption(id3) {
        var idx = options2.value.findIndex(function(radio) {
          return radio.id === id3;
        });
        if (idx === -1) return;
        options2.value.splice(idx, 1);
      }
    };
    provide(RadioGroupContext, api);
    useTreeWalker({
      container: computed(function() {
        return dom(radioGroupRef);
      }),
      accept: function accept(node) {
        if (node.getAttribute("role") === "radio") return NodeFilter.FILTER_REJECT;
        if (node.hasAttribute("role")) return NodeFilter.FILTER_SKIP;
        return NodeFilter.FILTER_ACCEPT;
      },
      walk: function walk2(node) {
        node.setAttribute("role", "none");
      }
    });
    function handleKeyDown(event) {
      if (!radioGroupRef.value) return;
      if (!radioGroupRef.value.contains(event.target)) return;
      var all = options2.value.filter(function(option) {
        return option.propsRef.disabled === false;
      }).map(function(radio) {
        return radio.element;
      });
      switch (event.key) {
        case Keys.ArrowLeft:
        case Keys.ArrowUp:
          {
            event.preventDefault();
            event.stopPropagation();
            var result = focusIn(all, Focus.Previous | Focus.WrapAround);
            if (result === FocusResult.Success) {
              var activeOption = options2.value.find(function(option) {
                return option.element === document.activeElement;
              });
              if (activeOption) api.change(activeOption.propsRef.value);
            }
          }
          break;
        case Keys.ArrowRight:
        case Keys.ArrowDown:
          {
            event.preventDefault();
            event.stopPropagation();
            var _result = focusIn(all, Focus.Next | Focus.WrapAround);
            if (_result === FocusResult.Success) {
              var _activeOption = options2.value.find(function(option) {
                return option.element === document.activeElement;
              });
              if (_activeOption) api.change(_activeOption.propsRef.value);
            }
          }
          break;
        case Keys.Space:
          {
            event.preventDefault();
            event.stopPropagation();
            var _activeOption2 = options2.value.find(function(option) {
              return option.element === document.activeElement;
            });
            if (_activeOption2) api.change(_activeOption2.propsRef.value);
          }
          break;
      }
    }
    var id2 = "headlessui-radiogroup-" + useId();
    return {
      id: id2,
      labelledby,
      describedby,
      el: radioGroupRef,
      handleKeyDown
    };
  }
});
var OptionState;
(function(OptionState2) {
  OptionState2[OptionState2["Empty"] = 1] = "Empty";
  OptionState2[OptionState2["Active"] = 2] = "Active";
})(OptionState || (OptionState = {}));
var RadioGroupOption = /* @__PURE__ */ defineComponent({
  name: "RadioGroupOption",
  props: {
    as: {
      type: [Object, String],
      "default": "div"
    },
    value: {
      type: [Object, String, Number, Boolean]
    },
    disabled: {
      type: Boolean,
      "default": false
    }
  },
  render: function render$119() {
    var _this$$props2 = this.$props, passThroughProps = _objectWithoutPropertiesLoose(_this$$props2, ["value", "disabled"]);
    var slot = {
      checked: this.checked,
      disabled: this.disabled,
      active: Boolean(this.state & OptionState.Active)
    };
    var propsWeControl = {
      id: this.id,
      ref: "el",
      role: "radio",
      "aria-checked": this.checked ? "true" : "false",
      "aria-labelledby": this.labelledby,
      "aria-describedby": this.describedby,
      "aria-disabled": this.disabled ? true : void 0,
      tabIndex: this.tabIndex,
      onClick: this.disabled ? void 0 : this.handleClick,
      onFocus: this.disabled ? void 0 : this.handleFocus,
      onBlur: this.disabled ? void 0 : this.handleBlur
    };
    return render$W({
      props: _extends$1({}, passThroughProps, propsWeControl),
      slot,
      attrs: this.$attrs,
      slots: this.$slots,
      name: "RadioGroupOption"
    });
  },
  setup: function setup28(props) {
    var api = useRadioGroupContext("RadioGroupOption");
    var id2 = "headlessui-radiogroup-option-" + useId();
    var labelledby = useLabels({
      name: "RadioGroupLabel"
    });
    var describedby = useDescriptions({
      name: "RadioGroupDescription"
    });
    var optionRef = ref(null);
    var propsRef = computed(function() {
      return {
        value: props.value,
        disabled: props.disabled
      };
    });
    var state = ref(OptionState.Empty);
    onMounted(function() {
      return api.registerOption({
        id: id2,
        element: optionRef,
        propsRef
      });
    });
    onUnmounted(function() {
      return api.unregisterOption(id2);
    });
    var isFirstOption = computed(function() {
      var _api$firstOption$valu;
      return ((_api$firstOption$valu = api.firstOption.value) == null ? void 0 : _api$firstOption$valu.id) === id2;
    });
    var disabled = computed(function() {
      return api.disabled.value || props.disabled;
    });
    var checked = computed(function() {
      return toRaw(api.value.value) === toRaw(props.value);
    });
    return {
      id: id2,
      el: optionRef,
      labelledby,
      describedby,
      state,
      disabled,
      checked,
      tabIndex: computed(function() {
        if (disabled.value) return -1;
        if (checked.value) return 0;
        if (!api.containsCheckedOption.value && isFirstOption.value) return 0;
        return -1;
      }),
      handleClick: function handleClick() {
        var _optionRef$value;
        if (!api.change(props.value)) return;
        state.value |= OptionState.Active;
        (_optionRef$value = optionRef.value) == null ? void 0 : _optionRef$value.focus();
      },
      handleFocus: function handleFocus() {
        state.value |= OptionState.Active;
      },
      handleBlur: function handleBlur() {
        state.value &= ~OptionState.Active;
      }
    };
  }
});
var RadioGroupLabel = Label;
var Reason;
(function(Reason2) {
  Reason2["Finished"] = "finished";
  Reason2["Cancelled"] = "cancelled";
})(Reason || (Reason = {}));
var TreeStates;
(function(TreeStates2) {
  TreeStates2["Visible"] = "visible";
  TreeStates2["Hidden"] = "hidden";
})(TreeStates || (TreeStates = {}));
Features.RenderStrategy;
var commonjsGlobal = typeof globalThis !== "undefined" ? globalThis : typeof window !== "undefined" ? window : typeof global !== "undefined" ? global : typeof self !== "undefined" ? self : {};
function getDefaultExportFromCjs(x2) {
  return x2 && x2.__esModule && Object.prototype.hasOwnProperty.call(x2, "default") ? x2["default"] : x2;
}
function getAugmentedNamespace(n2) {
  if (Object.prototype.hasOwnProperty.call(n2, "__esModule")) return n2;
  var f2 = n2.default;
  if (typeof f2 == "function") {
    var a2 = function a3() {
      if (this instanceof a3) {
        return Reflect.construct(f2, arguments, this.constructor);
      }
      return f2.apply(this, arguments);
    };
    a2.prototype = f2.prototype;
  } else a2 = {};
  Object.defineProperty(a2, "__esModule", { value: true });
  Object.keys(n2).forEach(function(k2) {
    var d2 = Object.getOwnPropertyDescriptor(n2, k2);
    Object.defineProperty(a2, k2, d2.get ? d2 : {
      enumerable: true,
      get: function() {
        return n2[k2];
      }
    });
  });
  return a2;
}
var lodash$1 = { exports: {} };
/**
 * @license
 * Lodash <https://lodash.com/>
 * Copyright OpenJS Foundation and other contributors <https://openjsf.org/>
 * Released under MIT license <https://lodash.com/license>
 * Based on Underscore.js 1.8.3 <http://underscorejs.org/LICENSE>
 * Copyright Jeremy Ashkenas, DocumentCloud and Investigative Reporters & Editors
 */
var lodash = lodash$1.exports;
var hasRequiredLodash;
function requireLodash() {
  if (hasRequiredLodash) return lodash$1.exports;
  hasRequiredLodash = 1;
  (function(module, exports) {
    (function() {
      var undefined$1;
      var VERSION2 = "4.17.21";
      var LARGE_ARRAY_SIZE = 200;
      var CORE_ERROR_TEXT = "Unsupported core-js use. Try https://npms.io/search?q=ponyfill.", FUNC_ERROR_TEXT = "Expected a function", INVALID_TEMPL_VAR_ERROR_TEXT = "Invalid `variable` option passed into `_.template`";
      var HASH_UNDEFINED = "__lodash_hash_undefined__";
      var MAX_MEMOIZE_SIZE = 500;
      var PLACEHOLDER = "__lodash_placeholder__";
      var CLONE_DEEP_FLAG = 1, CLONE_FLAT_FLAG = 2, CLONE_SYMBOLS_FLAG = 4;
      var COMPARE_PARTIAL_FLAG = 1, COMPARE_UNORDERED_FLAG = 2;
      var WRAP_BIND_FLAG = 1, WRAP_BIND_KEY_FLAG = 2, WRAP_CURRY_BOUND_FLAG = 4, WRAP_CURRY_FLAG = 8, WRAP_CURRY_RIGHT_FLAG = 16, WRAP_PARTIAL_FLAG = 32, WRAP_PARTIAL_RIGHT_FLAG = 64, WRAP_ARY_FLAG = 128, WRAP_REARG_FLAG = 256, WRAP_FLIP_FLAG = 512;
      var DEFAULT_TRUNC_LENGTH = 30, DEFAULT_TRUNC_OMISSION = "...";
      var HOT_COUNT = 800, HOT_SPAN = 16;
      var LAZY_FILTER_FLAG = 1, LAZY_MAP_FLAG = 2, LAZY_WHILE_FLAG = 3;
      var INFINITY = 1 / 0, MAX_SAFE_INTEGER = 9007199254740991, MAX_INTEGER = 17976931348623157e292, NAN = 0 / 0;
      var MAX_ARRAY_LENGTH2 = 4294967295, MAX_ARRAY_INDEX = MAX_ARRAY_LENGTH2 - 1, HALF_MAX_ARRAY_LENGTH = MAX_ARRAY_LENGTH2 >>> 1;
      var wrapFlags = [
        ["ary", WRAP_ARY_FLAG],
        ["bind", WRAP_BIND_FLAG],
        ["bindKey", WRAP_BIND_KEY_FLAG],
        ["curry", WRAP_CURRY_FLAG],
        ["curryRight", WRAP_CURRY_RIGHT_FLAG],
        ["flip", WRAP_FLIP_FLAG],
        ["partial", WRAP_PARTIAL_FLAG],
        ["partialRight", WRAP_PARTIAL_RIGHT_FLAG],
        ["rearg", WRAP_REARG_FLAG]
      ];
      var argsTag = "[object Arguments]", arrayTag = "[object Array]", asyncTag = "[object AsyncFunction]", boolTag = "[object Boolean]", dateTag = "[object Date]", domExcTag = "[object DOMException]", errorTag = "[object Error]", funcTag = "[object Function]", genTag = "[object GeneratorFunction]", mapTag = "[object Map]", numberTag = "[object Number]", nullTag2 = "[object Null]", objectTag = "[object Object]", promiseTag = "[object Promise]", proxyTag = "[object Proxy]", regexpTag = "[object RegExp]", setTag = "[object Set]", stringTag = "[object String]", symbolTag2 = "[object Symbol]", undefinedTag2 = "[object Undefined]", weakMapTag = "[object WeakMap]", weakSetTag = "[object WeakSet]";
      var arrayBufferTag = "[object ArrayBuffer]", dataViewTag = "[object DataView]", float32Tag = "[object Float32Array]", float64Tag = "[object Float64Array]", int8Tag = "[object Int8Array]", int16Tag = "[object Int16Array]", int32Tag = "[object Int32Array]", uint8Tag = "[object Uint8Array]", uint8ClampedTag = "[object Uint8ClampedArray]", uint16Tag = "[object Uint16Array]", uint32Tag = "[object Uint32Array]";
      var reEmptyStringLeading = /\b__p \+= '';/g, reEmptyStringMiddle = /\b(__p \+=) '' \+/g, reEmptyStringTrailing = /(__e\(.*?\)|\b__t\)) \+\n'';/g;
      var reEscapedHtml = /&(?:amp|lt|gt|quot|#39);/g, reUnescapedHtml = /[&<>"']/g, reHasEscapedHtml = RegExp(reEscapedHtml.source), reHasUnescapedHtml = RegExp(reUnescapedHtml.source);
      var reEscape = /<%-([\s\S]+?)%>/g, reEvaluate = /<%([\s\S]+?)%>/g, reInterpolate = /<%=([\s\S]+?)%>/g;
      var reIsDeepProp = /\.|\[(?:[^[\]]*|(["'])(?:(?!\1)[^\\]|\\.)*?\1)\]/, reIsPlainProp = /^\w*$/, rePropName = /[^.[\]]+|\[(?:(-?\d+(?:\.\d+)?)|(["'])((?:(?!\2)[^\\]|\\.)*?)\2)\]|(?=(?:\.|\[\])(?:\.|\[\]|$))/g;
      var reRegExpChar = /[\\^$.*+?()[\]{}|]/g, reHasRegExpChar = RegExp(reRegExpChar.source);
      var reTrimStart = /^\s+/;
      var reWhitespace = /\s/;
      var reWrapComment = /\{(?:\n\/\* \[wrapped with .+\] \*\/)?\n?/, reWrapDetails = /\{\n\/\* \[wrapped with (.+)\] \*/, reSplitDetails = /,? & /;
      var reAsciiWord2 = /[^\x00-\x2f\x3a-\x40\x5b-\x60\x7b-\x7f]+/g;
      var reForbiddenIdentifierChars = /[()=,{}\[\]\/\s]/;
      var reEscapeChar = /\\(\\)?/g;
      var reEsTemplate = /\$\{([^\\}]*(?:\\.[^\\}]*)*)\}/g;
      var reFlags = /\w*$/;
      var reIsBadHex = /^[-+]0x[0-9a-f]+$/i;
      var reIsBinary = /^0b[01]+$/i;
      var reIsHostCtor = /^\[object .+?Constructor\]$/;
      var reIsOctal = /^0o[0-7]+$/i;
      var reIsUint = /^(?:0|[1-9]\d*)$/;
      var reLatin2 = /[\xc0-\xd6\xd8-\xf6\xf8-\xff\u0100-\u017f]/g;
      var reNoMatch = /($^)/;
      var reUnescapedString = /['\n\r\u2028\u2029\\]/g;
      var rsAstralRange2 = "\\ud800-\\udfff", rsComboMarksRange2 = "\\u0300-\\u036f", reComboHalfMarksRange2 = "\\ufe20-\\ufe2f", rsComboSymbolsRange2 = "\\u20d0-\\u20ff", rsComboRange2 = rsComboMarksRange2 + reComboHalfMarksRange2 + rsComboSymbolsRange2, rsDingbatRange2 = "\\u2700-\\u27bf", rsLowerRange2 = "a-z\\xdf-\\xf6\\xf8-\\xff", rsMathOpRange2 = "\\xac\\xb1\\xd7\\xf7", rsNonCharRange2 = "\\x00-\\x2f\\x3a-\\x40\\x5b-\\x60\\x7b-\\xbf", rsPunctuationRange2 = "\\u2000-\\u206f", rsSpaceRange2 = " \\t\\x0b\\f\\xa0\\ufeff\\n\\r\\u2028\\u2029\\u1680\\u180e\\u2000\\u2001\\u2002\\u2003\\u2004\\u2005\\u2006\\u2007\\u2008\\u2009\\u200a\\u202f\\u205f\\u3000", rsUpperRange2 = "A-Z\\xc0-\\xd6\\xd8-\\xde", rsVarRange2 = "\\ufe0e\\ufe0f", rsBreakRange2 = rsMathOpRange2 + rsNonCharRange2 + rsPunctuationRange2 + rsSpaceRange2;
      var rsApos2 = "['’]", rsAstral2 = "[" + rsAstralRange2 + "]", rsBreak2 = "[" + rsBreakRange2 + "]", rsCombo2 = "[" + rsComboRange2 + "]", rsDigits2 = "\\d+", rsDingbat2 = "[" + rsDingbatRange2 + "]", rsLower2 = "[" + rsLowerRange2 + "]", rsMisc2 = "[^" + rsAstralRange2 + rsBreakRange2 + rsDigits2 + rsDingbatRange2 + rsLowerRange2 + rsUpperRange2 + "]", rsFitz2 = "\\ud83c[\\udffb-\\udfff]", rsModifier2 = "(?:" + rsCombo2 + "|" + rsFitz2 + ")", rsNonAstral2 = "[^" + rsAstralRange2 + "]", rsRegional2 = "(?:\\ud83c[\\udde6-\\uddff]){2}", rsSurrPair2 = "[\\ud800-\\udbff][\\udc00-\\udfff]", rsUpper2 = "[" + rsUpperRange2 + "]", rsZWJ2 = "\\u200d";
      var rsMiscLower2 = "(?:" + rsLower2 + "|" + rsMisc2 + ")", rsMiscUpper2 = "(?:" + rsUpper2 + "|" + rsMisc2 + ")", rsOptContrLower2 = "(?:" + rsApos2 + "(?:d|ll|m|re|s|t|ve))?", rsOptContrUpper2 = "(?:" + rsApos2 + "(?:D|LL|M|RE|S|T|VE))?", reOptMod2 = rsModifier2 + "?", rsOptVar2 = "[" + rsVarRange2 + "]?", rsOptJoin2 = "(?:" + rsZWJ2 + "(?:" + [rsNonAstral2, rsRegional2, rsSurrPair2].join("|") + ")" + rsOptVar2 + reOptMod2 + ")*", rsOrdLower2 = "\\d*(?:1st|2nd|3rd|(?![123])\\dth)(?=\\b|[A-Z_])", rsOrdUpper2 = "\\d*(?:1ST|2ND|3RD|(?![123])\\dTH)(?=\\b|[a-z_])", rsSeq2 = rsOptVar2 + reOptMod2 + rsOptJoin2, rsEmoji2 = "(?:" + [rsDingbat2, rsRegional2, rsSurrPair2].join("|") + ")" + rsSeq2, rsSymbol2 = "(?:" + [rsNonAstral2 + rsCombo2 + "?", rsCombo2, rsRegional2, rsSurrPair2, rsAstral2].join("|") + ")";
      var reApos2 = RegExp(rsApos2, "g");
      var reComboMark2 = RegExp(rsCombo2, "g");
      var reUnicode2 = RegExp(rsFitz2 + "(?=" + rsFitz2 + ")|" + rsSymbol2 + rsSeq2, "g");
      var reUnicodeWord2 = RegExp([
        rsUpper2 + "?" + rsLower2 + "+" + rsOptContrLower2 + "(?=" + [rsBreak2, rsUpper2, "$"].join("|") + ")",
        rsMiscUpper2 + "+" + rsOptContrUpper2 + "(?=" + [rsBreak2, rsUpper2 + rsMiscLower2, "$"].join("|") + ")",
        rsUpper2 + "?" + rsMiscLower2 + "+" + rsOptContrLower2,
        rsUpper2 + "+" + rsOptContrUpper2,
        rsOrdUpper2,
        rsOrdLower2,
        rsDigits2,
        rsEmoji2
      ].join("|"), "g");
      var reHasUnicode2 = RegExp("[" + rsZWJ2 + rsAstralRange2 + rsComboRange2 + rsVarRange2 + "]");
      var reHasUnicodeWord2 = /[a-z][A-Z]|[A-Z]{2}[a-z]|[0-9][a-zA-Z]|[a-zA-Z][0-9]|[^a-zA-Z0-9 ]/;
      var contextProps = [
        "Array",
        "Buffer",
        "DataView",
        "Date",
        "Error",
        "Float32Array",
        "Float64Array",
        "Function",
        "Int8Array",
        "Int16Array",
        "Int32Array",
        "Map",
        "Math",
        "Object",
        "Promise",
        "RegExp",
        "Set",
        "String",
        "Symbol",
        "TypeError",
        "Uint8Array",
        "Uint8ClampedArray",
        "Uint16Array",
        "Uint32Array",
        "WeakMap",
        "_",
        "clearTimeout",
        "isFinite",
        "parseInt",
        "setTimeout"
      ];
      var templateCounter = -1;
      var typedArrayTags = {};
      typedArrayTags[float32Tag] = typedArrayTags[float64Tag] = typedArrayTags[int8Tag] = typedArrayTags[int16Tag] = typedArrayTags[int32Tag] = typedArrayTags[uint8Tag] = typedArrayTags[uin