import openTelemetry, { diag } from '@opentelemetry/api';
import { Resource, detectResourcesSync, browserDetectorSync } from '@opentelemetry/resources';
import { SemanticResourceAttributes } from '@opentelemetry/semantic-conventions';
import { SimpleSpanProcessor } from '@opentelemetry/sdk-trace-base';
import { ExportResultCode } from '@opentelemetry/core';
import { WebTracerProvider } from '@opentelemetry/sdk-trace-web';
import { OTLPTraceExporter as OTLPTraceExporter$1 } from '@opentelemetry/exporter-trace-otlp-http';

/**
 * An implementation of the {@link SpanProcessor} that converts the {@link Span}
 * to {@link ReadableSpan} and passes it to the configured exporter.
 *
 * Only spans that are sampled are converted.
 */
class OnStartSpanProcessor extends SimpleSpanProcessor {
    onStart(span, _parentContext) {
        return this.onEnd(span);
    }
}

/**
 * Builds and returns a {@link SpanExporter} that logs Honeycomb URLs for completed traces
 *
 * @remark This is not for production use.
 * @param options The {@link HoneycombOptions} used to configure the exporter
 * @returns the configured {@link ConsoleTraceLinkExporter} instance
 */
class ConsoleTraceLinkExporter {
    _traceUrl = '';
    _uniqueTraces = {};
    // eslint-disable-next-line no-console
    _log = console.log;
    constructor({ serviceName, team, environment, }) {
        this._traceUrl = buildTraceUrl(serviceName, team, environment);
    }
    export(spans, resultCallback) {
        if (this._traceUrl) {
            spans.forEach((span) => {
                const { traceId, spanId } = span.spanContext();
                if (!span.ended) {
                    if (!Object.keys(this._uniqueTraces).includes(traceId)) {
                        this._uniqueTraces[traceId] = spanId;
                        this._log(`Trace start: [${span.name}] - ${this._traceUrl}=${span.spanContext().traceId}`);
                    }
                }
                else if (this._uniqueTraces[traceId] === spanId) {
                    this._log(`Trace end: [${span.name}] - ${this._traceUrl}=${span.spanContext().traceId}`);
                }
            });
        }
        resultCallback({ code: ExportResultCode.SUCCESS });
    }
    shutdown() {
        return Promise.resolve();
    }
}
/**
 * Builds and returns a URL that is used to log when a trace is completed in the {@link ConsoleTraceLinkExporter}.
 *
 * @param serviceName the Honeycomb service name (or classic dataset) where data is stored
 * @param team the Honeycomb team
 * @param environment the Honeycomb environment
 * @returns
 */
function buildTraceUrl(serviceName, team, environment) {
    return `https://ui.honeycomb.io/${team}/environments/${environment}/datasets/${serviceName}/trace?trace_id`;
}

const environment = (process.env.CYPRESS_CONFIG_ENV || process.env.CYPRESS_INTERNAL_ENV || 'development');
const SERVICE_NAME = 'cypress-app';
class Telemetry {
    tracer;
    spans;
    activeSpanQueue;
    rootContext;
    rootAttributes;
    provider;
    exporter;
    isVerbose;
    constructor({ namespace, Provider, detectors, rootContextObject, version, SpanProcessor, exporter, resources = {}, isVerbose = false, }) {
        // For troubleshooting, set the log level to DiagLogLevel.DEBUG
        // import { diag, DiagConsoleLogger, DiagLogLevel } from '@opentelemetry/api'
        // diag.setLogger(new DiagConsoleLogger(), DiagLogLevel.ALL)
        this.isVerbose = isVerbose;
        // Setup default resources
        const resource = Resource.default().merge(new Resource({
            ...resources,
            [SemanticResourceAttributes.SERVICE_NAME]: SERVICE_NAME,
            [SemanticResourceAttributes.SERVICE_NAMESPACE]: namespace,
            [SemanticResourceAttributes.SERVICE_VERSION]: version,
        }));
        // Merge resources and create a new provider of the desired type.
        this.provider = new Provider({
            resource: resource.merge(detectResourcesSync({ detectors })),
        });
        // Setup the exporter
        if (SpanProcessor.name === 'BatchSpanProcessor') {
            this.provider.addSpanProcessor(new SpanProcessor(exporter, {
                // Double the max queue size, We were seeing telemetry bursts that would result in loosing the top span.
                maxQueueSize: 4056,
            }));
        }
        else {
            this.provider.addSpanProcessor(new SpanProcessor(exporter));
        }
        // if local visualizations enabled, create composite exporter configured
        // to send to both local exporter and main exporter
        const honeyCombConsoleLinkExporter = new ConsoleTraceLinkExporter({
            serviceName: SERVICE_NAME,
            team: 'cypress',
            environment: (environment === 'production' ? 'cypress-app' : 'cypress-app-staging'),
        });
        this.provider.addSpanProcessor(new OnStartSpanProcessor(honeyCombConsoleLinkExporter));
        // Initialize the provider
        this.provider.register();
        // Save off the tracer
        this.tracer = openTelemetry.trace.getTracer('cypress', version);
        // store off the root context to apply to new spans
        this.setRootContext(rootContextObject);
        this.spans = {};
        this.activeSpanQueue = [];
        this.exporter = exporter;
    }
    /**
     * Starts a span with the given name. Stores off the span with the name as a key for later retrieval.
     * @param name - the span name
     * @param key - they key associated with the span, to be used to retrieve the span, if not specified, the name is used.
     * @param attachType - Should this span be attached as a new root span or a child of the previous root span.
     * @param name - Set true if this span should have child spans of it's own.
     * @param opts - pass through for otel span opts
     * @returns Span | undefined
     */
    startSpan({ name, attachType = 'child', active = false, parentSpan, opts = {}, key, isVerbose = false, }) {
        // Currently the latest span replaces any previous open or closed span and you can no longer access the replaced span.
        // This works well enough for now but may cause issue in the future.
        // if the span is declared in verbose mode, but verbosity is disabled, no-op the span creation
        if (isVerbose && !this.isVerbose) {
            return undefined;
        }
        let span;
        let parent;
        if (attachType === 'root' || (this.activeSpanQueue.length < 1 && !parentSpan)) {
            if (this.rootContext) {
                // Start span with external context
                span = this.tracer.startSpan(name, opts, this.rootContext);
                // This can only apply attributes set on the external root set up until the point at which it was sent.
                if (this.rootAttributes) {
                    span.setAttributes(this.rootAttributes);
                }
            }
            else {
                // Start span with no context
                span = this.tracer.startSpan(name, opts);
            }
        }
        else { // attach type must be child
            // Prefer passed in parent
            parent = parentSpan || this.activeSpanQueue[this.activeSpanQueue.length - 1];
            // Create a context from the active span.
            const ctx = openTelemetry.trace.setSpan(openTelemetry.context.active(), parent);
            // Start span with parent context.
            span = this.tracer.startSpan(name, opts, ctx);
        }
        //span keys must be unique, names do not.
        if (environment === 'development' && key && key in this.spans) {
            throw new Error(`Span key ${key} rejected. Span key already exists in spans map.`);
        }
        // Save off span
        const spanKey = key || name;
        this.spans[spanKey] = span;
        // Setup function on span to recursively get parent attributes.
        // Not bothering with types here since we only need this function within this function.
        // @ts-expect-error
        span.getAllAttributes = () => {
            // @ts-expect-error
            const parentAttributes = parent && parent.getAllAttributes ? parent.getAllAttributes() : {};
            const allAttributes = {
                // @ts-expect-error
                ...span.attributes,
                ...parentAttributes,
            };
            // never propagate name
            delete allAttributes['name'];
            return allAttributes;
        };
        // override the end function to allow us to pop the span off the queue if found.
        const _end = span.end;
        span.end = (endTime) => {
            if (active) {
                // find the span in the queue by spanId
                const index = this.activeSpanQueue.findIndex((element) => {
                    return element.spanContext().spanId === span.spanContext().spanId;
                });
                // if span exists, remove it from the queue
                if (index > -1) {
                    this.activeSpanQueue.splice(index, 1);
                }
            }
            // On span end recursively grab parent attributes
            // @ts-ignore
            if (parent && parent.getAllAttributes) {
                // @ts-ignore
                span.setAttributes(parent.getAllAttributes());
            }
            _end.call(span, endTime);
        };
        // If this is an active span, set it as the new active span
        if (active) {
            this.activeSpanQueue.push(span);
        }
        return span;
    }
    /**
     * Return requested span
     * @param name - span name to retrieve
     * @returns Span | undefined
     */
    getSpan(name) {
        return this.spans[name];
    }
    /**
     * Search the span queue for the active span that meets the criteria
     * @param fn - function to search the active spans
     * @returns Span | undefined
     */
    findActiveSpan(fn) {
        return this.activeSpanQueue.find(fn);
    }
    /**
     * Ends specified active span and any active child spans
     * @param span - span to end
     */
    endActiveSpanAndChildren(span) {
        if (!span) {
            return;
        }
        const startIndex = this.activeSpanQueue.findIndex((element) => {
            return element.spanContext().spanId === span.spanContext().spanId;
        });
        this.activeSpanQueue.slice(startIndex).forEach((spanToEnd) => {
            span.setAttribute('spanEndedPrematurely', true);
            spanToEnd?.end();
        });
    }
    /**
     * Returns the context object for the active span.
     * @returns the context
     */
    getActiveContextObject() {
        const rootSpan = this.activeSpanQueue[this.activeSpanQueue.length - 1];
        // If no root span, nothing to return
        if (!rootSpan) {
            return {};
        }
        const ctx = openTelemetry.trace.setSpan(openTelemetry.context.active(), rootSpan);
        let myCtx = {};
        openTelemetry.propagation.inject(ctx, myCtx);
        // @ts-expect-error
        return { context: myCtx, attributes: rootSpan.getAllAttributes() };
    }
    /**
     * Gets a list of the resources currently set on the provider.
     * @returns Attributes of resources
     */
    getResources() {
        return this.provider.resource.attributes;
    }
    /**
     * Shuts down telemetry and flushes any batched spans.
     * @returns promise
     */
    shutdown() {
        return this.provider.shutdown();
    }
    /**
     * Returns the assigned exporter
     * @returns SpanExporter
     */
    getExporter() {
        return this.exporter;
    }
    /**
     * Sets or resets the root context for spans
     * @param rootContextObject
     */
    setRootContext(rootContextObject) {
        // store off the root context to apply to new spans
        if (rootContextObject && rootContextObject.context && rootContextObject.context.traceparent) {
            this.rootContext = openTelemetry.propagation.extract(openTelemetry.context.active(), rootContextObject.context);
            this.rootAttributes = rootContextObject.attributes;
        }
    }
}
/**
 * The telemetry Noop class is used when telemetry is disabled.
 * It should mirror all the existing functions in telemetry, but no-op for
 * all operations.
 */
class TelemetryNoop {
    startSpan(arg) {
        return undefined;
    }
    getSpan(name) {
        return undefined;
    }
    findActiveSpan(fn) {
        return undefined;
    }
    endActiveSpanAndChildren(span) {
        return undefined;
    }
    getActiveContextObject() {
        return {};
    }
    getResources() {
        return {};
    }
    shutdown() {
        return Promise.resolve();
    }
    getExporter() {
        return undefined;
    }
    setRootContext(rootContextObject) {
        return undefined;
    }
}

/**
 * Collector Trace Exporter for Node
 */
class OTLPTraceExporter extends OTLPTraceExporter$1 {
    ws;
    delayedExport;
    constructor() {
        super({});
        this.delayedExport = [];
    }
    /**
     * attaches the websocket and replays any exports called without it
     * @param ws - the web socket.
     */
    attachWebSocket(ws) {
        this.ws = ws;
        this.delayedExport.forEach(({ items, resultCallback }) => {
            this.export(items, resultCallback);
        });
    }
    /**
     * Overrides export to delay sending spans if encryption is needed and there is no attached projectId
     * @param items
     * @param resultCallback
     */
    export(items, resultCallback) {
        if (!this.ws) {
            this.delayedExport.push({ items, resultCallback });
        }
        else {
            super.export(items, resultCallback);
        }
    }
    /**
     * Overrides the send method to use a websocket instead of http
     * @param objects
     * @param onSuccess
     * @param onError
     * @returns
     */
    send(objects, onSuccess, onError) {
        if (this._shutdownOnce.isCalled) {
            diag.debug('Shutdown already started. Cannot send objects');
            return;
        }
        const serviceRequest = JSON.stringify(this.convert(objects));
        const promise = Promise.resolve().then(() => {
            return new Promise((resolve, reject) => {
                this.ws.emit('backend:request', 'telemetry', serviceRequest, (res) => {
                    if (res && res.error) {
                        reject(res.error);
                    }
                    resolve();
                });
            });
        }).then(onSuccess, onError);
        this._sendingPromises.push(promise);
        const popPromise = () => {
            const index = this._sendingPromises.indexOf(promise);
            this._sendingPromises.splice(index, 1);
        };
        promise.then(popPromise, popPromise);
    }
}

let telemetryInstance = new TelemetryNoop;
/**
 * Initialize the telemetry singleton
 * @param namespace - namespace to apply to the singleton
 * @param config - object containing the version
 * @returns void
 */
const init = ({ namespace, config }) => {
    // If we don't have cypress_telemetry setup on window don't even bother making the global instance
    if (!window.__CYPRESS_TELEMETRY__) {
        return;
    }
    // Telemetry only needs to be initialized once.
    if (telemetryInstance instanceof Telemetry) {
        throw ('Telemetry instance has already be initialized');
    }
    const { context, resources, isVerbose } = window.__CYPRESS_TELEMETRY__;
    // We always use the websocket exporter for browser telemetry
    const exporter = new OTLPTraceExporter();
    telemetryInstance = new Telemetry({
        namespace,
        Provider: WebTracerProvider,
        detectors: [
            browserDetectorSync,
        ],
        rootContextObject: context,
        version: config?.version,
        exporter,
        // Because otel is lame we need to use the simple span processor instead of the batch processor
        // or we risk losing spans when the browser navigates.
        // TODO: create a browser batch span processor to account for navigation.
        // See https://github.com/open-telemetry/opentelemetry-js/issues/2613
        SpanProcessor: SimpleSpanProcessor,
        resources,
        isVerbose,
    });
    window.cypressTelemetrySingleton = telemetryInstance;
    return;
};
/**
 * If telemetry has already been setup, attach this singleton to this instance
 * @returns
 */
const attach = () => {
    if (window.cypressTelemetrySingleton) {
        telemetryInstance = window.cypressTelemetrySingleton;
        return;
    }
};
const telemetry = {
    init,
    attach,
    startSpan: (arg) => telemetryInstance.startSpan(arg),
    getSpan: (arg) => telemetryInstance.getSpan(arg),
    findActiveSpan: (arg) => telemetryInstance.findActiveSpan(arg),
    endActiveSpanAndChildren: (arg) => telemetryInstance.endActiveSpanAndChildren(arg),
    getActiveContextObject: () => telemetryInstance.getActiveContextObject(),
    shutdown: () => telemetryInstance.shutdown(),
    attachWebSocket: (ws) => telemetryInstance.getExporter()?.attachWebSocket(ws),
    setRootContext: (context) => (telemetryInstance.setRootContext(context)),
};

export { telemetry };
